/*******************************************************************************
 * Utils.h
 *
 * Header for utils.cpp.
 * Also contains type definitions and short inline functions.
 */

#ifndef __UTILS_H__
#define __UTILS_H__

#include <math.h>
#include <signal.h>
#include <vector>

#include "imageLib.h"

#ifndef uint
#define uint unsigned int
#endif


// read an image and perhaps tell the user you're doing so
void ReadImageVerb(CImage& img, const char* filename, int verbose);

// write out an image and perhaps tell the user you're doing so
void WriteImageVerb(CImage& img, const char* filename, int verbose);


//=========== Convenience classes ============

// Convenience class for accessing an array as if it were an image
template <class T>
struct ArrImage {
    int width, height, nBands, rowLen, length;
    T *arr;

    ArrImage() {}

    ArrImage(int _width, int _height, int _nBands) {
        setSize(_width, _height, _nBands);
    }

    ArrImage(int _width, int _height, int _nBands, T *_arr) {
        setSize(_width, _height, _nBands);
        setData(_arr);
    }

    void setSize(int _width, int _height, int _nBands) {
        height = _height;
        width = _width;
        nBands = _nBands;
        rowLen = width * nBands;
        length = rowLen * height;
    }
    
    void setData(T *_arr) {
        arr = _arr;
    }

    inline T &Pixel(int x, int y, int b) {
        return arr[y * rowLen + x * nBands + b];
    }

    inline T &Pixel(int x, int y) {  // can use ONLY IF nBands == 1
        return arr[y * width + x];
    }
};


//============ Primary utility functions ============

// Convert image types, optionally scale and offset.
// Performs proper rounding and boundary check.
template <class T1, class T2>
void convertImageType(CImageOf<T1> &src, CImageOf<T2> &dst, 
                      double scale = 1.0, double offset = 0.0);

// Utility for converting RGB to grey
template <class T>  // Channel must be B, G, R order
void convertRGBtoGrey(CImageOf<T> &src, CImageOf<T> &dst);

// Convert grey-scale or RGBA images to RGB
template <class T>
void convertToRGB(CImageOf<T> &src, CImageOf<T> &dst);
template <class T>
CImageOf<T> convertToRGB(CImageOf<T> &src);

// Converting RGB to YCbCr
template <class T1, class T2>
void convertRGBtoYCbCr(CImageOf<T1> &imRGB, CImageOf<T2> &imYCbCr);

// Converting RGB to YCbCr, producing three separate grey-scale image imY, imCb, and imCr
template <class T1, class T2>
void convertRGBtoYCbCr(CImageOf<T1> &imRGB, CImageOf<T2> &imY, CImageOf<T2> &imCb, CImageOf<T2> &imCr);

// Converting YCbCr to RGB
template <class T1, class T2>
void convertYCbCrtoRGB(CImageOf<T1> &imYCbCr, CImageOf<T2> &imRGB);

// Converting YCbCr to RGB, given three separate grey scale images imY, imCb, and imCr
template <class T1, class T2>
void convertYCbCrtoRGB(CImageOf<T1> &imY, CImageOf<T1> &imCb, CImageOf<T1> &imCr, CImageOf<T2> &imRGB);

// The gaussian function
double gaussian(double x, double mu, double sigma);

// Generate a gaussian sample of 0 mean, unit var. using Box-Muller transform
// See Wikipedia: Box-Muller transform (google)
double boxMullerGaussian();

// Adding gaussian noise to an image
template <class T>
void addGaussianNoise(CImageOf<T> &src, CImageOf<T> &dst, double sigma, 
                      double min_val, double max_val);

// Convolve an image with gaussian kernel
template <class T1, class T2>
void gaussianSmooth(CImageOf<T1> &src, CImageOf<T2> &dst, double sigma,
                     CByteImage *imMask = NULL, unsigned char maskVal = 0);

// Calculate the signal to noise ratio
template <class T>
double signal2noise(CImageOf<T> &imOrig, CImageOf<T> &imNoisy,
                    double *rms_error = NULL);

// Check whether the two given image has the same size (w*h)
template <class T1, class T2>
bool sizeMatch(CImageOf<T1> &im1, CImageOf<T2> &im2);

// Conversion between ArrImage and CImage (assume ArrImage is already allocated)
template <class TA, class TC>
void convertArrImageToCImage(ArrImage<TA> &src, CImageOf<TC> &dst);
template <class TC, class TA>
void convertCImageToArrImage(CImageOf<TC> &src, ArrImage<TA> &dst);


// ========== inline functions ==========

template <class T>
inline static T sq(T val) {
    return val * val;
}

template <class T>
inline static T cube(T val) {
    return val * val * val;
}

// Inner product of two vectors (arrays)
template <class T1, class T2>
inline static double inner_prod(T1 *v1, T2 *v2, int len) {
    double sum = 0;
    for(int i=0; i<len; i++)
        sum += v1[i] * v2[i];
    return sum;
}

#endif
