/*******************************************************************************
 * Utils.cpp
 *
 * Utility functions, provided with SLRF denoise demo.  
 * 
 * Contains some routines from early projects in collaboration with 
 * Daniel Scharstein and Amy Briggs.
 *
 * Yunpeng Li
 */

#include <stdlib.h>
#include <assert.h>
#include <signal.h>
#include <math.h>
#include <vector>
#include <map>

#ifdef WIN32
#include <direct.h>
#else
#include <unistd.h>
#endif

#include "Utils.h"

using namespace std;


// Color channel index corresponding to memory layout
// Scharstein-Szeliski library use B-G-R memory layout
#define R           2
#define G           1
#define B           0
#define ALPHA_CH    3



// read an image and perhaps tell the user you're doing so
void ReadImageVerb(CImage& img, const char* filename, int verbose) 
{
	if (verbose)
		fprintf(stdout, "Reading image %s\n", filename);
    fflush(stdout);
	ReadImage(img, filename);
}


// write out an image and perhaps tell the user you're doing so
void WriteImageVerb(CImage& img, const char* filename, int verbose) 
{
	if (verbose)
		fprintf(stdout, "Writing image %s\n", filename);
    fflush(stdout);
	WriteImage(img, filename);
}



//============ Primary utility functions ============


// Convert image types, optionally scale and offset.
// Performs proper rounding and boundary check.
template <class T1, class T2>
void convertImageType(CImageOf<T1> &src, CImageOf<T2> &dst, 
                      double scale, double offset)
{
    CShape sh = src.Shape();
    int w = sh.width, h = sh.height, nb = sh.nBands;
    dst.ReAllocate(sh);

    bool scale_offset = (scale != 1.0) || (offset != 0.0);
    const double min_val = dst.MinVal(), max_val = dst.MaxVal();
    const double bias = 0.5 - (T2)0.5;

    for(int y=0; y<h; y++) {
        T1 *s = &src.Pixel(0, y, 0);
        T2 *t = &dst.Pixel(0, y, 0);
        if(scale_offset) {
            for(int xb=0; xb<w*nb; xb++) {
                double val = s[xb] * scale + offset;
                val = max(min_val, min(max_val, val));
                if(bias != 0)
                    val = val > 0? val + bias : val - bias;
                t[xb] = (T2)val;
            }
        }
        else {
            for(int xb=0; xb<w*nb; xb++) {
                double val = max(min_val, min(max_val, (double)s[xb]));
                if(bias != 0)
                    val = val > 0? val + bias : val - bias;
                t[xb] = (T2)val;
            }
        }
    }
}


// Utility for converting RGB to grey (OK if already grey)
template <class T>  // Channel must be B-G-R order
void convertRGBtoGrey(CImageOf<T> &src, CImageOf<T> &dst) 
{
    CShape sShape = src.Shape();
    // Must be RGB image if not already grey
	int srcBands = sShape.nBands;
    if(srcBands == 1) {
        CopyPixels(src, dst);
        return;
    }
    if(srcBands < 3)
        throw CError("ConvertRGBtoGray: source does not have 1 or "
            "at least 3 bands\n");
    CShape dShape(sShape.width, sShape.height, 1);
    dst.ReAllocate(dShape);
    
    const double bias = 0.5 - (T)0.5;
    for(int y=0; y<dShape.height; y++) {
        T* xSrc = &src.Pixel(0, y, 0);
        T* xDst = &dst.Pixel(0, y, 0);
        for(int x = 0; x < dShape.width; x++, xSrc += srcBands, xDst++) {
            double val = 0.114 * (*xSrc) + 0.587 * (*(xSrc+1)) + 
                0.299 * (*(xSrc+2));
            *xDst = (T)(val + bias);
        }
    }
}


// Convert grey-scale or RGBA images to RGB
template <class T>
void convertToRGB(CImageOf<T> &src, CImageOf<T> &dst) 
{
    int w = src.Shape().width, h = src.Shape().height, nb = src.Shape().nBands;
    if(nb != 1 && nb != 3 && nb != 4)
        printf("Warning: In convertToRGB: Unexpected number of channels (%d) "
            "in input image\n", nb);
    dst.ReAllocate(CShape(w, h, 3));
    if(nb == 2) {
        dst.ClearPixels();
    }
    if(nb == 1) {
        for(int y=0; y<h; y++) {
            for(int x=0; x<w; x++) {
                dst.Pixel(x, y, R) = dst.Pixel(x, y, G) = 
                    dst.Pixel(x, y, B) = src.Pixel(x, y, 0);
            }
        }
    }
    else {
        for(int y=0; y<h; y++) {
            for(int x=0; x<w; x++) {
                for(int b=0; b<min(3,nb); b++) {
                    dst.Pixel(x, y, b) = src.Pixel(x, y, b);
                }
            }
        }
    }
}

template <class T>
CImageOf<T> convertToRGB(CImageOf<T> &src) 
{
    CImageOf<T> dst;
    convertToRGB(src, dst);
    return dst;
}


// RGB-YCbCr conversion coefficients (RGB order)
static const double Kb = 0.114;
static const double Kr = 0.299;
// YCbCr = Matrix * RGB + offset
static const double RGB2YCbCr_coeff[3][3] = {
    {Kr,                1-Kr-Kb,                Kb              },
    {-0.5*Kr/(1-Kb),    -0.5*(1-Kr-Kb)/(1-Kb),  0.5             },
    {0.5,               -0.5*(1-Kr-Kb)/(1-Kr),  -0.5*Kb/(1-Kr)  }
};
// Assume pixel value range is [0,255]
static const double digital_offset[3] = {0.0, 127.5, 127.5};
// Inverted matrix, assuming Kb = 0.114, Kr = 0.299
const double YCbCr2RGB_coeff[3][3] = {
    {1.0,   0.0,        1.4020},
    {1.0,   -0.3441,    -0.7141},
    {1.0,   1.7720,     0.0}
};


// Converting RGB to YCbCr
template <class T1, class T2>
void convertRGBtoYCbCr(CImageOf<T1> &imRGB, CImageOf<T2> &imYCbCr) 
{
    const double bias = 0.5 - (T2)0.5;
    CShape sh = imRGB.Shape();
    if(sh.nBands < 3)
        throw CError("ConvertRGBtoYCbCr: source does not have "
            "at least 3 bands\n");
    imYCbCr.ReAllocate(sh);

    for(int y=0; y<sh.height; y++) {
        T1 *src = &imRGB.Pixel(0, y, 0);
        T2 *dst = &imYCbCr.Pixel(0, y, 0);
        for(int x=0; x<sh.width*sh.nBands; x+=sh.nBands) {
            double Y = RGB2YCbCr_coeff[0][0] * src[x+R] + 
                RGB2YCbCr_coeff[0][1] * src[x+G] + 
                RGB2YCbCr_coeff[0][2] * src[x+B];
            double Cb = RGB2YCbCr_coeff[1][0] * src[x+R] + 
                RGB2YCbCr_coeff[1][1] * src[x+G] + 
                RGB2YCbCr_coeff[1][2] * src[x+B];
            double Cr = RGB2YCbCr_coeff[2][0] * src[x+R] + 
                RGB2YCbCr_coeff[2][1] * src[x+G] + 
                RGB2YCbCr_coeff[2][2] * src[x+B];
            dst[x] = (T2)(Y + bias);
            dst[x+1] = (T2)(Cb + (127.5 + bias));
            dst[x+2] = (T2)(Cr + (127.5 + bias));

            if(sh.nBands > 3) {
                for(int i=3; i<sh.nBands; i++)
                    dst[x+i] = (T2)(src[x+i] + bias);  // copy the rest
            }
        }
    }
}


// Converting RGB to YCbCr, producing three separate grey-scale image 
// imY, imCb, and imCr
template <class T1, class T2>
void convertRGBtoYCbCr(CImageOf<T1> &imRGB, CImageOf<T2> &imY, 
                       CImageOf<T2> &imCb, CImageOf<T2> &imCr) 
{
    CImageOf<T2> imYCbCr;
    convertRGBtoYCbCr(imRGB, imYCbCr);

    CShape sh = imYCbCr.Shape();
    imY.ReAllocate(CShape(sh.width, sh.height, 1));
    imCb.ReAllocate(CShape(sh.width, sh.height, 1));
    imCr.ReAllocate(CShape(sh.width, sh.height, 1));

    for(int y=0; y<sh.height; y++) {
        for(int x=0; x<sh.width; x++) {
            imY.Pixel(x, y, 0) = imYCbCr.Pixel(x, y, 0);
            imCb.Pixel(x, y, 0) = imYCbCr.Pixel(x, y, 1);
            imCr.Pixel(x, y, 0) = imYCbCr.Pixel(x, y, 2);
        }
    }
}


// Converting YCbCr to RGB
template <class T1, class T2>
void convertYCbCrtoRGB(CImageOf<T1> &imYCbCr, CImageOf<T2> &imRGB) 
{
    const double bias = 0.5 - (T2)0.5;
    CShape sh = imYCbCr.Shape();
    if(sh.nBands < 3)
        throw CError("ConvertYCbCrtoRGB: source does not have "
            "at least 3 bands\n");
    imRGB.ReAllocate(sh);

    for(int y=0; y<sh.height; y++) {
        T1 *src = &imYCbCr.Pixel(0, y, 0);
        T2 *dst = &imRGB.Pixel(0, y, 0);
        for(int x=0; x<sh.width*sh.nBands; x+=sh.nBands) {
            double red = 
                YCbCr2RGB_coeff[0][0] * src[x] +
                YCbCr2RGB_coeff[0][1] * (src[x+1] - 127.5) + 
                YCbCr2RGB_coeff[0][2] * (src[x+2] - 127.5);
            double green = 
                YCbCr2RGB_coeff[1][0] * src[x] +
                YCbCr2RGB_coeff[1][1] * (src[x+1] - 127.5) + 
                YCbCr2RGB_coeff[1][2] * (src[x+2] - 127.5);
            double blue = 
                YCbCr2RGB_coeff[2][0] * src[x] +
                YCbCr2RGB_coeff[2][1] * (src[x+1] - 127.5) + 
                YCbCr2RGB_coeff[2][2] * (src[x+2] - 127.5);
            dst[x+R] = (T2)(max(0.0, min(255.0, red)) + bias);
            dst[x+G] = (T2)(max(0.0, min(255.0, green)) + bias);
            dst[x+B] = (T2)(max(0.0, min(255.0, blue)) + bias);

            if(sh.nBands > 3) {
                for(int i=3; i<sh.nBands; i++)
                    dst[x+i] = (T2)(src[x+i] + bias);  // copy the rest
            }
        }
    }
}


// Converting YCbCr to RGB, given three separate grey scale images 
// imY, imCb, and imCr
template <class T1, class T2>
void convertYCbCrtoRGB(CImageOf<T1> &imY, CImageOf<T1> &imCb, 
                       CImageOf<T1> &imCr, CImageOf<T2> &imRGB) 
{
    if(imY.Shape() != imCb.Shape() || imY.Shape() != imCr.Shape())
        throw CError("ConvertYCbCrtoRGB: shapes of source images (Y,Cb,Cr) "
            "mismatch\n");
    if(imY.Shape().nBands != 1)
        throw CError("Input images (Y, Cb, Cr) are not single-channel\n");

    CShape sh = imY.Shape();
    CImageOf<T1> imYCbCr(CShape(sh.width, sh.height, 3));

    for(int y=0; y<sh.height; y++) {
        for(int x=0; x<sh.width; x++) {
            imYCbCr.Pixel(x, y, 0) = imY.Pixel(x, y, 0);
            imYCbCr.Pixel(x, y, 1) = imCb.Pixel(x, y, 0);
            imYCbCr.Pixel(x, y, 2) = imCr.Pixel(x, y, 0);
        }
    }

    convertYCbCrtoRGB(imYCbCr, imRGB);
}


// The gaussian function (unnormalized)
double gaussian(double x, double mu, double sigma) 
{
    return exp(- ((x - mu) / sigma) * ((x - mu) / sigma) * 0.5);
}


// Generate a gaussian sample of 0 mean, unit var. using Box-Muller transform
// See Wikipedia: Box-Muller transform (google)
double boxMullerGaussian() {
    const double pi = 3.14159265358979;
    double r = 1 - (double)rand() / ((double)RAND_MAX + 1);
    double f = 1 - (double)rand() / ((double)RAND_MAX + 1);
    return cos(2 * pi * f) * sqrt(- 2 * log(r));
}


// Adding gaussian noise to an image
template <class T>
void addGaussianNoise(CImageOf<T> &src, CImageOf<T> &dst, double sigma, 
                      double min_val, double max_val) 
{
    assert(min_val < max_val);
    dst.ReAllocate(src.Shape());
    int height = src.Shape().height;
    int width = src.Shape().width;
    int nb = src.Shape().nBands;
    const double bias = 0.5 - (T)0.5;
    for(int y=0; y<height; y++) {
        T *rowSrc = &src.Pixel(0, y, 0);
        T *rowDst = &dst.Pixel(0, y, 0);
        for(int x=0; x<width; x++) {
            for(int b=0; b<nb; b++) {
                T valSrc = rowSrc[x*nb+b];
                rowDst[x*nb+b] = (T)max(min_val, min(max_val, 
                    (valSrc + sigma * boxMullerGaussian() + bias)));
            }
        }
    }
}


// Convolve an image with gaussian kernel
// If mask is given (not NULL), masked pixels are counted out of the image
template <class T1, class T2>
void gaussianSmooth(CImageOf<T1> &src, CImageOf<T2> &dst, double sigma,
                     CByteImage *imMask, unsigned char maskVal) 
{
    const double bias = 0.5 - (T2)0.5;
    int w = src.Shape().width, h = src.Shape().height, nb = src.Shape().nBands;
    dst.ReAllocate(src.Shape());
    dst.FillPixels(0);
    CFloatImage temp(CShape(h, w, nb));

    if(sigma == 0) { // no smoothing
        for(int y=0; y<h; y++) {
            for(int x=0; x<w; x++) {
                for(int b=0; b<nb; b++) {
                    dst.Pixel(x, y, b) = (T2)(src.Pixel(x, y, b) + bias);
                }
            }
        }
        return;
    }

    // Fill the gaussin mask
    int kernel_rad = (int)ceil(4 * sigma);
    int kernel_size = 2 * kernel_rad + 1;
    float * const kernel_base = new float[kernel_size];
    float * const kernel = kernel_base + kernel_rad;
    for(int i=0; i<=kernel_rad; i++) {
        if(sigma != 0)
            kernel[i] = kernel[-i] = gaussian(i, 0, sigma);
        else if(i == 0)
            kernel[i] = 1;
        else
            kernel[i] = kernel[-i] = 0;
    }
    
    // Convolution 1st pass
    for(int y=0; y<h; y++) {
        T1 *row_src = &src.Pixel(0, y, 0);
        for(int x=0; x<w; x++) {
            for(int b=0; b<nb; b++) {
                if(imMask && (imMask->Pixel(x, y, b) & maskVal))
                    continue;
                double sum = 0, kernel_sum = 0;
                for(int i=max(-kernel_rad, -(w-1-x)); i<=min(kernel_rad, x); 
                    i++) 
                {
                    if(imMask && (imMask->Pixel(x - i, y, b) & maskVal))
                        continue;
                    kernel_sum += kernel[i];
                    sum += kernel[i] * row_src[(x - i) * nb + b];
                }
                temp.Pixel(y, x, b) = (float)(sum / kernel_sum);
            }
        }
    }
    // 2nd pass
    for(int x=0; x<w; x++) {
        float *row_temp = &temp.Pixel(0, x, 0);
        for(int y=0; y<h; y++) {
            for(int b=0; b<nb; b++) {
                if(imMask && (imMask->Pixel(x, y, b) & maskVal))
                    continue;
                double sum = 0, kernel_sum = 0;
                for(int i=max(-kernel_rad, -(h-1-y)); i<=min(kernel_rad, y); 
                    i++) 
                {
                    if(imMask && (imMask->Pixel(x, y - i, b) & maskVal))
                        continue;
                    kernel_sum += kernel[i];
                    sum += kernel[i] * row_temp[(y - i) * nb + b];
                }
                dst.Pixel(x, y, b) = (T2)(sum / kernel_sum + bias);
            }
        }
    }
    delete[] kernel_base;
}


// Calculate the signal to noise ratio
template <class T>
double signal2noise(CImageOf<T> &imOrig, CImageOf<T> &imNoisy, 
                    double *rms_error) 
{
    if(imOrig.Shape() != imNoisy.Shape()) {
        throw CError("Image dimension mismatch\n");
    }

    int w = imOrig.Shape().width, h = imOrig.Shape().height;
    int nb = imOrig.Shape().nBands;
    double sq_err_sum = 0;
    for(int y=0; y<h; y++) {
        for(int x=0; x<w; x++) {
            for(int b=0; b<nb; b++) {
                double diff = (double)imNoisy.Pixel(x, y, b) -
                    (double)imOrig.Pixel(x, y, b);
                sq_err_sum += diff * diff;
            }
        }
    }

    double rms_err = sqrt((double)sq_err_sum / (w * h * nb));
    double sig_noise = 20 * log(255.0 / rms_err) / log(10.0);
    if(rms_error)
        *rms_error = rms_err;
    return sig_noise;
}


// Check whether the two given image has the same size (w*h)
template <class T1, class T2>
bool sizeMatch(CImageOf<T1> &im1, CImageOf<T2> &im2)
{
    return im1.Shape().width == im2.Shape().width && 
        im1.Shape().height == im2.Shape().height;
}


// Conversion between ArrImage and CImage (assume ArrImage is already allocated)
template <class TA, class TC>
void convertArrImageToCImage(ArrImage<TA> &src, CImageOf<TC> &dst) 
{
    int w = src.width, h = src.height, nb = src.nBands;
    CImageOf<TA> temp(CShape(w, h, nb));
    for(int y=0; y<h; y++) {
        for(int x=0; x<w; x++) {
            for(int b=0; b<nb; b++) {
                temp.Pixel(x, y, b) = src.Pixel(x, y, b);
            }
        }
    }
    convertImageType(temp, dst);
}

template <class TC, class TA>
void convertCImageToArrImage(CImageOf<TC> &src, ArrImage<TA> &dst) 
{
    CShape sh = src.Shape();
    int w = sh.width, h = sh.height, nb = sh.nBands;
    CImageOf<TA> temp;
    convertImageType(src, temp);
    for(int y=0; y<h; y++) {
        for(int x=0; x<w; x++) {
            for(int b=0; b<nb; b++) {
                dst.Pixel(x, y, b) = temp.Pixel(x, y, b);
            }
        }
    }
}





// =========================================


// Support for double image
template <> double CImageOf<double>::MinVal(void)     { return -1e308; }
template <> double CImageOf<double>::MaxVal(void)     { return 1e308; }



// Functions calling the template functions to make them compiled
// -- Should never be called
template <class T1, class T2>
static void instantiateDual(CImageOf<T1> im1, CImageOf<T2> im2) {
    sizeMatch(im1, im2);
    convertImageType(im1, im2);
    gaussianSmooth(im1, im2, 0.0);
    ArrImage<T1> ima1;
    ArrImage<T2> ima2;
    convertArrImageToCImage(ima1, im2);
    convertCImageToArrImage(im1, ima2);
    convertRGBtoYCbCr(im1, im2);
    convertRGBtoYCbCr(im1, im2, im2, im2);
    convertYCbCrtoRGB(im1, im2);
    convertYCbCrtoRGB(im1, im1, im1, im2);
}

template <class T>
static void instantiateTemplates(CImageOf<T> im) {
    CByteImage imb;
    CIntImage imi;
    CFloatImage imf;
    CImageOf<double> imd;
	convertRGBtoGrey(im, im);
    convertToRGB(im);
    addGaussianNoise(im, im, 0, 0, 0);
    signal2noise(im, im);

    instantiateDual(im, imb);
    instantiateDual(im, imi);
    instantiateDual(im, imf);
    instantiateDual(im, imd);
}

void __utils_cpp__instantiateAllTemplates() {
	fprintf(stderr, "ERROR: THIS FUNCTION IS NOT TO BE CALLED ANYWHERE !!");
	instantiateTemplates(CByteImage());
	instantiateTemplates(CIntImage());
	instantiateTemplates(CFloatImage());
    instantiateTemplates(CImageOf<double>());
    assert(0);
}
