/*******************************************************************************
 * SLRF.h
 *
 * An implementation of the Sparse Long-range Random Field (SLRF) prior.
 * The model is described in:
 *
 *   "Sparse Long-range Random Field and its Application to Image Denoising"
 *   by Yunpeng Li and Daniel P. Huttenlocher, in ECCV 2008
 *
 * The function provided by the interface compute the value and the gradient
 * (i.e. derivative) of the energy function corresponding to SLRF, which can
 * then be used with one's favorite gradient descent based optimizer.
 *
 * Yunpeng Li
 */

#ifndef __SLRF_H__
#define __SLRF_H__

// The number of levels (of spatial extents) of SLRF
#define SLRF_LEVELS		3

/* Compute the SLRF energy and its gradient (derivative)
   - Return the energy value and store gradient in 'grad' (assume already 
     allocated)
   - Assume image in row-major order: row-column-channel

   Input: 
     img -      the image data (array of pointers to the rows)
	 w -        image width
	 h -        image height
	 nb -       number of channels (bands)
	 range -    the range of pixel values for the image (e.g. 255 for 8-bit RGB)
	 weight -   weighting factor of the SLRF prior, application dependent
   Output: 
	 grad -     *accumulated* (i.e. '+=') the gradient of SLRF's energy function 
	            (same dimension as 'img')
	 Return value: the value of the energy function
 */
template <class T>
double SLRFenergy(const T **imgrows, int w, int h, int nb, double weight, 
				  double range, T **grad);

/* Call this version if the image data is given as a contiguous array
 */
template <class T>
double SLRFenergy(const T *imgarr, int w, int h, int nb, double weight, 
				  double range, T *grad);

#endif
