/*******************************************************************************
 * Main.cpp
 *
 * Main entry for SLRF denoising demo, which implements the model described in:
 * - Yunpeng Li & Daniel P. Huttenlocher, Sparse Long-range Random Field and 
 *   its Application to Image Denoising, ECCV 2008.
 *
 * Yunpeng Li
 *
 * Know issues with the Szeliski/Scharstein image library + PNGLib (lpng):
 * - Compatibility issue with color management, causing image saved by 
 *   applications that use this feature (e.g. Photoshop) be read incorrectly.  
 *   If this occurs, convert the image to PPM (color) or PGM (gray) to get
 *   rid of color management.  (The Unix 'convert' will do.)
 */

#include <stdio.h>
#include <stdlib.h>
#include <time.h>

#include "Denoise.h"


static const char *info_str = 
    "\n"
    "SLRF denoise demo, written by Yunpeng Li (yuli@cs.cornell.edu),\n"
    "implementing the model described in: \n"
    "\n"
    "  Sparse Long-range Random Field and its Application to Image Denoising\n"
    "  - Yunpeng Li and Daniel P. Huttenlocher, ECCV 2008\n"
    "\n"
    "This software is intended for research purposes only and comes with\n"
    "ABSOLUTELY NO WARRANTY.  The author is not responsible for any \n"
    "consequences from the use of this software.\n"
    "\n";

static const char *usage_str = 
    "Usage: denoise [options] input output noiseSigma\n"
    "\n"
    "  input/output must be PNG/PPM/PGM images.\n"
    "  noiseSigma is the noise level (s.d. for Gaussian).\n"
    "\n"
    "Options:\n"
    "  -i int    -> the maximum number of gradient descent iterations\n"
    "               (default 20)\n"
    "  -v [0,1]  -> verbosity level (default 1)\n"
    "  -g string -> ground truth image file (if available)\n"
    "\n";


/* Command-line inputs
 */
struct ConsoleInputs 
{
    char *inputImage, *outputImage, *groundTruthImage;
    int maxIters;  // max number of gradient descent iterations
    int verbose;  // verbosity (feedback)
    double noiseSigma;  // noise level
};


/* Print program info
 */
static void print_info() 
{
    printf(info_str);
}


/* Print usage
 */
static void print_usage() 
{
    printf(usage_str);
}


/* Default
 */
static void set_default(ConsoleInputs *ci) 
{
    ci->maxIters = 20;
    ci->verbose = 1;
    ci->groundTruthImage = NULL;
    // Following must be set by user:
    ci->noiseSigma = -1;
    ci->inputImage = NULL;
    ci->outputImage = NULL;
}


/* Parse command-line inputs
 */
static void parse_input(int argc, char **argv, ConsoleInputs *ci) 
{
    set_default(ci);
    int i;
    for(i=1; i<argc && (argv[i])[0] == '-'; i++) {
        if(strcmp(argv[i], "-i") == 0) {
            ci->maxIters = atoi(argv[++i]);
        }
        else if(strcmp(argv[i], "-v") == 0) {
            ci->verbose = atoi(argv[++i]);
        }
        else if(strcmp(argv[i], "-g") == 0) {
            ci->groundTruthImage = argv[++i];
        }
    }
    if(i > argc - 3) {
        fprintf(stderr, "\nNot enough input argument\n\n");
        print_usage();
        exit(1);
    }
    ci->inputImage = argv[i];
    ci->outputImage = argv[i+1];
    ci->noiseSigma = atof(argv[i+2]);
}


/* Main function
 */
int main(int argc, char **argv) 
{
    if(argc < 2) {
        print_info();
        print_usage();
        exit(1);
    }

    // Parse command-line arguments
    ConsoleInputs ci;
    parse_input(argc, argv, &ci);
    
    // Reading inputs
    CByteImage im, imOut, imOrig, *imOrigPtr = NULL;
    ReadImageVerb(im, ci.inputImage, ci.verbose);
    if(im.Shape().nBands > 3) {
		CByteImage im2;
        convertToRGB(im, im2);
		im = im2;
    }
    if(ci.groundTruthImage) {
        ReadImageVerb(imOrig, ci.groundTruthImage, ci.verbose);
        if(imOrig.Shape().nBands > 3) {
			CByteImage im2;
            convertToRGB(imOrig, im2);
			imOrig = im2;
        }
        if(im.Shape() != imOrig.Shape()) {
            fprintf(stderr, "Input/ground truth image shape mismatch\n");
            exit(1);
        }
        imOrigPtr = &imOrig;
    }

    clock_t clk0 = clock();

    // Process the noisy image
    slrf_denoise(im, ci.noiseSigma, imOut, ci.maxIters, ci.verbose, imOrigPtr);

    if(ci.verbose) {
        printf("CPU time for denoising: %.2f seconds\n", 
            ((double)clock()-(double)clk0)/(double)CLOCKS_PER_SEC);
    }

    // Output
    WriteImageVerb(imOut, ci.outputImage, ci.verbose);

    return 0;
}
