/************************************************************************
 * Game.java
 *
 * Othello game flow control.
 * Also handles the iterative deepening and time management of the computer
 * player.
 *
 * Note: This one is event-triggered rather than loop-based
 *
 * TO DO: Use some multi-threading technique to be able to intervene
 * while computer is thinking. Current once the computer starts thinking
 * you have to wait.
 *
 *
 * Yunpeng Li
 *
 * */

package othelloGUI;

import java.io.*;
import java.awt.*;
import java.text.*;
import java.util.Date;

public class Game {
    /* Constants */
    public static final int HUMAN_PLAYER = 0, COMPUTER_PLAYER = 1;
    // Settings file name
    public static final String SETTINGS_FILE_NAME = "MiniOthello.ini";

    // Defaults
    public static final int DEF_PLAYER0 = HUMAN_PLAYER;
    public static final int DEF_PLAYER1 = COMPUTER_PLAYER;
    public static final int DEF_SEARCH_DEPTH = 8;
    public static final int DEF_END_GAME_DEPTH = 16;

    public static final boolean DEF_SHOW_STAT = true;
    public static final boolean DEF_SHOW_LAST_MOVE = true;
    public static final boolean DEF_SHOW_AVAIL_MOVES = true;
    public static final boolean DEF_MIRRORED_START = false;
    public static final boolean DEF_SAVE_SETTINGS = true;
    public static final boolean DEF_SCALE_EVAL = true;
    public static final boolean DEF_KEEP_MESSAGES = true;
    public static final boolean DEF_SHOW_SCORE_IN_EVAL = false;
    public static final boolean DEF_SAVE_LAST_GAME = true;
    public static final boolean DEF_LEFT_CLICK_TO_PASS = false;
    public static final boolean DEF_RIGHT_CLICK_TO_PASS = true;
    public static final boolean DEF_RIGHT_CLICK_FOR_FORCED_MOVE = false;

    public static final int START_DEPTH = 4;
    public static final int START_E_DEPTH = 12;

    // these two may get larger as computers evolve
    public static final int MAX_SEARCH_DEPTH = 14;
    public static final int MAX_END_GAME_DEPTH = 32;

    // They can be 10/2.0 with MPC, but need 20/2.5 without
    public static final double ESTIMATED_MAX_BRANCHING_FACTOR = 10; // empirical (actually this is from 2 level deeper)
    public static final double SMART_PLAY_SPENDING_FACTOR = 2.0;  // balance between depth and speed
    public static final double SMART_PLAY_SAVING_FACTOR = SMART_PLAY_SPENDING_FACTOR / ESTIMATED_MAX_BRANCHING_FACTOR;

    // Randomness levels
    public static final int RL_NONE = 0;
    public static final int RL_KNOWN_OPENINGS = 1;
    public static final int RL_SMALL = 2;
    public static final int RL_MEDIUM = 4;
    public static final int RL_LARGE = 8;
    public static final int[] R_LEVELS = {
        RL_NONE, RL_KNOWN_OPENINGS, RL_SMALL, RL_MEDIUM, RL_LARGE
    };
    public static final int DEF_RANDOMNESS_LEVEL = RL_KNOWN_OPENINGS;


    /************
     * Engine-specific fields! Update if the corresponding value in engine is
     * changed!!
     */
    public static final int MIN_EMPTIES_FOR_EVALUATION = 12; // as defined in minimax.h

    //***********


    /* Variables */
    private Board board;
    private OthelloFrame frame;

    public int[] player = new int[2];  // 0 - player0, 1  player1, hold value of HUMAN_PLAYER or COMPUTER_PLAYER
    public int searchDepth;
    public int endGameDepth;

    // Options
    public boolean showAvailMoves;
    public boolean showLastMove;
    public boolean showStat;
    public boolean mirroredStart;
    public boolean saveSettings;
    public boolean scaleEval;
    public boolean keepMessages;
    public boolean showScoreInEval;
    public boolean saveLastGame;
    public boolean leftClickToPass, rightClickToPass; // not mutually exclusive
    public boolean rightClickForForcedMove;

    private int currPlayer;  // 0 or 1
    private int currColor;  //Board.BLACK or WHITE
    private boolean waitingToStart;
    private boolean gameEnded;

    // The AI engine process
    private Process engine;
    public String engineVer;

    // Time management
    public boolean iterativeDeepening;
    public boolean timeDependent;  // whether using time management
    public int softTimeLimit;  // milliseconds -- used in timeDependent mode (0 disable)
    public int hardTimeLimit;  // milliseconds -- max time in all cases (0 disable)

    // randomness
    public int randomnessLevel;

    // stop think if it estimates that the next level can't be complete, even if
    // there is time left. But give some extra time if likely to complete next level soon.
    // This can largely reduced wasted searching time in time-constraint mode.
    public boolean smartPlay;

    // Game state variables
    private boolean canPlay;  // whether human player can make a move at this time
    private boolean isSelfPlaying;
    private boolean comNoMove;  // decide whether the computer should make no move when interrupted

    // Messages/stats for each move in the message box
    private String[] messages;
    private boolean[] messageAvailable;

    public Game() {
        canPlay = true;
        isSelfPlaying = false;
        comNoMove = false;

        // load default settings -- will be override when loading from .ini file
        player[0] = DEF_PLAYER0;
        player[1] = DEF_PLAYER1;

        searchDepth = DEF_SEARCH_DEPTH;
        endGameDepth = DEF_END_GAME_DEPTH;

        showStat = DEF_SHOW_STAT;
        showLastMove = DEF_SHOW_LAST_MOVE;
        showAvailMoves = DEF_SHOW_AVAIL_MOVES;
        mirroredStart = DEF_MIRRORED_START;
        saveSettings = DEF_SAVE_SETTINGS;
        scaleEval = DEF_SCALE_EVAL;
        keepMessages = DEF_KEEP_MESSAGES;
        showScoreInEval = DEF_SHOW_SCORE_IN_EVAL;
        saveLastGame = DEF_SAVE_LAST_GAME;
        leftClickToPass = DEF_LEFT_CLICK_TO_PASS;
        rightClickToPass = DEF_RIGHT_CLICK_TO_PASS;
        rightClickForForcedMove = DEF_RIGHT_CLICK_FOR_FORCED_MOVE;

        currPlayer = 0;
        currColor = Board.BLACK;
        waitingToStart = false;
        gameEnded = false;

        iterativeDeepening = true; // used by default - seems to be a good idea
        timeDependent = false;
        softTimeLimit = 2000;  // both milliseconds
        hardTimeLimit = 0;  // 0 = disabled
        smartPlay = true;

        randomnessLevel = DEF_RANDOMNESS_LEVEL;

        messages = new String[120];
        messages[0] = "Start of game";
        messageAvailable = new boolean[120];
        messageAvailable[0] = true;
        for(int i=1; i<120; i++)
            messageAvailable[i] = false;

        if(timeDependent) {
            searchDepth = MAX_SEARCH_DEPTH;
            endGameDepth = MAX_END_GAME_DEPTH;
        }

        engineVer = "Unknown";

        // Dumb execution of the engine once -- so it will sit in memory
        try {
            engine = Runtime.getRuntime().exec(MiniOthello.ENGINE_PATH +
                                               MiniOthello.ENGINE_NAME + " -vn");
            InputStream is = engine.getInputStream();
            InputStreamReader isr = new InputStreamReader(is);
            BufferedReader br = new BufferedReader(isr);
            String istr = br.readLine();
            engineVer = istr.split("\n")[0];
            engine.waitFor();
        }
        catch(IOException e) {
            sendEngineError("Cannot execute " + MiniOthello.ENGINE_PATH +
                                               MiniOthello.ENGINE_NAME);
        }
        catch(InterruptedException e) {
            sendEngineError("Interrupted");
            if(engine != null)
                engine.destroy();
        }

        // Instantiate the board and the frame
        loadSettings();
        board = new Board(mirroredStart);
        frame = new OthelloFrame(Util.WINDOW_TITLE, board, this);
        frame.repaint();
    }


    /* Methods as interface between Game and OthelloFrame */
    // test whether game is in a state able to play a human move
    public boolean canPlay() {
        return canPlay;
    }

    // Called when human makes a move. Return whether it can be played
    // It will wait for the AI engine thread to finish
    public synchronized boolean playMove(int x, int y) {
        if(!canPlay)
            return false;
        canPlay = false;
        if(!board.legalMove(x, y)) {
            canPlay = true;
            return false;
        }
        board.makeMove(x, y);
        storeMessage("Human's move: " + Board.move2str(x, y));
        currPlayer = 1 - currPlayer;
        currColor = board.whoseTurn();
        if(board.hasLegalMove()) {
            updateGraphics();
            if (player[currPlayer] == COMPUTER_PLAYER)
                playComputerMove();
            if(!board.hasLegalMove()) {
                // look ahead on step to see if the game should end
                boolean shouldEnd;
                board.switchSide();
                shouldEnd = !board.hasLegalMove();
                board.switchSide();
                if(shouldEnd) {
                    endGame();
                }
                else { // then human should pass
                    if(player[currPlayer] == HUMAN_PLAYER)
                        updateGraphics(); // cause the button to be shown
                }
            }
        }
        else { // next player has to pass
            board.makePass();
            currPlayer = 1 - currPlayer;
            currColor = board.whoseTurn();
            if(!board.hasLegalMove()) { // no one can move, game ends
                board.undoMoveIrreversible();
                currPlayer = 1 - currPlayer; // don't want to show the very last pass
                currColor = board.whoseTurn();
                endGame();
            }
            else {
                if (player[1-currPlayer] == COMPUTER_PLAYER) { // now the other player has made the pass, so it is 1-currPlayer
                    frame.getInfoBox().setText("Computer passed.");
                    storeMessage("Computer passed.");
                }
            }
            updateGraphics();
        }
        canPlay = true;
        return true;
    }

    // make a pass
    public synchronized boolean makePass() {
        if(!canPlay)
            return false;
        canPlay = false;
        if(board.hasLegalMove()) { // has legal move, not allowed to pass -- for bug detection
            System.out.println("Warning: Trying to pass while having legal moves !");
            canPlay = true;
            return false;
        }
        board.makePass();
        storeMessage("Human passed.");
        currPlayer = 1 - currPlayer;
        currColor = board.whoseTurn();
        frame.updateButtons(board.findLegalMoves(new boolean[64]));
        //updateGraphics();
        if(!board.hasLegalMove()) // no one can move, game ends
            endGame();
        else if(player[currPlayer] == COMPUTER_PLAYER) {
            playComputerMove();
            if(!board.hasLegalMove()) {
                // look ahead on step to see if the game should end
                boolean shouldEnd;
                board.switchSide();
                shouldEnd = !board.hasLegalMove();
                board.switchSide();
                if(shouldEnd) {
                    endGame();
                }
                else { // then human should pass
                    if(player[currPlayer] == HUMAN_PLAYER)
                        frame.updateButtons(0); // cause the button to be shown
                }
            }
        }
        canPlay = true;
        return true;
    }

    // Called when human and computer swapside
    public void rollSwapped() {
        waitingToStart = true;
    }

    // Called from the "start" button
    public synchronized void makeComMove() {
        if(!canPlay)
            return;
        canPlay = false;
        if(board.hasLegalMove()) {
            playComputerMove();
            // now see if game should end
            if(!board.hasLegalMove()) {
                board.switchSide();
                boolean shouldEnd = !board.hasLegalMove();
                board.switchSide();
                if(shouldEnd)
                    endGame();
            }
        }
        else {
            board.makePass();
            currPlayer = 1 - currPlayer;
            currColor = board.whoseTurn();
            if(!board.hasLegalMove()) { // game has ended
                board.undoMoveIrreversible(); // not want to show last pass
                currPlayer = 1 - currPlayer;
                currColor = board.whoseTurn();
                endGame();
            }
            else {
                frame.getInfoBox().setText("Computer passed.");
                storeMessage("Compter passed.");
            }
            updateGraphics();
        }
        canPlay = true;
    }

    // Called when user starts a self-played game
    public synchronized void selfPlay() {
        if(!canPlay)
            return;
        canPlay = false;
        isSelfPlaying = true;
        long startTime = (new Date()).getTime();
        while(isSelfPlaying && player[currPlayer] == COMPUTER_PLAYER) {
            if(board.hasLegalMove()) {
                playComputerMove();
                // now see if game should end
                if (!board.hasLegalMove()) {
                    board.switchSide();
                    boolean shouldEnd = !board.hasLegalMove();
                    board.switchSide();
                    if (shouldEnd) {
                        isSelfPlaying = false;
                        endGame();
                    }
                }
            }
            else {
                board.makePass();
                currPlayer = 1 - currPlayer;
                currColor = board.whoseTurn();
                if(!board.hasLegalMove()) { // no one can move, game over
                    board.undoMoveIrreversible();  // don't want the last pass to be displayed
                    currPlayer = 1 - currPlayer;
                    currColor = board.whoseTurn();
                    isSelfPlaying = false;
                    endGame();
                }
                else {
                    frame.getInfoBox().setText("Computer passed.");
                    storeMessage("Computer passed.");
                }
            }
        }
        long endTime = (new Date()).getTime();
        double timeUsed = (double)((endTime - startTime) / 10) / 100;
        frame.getInfoBox().append("\nTotal time: " + timeUsed + " sec. ");
        isSelfPlaying = false;
        canPlay = true;
        //System.out.println("player[0]: " + player[0] + ", player[1]: " + player[1]); // debug
        waitingToStart = true;
        updateGraphics();
    }

    // Call from OthelloFrame to stop self-playing
    public void stopSelfPlay() {
        isSelfPlaying = false;
        frame.updateButtons(1);
    }

    public void stopSelfPlayForced() {
        isSelfPlaying = false;
        frame.updateButtons(1);
        try {engine.destroy();} catch(NullPointerException e) {}
    }

    // See if a game is being self-played by the computer    public boolean isSelfPlaying() {
    public boolean isSelfPlaying() {
        return isSelfPlaying;
    }

    /* Undo/redo */
    public void undo() { // unsynchronized undo - for stop computer when it's thinking
        if(isSelfPlaying)
            return;  // make no sense to undo during a computer self-play
        stopComputerThinking();
        comNoMove = true;
        undo_sync();
        comNoMove = false;
    }

    public synchronized void undo_sync() {
        if(!canPlay)
            return;
        int count = 0;
        if(board.undoMove()) {
            currPlayer = 1 - currPlayer;
            currColor = board.whoseTurn();
            count++;
            if(player[currPlayer] == COMPUTER_PLAYER &&
               player[1-currPlayer] == HUMAN_PLAYER) {
                if(board.undoMove()) {
                    currPlayer = 1 - currPlayer;
                    currColor = board.whoseTurn();
                    count++;
                }
            }
            gameEnded = false; // because of the undo
        }
        if(keepMessages && hasAvailMessage())
            displayStoredMessage();
        else
            frame.getInfoBox().setText("Undo " + count +
                                       (count <= 1? " move. ":" moves."));
        if(player[currPlayer] == COMPUTER_PLAYER)
            waitingToStart = true;
        updateGraphics();
    }

    public synchronized void undoAll() {
        if(!canPlay)
            return;
        int count = board.undoAll();
        if(keepMessages && hasAvailMessage())
            displayStoredMessage();
        else
            frame.getInfoBox().setText("Undo " + count +
                                       (count <= 1? " move. ":" moves."));
        updateGraphics();
        currPlayer = 0;
        currColor = board.whoseTurn();
        gameEnded = false; // because of the undo
        if(player[currPlayer] == COMPUTER_PLAYER)
            waitingToStart = true;
        updateGraphics();
    }

    public synchronized void redo() {
        if(!canPlay)
            return;
        int count = 0;
        if(board.redoMove()) {
            currPlayer = 1 - currPlayer;
            currColor = board.whoseTurn();
            count++;
            if(player[currPlayer] == COMPUTER_PLAYER &&
               player[1-currPlayer] == HUMAN_PLAYER) {
                if(board.redoMove()) {
                    currPlayer = 1 - currPlayer;
                    currColor = board.whoseTurn();
                    count++;
                }
            }
            if(keepMessages && hasAvailMessage())
                displayStoredMessage();
            else
                frame.getInfoBox().setText("Redo " + count +
                                           (count <= 1? " move. ":" moves."));
        }
        if(!board.hasLegalMove()) {
            boolean shouldEnd;
            board.switchSide();
            shouldEnd = !board.hasLegalMove();
            board.switchSide();
            if(shouldEnd)
                endGame();
        }
        if(player[currPlayer] == COMPUTER_PLAYER)
            waitingToStart = true;
        updateGraphics();
    }

    public synchronized void redoAll() {
        if(!canPlay)
            return;
        int count = board.redoAll();
        if(keepMessages && hasAvailMessage())
            displayStoredMessage();
        else
            frame.getInfoBox().setText("Redo " + count +
                                       (count <= 1? " move. ":" moves."));
        currPlayer = (currPlayer + count) & 1;
        currColor = board.whoseTurn();
        if(!board.hasLegalMove()) {
            boolean shouldEnd;
            board.switchSide();
            shouldEnd = !board.hasLegalMove();
            board.switchSide();
            if(shouldEnd)
                endGame();
        }
        if(player[currPlayer] == COMPUTER_PLAYER)
            waitingToStart = true;
        updateGraphics();
    }

    /* update the graphics on the Frame */
    public void updateGraphics() {
        frame.paint(frame.getGraphics());
        // a bug detection in this frequently called method
        if(currColor != currPlayer + 1) {
            System.out.println("Error Warning: CurrColor and CurrPlayer mismatch in Game.java");
        }
    }

    /* Access methods */
    public Board getBoard() {
        return board;
    }

    public int getCurrPlayer() {
        return currPlayer;
    }

    public int getCurrColor() {
        return currColor;
    }

    public boolean isWaitingToStart() {
        return waitingToStart;
    }

    public boolean hasEnded() {
        return gameEnded;
    }


    /*** Careful with the following two methods ***/
    // Stop the computer thinking by killing the engine process
    public void stopComputerThinking() {
        try {
            engine.destroy();
        } catch(NullPointerException e) {}
    }

    // Should only be called while exiting the program
    public void terminate() {
        if(engine != null)
            engine.destroy();
    }

    /* New game,  load game, save game */
    public void newGame() {
        boolean forcedNewGame = false;
        if(!canPlay) {
            stopSelfPlayForced();
            forcedNewGame = true;
            while(!canPlay)
                try {Thread.sleep(10);} catch(InterruptedException e) {}
        }
        board = new Board(mirroredStart);
        currPlayer = 0;
        currColor = Board.BLACK;
        waitingToStart = false;
        gameEnded = false;
        if(player[0] == COMPUTER_PLAYER)
            waitingToStart = true;
        frame.initNewGame();
        for(int i=1; i<120; i++)
            messageAvailable[i] = false;
        if(forcedNewGame)
            updateGraphics();
    }

    // load game
    public void loadGame() {
        if(!canPlay)
            return;  // don't want to do this when game is not in a static situation
        boolean invalid = false;
        boolean error = false;
        FileDialog d = new FileDialog(frame, "Load Game", FileDialog.LOAD);
        d.show();
        String filename = d.getFile();
        if(filename == null) { // most likely user canceled
            return;
        }
        frame.getInfoBox().setText("Loading...");
        try {
            File f = new File(d.getDirectory() + filename);
            FileReader fr = new FileReader(f);
            char[] buf = new char[1000];
            fr.read(buf);
            String[] content = (new String(buf)).split(" ");
            boolean mirrored;
            int m, top;
            try {
                if(content.length < 3)
                    invalid = true;
                else {
                    mirrored = Integer.parseInt(content[0]) == 0 ? false : true;
                    m = Integer.parseInt(content[1]);
                    top = Integer.parseInt(content[2]);
                    if (! (0 <= m && m <= top && top <= 120)) {
                        // definitely invalid file
                        invalid = true;
                    }
                    if(top + 3 > content.length)
                        invalid = true;
                    if (!invalid) {
                        Board b = new Board(mirrored);
                        for (int i = 1; i <= top; i++) {
                            if (content[i+2].equals("pass")) {
                                if (b.hasLegalMove()) {
                                    invalid = true;
                                    break;
                                }
                                b.makePass();
                            }
                            else {
                                char[] c = content[i+2].toCharArray();
                                if (c.length != 2) {
                                    invalid = true;
                                    break;
                                }
                                int x = (int) c[0] - (int) 'a';
                                int y = (int) c[1] - (int) '1';
                                if (!b.legalMove(x, y)) {
                                    invalid = true;
                                    break;
                                }
                                b.makeMove(x, y);
                            }
                        }
                        // Update the board
                        if (!invalid && !error) {
                            // if there are passes at the very end (in old .sav file), remove them
                            while(b.gameOver() && b.getLastMove() == Board.PASS) {
                                b.undoMoveIrreversible();
                                top--;
                                if(m > top)
                                    m = top;
                            }
                            b.undo(top - m);
                            board = b; // set the board of game to the new board
                            gameEnded = board.gameOver();
                            currPlayer = board.whoseTurn() - 1; // since B = 1, W = 2, currPlayer = 0 or 1
                            currColor = board.whoseTurn();
                            if(player[currPlayer] == COMPUTER_PLAYER)
                                waitingToStart = true;
                            // update frame
                            frame.setBoard(board);
                            frame.getInfoBox().setText("Game loaded");
                            updateGraphics();
                            for(int i=1; i<120; i++)
                                messageAvailable[i] = false;
                        }
                    }
                }
            }
            catch(NumberFormatException e) {
                invalid = true;
                //System.out.println("Warning: Invalid save-file format");
            }
        }
        catch(IOException e) {
            error = true;
            //System.out.println("Warning: Error reading from file " + filename + ". Message: " + e.getMessage());
        }
        if(error) {
            frame.getInfoBox().append("\nCannot read from file " + filename + "!");
            return;
        }
        if(invalid) {
            frame.getInfoBox().append("\nInvalid save file!");
            return;
        }
    }

    // save game
    public void saveGame() {
        if(!canPlay)
            return;  // don't want to do this when game is not in a static situation
        boolean success = true;
        FileDialog d = new FileDialog(frame, "Save Game", FileDialog.SAVE);
        //d.setFile("*.gam");
        d.show();
        String filename = d.getFile();
        if(filename == null) { // most likely user canceled
            return;
        }
        try {
            File f = new File(d.getDirectory() + filename);
            FileWriter fw = new FileWriter(f);
            fw.write(board.isMirrored()? "1 " : "0 ");
            fw.write(board.getM() + " " + board.getTop() + " ");
            for(int i=1; i<=board.getTop(); i++) {
                int move = board.getRecordedMove(i);
                if(move == Board.PASS)
                    fw.write("pass ");
                else {
                    char[] c = new char[2];
                    c[0] = (char)((move & 7) + (byte)'a');
                    c[1] = (char)((move >> 3) + (byte)'1');
                    fw.write(c);
                    fw.write(" ");
                }
            }
            fw.flush();
        }
        catch(IOException e) {
            System.out.println("Warning: Error writing to file " + filename +
                               ". Message: " + e.getMessage());
            frame.getInfoBox().setText("Can't write to file " + filename + "!");
            success = false;
        }
        if(success) {
            frame.getInfoBox().append("\nGame saved");
        }
    }


    /* Save and load settings */
    // Save
    public void saveSettings() {
        try {
            File f = new File(SETTINGS_FILE_NAME);
            if (!f.exists())
                f.createNewFile();
            if (f.canWrite()) {
                FileWriter fw = new FileWriter(f);
                fw.write(
                    "searchDepth " + searchDepth + "\r\n" +
                    "endGameDepth " + endGameDepth + "\r\n" +
                    "timeDependent " + timeDependent + "\r\n" +
                    "softTimeLimit " + softTimeLimit + "\r\n" +
                    "hardTimeLimit " + hardTimeLimit + "\r\n" +
                    "smartPlay " + smartPlay + "\r\n" +
                    "randomnessLevel " + randomnessLevel + "\r\n" +
                    "\r\n" +
                    "showAvailMoves " + showAvailMoves + "\r\n" +
                    "showLastMove " + showLastMove + "\r\n" +
                    "showStat " + showStat + "\r\n" +
                    "mirroredStart " + mirroredStart + "\r\n" +
                    "saveSettings " + saveSettings + "\r\n" +
                    "scaleEval " + scaleEval + "\r\n" +
                    "keepMessages " + keepMessages + "\r\n" +
                    "showScoreInEval " + showScoreInEval + "\r\n" +
                    "saveLastGame " + saveLastGame + "\r\n" +
                    "leftClickToPass " + leftClickToPass + "\r\n" +
                    "rightClickToPass " + rightClickToPass + "\r\n" +
                    "rightClickForForcedMove " + rightClickForForcedMove + "\r\n");
                fw.flush();
                frame.getInfoBox().append("\nSettings saved");
            }
            else {
                System.out.println("Warning: Cannot write to " + SETTINGS_FILE_NAME);
            }
        }
        catch(IOException e) {
            System.out.println("Warning: Cannot write to " + SETTINGS_FILE_NAME +
                               ". Message: " + e.getMessage());
        }
    }

    // load settings -- automatic at the start
    public void loadSettings() {
        try {
            File f = new File(SETTINGS_FILE_NAME);
            if(!f.exists() || !f.canRead())
                return;
            long length = f.length();
            char[] buf = new char[(int)length];
            FileReader fr = new FileReader(f);
            fr.read(buf);
            String[] content = (new String(buf)).split("\r\n");
            for(int i=0; i<content.length; i++) {
                try {
                    String[] words = content[i].split(" ");
                    if(words.length < 2) // invalid entry
                        continue;
                    String var = words[0];
                    String str = words[1];
                    if(var.equals("searchDepth")) {
                        int val = Integer.parseInt(str);
                        if(val > 0)
                            searchDepth = val;
                    }
                    else if(var.equals("endGameDepth")) {
                        int val = Integer.parseInt(str);
                        if(val > 0)
                            endGameDepth = val;
                    }
                    else if(var.equals("timeDependent")) {
                        timeDependent = Boolean.valueOf(str).booleanValue();
                    }
                    else if(var.equals("softTimeLimit")) {
                        int val = Integer.parseInt(str);
                        if(val >= 0)
                            softTimeLimit = val;
                    }
                    else if(var.equals("hardTimeLimit")) {
                        int val = Integer.parseInt(str);
                        if(val >= 0)
                            hardTimeLimit = val;
                    }
                    else if(var.equals("smartPlay")) {
                        smartPlay = Boolean.valueOf(str).booleanValue();
                    }
                    else if(var.equals("randomnessLevel")) {
                        int val = Integer.parseInt(str);
                        if(val >= 0 && val <= 9)
                            randomnessLevel = val;
                    }
                    else if(var.equals("showAvailMoves")) {
                        showAvailMoves = Boolean.valueOf(str).booleanValue();
                    }
                    else if(var.equals("showLastMove")) {
                        showLastMove = Boolean.valueOf(str).booleanValue();
                    }
                    else if(var.equals("showStat")) {
                        showStat = Boolean.valueOf(str).booleanValue();
                    }
                    else if(var.equals("mirroredStart")) {
                        mirroredStart = Boolean.valueOf(str).booleanValue();
                    }
                    else if(var.equals("saveSettings")) {
                        saveSettings = Boolean.valueOf(str).booleanValue();
                    }
                    else if(var.equals("scaleEval")) {
                        scaleEval = Boolean.valueOf(str).booleanValue();
                    }
                    else if(var.equals("keepMessages")) {
                        keepMessages = Boolean.valueOf(str).booleanValue();
                    }
                    else if(var.equals("showScoreInEval")) {
                        showScoreInEval = Boolean.valueOf(str).booleanValue();
                    }
                    else if(var.equals("saveLastGame")) {
                        saveLastGame = Boolean.valueOf(str).booleanValue();
                    }
                    else if(var.equals("leftClickToPass")) {
                        leftClickToPass = Boolean.valueOf(str).booleanValue();
                    }
                    else if(var.equals("rightClickToPass")) {
                        rightClickToPass = Boolean.valueOf(str).booleanValue();
                    }
                    else if(var.equals("rightClickForForcedMove")) {
                        rightClickForForcedMove = Boolean.valueOf(str).booleanValue();
                    }
                    //System.out.println("var: " + var + ", str: " + str); //debug
                }
                catch(NumberFormatException e) {
                    // do nothing
                }
            }
        }
        catch(IOException e) {
            // do nothing
        }
    }

    // Recreate the frame (in case it doesn't get loaded properly
    public void recreateFrame() {
        frame.dispose();
        frame = new OthelloFrame(Util.WINDOW_TITLE, board, this);
    }


    /* Methods communicating with the AI engine */
    // Make the AI play a move, return true if successful.
    private boolean playComputerMove() {
        boolean success = true;
        waitingToStart = false;
        frame.updateButtons(1);
        Cursor oldCursor = frame.getCursor();
        frame.setCursor(new Cursor(Cursor.WAIT_CURSOR));
        String boardEncoding = board.getStringEncoding();

        TextArea infoBox = frame.getInfoBox();
        infoBox.setText("Computer is thinking...");

        // Time management variables / consts
        int startDepth = searchDepth;
        int startEdepth = endGameDepth;
        int successDepth = 0;
        if(timeDependent || iterativeDeepening) {
            startDepth = Math.min(searchDepth, START_DEPTH);
            startEdepth = Math.min(endGameDepth, START_E_DEPTH);
            iterativeDeepening = true;
        }
        int timeLimit = 10000000;
        if(hardTimeLimit != 0)
            timeLimit = hardTimeLimit;
        if(timeDependent && softTimeLimit != 0 &&
           (softTimeLimit < hardTimeLimit || hardTimeLimit == 0))
            timeLimit = softTimeLimit;
        if(smartPlay) {  // giving extra time
            timeLimit = (int) (timeLimit * SMART_PLAY_SPENDING_FACTOR);
            if(hardTimeLimit != 0 && hardTimeLimit < timeLimit)
                timeLimit = hardTimeLimit; // never exceed hard limit
        }
        long startTime = (new Date()).getTime();

        int aiMove = 64; // init
        String moveStr = "";

        // Init some stat
        int countTotalSearching = 0;
        double time = 0;
        // execute the engine
        int effectiveSearchDepth = Math.min(searchDepth,
                                            Math.max(4, board.getEmptyCount() - MIN_EMPTIES_FOR_EVALUATION));
        for (int depth = startDepth, eDepth = startEdepth;
             depth <= effectiveSearchDepth || eDepth <= endGameDepth;
             depth += 2, eDepth += 2) {
            //System.out.println(depth);  // debug
            if(depth > effectiveSearchDepth && eDepth < 64 - board.getTotalDiscCount() - 2) {
                // ... - 2 since the engine will do a WDL search 2 moves before exact search
                depth -= 2;
                continue; // can't reach the end game solve yet
            }
            if(depth > effectiveSearchDepth)
                depth -= 2;
            if(eDepth > endGameDepth) { // still haven't exceeded mid-game search limit
                eDepth = endGameDepth;
            }
            // Update time
            long iterStartTime = (new Date()).getTime();
            int timeElapsed = (int)(iterStartTime - startTime);
            int timeLeft = timeLimit - timeElapsed;
            if(timeLeft <= 0)  // no time left
                break;
            // Smart play check, see if perform this iteration
            if(smartPlay) {
                if(timeDependent &&
                   timeElapsed > timeLimit * (1 / ESTIMATED_MAX_BRANCHING_FACTOR))
                   // (most likely unnecessary) && eDepth < 64 - board.getTotalDiscCount()) // but want to search if possible to solve the game
                    break;
            }
            // build up the command
            String cmd = MiniOthello.ENGINE_PATH + MiniOthello.ENGINE_NAME;
            String enginePath = cmd;
            cmd += " -D " + depth;
            cmd += " -E " + eDepth;
            cmd += " -t -st";  // always a good idea to have the statistics info.
            cmd += " -r " + randomnessLevel;
            if(timeLeft < 10000000)
                cmd += " -tm " + timeLeft;
            cmd += " -gm " + boardEncoding;
            // declare vital statistics variables
            int countSearching = 0, countEval = 0, evalExact = 999;
            double eval = 999.0;
            int WDLresponse = 999;
            int speedKn = 0;
            try {
                // Invoke engine (native code)
                if(showStat) {
                    if (eDepth < 64 - board.getTotalDiscCount() - 2)
                        infoBox.append("\nTry depth " + depth + "... ");
                    else if(eDepth < 64 - board.getTotalDiscCount())
                        infoBox.append("\nTry WDL... ");
                    else
                        infoBox.append("\nTry solving... ");
                }
                engine = Runtime.getRuntime().exec(cmd);
                InputStream is = engine.getInputStream();
                InputStreamReader isr = new InputStreamReader(is);
                BufferedReader br = new BufferedReader(isr);
                String istr = br.readLine();
                //System.out.println("Result: " + istr); // debug
                String[] response = istr.split(" ");
                int moveReturned = Integer.parseInt(response[0]);
                if(moveReturned == 64) { // code for incomplete search due time expire
                    if(showStat) {
                        //infoBox.append("\n(Depth " + val + " Incomplete)");
                        // So this a time-dependent search
                        //infoBox.append("\n(Time expired @ depth " + depth + ")");
                        infoBox.append(smartPlay? "Abort" : "Time out");
                    }
                    break;
                }
                aiMove = moveReturned;
                moveStr = "" + (char) ( (aiMove & 7) + (byte) 'A') +
                    ( (aiMove >> 3) + 1);
                if (response.length >= 7) {
                    countSearching = Integer.parseInt(response[1]);
                    countTotalSearching += countSearching;
                    countEval = Integer.parseInt(response[2]);
                    eval = Double.parseDouble(response[3]);
                    evalExact = Integer.parseInt(response[4]);
                    //time += Double.parseDouble(response[5]); // time from engine
                    time = ((double)((new Date()).getTime() - startTime)) / 1000; // GUI time
                    WDLresponse = Integer.parseInt(response[6]);
                    if (time != 0)
                        speedKn = (int) (countTotalSearching / (1000 * time));
                        //information
                    DecimalFormat fm = new DecimalFormat("0.00");
                    DecimalFormat fm2 = new DecimalFormat("0.00");
                    DecimalFormat fm3 = new DecimalFormat("0.0");
                    //DecimalFormat fm2 = new DecimalFormat("0.0#########"); // if need more digit
                    if(showStat) {
                        infoBox.setText("");
                        if(board.getM() == 0) // MiniOthello engine actually doesn't search for the first move, it knows already
                            infoBox.append("* Indifferent Opening *");
                        else if(board.getM() == 1) { // second move - three openings
                            if(aiMove == 18 || aiMove == 21 || aiMove == 42 || aiMove == 45)
                                infoBox.append("* Diagonal Opening *");
                            else {
                                if(eval < 0)
                                    infoBox.append("* Parallel Opening *");
                                else
                                    infoBox.append("* Perpendicular Opening *");
                            }
                        }
                        else if(eDepth < 64 - board.getTotalDiscCount())
                            infoBox.append("Depth: " + depth);
                        else
                            infoBox.append("Depth: " + (64 - board.getTotalDiscCount()));
                        infoBox.append("\nBest move: " + moveStr);
                        if(countTotalSearching < 1000000)
                            infoBox.append("\nSearched: " + countTotalSearching / 1000 +
                                           "K nodes");
                        else if(countTotalSearching >= 10000000)
                            infoBox.append("\nSearched: " +
                                           fm3.format((double)countTotalSearching / 1000000) +
                                           "M nodes");
                        else
                            infoBox.append("\nSearched: " +
                                           fm.format((double)countTotalSearching / 1000000) +
                                           "M nodes");
                        infoBox.append("\nTime: " + fm.format(time) + " sec.");
                        if (time >= 0.01) // mininal time to give a useful search speed estimate
                            infoBox.append("\nSpeed: " + speedKn + " Kn/s");
                        else
                            infoBox.append("\nSpeed: N/A");
                        if (evalExact < 999) {
                            infoBox.append("\nEval: " + (evalExact>0?"+":"") + evalExact);
                            if(showScoreInEval) {
                                infoBox.append(" (" + (32 + evalExact / 2) + " - " +
                                               (32 - evalExact / 2) + ")");
                            }
                        }
                        else if(evalExact > 999) // encoding for mid-game win
                            infoBox.append("\nEval: >= +" + (evalExact >> 10));
                        else if (eval != 999) { // not solved in end-game
                            if(false && (eval > 50 || eval < -50) && eval == (int)eval)  // most like solved in midd-game
                                infoBox.append("\nEval: " + (int) eval); // may not be need, not exactly sure
                            else {
                                if(scaleEval && WDLresponse != 2 && WDLresponse != -2)
                                    infoBox.append("\nEval: " + (eval>0?"+":"") +
                                        fm2.format(scaleEvaluation(eval)));
                                else
                                    infoBox.append("\nEval: " + (eval>0?"+":"") +
                                        fm2.format(eval));
                            }
                            // WDL info
                            int nEmpty = 64 - board.getTotalDiscCount();
                            if(eDepth + 2 >= nEmpty && eDepth < nEmpty && WDLresponse != 999) {
                                infoBox.append(eval > 0? " (Win)" : (eval < 0? " (Loss)" : " (Draw)"));
                            }
                        }
                        else  // first move
                            infoBox.append("\nEval: 0.0");
                    }
                }
                int exitVal = engine.waitFor();
                if (exitVal != 0) {
                    System.out.println("Warnig: Engine return value is " +
                                       exitVal);
                }
            }
            catch (IOException e) {
                sendEngineError("Cannot execute " + MiniOthello.ENGINE_PATH +
                                MiniOthello.ENGINE_NAME);
                success = false;
                break;
            }
            catch (InterruptedException e) {
                //System.out.println("Engine process interrupted"); -- debug
                if (engine != null)
                    engine.destroy();
                infoBox.append("\n(Interrupted at Depth " + depth + ")");
                success = false;
                break;
            }
            catch (NumberFormatException e) {
                sendEngineError("Invalid Response from Engine. " + e.getMessage());
                if (engine != null)
                    engine.destroy();
                success = false;
                break;
            }
            catch(NullPointerException e) { // dangerous, but needed
                // Engine terminated by user (if there is no bug)
                infoBox.append("\n(Interrupted at Depth " + depth + ")");
                success = false;
                break;
            }
            successDepth = depth;
            if(eDepth >= 64 - board.getTotalDiscCount()) { // have solved exactly
                successDepth = eDepth;
                break;
            }
            // already solved, no need to continue searching
            if(evalExact < 999) {
                // (engine returns >= 999 if cannot solve)
                break;
            }
            // not need to search deeper for 1st and 2nd move
            if(board.getM() < 2) {
                break;
            }
        }

        if (!board.legalMove(aiMove)) {
            sendEngineError(
                "Illegal Move Received from Engine. Value: " + aiMove);
            success = false;
        }
        else if(successDepth == 0) {  // got no successful search
            sendEngineError("No Search Completed");
            success = false;
        }
        else { // can get a legal move from computer
            if(!comNoMove) {
                // play this move
                if(!showStat)
                    infoBox.setText("Computer's Move: " + moveStr);
                board.makeMove(aiMove);
                storeMessage(infoBox.getText());
                currPlayer = 1 - currPlayer;
                currColor = board.whoseTurn();
                updateGraphics();
            }
        }

        frame.setCursor(oldCursor);
        return success;
    }

    // Update the OthelloFrame window
    private void updateGUI() {
        frame.repaint();
    }

    // When not able to communicate with the AI engine
    private void sendEngineError(String msg) {
        /** To be implemented **/
        System.out.println("Engine Error Message: " + msg);
    }

    // When the game ends
    private void endGame() {
        gameEnded = true;
        int nb = board.getDiscCount(Board.BLACK);
        int nw = board.getDiscCount(Board.WHITE);
        String result = "";
        if(nb > nw)
            result = "BLACK WINS BY: " + (64-nw) + " - " + nw;
        else if(nb < nw)
            result = "WHITE WINS BY: " + (64-nb) + " - " + nb;
        else // draw
            result = "GAME IS DRAWN AT: " + nb + " - " + nw;
        frame.getInfoBox().append("\n" + result);
        updateGraphics();
        if(saveLastGame)
            saveLastGame();
    }

    // Function scales the evaluation to approximate score -- very inaccurate
    private double scaleEvaluation(double eval) {
        if(eval == 0)
            return eval;
        double result = (eval/Math.abs(eval)) * 10*Math.sqrt(Math.abs(eval));
        if(eval == 2.0 || eval == -2.0) // exeption
            return eval;
        return result;
    }

    // Helper function that stores and displays info box message from the message stack
    private void storeMessage(String msg) {
        messages[board.getM()] = msg;
        messageAvailable[board.getM()] = true;
    }

    private void displayStoredMessage() {
        if(hasAvailMessage())
            frame.getInfoBox().setText(messages[board.getM()]);
        else
            frame.getInfoBox().setText("");
    }

    private boolean hasAvailMessage() {
        return messageAvailable[board.getM()];
    }

    // Auto saves the last game
    private void saveLastGame() {
        String filename = MiniOthello.SAVE_GAME_PATH + MiniOthello.LAST_GAME_FILE_NAME;
        try {
            File f = new File(filename);
            FileWriter fw = new FileWriter(f);
            fw.write(board.isMirrored()? "1 " : "0 ");
            fw.write("" + 0 + " " + board.getTop() + " "); // always rewind the last game
            for(int i=1; i<=board.getTop(); i++) {
                int move = board.getRecordedMove(i);
                if(move == Board.PASS)
                    fw.write("pass ");
                else {
                    char[] c = new char[2];
                    c[0] = (char)((move & 7) + (byte)'a');
                    c[1] = (char)((move >> 3) + (byte)'1');
                    fw.write(c);
                    fw.write(" ");
                }
            }
            fw.flush();
        }
        catch(IOException e) {
            System.out.println("Warning: Error writing to file " + filename +
                               ". Message: " + e.getMessage());
            //frame.getInfoBox().setText("Can't write to file " + filename + "!");
        }
    }

/* Never mind ...
    // A pair holding an int and a double
    private class IntDouble {
        int d;
        double f;
        public IntDouble(int _d, double _f) {d = _d; f = _f;}
    }

    // generate a known opening move without calling the engine
    private IntDouble getOpeningMove() {
        if(board.getM() == 0) {  // to generate the first move

        }
    }
        */
}
