/***************************************************************************
 * OthelloFrame.java
 *
 * The displayed window object.
 * This is the GUI.
 *
 * Note: it should never directly modify the state of the board
 *
 * Yunpeng Li
 */

package othelloGUI;

import java.io.*;
import java.awt.*;
import java.awt.event.*;
import java.util.*;
import java.text.*;
import java.applet.Applet;
import javax.swing.*;

public class OthelloFrame extends Frame {
    // --- Constants ---
    public static int DRAWING_AREA_X_OFFSET = 0;
    public static int DRAWING_AREA_Y_OFFSET = 42;
    public static int TOOLBAR_X_OFFSET = 0;
    public static int TOOLBAR_Y_OFFSET = 0;  // relative to drawing area
    public static int TOOLBAR_X_HEIGHT = 0;
    public static int BOARD_FRAME_X_OFFSET = 30;
    public static int BOARD_FRAME_Y_OFFSET = 55;  // absolute
    public static int BOARD_FRAME_WIDTH = 40;

    public static int LOWER_MARGIN = 15;  // from bottom of board frame to window edge
    public static int RIGHT_MARGIN = 250;  // area to the right of board frame, will have some AWT components

    public static int BOARD_CELL_SIZE = 60;
    public static int DISC_RADIUS = 25;
    public static int GRAY_DOT_RADIUS = 4;
    public static int BLUE_DOT_RADIUS = 4;

    // Derived constants
    public static int BOARD_SIZE = BOARD_CELL_SIZE * 8;
    public static int BOARD_X_OFFSET = BOARD_FRAME_X_OFFSET + BOARD_FRAME_WIDTH;
    public static int BOARD_Y_OFFSET = BOARD_FRAME_Y_OFFSET + BOARD_FRAME_WIDTH;
    public static int BOARD_FRAME_SIZE = BOARD_SIZE + 2 * BOARD_FRAME_WIDTH;
    public static int BOARD_FRAME_RIGHT_POS = BOARD_FRAME_X_OFFSET + BOARD_FRAME_SIZE;
    public static int BOARD_FRAME_LOWER_POS = BOARD_FRAME_Y_OFFSET + BOARD_FRAME_SIZE;
    public static int WINDOW_SIZE_X = BOARD_FRAME_RIGHT_POS + RIGHT_MARGIN;
    public static int WINDOW_SIZE_Y = BOARD_FRAME_LOWER_POS + LOWER_MARGIN;

    // Button size and positions
    public static int BIG_BUTTON_WIDTH = 130;
    public static int BIG_BUTTON_HEIGHT = 55;
    public static int BIG_BUTTON_POS_X = WINDOW_SIZE_X - (RIGHT_MARGIN + BIG_BUTTON_WIDTH) / 2;
    public static int BIG_BUTTON_POS_Y = WINDOW_SIZE_Y - 5 - LOWER_MARGIN - BIG_BUTTON_HEIGHT;

    public static int BUTTON_WIDTH = 40;
    public static int BUTTON_HEIGHT = 30;
    public static int BUTTON_SPACE_BETWEEN = 5;
    public static int BUTTON_GROUP_WIDTH = 4*BUTTON_WIDTH + 3*BUTTON_SPACE_BETWEEN;
    public static int BUTTON_POS_X = WINDOW_SIZE_X - (RIGHT_MARGIN + BUTTON_GROUP_WIDTH) / 2;
    public static int BUTTON_POS_Y = BIG_BUTTON_POS_Y - BUTTON_HEIGHT - 20;

    // Legal move cross
    public static int CROSS_DIM_S = 6;
    public static int CROSS_DIM_L = 7;

    // Text Area
    public static int INFOBOX_POS_X = BOARD_FRAME_RIGHT_POS + 30;
    public static int INFOBOX_POS_Y = BUTTON_POS_Y - 200;

    // search and end-game depth index table -- for diff level menu listens
    public static int[][] DIFFICULTY_LEVEL = {
        {1, 8}, {2, 10}, {4, 12}, {6, 14}, {8, 16}, {10, 18}, {12, 20}
    };
    // computer role index table
    public static int[][] PLAYER_ROLES = {
        {Game.COMPUTER_PLAYER, Game.HUMAN_PLAYER},
        {Game.HUMAN_PLAYER, Game.COMPUTER_PLAYER},
        {Game.HUMAN_PLAYER, Game.HUMAN_PLAYER},
        {Game.COMPUTER_PLAYER, Game.COMPUTER_PLAYER}
    };

    // Menu indices
    public static int STOOGE = 0, MINDLESS = 1, NOVICE = 2, BEGINNER = 3,
        AMATEUR = 4, AMATEUR_P = 5, EXPERIENCED = 6, EXPERIENCED_P = 7,
        ADVANCED = 8, CUSTOMIZE = 9;
    public static int COM_HUM = 0, HUM_COM = 1, HUM_HUM = 2, COM_COM = 3;

    // Time limits - unit: milliseconds
    public static int[] SOFT_LIMIT = {1000, 2000, 4000, 8000, 15000, 30000, 60000, 0};
    public static int[] HARD_LIMIT = {2000, 5000, 10000, 30000, 60000, 0};


    // --- Variables ---
    private Image screen;  // double buffering to avoid blinking of the Frame
    private Graphics g;
    private Graphics2D g2;

    // Drive the game, when computer makes moves (another thread)
    private GameDriver gd;
    private OthelloFrame frame;  // reference to self

    // Menus
    private MenuBar menu;
    private Menu fileM, gameM, difficultyM, optionsM, helpM; // menus on the menubar
    private MenuItem newGameM, loadGameM, saveGameM, exitM;  // file
    private CheckboxMenuItem[] comPlayerM; // comBM, comWM, comNM, comVScom;  // game
    private MenuItem undoM, redoM, undoAllM, redoAllM;  // game
    private MenuItem stopComThinkingM;  // game
    private MenuItem comPlayMoveM;  // game
    private CheckboxMenuItem[] levelMs;  // difficulty level
    private CheckboxMenuItem flexibleTimeM;  // difficulty level
    private CheckboxMenuItem showAvailMovesM, showLastMoveM, showStatM;  // options
    private CheckboxMenuItem mirroredStartM;  // options
    private CheckboxMenuItem saveSettingsM;  // options
    private MenuItem saveSettingsNowM, refreshScreenM;  // options
    private MenuItem helpContentM, aboutMiniOthelloM;

    // Sub-menus
    // Time management
    private Menu fixedDepthM;
    private Menu softLimitM, hardLimitM;
    private CheckboxMenuItem[] sItems;
    private CheckboxMenuItem[] hItems;

    // text fields, buttons
    private TextArea pastMoves, infoBox;
    private Button startB, undoB, redoB, undoAllB, redoAllB;

    // state variables
    private int comPlayerState;  // 0 - computer players black, 1 W, 2 None, 3, both
    private int level;  // 0-(1,8) 2-(2,10) 3-(4, 12) 4-(6,14) 5-(8,16) 6-(10,18)
    //private boolean showAvailMoves, showLastMove, showStat, mirroredStart; // defined in Game
    private boolean gameStarting;

    // game info/control
    private Board board;
    private Game game;

    // GUI boolean variables are declared in Game
    // .....

    /* Constructor */
    public OthelloFrame(String _title, Board _board, Game _game) {
        /* Initialize the frame */
        super(_title);
        board = _board;
        game = _game;

        setSize(WINDOW_SIZE_X, WINDOW_SIZE_Y);
        setVisible(true);
        setResizable(false); // no resize

        // enable awt events that might be needed
        enableEvents(AWTEvent.MOUSE_EVENT_MASK);
        enableEvents(AWTEvent.KEY_EVENT_MASK);
        enableEvents(AWTEvent.WINDOW_EVENT_MASK);
        enableEvents(AWTEvent.MOUSE_MOTION_EVENT_MASK);
        enableEvents(AWTEvent.ACTION_EVENT_MASK);

        setBackground(Util.WINDOW_BACKGROUND);

        screen = createImage(getSize().width, getSize().height);
        g = getGraphics();
        g2 = (Graphics2D)(screen.getGraphics());
        g2.setRenderingHint(RenderingHints.KEY_ANTIALIASING,
                            RenderingHints.VALUE_ANTIALIAS_ON);
        Image icon = Toolkit.getDefaultToolkit().getImage(MiniOthello.PICTURE_PATH +
            MiniOthello.ICON_FILE);
        setIconImage(icon);


        /* Instantiate Menus */
        // menubar
        menu = new MenuBar();

        // menus
        fileM = new Menu("File");
        gameM = new Menu("Game");
        difficultyM = new Menu("AI Settings");
        optionsM = new Menu("Options");
        helpM = new Menu("Help");

        // menuitems
        // file
        newGameM = new MenuItem("New Game", new MenuShortcut( (int) 'N'));
        loadGameM = new MenuItem("Load Game...");
        //loadGameM.setEnabled(false);  // not yet implemented
        saveGameM = new MenuItem("Save Game...");
        //saveGameM.setEnabled(false);  // not yet implemented
        exitM = new MenuItem("Exit");

        // game
        comPlayerM = new CheckboxMenuItem[4];
        comPlayerM[HUM_COM] = new CheckboxMenuItem("Computer Plays White",
            game.player[0] == Game.HUMAN_PLAYER &&
            game.player[1] == Game.COMPUTER_PLAYER);
        comPlayerM[COM_HUM] = new CheckboxMenuItem("Computer Plays Black",
            game.player[0] == Game.COMPUTER_PLAYER &&
            game.player[1] == Game.HUMAN_PLAYER);
        comPlayerM[HUM_HUM] = new CheckboxMenuItem("Computer Doesn't Play",
            game.player[0] == Game.HUMAN_PLAYER &&
            game.player[1] == Game.HUMAN_PLAYER);
        comPlayerM[COM_COM] = new CheckboxMenuItem("Computer Self-plays",
            game.player[0] == Game.COMPUTER_PLAYER &&
            game.player[1] == Game.COMPUTER_PLAYER);
        //--- currently not using ---
        undoM = new MenuItem("Undo", new MenuShortcut( (int) 'B'));
        redoM = new MenuItem("Redo", new MenuShortcut( (int) 'F'));
        undoAllM = new MenuItem("Undo All Moves");
        redoAllM = new MenuItem("Redo All Moves");
        //---------------------------
        comPlayMoveM = new MenuItem("Computer Plays This Move",
                                    new MenuShortcut( (int) 'D'));
        stopComThinkingM = new MenuItem("Force Computer to Move (if it is thinking too long)");

        // Difficulty level
        levelMs = new CheckboxMenuItem[CUSTOMIZE + 1];
        boolean tempbool = false;
        boolean tempbool2;
        tempbool2 = game.searchDepth == 1 && game.endGameDepth == 8;
        tempbool |= tempbool2;
        levelMs[STOOGE] = new CheckboxMenuItem("1-Ply, 8-Ply End  (Fastest, Easiest)", tempbool2);
        tempbool2 = game.searchDepth == 2 && game.endGameDepth == 10;
        tempbool |= tempbool2;
        levelMs[MINDLESS] = new CheckboxMenuItem("2-Ply, 10-Ply End", tempbool2);
        tempbool2 = game.searchDepth == 4 && game.endGameDepth == 12;
        tempbool |= tempbool2;
        levelMs[NOVICE] = new CheckboxMenuItem("4-Ply, 12-Ply End", tempbool2);
        tempbool2 = game.searchDepth == 6 && game.endGameDepth == 14;
        tempbool |= tempbool2;
        levelMs[BEGINNER] = new CheckboxMenuItem("6-Ply, 14-Ply End", tempbool2);
        tempbool2 = game.searchDepth == 8 && game.endGameDepth == 16;
        tempbool |= tempbool2;
        levelMs[AMATEUR] = new CheckboxMenuItem("8-Ply, 16-Ply End", tempbool2);
        tempbool2 = game.searchDepth == 8 && game.endGameDepth == 18;
        tempbool |= tempbool2;
        levelMs[AMATEUR_P] = new CheckboxMenuItem("8-Ply, 18-Ply End", tempbool2);
        tempbool2 = game.searchDepth == 10 && game.endGameDepth == 18;
        tempbool |= tempbool2;
        levelMs[EXPERIENCED] = new CheckboxMenuItem("10-Ply, 18-Ply End  (Occasionally slow)", tempbool2);
        tempbool2 = game.searchDepth == 10 && game.endGameDepth == 20;
        tempbool |= tempbool2;
        levelMs[EXPERIENCED_P] = new CheckboxMenuItem("10-Ply, 20-Ply End", tempbool2);
        tempbool2 = game.searchDepth == 12 && game.endGameDepth == 22;
        tempbool |= tempbool2;
        levelMs[ADVANCED] = new CheckboxMenuItem("12-Ply, 22-Ply End  ( Very Slow !! )", tempbool2);
        // the CUSTOMIZE menu item
        levelMs[CUSTOMIZE] = new CheckboxMenuItem("Customize...", !tempbool);
        levelMs[CUSTOMIZE].setEnabled(false);  // not yet implemented
        // the flexible time (smart play) item
        flexibleTimeM = new CheckboxMenuItem("Flexible Time Limit (more efficient)");

        // A sub-menu
        fixedDepthM = new Menu("Fixed-depth Search");

        // Options
        showAvailMovesM = new CheckboxMenuItem("Show Available Moves", game.showAvailMoves);
        showLastMoveM = new CheckboxMenuItem("Show Last Move", game.showLastMove);
        showStatM = new CheckboxMenuItem("Show Statistics", game.showStat);
        mirroredStartM = new CheckboxMenuItem("Mirrored Starting Position", game.mirroredStart);

        saveSettingsM = new CheckboxMenuItem("Save Settings on Exit"); //game.saveSettings);
        saveSettingsNowM = new MenuItem("Save Settings Now");
        refreshScreenM = new MenuItem("Refresh Screen", new MenuShortcut((int)'R'));

        // Help
        helpContentM = new MenuItem("Content...", new MenuShortcut(KeyEvent.VK_F1)); //Ctrl+F1
        helpContentM.setEnabled(false); //not yet implemented
        aboutMiniOthelloM = new MenuItem("About MiniOthello...");

        /* Submenus */
        softLimitM = new Menu("Time-dependent Search");
        hardLimitM = new Menu("Hard Limit (per-Move)");
        sItems = new CheckboxMenuItem[SOFT_LIMIT.length];
        hItems = new CheckboxMenuItem[HARD_LIMIT.length];
        sItems[0] = new CheckboxMenuItem("1 Second of Move");
        for(int i=1; i<sItems.length-1; i++)
            sItems[i] = new CheckboxMenuItem(SOFT_LIMIT[i]/1000 + " Seconds per Move", false);
        sItems[sItems.length-1] = new CheckboxMenuItem("Customize...", false);
        sItems[sItems.length-1].setEnabled(false); // not yet implemented
        for(int i=0; i<hItems.length-1; i++)
            hItems[i] = new CheckboxMenuItem(HARD_LIMIT[i]/1000 + " Seconds Maximum", false);
        hItems[hItems.length-1] = new CheckboxMenuItem("None", false);


        /* TextArea */
        pastMoves = new TextArea("", 40, 25, TextArea.SCROLLBARS_VERTICAL_ONLY);
        pastMoves.setBackground(Color.white);
        pastMoves.setFont(new Font("Courier", Font.BOLD, 12));
        pastMoves.setSize(190, 185);
        pastMoves.setLocation(BOARD_FRAME_RIGHT_POS + 30, BUTTON_POS_Y - 345);
        pastMoves.setEditable(false);
        pastMoves.setFocusable(false);
        pastMoves.setCursor(new Cursor(Cursor.DEFAULT_CURSOR));
        infoBox = new TextArea("Message", 6, 25, TextArea.SCROLLBARS_NONE);
        infoBox.setBackground(Color.white);
        infoBox.setFont(new Font("Courier", Font.BOLD, 12));
        infoBox.setSize(190, 125);
        infoBox.setLocation(BOARD_FRAME_RIGHT_POS + 30, BUTTON_POS_Y - 140);
        infoBox.setEditable(false);
        infoBox.setFocusable(false);
        infoBox.setCursor(new Cursor(Cursor.DEFAULT_CURSOR));

        /* Instantiate and initialize the buttons */
        startB = new Button("PASS");
        undoB = new Button("<-");
        redoB = new Button("->");
        undoAllB = new Button("|<<");
        redoAllB = new Button(">>|");
        // initialize
        startB.setFont(new Font("Dialog", Font.BOLD, 32));
        startB.setLocation(BIG_BUTTON_POS_X, BIG_BUTTON_POS_Y);
        startB.setSize(BIG_BUTTON_WIDTH, BIG_BUTTON_HEIGHT);
        Font buttonFont = new Font("monospaced", Font.BOLD, 18);
        undoB.setFont(buttonFont);
        redoB.setFont(buttonFont);
        undoAllB.setFont(buttonFont);
        redoAllB.setFont(buttonFont);
        undoAllB.setLocation(BUTTON_POS_X, BUTTON_POS_Y);
        undoB.setLocation(BUTTON_POS_X + 1 * (BUTTON_WIDTH + BUTTON_SPACE_BETWEEN), BUTTON_POS_Y);
        redoB.setLocation(BUTTON_POS_X + 2 * (BUTTON_WIDTH + BUTTON_SPACE_BETWEEN), BUTTON_POS_Y);
        redoAllB.setLocation(BUTTON_POS_X + 3 * (BUTTON_WIDTH + BUTTON_SPACE_BETWEEN), BUTTON_POS_Y);
        undoB.setSize(BUTTON_WIDTH, BUTTON_HEIGHT);
        redoB.setSize(BUTTON_WIDTH, BUTTON_HEIGHT);
        undoAllB.setSize(BUTTON_WIDTH, BUTTON_HEIGHT);
        redoAllB.setSize(BUTTON_WIDTH, BUTTON_HEIGHT);
        undoB.setEnabled(false);
        redoB.setEnabled(false);
        undoAllB.setEnabled(false);
        redoAllB.setEnabled(false);
        startB.setEnabled(false);



        /* Menu ActionListeners/ItemListeners */
        // File
        newGameM.addActionListener(new ActionListener() {
            public void actionPerformed(ActionEvent e) {
                game.newGame();
            }
        });
        loadGameM.addActionListener(new ActionListener() {
            public void actionPerformed(ActionEvent e) {
                game.loadGame();
            }
        });
        saveGameM.addActionListener(new ActionListener() {
            public void actionPerformed(ActionEvent e) {
                game.saveGame();
            }
        });
        exitM.addActionListener(new ActionListener() {
            public void actionPerformed(ActionEvent e) {
                exit();
            }
        });

        //Game - computer role
        comPlayerM[COM_HUM].addItemListener(new ItemListener() {
            public void itemStateChanged(ItemEvent e) {
                game.player[0] = PLAYER_ROLES[COM_HUM][0];
                game.player[1] = PLAYER_ROLES[COM_HUM][1];
                game.rollSwapped();
                gameStarting = false;
                paint(getGraphics());
            }
        });
        comPlayerM[HUM_COM].addItemListener(new ItemListener() {
            public void itemStateChanged(ItemEvent e) {
                game.player[0] = PLAYER_ROLES[HUM_COM][0];
                game.player[1] = PLAYER_ROLES[HUM_COM][1];
                game.rollSwapped();
                gameStarting = false;
                paint(getGraphics());
            }
        });
        comPlayerM[HUM_HUM].addItemListener(new ItemListener() {
            public void itemStateChanged(ItemEvent e) {
                game.player[0] = PLAYER_ROLES[HUM_HUM][0];
                game.player[1] = PLAYER_ROLES[HUM_HUM][1];
                gameStarting = false;
                paint(getGraphics());
            }
        });
        comPlayerM[COM_COM].addItemListener(new ItemListener() {
            public void itemStateChanged(ItemEvent e) {
                game.player[0] = PLAYER_ROLES[COM_COM][0];
                game.player[1] = PLAYER_ROLES[COM_COM][1];
                game.rollSwapped();
                gameStarting = false;
                paint(getGraphics());
            }
        });
        comPlayMoveM.addActionListener(new ActionListener() {
            public void actionPerformed(ActionEvent e) {
                //System.out.println("currPlayer: " + game.getCurrPlayer() + " now player: " + game.player[game.getCurrPlayer()]); // debug
                if(game.player[game.getCurrPlayer()] != Game.HUMAN_PLAYER ||
                   !game.canPlay())
                    return;
                if(board.hasLegalMove()) {
                    gd = new GameDriver(frame, game, board);
                    gd.setState(GameDriver.COM_HELP_MAKE_MOVE);
                    gd.start();
                }
                else {
                    board.switchSide();
                    boolean shouldEnd = !board.hasLegalMove();
                    board.switchSide();
                    if(shouldEnd) {/* do nothing -- ideally shouldn't get here */}
                    else {
                        gd = new GameDriver(frame, game, board);
                        gd.setState(GameDriver.COM_HELP_MAKE_PASS);
                        gd.start();
                    }
                }
            }
        });
        stopComThinkingM.addActionListener(new ActionListener() {
            public void actionPerformed(ActionEvent e) {
                game.stopComupterThinking();
            }
        });

        // difficulty level
        levelMs[STOOGE].addItemListener(new ItemListener() {
            public void itemStateChanged(ItemEvent e) {
                game.searchDepth = 1;
                game.endGameDepth = 8;
                game.timeDependent = false;
                game.updateGraphics();
            }
        });
        levelMs[MINDLESS].addItemListener(new ItemListener() {
            public void itemStateChanged(ItemEvent e) {
                game.searchDepth = 2;
                game.endGameDepth = 10;
                game.timeDependent = false;
                game.updateGraphics();
            }
        });
        levelMs[NOVICE].addItemListener(new ItemListener() {
            public void itemStateChanged(ItemEvent e) {
                game.searchDepth = 4;
                game.endGameDepth = 12;
                game.timeDependent = false;
                game.updateGraphics();
            }
        });
        levelMs[BEGINNER].addItemListener(new ItemListener() {
            public void itemStateChanged(ItemEvent e) {
                game.searchDepth = 6;
                game.endGameDepth = 14;
                game.timeDependent = false;
                game.updateGraphics();
            }
        });
        levelMs[AMATEUR].addItemListener(new ItemListener() {
            public void itemStateChanged(ItemEvent e) {
                game.searchDepth = 8;
                game.endGameDepth = 16;
                game.timeDependent = false;
                game.updateGraphics();
            }
        });
        levelMs[AMATEUR_P].addItemListener(new ItemListener() {
            public void itemStateChanged(ItemEvent e) {
                game.searchDepth = 8;
                game.endGameDepth = 18;
                game.timeDependent = false;
                game.updateGraphics();
            }
        });
        levelMs[EXPERIENCED].addItemListener(new ItemListener() {
            public void itemStateChanged(ItemEvent e) {
                game.searchDepth = 10;
                game.endGameDepth = 18;
                game.timeDependent = false;
                game.updateGraphics();
            }
        });
        levelMs[EXPERIENCED_P].addItemListener(new ItemListener() {
            public void itemStateChanged(ItemEvent e) {
                game.searchDepth = 10;
                game.endGameDepth = 20;
                game.timeDependent = false;
                game.updateGraphics();
            }
        });
        levelMs[ADVANCED].addItemListener(new ItemListener() {
            public void itemStateChanged(ItemEvent e) {
                game.searchDepth = 12;
                game.endGameDepth = 22;
                game.timeDependent = false;
                game.updateGraphics();
            }
        });
        levelMs[CUSTOMIZE].addItemListener(new ItemListener() {
            public void itemStateChanged(ItemEvent e) {
                // not yet implemented
            }
        });
        flexibleTimeM.addItemListener(new ItemListener() {
            public void itemStateChanged(ItemEvent e) {
                game.smartPlay = !game.smartPlay;
                flexibleTimeM.setState(game.smartPlay);
            }
        });

        // options
        showAvailMovesM.addItemListener(new ItemListener() {
            public void itemStateChanged(ItemEvent e) {
                game.showAvailMoves = !game.showAvailMoves;
                showAvailMovesM.setState(game.showAvailMoves);
                game.updateGraphics();
            }
        });
        showLastMoveM.addItemListener(new ItemListener() {
            public void itemStateChanged(ItemEvent e) {
                game.showLastMove = !game.showLastMove;
                showLastMoveM.setState(game.showLastMove);
                game.updateGraphics();
            }
        });
        showStatM.addItemListener(new ItemListener() {
            public void itemStateChanged(ItemEvent e) {
                game.showStat = !game.showStat;
                showStatM.setState(game.showStat);
                game.updateGraphics();
            }
        });
        mirroredStartM.addItemListener(new ItemListener() {
            public void itemStateChanged(ItemEvent e) {
                game.mirroredStart = !game.mirroredStart;
                mirroredStartM.setState(game.mirroredStart);
            }
        });
        saveSettingsM.addItemListener(new ItemListener() {
            public void itemStateChanged(ItemEvent e) {
                game.saveSettings = !game.saveSettings;
                saveSettingsM.setState(game.saveSettings);
            }
        });
        saveSettingsNowM.addActionListener(new ActionListener() {
            public void actionPerformed(ActionEvent e) {
                game.saveSettings();
            }
        });
        refreshScreenM.addActionListener(new ActionListener() {
            public void actionPerformed(ActionEvent e) {
                game.recreateFrame();
            }
        });

        // About
        aboutMiniOthelloM.addActionListener(new ActionListener() {
            public void actionPerformed(ActionEvent e) {
                showAbout();
            }
        });


        /* Button ActionListeners */
        startB.addActionListener(new ActionListener() {
            public void actionPerformed(ActionEvent e) {
                startB.setSize(0, 0);
                startB.setEnabled(false);
                if(startB.getLabel().equals("PASS")) {
                    if(!game.canPlay())
                        return;  // should really never happen, but just in case this is a bug
                    gd = new GameDriver(frame, game, board);
                    gd.setState(GameDriver.MAKE_HUMAN_PASS);
                    gd.start();
                }
                else if(startB.getLabel().equals("Start")) {
                    if(!game.canPlay())
                        return;  // should really never happen, but just in case this is a bug
                    if(game.player[0] == Game.COMPUTER_PLAYER &&
                       game.player[1] == Game.COMPUTER_PLAYER) {
                        gd = new GameDriver(frame, game, board);
                        gd.setState(GameDriver.SELF_PLAY);
                        gd.start();
                    }
                    else {
                        gd = new GameDriver(frame, game, board);
                        gd.setState(GameDriver.MAKE_COM_MOVE);
                        gd.start();
                    }
                }
                else if(startB.getLabel().equals("New Game")) {
                    if(!game.canPlay())
                        return;  // should really never happen, but just in case this is a bug
                    game.newGame();
                }
                else if(startB.getLabel().equals("Stop")) {
                    game.stopSelfPlay();
                }
            }
        });
        // undo/redo
        undoB.addActionListener(new ActionListener() {
            public void actionPerformed(ActionEvent e) {
                if(game.canPlay()) {
                    gd = new GameDriver(frame, game, board);
                    gd.setState(GameDriver.UNDO);
                    gd.start();
                }
            }
        });
        undoAllB.addActionListener(new ActionListener() {
            public void actionPerformed(ActionEvent e) {
                if(game.canPlay()) {
                    gd = new GameDriver(frame, game, board);
                    gd.setState(GameDriver.UNDO_ALL);
                    gd.start();
                }
            }
        });
        redoB.addActionListener(new ActionListener() {
            public void actionPerformed(ActionEvent e) {
                if(game.canPlay()) {
                    gd = new GameDriver(frame, game, board);
                    gd.setState(GameDriver.REDO);
                    gd.start();
                }
            }
        });
        redoAllB.addActionListener(new ActionListener() {
            public void actionPerformed(ActionEvent e) {
                if(game.canPlay()) {
                    gd = new GameDriver(frame, game, board);
                    gd.setState(GameDriver.REDO_ALL);
                    gd.start();
                }
            }
        });

        /* Submenu action listeners */
        sItems[0].addItemListener(new ItemListener() {
            public void itemStateChanged(ItemEvent e) {
                game.timeDependent = true;
                game.softTimeLimit = SOFT_LIMIT[0];
                game.searchDepth = Game.MAX_SEARCH_DEPTH;
                game.endGameDepth = Game.MAX_END_GAME_DEPTH;
                updateMenu();
            }
        });
        sItems[1].addItemListener(new ItemListener() {
            public void itemStateChanged(ItemEvent e) {
                game.timeDependent = true;
                game.softTimeLimit = SOFT_LIMIT[1];
                game.searchDepth = Game.MAX_SEARCH_DEPTH;
                game.endGameDepth = Game.MAX_END_GAME_DEPTH;
                updateMenu();
            }
        });
        sItems[2].addItemListener(new ItemListener() {
            public void itemStateChanged(ItemEvent e) {
                game.timeDependent = true;
                game.softTimeLimit = SOFT_LIMIT[2];
                game.searchDepth = Game.MAX_SEARCH_DEPTH;
                game.endGameDepth = Game.MAX_END_GAME_DEPTH;
                updateMenu();
            }
        });
        sItems[3].addItemListener(new ItemListener() {
            public void itemStateChanged(ItemEvent e) {
                game.timeDependent = true;
                game.softTimeLimit = SOFT_LIMIT[3];
                game.searchDepth = Game.MAX_SEARCH_DEPTH;
                game.endGameDepth = Game.MAX_END_GAME_DEPTH;
                updateMenu();
            }
        });
        sItems[4].addItemListener(new ItemListener() {
            public void itemStateChanged(ItemEvent e) {
                game.timeDependent = true;
                game.softTimeLimit = SOFT_LIMIT[4];
                game.searchDepth = Game.MAX_SEARCH_DEPTH;
                game.endGameDepth = Game.MAX_END_GAME_DEPTH;
                updateMenu();
            }
        });
        sItems[5].addItemListener(new ItemListener() {
            public void itemStateChanged(ItemEvent e) {
                game.timeDependent = true;
                game.softTimeLimit = SOFT_LIMIT[5];
                game.searchDepth = Game.MAX_SEARCH_DEPTH;
                game.endGameDepth = Game.MAX_END_GAME_DEPTH;
                updateMenu();
            }
        });
        sItems[6].addItemListener(new ItemListener() {
            public void itemStateChanged(ItemEvent e) {
                game.timeDependent = true;
                game.softTimeLimit = SOFT_LIMIT[6];
                game.searchDepth = Game.MAX_SEARCH_DEPTH;
                game.endGameDepth = Game.MAX_END_GAME_DEPTH;
                updateMenu();
            }
        });
        sItems[7].addItemListener(new ItemListener() {
            public void itemStateChanged(ItemEvent e) {
                game.timeDependent = true;
                // ...
                updateMenu();
            }
        });  // customized item, not yet implemented

        hItems[0].addItemListener(new ItemListener() {
            public void itemStateChanged(ItemEvent e) {
                game.hardTimeLimit = HARD_LIMIT[0];
                updateMenu();
            }
        });
        hItems[1].addItemListener(new ItemListener() {
            public void itemStateChanged(ItemEvent e) {
                game.hardTimeLimit = HARD_LIMIT[1];
                updateMenu();
            }
        });
        hItems[2].addItemListener(new ItemListener() {
            public void itemStateChanged(ItemEvent e) {
                game.hardTimeLimit = HARD_LIMIT[2];
                updateMenu();
            }
        });
        hItems[3].addItemListener(new ItemListener() {
            public void itemStateChanged(ItemEvent e) {
                game.hardTimeLimit = HARD_LIMIT[3];
                updateMenu();
            }
        });
        hItems[4].addItemListener(new ItemListener() {
            public void itemStateChanged(ItemEvent e) {
                game.hardTimeLimit = HARD_LIMIT[4];
                updateMenu();
            }
        });
        hItems[5].addItemListener(new ItemListener() {
            public void itemStateChanged(ItemEvent e) {
                game.hardTimeLimit = HARD_LIMIT[5];
                updateMenu();
            }
        });




        /* Adding AWT components */

        /* Add items to menus and menus to the bar */
        // file
        fileM.add(newGameM);
        fileM.addSeparator();
        fileM.add(loadGameM);
        fileM.add(saveGameM);
        fileM.addSeparator();
        fileM.add(exitM);
        // game
        for (int i = 0; i < 4; i++)
            gameM.add(comPlayerM[i]);
        gameM.addSeparator();
        gameM.add(comPlayMoveM);
        gameM.addSeparator();
        gameM.add(stopComThinkingM);
        // difficulty level
        // submenu
        for(int i=0; i<sItems.length-1; i++)
            softLimitM.add(sItems[i]);
        softLimitM.addSeparator();
        softLimitM.add(sItems[sItems.length-1]);
        softLimitM.addSeparator();
        softLimitM.add(flexibleTimeM);  // the flexible-time menu item
        for(int i=0; i<hItems.length; i++)
            hardLimitM.add(hItems[i]);
        // the diff menu
        for (int i = 0; i < CUSTOMIZE; i++)
            fixedDepthM.add(levelMs[i]);
        fixedDepthM.addSeparator();
        fixedDepthM.add(levelMs[CUSTOMIZE]);
        difficultyM.add(fixedDepthM);
        //difficultyM.addSeparator();
        difficultyM.add(softLimitM);
        difficultyM.addSeparator();
        difficultyM.add(hardLimitM);

        // options
        optionsM.add(showAvailMovesM);
        optionsM.add(showLastMoveM);
        optionsM.add(showStatM);
        optionsM.addSeparator();
        optionsM.add(mirroredStartM);
        //optionsM.addSeparator();
        optionsM.add(saveSettingsM);
        optionsM.addSeparator();
        optionsM.add(saveSettingsNowM);
        optionsM.add(refreshScreenM);
        // help
        helpM.add(helpContentM);
        helpM.addSeparator();
        helpM.add(aboutMiniOthelloM);
        // add menus to the bar
        menu.add(fileM);
        menu.add(gameM);
        menu.add(difficultyM);
        menu.add(optionsM);
        menu.add(helpM);

        // Lastly initialize the menubar and add it to the frame
        menu.setFont(new Font("Dialog", Font.PLAIN, 11));
        setMenuBar(menu);

        // add text fields
        if(MiniOthello.osName.equalsIgnoreCase("Linux") &&
           MiniOthello.DISABLE_TEXT_AREA_IN_LINUX) {
            // have problems with text components in Linux (as tested in BiH 632)
            // don't know why.
            pastMoves.setVisible(false);
            infoBox.setVisible(false);
        }
        add(pastMoves);
        add(infoBox);

        // add buttons
        //startB.setVisible(true);
        add(undoB);
        add(redoB);
        add(undoAllB);
        add(redoAllB);
        add(startB);

        /* Final itialization */
        initNewGame();
        frame = this;
    }

    /* Paint method, Need to override to draw the board (lines, buttons, etc). */
    public void paint(Graphics g) {
        // some funny null pointer stuff
        if(screen == null)
            screen = createImage(getSize().width, getSize().height);
        if(g2 == null) {
            g2 = (Graphics2D) (screen.getGraphics());
            g2.setBackground(Util.WINDOW_BACKGROUND);
        }
        if(g == null)
            g = getGraphics();
        // Pre-compute some information for what needs to be drawn
        int nMoves = 0;
        boolean[] lm = new boolean[64];
        nMoves = board.findLegalMoves(lm);
        // turn on rendering
        g2.setRenderingHint(RenderingHints.KEY_ANTIALIASING,
                            RenderingHints.VALUE_ANTIALIAS_ON);
        // draw background (image sometimes looks no very clean without this step
        g2.setColor(Util.WINDOW_BACKGROUND);
        g2.fillRect(0, 0, getSize().width, getSize().height);
        // draw the top seperation line
        g2.setColor(Util.WINDOW_BACKGROUND);
        g2.draw3DRect(3, DRAWING_AREA_Y_OFFSET, WINDOW_SIZE_X - 7, 1, false);
        // draw board and player info
        updateButtons(nMoves);
        updateTextAreas();
        drawBoard();
        drawDiscs();
        if(game.showAvailMoves)
            drawAvailMoves(lm);
        if(game.showLastMove)
            drawLastMove();
        drawPlayerStat();
        updateMenu();
        // draw every in the buffer onto the frame
        if(g != null) // only to avoid null pointer during initialization of the frame
            g.drawImage(screen, 0, 0, this);
    }

    /* Drawing methods */
    public void drawBoard() {
        // Draw frame
        g2.setColor(Util.BOARD_FRAME);
        g2.fill3DRect(BOARD_FRAME_X_OFFSET, BOARD_FRAME_Y_OFFSET,
                      BOARD_FRAME_SIZE, BOARD_FRAME_SIZE, true);
        // draw board
        g2.setColor(Util.BOARD_BACKGROUND);
        g2.fillRect(BOARD_X_OFFSET, BOARD_Y_OFFSET, BOARD_SIZE, BOARD_SIZE);
        // draw grid
        g2.setColor(Util.BOARD_GRID);
        for(int i=BOARD_Y_OFFSET+BOARD_CELL_SIZE; i<=BOARD_Y_OFFSET+7*BOARD_CELL_SIZE; i+=BOARD_CELL_SIZE) {
            g2.drawLine(BOARD_X_OFFSET + 1, i - 1, BOARD_X_OFFSET + BOARD_SIZE - 2, i - 1);
            g2.drawLine(BOARD_X_OFFSET + 1, i, BOARD_X_OFFSET + BOARD_SIZE - 2, i);
        }
        for(int i=BOARD_X_OFFSET+BOARD_CELL_SIZE; i<=BOARD_X_OFFSET+7*BOARD_CELL_SIZE; i+=BOARD_CELL_SIZE) {
            g2.drawLine(i - 1, BOARD_Y_OFFSET + 1, i - 1, BOARD_Y_OFFSET + BOARD_SIZE - 2);
            g2.drawLine(i, BOARD_Y_OFFSET + 1, i, BOARD_Y_OFFSET + BOARD_SIZE - 2);
        }
        g2.drawRect(BOARD_X_OFFSET, BOARD_Y_OFFSET, BOARD_SIZE - 1, BOARD_SIZE - 1);
        g2.setColor(Util.FOUR_DOTS);
        drawDisc(BOARD_X_OFFSET + 2*BOARD_CELL_SIZE, BOARD_Y_OFFSET + 2*BOARD_CELL_SIZE, GRAY_DOT_RADIUS);
        drawDisc(BOARD_X_OFFSET + 2*BOARD_CELL_SIZE, BOARD_Y_OFFSET + 6*BOARD_CELL_SIZE, GRAY_DOT_RADIUS);
        drawDisc(BOARD_X_OFFSET + 6*BOARD_CELL_SIZE, BOARD_Y_OFFSET + 2*BOARD_CELL_SIZE, GRAY_DOT_RADIUS);
        drawDisc(BOARD_X_OFFSET + 6*BOARD_CELL_SIZE, BOARD_Y_OFFSET + 6*BOARD_CELL_SIZE, GRAY_DOT_RADIUS);

        //draw the number/letter label
        g2.setFont(new Font("Dialog", Font.BOLD, 24));
        for(int i=0; i<8; i++) {
            g2.drawString("" + (char)((byte)'A' + i),
                          BOARD_X_OFFSET + i*BOARD_CELL_SIZE + 20,
                          BOARD_FRAME_Y_OFFSET + 33);
            g2.drawString("" + (char)((byte)'1' + i),
                          BOARD_FRAME_X_OFFSET + 16,
                          BOARD_Y_OFFSET + i*BOARD_CELL_SIZE + 37);
        }
    }

    public void drawDiscs(boolean b) {
        for(int y=0; y<8; y++)
            for(int x=0; x<8; x++) {
                int piece = board.getPiece(x, y);
                if(piece == Board.BLACK)
                    drawDisc(x, y, Util.BLACK);
                else if(piece == Board.WHITE)
                    drawDisc(x, y, Util.WHITE);
                else if(b)
                    drawDisc(x, y, Util.BOARD_BACKGROUND);
            }
    }

    public void drawDiscs() {
        for(int y=0; y<8; y++)
            for(int x=0; x<8; x++) {
                int piece = board.getPiece(x, y);
                if(piece == Board.BLACK)
                    drawDisc(x, y, Util.BLACK);
                else if(piece == Board.WHITE)
                    drawDisc(x, y, Util.WHITE);
            }
    }

    public void drawPlayerStat() {
        g2.setColor(board.whoseTurn() == Board.BLACK? Color.black : Color.white);
        g2.fillOval(BOARD_FRAME_RIGHT_POS + 60, BOARD_FRAME_Y_OFFSET + 5, 32, 32);
        g2.setColor(Color.black);
        g2.setFont(new Font("Dialog", Font.PLAIN, 18));
        g2.drawString("to Move", BOARD_FRAME_RIGHT_POS + 103,
                      BOARD_FRAME_Y_OFFSET + 30);
        g2.setFont(new Font("Dialog", Font.PLAIN, 18));
        g2.drawString("Black:", BOARD_FRAME_RIGHT_POS + 60, BOARD_FRAME_Y_OFFSET + 62);
        g2.drawString("White:", BOARD_FRAME_RIGHT_POS + 60, BOARD_FRAME_Y_OFFSET + 85);
        g2.drawString(board.getDiscCount(Board.BLACK) + " discs",
                      BOARD_FRAME_RIGHT_POS + 120, BOARD_FRAME_Y_OFFSET + 62);
        g2.drawString(board.getDiscCount(Board.WHITE) + " discs",
                      BOARD_FRAME_RIGHT_POS + 120, BOARD_FRAME_Y_OFFSET + 85);
    }

    public void updateButtons(int nMoves) {
        // update the buttons condition
        if(undoB != null) undoB.setEnabled(board.canUndo());
        if(redoB != null) redoB.setEnabled(board.canRedo());
        if(undoAllB != null) undoAllB.setEnabled(board.canUndo());
        if(redoAllB != null) redoAllB.setEnabled(board.canRedo());
        if(startB != null) {
            if(game.hasEnded()) {
                // maybe should be disabled, since I myself sometimes click it
                // by mistake (wanted to click PASS)
                startB.setEnabled(false);
                startB.setSize(BIG_BUTTON_WIDTH, BIG_BUTTON_HEIGHT);
                if(MiniOthello.osName.equalsIgnoreCase("Linux"))
                    startB.setFont(new Font("Dialog", Font.BOLD, 15));
                else
                    startB.setFont(new Font("Dialog", Font.BOLD, 20));
                startB.setLabel("Game Over");
            }
            else if (game.player[0] == Game.COMPUTER_PLAYER &&
                     game.player[1] == Game.COMPUTER_PLAYER &&
                     game.isSelfPlaying()) {  // disabled
                startB.setEnabled(true);
                startB.setSize(BIG_BUTTON_WIDTH, BIG_BUTTON_HEIGHT);
                startB.setFont(new Font("Dialog", Font.BOLD, 28));
                startB.setLabel("Stop");
            }
            else if(game.player[game.getCurrPlayer()] == Game.COMPUTER_PLAYER &&
                    game.isWaitingToStart() && !game.hasEnded()) {
                startB.setEnabled(true);
                startB.setSize(BIG_BUTTON_WIDTH, BIG_BUTTON_HEIGHT);
                startB.setFont(new Font("Dialog", Font.BOLD, 28));
                startB.setLabel("Start");
            }
            else if (game.player[game.getCurrPlayer()] == Game.HUMAN_PLAYER &&
                     nMoves == 0 && !game.hasEnded()) {
                startB.setEnabled(true);
                startB.setSize(BIG_BUTTON_WIDTH, BIG_BUTTON_HEIGHT);
                startB.setFont(new Font("Dialog", Font.BOLD, 28));
                startB.setLabel("PASS");
            }
            else if(!game.canPlay() &&
                    (startB.getLabel().equals("Stop") || startB.getLabel().equals("Stopping..."))) {
                if(MiniOthello.osName.equalsIgnoreCase("Linux"))
                    startB.setFont(new Font("Dialog", Font.BOLD, 15));
                else
                    startB.setFont(new Font("Dialog", Font.BOLD, 22));
                startB.setSize(BIG_BUTTON_WIDTH, BIG_BUTTON_HEIGHT);
                startB.setLabel("Stopping...");
                startB.setEnabled(false);
            }
            else {
                //startB.setFont(new Font("Dialog", Font.BOLD, 28));
                startB.setEnabled(false);
                startB.setLabel("");
                startB.setSize(0,0);
            }
        }
    }

    public void updateMenu() {
        // avoiding null pointer during init
        if(comPlayerM == null || comPlayerM[COM_COM] == null) return;
        if(levelMs == null || levelMs[CUSTOMIZE] == null) return;
        // Player role menu
        comPlayerM[HUM_COM].setState(game.player[0] == Game.HUMAN_PLAYER &&
                                     game.player[1] == Game.COMPUTER_PLAYER);
        comPlayerM[COM_HUM].setState(game.player[0] == Game.COMPUTER_PLAYER &&
                                     game.player[1] == Game.HUMAN_PLAYER);
        comPlayerM[HUM_HUM].setState(game.player[0] == Game.HUMAN_PLAYER &&
                                     game.player[1] == Game.HUMAN_PLAYER);
        comPlayerM[COM_COM].setState(game.player[0] == Game.COMPUTER_PLAYER &&
                                     game.player[1] == Game.COMPUTER_PLAYER);
        // Difficulty level menu
        boolean tempbool = false;
        boolean tempbool2;
        tempbool2 = game.searchDepth == 1 && game.endGameDepth == 8;
        tempbool |= tempbool2;
        levelMs[STOOGE].setState(tempbool2);
        tempbool2 = game.searchDepth == 2 && game.endGameDepth == 10;
        tempbool |= tempbool2;
        levelMs[MINDLESS].setState(tempbool2);
        tempbool2 = game.searchDepth == 4 && game.endGameDepth == 12;
        tempbool |= tempbool2;
        levelMs[NOVICE].setState(tempbool2);
        tempbool2 = game.searchDepth == 6 && game.endGameDepth == 14;
        tempbool |= tempbool2;
        levelMs[BEGINNER].setState(tempbool2);
        tempbool2 = game.searchDepth == 8 && game.endGameDepth == 16;
        tempbool |= tempbool2;
        levelMs[AMATEUR].setState(tempbool2);
        tempbool2 = game.searchDepth == 8 && game.endGameDepth == 18;
        tempbool |= tempbool2;
        levelMs[AMATEUR_P].setState(tempbool2);
        tempbool2 = game.searchDepth == 10 && game.endGameDepth == 18;
        tempbool |= tempbool2;
        levelMs[EXPERIENCED].setState(tempbool2);
        tempbool2 = game.searchDepth == 10 && game.endGameDepth == 20;
        tempbool |= tempbool2;
        levelMs[EXPERIENCED_P].setState(tempbool2);
        tempbool2 = game.searchDepth == 12 && game.endGameDepth == 22;
        tempbool |= tempbool2;
        levelMs[ADVANCED].setState(tempbool2);
        // lastly the CUSTOMIZE menu item
        levelMs[CUSTOMIZE].setState(!tempbool && !game.timeDependent);
        // .. and flexible-time itme
        flexibleTimeM.setState(game.smartPlay);

        // update time-dependent sub-menus
        tempbool = false;
        for(int i=0; i<sItems.length-1; i++) {
            tempbool2 = game.timeDependent && game.softTimeLimit == SOFT_LIMIT[i];
            tempbool |= tempbool2;
            sItems[i].setState(tempbool2);
        }
        sItems[sItems.length-1].setState(!tempbool && game.timeDependent);

        for(int i=0; i<hItems.length; i++) {
            hItems[i].setState(game.hardTimeLimit == HARD_LIMIT[i]);
        }

        // update the options menu
        showAvailMovesM.setState(game.showAvailMoves);
        showLastMoveM.setState(game.showLastMove);
        showStatM.setState(game.showStat);
        mirroredStartM.setState(game.mirroredStart);
        saveSettingsM.setState(game.saveSettings);
    }

    public void updateTextAreas() {
        String s = "";
        int nMoves = board.getM();
        for(int i=1; i<=nMoves; i++) {
            if((i & 1) == 0)
                s += "\t";
            else if(i != 1)
                s += "\n";
            s += (i < 10? "  " : " ") + i + ". ";
            int mv = board.getRecordedMove(i);
            if(mv == Board.PASS)
                s += "Pass";
            else
                s += "" + (char)((mv & 7)+(byte)'A') + ((mv >> 3) + 1);
        }
        if(pastMoves != null) {
            if(!gameStarting || board.getM() > 0) {
                pastMoves.setText("");
                pastMoves.append(s);
                gameStarting = false;
            }
        }
    }

    public void drawAvailMoves(boolean[] lm) {
        int length = lm.length;
        if(length != 64)
            System.out.println("Error: incorrect boolean array size in drawAvailMoves");
        for(int i=0; i<length; i++)
            if(lm[i]) {
                int centx = (i & 7) * BOARD_CELL_SIZE + BOARD_X_OFFSET + BOARD_CELL_SIZE / 2;
                int centy = (i >> 3) * BOARD_CELL_SIZE + BOARD_Y_OFFSET + BOARD_CELL_SIZE / 2;
                drawCross(centx, centy);
            }
    }

    public void drawLastMove() { // x, y are 0 - 7
        int lastMove = board.getLastMove();
        if(lastMove == Board.PASS)
            return;
        Point p = Board.conv1to2(lastMove);
        int x = p.x, y = p.y;
        g2.setColor(Util.LAST_MOVE_DOT);
        drawDisc(BOARD_X_OFFSET + (BLUE_DOT_RADIUS + 2) + x * BOARD_CELL_SIZE,
                 BOARD_Y_OFFSET - (BLUE_DOT_RADIUS + 2) + (y+1) * BOARD_CELL_SIZE,
                 BLUE_DOT_RADIUS);
    }

/*
    // No longer used --  replaced with calling paint(Graphics)
    public void drawMinGraphics() { // not to do repaint, avoid flashing
        drawBoard();
        drawDiscs();
        g2.setColor(Util.WINDOW_BACKGROUND);
        g2.fillRect(BOARD_FRAME_RIGHT_POS + 30, BOARD_FRAME_Y_OFFSET,
                    300, 200);
        drawPlayerStat();
        updateMenu();
        updateuttons(board.findLegalMoves(new boolean[64]));
        g.drawImage(screen, 0, 0, this);
    }
 */


    /* helper gui methoeds */
    public void drawDisc(int centx, int centy, int radius) {
        g2.fillOval(centx-radius, centy-radius, 2*radius, 2*radius);
    }

    public void drawDisc(int x, int y, Color color) {
        // x, y are 0 - 7 on boar coordinates, color is Util.BLACK or Util.WHITE
        g2.setColor(color);
        drawDisc(BOARD_X_OFFSET + BOARD_CELL_SIZE / 2 + x * BOARD_CELL_SIZE,
                 BOARD_Y_OFFSET + BOARD_CELL_SIZE / 2 + y * BOARD_CELL_SIZE,
                 DISC_RADIUS);
    }

    public void drawCross(int centx, int centy) {
        int[] x1 = {centx-CROSS_DIM_L, centx-CROSS_DIM_S, centx+CROSS_DIM_L, centx+CROSS_DIM_S};
        int[] y1 = {centy-CROSS_DIM_S, centy-CROSS_DIM_L, centy+CROSS_DIM_S, centy+CROSS_DIM_L};
        int[] x2 = {centx-CROSS_DIM_S, centx-CROSS_DIM_L, centx+CROSS_DIM_S, centx+CROSS_DIM_L};
        int[] y2 = {centy+CROSS_DIM_L, centy+CROSS_DIM_S, centy-CROSS_DIM_L, centy-CROSS_DIM_S};
        g2.setColor(Util.LEGAL_MOVE_CROSS);
        g2.drawPolygon(x1, y1, 4);
        g2.drawPolygon(x2, y2, 4);
    }


    /* Not want user to click undo/redo when computer is thinking */
    // Not used. Found other way round.
    public void disableButtons() {
        undoB.setEnabled(false);
        redoB.setEnabled(false);
        undoAllB.setEnabled(false);
        redoAllB.setEnabled(false);
    }

    /* Change of game states */
    public void initNewGame() {
        gameStarting = true;
        board = game.getBoard();
        gd = new GameDriver(frame, game, board);
        pastMoves.setText("--- New Game ---\nBlack: " +
                        (game.player[0] == Game.HUMAN_PLAYER? "Human\n" : "Computer\n") +
                        "White: " + (game.player[1] == Game.HUMAN_PLAYER? "Human\n" : "Computer\n"));
        if(game.player[0] == Game.COMPUTER_PLAYER || game.player[1] == Game.COMPUTER_PLAYER) {
            if(game.timeDependent) {
                if(game.softTimeLimit <= game.hardTimeLimit || game.hardTimeLimit == 0) {
                    pastMoves.append("Time: " + game.softTimeLimit/1000 + " sec. per Move");
                    if(game.smartPlay)
                        pastMoves.append(" (flexible)\n");
                    else
                        pastMoves.append(" (fixed)\n");
                }
                else
                    pastMoves.append("Time (Hard limit): " + game.hardTimeLimit/1000 + " sec per Move\n");
            }
            else
                pastMoves.append("Search: " + game.searchDepth +
                                 "-ply, (" + game.endGameDepth + " End)\n");
        }
        pastMoves.append("Board Mirrored: " + (game.mirroredStart? "Yes" : "No"));
        infoBox.setText("Message");
        paint(getGraphics());
    }

    // used when loading a game
    public void setBoard(Board b) {
        board = b;
    }

    // Show the "About" info
    public void showAbout() {
        String msg = "";
        msg += "*** MiniOthello GUI Version ***\n" +
            "Written by: Yunpeng Li (yl@middlebury.edu)\n" +
            "GUI version: " + MiniOthello.VERSION + "\n" +
            "Engine version: " + game.engineVer + "\n" +
            "(Grey menu item = Not yet implemented. Sorry... :)";
        InfoDialog d = new InfoDialog(this, "About...", false, msg, 20, 60,
                                      new Font("Dialog", Font.PLAIN, 14));
        //d.setLocation(getLocation().x + 200, getLocation().y + 200);
        //d.setSize(400, 180);
        d.show();
    }

    /* Access methods */
    public TextArea getPastMoves() {
        return pastMoves;
    }

    public TextArea getInfoBox() {
        return infoBox;
    }

    public Game getGame() {
        return game;
    }

    /***** Exit ******/
    public void exit() {
        if(game.saveSettings) {
            game.saveSettings();
        }
        game.terminate();
        dispose();
        System.exit(0);
    }


    //***************************************************************
    // BoardFrame extends Frame, which incorporates the following methods.

    /* Process user's click to close the window. */
    protected void processWindowEvent(WindowEvent e) {
        if (e.getID() == e.WINDOW_CLOSING) {
            exit();
        }
    }

    /* Process action events. Nothing to do. */
    protected void processActionEvent(AWTEvent e) {
        //System.out.println("AWTEvent: " + e.getID()); // info
    }

    /* Process key events. Nothing to do. */
    protected void processKeyEvent(KeyEvent e) {
        //System.out.println("KeyEvent: " + e.getKeyCode()); // info
    }

    /* Process mouse motion events. */
    protected void processMouseMotionEvent(MouseEvent e) {
    }

    /* Process mouse events, for game playing with mouse click. */
    protected void processMouseEvent(MouseEvent e) {
        if(e.getID() == MouseEvent.MOUSE_PRESSED && e.getButton() == MouseEvent.BUTTON1 &&
           game.player[game.getCurrPlayer()] == Game.HUMAN_PLAYER &&
           game.canPlay()) {
            int x = (e.getX() - BOARD_X_OFFSET) / BOARD_CELL_SIZE;
            int y = (e.getY() - BOARD_Y_OFFSET) / BOARD_CELL_SIZE;
            //drawDisc(x, y, board.whoseTurn() == Board.BLACK? Util.BLACK : Util.WHITE);
            gd = new GameDriver(frame, game, board);
            gd.setState(GameDriver.PLAY_HUMAN_MOVE);
            gd.setCoord(x, y);
            gd.start();
        }
    }





    /* Method for experiementing */
    public void drawSomething() {
        if(screen == null)
            screen = createImage(getSize().width, getSize().height);
        Graphics g = screen.getGraphics();
        Graphics2D g2 = (Graphics2D)g;
        g2.setRenderingHint(RenderingHints.KEY_ANTIALIASING,
                            RenderingHints.VALUE_ANTIALIAS_ON);
        g2.setBackground(Util.WINDOW_BACKGROUND);
        g2.setColor(Util.WINDOW_BACKGROUND);
        g2.fillRect(0, 0, 700, 600);
        g2.setColor(Util.TABLE_BACKGROUND);
        //g2.fillRect(0, 0, 600, 600);
        g2.setColor(Util.BOARD_FRAME);
        g2.fill3DRect(30, 50, 540, 540, true);
        g.setColor(Util.BOARD_BACKGROUND);
        g2.fillRect(60, 60, 480, 480);
        g2.setColor(Util.BLACK);
        g2.fillOval(125, 185, 50, 50);
        g2.fillOval(65, 65, 50, 50);
        g2.fillOval(305, 245, 50, 50);
        g2.fillOval(245, 305, 50, 50);
        g2.setColor(Util.WHITE);
        g2.fillOval(185, 185, 50, 50);
        g2.fillOval(485, 485, 50, 50);
        g2.fillOval(245, 245, 50, 50);
        g2.fillOval(305, 305, 50, 50);
        g2.setColor(Util.BOARD_GRID);
        g2.drawRect(60, 60, 479, 479);
        for(int i=120; i<=480; i+=60) {
            g2.drawLine(61, i, 538, i);
            g2.drawLine(61, i-1, 538, i-1);
        }
        for(int i=120; i<=480; i+=60) {
            g2.drawLine(i, 61, i, 538);
            g2.drawLine(i-1, 61, i-1, 538);
        }
        g2.setColor(Util.FOUR_DOTS);
        g2.fillOval(175, 175, 10, 10);
        g2.fillOval(415, 175, 10, 10);
        g2.fillOval(175, 415, 10, 10);
        g2.fillOval(415, 415, 10, 10);
        //g2.drawLine(60, 230, 390, 230);
        //g2.drawLine(60, 231, 390, 231);
        //g2.drawLine(60, 250, 390, 440);
        //g.drawImage(screen, 400, 400, this);
        //g2.drawLine(600, 41, 700, 41);
        g2.setColor(Util.WINDOW_BACKGROUND);
        g2.draw3DRect(3, 42, 693, 1, false);

        getGraphics().drawImage(screen, 0, 0, this);
        //System.out.println(new String((""+0.1234567).getBytes(), 0, 5));
        DecimalFormat formatter = new DecimalFormat("#.###");
        //System.out.println(formatter.format(12.34567));

        Image icon = Toolkit.getDefaultToolkit().getImage(MiniOthello.PICTURE_PATH + "icon3.gif");
        //System.out.println("icon == null? " + (icon == null));
        //Toolkit.getDefaultToolkit().prepareImage(icon, 24, 24, this);
        setIconImage(icon);

    }


}



/*
menu = new MenuBar();
file = new Menu("File");
edit = new Menu("Edit");
//String[] fonts = getToolkit().getFontList();
//for(int i=0; i<fonts.length; i++) System.out.println(fonts[i]);
menu.setFont(new Font("Dialog", Font.PLAIN, 11));
//System.out.println(menu.getFont().getName() + ", " + menu.getFont().getFontName());
newGame = new MenuItem("New game", new MenuShortcut((int)'N'));
quit = new MenuItem("Exit");
undo = new MenuItem("Undo", new MenuShortcut((int)'B'));
showM = new CheckboxMenuItem("Show Available Moves", true);



file.add(newGame);
file.add(quit);
edit.add(undo);
edit.add(showM);
menu.add(file);
menu.add(edit);
setMenuBar(menu);
*/

/*



    // experiment vars
    private Menu file, edit;
    private MenuItem newGame, undo;
    private MenuItem quit;
    private CheckboxMenuItem showM;
*/

