/*************************************************************
 * Board.java
 *
 * Holding information of the game board
 *
 * Yunpeng Li
 *
 * */

package othelloGUI;

import java.awt.*;

public class Board {
    public static final byte EMPTY = 0, BLACK = 1, WHITE = 2; // must be the same as the C file
    public static final byte PASS = 99 - 10; // since a move will be added by 10 when passed to te C program
    public static final byte[][] DIRECTION =
        {{1, 0}, {1, 1}, {0, 1}, {-1, 1}, {-1, 0}, {-1, -1}, {0, -1}, {1, -1}};

    private byte a[][];  // stack of board array
    private byte moves[];  // sequence of moves include PASS until this point.
    private int n;  // number of actual moves made (NOT incl. PASS).
    private int m;  // number of moves incl. PASSes made so far.
    private int top;  // redo possible if m < top.
    private byte wt;  // whose turn it is to move.
    private boolean mirrored;  // whether the board is mirrored


    /* Constructors */
    public Board() {
        this(false);
    }

    public Board(boolean _mirrored) {
        a = new byte[61][64];
        moves = new byte[128];
        n = 0;
        m = 0;
        top = 0;
        wt = BLACK;
        mirrored = _mirrored;

        byte[] aa = a[0];
        for(int i=0; i<64; i++)
            aa[i] = EMPTY;
        if(mirrored) {
            aa[27] = BLACK;
            aa[28] = WHITE;
            aa[35] = WHITE;
            aa[36] = BLACK;
        }
        else {
            aa[27] = WHITE;
            aa[28] = BLACK;
            aa[35] = BLACK;
            aa[36] = WHITE;
        }
        moves[0] = PASS;  // 0th move doesn't exist, so just put PASS in it.
    }


    /* Access methods */
    // Set piece
    public void setPiece(int place, int player) {
        a[n][place] = (byte)player;
    }

    public void setPiece(int x, int y, int player) {
        setPiece(conv2to1(x, y), player);
    }

    // get piece
    public int getPiece(int place) {
        return a[n][place];
    }

    public int getPiece(int x, int y) {
        return a[n][conv2to1(x, y)];
    }

    // flip piece
    public void flipPiece(int place) {
        a[n][place] = other(a[n][place]);
    }

    public void flipPiece(int x, int y) {
        flipPiece(conv2to1(x, y));
    }

    // disc count
    public int getDiscCount(int player) {
        int count = 0;
        byte[] aa = a[n];
        for(int i=0; i<64; i++)
            if(aa[i] == player)
                count++;
        return count;
    }

    public int getTotalDiscCount() {
        return n + 4;
    }

    // recorded moves
    public int getRecordedMove(int number) {
        return moves[number];
    }

    // other board variables
    public int getN() { return n; }
    public int getM() { return m; }
    public int getTop() { return top; }
    public int whoseTurn() { return wt; }
    public boolean isMirrored() {return mirrored; }
    public int getLastMove() {return moves[m]; }


    /* Test if a move is legal or not */
    public boolean legalMove(int x, int y) {
        byte[] aa = a[n];
        byte place = conv2to1(x, y);
        if (x < 0 || x > 7 || y < 0 || y > 7)
            return false;
        if (aa[place] != EMPTY)
            return false;
        /* test for possible flips */
        boolean result = false;
        for(int dir = 0; dir < 8; dir++) {
            int dx = DIRECTION[dir][0];
            int dy = DIRECTION[dir][1];
            int tx = x + 2 * dx;
            int ty = y + 2 * dy;
            /* need to be at least 2 grids away from the edge and a oppenent piece
              adjacent in the direction */
            if (!onBoard(tx, ty) || aa[conv2to1(x + dx, y + dy)] != other(wt))
                continue;
            while (onBoard(tx, ty) && aa[conv2to1(tx, ty)] == other(wt)) {
                tx += dx;
                ty += dy;
            }
            if (onBoard(tx, ty) && aa[conv2to1(tx, ty)] == wt) {
                result = true;
                break;
            }
        }
        return result;
    }

    public boolean legalMove(int place) {
        Point p = conv1to2(place);
        return legalMove(p.x, p.y);
    }

    /* get all legal moves, store legal move bit map in boolean[] lm */
    public int findLegalMoves(boolean[] lm) {
        int result = 0;
        for(int y=0; y<8; y++) {
            for(int x=0; x<8; x++) {
                int place = conv2to1(x, y);
                lm[place] = legalMove(x, y);
                if(lm[place])
                    result++;
            }
        }
        return result;
    }

    /* test whether there is a legal move */
    public boolean hasLegalMove() {
        return findLegalMoves(new boolean[64]) > 0;
    }

    /* test whether the game should end since no one can move */
    public boolean gameOver() {
        if(hasLegalMove())
            return false;
        switchSide();
        boolean shouldEnd = !hasLegalMove();
        switchSide();
        return shouldEnd;
    }

    /* make a move, (x, y) must be a legal move */
    public void makeMove(int x, int y) {
        byte[] aa = a[n];
        byte[] a1 = a[n+1];
        byte place = conv2to1(x, y);
        int flipCount = 0;
        for(int i=0; i<64; i++)
            a1[i] = aa[i];
        for(int dir=0; dir<8; dir++) {
            int dx = DIRECTION[dir][0];
            int dy = DIRECTION[dir][1];
            int tx = x+2*dx;
            int ty = y+2*dy;
            /* need to be at least 2 grids away from the edge and a oppenent piece
                   adjacent in the direction to make flips in this direction. */
            if(!onBoard(tx, ty) || aa[conv2to1(x+dx, y+dy)] != other(wt))
                continue;
            while(onBoard(tx, ty) && aa[conv2to1(tx, ty)] == other(wt)) {
                tx += dx;
                ty += dy;
            }
            /* go back and flip the pieces if it should happen */
            if(onBoard(tx, ty) && aa[conv2to1(tx, ty)] == wt) {
                tx -= dx;
                ty -= dy;
                while(aa[conv2to1(tx, ty)] == other(wt)) {
                    a1[conv2to1(tx, ty)] = wt;
                    tx -= dx;
                    ty -= dy;
                    flipCount++;
                }
            }
        }

        if(flipCount == 0) { // played an illegal move -- should never happen
            System.err.println("ERROR: Illegal move received in makeMove()!");
            System.exit(1);
        }

        /* update the board to the next step */
        a1[place] = wt;
        wt = other(wt);
        n += 1;  // actual move pointer
        m += 1;  // total move pointer
        if(m > top || moves[m] != place) {
            top = m;  // whenever a different move is made, no 'redo' possible
            moves[m] = place;  // record this move
        }
    }

    public void makeMove(int place) {
        Point p = conv1to2(place);
        makeMove(p.x, p.y);
    }

    /* make a pass */
    public void makePass() {
        m += 1;
        wt = other(wt);
        if(m > top || moves[m] != PASS) {
            top = m;
            moves[m] = PASS;
        }
    }

    /* Should only be used when need to determine if a game will end
    should always undo it afterwards by callin it gain */
    public void switchSide() {
        wt = other(wt);
    }


    /* try to undo a move -- return true if it can be done, false otherwise */
    public boolean undoMove() {
      if(m == 0)  // no move to undo
        return false;
      if(moves[m] == PASS) {  // undo a pass
        m -= 1;
        wt = other(wt);
      }
      else {  // undo an actual move
        m -= 1;
        n -= 1;
        wt = other(wt);
      }
      return true;
    }

    // undo that doesn't allow redo
    public boolean undoMoveIrreversible() {
      if(m == 0)  // no move to undo
        return false;
      if(moves[m] == PASS) {  // undo a pass
        m -= 1;
        wt = other(wt);
      }
      else {  // undo an actual move
        m -= 1;
        n -= 1;
        wt = other(wt);
      }
      top = m;
      return true;
    }

    /* try to redo a move -- return true if redo is possible, false otherwise */
    public boolean redoMove() {
      if(m == top)  // no move to redo
        return false;
      if(moves[m+1] == PASS) {  // redo a pass
        m += 1;
        wt = other(wt);
      }
      else {  // redo an actual move
        m += 1;
        n += 1;
        wt = other(wt);
      }
      return true;
    }

    /* undo all moves -- return the number of moves one can undo (incl. PASS) */
    public int undoAll() {
        int count = 0;
        while(undoMove())
            count++;
        return count;
    }

    /* redo all moves -- return the number of moves one can redo (incl. PASS) */
    public int redoAll() {
        int count = 0;
        while(redoMove())
            count++;
        return count;
    }

    /* try to undo a certain number of moves -- return the actual number of
     moves one can undo */
    public int undo(int number) {
        int count = 0;
        while(count < number && undoMove())
            count++;
        return count;
    }

    /* try to redo a certain number of moves -- return the actual number of
     moves one can redo */
    public int redo(int number) {
        int count = 0;
        while(count < number && redoMove())
            count++;
        return count;
    }

    /* get the string encoding of the current board state to be passed to AI */
    public String getStringEncoding() {
        String result = "";
        byte[] aa = a[n];
        for(int i=0; i<64; i++)
            result += aa[i];
        result += wt;
        result += 10 + moves[m];
        if(result.length() != 67) { // debug
            System.out.println("ERROR: length == " + result.length());
            System.exit(1);
        }
        return result;
    }

    /* Auxiliary methods */
    public boolean canUndo() { return m > 0; }
    public boolean canRedo() { return m < top; }


    /* Utility methods */
    public static byte conv2to1(int x, int y) {
        return (byte)(x + (y << 3));
    }

    public static Point conv1to2(int place) {
        return new Point(place & 7, place >> 3);
    }

    public static byte other(int player) {
        return (byte)(BLACK + WHITE - player);
    }

    public static boolean onBoard(int x, int y) {
        return x >= 0 && x < 8 && y >= 0 && y < 8;
    }


    /***** For debugging use *******/
    /* print out the board */
    public void print() {
        boolean[] legalMoves = new boolean[64];
        findLegalMoves(legalMoves);
        byte[] aa = a[n];
        byte lastMove = moves[m];
        byte place;
        String s;
        int nb, nw;
        nb = getDiscCount(BLACK);
        nw = getDiscCount(WHITE);
        System.out.print("\n     A B C D E F G H\n");
        for(int y=0; y<8; y++) {
            if(lastMove == 8*y)
                System.out.print("  " + (y+1) +" (");
            else
                System.out.print("  " + (y+1) + "  ");
            for(int x=0; x<8; x++) {
                place = conv2to1(x, y);
                if(aa[place] == BLACK)
                    s = "X";
                else if(aa[place] == WHITE)
                    s = "O";
                else if(legalMoves[place])
                    s = "+";
                else
                    s = ".";
                if(place+1 == lastMove && x != 7)
                    System.out.print(s + "(");
                else if(place == lastMove)
                    System.out.print(s + ")");
                else
                    System.out.print(s + " ");
            }
            // printf(" %d", y+1);
            if(y == 0)
                System.out.print("\tBlack: " + nb);
            else if(y == 1)
                System.out.print("\tWhite: " + nw);
            System.out.print("\n");
        }
        // printf("   A B C D E F G H\n");
        System.out.print("\n");
    }

}
