/* mobility.cpp

  Generates the 3^8 (row/column/diag) mobility lookup table mob_table.hpp 
  and provides routines for initialize the 2^16 table.
  Use 2^16-entry table in the memory for more convenient (faster) access
  
  */

#include <stdio.h>
#include <stdlib.h>
#include <assert.h>

/* Binary-based (4-based) mob table: Two bits per square, 1=01=B, 2=10=W, 0=00=E.
  Used for bitmap indexing. */
unsigned short mobTableBin[(1<<16)];

/* Initialize the binary table from the tertiary one.
  Run once only each execution of the program*/
void initMobTableBinFrom(unsigned short *mt3) {
  unsigned short *mt = mobTableBin;
  for(int i=0; i<6561; i++) {
    // convert 3-based table index to 4-based ones
    int index = 0, k = i, nd = 0; // index is a bitmap index
    while(k > 0) {
      int q = k / 3;
      int r = k - 3 * q;
      k = q;
      index |= r << (nd++ << 1);
    }
    mt[index] = mt3[i];
  }
}

/* Initialized and print out the 3-based mobility table as a .hpp file.
  Only do it once and for all. */
void main_genMobTable() { // rename to main() to generate the file
  FILE *fp = fopen("mob_table.hpp", "w");
  if(fp == NULL) {
    fprintf(stderr, "Unable to open file mob_table.hpp for writing\n");
    exit(1);
  }
  fprintf(fp, "\
/*****************************************\n\
  mob_table.hpp -- this file is automatically generated from mobility.cpp\n\
  \n\
  3-based row/column/diag mobility lookup table\n\
*/\n\n");
  fprintf(fp, "unsigned short mobTable[] = {\n");
  for(int i=0; i<6561; i++) {
    int row = 0, k = i, nd = 0; // row is the config of a row using bitmap index (16-bit)
    char comments[11];
    comments[0] = comments[1] = '/'; comments[10] = (char)0;
    for(int j=2; j<10; j++)
      comments[j] = '.';
    while(k > 0) {
      int q = k / 3;
      int r = k - 3 * q;
      k = q;
      row |= r << (nd << 1);
      comments[2+nd++] = r==0? '.' : r==1? 'X' : 'O';
    }
    int mob = 0;  // 16 bits: 8 low bits for black, 8 high bits for white
    for(int j=0; j<8; j++) {
      if(!(row & (3 << (j << 1)))) { // square is empty
        // Black flip
        // flip right?
        int k = j + 1;
        while(k < 8 && ((row & (3 << (k << 1))) >> (k << 1)) == 2)
          k++;
        if(k < 8 && k > j+1 && ((row & (3 << (k << 1))) >> (k << 1)) == 1)
          mob |= 1 << j;  // j position can flip right
        // flip left?
        k = j - 1;
        while(k >= 0 && ((row & (3 << (k << 1))) >> (k << 1)) == 2)
          k--;
        if(k >= 0 && k < j-1 && ((row & (3 << (k << 1))) >> (k << 1)) == 1)
          mob |= 1 << j;  // j position can flip left
          
        // White flip
        // flip right?
        k = j + 1;
        while(k < 8 && ((row & (3 << (k << 1))) >> (k << 1)) == 1)
          k++;
        if(k < 8 && k > j+1 && ((row & (3 << (k << 1))) >> (k << 1)) == 2)
          mob |= (1 << j) << 8;  // j position can flip right
        // flip left?
        k = j - 1;
        while(k >= 0 && ((row & (3 << (k << 1))) >> (k << 1)) == 1)
          k--;
        if(k >= 0 && k < j-1 && ((row & (3 << (k << 1))) >> (k << 1)) == 2)
          mob |= (1 << j) << 8;  // j position can flip left
      }
    }
    fprintf(fp, "  0x%04x,     %s\n", mob, comments);
  }
  fprintf(fp, "};\n");
}

/*** Code to computer configEntryMap ***/

void computeConfigEntryMap() {
    FILE *fp = fopen("tempout.txt", "w");
    assert(fp != NULL);
    for(int i=10; i<81; i+=9) {
        for(int j=0; j<8; j++) {
            int r = (i+j) / 9 - 1;
            int c = (i+j) % 9 - 1;
            assert(r >= 0 && c >= 0);
            int d1 = 7 + c - r;
            int d2 = c + r;
            int index = r | ((c+8) << 6) | ((d1+16) << 12) | ((d2+31) << 18);
            fprintf(fp, "0x%06x, ", index);
        }
        fprintf(fp, "\n");
    }
}



/*************** Testing / Run ***************/
/*
int main(int argc, char **argv) {
  genMobTable();
  return 0;
}
*/


