/****
  Minimax.hpp
  
  */
  
#ifndef _MINIMAX_HPP_
#define _MINIMAX_HPP_

#include <time.h>
#include <assert.h>
#include "board.hpp"
#include "endgamecx.h"

#define MAX_INT 2147483647
#define MIN_INT -2147483648
#define LARGE_FLOAT 1.0e35
#define SMALL_FLOAT -1.0e35
#define HUGE_FLOAT 2.0e38
#define TINY_FLOAT -2.0e38

#define HARD_WIRED_FIRST_MOVE 37  // F-5 position
#define HARD_WIRED_FIRST_MOVE_BOARD_FLIPPED 34  // C-5

/* --- Evaluation parameters --- */
/* Manually given weight -- ideally should be automatically learnt */
#define DISC_WEIGHT_TEST -0.28  // was used when testing (no longer used)
#define DISC_WEIGHT_STAGE_1 -0.28
#define DISC_WEIGHT_STAGE_2 -0.10
#define DISC_WEIGHT_STAGE_3 -0.05
#define DISC_WEIGHT_STAGE_4  0.001
#define STAGE_1_END    24
#define STAGE_2_START  30
#define STAGE_2_END    38
#define STAGE_3_START  42
#define STAGE_3_END    48
#define STAGE_4_START  50

// computer stop give negative weight to disc difference when oppDisc - selfDisc > this value
/* Has no effect when playing against a strong player, but might help win more against 
  bad players. To disable this criteria, simply give it a big value, e.g. 1000. */
#define DISC_DIFF_THRESHOLD 20  // estimate

/* Next two boolean variable should not be changed unless there is clear reason */
#define USE_MOBILITY_RATIO 1    // default 1
#define USE_DISC_RATIO 0        // default 0

/* Whether given small negative value to squares adjacent to corners */
#define COUNT_FOR_NEAR_TO_CORNER_EDGE 1

/* More coefficients -- ideally should all be automatically learnt */
#define ESTIMATED_CORNER_WORTH 10.0  // wild estimate
#define NEAR_TO_CORNER_NEG_DIAG 0.5
#define NEAR_TO_CORNER_NEG_EDGE 0.1
#define NEAR_TO_CORNER_NEG_DIAG_COEFF 0.01
#define NEAR_TO_CORNER_NEG_EDGE_COEFF 0.0001  // not sure if it really should be accounted for
#define DENOMINATOR_EXTRA_DOF 0.8  // avoiding divide by 0
#define RANDOM_FACTOR_CONSTANT 0.05

/* Multi-probability Cutoff (experimental) */
// Current combination tends to be conservative (cut off fewer)
#define USE_MPC 0  // seems not quite safe -- more accurate evaluation function needed!

#define MPC_THRESHOLD 6.0  // a guess (will be divided by ordering search depth when used)
#define MIN_DEPTH_FOR_MPC 6
#define MIN_EMPTIES_FOR_MPC 25
#define MIN_LEGAL_MOVES_FOR_MPC 3
#define TOP_VALUE_LOWER_BOUND -1.5
#define TOP_VALUE_UPPER_BOUND 1.5

/* Quiescence search */
/* Not quite mature and almost doubles running time if used. However, it should 
  be done in order to give more accurate evaluation and avoid big blunders. */
#define DO_USE_QUIESCENCE_SEARCH  1
#define QUIESCENCE_START 26

/* Move ordering Part */
// need to be experimented for best values
#define USE_MOVE_ORDERING 1 // proved to be very necessary

#define MIN_DEPTH_FOR_ORDERING 6  // min depth to use ordering
#define MAX_ORDERING_SEARCH_DEPTH 8 // max depth of an ordering search
#define ORDERING_SEARCH_DEPTH_OFFSET 4 // how many plies fewer is ordering search than normal search
// Following two marcos are alreday experimented for optimal values, (12, 10) also good
#define MIN_EMPTIES_FOR_ORDERING 14 
#define MIN_EMPTIES_FOR_EVALUATION 12


/* test/debug */
#define DEBUG_MINIMAX 0
#define AB_DEBUG 0
#define DEEP_DEBUG 0
#define CURRENT_DEBUG 0
#define CURRENT_DEBUG2 0

/* Whether or not use WDL search to secure a win/draw if can be done. Draw back is 
  that this takes extra time and may decrease winning margin. Default is 1 */
#define USE_WDL_SEARCH 1

/* New end of game solver */
// Uses a mover ordering scheme experimentally determined by experts. (worst2best[64])
// It is experimentally proven faster than using the minimax core for mid game, but
// still much slower than the special end game solver from the web.
#define USE_NEW_END_GAME_SOLVER 1

/* ---- produce some statistics --- */
#define MPC_FEEDBACK 0

// actually counts also nodes searched and positions evaluated
#define COUNT_PRUNING 1

extern char searchDepth;
extern char originalSearchDepth;
extern char bruteForceDepth; // for approaching the end of game.
extern char mpcDepth;
extern bool winLarge;
extern bool loseSmall;
extern char randomnessLevel;
extern bool useAndersson;  // use Andersson's sophisticated end game solver

extern bool boardFlipped;
extern bool showDots;  // simply for output
extern bool showStatistics;

// for the new end of game solver
struct EmptyList {
  EmptyList *prev;
  EmptyList *next;
  char square;
};


char getMinimaxMove(Board *board, bool *lm);
float getMin(char *a, char lastMove, char color, char depth, char passes, 
             char prevmaxDOF, char prevminDOF, //Mobility in previous getMax and getMin;
             char selfPieces, char oppPieces, float alpha, float beta);
float getMax(char *a, char lastMove, char color, char depth, char passes, 
             char prevmaxDOF, char prevminDOF, 
             char selfPieces, char oppPieces, float alpha, float beta);

void copyBoardArray(char *to, char *from);
bool canFlip(char *a, char place, char dir, char color, char oppcolor);
bool legalMove(char *a, char color, char place);
char findLegalMoves(char *a, char color, char *legalMoves);
char findLegalMoves(char *a, char color);
char tryMove(char *a, char color, char place, char *flipped);
inline void undo(char *a, char color, char place, char *flipped, char nFlipped);

int evaluateEndGame(char selfPieces, char oppPieces);
int evaluateEndGameWDL(char selfPieces, char oppPieces);
float evaluateBoard(char *a, char forWhom, char whoseTurn, char prevmaxDOF, 
                    char prevminDOF, char selfPieces, char oppPieces);

void printBoardArray(char *a);
int strongEndGameSolve(char *a, char color, char selfPieces, char oppPieces, 
                      char prevNotPass, float alpha, float beta);

// For faster exact solve end of game
inline char flipEnd(char *placePointer, char dir, char color, char oppcolor, 
                char *flipped, char cumulativeFlips);
char tryMoveEnd(char *a, char color, char place, char *flipped, EmptyList *currNode);
inline void undo_end(char *a, char color, char place, char *flipped, char nFlipped,
                    EmptyList *currNode);
char getExactMove(char *a, char color, char selfPieces, char oppPieces, char *legalMoves,
                  char nLegalMoves);
float getMin_exact(char *a, char lastMove, char color, char selfPieces, char oppPieces, 
                  float alpha, float beta);
float getMax_exact(char *a, char lastMove, char color, char selfPieces, char oppPieces, 
                  float alpha, float beta);
inline int evaluateEndGame_exact(char selfPieces, char oppPieces);

#endif
