/*******
  minimax.cpp
  This is the AI minimax search and evaluation function for Othello.
  
  Note: Andersson's end-of-game solver is very good at solving end of game 
   -- about 1-2 steps more than my simple minimax. 
  */

#include <math.h>
#include "minimax.hpp"

//#ifndef NDEBUG
//#define NDEBUG  // uncomment to disable 'assert()' debug
//#endif

static char mmBoard[64];
static char base;
static unsigned int countPruning;
static unsigned int countSearching, countEval;
static bool quiescenceSearch;
static float inEndGame;
static float bestValue_global;

static float extra;  // to avoid zero denominator
static float cornerValue;
static float discWeight;
static float near2cornerEdgeCoeff;
static float near2cornerDiagCoeff;

static const char directions[] = {1, -1, 7, -7, 8, -8, 9, -9};
// Indicate towards what directions a disc played into a certain place can make flipped
static const uchar dirmask[] = {
  81, 81, 87, 87, 87, 87, 22, 22,
  81, 81, 87, 87, 87, 87, 22, 22,
  121,121,255,255,255,255,182,182,
  121,121,255,255,255,255,182,182,
  121,121,255,255,255,255,182,182,
  121,121,255,255,255,255,182,182,
  41, 41, 171,171,171,171,162,162,
  41, 41, 171,171,171,171,162,162
};
// Bound tables for different directions
static const bool boundTables[3][64] = {
 {1, 1, 1, 1, 1, 1, 1, 1,
  1, 0, 0, 0, 0, 0, 0, 1,
  1, 0, 0, 0, 0, 0, 0, 1,
  1, 0, 0, 0, 0, 0, 0, 1,
  1, 0, 0, 0, 0, 0, 0, 1,
  1, 0, 0, 0, 0, 0, 0, 1,
  1, 0, 0, 0, 0, 0, 0, 1,
  1, 1, 1, 1, 1, 1, 1, 1},
  
 {1, 0, 0, 0, 0, 0, 0, 1,
  1, 0, 0, 0, 0, 0, 0, 1,
  1, 0, 0, 0, 0, 0, 0, 1,
  1, 0, 0, 0, 0, 0, 0, 1,
  1, 0, 0, 0, 0, 0, 0, 1,
  1, 0, 0, 0, 0, 0, 0, 1,
  1, 0, 0, 0, 0, 0, 0, 1,
  1, 0, 0, 0, 0, 0, 0, 1},
  
 {1, 1, 1, 1, 1, 1, 1, 1,
  0, 0, 0, 0, 0, 0, 0, 0,
  0, 0, 0, 0, 0, 0, 0, 0,
  0, 0, 0, 0, 0, 0, 0, 0,
  0, 0, 0, 0, 0, 0, 0, 0,
  0, 0, 0, 0, 0, 0, 0, 0,
  0, 0, 0, 0, 0, 0, 0, 0,
  1, 1, 1, 1, 1, 1, 1, 1}
};
// The index tells which of the 3 tables to use with a certain direction.
static const char dirBoundTableIndex[] = {1, 1, 0, 0, 2, 2, 0, 0};

static const char inCorner[] = {
  1, 0, 0, 0, 0, 0, 0, 2,
  0, 0, 0, 0, 0, 0, 0, 0,
  0, 0, 0, 0, 0, 0, 0, 0,
  0, 0, 0, 0, 0, 0, 0, 0,
  0, 0, 0, 0, 0, 0, 0, 0,
  0, 0, 0, 0, 0, 0, 0, 0,
  0, 0, 0, 0, 0, 0, 0, 0,
  3, 0, 0, 0, 0, 0, 0, 4
};
static const char adjCorner[5][2] = {
  {-1, -1},  // dummy
  {1, 8}, {6, 15}, {48, 57}, {55, 62}
};
static const char corners[] = {0, 7, 56, 63};

static char empties[64];
static char nEmpties;
static char emptiesIndex[64];

static bool useMidGameOrdingInEndGame;


static void initDiscWeight();
/*
static float getQuiescenceValueMax(char *a, char lastMove, char color, char prevmaxDOF,
                char prevminDOF, char selfPieces, char oppPieces, float alpha, float beta);
static float getQuiescenceValueMin(char *a, char lastMove, char color, char prevmaxDOF,
                char prevminDOF, char selfPieces, char oppPieces, float alpha, float beta);
*/

/* Return a move using minimax search. -- Called only once per move made */
char getMinimaxMove(Board *board, bool *lm) {
  /* Initialization */
  countPruning = countSearching = countEval = 0;
  base = board->n;
  useMidGameOrdingInEndGame = false; // experimental (concerning both speed and accuracy)
  const char depth = 0; // constant within this function
  const char passes = 0;
  const char color = board->wt;
  float currValue;
  float bestValue = 2* SMALL_FLOAT; // avoid return a PASS while there is still legal moves
  char bestMove = PASS;
  float alpha = SMALL_FLOAT;
  float beta = LARGE_FLOAT;
  /* If end of game is within the brute force depth, search all the way to the end. */
  if(base + 4 + bruteForceDepth >= 64) {
    searchDepth = 64;
    inEndGame = true;
  }
  else {
    searchDepth = originalSearchDepth;
    inEndGame = false;
  }
  if(base >= QUIESCENCE_START) {
    quiescenceSearch = DO_USE_QUIESCENCE_SEARCH;
  }
  else {
    quiescenceSearch = false;
  }
  initDiscWeight();
  /* extra compensation if search is very shallow -- would not need if quiescence
    search is implemented. */
  extra = DENOMINATOR_EXTRA_DOF;
  near2cornerEdgeCoeff = NEAR_TO_CORNER_NEG_EDGE_COEFF;
  near2cornerDiagCoeff = NEAR_TO_CORNER_NEG_DIAG_COEFF;
  if(searchDepth <= 4)
    near2cornerDiagCoeff = 10 * NEAR_TO_CORNER_NEG_DIAG_COEFF;
  else if(searchDepth <= 6)
    near2cornerDiagCoeff = 4 * NEAR_TO_CORNER_NEG_DIAG_COEFF;
  // Add a certain randomness to corner value 
  cornerValue = ESTIMATED_CORNER_WORTH;
  // initialize the arrays to the board configuration.
  char *a = board->a[base];
  char *b = mmBoard;
  copyBoardArray(b, a);
  // Find the empty sqares and create indices.
  nEmpties = 0;
  for(char i=0; i<64; i++) {
    if(b[i] == EMPTY) {
      empties[nEmpties] = i;
      emptiesIndex[i] = nEmpties;
      nEmpties++;
    }
    else {
      emptiesIndex[i] = -1;
    }
  }
  // get disc counts
  char selfPieces, oppPieces;
  if(color == BLACK)
    countPieces(b, &selfPieces, &oppPieces, base+depth+4);
  else
    countPieces(b, &oppPieces, &selfPieces, base+depth+4);
  // Find all legal moves
  char *legalMoves, legalMoves1[64], flipped[20], nFlips;
  char nLegalMoves = findLegalMoves(b, color, legalMoves1);
  legalMoves = legalMoves1;
  if(CURRENT_DEBUG) {
    printf("legalMoves for color %d:\n", color);
    for(int i=0; i<nLegalMoves; i++)
      printf("%d, ", legalMoves[i]);
    printf("\n");
  }
  if(base == 0) {  // don't want to search for the first move.
    if(!randomnessLevel) {  
      if(boardFlipped)
        return HARD_WIRED_FIRST_MOVE_BOARD_FLIPPED;
      return HARD_WIRED_FIRST_MOVE;
    }
    else {
      return legalMoves[rand()%nLegalMoves];
    }
  }
  /* Shallow search for move ordering and maybe MPC cutoff */
  char remainingDepth = searchDepth - depth;
  // may need to experiment different searchDepth - depth threshhold values.
  if(USE_MOVE_ORDERING && remainingDepth >= MIN_DEPTH_FOR_ORDERING && 
      base + depth <= 64 - MIN_EMPTIES_FOR_ORDERING) {
    useMidGameOrdingInEndGame = true;
    // back-up variables to be changed in the shallow search.
    float oldAlpha = alpha, oldBeta = beta;
    char oldSearchDepth = searchDepth;
    char oldQuiescenceSearch = quiescenceSearch;
    // change some variables
    quiescenceSearch = false;
    char orderingSearchDepth = remainingDepth - ORDERING_SEARCH_DEPTH_OFFSET;
    orderingSearchDepth = orderingSearchDepth < MAX_ORDERING_SEARCH_DEPTH? 
              orderingSearchDepth : MAX_ORDERING_SEARCH_DEPTH;  // shallow search, with MAX
    char upperBound = nEmpties - MIN_EMPTIES_FOR_EVALUATION;
    if(orderingSearchDepth > upperBound)
      orderingSearchDepth = upperBound;
    searchDepth = depth + orderingSearchDepth;
    char orderedMoves[64];
    float orderedMoveValues[64];
    char index;
    for(int i=0; i<nLegalMoves; i++) {
      char place = legalMoves[i];
      assert(lm[place]);
      nFlips = tryMove(b, color, place, flipped);
      currValue = getMin(b, place, OTHER(color), depth+1, passes, nLegalMoves, nLegalMoves, 
                        selfPieces+nFlips+1, oppPieces-nFlips, alpha, beta);
      if(DEBUG_MINIMAX) {
        printf("getMin returned: %e, (depth: %d, in ordering search)\n", currValue, depth+1);
      }
      undo(b, color, place, flipped, nFlips);
      if(currValue > bestValue) {
        bestValue = currValue;
      }
      if(bestValue > alpha) {
        alpha = bestValue;
      }
      // keep the order, best to worst
      index = 0;
      while(index < i && currValue <= orderedMoveValues[index]) 
        index++;
      // insert this move, shift verything up
      for(int j=i; j>index; j--) {
        orderedMoves[j] = orderedMoves[j-1];
        orderedMoveValues[j] = orderedMoveValues[j-1];
      }
      orderedMoves[index] = place;
      orderedMoveValues[index] = currValue;
    }
    /* Possibly use MPC */
    if(USE_MPC && !inEndGame && orderingSearchDepth >= MIN_DEPTH_FOR_MPC && 
        nLegalMoves >= MIN_LEGAL_MOVES_FOR_MPC && nEmpties >= MIN_EMPTIES_FOR_MPC) {
      float topValue = orderedMoveValues[MIN_LEGAL_MOVES_FOR_MPC - 1];
      if(topValue >= TOP_VALUE_LOWER_BOUND && topValue <= TOP_VALUE_UPPER_BOUND) {  
        float threshold = ((float)MPC_THRESHOLD) / orderingSearchDepth;
        char j = 1;
        while(j < nLegalMoves && topValue - orderedMoveValues[j] <= threshold)
          j++;
        if(MPC_FEEDBACK) {
          printf("Ordered moves:\n");
          for(char k=0; k<nLegalMoves; k++) {
            char mv[3];
            mv[0] = 'a' + (orderedMoves[k] & 7);
            mv[1] = '1' + (orderedMoves[k] >> 3);
            mv[2] = 0;
            printf("%s, %f\n", mv, orderedMoveValues[k]);
          }
          printf("Cut off: ");
          for(char k=j; k<nLegalMoves; k++) {
            char mv[3];
            mv[0] = 'a' + (orderedMoves[k] & 7);
            mv[1] = '1' + (orderedMoves[k] >> 3);
            mv[2] = 0;
            if(k == nLegalMoves - 1)
              printf("%s", mv);
            else
              printf("%s, ", mv);
          }
          printf("\n");
        }
        nLegalMoves = j; // i.e. low-valued moves are cut off.
      }
    }
    // restore values changed in the ordering search.
    alpha = oldAlpha;
    beta = oldBeta;
    quiescenceSearch = oldQuiescenceSearch;
    searchDepth = oldSearchDepth; 
    bestValue = 2* SMALL_FLOAT; // avoid return a PASS while there is still legal moves
    // update the legalMoves array (now point to the ordered moves).
    legalMoves = orderedMoves;
  }
  if(USE_NEW_END_GAME_SOLVER && inEndGame && !useAndersson) {
    char result = getExactMove(b, color, selfPieces, oppPieces, legalMoves, nLegalMoves);
    if(showDots)
      printf("\n");
    if(COUNT_PRUNING && showStatistics)
      printf("searched: %d, evaluated: %d, bestValue: %f\n", 
              countSearching, countEval, bestValue_global);
    return result;
  }
//  printf("base: %d, depth: %d, searchDepth: %d, bruteForceDepth: %d\n",
//          base, depth, searchDepth, bruteForceDepth);  // debug
  /* try the ordered legal moves */
  for(int i=0; i<nLegalMoves; i++) {
    char place = legalMoves[i];
    assert(lm[place]);
    nFlips = tryMove(b, color, place, flipped);
    if(showDots) { // show progress
      printf(".");
      fflush(stdout);
    }
    if(base + depth + 4 == 63) {  // just filled the board.
      currValue = evaluateEndGame(selfPieces+nFlips+1, oppPieces-nFlips) / (1 << 10);
    }
    else if(base + 4 + bruteForceDepth >= 64) {
      if(useAndersson) {
        if(winLarge)
          currValue = strongEndGameSolve(b, OTHER(color), selfPieces+nFlips+1, 
                                        oppPieces-nFlips, 1, alpha, beta);
        else if(loseSmall)
          currValue = strongEndGameSolve(b, OTHER(color), selfPieces+nFlips+1, 
                                        oppPieces-nFlips, 1, alpha, 1);
        else {
          float lowerBound = alpha > -1? alpha : -1;
          currValue = strongEndGameSolve(b, OTHER(color), selfPieces+nFlips+1, 
                                        oppPieces-nFlips, 1, lowerBound, 1);
        }
      }
      else { // own end of game search
        if(winLarge)
          currValue = getMin(b, place, OTHER(color), depth+1, 0, nLegalMoves, nLegalMoves, 
                        selfPieces+nFlips+1, oppPieces-nFlips, alpha, beta);
        else if(loseSmall)
          currValue = getMin(b, place, OTHER(color), depth+1, 0, nLegalMoves, nLegalMoves, 
                        selfPieces+nFlips+1, oppPieces-nFlips, alpha, 1);
        else {
          float lowerBound = alpha > -1? alpha : -1;
          currValue = getMin(b, place, OTHER(color), depth+1, 0, nLegalMoves, nLegalMoves, 
                        selfPieces+nFlips+1, oppPieces-nFlips, lowerBound, 1);
        }
      }
    }
    else {
      currValue = getMin(b, place, OTHER(color), depth+1, 0, nLegalMoves, nLegalMoves, 
                        selfPieces+nFlips+1, oppPieces-nFlips, alpha, beta);
      if(DEBUG_MINIMAX || CURRENT_DEBUG2) {
        printf("getMin returned: %e, for move: %d (depth: %d)\n", currValue,
              place, depth+1);
      }
    }
    undo(b, color, place, flipped, nFlips);
    if(currValue > bestValue) {
      bestMove = place;
      bestValue = currValue;
    }
    if(bestValue > alpha)
      alpha = bestValue;
    if(!winLarge && alpha >= 1)
      break;
  }
  /* Use WDL (win-draw-loss) search to see if the estimated best move is a winning move */
  float bestMoveWDLvalue = 0;
  bool divide = false;  // for better statistics only
  if(USE_WDL_SEARCH && base+4+bruteForceDepth < 64 && base+4+bruteForceDepth >= 62) {
    nFlips = tryMove(b, color, bestMove, flipped);
    if(USE_NEW_END_GAME_SOLVER && !useAndersson) {
      bool winLarge_bkp = winLarge;
      bool loseSmall_bkp = loseSmall;
      winLarge = false;
      loseSmall = false;
      char lm[64], nlm;
      nlm = findLegalMoves(b, OTHER(color), lm);
      getExactMove(b, OTHER(color), oppPieces-nFlips, selfPieces+nFlips+1, 
                  lm, nlm); // since it is for opponent
      bestMoveWDLvalue = 0 - bestValue_global;
      winLarge = winLarge_bkp;
      loseSmall = loseSmall_bkp;
    }
    else {
      if(useAndersson) {
        bestMoveWDLvalue = strongEndGameSolve(b, OTHER(color), selfPieces+nFlips+1, 
                                        oppPieces-nFlips, 1, -1, 1);
      }
      else {
        searchDepth = 64;
        bestMoveWDLvalue = getMin(b, bestMove, OTHER(color), depth+1, 0, nLegalMoves, 
                                nLegalMoves, selfPieces+nFlips+1, oppPieces-nFlips, -1, 1);
      }
    }
    undo(b, color, bestMove, flipped, nFlips);
    if(bestValue < bestMoveWDLvalue)
      bestValue = bestMoveWDLvalue;
    if(!USE_NEW_END_GAME_SOLVER && !useAndersson)
      divide = true;
  }
  /* if estimated best move is not a winning one, do WDL search at 2 steps 
    ahead of End-of-game search */
  if(USE_WDL_SEARCH && base+4+bruteForceDepth < 64 && base+4+bruteForceDepth >= 62 &&
     bestMoveWDLvalue <= 0) {
    /* At this point, the estimated best move is not a winning move */
    /* initiate WDL search to see if a win or draw can be secured */
    searchDepth = 64; // search to the end
    float WDL_value = SMALL_FLOAT;
    char bestWDLmove = -2; // init value
    alpha = -1;
    beta = 1;
    if(USE_NEW_END_GAME_SOLVER && !useAndersson) {
      bool winLarge_bkp = winLarge;
      bool loseSmall_bkp = loseSmall;
      winLarge = false;
      loseSmall = false;
      bestWDLmove = getExactMove(b, color, selfPieces, oppPieces, legalMoves, nLegalMoves);
      WDL_value = bestValue_global;
      winLarge = winLarge_bkp;
      loseSmall = loseSmall_bkp;
    }
    else {  
      for(int i=0; i<nLegalMoves; i++) {
        char place = legalMoves[i];
        nFlips = tryMove(b, color, place, flipped);
        if(showDots) { // show progress
          printf(".");
          fflush(stdout);
        }
        if(useAndersson) {
          currValue = strongEndGameSolve(b, OTHER(color), selfPieces+nFlips+1, 
                                          oppPieces-nFlips, 1, alpha, beta);
        }
        else {
          currValue = getMin(b, place, OTHER(color), depth+1, 0, nLegalMoves, nLegalMoves, 
                          selfPieces+nFlips+1, oppPieces-nFlips, alpha, beta);
        }
        undo(b, color, place, flipped, nFlips);
        if(currValue > WDL_value) {
          bestWDLmove = place;
          WDL_value = currValue;
        }
        if(WDL_value > alpha)
          alpha = WDL_value;
        if(alpha >= beta)
          break;
      }
    }
    if(WDL_value >= 0) {
      bestMove = bestWDLmove;
      bestValue = WDL_value;
    }
    else { // losing situation anyway
      if(bestValue > WDL_value)  // just for statistical purpose
        bestValue = WDL_value;
    }
    if(!USE_NEW_END_GAME_SOLVER && !useAndersson)
      divide = true;
  }
  if(showDots)
    printf("\n");
  if(COUNT_PRUNING && showStatistics) {
    if((!USE_NEW_END_GAME_SOLVER && inEndGame && !useAndersson) || divide)
      bestValue /= (1 << 10); // simply for statistical purpose
    if(bestValue > 64)
      bestValue = 64;
    printf("searched: %d, evaluated: %d, bestValue: %f\n", 
            countSearching, countEval, bestValue);
  }
  return bestMove;
}

/***************************************************************
                    The minimax core      
 *****************************************************************
 */

/**************************************************
  Function: getMin(...)
  ---- The MIN part of Minimax ---- 
  getMin always plays for "opponent"
*/
/* NOTE: passes = 1 if there has been odd number of passes on the search path, 
  zero otherwise (same in getMax) */
  
float getMin(char *a, char lastMove, char color, char depth, char passes, 
             char prevmaxDOF, char prevminDOF, //DOF found in previous getMax and getMin;
             char selfPieces, char oppPieces,
             float alpha, float beta) {
  if(DEBUG_MINIMAX || AB_DEBUG || CURRENT_DEBUG) {
    printf("In getMin, at depth %d -- alpha: %e, beta: %e\n", depth, alpha, beta);
  }
  if(COUNT_PRUNING && showStatistics)
    countSearching++;
  // Initialization started.
  uchar uPieces = base + depth + 4;
  assert(uPieces + nEmpties == 64);  // bug detection
  char place;
  char flipped[20], nFlips;
  /* If there is only one move left, simply player it (if possible) and count scores */
  if(uPieces == 63) {
    float score;
    place = empties[0];
    nFlips = tryMove(a, color, place, flipped);
    if(nFlips) { // i.e. move can be played (by min) (min == opponent)
      score = evaluateEndGame(selfPieces-nFlips, oppPieces+nFlips+1);
      undo(a, color, place, flipped, nFlips);
    }
    else { // see if the other player (max) can play that move
      nFlips = tryMove(a, OTHER(color), place, flipped);
      if(nFlips) { // i.e. move can be played by the other player
        score = evaluateEndGame(selfPieces+nFlips+1, oppPieces-nFlips);
        undo(a, OTHER(color), place, flipped, nFlips);
      }
      else {  // no one can play this move
        score = evaluateEndGame(selfPieces, oppPieces);
      }
    }
    return score;
  }
  /* Prepare for deeper search */
  char *legalMoves, legalMoves1[64], nLegalMoves;
  nLegalMoves = findLegalMoves(a, color, legalMoves1);
  legalMoves = legalMoves1;
  if(CURRENT_DEBUG) {
    printf("legalMoves for color %d:\n", color);
    for(int i=0; i<nLegalMoves; i++)
      printf("%d, ", legalMoves[i]);
    printf("\n");
  }
  /* test if there is any legal move posssible */
  if(!nLegalMoves) {
    float score;
    if(lastMove == PASS) // last step is also pass, this branch ends here.
      score = evaluateEndGame(selfPieces, oppPieces);
    else // make a pass (set lastx to -1), depth NOT incremented.
      score = getMax(a, PASS, OTHER(color), depth, 1-passes, prevmaxDOF, 0, 
                    selfPieces, oppPieces, alpha, beta);
    return score;
  }
  /* Now there are legal moves to make */
  float minValue = LARGE_FLOAT;
  float currValue = LARGE_FLOAT;
  /* Shallow search for move ordering and maybe MPC cutoff */
  char remainingDepth = searchDepth - depth;
  // may need to experiment different searchDepth - depth threshhold values.
  if(USE_MOVE_ORDERING && remainingDepth >= MIN_DEPTH_FOR_ORDERING && 
      base + depth <= 64 - MIN_EMPTIES_FOR_ORDERING) {
    // back-up variables to be changed in the shallow search.
    float oldAlpha = alpha, oldBeta = beta;
    char oldSearchDepth = searchDepth;
    char oldQuiescenceSearch = quiescenceSearch;
    // change some variables
    quiescenceSearch = false;
    char orderingSearchDepth = remainingDepth - ORDERING_SEARCH_DEPTH_OFFSET;
    if(orderingSearchDepth > MAX_ORDERING_SEARCH_DEPTH)
      orderingSearchDepth = MAX_ORDERING_SEARCH_DEPTH;  // shallow search, with MAX
    char upperBound = nEmpties - MIN_EMPTIES_FOR_EVALUATION;
    if(orderingSearchDepth > upperBound)
      orderingSearchDepth = upperBound;
    searchDepth = depth + orderingSearchDepth;
    char orderedMoves[64];
    float orderedMoveValues[64];
    char index;
    for(int i=0; i<nLegalMoves; i++) {
      char place = legalMoves[i];
      nFlips = tryMove(a, color, place, flipped);
      // can never reach cut-off depth here.
      currValue = getMax(a, place, OTHER(color), depth+1, passes, prevmaxDOF, nLegalMoves, 
                        selfPieces-nFlips, oppPieces+nFlips+1, alpha, beta);
      undo(a, color, place, flipped, nFlips);
      if(currValue < minValue) {
        minValue = currValue;
      }
      if(minValue < beta) {
        beta = minValue;
      }
      // keep the order, best to worst (small to large, since in getmin)
      index = 0;
      while(index < i && currValue >= orderedMoveValues[index]) 
        index++;
      // insert this move, shift verything up
      for(int j=i; j>index; j--) {
        orderedMoves[j] = orderedMoves[j-1];
        orderedMoveValues[j] = orderedMoveValues[j-1];
      }
      orderedMoves[index] = place;
      orderedMoveValues[index] = currValue;
    }
    /* Possibly use MPC */
    if(USE_MPC && !inEndGame && orderingSearchDepth >= MIN_DEPTH_FOR_MPC &&
        nLegalMoves >= MIN_LEGAL_MOVES_FOR_MPC && nEmpties >= MIN_EMPTIES_FOR_MPC) {
      float topValue = orderedMoveValues[MIN_LEGAL_MOVES_FOR_MPC - 1]; // smallest  
      if(topValue >= TOP_VALUE_LOWER_BOUND && topValue <= TOP_VALUE_UPPER_BOUND) {  
        float threshold = ((float)MPC_THRESHOLD) / orderingSearchDepth;
        char j = 1;
        while(j < nLegalMoves && orderedMoveValues[j] - topValue <= threshold)
          j++;
        nLegalMoves = j; // i.e. low-valued moves are cut off.
      }
    }
    // restore values changed in the ordering search.
    alpha = oldAlpha;
    beta = oldBeta;
    quiescenceSearch = oldQuiescenceSearch;
    searchDepth = oldSearchDepth; 
    minValue = LARGE_FLOAT;
    // update the legalMoves array (now point to the ordered moves).
    legalMoves = orderedMoves;
  }
  if(AB_DEBUG) {
    printf("after ordering, at depth %d -- alpha: %e, beta: %e\n", depth, alpha, beta);
  }
  /* try the ordered legal moves (or unordered moves if ordering didn't happen) */
  for(int i=0; i<nLegalMoves; i++) {
    char place = legalMoves[i];
    nFlips = tryMove(a, color, place, flipped);
    /* test if cut-off depth will be reached. If so, avoid an extra recursive call. */
    if(depth + 1 >= searchDepth + passes) { // always evaluate when it is one side's turn
      // Quiescence search for corner trade
      char lastMoveIndex = inCorner[lastMove];
      char placeIndex = inCorner[place];
      if(DO_USE_QUIESCENCE_SEARCH && quiescenceSearch && (lastMoveIndex || placeIndex)) {
        bool currValueUpdated = false;
        if(placeIndex) {
          searchDepth += 3;
          char fpd[20];
          float newValue;
          // get max part, but limit moves to squares adjacent to corners
          currValue = SMALL_FLOAT;
          for(char j=0; j<2; j++) {
            char adjCornerPlace = adjCorner[inCorner[place]][j];
            char nfpd = tryMove(a, OTHER(color), adjCornerPlace, fpd);
            if(nfpd) {
              newValue = getMin(a, adjCornerPlace, color, depth+2, passes, prevmaxDOF,
                                nLegalMoves, selfPieces-nFlips+nfpd+1, 
                                oppPieces+nFlips+1-nfpd, alpha, beta);
              currValueUpdated = true;
              if(newValue > currValue)
                currValue = newValue;
              undo(a, OTHER(color), adjCornerPlace, fpd, nfpd);
            }
          }
          searchDepth -= 3;
        }
        else if(lastMoveIndex) { // search 2 extra plies to discover a corner trade.
          if(place == adjCorner[lastMoveIndex][0] || place == adjCorner[lastMoveIndex][1]) {
            searchDepth += 2;
            currValue = getMax(a, place, OTHER(color), depth+1, passes, prevmaxDOF, 
                          nLegalMoves, selfPieces-nFlips, oppPieces+nFlips+1, alpha, beta);
            currValueUpdated = true;
            searchDepth -= 2;
          }
        }
        // make sure the position gets evaluated somehow.
        if(!currValueUpdated) { // used normal eval. if quiescence search doesn't happen
          currValue = evaluateBoard(a, OTHER(color), OTHER(color), prevmaxDOF, nLegalMoves, 
                                    selfPieces-nFlips, oppPieces+nFlips+1);
        }
      }
      else {
        currValue = evaluateBoard(a, OTHER(color), OTHER(color), prevmaxDOF, nLegalMoves, 
                                  selfPieces-nFlips, oppPieces+nFlips+1);
      }
    }
    else {
      currValue = getMax(a, place, OTHER(color), depth+1, passes, prevmaxDOF, nLegalMoves, 
                        selfPieces-nFlips, oppPieces+nFlips+1, alpha, beta);
    }
    if(DEBUG_MINIMAX || CURRENT_DEBUG) {
      printf("getMax returned: %e, for move: %d. (depth: %d)\n", currValue, 
            legalMoves[i], depth+1);
    }
    undo(a, color, place, flipped, nFlips);
    if(currValue < minValue)
      minValue = currValue;
    if(minValue < beta)
      beta = minValue;
    if(alpha >= beta) {
      if(COUNT_PRUNING && showStatistics)
        countPruning++;
      return beta;
    }
  }
  return minValue;
}

/***********************************************/
  /**************************************************
  Function: getMax(...)
  ---- The MAX par of Minimax ---- 
  getMax always plays for "self"
*/
float getMax(char *a, char lastMove, char color, char depth, char passes, 
             char prevmaxDOF, char prevminDOF, 
             char selfPieces, char oppPieces,
             float alpha, float beta) {
  if(DEBUG_MINIMAX || AB_DEBUG || CURRENT_DEBUG) {
    printf("In getMax, at depth %d -- alpha: %e, beta: %e\n", depth, alpha, beta);
    // printf("DEBUG_MINIMAX: %d", DEBUG_MINIMAX);
  }
  if(COUNT_PRUNING && showStatistics)
    countSearching++;
  // Initialization started.
  uchar uPieces = base + depth + 4;
  assert(uPieces + nEmpties == 64);  // bug detection
  char place;
  char flipped[20], nFlips;
  /* If there is only one move left, simply player it (if possible) and count scores */
  if(uPieces == 63) {
    float score;
    place = empties[0];
    nFlips = tryMove(a, color, place, flipped);
    if(nFlips) { // i.e. move can be played (by min) (min == opponent)
      score = evaluateEndGame(selfPieces+nFlips+1, oppPieces-nFlips);
      undo(a, color, place, flipped, nFlips);
    }
    else { // see if the other player (max) can play that move
      nFlips = tryMove(a, OTHER(color), place, flipped);
      if(nFlips) { // i.e. move can be played by the other player
        score = evaluateEndGame(selfPieces-nFlips, oppPieces+nFlips+1);
        undo(a, OTHER(color), place, flipped, nFlips);
      }
      else {  // no one can play this move
        score = evaluateEndGame(selfPieces, oppPieces);
      }
    }
    return score;
  }
  /* Prepare for deeper search */
  char *legalMoves, legalMoves1[64], nLegalMoves;
  nLegalMoves = findLegalMoves(a, color, legalMoves1);
  legalMoves = legalMoves1;
  if(CURRENT_DEBUG) {
    printf("legalMoves for color %d:\n", color);
    for(int i=0; i<nLegalMoves; i++)
      printf("%d, ", legalMoves[i]);
    printf("\n");
  }
  /* test if there is any legal move posssible */
  if(!nLegalMoves) {
    float score;
    if(lastMove == PASS) // last step is also pass, this branch ends here.
      score = evaluateEndGame(selfPieces, oppPieces);
    else // make a pass (set lastx to -1), depth NOT incremented.
      score = getMin(a, PASS, OTHER(color), depth, 1-passes, 0, prevminDOF, 
                    selfPieces, oppPieces, alpha, beta);
    return score;
  }
  /* Now there are legal moves to make */
  float maxValue = SMALL_FLOAT;
  float currValue = SMALL_FLOAT;
  /* Shallow search for move ordering and maybe MPC cutoff */
  char remainingDepth = searchDepth - depth;
  // may need to experiment different searchDepth - depth threshhold values.
  if(USE_MOVE_ORDERING && remainingDepth >= MIN_DEPTH_FOR_ORDERING && 
      base + depth <= 64 - MIN_EMPTIES_FOR_ORDERING) { 
              // at least 10 moves left
    // back-up variables to be changed in the shallow search.
    float oldAlpha = alpha, oldBeta = beta;
    char oldSearchDepth = searchDepth;
    char oldQuiescenceSearch = quiescenceSearch;
    // change some variables
    quiescenceSearch = false;
    char orderingSearchDepth = remainingDepth - ORDERING_SEARCH_DEPTH_OFFSET;
    if(orderingSearchDepth > MAX_ORDERING_SEARCH_DEPTH)
      orderingSearchDepth = MAX_ORDERING_SEARCH_DEPTH;  // shallow search, with MAX
    char upperBound = nEmpties - MIN_EMPTIES_FOR_EVALUATION;
    if(orderingSearchDepth > upperBound)
      orderingSearchDepth = upperBound;
    searchDepth = depth + orderingSearchDepth;
    char orderedMoves[64];
    float orderedMoveValues[64];
    char index;
    for(int i=0; i<nLegalMoves; i++) {
      char place = legalMoves[i];
      nFlips = tryMove(a, color, place, flipped);
      // can never reach cut-off depth here.
      currValue = getMin(a, place, OTHER(color), depth+1, passes, nLegalMoves, prevminDOF, 
                        selfPieces+nFlips+1, oppPieces-nFlips, alpha, beta);
      undo(a, color, place, flipped, nFlips);
      if(currValue > maxValue) {
        maxValue = currValue;
      }
      if(maxValue > alpha) {
        alpha = maxValue;
      }
      // keep the order, best to worst (small to large, since in getmin)
      index = 0;
      while(index < i && currValue <= orderedMoveValues[index]) 
        index++;
      // insert this move, shift verything up
      for(int j=i; j>index; j--) {
        orderedMoves[j] = orderedMoves[j-1];
        orderedMoveValues[j] = orderedMoveValues[j-1];
      }
      orderedMoves[index] = place;
      orderedMoveValues[index] = currValue;
    }
    /* Possibly use MPC */
    if(USE_MPC && !inEndGame && orderingSearchDepth >= MIN_DEPTH_FOR_MPC &&
        nLegalMoves >= MIN_LEGAL_MOVES_FOR_MPC && nEmpties >= MIN_EMPTIES_FOR_MPC) {
      float topValue = orderedMoveValues[0];
      if(topValue >= TOP_VALUE_LOWER_BOUND && topValue <= TOP_VALUE_UPPER_BOUND) {    
        float threshold = ((float)MPC_THRESHOLD) / orderingSearchDepth;
        char j = 1;
        while(j < nLegalMoves && topValue - orderedMoveValues[j] <= threshold)
          j++;
        nLegalMoves = j; // i.e. low-valued moves are cut off.
      }
    }
    // restore values changed in the ordering search.
    alpha = oldAlpha;
    beta = oldBeta;
    quiescenceSearch = oldQuiescenceSearch;
    searchDepth = oldSearchDepth; 
    maxValue = SMALL_FLOAT;
    // update the legalMoves array (now point to the ordered moves).
    legalMoves = orderedMoves;
  }
  if(AB_DEBUG) {
    printf("After ordering, at depth %d -- alpha: %e, beta: %e\n", depth, alpha, beta);
  }
  /* try the ordered legal moves (or unordered moves if ordering didn't happen) */
  for(int i=0; i<nLegalMoves; i++) {
    char place = legalMoves[i];
    nFlips = tryMove(a, color, place, flipped);
    /* test if cut-off depth will be reached. If so, avoid an extra recursive call. */
    if(depth + 1 >= searchDepth + passes) { // always evaluate when it is one side's turn
      // Quiescence search for corner trade
      char lastMoveIndex = inCorner[lastMove];
      char placeIndex = inCorner[place];
      if(DO_USE_QUIESCENCE_SEARCH && quiescenceSearch && (lastMoveIndex || placeIndex)) {
        bool currValueUpdated = false;
        if(placeIndex) {
          searchDepth += 3;
          char fpd[20];
          float newValue;
          // get min part, but limit moves to squares adjacent to corners
          currValue = LARGE_FLOAT;
          for(char j=0; j<2; j++) {
            char adjCornerPlace = adjCorner[inCorner[place]][j];
            char nfpd = tryMove(a, OTHER(color), adjCornerPlace, fpd);
            if(nfpd) {
              newValue = getMax(a, adjCornerPlace, color, depth+2, passes, nLegalMoves,
                                prevminDOF, selfPieces+nFlips-nfpd+1, 
                                oppPieces-nFlips+nfpd+1, alpha, beta);
              currValueUpdated = true;
              if(newValue < currValue)
                currValue = newValue;
              undo(a, OTHER(color), adjCornerPlace, fpd, nfpd);
            }
          }
          searchDepth -= 3;
        }
        else if(lastMoveIndex) { // search 2 extra plies to discover a corner trade.
          if(place == adjCorner[lastMoveIndex][0] || place == adjCorner[lastMoveIndex][1]) {
            searchDepth += 2;
            currValue = getMin(a, place, OTHER(color), depth+1, passes, nLegalMoves, 
                          prevminDOF, selfPieces+nFlips+1, oppPieces-nFlips, alpha, beta);
            currValueUpdated = true;
            searchDepth -= 2;
          }
        }
        // make sure the position gets evaluated somehow.
        if(!currValueUpdated) { // used normal eval. if quiescence search doesn't happen
          currValue = evaluateBoard(a, color, OTHER(color), nLegalMoves, prevminDOF, 
                                    selfPieces+nFlips+1, oppPieces-nFlips);
        }
      }
      else {
        currValue = evaluateBoard(a, color, OTHER(color), nLegalMoves, prevminDOF, 
                                  selfPieces+nFlips+1, oppPieces-nFlips);
      }
    }
    else {
      currValue = getMin(a, place, OTHER(color), depth+1, passes, nLegalMoves, prevminDOF, 
                        selfPieces+nFlips+1, oppPieces-nFlips, alpha, beta);
    }
    if(DEBUG_MINIMAX || CURRENT_DEBUG) {
      printf("getMin returned: %e, for move: %d. (depth: %d)\n", currValue,
            legalMoves[i], depth+1);
    }
    undo(a, color, place, flipped, nFlips);
    if(currValue > maxValue)
      maxValue = currValue;
    if(maxValue > alpha)
      alpha = maxValue;
    if(alpha >= beta) {
      if(COUNT_PRUNING && showStatistics)
        countPruning++;
      return alpha;
    }
  }
  return maxValue;
}

/************************ End of Minimax core ***********************/


/* copy a board array */
void copyBoardArray(char *to, char *from) {
  for(int i=0; i<60; i+=6) { // some loop unrolling
    to[i] = from[i];
    to[i+1] = from[i+1];
    to[i+2] = from[i+2];
    to[i+3] = from[i+3];
    to[i+4] = from[i+4];
    to[i+5] = from[i+5];
  }
  to[60] = from[60];
  to[61] = from[61];
  to[62] = from[62];
  to[63] = from[63];
}

/* test if a move is legal one on a board array for the given color */
bool legalMove(char *a, char color, char place) {
  if(a[place] != EMPTY)
    return false;
  /* test left for possible flips */
  bool result = false;
  char oppcolor = OTHER(color);
  uchar dirs = dirmask[place];
  if(dirs & 1) {
    result = canFlip(a, place, 0, color, oppcolor);
    if(result)
      return result;
  }
  if(dirs & 2) {
    result = canFlip(a, place, 1, color, oppcolor);
    if(result)
      return result;
  }
  if(dirs & 4) {
    result = canFlip(a, place, 2, color, oppcolor);
    if(result)
      return result;
  }
  if(dirs & 8) {
    result = canFlip(a, place, 3, color, oppcolor);
    if(result)
      return result;
  }
  if(dirs & 16) {
    result = canFlip(a, place, 4, color, oppcolor);
    if(result)
      return result;
  }
  if(dirs & 32) {
    result = canFlip(a, place, 5, color, oppcolor);
    if(result)
      return result;
  }
  if(dirs & 64) {
    result = canFlip(a, place, 6, color, oppcolor);
    if(result)
      return result;
  }
  if(dirs & 128) {
    result = canFlip(a, place, 7, color, oppcolor);
    if(result)
      return result;
  }
  return result;
}

/* test if flip in one direction is possible */
inline bool canFlip(char *a, char place, char dirIndex, char color, char oppcolor) {
  const bool *atBound = boundTables[dirBoundTableIndex[dirIndex]];
  char dir = directions[dirIndex];
  char p = place + dir;
  if(a[p] == oppcolor && !atBound[p]) {
    p += dir;
    if(a[p] == oppcolor && !atBound[p]) {
      p += dir;
      if(a[p] == oppcolor && !atBound[p]) {
        p += dir;
        if(a[p] == oppcolor && !atBound[p]) {
          p += dir;
          if(a[p] == oppcolor && !atBound[p]) {
            p += dir;
            if(a[p] == oppcolor && !atBound[p]) {
              p += dir;
            }
          }
        }
      }
    }
    if(a[p] == color)
      return true;
  }
  return false;
}

/* flip all the discs in a certain direction, played at 'place'. return the 
  count of discs flipped. 'flipCount' is used as the entry index for the 
  'flipped' array. */
inline char flip(char *a, char place, char dirIndex, char color, char oppcolor, 
                char *flipped, char flipCount) {
  const bool *atBound = boundTables[dirBoundTableIndex[dirIndex]];
  char dir = directions[dirIndex];
  char count = flipCount;
  char p = place + dir;
  if(a[p] == oppcolor && !atBound[p]) {
    p += dir;
    if(a[p] == oppcolor && !atBound[p]) {
      p += dir;
      if(a[p] == oppcolor && !atBound[p]) {
        p += dir;
        if(a[p] == oppcolor && !atBound[p]) {
          p += dir;
          if(a[p] == oppcolor && !atBound[p]) {
            p += dir;
            if(a[p] == oppcolor && !atBound[p]) {
              p += dir;
            }
          }
        }
      }
    }
    if(a[p] == color) {
      p -= dir;
      while(p != place) {
        a[p] = color;
        flipped[count] = p;
        count++;
        p -= dir;
      }
    }
  }
  // number of element in the current 'flipped' array minus previous count.
  return count - flipCount;
}

/* find all the legal moves, stored as bitmask of two ints. Returns the number 
  of legalMoves found */
char findLegalMoves(char *a, char color, char *legalMoves) {
  char count = 0;
  char *emp = empties, nEmp = nEmpties;
  for(char i=0; i<nEmp; i++) {
    if(legalMove(a, color, emp[i])) {
      legalMoves[count] = emp[i];
      count++;
    }
  }
  return count;
}

/* find the number of legalmoves without storing them, only want the count */
char findLegalMoves(char *a, char color) {
  char count = 0;
  char *emp = empties, nEmp = nEmpties;
  for(char i=0; i<nEmp; i++)
    count += legalMove(a, color, emp[i]);
  return count;
}

/* find the number of legalmoves without storing them, only want the count and 
  the corner moves (also find the number of corner moves). */
char findLegalMoves(char *a, char color, char *cornerMoves, char *nCornerMoves) {
  char count = 0;
  char countCM = 0;
  char *emp = empties, nEmp = nEmpties;
  for(char i=0; i<nEmp; i++) {
    if(legalMove(a, color, emp[i])) {
      count++;
      if(i == 0 || i == 7 || i == 56 || i == 63) {
        cornerMoves[countCM++] = i;
      }
    }
  }
  *nCornerMoves = countCM;
  return count;
}

/* try a move on the minimax's board. Return the number of pieces flipped. 
  in this whole project, tryMove is only called after we know this move to 
  be made is legal. */
char tryMove(char *a, char color, char place, char *flipped) {
  if(DEBUG_MINIMAX) {
    printf("tryMove - place: %d, color %d\n", place, color);
  }
  if(a[place] != EMPTY)
    return 0;
  char flipCount = 0;
  char oppcolor = OTHER(color);
  char dirs = dirmask[place];
  if(dirs & 1) {
    flipCount += flip(a, place, 0, color, oppcolor, flipped, flipCount);
  }
  if(dirs & 2) {
    flipCount += flip(a, place, 1, color, oppcolor, flipped, flipCount);
  }
  if(dirs & 4) {
    flipCount += flip(a, place, 2, color, oppcolor, flipped, flipCount);
  }
  if(dirs & 8) {
    flipCount += flip(a, place, 3, color, oppcolor, flipped, flipCount);
  }
  if(dirs & 16) {
    flipCount += flip(a, place, 4, color, oppcolor, flipped, flipCount);
  }
  if(dirs & 32) {
    flipCount += flip(a, place, 5, color, oppcolor, flipped, flipCount);
  }
  if(dirs & 64) {
    flipCount += flip(a, place, 6, color, oppcolor, flipped, flipCount);
  }
  if(dirs & 128) {
    flipCount += flip(a, place, 7, color, oppcolor, flipped, flipCount);
  }
  if(flipCount) {
    a[place] = color;
    char placeIndex = emptiesIndex[place];
    assert(placeIndex >= 0);
    char lastEmpty = empties[--nEmpties]; // nEmpties decremented
    empties[placeIndex] = lastEmpty; // remove from empty list 'place'
    emptiesIndex[lastEmpty] = placeIndex;
    emptiesIndex[place] = -1;  // NOT necessary, only for bug detection. (for assert())
  }
  if(DEBUG_MINIMAX) {
    printBoardArray(a);
    printf("flipCount: %d\n", flipCount);
  }
  return flipCount;
}

/* undo a move previously made by 'color' */
inline void undo(char *a, char color, char place, char *flipped, char nFlipped) {
  empties[nEmpties] = place;
  assert(nFlipped > 0);
  assert(emptiesIndex[place] == -1);
  emptiesIndex[place] = nEmpties;
  nEmpties++;
  a[place] = EMPTY;
  char otherColor = OTHER(color);
  for(char i=0; i<nFlipped; i++) {
    a[flipped[i]] = otherColor;
  }
  if(DEBUG_MINIMAX) {
    printf("undo move: %d\n", place);
    printBoardArray(a);
  }
}


/* Premitive evaluation functions -- not good at all, just to make game going */
/* Evaluate an end-of-game situation (just count) */
inline int evaluateEndGame(char selfPieces, char oppPieces) {
  if(COUNT_PRUNING && showStatistics)
    countEval++; 
  int score;
  if(selfPieces > oppPieces)
    score = 64 - oppPieces - oppPieces;
  else if(selfPieces < oppPieces)
    score = selfPieces + selfPieces - 64;
  else
    score = 0;
  return score << 10; // make sure a secured win out-weighs any estimated advantage.
}

/* Only ask if it is Win, Draw or Loss, > 0 means win. */
inline int evaluateEndGameWDL(char selfPieces, char oppPieces) {
  return selfPieces - oppPieces;
}

/* Evaluate an board situation (game still in progress) */
float evaluateBoard(char *a, char forWhom, char whoseTurn, char prevmaxDOF, 
                    char prevminDOF, char selfPieces, char oppPieces) {
  if(DEBUG_MINIMAX) {
    printBoardArray(a);
    printf("Depth: %d, disc diff. value: %e\n", selfPieces+oppPieces-base-4, 
            (float)(selfPieces-oppPieces));
  }
  if(COUNT_PRUNING && showStatistics) {
    countEval++;
    countSearching++;
  }
  // One side wins large if the other is anihilated
  if(selfPieces == 0)
    return SMALL_FLOAT;
  else if(oppPieces == 0)
    return LARGE_FLOAT;
  
  char nEmp = 64 - selfPieces - oppPieces;
  
  char nLegalMoves = findLegalMoves(a, whoseTurn); // slow step!
  // in mamimax, always max for self, min for opponent.
  if(forWhom == whoseTurn)
    prevmaxDOF = nLegalMoves;
  else
    prevminDOF = nLegalMoves;
  //*/
  
  float result = 0;
  /* First evaluate for mobility */
  if(USE_MOBILITY_RATIO) {  // *** DEFAULT ***
    float mobilityRatio = ((prevmaxDOF + extra) / (prevminDOF + extra));
    result += mobilityRatio - 1/mobilityRatio;
  }
  else { // use mobility difference
    result += prevmaxDOF - prevminDOF;
  }
  /* Evaluate for disc difference */
  if(USE_DISC_RATIO) { // experimental (result: doesn't look good)
    result += discWeight * ((float)selfPieces/oppPieces - (float)oppPieces/selfPieces);
  }
  else { // *** DEFAULT ***
    int deficit = oppPieces - selfPieces;
    if(deficit > DISC_DIFF_THRESHOLD) {
      deficit = DISC_DIFF_THRESHOLD;
    }
    result += discWeight * (- deficit); // discWeigh is negative most of the times
  }
  /* A little simple account for corners */
  /* If in advantage, we are not too keen to capture a corner immediately, since
    it is likely to be ours anyway; but we are very avert to let opponent 
    a corner since we would probably be able to prevent that. And intuitively 
    it is the opposite when in disadvantage. */
  // when mobility is high cornerValue is effective devaluated, and vice versa
  /* ! -- The above idea is currently NOT reflected in the following evaluation! --*/
  
  float corner = cornerValue;
  float near2cornerEdge = near2cornerEdgeCoeff * nEmp;
  float near2cornerDiag = near2cornerDiagCoeff * nEmp;
  char self = forWhom, opp = OTHER(forWhom);
  char c1 = a[0], c2 = a[7], c3 = a[56], c4 = a[63];
  // Account for corners and near-to-corner squares.
  /* upper-left */
  if(c1 == self) {
    result += corner;
  }
  else if(c1 == opp) {
    result -= corner;
  }
  else { // this corner is still empty
    char near0 = a[9];
    if(near0 == self)
      result -= near2cornerDiag;
    else if(near0 == opp)
      result += near2cornerDiag;
    if(COUNT_FOR_NEAR_TO_CORNER_EDGE) {  
      if(c2 == EMPTY) {
        char near1 = a[1];
        if(near1 == self)
          result -= near2cornerEdge;
        else if(near1 == opp)
          result += near2cornerEdge;
      }
      if(c3 == EMPTY) {
        char near2 = a[8];
        if(near2 == self)
          result -= near2cornerEdge;
        else if(near2 == opp)
          result += near2cornerEdge;
      }
    }
  }
  /* upper-right */
  if(c2 == self) {
    result += corner;
  }
  else if(c2 == opp) {
    result -= corner;
  }
  else { // this corner is still empty
    char near0 = a[14];
    if(near0 == self)
      result -= near2cornerDiag;
    else if(near0 == opp)
      result += near2cornerDiag;
    if(COUNT_FOR_NEAR_TO_CORNER_EDGE) {
      if(c1 == EMPTY) {
        char near1 = a[6];
        if(near1 == self)
          result -= near2cornerEdge;
        else if(near1 == opp)
          result += near2cornerEdge;
      }
      if(c4 == EMPTY) {
        char near2 = a[15];
        if(near2 == self)
          result -= near2cornerEdge;
        else if(near2 == opp)
          result += near2cornerEdge;
      }
    }
  }
  /* lower-left */
  if(c3 == self) {
    result += corner;
  }
  else if(c3 == opp) {
    result -= corner;
  }
  else { // this corner is still empty
    char near0 = a[49];
    if(near0 == self)
      result -= near2cornerDiag;
    else if(near0 == opp)
      result += near2cornerDiag;
    if(COUNT_FOR_NEAR_TO_CORNER_EDGE) {  
      if(c1 == EMPTY) {
        char near1 = a[48];
        if(near1 == self)
          result -= near2cornerEdge;
        else if(near1 == opp)
          result += near2cornerEdge;
      }
      if(c4 == EMPTY) {
        char near2 = a[57];
        if(near2 == self)
          result -= near2cornerEdge;
        else if(near2 == opp)
          result += near2cornerEdge;
      }
    }
  }
  /* lower-right */
  if(c4 == self) {
    result += corner;
  }
  else if(c4 == opp) {
    result -= corner;
  }
  else { // this corner is still empty
    char near0 = a[54];
    if(near0 == self)
      result -= near2cornerDiag;
    else if(near0 == opp)
      result += near2cornerDiag;
    if(COUNT_FOR_NEAR_TO_CORNER_EDGE) {  
      if(c2 == EMPTY) {
        char near1 = a[55];
        if(near1 == self)
          result -= near2cornerEdge;
        else if(near1 == opp)
          result += near2cornerEdge;
      }
      if(c3 == EMPTY) {
        char near2 = a[62];
        if(near2 == self)
          result -= near2cornerEdge;
        else if(near2 == opp)
          result += near2cornerEdge;
      }
    }
  }
  
  /* ---- Old ------
  float selfCorner = cornerValue;
  float oppCorner = cornerValue;
  result += (a[0] == forWhom) * selfCorner;
  result -= (a[0] == OTHER(forWhom)) * oppCorner;
  result += (a[7] == forWhom) * selfCorner;
  result -= (a[7] == OTHER(forWhom)) * oppCorner;
  result += (a[56] == forWhom) * selfCorner;
  result -= (a[56] == OTHER(forWhom)) * oppCorner;
  result += (a[63] == forWhom) * selfCorner;
  result -= (a[63] == OTHER(forWhom)) * oppCorner;
  */
  
  /* The following doesn't change anything within a getMinimax call, and hence 
    has no effect on the choice of moves. Howver it makes 
    the score more consistant throughout the game. */
  // result += scoreCorrection; 
  
  //char sign;
//  if(forWhom == BLACK)
//    sign = 1; 
//  else
//    sign = -1;
//  // BLACK = 1, WHITE = 2, EMPTY = 0
//  // Approach 1 -- About the same speed as Approch 2
//  char mean = EMPTY+1;
//  result += (((a[0]+1) % 3 - mean) * sign) * ESTIMATED_CORNER_WORTH;
//  result += (((a[7]+1) % 3 - mean) * sign) * ESTIMATED_CORNER_WORTH;
//  result += (((a[56]+1) % 3 - mean) * sign) * ESTIMATED_CORNER_WORTH;
//  result += (((a[63]+1) % 3 - mean) * sign) * ESTIMATED_CORNER_WORTH;
  // */
  /* // Approach 2 -- need to set sign = -2.0 for BLACK, 2.0 for WHITE
  float mean = ((float)BLACK + (float)WHITE) * 0.5;
  result += ((float)(a[0]) - mean) * (a[0] / (a[0] + 1.0e-20)) * sign * cornerWorth;
  result += ((float)(a[7]) - mean) * (a[7] / (a[7] + 1.0e-20)) * sign * cornerWorth;
  result += ((float)(a[56]) - mean) * (a[56] / (a[56] + 1.0e-20)) * sign * cornerWorth;
  result += ((float)(a[63]) - mean) * (a[63] / (a[63] + 1.0e-20)) * sign * cornerWorth;
  // */
  return result;
}

/* Print out the board array -- for debugging purpose */
void printBoardArray(char *a) {
  char place;
  printf("\n   0 1 2 3 4 5 6 7\n");
  for(char y=0; y<8; y++) {
    printf("%d  ", y);
    for(char x=0; x<8; x++) {
      place = CONV_21(x, y);
      if(a[place] == BLACK)
        printf("X ");
      else if(a[place] == WHITE)
        printf("O ");
      else
        printf(". ");
    }
    printf("\n");
  }
  printf("\n");
}

/* Uses Andersson's complicated end of game solver 
  Here 'color' is the opponent color as viewed by Minimax, but is 'self' 
  as viewed by EndSolve. */
  /* NOTE: In core minimax, alpha is for max and beta for min, but here 
   since EndSolve is actually evaluating for Min but return a positve
   value (and we invert that value as return value), both alpha and beta must
   be swapped and then inverted (0 - ..) before passing to EndSolve. */
int strongEndGameSolve(char *a, char color, char selfPieces, char oppPieces, 
                       char prevNotPass, float alpha, float beta) {
  int result;
  int nEmpt = 64 - selfPieces - oppPieces;
  int difference = oppPieces - selfPieces;
  int ecolor = color == BLACK? END_BLACK : END_WHITE;
  uchar eboard[91]; // board specified by the end of game solver
  for(int i=0; i<91; i++)
    eboard[i] = DUMMY;
  for(int y=0; y<8; y++)
    for(int x=0; x<8; x++) {
      if(a[CONV_21(x, y)] == BLACK)
        eboard[10+x+9*y] = END_BLACK;
      else if(a[CONV_21(x, y)] == WHITE)
        eboard[10+x+9*y] = END_WHITE;
      else
        eboard[10+x+9*y] = END_EMPTY;
    }
  float end_alpha = 0 - beta;
  float end_beta = 0 - alpha;
  PrepareToSolve(eboard);
  result = EndSolve(eboard, end_alpha, end_beta, ecolor, nEmpt, difference, prevNotPass);
  return 0 - result; // since result is as viewed by opponent.
}

/* Initialize the disc weight (relative to mobility -- staged) */
static void initDiscWeight() {
  char evalDepth = base + searchDepth;
  if(evalDepth < STAGE_1_END) {
    discWeight = DISC_WEIGHT_STAGE_1;
  }
  else if(evalDepth < STAGE_2_START) {
    discWeight = DISC_WEIGHT_STAGE_1 + (DISC_WEIGHT_STAGE_2 - DISC_WEIGHT_STAGE_1) *
                  (evalDepth - STAGE_1_END) / (STAGE_2_START - STAGE_1_END);
  }
  else if(evalDepth < STAGE_2_END) {
    discWeight = DISC_WEIGHT_STAGE_2;
  }
  else if(evalDepth < STAGE_3_START) {
    discWeight = DISC_WEIGHT_STAGE_2 + (DISC_WEIGHT_STAGE_3 - DISC_WEIGHT_STAGE_2) *
                  (evalDepth - STAGE_2_END) / (STAGE_3_START - STAGE_2_END);
  }
  else if(evalDepth < STAGE_3_END) {
    discWeight = DISC_WEIGHT_STAGE_3;
  }
  else if(evalDepth < STAGE_4_START) {
    discWeight = DISC_WEIGHT_STAGE_3 + (DISC_WEIGHT_STAGE_4 - DISC_WEIGHT_STAGE_3) *
                  (evalDepth - STAGE_3_END) / (STAGE_4_START - STAGE_3_END);
  }
  else {
    discWeight = DISC_WEIGHT_STAGE_4;
  }
  // add some randomness
  if(randomnessLevel) {
    float randomFactor = ((float)(rand() % 4096) / 4096) - 0.5; // +- 0.5
    float randomCoeff = 1 + randomnessLevel * randomFactor * RANDOM_FACTOR_CONSTANT;
    discWeight *= randomCoeff;
  }
  //discWeight = DISC_WEIGHT_TEST;  // debug / test
}


/* ---------------------------------------------
  Exactly solve end-of-game
  Borrowed some idea from Andersson's end of game solver.
  It is still rather slow, since I haven't fully understood how exactly his 
  code works. However, it is better than simply using the minimax core for 
  mid-game. 
  -----------------------------------------------*/

// Use 91-square board -- more efficient (idea from Andersson's code)
#ifndef DUMMY
#define DUMMY 3
#endif
static const char directionsEnd[] = {1, -1, 8, -8, 9, -9, 10, -10};
static const uchar dirmask91[] = {
  0,  0,  0,  0,  0,  0,  0,  0,  0,
  0,  81, 81, 87, 87, 87, 87, 22, 22,
  0,  81, 81, 87, 87, 87, 87, 22, 22,
  0,  121,121,255,255,255,255,182,182,
  0,  121,121,255,255,255,255,182,182,
  0,  121,121,255,255,255,255,182,182,
  0,  121,121,255,255,255,255,182,182,
  0,  41, 41, 171,171,171,171,162,162,
  0,  41, 41, 171,171,171,171,162,162,
  0,  0,  0,  0,  0,  0,  0,  0,  0,  0
};
/* fixed square ordering: */
/* jcw's order, which is the best of 4 tried: */
static char worst2best[64] =
{
/*B2*/      20 , 25 , 65 , 70 ,
/*B1*/      11 , 16 , 19 , 26 , 64 , 71 , 74 , 79 ,
/*C2*/      21 , 24 , 29 , 34 , 56 , 61 , 66 , 69 ,
/*D2*/      22 , 23 , 38 , 43 , 47 , 52 , 67 , 68 ,
/*D3*/      31 , 32 , 39 , 42 , 48 , 51 , 58 , 59 ,
/*D1*/      13 , 14 , 37 , 44 , 46 , 53 , 76 , 77 ,
/*C3*/      30 , 33 , 57 , 60 ,
/*C1*/      12 , 15 , 28 , 35 , 55 , 62 , 75 , 78 ,
/*A1*/      10 , 17 , 73 , 80 , 
/*D4*/      40 , 41 , 49 , 50
};
// Set up according to worst2best, higher == better
/* --- Now not used ---
static char fixedGoodness[91] = {
  0, 0, 0, 0, 0, 0, 0, 0, 0,
  0, 9, 2, 8, 6, 6, 8, 2, 9,
  0, 2, 1, 3, 4, 4, 3, 1, 2,
  0, 8, 3, 7, 5, 5, 7, 3, 8,
  0, 6, 4, 5, 10,10,5, 4, 6,
  0, 6, 4, 5, 10,10,5, 4, 6,
  0, 8, 3, 7, 5, 5, 7, 3, 8,
  0, 2, 1, 3, 4, 4, 3, 1, 2,
  0, 9, 2, 8, 6, 6, 8, 2, 9,
  0, 0, 0, 0, 0, 0, 0, 0, 0, 0
}
*/
static EmptyList *emHead; // first (and dummy) node in the list
static char emSize;

/* Return the cumulative disc flips */
inline char flipEnd(char *a, char *placePointer, char dir, char color, char oppcolor, 
                char *flipped, char cumulativeFlips) {
  char count = cumulativeFlips;
  char *p = placePointer + dir;
  if(*p == oppcolor) {
    p += dir;
    if(*p == oppcolor) {
      p += dir;
      if(*p == oppcolor) {
        p += dir;
        if(*p == oppcolor) {
          p += dir;
          if(*p == oppcolor) {
            p += dir;
            if(*p == oppcolor) {
              p += dir;
            }
          }
        }
      }
    }
    if(*p == color) {
      p -= dir;
      while(p != placePointer) {
        *p = color;
        flipped[count] = p - a;
        count++;
        p -= dir;
      }
    }
  }
  // number of element in the current 'flipped' array minus previous count.
  return count;
}

/* Return the number discs flipped (91-square board) */
char tryMoveEnd(char *a, char color, char place, char *flipped, EmptyList *currNode) {
  if(DEBUG_MINIMAX) {
    printf("tryMoveEnd - place: %d, color %d\n", place, color);
  }
  char *p = a + place; // place pointer
  if(*p != EMPTY)
    return 0;
  char flipCount = 0;
  char oppcolor = OTHER(color);
  uchar dirs = dirmask91[place];
  if(dirs & 1) {
    flipCount = flipEnd(a, p, directionsEnd[0], color, oppcolor, flipped, flipCount);
  }
  if(dirs & 2) {
    flipCount = flipEnd(a, p, directionsEnd[1], color, oppcolor, flipped, flipCount);
  }
  if(dirs & 4) {
    flipCount = flipEnd(a, p, directionsEnd[2], color, oppcolor, flipped, flipCount);
  }
  if(dirs & 8) {
    flipCount = flipEnd(a, p, directionsEnd[3], color, oppcolor, flipped, flipCount);
  }
  if(dirs & 16) {
    flipCount = flipEnd(a, p, directionsEnd[4], color, oppcolor, flipped, flipCount);
  }
  if(dirs & 32) {
    flipCount = flipEnd(a, p, directionsEnd[5], color, oppcolor, flipped, flipCount);
  }
  if(dirs & 64) {
    flipCount = flipEnd(a, p, directionsEnd[6], color, oppcolor, flipped, flipCount);
  }
  if(dirs & 128) {
    flipCount = flipEnd(a, p, directionsEnd[7], color, oppcolor, flipped, flipCount);
  }
  if(flipCount) {
    *p = color;
    // remove the node from the list
    EmptyList *prev = currNode->prev;
    EmptyList *next = currNode->next;
    prev->next = next;
    if(next != NULL)
      next->prev = prev;
    emSize--;
  }
  if(DEBUG_MINIMAX) {
    printBoardArray(a);
    printf("flipCount: %d\n", flipCount);
  }
  return flipCount;
}

/* For 91-square board */
inline void undo_end(char *a, char color, char place, char *flipped, char nFlipped,
                    EmptyList *currNode) {
  a[place] = EMPTY;
  char otherColor = OTHER(color);
  for(char i=0; i<nFlipped; i++) {
    a[flipped[i]] = otherColor;
  }
  // insert this node back to the list
  currNode->prev->next = currNode;
  EmptyList *next = currNode->next;
  if(next != NULL)
    next->prev = currNode;
  if(nFlipped)
    emSize++;
  if(DEBUG_MINIMAX) {
    //printf("undo move: %d\n", place);
    //printBoardArray(a);
  }
}

/* Near the end of game, perfect move */
char getExactMove(char *a, char color, char selfPieces, char oppPieces, char *legalMoves, 
                  char nLegalMoves) {
  float alpha, beta;
  float bestValue, currValue;
  char bestMove;
  if(winLarge) {
    alpha = SMALL_FLOAT;
    beta = LARGE_FLOAT;
  }
  else if(loseSmall) {
    alpha = SMALL_FLOAT;
    beta = 1;
  }
  else {  // WDL only
    alpha = -1;
    beta = 1;
  }
  bestValue = SMALL_FLOAT;
  bestMove = PASS;
  // Initialize the 91-square board
  char b[91]; // board specified by the end of game solver
  for(int i=0; i<91; i++)
    b[i] = DUMMY;
  for(int y=0; y<8; y++) {
    for(int x=0; x<8; x++) {
      b[10+x+9*y] = a[CONV_21(x, y)];
    }
  }
  // init EmptyList -- best to worst order
  emHead = (EmptyList*)malloc(sizeof(EmptyList));
  emHead->prev = NULL;
  emHead->next = NULL;
  emSize = 0;
  EmptyList *currNode = emHead;
  // (Want to west first the move estimated to be best by move ordering)
  char firstMove = legalMoves[0];
  EmptyList *firstNode = emHead->next;
  for(int i=63; i>=0; i--) {
    char place = worst2best[i];
    if(b[place] == EMPTY) {
      EmptyList *newNode = (EmptyList*)malloc(sizeof(EmptyList));
      newNode->square = place;
      newNode->next = NULL;
      newNode->prev = currNode;
      currNode->next = newNode;
      currNode = newNode;
      emSize++;
      if(place == firstMove) {
        firstNode = newNode;
      }
    }
  }
  // Test first the move estimated to be best by move ordering
  
  // bug detection
  assert(selfPieces + oppPieces + emSize == 64);
  assert(selfPieces >= 0 && oppPieces >= 0);
  // Try for every emptie squares
  char oppcolor = OTHER(color);
  char flipped[20], nFlips;
  // Test firstNode first
  if(useMidGameOrdingInEndGame) {
    currNode = firstNode;
    char place = firstMove;
    nFlips = tryMoveEnd(b, color, place, flipped, currNode);
    if(showDots) { // show progress
      printf(".");
      fflush(stdout);
    }
    if(nFlips) {
      currValue = getMin_exact(b, place, oppcolor, selfPieces+nFlips+1, oppPieces-nFlips, 
                              alpha, beta);
      undo_end(b, color, place, flipped, nFlips, currNode);
      //printf("move: %d, value: %f\n", place, currValue);  // debug
      if(currValue > bestValue) {
        bestValue = currValue;
        bestMove = place;
      }
      if(bestValue > alpha)
        alpha = bestValue;
    }
  }
  // Try the rest moves
  currNode = emHead;
  for(int i=0; i<emSize; i++) {
    currNode = currNode->next;
    char place = currNode->square;
    if(useMidGameOrdingInEndGame && place == firstMove)
      continue;
    nFlips = tryMoveEnd(b, color, place, flipped, currNode);
    if(showDots) { // show progress
      printf(".");
      fflush(stdout);
    }
    if(nFlips) {
      currValue = getMin_exact(b, place, oppcolor, selfPieces+nFlips+1, oppPieces-nFlips, 
                              alpha, beta);
      undo_end(b, color, place, flipped, nFlips, currNode);
      //printf("move: %d, value: %f\n", place, currValue);  // debug
      if(currValue > bestValue) {
        bestValue = currValue;
        bestMove = place;
      }
      if(bestValue > alpha)
        alpha = bestValue;
      if(alpha >= beta)
        break;
    }
  }
  // free the memory taken by the EmptyList
  currNode = emHead;
  while(currNode != NULL) {
    EmptyList *nextNode = currNode->next;
    free(currNode);
    currNode = nextNode;
  }
  //printf("bestMove: %d\n", bestMove); // debug
  bestValue_global = bestValue;
  char result = CONV_21((bestMove-9) % 9 - 1, (bestMove-9) / 9);
  assert(selfPieces + oppPieces + emSize == 64);
  assert(selfPieces >= 0 && oppPieces >= 0);
  return result;
}

/* --- min part --- */
float getMin_exact(char *a, char lastMove, char color, char selfPieces, char oppPieces, 
                  float alpha, float beta) {
  if(COUNT_PRUNING && showStatistics)
    countSearching++;
  char uPieces = selfPieces + oppPieces;
  //assert(uPieces + nEmpties_end == 64);
  //assert(selfPieces >= 0 && oppPieces >= 0);
  char nFlips, flipped[20];
  char oppcolor = OTHER(color);
  char place;
  EmptyList *currNode;
  /* Only the last square left */
  if(uPieces == 63) {
    float score;
    currNode = emHead->next;
    place = currNode->square;
    nFlips = tryMoveEnd(a, color, place, flipped, currNode);
    if(nFlips) { // i.e. move can be played (by min) (min == opponent)
      score = evaluateEndGame_exact(selfPieces-nFlips, oppPieces+nFlips+1);
      undo_end(a, color, place, flipped, nFlips, currNode);
    }
    else { // see if the other player (max) can play that move
      nFlips = tryMoveEnd(a, oppcolor, place, flipped, currNode);
      if(nFlips) { // i.e. move can be played by the other player
        score = evaluateEndGame_exact(selfPieces+nFlips+1, oppPieces-nFlips);
        undo_end(a, oppcolor, place, flipped, nFlips, currNode);
      }
      else {  // no one can play this move
        score = evaluateEndGame_exact(selfPieces, oppPieces);
      }
    }
    return score;
  }
  /* search for every empty place */
  bool hasLegalMove = false;
  float minValue, currValue;
  minValue = LARGE_FLOAT;
  currNode = emHead;
  for(char i=0; i<emSize; i++) {
    currNode = currNode->next;
    place = currNode->square;
    nFlips = tryMoveEnd(a, color, place, flipped, currNode);
    if(nFlips) {
      hasLegalMove = true;
      currValue = getMax_exact(a, place, oppcolor, selfPieces-nFlips, oppPieces+nFlips+1,
                              alpha, beta);
      undo_end(a, color, place, flipped, nFlips, currNode);
      if(currValue < minValue)
        minValue = currValue;
      if(minValue < beta)
        beta = minValue;
      if(beta <= alpha) {
        if(COUNT_PRUNING && showStatistics)
          countPruning++;
        return beta;
      }
    }
  }
  /* if no legal move, see if should search on or not */
  if(!hasLegalMove) {
    if(lastMove == PASS) {
      return evaluateEndGame_exact(selfPieces, oppPieces);
    }
    else {
      return getMax_exact(a, PASS, oppcolor, selfPieces, oppPieces, alpha, beta);
    }
  }
  return minValue;
}

/* --- max part --- */
float getMax_exact(char *a, char lastMove, char color, char selfPieces, char oppPieces, 
                  float alpha, float beta) {
  if(COUNT_PRUNING && showStatistics)
    countSearching++;
  char uPieces = selfPieces + oppPieces;
  //assert(uPieces + nEmpties_end == 64);
  //assert(selfPieces >= 0 && oppPieces >= 0);
  char nFlips, flipped[20];
  char oppcolor = OTHER(color);
  char place;
  EmptyList *currNode;
  /* Only the last square left */
  if(uPieces == 63) {
    float score;
    currNode = emHead->next;
    place = currNode->square;
    nFlips = tryMoveEnd(a, color, place, flipped, currNode);
    if(nFlips) { // i.e. move can be played (by min) (min == opponent)
      score = evaluateEndGame_exact(selfPieces+nFlips+1, oppPieces-nFlips);
      undo_end(a, color, place, flipped, nFlips, currNode);
    }
    else { // see if the other player (max) can play that move
      nFlips = tryMoveEnd(a, oppcolor, place, flipped, currNode);
      if(nFlips) { // i.e. move can be played by the other player
        score = evaluateEndGame_exact(selfPieces-nFlips, oppPieces+nFlips+1);
        undo_end(a, oppcolor, place, flipped, nFlips, currNode);
      }
      else {  // no one can play this move
        score = evaluateEndGame_exact(selfPieces, oppPieces);
      }
    }
    return score;
  }
  /* search for every empty place */
  bool hasLegalMove = false;
  float maxValue, currValue;
  maxValue = SMALL_FLOAT;
  currNode = emHead;
  for(char i=0; i<emSize; i++) {
    currNode = currNode->next;
    place = currNode->square;
    nFlips = tryMoveEnd(a, color, place, flipped, currNode);
    if(nFlips) {
      hasLegalMove = true;
      currValue = getMin_exact(a, place, oppcolor, selfPieces+nFlips+1, oppPieces-nFlips,
                              alpha, beta);
      undo_end(a, color, place, flipped, nFlips, currNode);
      if(currValue > maxValue)
        maxValue = currValue;
      if(maxValue > alpha)
        alpha = maxValue;
      if(alpha >= beta) {
        if(COUNT_PRUNING && showStatistics)
          countPruning++;
        return alpha;
      }
    }
  }
  if(!hasLegalMove) {
    if(lastMove == PASS) {
      return evaluateEndGame_exact(selfPieces, oppPieces);
    }
    else {
      return getMin_exact(a, PASS, oppcolor, selfPieces, oppPieces, alpha, beta);
    }
  }
  return maxValue;
}

/* --- Return exact score --- */
inline int evaluateEndGame_exact(char selfPieces, char oppPieces) {
  if(COUNT_PRUNING && showStatistics)
    countEval++; 
  int score;
  if(selfPieces > oppPieces)
    score = 64 - oppPieces - oppPieces;
  else if(selfPieces < oppPieces)
    score = selfPieces + selfPieces - 64;
  else
    score = 0;
  //assert(selfPieces+oppPieces+nEmpties_end == 64); // debug
  //assert(selfPieces >= 0 && oppPieces >= 0);
  return score; // make sure a secured win out-weighs any estimated advantage.
}


/* Quiescence search, mainly for avoiding horizon effect in corner trade */
/* --- Not quite complete (Currently not used) --- 
static float getQuiescenceValueMax(char *a, char lastMove, char color, char prevmaxDOF,
                char prevminDOF, char selfPieces, char oppPieces, float alpha, float beta) {
  float result;
  float value1, value2;
  bool updated;
  char oppcolor;
  char nLegalMoves1, nLegalMoves2;
  char legalMoves1[64];
  char nFlips1, nFlips2;
  char flipped1[20], flipped2[20];
  // Init
  updated = false;
  oppcolor = OTHER(color);
  result = SMALL_FLOAT;
  nLegalMoves1 = findLegalMoves(a, color, legalMoves1);
  for(char i=0; i<nLegalMoves1; i++) { // getmax loop
    char place = legalMoves1[i];
    value1 = LARGE_FLOAT;
    bool updated2 = false; // whether value1 is updated in this iteration
    nFlips1 = tryMove(a, color, place, flipped1);
    bool searchOn = false;
    for(char j=0; j<4; j++) {
      if(legalMove(a, oppcolor, corners[j]))
        searchOn = true;
    }
    if(!searchOn) { // i.e. no further corner moves possible
      undo(a, color, place, flipped1, nFlips1);
      continue;
    }
    nLegalMoves2 = findLegalMoves(a, color);
    for(char j=0; j<4; j++) { // getmin loop
      char cornerPlace = corners[j];
      nFlips2 = tryMove(a, oppcolor, cornerPlace, flipped2);
      if(nFlips2) {
        updated = true;
        updated2 = true;
        value2 = evaluateBoard(a, color, color, nLegalMoves1, nLegalMoves2,
                    selfPieces+nFlips1+1-nFlips2, oppPieces-nFlips1+nFlips2+1);
        if(value2 < value1)
          value1 = value2;
        undo(a, oppcolor, cornerPlace, flipped2, nFlips2);
      }
    }
    undo(a, color, place, flipped1, nFlips1);
    if(updated2 && value1 > result)
      result = value1;
    if(result > alpha)
      alpha = result;
    if(alpha >= beta)
      break;
  }
  if(!updated) { // if already quiescent, i.e. no evaluation for further search.
    result = evaluateBoard(a, color, color, prevmaxDOF, prevminDOF, selfPieces, oppPieces);
  }
  assert(selfPieces+oppPieces+nEmpties == 64);
  return result;
}

// min part -- by changing prospect
static float getQuiescenceValueMin(char *a, char lastMove, char color, char prevmaxDOF,
                char prevminDOF, char selfPieces, char oppPieces, float alpha, float beta) {
  return 0 - getQuiescenceValueMax(a, lastMove, color, prevminDOF, prevmaxDOF, oppPieces, 
                                   selfPieces, 0-beta, 0-alpha);
}
*/

/* test /degug */
//int main() {
//  char b[64];
//  nEmpties = 0;
//  for(int i=0; i<64; i++) {
//    b[64] = EMPTY;
//  }
//  b[27] = WHITE;
//  b[28] = BLACK;
//  b[31] = BLACK;
//  b[32] = WHITE;
//  for(int i=0; i<64; i++) {
//    if(b[i] == EMPTY)
//      empties[nEmpties++] = i;
//  }
//  printBoardArray(b);
//  char lm[64], nlm;
//  nlm = findLegalMoves(b, BLACK, lm);
//  for(int i=0; i<nlm; i++)
//    printf("%d\n", lm[i]);
//  return 0;
//}




/******** Experiment code now not used ******************

#define MOBILITY_RATIO_EXPONENT 0.5 // with what factor M.R. affects corner weights.
#define RAW_SCORE_NORMALIZATION 0.05 // also used for corner weights.

static float situationFactor;  // adjucting corner weight
static float scoreCorrection;  // doesn't really affect the result of minimax

* Make a rough estimation of the situation, so as to see if the computer 
    should favor a trade of corners or be avert to it. *
  if(searchDepth == origialSearchDepth &&  // don't do this if brute-forcing
     searchDepth >= 8) { // only do this with > certain depth level
    searchDepth -= 4;  // shallow search estimate
    float estiScore, currScore1, currScore2;
    estiScore = getMax(lastMove, color, 0, 0, nLegalMoves, nLegalMoves, selfPieces, 
                        oppPieces, SMALL_FLOAT, LARGE_FLOAT);
    situationFactor = (float)pow(2, estiScore * RAW_SCORE_NORMALIZATION);
    currScore2 = getMax(lastMove, color, 0, 0, nLegalMoves, nLegalMoves, selfPieces, 
                        oppPieces, SMALL_FLOAT, LARGE_FLOAT);
    scoreCorrection = currScore1 - currScore2;
    searchDepth += 4;  // restore
  }


*/

/* --- Old Evaluation Function --- */
//float evaluateBoard__(char *a, char forWhom, char whoseTurn, char prevmaxDOF, 
//                    char prevminDOF, char selfPieces, char oppPieces) {
//  if(DEBUG_MINIMAX) {
//    printBoardArray(a);
//    printf("Depth: %d, eval. value: %e\n", selfPieces+oppPieces-base-4, 
//            (float)(selfPieces-oppPieces));
//  }
//  if(COUNT_PRUNING) {
//    countEval++;
//    countSearching++;
//  }
//  
//  char nLegalMoves = findLegalMoves(a, whoseTurn); // slow step.
//  if(forWhom == whoseTurn)
//    prevmaxDOF = nLegalMoves;
//  else
//    prevminDOF = nLegalMoves;
//  
//  // disc count unimportant during midd
//  float result = (0.01 * (selfPieces - oppPieces)) + 
//                (prevmaxDOF / (prevminDOF + extra));
//  /* A little simple account for corners */
//  char sign;
//  if(forWhom == BLACK)
//    sign = 1; 
//  else
//    sign = -1;
//  // BLACK = 1, WHITE = 2, EMPTY = 0
//  // Approach 1 -- About the same speed as Approch 2
//  char mean = EMPTY+1;
//  result += (((a[0]+1) % 3 - mean) * sign) * ESTIMATED_CORNER_WORTH;
//  result += (((a[7]+1) % 3 - mean) * sign) * ESTIMATED_CORNER_WORTH;
//  result += (((a[56]+1) % 3 - mean) * sign) * ESTIMATED_CORNER_WORTH;
//  result += (((a[63]+1) % 3 - mean) * sign) * ESTIMATED_CORNER_WORTH;
//  // */
//  /* // Approach 2 -- need to set sign = -2.0 for BLACK, 2.0 for WHITE
//  float mean = ((float)BLACK + (float)WHITE) * 0.5;
//  result += ((float)(a[0]) - mean) * (a[0] / (a[0] + 1.0e-20)) * sign * cornerWorth;
//  result += ((float)(a[7]) - mean) * (a[7] / (a[7] + 1.0e-20)) * sign * cornerWorth;
//  result += ((float)(a[56]) - mean) * (a[56] / (a[56] + 1.0e-20)) * sign * cornerWorth;
//  result += ((float)(a[63]) - mean) * (a[63] / (a[63] + 1.0e-20)) * sign * cornerWorth;
//  // */
//  return result;
//}

/*
          char lm[64];
          char nlm = findLegalMoves(a, OTHER(color), lm);
          for(char j=0; j<nlm; j++) {
            // This is in fact the getMax part.
            char cornerPlace = corners[j];
            char nfpd = tryMove(a, OTHER(color), cornerPlace, fpd);
            if(nfpd) {
              currValue = getMin(a, cornerPlace, color, depth+2, passes, nlm, nLegalMoves,
                                  alpha, beta);
              undo(a, OTHER(color), cornerPlace, fpd, nfpd);
            }
          }
*/
