/*******
  minimax.cpp
  This is the AI minimax search for Othello.
  
  Note: Andersson's end-of-game solver is very good at solving end of game 
   -- about 2 steps more than my simple minimax. 
  */

#include <math.h>
#include "minimax.hpp"

//#ifndef NDEBUG
//#define NDEBUG  // uncomment to disable 'assert()' debug
//#endif

static char mmBoard[64];
static char base;
static unsigned int countPruning;
static unsigned int countSearching, countEval;
static float extra;  // to avoid zero denominator
static float cornerValue;
static bool quiescenceSearch;

static const char directions[] = {1, -1, 7, -7, 8, -8, 9, -9};
// Indicate towards what directions a disc played into a certain place can make flipped
static const uchar dirmask[] = {
  81, 81, 87, 87, 87, 87, 22, 22,
  81, 81, 87, 87, 87, 87, 22, 22,
  121,121,255,255,255,255,182,182,
  121,121,255,255,255,255,182,182,
  121,121,255,255,255,255,182,182,
  121,121,255,255,255,255,182,182,
  41, 41, 171,171,171,171,162,162,
  41, 41, 171,171,171,171,162,162
};
// Bound tables for different directions
static const bool boundTables[3][64] = {
 {1, 1, 1, 1, 1, 1, 1, 1,
  1, 0, 0, 0, 0, 0, 0, 1,
  1, 0, 0, 0, 0, 0, 0, 1,
  1, 0, 0, 0, 0, 0, 0, 1,
  1, 0, 0, 0, 0, 0, 0, 1,
  1, 0, 0, 0, 0, 0, 0, 1,
  1, 0, 0, 0, 0, 0, 0, 1,
  1, 1, 1, 1, 1, 1, 1, 1},
  
 {1, 0, 0, 0, 0, 0, 0, 1,
  1, 0, 0, 0, 0, 0, 0, 1,
  1, 0, 0, 0, 0, 0, 0, 1,
  1, 0, 0, 0, 0, 0, 0, 1,
  1, 0, 0, 0, 0, 0, 0, 1,
  1, 0, 0, 0, 0, 0, 0, 1,
  1, 0, 0, 0, 0, 0, 0, 1,
  1, 0, 0, 0, 0, 0, 0, 1},
  
 {1, 1, 1, 1, 1, 1, 1, 1,
  0, 0, 0, 0, 0, 0, 0, 0,
  0, 0, 0, 0, 0, 0, 0, 0,
  0, 0, 0, 0, 0, 0, 0, 0,
  0, 0, 0, 0, 0, 0, 0, 0,
  0, 0, 0, 0, 0, 0, 0, 0,
  0, 0, 0, 0, 0, 0, 0, 0,
  1, 1, 1, 1, 1, 1, 1, 1}
};
// The index tells which of the 3 tables to use with a certain direction.
static const char dirBoundTableIndex[] = {1, 1, 0, 0, 2, 2, 0, 0};

static char empties[64];
static char nEmpties;
static char emptiesIndex[64];


/* Return a move using minimax search. -- Called only once per move made */
char getMinimaxMove(Board *board, bool *lm) {
  /* Initialization */
  countPruning = countSearching = countEval = 0;
  base = board->n;
  quiescenceSearch = DO_USE_QUIESCENCE_SEARCH;
  const char depth = 0; // constant within this function
  const char passes = 0;
  const char color = board->wt;
  float currValue;
  float bestValue = 2* SMALL_FLOAT; // avoid return a PASS while there is still legal moves
  char bestMove = PASS;
  float alpha = SMALL_FLOAT;
  float beta = LARGE_FLOAT;
  /* If end of game is within the brute force depth, search all the way to the end. */
  if(base + 4 + bruteForceDepth >= 64) {
    searchDepth = bruteForceDepth;
  }
  else {
    searchDepth = originalSearchDepth;
  }
  extra = DENOMINATOR_EXTRA_PREV_MIN_DOF;
  // Add a certain randomness to corner value 
  cornerValue = ESTIMATED_CORNER_WORTH;
  if(randomnessLevel) {
    float temprand = ((float)(rand() % 4001)) / 4001 - 0.5;  // +- 0.5
    cornerValue = ESTIMATED_CORNER_WORTH * (1 + 0.5 * temprand * randomnessLevel);
  }
  // initialize the arrays to the board configuration.
  char *a = board->a[base];
  char *b = mmBoard;
  copyBoardArray(b, a);
  if(base == 0) {  // don't want to search for the first move.
    if(boardFlipped)
      return HARD_WIRED_FIRST_MOVE_BOARD_FLIPPED;
    return HARD_WIRED_FIRST_MOVE;
  }
  // Find the empty sqares and create indices.
  nEmpties = 0;
  for(char i=0; i<64; i++) {
    if(b[i] == EMPTY) {
      empties[nEmpties] = i;
      emptiesIndex[i] = nEmpties;
      nEmpties++;
    }
    else {
      emptiesIndex[i] = -1;
    }
  }
  // get disc counts
  char selfPieces, oppPieces;
  if(color == BLACK)
    countPieces(b, &selfPieces, &oppPieces, base+depth+4);
  else
    countPieces(b, &oppPieces, &selfPieces, base+depth+4);
  // Find all legal moves
  char *legalMoves, legalMoves1[64], flipped[20], nFlips;
  char nLegalMoves = findLegalMoves(b, color, legalMoves1);
  legalMoves = legalMoves1;
  if(CURRENT_DEBUG) {
    printf("legalMoves for color %d:\n", color);
    for(int i=0; i<nLegalMoves; i++)
      printf("%d, ", legalMoves[i]);
    printf("\n");
  }
  /* Shallow search for move ordering and maybe MPC cutoff */
  char remainingDepth = searchDepth - depth;
  // may need to experiment different searchDepth - depth threshhold values.
  if(USE_MOVE_ORDERING && remainingDepth >= MIN_DEPTH_FOR_ORDERING && 
      base + depth <= 64 - MIN_EMPTIES_FOR_ORDERING) {
    // back-up variables to be changed in the shallow search.
    float oldAlpha = alpha, oldBeta = beta;
    char oldSearchDepth = searchDepth;
    char oldQuiescenceSearch = quiescenceSearch;
    // change some variables
    quiescenceSearch = false;
    char orderingSearchDepth = remainingDepth - ORDERING_SEARCH_DEPTH_OFFSET;
    orderingSearchDepth = orderingSearchDepth < MAX_ORDERING_SEARCH_DEPTH? 
              orderingSearchDepth : MAX_ORDERING_SEARCH_DEPTH;  // shallow search, with MAX
    searchDepth = depth + orderingSearchDepth;
    char orderedMoves[64];
    float orderedMoveValues[64];
    char index;
    for(int i=0; i<nLegalMoves; i++) {
      char place = legalMoves[i];
      assert(lm[place]);
      nFlips = tryMove(b, color, place, flipped);
      currValue = getMin(b, place, OTHER(color), depth+1, passes, nLegalMoves, nLegalMoves, 
                        selfPieces+nFlips+1, oppPieces-nFlips, alpha, beta);
      if(DEBUG_MINIMAX) {
        printf("getMin returned: %e, (depth: %d, in ordering search)\n", currValue, depth+1);
      }
      undo(b, color, place, flipped, nFlips);
      if(currValue > bestValue) {
        bestValue = currValue;
      }
      if(bestValue > alpha) {
        alpha = bestValue;
      }
      // keep the order, best to worst
      index = 0;
      while(index < i && currValue <= orderedMoveValues[index]) 
        index++;
      // insert this move, shift verything up
      for(int j=i; j>index; j--) {
        orderedMoves[j] = orderedMoves[j-1];
        orderedMoveValues[j] = orderedMoveValues[j-1];
      }
      orderedMoves[index] = place;
      orderedMoveValues[index] = currValue;
    }
    // restore values changed in the ordering search.
    alpha = oldAlpha;
    beta = oldBeta;
    quiescenceSearch = oldQuiescenceSearch;
    searchDepth = oldSearchDepth; 
    bestValue = 2* SMALL_FLOAT; // avoid return a PASS while there is still legal moves
    // update the legalMoves array (now point to the ordered moves).
    legalMoves = orderedMoves;
  }
//  printf("base: %d, depth: %d, searchDepth: %d, bruteForceDepth: %d\n",
//          base, depth, searchDepth, bruteForceDepth);  // debug
  /* try the ordered legal moves */
  for(int i=0; i<nLegalMoves; i++) {
    char place = legalMoves[i];
    assert(lm[place]);
    nFlips = tryMove(b, color, place, flipped);
    if(showDots) // show progress
      printf(".");
    if(base + depth + 4 == 63) {  // just filled the board.
      currValue = evaluateEndGame(selfPieces+nFlips+1, oppPieces-nFlips);
    }
    else if(base + 4 + bruteForceDepth >= 64) {
      if(useAndersson) {
        if(winLarge)
          currValue = strongEndGameSolve(b, OTHER(color), selfPieces+nFlips+1, 
                                        oppPieces-nFlips, 1, -64, 64);
        else if(loseSmall)
          currValue = strongEndGameSolve(b, OTHER(color), selfPieces+nFlips+1, 
                                        oppPieces-nFlips, 1, -64, 1);
        else
          currValue = strongEndGameSolve(b, OTHER(color), selfPieces+nFlips+1, 
                                        oppPieces-nFlips, 1, -1, 1);
      }
      else { // own end of game search
        if(winLarge)
          currValue = getMin(b, place, OTHER(color), depth+1, 0, nLegalMoves, nLegalMoves, 
                        selfPieces+nFlips+1, oppPieces-nFlips, alpha, beta);
        else if(loseSmall)
          currValue = getMin(b, place, OTHER(color), depth+1, 0, nLegalMoves, nLegalMoves, 
                        selfPieces+nFlips+1, oppPieces-nFlips, alpha, 1);
        else
          currValue = getMin(b, place, OTHER(color), depth+1, 0, nLegalMoves, nLegalMoves, 
                        selfPieces+nFlips+1, oppPieces-nFlips, -1, 1);
      }
    }
    else {
      currValue = getMin(b, place, OTHER(color), depth+1, 0, nLegalMoves, nLegalMoves, 
                        selfPieces+nFlips+1, oppPieces-nFlips, alpha, beta);
      if(DEBUG_MINIMAX || CURRENT_DEBUG2) {
        printf("getMin returned: %e, for move: %d (depth: %d)\n", currValue,
              place, depth+1);
      }
    }
    undo(b, color, place, flipped, nFlips);
    if(currValue > bestValue) {
      bestMove = place;
      bestValue = currValue;
    }
    if(bestValue > alpha)
      alpha = bestValue;
  }
  if(showDots)
    printf("\n");
  if(COUNT_PRUNING)
    printf("Nodes searched: %d, evaluated: %d, pruned: %d\n", countSearching, countEval, 
            countPruning);
  return bestMove;
}

/***************************************************************
                    The minimax core      
 *****************************************************************
 */

/**************************************************
  Function: getMin(...)
  ---- The MIN part of Minimax ---- 
  getMin always plays for "opponent"
*/
/* NOTE: passes = 1 if there has been odd number of passes on the search path, 
  zero otherwise (same in getMax) */
  
float getMin(char *a, char lastMove, char color, char depth, char passes, 
             char prevmaxDOF, char prevminDOF, //DOF found in previous getMax and getMin;
             char selfPieces, char oppPieces,
             float alpha, float beta) {
  if(DEBUG_MINIMAX || AB_DEBUG || CURRENT_DEBUG) {
    printf("In getMin, at depth %d -- alpha: %e, beta: %e\n", depth, alpha, beta);
  }
  if(COUNT_PRUNING)
    countSearching++;
  // Initialization started.
  uchar uPieces = base + depth + 4;
  assert(uPieces + nEmpties == 64);  // bug detection
  char place;
  char flipped[20], nFlips;
  /* If there is only one move left, simply player it (if possible) and count scores */
  if(uPieces == 63) {
    float score;
    place = empties[0];
    nFlips = tryMove(a, color, place, flipped);
    if(nFlips) { // i.e. move can be played (by min) (min == opponent)
      score = evaluateEndGame(selfPieces-nFlips, oppPieces+nFlips+1);
      undo(a, color, place, flipped, nFlips);
    }
    else { // see if the other player (max) can play that move
      nFlips = tryMove(a, OTHER(color), place, flipped);
      if(nFlips) { // i.e. move can be played by the other player
        score = evaluateEndGame(selfPieces+nFlips+1, oppPieces-nFlips);
        undo(a, OTHER(color), place, flipped, nFlips);
      }
      else {  // no one can play this move
        score = evaluateEndGame(selfPieces, oppPieces);
      }
    }
    return score;
  }
  /* Prepare for deeper search */
  char *legalMoves, legalMoves1[64], nLegalMoves;
  nLegalMoves = findLegalMoves(a, color, legalMoves1);
  legalMoves = legalMoves1;
  if(CURRENT_DEBUG) {
    printf("legalMoves for color %d:\n", color);
    for(int i=0; i<nLegalMoves; i++)
      printf("%d, ", legalMoves[i]);
    printf("\n");
  }
  /* test if there is any legal move posssible */
  if(!nLegalMoves) {
    float score;
    if(lastMove == PASS) // last step is also pass, this branch ends here.
      score = evaluateEndGame(selfPieces, oppPieces);
    else // make a pass (set lastx to -1), depth NOT incremented.
      score = getMax(a, PASS, OTHER(color), depth, 1-passes, prevmaxDOF, 0, 
                    selfPieces, oppPieces, alpha, beta);
    return score;
  }
  /* Now there are legal moves to make */
  float minValue = LARGE_FLOAT;
  float currValue;
  /* Shallow search for move ordering and maybe MPC cutoff */
  char remainingDepth = searchDepth - depth;
  // may need to experiment different searchDepth - depth threshhold values.
  if(USE_MOVE_ORDERING && remainingDepth >= MIN_DEPTH_FOR_ORDERING && 
      base + depth <= 64 - MIN_EMPTIES_FOR_ORDERING) {
    // back-up variables to be changed in the shallow search.
    float oldAlpha = alpha, oldBeta = beta;
    char oldSearchDepth = searchDepth;
    char oldQuiescenceSearch = quiescenceSearch;
    // change some variables
    quiescenceSearch = false;
    char orderingSearchDepth = remainingDepth - ORDERING_SEARCH_DEPTH_OFFSET;
    if(orderingSearchDepth > MAX_ORDERING_SEARCH_DEPTH)
      orderingSearchDepth = MAX_ORDERING_SEARCH_DEPTH;  // shallow search, with MAX
    searchDepth = depth + orderingSearchDepth;
    char orderedMoves[64];
    float orderedMoveValues[64];
    char index;
    for(int i=0; i<nLegalMoves; i++) {
      char place = legalMoves[i];
      nFlips = tryMove(a, color, place, flipped);
      // can never reach cut-off depth here.
      currValue = getMax(a, place, OTHER(color), depth+1, passes, prevmaxDOF, nLegalMoves, 
                        selfPieces-nFlips, oppPieces+nFlips+1, alpha, beta);
      undo(a, color, place, flipped, nFlips);
      if(currValue < minValue) {
        minValue = currValue;
      }
      if(minValue < beta) {
        beta = minValue;
      }
      // keep the order, best to worst (small to large, since in getmin)
      index = 0;
      while(index < i && currValue >= orderedMoveValues[index]) 
        index++;
      // insert this move, shift verything up
      for(int j=i; j>index; j--) {
        orderedMoves[j] = orderedMoves[j-1];
        orderedMoveValues[j] = orderedMoveValues[j-1];
      }
      orderedMoves[index] = place;
      orderedMoveValues[index] = currValue;
    }
    // restore values changed in the ordering search.
    alpha = oldAlpha;
    beta = oldBeta;
    quiescenceSearch = oldQuiescenceSearch;
    searchDepth = oldSearchDepth; 
    minValue = LARGE_FLOAT;
    // update the legalMoves array (now point to the ordered moves).
    legalMoves = orderedMoves;
  }
  if(AB_DEBUG) {
    printf("after ordering, at depth %d -- alpha: %e, beta: %e\n", depth, alpha, beta);
  }
  /* try the ordered legal moves (or unordered moves if ordering didn't happen) */
  for(int i=0; i<nLegalMoves; i++) {
    char place = legalMoves[i];
    nFlips = tryMove(a, color, place, flipped);
    /* test if cut-off depth will be reached. If so, avoid an extra recursive call. */
    if(depth + 1 >= searchDepth + passes) // always evaluate when it is a certain side's turn
      currValue = evaluateBoard(a, OTHER(color), OTHER(color), prevmaxDOF, nLegalMoves, 
                                selfPieces-nFlips, oppPieces+nFlips+1);
    else
      currValue = getMax(a, place, OTHER(color), depth+1, passes, prevmaxDOF, nLegalMoves, 
                        selfPieces-nFlips, oppPieces+nFlips+1, alpha, beta);
    if(DEBUG_MINIMAX || CURRENT_DEBUG) {
      printf("getMax returned: %e, for move: %d. (depth: %d)\n", currValue, 
            legalMoves[i], depth+1);
    }
    undo(a, color, place, flipped, nFlips);
    if(currValue < minValue)
      minValue = currValue;
    if(minValue < beta)
      beta = minValue;
    if(alpha >= beta) {
      if(COUNT_PRUNING)
        countPruning++;
      return beta;
    }
  }
  return minValue;
}

/***********************************************/
  /**************************************************
  Function: getMax(...)
  ---- The MAX par of Minimax ---- 
  getMax always plays for "self"
*/
float getMax(char *a, char lastMove, char color, char depth, char passes, 
             char prevmaxDOF, char prevminDOF, 
             char selfPieces, char oppPieces,
             float alpha, float beta) {
  if(DEBUG_MINIMAX || AB_DEBUG || CURRENT_DEBUG) {
    printf("In getMax, at depth %d -- alpha: %e, beta: %e\n", depth, alpha, beta);
    // printf("DEBUG_MINIMAX: %d", DEBUG_MINIMAX);
  }
  if(COUNT_PRUNING)
    countSearching++;
  // Initialization started.
  uchar uPieces = base + depth + 4;
  assert(uPieces + nEmpties == 64);  // bug detection
  char place;
  char flipped[20], nFlips;
  /* If there is only one move left, simply player it (if possible) and count scores */
  if(uPieces == 63) {
    float score;
    place = empties[0];
    nFlips = tryMove(a, color, place, flipped);
    if(nFlips) { // i.e. move can be played (by min) (min == opponent)
      score = evaluateEndGame(selfPieces+nFlips+1, oppPieces-nFlips);
      undo(a, color, place, flipped, nFlips);
    }
    else { // see if the other player (max) can play that move
      nFlips = tryMove(a, OTHER(color), place, flipped);
      if(nFlips) { // i.e. move can be played by the other player
        score = evaluateEndGame(selfPieces-nFlips, oppPieces+nFlips+1);
        undo(a, OTHER(color), place, flipped, nFlips);
      }
      else {  // no one can play this move
        score = evaluateEndGame(selfPieces, oppPieces);
      }
    }
    return score;
  }
  /* Prepare for deeper search */
  char *legalMoves, legalMoves1[64], nLegalMoves;
  nLegalMoves = findLegalMoves(a, color, legalMoves1);
  legalMoves = legalMoves1;
  if(CURRENT_DEBUG) {
    printf("legalMoves for color %d:\n", color);
    for(int i=0; i<nLegalMoves; i++)
      printf("%d, ", legalMoves[i]);
    printf("\n");
  }
  /* test if there is any legal move posssible */
  if(!nLegalMoves) {
    float score;
    if(lastMove == PASS) // last step is also pass, this branch ends here.
      score = evaluateEndGame(selfPieces, oppPieces);
    else // make a pass (set lastx to -1), depth NOT incremented.
      score = getMin(a, PASS, OTHER(color), depth, 1-passes, 0, prevminDOF, 
                    selfPieces, oppPieces, alpha, beta);
    return score;
  }
  /* Now there are legal moves to make */
  float maxValue = SMALL_FLOAT;
  float currValue;
  /* Shallow search for move ordering and maybe MPC cutoff */
  char remainingDepth = searchDepth - depth;
  // may need to experiment different searchDepth - depth threshhold values.
  if(USE_MOVE_ORDERING && remainingDepth >= MIN_DEPTH_FOR_ORDERING && 
      base + depth <= 64 - MIN_EMPTIES_FOR_ORDERING) { 
              // at least 10 moves left
    // back-up variables to be changed in the shallow search.
    float oldAlpha = alpha, oldBeta = beta;
    char oldSearchDepth = searchDepth;
    char oldQuiescenceSearch = quiescenceSearch;
    // change some variables
    quiescenceSearch = false;
    char orderingSearchDepth = remainingDepth - ORDERING_SEARCH_DEPTH_OFFSET;
    if(orderingSearchDepth > MAX_ORDERING_SEARCH_DEPTH)
      orderingSearchDepth = MAX_ORDERING_SEARCH_DEPTH;  // shallow search, with MAX
    searchDepth = depth + orderingSearchDepth;
    char orderedMoves[64];
    float orderedMoveValues[64];
    char index;
    for(int i=0; i<nLegalMoves; i++) {
      char place = legalMoves[i];
      nFlips = tryMove(a, color, place, flipped);
      // can never reach cut-off depth here.
      currValue = getMin(a, place, OTHER(color), depth+1, passes, nLegalMoves, prevminDOF, 
                        selfPieces+nFlips+1, oppPieces-nFlips, alpha, beta);
      undo(a, color, place, flipped, nFlips);
      if(currValue > maxValue) {
        maxValue = currValue;
      }
      if(maxValue > alpha) {
        alpha = maxValue;
      }
      // keep the order, best to worst (small to large, since in getmin)
      index = 0;
      while(index < i && currValue <= orderedMoveValues[index]) 
        index++;
      // insert this move, shift verything up
      for(int j=i; j>index; j--) {
        orderedMoves[j] = orderedMoves[j-1];
        orderedMoveValues[j] = orderedMoveValues[j-1];
      }
      orderedMoves[index] = place;
      orderedMoveValues[index] = currValue;
    }
    // restore values changed in the ordering search.
    alpha = oldAlpha;
    beta = oldBeta;
    quiescenceSearch = oldQuiescenceSearch;
    searchDepth = oldSearchDepth; 
    maxValue = SMALL_FLOAT;
    // update the legalMoves array (now point to the ordered moves).
    legalMoves = orderedMoves;
  }
  if(AB_DEBUG) {
    printf("After ordering, at depth %d -- alpha: %e, beta: %e\n", depth, alpha, beta);
  }
  /* try the ordered legal moves (or unordered moves if ordering didn't happen) */
  for(int i=0; i<nLegalMoves; i++) {
    char place = legalMoves[i];
    nFlips = tryMove(a, color, place, flipped);
    /* test if cut-off depth will be reached. If so, avoid an extra recursive call. */
    if(depth + 1 >= searchDepth + passes) // always evaluate when it is a certain side's turn
      currValue = evaluateBoard(a, color, OTHER(color), nLegalMoves, prevminDOF, 
                                selfPieces+nFlips+1, oppPieces-nFlips);
    else
      currValue = getMin(a, place, OTHER(color), depth+1, passes, nLegalMoves, prevminDOF, 
                        selfPieces+nFlips+1, oppPieces-nFlips, alpha, beta);
    if(DEBUG_MINIMAX || CURRENT_DEBUG) {
      printf("getMin returned: %e, for move: %d. (depth: %d)\n", currValue,
            legalMoves[i], depth+1);
    }
    undo(a, color, place, flipped, nFlips);
    if(currValue > maxValue)
      maxValue = currValue;
    if(maxValue > alpha)
      alpha = maxValue;
    if(alpha >= beta) {
      if(COUNT_PRUNING)
        countPruning++;
      return alpha;
    }
  }
  return maxValue;
}

/************************ End of Minimax core ***********************/


/* copy a board array */
void copyBoardArray(char *to, char *from) {
  for(int i=0; i<60; i+=6) { // some loop unrolling
    to[i] = from[i];
    to[i+1] = from[i+1];
    to[i+2] = from[i+2];
    to[i+3] = from[i+3];
    to[i+4] = from[i+4];
    to[i+5] = from[i+5];
  }
  to[60] = from[60];
  to[61] = from[61];
  to[62] = from[62];
  to[63] = from[63];
}

/* test if a move is legal one on a board array for the given color */
bool legalMove(char *a, char color, char place) {
  if(a[place] != EMPTY)
    return false;
  /* test left for possible flips */
  bool result = false;
  char oppcolor = OTHER(color);
  char dirs = dirmask[place];
  if(dirs & 1) {
    result = canFlip(a, place, 0, color, oppcolor);
    if(result)
      return result;
  }
  if(dirs & 2) {
    result = canFlip(a, place, 1, color, oppcolor);
    if(result)
      return result;
  }
  if(dirs & 4) {
    result = canFlip(a, place, 2, color, oppcolor);
    if(result)
      return result;
  }
  if(dirs & 8) {
    result = canFlip(a, place, 3, color, oppcolor);
    if(result)
      return result;
  }
  if(dirs & 16) {
    result = canFlip(a, place, 4, color, oppcolor);
    if(result)
      return result;
  }
  if(dirs & 32) {
    result = canFlip(a, place, 5, color, oppcolor);
    if(result)
      return result;
  }
  if(dirs & 64) {
    result = canFlip(a, place, 6, color, oppcolor);
    if(result)
      return result;
  }
  if(dirs & 128) {
    result = canFlip(a, place, 7, color, oppcolor);
    if(result)
      return result;
  }
  return result;
}

/* test if flip in one direction is possible */
inline bool canFlip(char *a, char place, char dirIndex, char color, char oppcolor) {
  const bool *atBound = boundTables[dirBoundTableIndex[dirIndex]];
  char dir = directions[dirIndex];
  char p = place + dir;
  if(a[p] == oppcolor && !atBound[p]) {
    p += dir;
    if(a[p] == oppcolor && !atBound[p]) {
      p += dir;
      if(a[p] == oppcolor && !atBound[p]) {
        p += dir;
        if(a[p] == oppcolor && !atBound[p]) {
          p += dir;
          if(a[p] == oppcolor && !atBound[p]) {
            p += dir;
            if(a[p] == oppcolor && !atBound[p]) {
              p += dir;
            }
          }
        }
      }
    }
    if(a[p] == color)
      return true;
  }
  return false;
}

/* flip all the discs in a certain direction, played at 'place'. return the 
  count of discs flipped. 'flipCount' is used as the entry index for the 
  'flipped' array. */
inline char flip(char *a, char place, char dirIndex, char color, char oppcolor, 
                char *flipped, char flipCount) {
  const bool *atBound = boundTables[dirBoundTableIndex[dirIndex]];
  char dir = directions[dirIndex];
  char count = flipCount;
  char p = place + dir;
  if(a[p] == oppcolor && !atBound[p]) {
    p += dir;
    if(a[p] == oppcolor && !atBound[p]) {
      p += dir;
      if(a[p] == oppcolor && !atBound[p]) {
        p += dir;
        if(a[p] == oppcolor && !atBound[p]) {
          p += dir;
          if(a[p] == oppcolor && !atBound[p]) {
            p += dir;
            if(a[p] == oppcolor && !atBound[p]) {
              p += dir;
            }
          }
        }
      }
    }
    if(a[p] == color) {
      p -= dir;
      while(p != place) {
        a[p] = color;
        flipped[count] = p;
        count++;
        p -= dir;
      }
    }
  }
  // number of element in the current 'flipped' array minus previous count.
  return count - flipCount;
}

/* find all the legal moves, stored as bitmask of two ints. Returns the number 
  of legalMoves found */
char findLegalMoves(char *a, char color, char *legalMoves) {
  char count = 0;
  char *emp = empties, nEmp = nEmpties;
  for(char i=0; i<nEmp; i++) {
    if(legalMove(a, color, emp[i])) {
      legalMoves[count] = emp[i];
      count++;
    }
  }
  return count;
}

/* find the number of legalmoves without storing them, only want the count */
char findLegalMoves(char *a, char color) {
  char count = 0;
  char *emp = empties, nEmp = nEmpties;
  for(char i=0; i<nEmp; i++)
    count += legalMove(a, color, emp[i]);
  return count;
}

/* find the number of legalmoves without storing them, only want the count and 
  the corner moves (also find the number of corner moves). */
char findLegalMoves(char *a, char color, char *cornerMoves, char *nCornerMoves) {
  char count = 0;
  char countCM = 0;
  char *emp = empties, nEmp = nEmpties;
  for(char i=0; i<nEmp; i++) {
    if(legalMove(a, color, emp[i])) {
      count++;
      if(i == 0 || i == 7 || i == 56 || i == 63) {
        cornerMoves[countCM++] = i;
      }
    }
  }
  *nCornerMoves = countCM;
  return count;
}

/* try a move on the minimax's board. Return the number of pieces flipped. 
  in this whole project, tryMove is only called after we know this move to 
  be made is legal. */
char tryMove(char *a, char color, char place, char *flipped) {
  if(DEBUG_MINIMAX) {
    printf("tryMove - place: %d, color %d\n", place, color);
  }
  if(a[place] != EMPTY)
    return 0;
  char flipCount = 0;
  char oppcolor = OTHER(color);
  char dirs = dirmask[place];
  if(dirs & 1) {
    flipCount += flip(a, place, 0, color, oppcolor, flipped, flipCount);
  }
  if(dirs & 2) {
    flipCount += flip(a, place, 1, color, oppcolor, flipped, flipCount);
  }
  if(dirs & 4) {
    flipCount += flip(a, place, 2, color, oppcolor, flipped, flipCount);
  }
  if(dirs & 8) {
    flipCount += flip(a, place, 3, color, oppcolor, flipped, flipCount);
  }
  if(dirs & 16) {
    flipCount += flip(a, place, 4, color, oppcolor, flipped, flipCount);
  }
  if(dirs & 32) {
    flipCount += flip(a, place, 5, color, oppcolor, flipped, flipCount);
  }
  if(dirs & 64) {
    flipCount += flip(a, place, 6, color, oppcolor, flipped, flipCount);
  }
  if(dirs & 128) {
    flipCount += flip(a, place, 7, color, oppcolor, flipped, flipCount);
  }
  if(flipCount) {
    a[place] = color;
    char placeIndex = emptiesIndex[place];
    assert(placeIndex >= 0);
    char lastEmpty = empties[--nEmpties]; // nEmpties decremented
    empties[placeIndex] = lastEmpty; // remove from empty list 'place'
    emptiesIndex[lastEmpty] = placeIndex;
    emptiesIndex[place] = -1;  // NOT necessary, only for bug detection. (for assert())
  }
  if(DEBUG_MINIMAX) {
    printBoardArray(a);
    printf("flipCount: %d\n", flipCount);
  }
  return flipCount;
}

/* undo a move previously made by 'color' */
inline void undo(char *a, char color, char place, char *flipped, char nFlipped) {
  empties[nEmpties] = place;
  assert(nFlipped > 0);
  assert(emptiesIndex[place] == -1);
  emptiesIndex[place] = nEmpties;
  nEmpties++;
  a[place] = EMPTY;
  char otherColor = OTHER(color);
  for(char i=0; i<nFlipped; i++) {
    a[flipped[i]] = otherColor;
  }
  if(DEBUG_MINIMAX) {
    printf("undo move: %d\n", place);
    printBoardArray(a);
  }
}


/* Premitive evaluation functions -- not good at all, just to make game going */
/* Evaluate an end-of-game situation (just count) */
inline int evaluateEndGame(char selfPieces, char oppPieces) {
  if(COUNT_PRUNING)
    countEval++;
  int score;
  if(selfPieces > oppPieces)
    score = 64 - oppPieces - oppPieces;
  else if(selfPieces < oppPieces)
    score = selfPieces + selfPieces - 64;
  else
    score = 0;
  return score << 10; // make sure a secured win out-weighs any estimated advantage.
}

/* Only ask if it is Win, Draw or Loss, > 0 means win. */
inline int evaluateEndGameWDL(char selfPieces, char oppPieces) {
  return selfPieces - oppPieces;
}

/* Evaluate an board situation (game still in progress) */
float evaluateBoard(char *a, char forWhom, char whoseTurn, char prevmaxDOF, 
                    char prevminDOF, char selfPieces, char oppPieces) {
  if(DEBUG_MINIMAX) {
    printBoardArray(a);
    printf("Depth: %d, disc diff. value: %e\n", selfPieces+oppPieces-base-4, 
            (float)(selfPieces-oppPieces));
  }
  if(COUNT_PRUNING) {
    countEval++;
    countSearching++;
  }
  
  char nLegalMoves = findLegalMoves(a, whoseTurn); // slow step!
  // in mamimax, always max for self, min for opponent.
  if(forWhom == whoseTurn)
    prevmaxDOF = nLegalMoves;
  else
    prevminDOF = nLegalMoves;
  
  /* --------------- 
    Quiescence search to be implemented ! 
  
    ------------------*/
    
  // */   
  float mobilityRatio = ((prevmaxDOF + extra) / (prevminDOF + extra));
  // disc count unimportant during midd
  float result = (0.001 * (selfPieces - oppPieces)) + mobilityRatio - 1/mobilityRatio;
  /* A little simple account for corners */
  /* If in advantage, we are not too keen to capture a corner immediately, since
    it is likely to be ours anyway; but we are very avert to let opponent 
    a corner since we would probably be able to prevent that. And intuitively 
    it is the opposite when in disadvantage. */
  // when mobility is high cornerValue is effective devaluated, and vice versa
  /* ! -- The above idea is currently NOT reflected in the following evaluation! --*/
  float selfCorner = cornerValue;
  float oppCorner = cornerValue;
  result += (a[0] == forWhom) * selfCorner;
  result -= (a[0] == OTHER(forWhom)) * oppCorner;
  result += (a[7] == forWhom) * selfCorner;
  result -= (a[7] == OTHER(forWhom)) * oppCorner;
  result += (a[56] == forWhom) * selfCorner;
  result -= (a[56] == OTHER(forWhom)) * oppCorner;
  result += (a[63] == forWhom) * selfCorner;
  result -= (a[63] == OTHER(forWhom)) * oppCorner;
  
  /* The following doesn't change anything within a getMinimax call, and hence 
    has no effect on the choice of moves. Howver it makes 
    the score more consistant throughout the game. */
  // result += scoreCorrection; 
  
  //char sign;
//  if(forWhom == BLACK)
//    sign = 1; 
//  else
//    sign = -1;
//  // BLACK = 1, WHITE = 2, EMPTY = 0
//  // Approach 1 -- About the same speed as Approch 2
//  char mean = EMPTY+1;
//  result += (((a[0]+1) % 3 - mean) * sign) * ESTIMATED_CORNER_WORTH;
//  result += (((a[7]+1) % 3 - mean) * sign) * ESTIMATED_CORNER_WORTH;
//  result += (((a[56]+1) % 3 - mean) * sign) * ESTIMATED_CORNER_WORTH;
//  result += (((a[63]+1) % 3 - mean) * sign) * ESTIMATED_CORNER_WORTH;
  // */
  /* // Approach 2 -- need to set sign = -2.0 for BLACK, 2.0 for WHITE
  float mean = ((float)BLACK + (float)WHITE) * 0.5;
  result += ((float)(a[0]) - mean) * (a[0] / (a[0] + 1.0e-20)) * sign * cornerWorth;
  result += ((float)(a[7]) - mean) * (a[7] / (a[7] + 1.0e-20)) * sign * cornerWorth;
  result += ((float)(a[56]) - mean) * (a[56] / (a[56] + 1.0e-20)) * sign * cornerWorth;
  result += ((float)(a[63]) - mean) * (a[63] / (a[63] + 1.0e-20)) * sign * cornerWorth;
  // */
  return result;
}

/* Print out the board array -- for debugging purpose */
void printBoardArray(char *a) {
  char place;
  printf("\n   0 1 2 3 4 5 6 7\n");
  for(char y=0; y<8; y++) {
    printf("%d  ", y);
    for(char x=0; x<8; x++) {
      place = CONV_21(x, y);
      if(a[place] == BLACK)
        printf("X ");
      else if(a[place] == WHITE)
        printf("O ");
      else
        printf(". ");
    }
    printf("\n");
  }
  printf("\n");
}

/* Uses Andersson's complicated end of game solver 
  Here 'color' is the opponent color as viewed by Minimax, but is 'self' 
  as viewed by EndSolve. */
  /* NOTE: In core minimax, alpha is for max and beta for min, but here 
   since EndSolve is actually evaluating for Min but return a positve
   value (and we invert that value as return value), both alpha and beta must
   be swapped and then inverted (0 - ..) before passing to EndSolve. */
int strongEndGameSolve(char *a, char color, char selfPieces, char oppPieces, 
                       char prevNotPass, float alpha, float beta) {
  int result;
  int nEmpt = 64 - selfPieces - oppPieces;
  int difference = oppPieces - selfPieces;
  int ecolor = color == BLACK? END_BLACK : END_WHITE;
  uchar eboard[91]; // board specified by the end of game solver
  for(int i=0; i<91; i++)
    eboard[i] = DUMMY;
  for(int y=0; y<8; y++)
    for(int x=0; x<8; x++) {
      if(a[CONV_21(x, y)] == BLACK)
        eboard[10+x+9*y] = END_BLACK;
      else if(a[CONV_21(x, y)] == WHITE)
        eboard[10+x+9*y] = END_WHITE;
      else
        eboard[10+x+9*y] = END_EMPTY;
    }
  float end_alpha = 0 - beta;
  float end_beta = 0 - alpha;
  PrepareToSolve(eboard);
  result = EndSolve(eboard, end_alpha, end_beta, ecolor, nEmpt, difference, prevNotPass);
  return 0 - result; // since result is as viewed by opponent.
}

/* test /degug */
//int main() {
//  char b[64];
//  nEmpties = 0;
//  for(int i=0; i<64; i++) {
//    b[64] = EMPTY;
//  }
//  b[27] = WHITE;
//  b[28] = BLACK;
//  b[31] = BLACK;
//  b[32] = WHITE;
//  for(int i=0; i<64; i++) {
//    if(b[i] == EMPTY)
//      empties[nEmpties++] = i;
//  }
//  printBoardArray(b);
//  char lm[64], nlm;
//  nlm = findLegalMoves(b, BLACK, lm);
//  for(int i=0; i<nlm; i++)
//    printf("%d\n", lm[i]);
//  return 0;
//}




/******** Experiment code now not used ******************

#define MOBILITY_RATIO_EXPONENT 0.5 // with what factor M.R. affects corner weights.
#define RAW_SCORE_NORMALIZATION 0.05 // also used for corner weights.

static float situationFactor;  // adjucting corner weight
static float scoreCorrection;  // doesn't really affect the result of minimax

* Make a rough estimation of the situation, so as to see if the computer 
    should favor a trade of corners or be avert to it. *
  if(searchDepth == origialSearchDepth &&  // don't do this if brute-forcing
     searchDepth >= 8) { // only do this with > certain depth level
    searchDepth -= 4;  // shallow search estimate
    float estiScore, currScore1, currScore2;
    estiScore = getMax(lastMove, color, 0, 0, nLegalMoves, nLegalMoves, selfPieces, 
                        oppPieces, SMALL_FLOAT, LARGE_FLOAT);
    situationFactor = (float)pow(2, estiScore * RAW_SCORE_NORMALIZATION);
    currScore2 = getMax(lastMove, color, 0, 0, nLegalMoves, nLegalMoves, selfPieces, 
                        oppPieces, SMALL_FLOAT, LARGE_FLOAT);
    scoreCorrection = currScore1 - currScore2;
    searchDepth += 4;  // restore
  }


*/

float evaluateBoard__(char *a, char forWhom, char whoseTurn, char prevmaxDOF, 
                    char prevminDOF, char selfPieces, char oppPieces) {
  if(DEBUG_MINIMAX) {
    printBoardArray(a);
    printf("Depth: %d, eval. value: %e\n", selfPieces+oppPieces-base-4, 
            (float)(selfPieces-oppPieces));
  }
  if(COUNT_PRUNING) {
    countEval++;
    countSearching++;
  }
  
  char nLegalMoves = findLegalMoves(a, whoseTurn); // slow step.
  if(forWhom == whoseTurn)
    prevmaxDOF = nLegalMoves;
  else
    prevminDOF = nLegalMoves;
  // */   
  // disc count unimportant during midd
  float result = (0.01 * (selfPieces - oppPieces)) + 
                (prevmaxDOF / (prevminDOF + extra));
  /* A little simple account for corners */
  char sign;
  if(forWhom == BLACK)
    sign = 1; 
  else
    sign = -1;
  // BLACK = 1, WHITE = 2, EMPTY = 0
  // Approach 1 -- About the same speed as Approch 2
  char mean = EMPTY+1;
  result += (((a[0]+1) % 3 - mean) * sign) * ESTIMATED_CORNER_WORTH;
  result += (((a[7]+1) % 3 - mean) * sign) * ESTIMATED_CORNER_WORTH;
  result += (((a[56]+1) % 3 - mean) * sign) * ESTIMATED_CORNER_WORTH;
  result += (((a[63]+1) % 3 - mean) * sign) * ESTIMATED_CORNER_WORTH;
  // */
  /* // Approach 2 -- need to set sign = -2.0 for BLACK, 2.0 for WHITE
  float mean = ((float)BLACK + (float)WHITE) * 0.5;
  result += ((float)(a[0]) - mean) * (a[0] / (a[0] + 1.0e-20)) * sign * cornerWorth;
  result += ((float)(a[7]) - mean) * (a[7] / (a[7] + 1.0e-20)) * sign * cornerWorth;
  result += ((float)(a[56]) - mean) * (a[56] / (a[56] + 1.0e-20)) * sign * cornerWorth;
  result += ((float)(a[63]) - mean) * (a[63] / (a[63] + 1.0e-20)) * sign * cornerWorth;
  // */
  return result;
}

