/***********************************************************************
 * GameDriver.java
 *
 * A class for multi-threaded game playing (so computer move doesn't freeze
 * the GUI.
 *
 * Any potentially blocking operation (CPU-bound or calling a synchronized
 * method) should start a new thread using this class. Should always check
 * game.canPlay() before start a new thread.
 *
 * Yunpeng Li
 *
 */

package othelloGUI;

import java.io.*;
import java.awt.*;
import java.awt.event.*;
import java.util.*;
import java.text.*;
import java.applet.Applet;
import javax.swing.*;

public class GameDriver extends Thread {
    public static final int SELF_PLAY = 1;
    public static final int MAKE_COM_MOVE = 2;
    public static final int COM_HELP_MAKE_MOVE = 3;
    public static final int COM_HELP_MAKE_PASS = 4;
    public static final int PLAY_HUMAN_MOVE = 5;
    public static final int MAKE_HUMAN_PASS = 6;
    public static final int UNDO = 7;
    public static final int REDO = 8;
    public static final int UNDO_ALL = 9;
    public static final int REDO_ALL = 10;

    public static final int NULL_STATE = 0;

    private int state;
    private Board board;
    private Game game;
    private OthelloFrame frame;

    private int x, y;

    public GameDriver(OthelloFrame _frame, Game _game, Board _board) {
        game = _game;
        board = _board;
        frame = _frame;
        state = NULL_STATE;
    }

    // The start() method should be called to invoke this thread whenever there
    // is computer move involved in the subsequent operations
    public void run() {
        //System.out.println("run."); // debug
        //frame.disableButtons(); // actually no use here, they get updated in paint()
        switch(state) {
            case SELF_PLAY:
                game.selfPlay();
                break;
            case MAKE_COM_MOVE:
                game.makeComMove();
                break;
            case COM_HELP_MAKE_MOVE:
                game.makeComMove();
                if(game.player[game.getCurrPlayer()] == Game.COMPUTER_PLAYER)
                    game.makeComMove();
                break;
            case COM_HELP_MAKE_PASS:
                game.makePass(); // so knows the other play can move after this pass
                if(game.player[game.getCurrPlayer()] == Game.COMPUTER_PLAYER)
                    game.makeComMove();
                break;
            case PLAY_HUMAN_MOVE:
                game.playMove(x, y);
                break;
            case MAKE_HUMAN_PASS:
                game.makePass();
                break;
            case UNDO:
                game.undo();
                break;
            case REDO:
                game.redo();
                break;
            case UNDO_ALL:
                game.undoAll();
                break;
            case REDO_ALL:
                game.redoAll();
                break;
            default:
                System.out.println("Unkown state (or NULL) state in GameDriver");
        }
        state = NULL_STATE;
    }

    // needed when human make a move -- need to know where it is
    public void setCoord(int x, int y) {
        this.x = x;
        this.y = y;
    }

    // access methods
    public void setState(int s) {
        state = s;
    }
}

