/* 
 *  Copyright (c) 2008  Noah Snavely (snavely (at) cs.washington.edu)
 *    and the University of Washington
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 */

/* keys.h */
/* Class for SIFT keypoints */

#ifndef __keys_h__
#define __keys_h__

#include <vector>
#include <stdio.h>
#include <zlib.h>



class Keypoint {
public:    
    Keypoint()  
    { m_x = 0.0; m_y = 0.0; m_r = 0; m_g = 0; m_b = 0; m_d = NULL; 
        m_extra = -1; m_track = -1; }

    Keypoint(float x, float y, unsigned char *d) :
	m_x(x), m_y(y), m_d(d), m_extra(-1), m_track(-1)
    { m_r = 0; m_b = 0; m_g = 0; }

    float m_x, m_y;        /* Subpixel location of keypoint. */
    // float m_scale, m_ori;  /* Scale and orientation (range [-PI,PI]) */
    unsigned char *m_d;    /* Vector of descriptor values */
    unsigned char m_r, m_g, m_b;          /* Color of this key */

    int m_extra;  /* 4 bytes of extra storage */
    int m_track;  /* Track index this point corresponds to */
};

class KeypointWithScaleRot : public Keypoint
{
public:
    float m_scale, m_orient;
};

/* Data struct for matches */
class KeypointMatch {
public:
    KeypointMatch() 
    { }

    KeypointMatch(int idx1, int idx2) :
	m_idx1(idx1), m_idx2(idx2)
    { }

    int m_idx1, m_idx2;
};

/* Data struct for matches (with score field) */
class KeypointMatchWithScore : public KeypointMatch {
public:
    KeypointMatchWithScore() 
    { }

    KeypointMatchWithScore (int idx1, int idx2, float score) :
	KeypointMatch(idx1, idx2), m_score(score)
    { }

    // int m_idx1, m_idx2;
    float m_score;
};

/* Returns the number of keys in a file */
int GetNumberOfKeys(const char *filename);

/* This reads a keypoint file from a given filename and returns the list
 * of keypoints. */
std::vector<Keypoint> ReadKeyFile(const char *filename, bool descriptor);
std::vector<KeypointWithScaleRot> 
    ReadKeyFileWithScaleRot(const char *filename, bool descriptor);

/* Read keypoints from the given file pointer and return the list of
 * keypoints.  The file format starts with 2 integers giving the total
 * number of keypoints and the size of descriptor vector for each
 * keypoint (currently assumed to be 128). Then each keypoint is
 * specified by 4 floating point numbers giving subpixel row and
 * column location, scale, and orientation (in radians from -PI to
 * PI).  Then the descriptor vector for each keypoint is given as a
 * list of integers in range [0,255]. */
std::vector<Keypoint> ReadKeys(FILE *fp, bool descriptor);

/* Read keys more quickly */
std::vector<Keypoint> ReadKeysFast(FILE *fp, bool descriptor, 
                                   float **scale = NULL, 
                                   float **orient = NULL);
std::vector<Keypoint> ReadKeysFastGzip(gzFile fp, bool descriptor,
                                       float **scale = NULL, 
                                       float **orient = NULL);

/* Read keys using MMAP to speed things up */
std::vector<Keypoint> ReadKeysMMAP(FILE *fp);





#endif /* __keys_h__ */
