/* 
 *  Copyright (c) 2008  Noah Snavely (snavely (at) cs.washington.edu)
 *    and the University of Washington
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 */

/* BaseApp.h */
/* Base application */

#ifndef __baseapp_h__
#define __baseapp_h__


#ifndef __USE_ANN__
#include "BruteForceSearch.h"
#else
#include "anniface.h"
#endif /* __USE_ANN__ */

#include "ImageData.h"

#include "sfm.h"

#ifndef WIN32_MVC
#include <ext/hash_map>
#include <ext/hash_set>
#else
#include <hash_map>
#include <hash_set>
#endif

#ifndef __DEMO__
#ifdef __USE_BOOST__
#include "boost/graph/graph_traits.hpp"
#include "boost/graph/adjacency_list.hpp"
using namespace boost;
#endif /* __USE_BOOST__ */
#endif /* __DEMO__ */

#ifdef __USE_BOOST__
typedef adjacency_list<vecS, vecS, 
                       undirectedS, 
                       property<vertex_color_t, int>, 
                       property<edge_weight_t, int> > ImageGraph;
#endif

class TransformInfo {
public:

    /* File IO routines */
    void ReadFromFile(FILE *f);
    void WriteToFile(FILE *f);

    /* For object movies */
    double m_fmatrix[9];
    double m_ematrix[9];

    /* For homographies */
    double m_H[9];
    double m_inlier_ratio;
    int m_num_inliers;

    /* For color correction */
    double m_gain[3], m_bias[3];
};

typedef std::pair<unsigned long, unsigned long> MatchIndex;
// typedef unsigned long long MatchIndex;

namespace __gnu_cxx {
    template<>
    struct hash<MatchIndex> {
        size_t
        operator()(MatchIndex __x) const
        { return __x.first * 1529 + __x.second; }
    };
}

/* Table containing information about which pairs of images match */
#ifndef WIN32_MVC
class MatchTable : private  __gnu_cxx::hash_set<MatchIndex>
#else
class MatchTable : private  stdext::hash_set<MatchIndex>
#endif
{
public:
    void SetMatch(MatchIndex idx) { 
        insert(idx);
    }

    void RemoveMatch(MatchIndex idx) {
        erase(idx);
    }

    bool Contains(MatchIndex idx) {
        return (find(idx) != end());
    }

    void Clear() {
        clear();
    }
};

class BaseApp 
{
public:
    virtual ~BaseApp() { }

    virtual bool OnInit() = 0;

    /* Process command line options */
    virtual void ProcessOptions(int argc, char **argv) = 0;

    /* Return the number of images */
    int GetNumImages();
    int GetNumOriginalImages();
    int GetNumMatches(int i1, int i2);

    /* Get match information */
    MatchIndex GetMatchIndex(int i1, int i2);
    void SetMatch(int i1, int i2);
    void RemoveMatch(int i1, int i2);
    bool ImagesMatch(int i1, int i2);

    /* Get keys */
    Keypoint &GetKey(int img, int key);
    int GetNumKeys(int img);
    /* Get the index of a registered camera */
    int GetRegisteredCameraIndex(int cam);

    /* Load a list of image names from a file */
    void LoadImageNamesFromFile(FILE *f);

    /* Load matches from files */
    void LoadMatches();
    void ReadMatchFile(int i, int j);
    void LoadMatchTable(const char *filename);
    /* Load keys from files */
    void LoadKeys(bool descriptor = true);
    void ClearMatches();
    /* Prune points that match to multiple targets */
    void PruneDoubleMatches();

    /* Make match lists symmetric */
    void MakeMatchListsSymmetric();

    /* Setup data structures storing the points and lines visible to
     * each image */
    void SetupImagePoints(int min_views = 1);

    /* Output routines */

    /* IO routines */
    void ReadGeometricConstraints(char *filename);
    void WriteGeometricConstraints(char *filename);
    void WriteTracksForSameer(char *filename);
    void WriteTracks(char *filename);
    void ReadCameraConstraints();
    void ReadPointConstraints();
    void ReadIntrinsicsFile();

    /* Read the ignore file */
    void ReadIgnoreFile();

    /* Grab the color of each keypoint */
    void ReadKeyColors();

    /* Read in information about the world */
    void ReadBundleFile(char *filename);
    void ReloadBundleFile (char *filename);

    /* Clear the current model */
    void ClearModel();

    /* Write the match table in Drew's format */
    void WriteMatchTableDrew(const char *append = "");
    void ReadMatchTableDrew(const char *append = "");

    /* Initialize images read from a file without performing bundle
     * adjustment */
    void InitializeImagesFromFile(FILE *f);

    void SetTracks(int image);
    /* Set the track field of each keypoint given the 3D points */
    void CreateTracksFromPoints();
    void SetTracksFromPoints();
    int SetTracksFromPoints(int image);
    /* Use tracks to setup matches */
    void SetMatchesFromTracks();
    void SetMatchesFromTracks(int img1, int img2);
     void ClearMatches(MatchIndex idx);
    int GetNumTrackMatches(int img1, int img2);

    /* Use the bundle-adjusted points to create a new set of matches */
    void SetMatchesFromPoints(int threshold = 0);

    /* Create a search tree on cameras */
#ifdef __USE_ANN__
    ANNkd_tree *CreateCameraSearchTree();
#else
    BruteForceSearch *CreateCameraSearchTree();
#endif

#ifndef __DEMO__
    /* Write point files to a ply file */
    void DumpPointsToPly(char *output_directory, char *filename, 
                         int num_points, int num_cameras, 
			 v3_t *points, v3_t *colors, camera_params_t *cameras
                         /*bool reflect = true*/);

    /* Dump an output file containing information about the current
     * state of the world */
    void DumpOutputFile(char *output_dir, char *filename, 
			int num_images, int num_cameras, int num_points,
			int *added_order, 
			camera_params_t *cameras, v3_t *points, v3_t *colors,
			std::vector<ImageKeyVector> &pt_views
                        /*bool output_radial_distortion = false*/);

#endif /* __DEMO__ */

    /* XML output routines */
    void WritePointsXML(const char *filename);
    void WriteCamerasXML(const char *filename);
    void WriteCamerasGeoXML(const char *filename);

    /* Write POINTS in a FILE */
    void WritePoints(const char* filename);
    
    /* Write camera parameters in a FILE */
    void WriteCameras(const char* filename);

    /* Fix cameras (reflection bug) */
    void FixReflectionBug();

    void RemoveBadImages(int min_num_points);

    /* Estimate point normals and confidences */
    void EstimatePointNormalsConfidence();
    void EstimatePointNormals();

    /* Estimate the axes based on the image orientations */
    void EstimateAxes(double *xaxis, double *yaxis, double *zaxis);
    /* Analyze the scene and return various attributes */
    void SetupScene(double *center, double *up, 
		    double *x_axis, double *z_axis, double &scale);

    void SetupSceneGroundPlane(double *center, double *up, 
                               double *x_axis, double *z_axis, double &scale);

    void TransformWorldReal();
    virtual void RepositionScene(double *center_out, double *R_out, 
                                 double &scale_out);
    void TransformSceneCanonical(int c1, int c2);
    void UnscaleCameras(int start_camera);

    /* Compute orientations for each image */
    void ComputeImageRotations();


    /* **** Data **** */

    double m_bundle_version;

    /* Geometry data */

    std::vector<ImageData> m_image_data;   /* Image data */
    int m_num_original_images;

    std::vector<PointData> m_point_data;   /* Information about 3D
					    * points in the scene */

    std::vector<int> m_num_views_orig;

    std::vector<TrackData> m_track_data;   /* Information about the
                                            * detected 3D tracks */

    ImageKeyVector m_outliers;             /* Outliers detected among
					    * the feature points */    

    double m_repos_R[9];
    double m_repos_d[3];
    double m_repos_scale;

    double m_xform[16];       /* Scene transform */

    /* Point constraint fields */
    bool m_use_point_constraints;
    v3_t *m_point_constraints;
    double m_point_constraint_weight;
    char *m_point_constraint_file;

    bool m_matches_loaded;    /* Have the matches been loaded? */
    bool m_matches_computed;  /* Have the matches been computed? */
    // bool *m_matches;          /* Match matrix */
    MatchTable m_matches;

#ifndef WIN32_MVC
    __gnu_cxx::hash_map<MatchIndex, std::vector<KeypointMatch> > m_match_lists;
#else
    stdext::hash_map<MatchIndex, std::vector<KeypointMatch> > m_match_lists;
#endif

#ifndef WIN32_MVC
    __gnu_cxx::hash_map<MatchIndex, TransformInfo> m_transforms;
#else
    stdext::hash_map<MatchIndex, TransformInfo> m_transforms;
#endif

    /* **** Options **** */

    double m_scale;              /* Scene scale */
    bool m_metric;               /* Is the scene in meters? */

    bool m_fisheye;              /* Use fisheye images? */
    char *m_fisheye_params;      /* Fisheye parameter file */

    char *m_ignore_file;         /* Contains the images to ignore during
                                  * bundle adjustment */
    bool m_use_intrinsics;
    char *m_intrinsics_file;     /* File containing intrinsics */     

    bool m_bundle_provided;      /* Was a bundle adjustment file given? */
    char *m_bundle_file;         /* Bundle file */

    char *m_match_directory;     /* Which directory are the matches
				  * stored in? */
    char *m_match_table;         /* File where match table is stored */
    char *m_key_directory;
    char *m_image_directory;
    char *m_sift_binary;         /* Where can we find the sift binary? */

    bool m_estimate_up_vector_szeliski;  /* Estimate the up vector
					  * using Rick's method? */

    int m_min_track_views;           /* Minimum number of observations
				      * for each track */
    int m_max_track_views;           /* Minimum number of observations
				      * for each track */

    int m_up_image;                  /* What image should we use to
				      * estimate the up vector (if
				      * any?) */

#ifdef __USE_BOOST__
    /* Graph fields */    
    ImageGraph m_image_graph;
    ImageGraph m_working_image_graph;
    std::vector<int> m_working_images;

    std::vector<int> m_graph_components;
    int m_max_graph_component;
#endif /* __DEMO__ */
};

#endif /* __baseapp_h__ */
