# Machine Definitions
#
# 0.1 - Brittany Nkounkou, 18 Jun 2014
#       - initial machine structure
#       - using (mutable) recordtype instead of (immutable) namedtuple
# 0.2 - Stephen Longfield, 23 Jun 2014
#       - Adding edge label type

from recordtype import recordtype

## Machine Types
# M_SMP : Simple machine
# M_SEQ : Sequence 
# M_PAR : Parallel 
# M_SEL : Selection
# M_NFA : Non-deterministic finite automata
# M_GRD : Guard
M_SMP, M_SEQ, M_PAR, M_SEL, M_NFA, M_GRD = range(6)

# Machines
# error   : True or False
# type    : Machine Type
# prbs,snds,rcvs,occs : sets of channel names
# rep     : True or False
# content : None, list of Machines, NFA, or list of exclusive probes for guard
Machine = recordtype("Machine", "error type prbs snds rcvs occs rep content") 

# NFA Machine Edge
Edge = recordtype("Edge", "head label tail")

### NFA Edge labels
## Basic labels
# L_STB : Start! 0
# L_STQ : Start?
# L_USB : Use!
# L_USQ : Use?
# L_FIB : Finish!
# L_FIQ : Finish?
## Special Labels
# L_SND : Send 6
# L_RCV : Receive
# L_OCC : Occupy
# L_PRB : Probe
# L_XPR : Exclusive Probe
#         chan is the probed channel
#         content is the exclusive channels
## Label composition
# L_SEQ : Sequence of special labels 11
# L_PAR : Parallel composition of non-interacting special labels or L_SEQs
# L_SET : Set of (possibly interacting) basic labels.
# L_NUL : No label, result of reduction after New
## Error labels
# L_IST : Instability 15
# L_INT : Interference
# L_SER : Selection error
L_STB, L_STQ, L_USB, L_USQ, L_FIB, L_FIQ, L_SND, L_RCV, L_OCC, L_PRB, L_XPR, L_SEQ, L_PAR, L_SET, L_NUL, L_IST, L_INT, L_SER = range(18)
## Edge Label
# Type    : one of the labels above
# Chan    : The channel for one of the basic/special labels, set of chans
#           label composition or probes.
# Content : For label composition, this is an array of more labels. 
#           For exclusive probe, this is a list of more channels.
Label = recordtype("Label", "type chan content")

def compose_special(label1, label2):
	if label1.type == L_SND:
		if label2.type == L_SND:
			return Label(L_INT, label1.chan, [])
		elif label2.type == L_RCV:
			return Label(L_OCC, label1.chan, [])
		elif label2.type == L_OCC:
			return Label(L_INT, label1.chan, [])
	elif label1.type == L_RCV:
		if label2.type == L_SND:
			return Label(L_OCC, label1.chan, [])
		elif label2.type == L_RCV:
			return Label(L_INT, label1.chan, [])
		elif label2.type == L_OCC:
			return Label(L_INT, label1.chan, [])
	elif label1.type == L_OCC:
		if label2.type == L_SND:
			return Label(L_INT, label1.chan, [])
		elif label2.type == L_RCV:
			return Label(L_INT, label1.chan, [])
		elif label2.type == L_OCC:
			return Label(L_INT, label1.chan, [])

def composeLabels(label1, label2):
	'''Composes two special labels together.  Not complete composition yet...'''

	#print "LABEL1", label1
	#print "LABEL2", label2
	#print "----"

	# short-circuit on errors
	if label1.type in [L_IST, L_INT, L_SER]:
		return label1

	if label2.type in [L_IST, L_INT, L_SER]:
		return label2

	if label1.type == L_NUL:
		return label2

	if label2.type == L_NUL:
		return label1

	# Raise exceptions for more interesting cases than we've implemented....
	if label1.type not in [L_STB, L_STQ, L_USB, L_USQ, L_FIB, L_FIQ, L_SND, L_RCV, L_OCC, L_PRB, L_XPR, L_PAR]:
		raise NotImplementedError("Label 1 is too interesting to compose trivially")
	if label2.type not in [L_STB, L_STQ, L_USB, L_USQ, L_FIB, L_FIQ, L_SND, L_RCV, L_OCC, L_PRB, L_XPR, L_PAR]:
		raise NotImplementedError("Label 2 is too interesting to compose trivially")

	bp = False

	# Composition is only interesting if the channels are the same
	if label1.chan != label2.chan:
		if label1.type == L_PAR or label2.type == L_PAR:
			pass
		else:
			if label1.chan != "" and label2.chan != "":
				rl = Label(L_PAR, set([label1.chan, label2.chan]), [label1, label2])
			elif label1.chan == "":
				rl = Label(L_PAR, set([label2.chan]), [label1, label2])
			elif label2.chan == "":
				rl = Label(L_PAR, set([label1.chan]), [label1, label2])
			elif label1.chan == label2.chan == "":
				rl = Label(L_PAR, set([]), [label1, label2])
			else:
				print label1
				print label2
				assert False

			cont = False
			for lb in rl.content:
				if lb.type == L_XPR:
					content = rl.content
					cont = True
					bp = True
					break
			if not cont:
				return rl
	

	# Special labels are simple to compose
	if label1.type in [L_SND, L_RCV, L_OCC] and label2.type in [L_SND, L_RCV, L_OCC]:
		return compose_special(label1, label2)

	# If one of the labels is parallel composition, it's a pain. First, we collect up the parts 
	p_basic = []
	p_special = []
	xpr = []
	prb = []
	chans = set([])

	if label1.type == L_PAR and label2.type == L_PAR:
		if label1.chan.isdisjoint(label2.chan):
			cont = False
			for lb in (label1.content + label2.content):
				if lb.type == L_XPR:
					content = label1.content + label2.content
					cont = True
					break
			if not cont:
				return Label(L_PAR, label1.chan | label2.chan, label1.content+label2.content)
		else:
			content = label1.content + label2.content
	elif label1.type == L_PAR:
		rl = False
		if label2.chan == "":
			rl = True
			rl = Label(L_PAR, label1.chan, label1.content+[label2])
		elif label2.chan not in label1.chan:
			rl = True
			rl = Label(L_PAR, label1.chan | set([label2.chan]), label1.content+[label2])
		else:
			content = label1.content + [label2]
		if rl:
			cont = False
			for lb in rl.content:
				if lb.type == L_XPR:
					content = rl.content
					cont = True
					break
			if not cont:
				return rl
	elif label2.type == L_PAR:
		rl = False
		if label1.chan == "":
			rl = True
			rl = Label(L_PAR, label2.chan, label2.content+[label1])
		elif label1.chan not in label2.chan:
			rl = True
			rl = Label(L_PAR, label2.chan | set([label1.chan]), label2.content+[label1])
		else:
			content = label2.content + [label1]
		if rl:
			cont = False
			for lb in rl.content:
				if lb.type == L_XPR:
					content = rl.content
					cont = True
					break
			if not cont:
				return rl
	else:
		if bp:
			pass
		else:
			content = [label1, label2]
	
	# Then classfiy each one of the labels
	for l in content:
		if l.chan != "":
			chans.add(l.chan)
		if l.type in [L_IST, L_INT, L_SER]:
			return l
		elif l.type == L_XPR:
			xpr.append(l)
		elif l.type == L_PRB:
			prb.append(l)
		elif l.type in [L_STB, L_STQ, L_USB, L_USQ, L_FIB, L_FIQ]:
			p_basic.append(l)
		elif l.type in [L_SND, L_RCV, L_OCC]:
			p_special.append(l)
		else:
			print printLabel(l)
			#assert l.type in [L_XPR, L_PRB] and l.chan == ""
			assert False

	# Then start processing by first going through all of the special labels, and 
  #  making a mapping from channel to label.
	ch_special = {}
	for l in p_special:
		if l.chan in ch_special:
			ch_special[l.chan].append(l)
		else:
			ch_special[l.chan] = [l]

	# Then go through each channel, check if there are two or fewer labels, and if there
  #  are exactly two, compose them with compose_special. Use this to create p_special2,
  #  the processed special labels
	p_special2 = []
	for ch in ch_special.keys():
		if len(ch_special[ch]) > 2:
			return Label(L_INT, ch, [])
		elif len(ch_special[ch]) == 1:
			p_special2.append(ch_special[ch][0])
		elif len(ch_special[ch]) == 0:
			pass
		else:
			assert len(ch_special[ch]) == 2
			p_special2.append(compose_special(ch_special[ch][0], ch_special[ch][1]))

	# Go through each label in the basic labels, and build up the set-true and set-false 
  #  sets, checking for interference or instability
	# (chan, 0) is \bar{chan}, (chan, 1) is \hat{chan}
	strue = set([])
	sfalse = set([])
	
	for l in p_basic:
		if l.type == L_STB:
			strue.add((l.chan, 0))
			if (l.chan, 0) in sfalse:
				return Label(L_INT, l.chan, [])
#			if (l.chan, 1) in strue:
#				return Label(L_IST, l.chan, [])
		elif l.type == L_STQ:
			strue.add((l.chan, 1))
			if (l.chan, 1) in sfalse:
				return Label(L_INT, l.chan, [])
#			if (l.chan, 0) in sfalse:
#				return Label(L_IST, l.chan, [])
		elif l.type == L_USB:
			strue.add((l.chan, 0))
			if (l.chan, 0) in sfalse:
				return Label(L_INT, l.chan, [])
		elif l.type == L_USQ:
			strue.add((l.chan, 1))
			if (l.chan, 1) in sfalse:
				return Label(L_INT, l.chan, [])
		elif l.type == L_FIB:
			sfalse.add((l.chan, 0))
			if (l.chan, 0) in strue:
				return Label(L_INT, l.chan, [])
#			if (l.chan, 1) in sfalse:
#				return Label(L_IST, l.chan, [])
		elif l.type == L_FIQ:
			sfalse.add((l.chan, 1))
			if (l.chan, 1) in strue:
				return Label(L_INT, l.chan, [])
#			if (l.chan, 0) in strue:
#				return Label(L_IST, l.chan, [])
		else:
			assert l.type in [L_SND, L_RCV, L_OCC, L_PRB, L_XPR]
			pass

	# Go through the list of probes, and check for instability (set-false)
	for l in prb:
		if (l.chan, 0) in sfalse and (l.chan, 1) in strue:
			return Label(L_IST, l.chan, [])


	# Go through the exclusive probes, and check to see if the exclusion is satisfied,
  #  and that there isn't an instability
	for l in xpr:
		if (l.chan, 0) in sfalse and (l.chan, 1) in strue:
			return Label(L_IST, l.chan, [])
		for ch in l.content:
			if (ch, 0) in strue:
				return Label(L_SER, ch, [])
			if (ch, 0) in sfalse:
				return Label(L_SER, ch, [])

	# If nothing bad happens, just return the label with everything composed together
	return Label(L_PAR, chans, p_basic + p_special2 + prb + xpr)


def compliantLabel(label, chan):
	'''Checks to see if a label is compliant with a channel.  Returns a complaint
  value and a possible new label. 
     Value 0:
       Non-compliant, no label returned
     Value 1:
       Compliant, channel matched, new label w/o that channel returned
     Value 2:
       Complaint, no match, no label returned'''
	#print chan, label
	if label.type in [L_SND, L_RCV, L_PRB, L_XPR]:
		# Simple labels can never be compliant
		if label.chan == chan:
			return (0, None)
		else: 
			return (2, None)
	elif label.type in [L_STQ, L_FIB]:
		# These need other parts to be compliant
		if label.chan == chan:
			return (0, None)
		else: 
			return (2, None)
	elif label.type in [L_STB, L_USB, L_USQ, L_FIQ]:
		# These basic labels are always complaint
		if label.chan == chan:
			return (1, Label(L_NUL, [], []))
		else:
			return (2, None)
	elif label.type == L_OCC:
		# Occupy is always compliant
		if label.chan == chan:
			return (1, Label(L_NUL, [], []))
		else:
			return (2, None)
	elif label.type == L_PAR:
		#print "LABEL_PAR", label
		#print chan
		#print label.chan
		#print (chan in label.chan)
		if chan in label.chan:
			# Collect all the labels together for this channel, and the
			#  labels for the uninvolved channels
			labs = []
			non_labs = []
			xpr = []
			xprc = set([])
			for l in label.content:
				# Collect up the XPR labels, since they need to be kept around even in the case
				#  of non-compliant edges
				if l.type == L_XPR:
					xprc.add(l.chan)
					xpr.append(l)
				if l.chan == chan:
					labs.append(l)
				else:
					non_labs.append(l)
			# Build the label for if this edge is compliant out of the uninvolved edges
			if len(non_labs) > 0:
				l_comp = Label(L_PAR, label.chan-set([chan]), non_labs)
			else:
				l_comp = Label(L_NUL, [], [])
			if len(xpr) > 0:
				xpr = Label(L_PAR, xprc, xpr)
			else:
				xpr = None
			# First, check if any of the speical labels are in the set, if so, see if
			#  they are satisfied
			for i in range(len(labs)):
				l = labs[i]
				if l.type == L_OCC:
					if len(labs) == 1:
						return (1, l_comp)
					else:
						return (0, None)
				elif l.type == L_SND:
					if len(labs) == 2:
						if i == 0:
							if labs[1].type == L_RCV:
								return (1, l_comp)
							else:
								return (0, None)
						elif i == 1:
							if labs[0].type == L_RCV:
								return (1, l_comp)
							else:
								return (0, None)
						else:
							assert False
					else:
						return (0, None)
				elif l.type == L_RCV:
					if len(labs) == 2:
						if i == 0:
							if labs[1].type == L_SND:
								return (1, l_comp)
							else:
								return (0, None)
						elif i == 1:
							if labs[0].type == L_SND:
								return (1, l_comp)
							else:
								return (0, None)
						else:
							assert False
					else:
						return (0, None)
				#else:
			#		print "WHAT1", l
			
			# Collect up pre. (chan, 0) is \bar{chan}, (chan, 1) is \hat{chan}
			pre = set([])
			for l in labs:
				if l.type == L_USB:
					pre.add((l.chan, 0))
				elif l.type == L_USQ:
					pre.add((l.chan, 1))
				elif l.type == L_FIB:
					pre.add((l.chan, 0))
				elif l.type == L_FIQ:
					pre.add((l.chan, 1))
			# Check req
			comp = True
			for l in labs:
				if l.type == L_STB:
					if (l.chan, 1) in pre:
						comp = False
				elif l.type == L_STQ:
					if (l.chan, 0) not in pre:
						comp = False
				elif l.type == L_FIB:
					if (l.chan, 1) not in pre:
						comp = False
				elif l.type == L_FIQ:
					if (l.chan, 0) in pre:
						comp = False
				elif l.type == L_PRB:
					if (l.chan, 0) not in pre:
						comp = False
				elif l.type == L_XPR:
					if (l.chan, 0) not in pre:
						comp = False
			if comp:
				return (1, l_comp)
			if xpr is not None:
				return (3, xpr)
			return (0, None)
		else:
			return (2, None)
	elif label.type in [L_IST, L_INT, L_SER]:
		# Treat these as compliant, so they aren't deleted
		return (2, None)
	elif label.type == L_NUL:
		return (2, None)
	else:
		print label.type, printLabel(label), label
		quit()

def updateStats(prbs, snds, rcvs, occs, label):
	'''Updates the prvs, snds, rcvs, and occs stats based on a label.'''
	if label.type == L_SND:
		snds.add(label.chan)
	elif label.type == L_RCV:
		rcvs.add(label.chan)
	elif label.type == L_OCC:
		occs.add(label.chan)
	elif label.type == L_PRB:
		for c in label.chan:
			prbs.add(c)
	elif label.type == L_XPR:
		for c in label.chan:
			prbs.add(c)
	elif label.type == L_PAR:
		for l in label.content:
			(prbs, snds, rcvs, occs) = updateStats(prbs, snds, rcvs, occs, l)
	elif label.type in [L_STB, L_USB, L_FIB]:
		snds.add(label.chan)
	elif label.type in [L_STQ, L_USQ, L_FIQ]:
		rcvs.add(label.chan)
	elif label.type in [L_INT, L_IST, L_SER, L_NUL]:
		pass
	else:
		print label.type
		raise NotImplementedError("Label is too interesting to update trivially")
	return (prbs, snds, rcvs, occs)

def eqLabel(label1, label2):
	'''Label equality'''
	if label1.type != label2.type:
		return False
	if label1.type in [L_NUL]:
		return True
	if label1.type in [L_STB, L_STQ, L_USB, L_USQ, L_FIB, L_FIQ, L_SND, L_RCV, L_OCC, L_PRB, L_IST, L_INT, L_SER]:
		if label1.chan == label2.chan:
			return True
		return False
	if label1.type == L_PAR:
		if len(label1.content) != len(label2.content):
			return False
		for i in range(len(label1.content)):
			if not eqLabel(label1.content[i], label2.content[i]):
				return False
		return True
	if label1.type == L_XPR:
		if label1.chan != label2.chan:
			return False
		if label1.content != label2.content:
			return False
		return True
	assert False
	

def printLabel(label):
	'''Pretty-printer for labels.'''
	if label.type == L_STB:
		return "Start!"+label.chan
	elif label.type == L_STQ:
		return "Start?"+label.chan
	elif label.type == L_USB:
		return "Use!"+label.chan
	elif label.type == L_USQ:
		return "Use?"+label.chan
	elif label.type == L_FIB:
		return "Finish!"+label.chan
	elif label.type == L_FIQ:
		return "Finish?"+label.chan
	elif label.type == L_SND:
		return label.chan+"!"
	elif label.type == L_RCV:
		return label.chan+"?"
	elif label.type == L_OCC:
		return label.chan+"!?"
	elif label.type == L_PRB:
		me = ""
		for c in label.chan:
			me += "#" + c + " "
		if me == "": return "PRB"
		return me
	elif label.type == L_XPR:
		me = ""
		if len(label.chan) == len(label.content) == 0:
			return "Bool"
		me += "#" + label.chan + " "
		me += " w/o "
		for c in label.content:
			me += "#" + c + " "
		if me == "": return "XPR"
		return me
	elif label.type == L_SEQ:
		me = ""
		for l in label.content:
			me += printLabel(l) + "; "
		me = me[:-2]
		if me == "": return "SEQ"
		return me
	elif label.type == L_PAR:
		me = ""
		for l in label.content:
			me += printLabel(l) + "|| "
		me = me[:-3]
		if me == "": return "PAR"
		return me
	elif label.type == L_SET:
		me = ""
		for l in label.content:
			me += " " + printLabel(l)
		if me == "": return "SET"
		return me
	elif label.type == L_NUL:
		return "NULL"
	elif label.type == L_IST:
		return "Instability" + label.chan
	elif label.type == L_INT:
		return "Interference" + label.chan
	elif label.type == L_SER:
		return "SEL error " + label.chan
	else:
		raise NotImplementedError

# NFA Machine
class NFA:
	'''Non-deterministic finite automata for analysis of CHP programs'''
	def __init__(self):
		'''Initializes an empty NFA. Takes no arguments.'''
		self.start = 0
		self.accept = 0
		self.prevNodeId = 0
		self.prevEdgeId = 0
		self.edges = set([])
		self.edgeLabel = {}
		#self.edges = []
		self.incoming = {0 : set([])}
		self.outgoing = {0 : set([])}
	
	def addNode(self):
		'''Add a new node. Returns the node number that was just added'''
		self.prevNodeId += 1
		self.incoming[self.prevNodeId] = set([])
		self.outgoing[self.prevNodeId] = set([])
		return self.prevNodeId
	
	def addEdge(self, edge):
		'''Add a new edge.'''
		self.prevEdgeId += 1
		self.edges.add(self.prevEdgeId)
		self.edgeLabel[self.prevEdgeId] = edge
		self.outgoing[edge.head].add(self.prevEdgeId)
		self.incoming[edge.tail].add(self.prevEdgeId)
		#self.edges += [edge]
	
	def remEdge(self, edge):
		'''Remove an edge.'''
		self.edges.remove(edge)
		el = self.edgeLabel[edge]
		self.incoming[el.tail].discard(edge)
		self.outgoing[el.head].discard(edge)
		self.edgeLabel.pop(edge)

	def buildInOut(self):
		'''Rebuild the incoming and outgoing sets'''
		self.incoming = {}
		self.outgoing = {}
		for e in self.edges:
			el = self.edgeLabel[e]
			if el.head in self.outgoing:
				self.outgoing[el.head].add(e)
			else:
				self.outgoing[el.head] = set([e])
			if el.tail not in self.outgoing:
				self.outgoing[el.tail] = set([])

			if el.tail in self.incoming:
				self.incoming[el.tail].add(e)
			else:
				self.incoming[el.tail] = set([e])
			if el.head not in self.incoming:
				self.incoming[el.head] = set([])

	# decides whether or not there is a path from start to accept using DFS
	def canTerminate(self):
		'''Decide wheater or not there is a path from start to accept using DFS.
  Quick-return if accept is None or the start node.'''
		if self.accept is None:
			return False
		if self.accept == self.start:
			return True
		#self.buildOutgoing()
		stack = [self.start]
		seen = set([])
		while len(stack) > 0:
			n = stack.pop()
			out = self.outgoing[n]
			for e in out:
				e = self.edgeLabel[e]
				tgt = e.tail
				if tgt == self.accept:
					return True
				if tgt not in seen:
					stack.append(tgt)
					seen.add(tgt)
		return False

	def filterReachable(self):
		# Note: This returns if accept was found or not, so can be used
		#       as a variant of canTerminate, that reduces the graph, 
		#       though it has worse average-case performance.
		'''Starting from the start node, use DFS to find reachable edges.
  Edges that are not reachable are removed from the NFA.'''
		#self.buildOutgoing()
		newEdges = set([])
		stack = [self.start]
		seen = set([])
		acceptFound = False
		while len(stack) > 0:
			n = stack.pop()
			if n not in self.outgoing:
				print self.edges
			out = self.outgoing[n]
			for e in out:
				newEdges.add(e)
				e = self.edgeLabel[e]
				if e.label.type in [L_IST, L_INT, L_SER]:
					continue
				tgt = e.tail
				if tgt == self.accept:
					acceptFound = True
				if tgt not in seen:
					stack.append(tgt)
					seen.add(tgt)
		rem = []
		for e in self.edges:
			if e not in newEdges:
				rem.append(e)
		for r in rem:
			self.remEdge(r)
		#self.edges = list(newEdges)
		if not acceptFound:
			self.accept = None
		return acceptFound

	def mergeNodes(self, nodes, node):
		'''Takes in a list of nodes and a single node. Goes through the edge set,
  replacing nodes in the nodes list with the specified node.'''
		snodes = set(nodes)

		if node not in self.incoming:
			self.incoming[node] = set([])
		if node not in self.outgoing:
			self.outgoing[node] = set([])

		#print node, nodes, self.incoming, self.outgoing

		for n in snodes:
			if n == node:
				continue
			for e in self.incoming[n]:
				self.incoming[node].add(e)
				self.edgeLabel[e].tail = node
			for e in self.outgoing[n]:
				self.outgoing[node].add(e)
				self.edgeLabel[e].head = node
			self.outgoing.pop(n)
			self.incoming.pop(n)

		if self.start in nodes:
			self.start = node
		#	self.incoming.pop(n)
		#	self.outgoing.pop(n)

		#for e in self.edges:
		#	el = self.edgeLabel[e]
		#	if el.head in snodes:
		#		self.edgeLabel[e].head = node
		#	if el.tail in snodes:
		#		self.edgeLabel[e].tail = node
		#self.buildInOut()

	def filterNull(self):
		'''Connects the head and tail of Null nodes'''
		null = []
		rem = []
		#self.buildOutgoing()
		for e in self.edges:
			if self.edgeLabel[e].label.type == L_NUL:
				null.append(e)

		for e in rem:
			self.remEdge(e)

		for e in null:
			e = self.edgeLabel[e]
			self.mergeNodes([e.tail], e.head)

	def mergeNull(self):
		'''Go through each entry in outgoing. Collect each of the Null-labeled edges.
  If there is more than one connecting to the same target, remove the duplicates.'''

		rem = []
		#self.buildOutgoing()

		for n in self.outgoing.keys():
			null = [e for e in self.outgoing[n] if self.edgeLabel[e].label.type == L_NUL]
			tgts = set([])
			for e in null:
				if self.edgeLabel[e].tail in tgts:
					rem.append(e)
				else:
					tgts.add(self.edgeLabel[e].tail)

		for e in rem:
			self.remEdge(e)

	def removeNullSelf(self):
		'''Removes null self-edges'''
		rem = []
	
		for e in self.edges:
			el = self.edgeLabel[e]
			if el.label.type == L_NUL:
				if el.head == el.tail:
					rem.append(e)

		for e in rem:
			self.remEdge(e)

	def mergeSame(self):
		'''Go through every entry in outgoing. Collect each of the edges. If there is
  an edge with the exact same label and target, remove that edge.'''

		rem = set([])
		replace = []
		#self.buildOutgoing()

		for n in self.outgoing.keys():
			for e1 in self.outgoing[n]:
				if e1 in rem:
					continue
				for e2 in self.outgoing[n]:
					if e1 is e2:
						continue
					if e2 in rem:
						continue
					el1 = self.edgeLabel[e1]
					el2 = self.edgeLabel[e2]
					if el1.head == el2.head:
						if el1.tail == el2.tail:
							if eqLabel(el1.label, el2.label):
								rem.add(e2)

		for e in rem:
			self.remEdge(e)

	def mergeParallel(self):
		'''Go through every entry in outgoing. For each of these entries, collect up
  the mapping from targets to edges. If there is more than one non-error edge
  going to the same target, merge them together.'''
		merge = []
		rem = set([])

		for n in self.outgoing.keys():
			map = {}
			reduce = set([])
			for e in self.outgoing[n]:
				el = self.edgeLabel[e]
				if el.label.type in [L_IST, L_INT, L_SER]:
					continue
				if e in rem:
					continue
				if el.tail in map:
					reduce.add(el.tail)
					map[el.tail].append(e)
				else:
					map[el.tail] = [e]
			for t in reduce:
				l = map[t]
				label = self.edgeLabel[l[0]].label
				head = self.edgeLabel[l[0]].head
				for e in l[1:]:
					label = composeLabels(self.edgeLabel[e].label, label)
				if label.type in [L_IST, L_INT, L_SER]:
					continue
				for e in l:
					self.remEdge(e)
				self.addEdge(Edge(head, label, t))

	def reduceNFA(self):
		# Filter out edges that have become unreachable
		self.filterReachable()

		# Collapse Null edges
		self.filterNull()

		# Merge parallel Null edges
		self.mergeNull()

		# Remove Null self edges
		self.removeNullSelf()

		# Merge same
		self.mergeSame()

		# Merge parallel
		#self.mergeParallel()


	def filterCompliant(self, chan):
		'''Filters using the rules of compliant.  If an edge is non-compliant, it is
  removed. If it is compliant, the labels relating to that channel are removed.  If 
  needed, this may create more edges with appropriate labels (e.g. changing a SEND
  into a START;USE;FINISH to comply with a probe).'''
		# Check every edge to see if it's compliant.
		remove = []
		#for i in range(len(self.edges)):
		#	e = self.edges[i]
		for e in self.edges:
			el = self.edgeLabel[e]
			(compliant, l) = compliantLabel(el.label, chan)

			#print chan, l, compliant, el.head, el.tail, printLabel(el.label)
			#print "-------"
			if compliant in [0,3]:
				remove.append(e)
			elif compliant == 1:
				#self.edges[i].label = l
				self.edgeLabel[e].label = l
			elif compliant == 2:
				pass
			else:
				assert False

		for r in remove:
			self.remEdge(r)

		self.reduceNFA()

	def expandEdges(self, chans):
		'''Turns L_SND and L_RCV (and occurrences of these in L_PAR) into
  their expanded forms, using L_STB..L_FIQ for some list of channels.'''
		
		# First, we collect up the edges to expand
		expand = []
		for e in self.edges:
			el = self.edgeLabel[e]
			if el.label.type == L_SND:
				if el.label.chan in chans:
					expand.append(e)
			elif el.label.type == L_RCV:
				if el.label.chan in chans:
					expand.append(e)
			elif el.label.type == L_NUL:
				pass
			elif el.label.type == L_PAR:
				# Shouldn't actually happen
				raise NotImplementedError("Trying to expand a PAR")
			elif el.label.type in [L_STB, L_STQ, L_USB, L_USQ, L_FIB, L_FIQ]:
				raise NotImplementedError("Trying to expand an already expanded label")
			elif el.label.type in [L_SEQ, L_PAR, L_SET, L_NUL, L_IST, L_INT]:
				print el.label
				print printLabel(el.label)
				raise NotImplementedError("Trying to expand a too-complex label")
			else:
				assert el.label.type in [L_PRB, L_XPR]
				pass

		# Then we expand them.  Each becomes a L_STx, L_USx, L_FIx.  Need to add a new
		#  node for the L_USx self-loop, but otherwise don't change much.
		for e in expand:
			el = self.edgeLabel[e]
			head = el.head
			tail = el.tail
			mid = self.addNode()
			if el.label.type == L_SND:
				l1 = Label(L_STB, el.label.chan, [])
				l2 = Label(L_USB, el.label.chan, [])
				l3 = Label(L_FIB, el.label.chan, [])
			else:
				assert el.label.type == L_RCV
				l1 = Label(L_STQ, el.label.chan, [])
				l2 = Label(L_USQ, el.label.chan, [])
				l3 = Label(L_FIQ, el.label.chan, [])
			self.remEdge(e)
			e1 = Edge(head, l1, mid)
			e2 = Edge(mid, l2, mid)
			e3 = Edge(mid, l3, tail)
			self.addEdge(e1)
			self.addEdge(e2)
			self.addEdge(e3)

	def count_states(self):
		states = set([])
		for e in self.edges:
			el = self.edgeLabel[e]
			states.add(el.head)
			states.add(el.tail)

		return len(states)
	
def checkInteract(machine1, machine2):
	'''Checks to see if two machines interact, using the probes, sends, recieves and
  occupies sets. Returns a 5-tuple, where the 1st element is True if they interact,
  and False if they do not.  If they do not, the other elements of the 5-tuple are
  the union of the prbs, snds, revs, and occrs sets'''
	prbs1 = machine1.prbs
	snds1 = machine1.snds
	rcvs1 = machine1.rcvs
	occs1 = machine1.occs

	prbs2 = machine2.prbs
	snds2 = machine2.snds
	rcvs2 = machine2.rcvs
	occs2 = machine2.occs

	if ( # Probes interact with rcvs and occurs
		(prbs1.isdisjoint(rcvs2) and prbs1.isdisjoint(occs2))
		and # Sends interact with snds, rcvs, and occs
			(snds1.isdisjoint(snds2) and snds1.isdisjoint(rcvs2) and 
			 snds1.isdisjoint(occs2))
		and # Recvs interact with prbs, snds, recvs, and occs
			(rcvs1.isdisjoint(prbs2) and rcvs1.isdisjoint(snds2) and 
			 rcvs1.isdisjoint(rcvs2) and rcvs1.isdisjoint(occs2))
		and # Occs interact with prbs, snds, recvs and occs
			(occs1.isdisjoint(prbs2) and occs1.isdisjoint(snds2) and 
			 occs1.isdisjoint(rcvs2) and occs1.isdisjoint(occs2))):
		return False
	else:
		return True

			

def buildCache(machine1, machine2):
	'''Builds the prbs, snds, rcvs, and occs sets'''
	prbs = machine1.prbs.union(machine2.prbs)
	snds = machine1.snds.union(machine2.snds)
	rcvs = machine1.rcvs.union(machine2.rcvs)
	occs = machine1.occs.union(machine2.occs)
	return (prbs, snds, rcvs, occs)

