# CHP to Promela Translator
#
# 0.1 - Stephen Longfield 29 July 2014
#       - Language translation
#

from chp_funs import *
import chp_parse
import sys

def printChans(ctype, probed, shared):
	'''Prints out the channels and their types.  If a channel's type is not well-defined, 
  asks the user to input it.'''
	chans = ""
	for k in ctype.keys():
		# Symbolic channel to use for occupancy checks and probes
		if k in shared:
			chans += "chan probe" + k + " = [3] of {mtype};\n"
		elif k in probed:
			chans += "chan probe" + k + " = [2] of {mtype};\n"
		# Real channel
		chans += "chan " + k + " = [0] of "
		if ctype[k][1] == C_BOOL:
			# an int can always hold a bool
			chans += "{int};\n"
			#chans += "{bool};\n"
		elif ctype[k][1] == C_INT:
			chans += "{int};\n"
		elif ctype[k][1] == C_CTRL:
			chans += "{mtype};\n"
		else:
			assert ctype[k][1] == C_DATA
			type = ""
			while type not in ["0", "1"]:
				print "What is the type of channel " + k + "? 0: Bool, 1: Int"
				type = raw_input()
				if type == "0":
					ctype[k] = (ctype[k][0], C_BOOL)
					#chans += "{bool};\n"
					# same here
					chans += "{int};\n"
				elif type == "1":
					ctype[k] = (ctype[k][0], C_INT)
					chans += "{int};\n"
				else:
					print "Unrecognized type"
	return (chans, ctype)

def printVars(vtype):
	'''Prints out the variables and their types. If a variable's type is not well-defined,
	asks the user to input it.'''
	vars = ""
	for k in vtype.keys():
		if vtype[k] == V_INT:
			vars += "int " + k + ";\n"
		elif vtype[k] == V_BOOL:
			# same
			#vars += "bool " + k + ";\n"
			vars += "int " + k + ";\n"
		elif vtype[k] == V_DATA:
			type = ""
			while type not in ["0", "1"]:
				print "What is the type of variable " + k + "? 0: Bool, 1: Int"
				type = raw_input()
				if type == "0":
					# yep
					#vars += "bool " + k + ";\n"
					vars += "int " + k + ";\n"
				elif type == "1":
					vars += "int " + k + ";\n"
				else:
					print "Unrecgonized type"
	return (vars, vtype)

def printProExpr(expr):
	'''Prints a CHP expression as a Promela expression.  Also used to print guards.'''
	e = "" 
	if expr[0] == "EXPR":
		if len(expr) == 3:
			e += printProExpr(expr[1]) 
			e += " && " + printProExpr(expr[2])
		else:
			e += printProExpr(expr[1]) 
	elif expr[0] == "CONJ":
		if len(expr) == 3:
			e += printProExpr(expr[1]) 
			e += " || " + printProExpr(expr[2])
		else:
			e += printProExpr(expr[1]) 
	elif expr[0] == "PRIM":
		if expr[1] == True:
			e += "true"
		elif expr[1] == False:
			e += "false"
		elif expr[1] == "~":
			e += "!" + printProExpr(expr[2])
		else:
			e += printProExpr(expr[1])
	elif expr[0] == "PRIM_p":
		e += "("
		e += printProExpr(expr[1])
		e += ")"
	elif expr[0] in ["PLUS", "PLUS_ID", "PLUS_ID0", "PLUS_ID1"]:
		e += printProExpr(expr[1]) 
		e += " + " + printProExpr(expr[2])
	elif expr[0] == "BEQ":
		e += printProExpr(expr[1]) 
		e += " == " + printProExpr(expr[2])
	elif expr[0] == "INT":
		e += str(expr[1])
	elif expr[0] == "GUARD":
		e += "nempty(probe" + expr[1] + ") && "
		e += printProExpr(expr[2])
	elif expr[0] == "GUARD_p":
		e += "nempty(probe" + expr[1] + ")" 
	elif expr[0] == "GUARD_e":
		e += printProExpr(expr[1])
	elif expr == "ELSE":
		e += "else"
	elif isinstance(expr, str):
		# ID
		e += expr
	else:
		raise NotImplementedError("Unrecognized expression", str(expr))
	return e

def printProComm(comm, probed, shared):
	'''Prints a CHP communication action as a Promela communication action'''
	c = ""
	if comm[0] == "SEND_C" and (comm[1] in probed):
		c += "probe" + comm[1] + "!e;"
		c += "atomic{" + comm[1] + "!ctrl;"
		c += "probe" + comm[1] + "?_;"
		c += "(empty(probe" + comm[1] + "))}"
	elif comm[0] == "SEND_C" and (comm[1] in shared):
		c += "probe" + comm[1] + "!e;"
		c += "atomic{" + comm[1] + "!ctrl;"
		c += "probe" + comm[1] + "?_}"
	elif comm[0] == "SEND_C":
		c += comm[1] + "!ctrl"
	elif comm[0] == "RECV_C" and (comm[1] in probed):
		c += "probe" + comm[1] + "!e;"
		c += "atomic{" + comm[1] + "?ctrl;"
		c += "probe" + comm[1] + "?_;"
		c += "(empty(probe" + comm[1] + "))}"
	elif comm[0] == "RECV_C" and (comm[1] in shared):
		c += "probe" + comm[1] + "!e;"
		c += "atomic{" + comm[1] + "?ctrl;"
		c += "probe" + comm[1] + "?_}"
	elif comm[0] == "RECV_C":
		c += comm[1] + "?ctrl"
	elif comm[0] == "SEND" and ((comm[1] in probed) or (comm[1] in shared)):
		c += "probe" + comm[1] + "!e;"
		c += "atomic{" + comm[1] + "!" + printProExpr(comm[2]) + ";"
		c += "probe" + comm[1] + "?_;"
		c += "(empty(probe" + comm[1] + "))}"
	elif comm[0] == "SEND":
		c += comm[1] + "!" + printProExpr(comm[2]) + ";"
	elif comm[0] == "RECV" and ((comm[1] in probed) or (comm[1] in shared)):
		c += "probe" + comm[1] + "!e;"
		c += "atomic{" + comm[1] + "?" + comm[2] + ";"
		c += "probe" + comm[1] + "?_;"
		c += "(empty(probe" + comm[1] + "))}"
	elif comm[0] == "RECV":
		c += comm[1] + "?" + comm[2]
	else:
		raise NotImplementedError("Unrecognized communication on ", comm[1])
	return c

# Counter to keep track of how many repetition markers we've put down.
repcount = 0
# And for processes
proccount = 0

def guardAssert(gcs):
	'''Given a set of guarded commands, prints out the assertion that the guards
  are mutex.'''
	me = ""
	probed = []
	for i in range(len(gcs)):
		gc1 = gcs[i]
		if gc1[1] == "ELSE":
			continue
		probed = collectProbedGuard(gc1[1], probed)
		for j in range(len(gcs[i:])-1):
			gc2 = gcs[i+j+1]
			if gc2[1] == "ELSE":
				continue
			g1e = printProExpr(gc1[1])
			g2e = printProExpr(gc2[1])
			# What we want to check is that no more than one of these is true,
			#  or, NAND(g1e, g2e).  However, we use probes for nempty, and SPIN doesn't
			#  support negated nempty for some reason. So, using demorgan's, if there are
			#  nempty in either of them, we translate it into using empty, and negate it,
			#  then "||" the two of them together.
			# XXX: Assumes only a single probe in each expression.
			if g1e[0:6] == "nempty" and g2e[0:6] == "nempty":
				me += "(("
				me += g1e[1:]
				me += ") || ("
				me += g2e[1:]
				me += ")) &&"
			elif g1e[0:6] == "nempty":
				me += "(("
				me += g1e[1:]
				me += ") || (!("
				me += g2e
				me += "))) &&"
			elif g2e[0:6] == "nempty":
				me += "((!("
				me += g1e
				me += ")) || ("
				me += g2e[1:]
				me += ")) &&"
			else:
				me += "!(("
				me += g1e
				me += ") && ("
				me += g2e
				me += ")) && "
	if me == "":
		# If there are no guards for some reason, assume trivial guard
		me = "true"
	else:
		# Otherwise, remove the trailing ||
		me = me[:-3]
	return (me, probed)

def printProbed(prbd):
	'''Prints out an assertion that any channels probed in the guard never have
  more than a single active user of the channel when the probe is active. This
  serves to identify guard probe instabiltiies.'''
	if len(prbd) > 0:
		me = "&& ("
	else:
		return ""
	for p in prbd:
		me += "(len(probe" + p + ") <= 1) &&"
	# remove trailing && and add the closing paren
	me = me[:-3]
	me += ")"
	return me

def printPromela(program, probed, shared):
	global repcount
	global proccount
	'''Prints a CHP program as a Promela program'''
	prog = ""
	if program[0] == "NEW":
		# no great notion of scoping in Promela... throw this information out =/
		# NOTE: This doesn't work great when mixing parallel and new... some exampes that have 
		#   odd nesting don't work right with this.... (e.g. Ned6)
		prog += printPromela(program[2], probed, shared)
	elif program[0] == "REP":
		# Put down a marker, then print the program, then a goto
		mark = "mark" + str(repcount)
		repcount += 1
		prog += mark + ":\n"
		prog += printPromela(program[1], probed, shared) + ";\n"
		prog += "goto " + mark + "\n"
	elif program[0] == "SEQ":
		pr = ""
		for p in program[1]:
			pr += printPromela(p, probed, shared) + ";"
		# Strip off last semicolon
		prog += pr[:-1]
	elif program[0] == "PAR":
		# Create a separate active process for each element in the par block
		for p in program[1]:
			prog += "active proctype P" + str(proccount) + "() {\n"
			proccount += 1
			prog += printPromela(p, probed, shared)
			prog += "\n}\n"
	elif program[0] == "GC":
		# Guarded commands are always preceeded with ::
		prog += "  ::" + printProExpr(program[1]) + " -> " + printPromela(program[2], probed, shared) + "\n"
	elif program[0] == "SELECT_ONE":
		prog += "if \n"
		prog += printPromela(program[1], probed, shared)
		prog += "fi\n"
	elif program[0] in ["SELECT_DET", "SELECT_UDET"]:
		if program[0] == "SELECT_DET":
			prog += "assert(" 
			(gassert, prbd) = guardAssert(program[1][1])
			prog += gassert
			prog += printProbed(prbd)
			prog += ");\n"
		prog += "if \n"
		for gc in program[1][1]:
			prog += printPromela(gc, probed, shared)
		prog += "fi \n"
	elif program[0] in "COMM":
		prog += printProComm(program[1], probed, shared)
	elif program == "SKIP":
		prog += "skip"
	else:
		raise NotImplementedError("Unexpected Program: " + str(program))
	return prog

# A few definitions of environmental processes, to instantiate if needed
prb_src_bool = """proctype prb_source_bool (chan probeS; chan S) {
  endmark: 
		probeS!e; atomic{
    if
      :: S!true
      :: S!false
    fi; probeS?_; (empty(probeS))};
		goto endmark
}\n"""

prb_src_ctrl = """proctype prb_source_ctrl (chan probeS; chan S) {
  endmark:
		probeS!e; atomic{S!ctrl; probeS?_; (empty(probeS))};
		goto endmark
}\n"""

prb_sink_bool = """proctype prb_sink_bool (chan probeS; chan S) {
  bool sbtemp;
  endmark:
		probeS!e; atomic{S?_; probeS?_; (empty(probeS))};
		goto endmark
}\n"""

prb_sink_ctrl = """proctype prb_sink_ctrl (chan probeS; chan S) {
  endmark:
    probeS!e; atomic{S?_; probeS?_; (empty(probeS))};
		goto endmark
}\n"""

prb_sink_int = """proctype prb_sink_int (chan probeS; chan S) {
	endmark:
		probeS!e; atomic{S?_; probeS?_; (empty(probeS))};
		goto endmark
}\n"""

src_bool = """proctype source_bool (chan S) {
  endmark: 
    if
      :: S!true
      :: S!false
    fi; 
		goto endmark
}\n"""

src_ctrl = """proctype source_ctrl (chan S) {
  endmark:
		S!ctrl;
		goto endmark
}\n"""

sink_bool = """proctype sink_bool (chan S) {
  endmark:
		S?_;
		goto endmark
}\n"""

sink_ctrl = """proctype sink_ctrl (chan S) {
  endmark:
    S?_;
		goto endmark
}\n"""

sink_int = """proctype sink_int (chan S) {
	endmark:
		S?_;
		goto endmark
}\n"""
def printEnv(program, probed, shared):
	'''Prints out an environment for senders and recievers.'''
	printed = []
	init = False
	me = ""

	# First do the integer sources, since we need to know the maximum value
	sc = 0
	for k in ctype.keys():
		if ctype[k][0] == C_RECV and ctype[k][1] == C_INT:
			v = ""
			while not v.isdigit():
				print "What is the maximum value of integer channel " + k + "?"
				v = raw_input()
				if not v.isdigit():
					print "Not a valid digit."
			v = int(v)
			me += "active proctype SI" + str(sc) + "() {\n"
			me += "  do\n"
			for i in range(v+1):
				me += "    :: " + k + "!" + str(i) + "\n"
			me += "  od\n"
			me += "}\n"
			sc += 1

	# Next, check to see which of the processes we'll need
	for k in ctype.keys():
		if ((k not in probed) and (k not in shared)):
			if ctype[k][0] == C_RECV:
				if ctype[k][1] == C_CTRL:
					if 0 not in printed:
						me += src_ctrl
						printed.append(0)
				elif ctype[k][1] == C_BOOL:
					if 1 not in printed:
						me += src_bool
						printed.append(1)
			elif ctype[k][0] == C_SEND:
				if ctype[k][1] == C_CTRL:
					if 2 not in printed:
						me += sink_ctrl
						printed.append(2)
				elif ctype[k][1] == C_BOOL:
					if 3 not in printed:
						me += sink_bool
						printed.append(3)
				elif ctype[k][1] == C_INT:
					if 4 not in printed:
						me += sink_int
						printed.append(4)
		else:
			# k in probed or shared
			if ctype[k][0] == C_RECV:
				if ctype[k][1] == C_CTRL:
					if 5 not in printed:
						me += prb_src_ctrl
						printed.append(5)
				elif ctype[k][1] == C_BOOL:
					if 6 not in printed:
						me += prb_src_bool
						printed.append(6)
			elif ctype[k][0] == C_SEND:
				if ctype[k][1] == C_CTRL:
					if 7 not in printed:
						me += prb_sink_ctrl
						printed.append(7)
				elif ctype[k][1] == C_BOOL:
					if 8 not in printed:
						me += prb_sink_bool
						printed.append(8)
				elif ctype[k][1] == C_INT:
					if 9 not in printed:
						me += prb_sink_int
						printed.append(9)

	# Instantiate the sources and sinks
	for k in ctype.keys():
		if ctype[k][1] == C_DATA:
			raise Exception("Untyped channel!\n")
		if ctype[k][0] in [C_SEND, C_RECV] and not init:
			if ctype[k][0] == C_RECV and ctype[k][1] == C_INT:
				pass
			else:
				me += "init {\n"
				init = True
		if ((k not in probed) and (k not in shared)):
			# Sends
			if ctype[k][0] == C_RECV and ctype[k][1] == C_CTRL:
				me += "  run source_ctrl(" + k + ")\n"
			elif ctype[k][0] == C_RECV and ctype[k][1] == C_BOOL:
				me += "  run source_bool(" + k + ")\n"
			# Recvs
			elif ctype[k][0] == C_SEND and ctype[k][1] == C_CTRL:
				me += "  run sink_ctrl(" + k + ")\n"
			elif ctype[k][0] == C_SEND and ctype[k][1] == C_BOOL:
				me += "  run sink_bool(" + k + ")\n"
			elif ctype[k][0] == C_SEND and ctype[k][1] == C_INT:
				me += "  run sink_int(" + k + ")\n"
			else:
				pass
		else:
			# k in probed or shared
			# Sends
			if ctype[k][0] == C_RECV and ctype[k][1] == C_CTRL:
				me += "  run prb_source_ctrl(probe" + k + ", " + k + ")\n"
			elif ctype[k][0] == C_RECV and ctype[k][1] == C_BOOL:
				me += "  run prb_source_bool(probe" + k + ", " + k + ")\n"
			# Recvs
			elif ctype[k][0] == C_SEND and ctype[k][1] == C_CTRL:
				me += "  run prb_sink_ctrl(probe" + k + ", " + k + ")\n"
			elif ctype[k][0] == C_SEND and ctype[k][1] == C_BOOL:
				me += "  run prb_sink_bool(probe" + k + ", " + k + ")\n"
			elif ctype[k][0] == C_SEND and ctype[k][1] == C_INT:
				me += "  run prb_sink_int(probe" + k + ", " + k + ")\n"
			else:
				pass
	if init:
		me += "}\n"

	# Instantiate the safety-checking proces
	me += """never { 
  do 
    :: true
"""
	for k in ctype.keys():
		if k in shared:
			me += "    :: full(probe"
			me += k 
			me += ") -> break\n"
	me += """  od
}"""

	return me

# Testing
if __name__ == "__main__":
	if len(sys.argv) < 3:
		print "Usage: chp_promela.py input.chp output.pml"
		quit()

	fin = open(sys.argv[1], 'r')
	if fin is None:
		print "Could not open file:", sys.argv[1]
		quit()

	fout = open(sys.argv[2], 'w')
	if fout is None:
		print "Could not open file:", sys.argv[2]
		quit()

	input = fin.read()
	fin.close()
	program = chp_parse.parser.parse(input)

	probed = set(collectProbed(program))
	shared = collectShared(program)

	# ctrl - Data type for dataless channels
	# e -- epsilon for probe channels
	if len(shared) > 0 or len(probed) > 0:
		outp = "mtype {ctrl, e};\n"
	else:
		outp = "mtype {ctrl};\n"

	# Print out the channel delcarations
	ctype = channelType(program)
	(cp, ctype) = printChans(ctype, probed, shared)
	if cp is not None:
		outp += cp

	# Print out the variable declarations
	vtype = varType(program, ctype)
	(vp, vtype) = printVars(vtype)
	if vp is not None:
		outp += vp

	# Print out the program in the promela language
	if program[0] not in ["PAR", "NEW"]:
		pp = "active proctype P() {\n"
		pp += printPromela(program, probed, shared)
		pp += "\n}\n"
	
	else:
		pp = printPromela(program, probed, shared)

	if pp is not None:
		outp += pp

	# Print out an envrionement if needed, as well as the safety properties
	#  SPIN can't handle non-closed programs, so these envrionments are needed to close it
	pe = printEnv(ctype, probed, shared)
	if pe is not None:
		outp += pe

	fout.write(outp)
	fout.close()
