# Parser for CHP for POPL'14 project
#
#  Depends on ply
#
# 0.1 - Stephen Longfield, 10 Dec 2012
#       - Initial creation
# 0.2 - Stephen Longfield, 10 June 2014
#       - Changing repeat to use *( P ) syntax
#       - Adding non-deterministic choice
#       - Making booleans booleans instead of strings
#       - Adding separate data and dataless channel actions
# 0.3 - Brittany Nkounkou, 17 June 2014
#       - commented out ASSIGN
# 0.4 - Stephen Longfield, 23 June 2014
#       - Making PAR and SEQ arrays instead of pairs

import ply.yacc as yacc
import chp_lex
import sys

tokens = chp_lex.tokens

# Precidence relations
precedence = ( 
  ('left', 'AND'),
  ('left', 'OR'),
  ('left', 'PLUS'),
  ('left', 'EQUALS'),
  ('left', 'ARROW'),
  ('left', 'SEMICOLON'),
)

#Program Constructors
def p_program_nu(b):
  '''program : NEW ID DOT OPEN_PAREN program CLOSE_PAREN'''
  b[0] = ("NEW", b[2], b[5])

def p_program_par(b):
  '''program : program OR program'''
  if b[1][0] == "PAR" and b[3][0] == "PAR":
    b[0] = ("PAR", b[1][1] + b[3][1], 0)
  elif b[1][0] == "PAR":
    b[0] = ("PAR", b[1][1] + [b[3]], 0)
  elif b[3][0] == "PAR":
    b[0] = ("PAR", [b[1]] + b[3][1], 0)
  else:
    b[0] = ("PAR", [b[1], b[3]], 0)

#def p_program_assign(b):
#  '''program : ID GETS bexpr'''
#  b[0] = ("ASSIGN", b[1], b[3])

def p_program_comm(b):
  '''program : comm'''
  b[0] = ("COMM", b[1])

def p_gc(b):
  '''gc : guard ARROW program'''
  b[0] = ("GC", b[1], b[3])

def p_gcdet(b):
  '''gcdet : gc BOX gc
           | gcdet BOX gc'''
  if b[1][0] == "GCDET":
    b[0] = ("GCDET", b[1][1]+[b[3]])
  else:
    b[0] = ("GCDET", [b[1], b[3]])

def p_gcudet(b):
  '''gcudet : gc BAR gc
            | gcudet BAR gc'''
  if b[1][0] == "GCUDET":
    b[0] = ("GCUDET", b[1][1]+[b[3]])
  else:
    b[0] = ("GCUDET", [b[1], b[3]])

def p_program_selectone(b):
  '''program : OPEN_BRACKET gc CLOSE_BRACKET'''
  b[0] = ("SELECT_ONE", b[2])

def p_program_selectdet(b):
  '''program : OPEN_BRACKET gcdet CLOSE_BRACKET'''
  b[0] = ("SELECT_DET", b[2])

def p_program_selectudet(b):
  '''program : OPEN_BRACKET gcudet CLOSE_BRACKET'''
  b[0] = ("SELECT_UDET", b[2])

def p_program_seq(b):
  '''program : program SEMICOLON program'''
  if b[1][0] == "SEQ" and b[3][0] == "SEQ":
    b[0] = ("SEQ", b[1][1] + b[3][1])
  elif b[1][0] == "SEQ":
    b[0] = ("SEQ", b[1][1] + [b[3]])
  elif b[3][0] == "SEQ":
    b[0] = ("SEQ", [b[1]] + b[3][1])
  else:
    b[0] = ("SEQ", [b[1], b[3]])

def p_program_rep(b):
  '''program : STAR OPEN_PAREN program CLOSE_PAREN'''
  b[0] = ("REP", b[3])

def p_program_skip(b): 
  '''program : SKIP'''
  b[0] = ("SKIP")

def p_program_paren(b): 
  '''program : OPEN_PAREN program CLOSE_PAREN'''
  b[0] = b[2]

# Communication Constructors
def p_comm_send_data(b):
  '''comm : ID BANG bexpr
          | ID BANG iexpr'''
  b[0] = ("SEND", b[1], b[3])

def p_comm_recv_data(b):
  '''comm : ID QUESTION ID'''
  b[0] = ("RECV", b[1], b[3])

def p_comm_send_ctrl(b):
  '''comm : ID BANG BANG'''
  b[0] = ("SEND_C", b[1])

def p_comm_recv_ctrl(b):
  '''comm : ID QUESTION QUESTION'''
  b[0] = ("RECV_C", b[1])

# Guard Constructors
def p_guard_expr(b):
  '''guard : bexpr 
           | PROBE ID
           | PROBE ID AND guard'''
  if len(b) == 2:
    b[0] = ("GUARD_e", b[1])
  elif len(b) == 3:
    b[0] = ("GUARD_p", b[2])
  else:
    b[0] = ("GUARD", b[2], b[4])

def p_guard_else(b):
  '''guard : ELSE'''
  b[0] = ("ELSE")

# Expression Constructors
def p_bexpr(b):
  '''bexpr : conj
           | conj AND bexpr'''
  if len(b) == 2:
    b[0] = ("EXPR", b[1])
  else:
    b[0] = ("EXPR", b[1], b[3])

# Conjunction Constructors
def p_conj(b):
  '''conj : prim
          | prim OR conj'''
  if len(b) == 2:
    b[0] = ("CONJ", b[1])
  else:
    b[0] = ("CONJ", b[1], b[3])

# Primary
def p_bprim_true(b):
  '''prim : TRUE'''
  b[0] = ("PRIM", True)

def p_bprim_false(b):
  '''prim : FALSE'''
  b[0] = ("PRIM", False)

def p_bprim_b(b):
  '''prim : ID
          | NOT prim
          | OPEN_PAREN bexpr CLOSE_PAREN'''
  if len(b) == 2:
    b[0] = ("PRIM", b[1])
  elif len(b) == 3:
    b[0] = ("PRIM", b[1], b[2])
  elif len(b) == 4:
    b[0] = ("PRIM_p", b[2]) 

def p_bprim_i(b):
  '''prim : iexpr EQUALS iexpr
          | ID    EQUALS iexpr
          | iexpr EQUALS ID'''
  b[0] = ("BEQ", b[1], b[3])

def p_iexpr(b):
  '''iexpr : INT
           | ID PLUS iexpr
           | iexpr PLUS ID
           | iexpr PLUS iexpr
           | ID PLUS ID'''
  if len(b) == 2:
    b[0] = ("INT", b[1])
  elif b[1][0][0:4] == "PLUS":
    if b[3][0][0:4] == "PLUS":
      b[0] = ("PLUS", b[1], b[3])
    else:
      b[0] = ("PLUS_ID1", b[1], b[3])
  elif b[3][0][0:4] == "PLUS":
    b[0] = ("PLUS_ID0", b[1], b[3])
  else:
    b[0] = ("PLUS_ID", b[1], b[3])

def p_iexpr_paren(b):
  '''iexpr : OPEN_PAREN iexpr CLOSE_PAREN'''
  b[0] = b[2]

# Junky error handling
def p_error(p):
  if not p:
    print('Syntax error at EOF')
  else:
    print('ERROR!')
    print p

# Build the parser
parser = yacc.yacc()

# Test
if __name__ == '__main__':
  inp = sys.stdin.read()
  print parser.parse(inp)
