# CHP to Machine Translator
#
# 0.1 - Brittany Nkounkou, 18 Jun 2014
#       - Basic layout with TODO's
#       - added Test
# 0.2 - Stephen Longfield, 23 June 2014
#       - Extending to handle N-Ary PAR and SEQ

#from sets import Set
from machine_defs import *
from machine_funs import *
from chp_funs import *
import chp_parse
import sys
import copy
import operator
import psutil

# For counting states
states = 0
max_states = 0

# Transforms a M_SMP, M_SEQ, M_SEL or M_PAR into a M_NFA
#
# Uses a stack-based method to traverse the tree, with three stacks: 
#
#  working - working stack
#          - Elements are tuples: (Machine, start, Continuation, repeat)
#            - First element is a machine, one of M_SMP, M_SEQ, M_SEL, or M_PAR
#            - Second element is a node number in the NFA, representing the previous
#              element's output
#            - Third element is a continuation (possibly None) representing the
#              next action to process after the first
#            - Fourth element is None if the element doesn't repeat, and the node
#              number of the beginning of the repeat if it does
#  continuation - continuation stack
#               - Elements are stacks
#               - This is for instances where we have SEL; P or PAR; P, since
#                 we need to process all elements of SEL/PAR before processing P
#               - First element in the stack should have start = None
#  final - Array to be used with continuations. Whenever a element on the working
#          stack has None as a continuation, the last-introduced node is appended
#          to this list
#  final_stack - stack of final arrays.
# 
#
def NFA_Machine(machine, probed):
	global states
	global max_states
	temp_states = states

	if machine.type == M_NFA:
		return machine

	assert machine.type in [M_SMP, M_SEQ, M_SEL, M_PAR]
	# Initialize an empty NFA
	nfa = NFA()

	# Initialize stacks
	working = [(machine, nfa.start, None, None)]
	continuation = []
	final = []
	final_stack = []
	start = None
	states += 1

	while len(working) > 0 or len(continuation) > 0:
		#print "WORKING", working
		if len(working) > 0:
			(m, s, c, r) = working.pop()
			# Check to see if this is a repeat
			rold = r
			if m.rep:
				r = s
			# Simple machines are either send or recieve
			if m.type == M_SMP:
				#print "SIMPLE"
				if len(m.snds) == 1:
					n = nfa.addNode()
					states += 1
					chan = list(m.snds)[0]
					l = Label(L_SND, chan, None)
					edge = Edge(s, l, n)
					nfa.addEdge(edge)
					if c is None:
						if r is None:
							final.append(n)
						else:
							nfa.mergeNodes([n], r)
					else:
						working.append((c, n, None, r))
				elif len(m.rcvs) == 1:
					n = nfa.addNode()
					states += 1
					chan = list(m.rcvs)[0]
					l = Label(L_RCV, chan, None)
					edge = Edge(s, l, n)
					nfa.addEdge(edge)
					if c is None:
						if r is None:
							final.append(n)
						else:
							nfa.mergeNodes([n], r)
					else:
						working.append((c, n, None, r))
				# Skip case, no label on edge
				elif len(m.prbs) == len(m.snds) == len(m.rcvs) == len(m.occs) == 0:
					n = nfa.addNode()
					states += 1
					l = Label(L_NUL, None, None)
					edge = Edge(s, l, n)
					nfa.addEdge(edge)
					if c is None:
						if r is None:
							final.append(n)
						else:
							nfa.mergeNodes([n], r)
					else:
						working.append((c, n, None, r))
				else:
					raise NotImplementedError("Badly formed M_SMP")
			# Guards, which may either be exclusive or non-exclusive
			elif m.type == M_GRD:
				#print "GUARD"
				# Guards should never appear as the last element in a chain
				assert c is not None
				# Content will be populated on exclusive guards, empty on non-exclusive
				if m.content is not None:
					n = nfa.addNode()
					states += 1
					assert len(list(m.prbs)) <= 1
					if len(list(m.prbs)) == 0:
						l = Label(L_XPR, "", m.content)
					else:
						l = Label(L_XPR, list(m.prbs)[0], m.content)
					edge = Edge(s, l, n)
					nfa.addEdge(edge)
					working.append((c, n, None, r))
				else:
					n = nfa.addNode()
					states += 1
					assert len(list(m.prbs)) <= 1
					if len(list(m.prbs)) == 0:
						l = Label(L_PRB, "", [])
					else:
						l = Label(L_PRB, list(m.prbs)[0], [])
					edge = Edge(s, l, n)
					nfa.addEdge(edge)
					working.append((c, n, None, r))
			# Sequence just moves the first element into the to-process place and puts the
			#  rest as a continuation. If there is only a single element in the remainder,
			#  this removes the M_SEQ wrapper
			elif m.type == M_SEQ:
				#print "SEQUENCE"
				# M_SEQ should always have at least two elements
				assert len(m.content) > 1
				if len(m.content) == 2:
					nxt = m.content[0]
					cnt = m.content[1]
					working.append((nxt, s, cnt, r))
				else:
					nxt = m.content[0]
					# Make the continuation, toss out prbs, snds, rcvs, occs, and rep
					cnt = Machine(False, M_SEQ, None, None, None, None, False, m.content[1:])
					working.append((nxt, s, cnt, r))
			# Selection puts its continuation at the head of the queue, and then pushes
      #  the current value of final onto the final stack, as well as pushing the
      #  entire stack onto the continuation stack.  It then populates the working
      #  stack with the branches of the selection.  PAR is the same.
			elif m.type in [M_SEL, M_PAR]:
				if c is not None:
					working.append((c, None, None, r))

				continuation.append(copy.copy(working))
				working = []

				final_stack.append(copy.copy(final))
				final = []

				for d in m.content:
					# If there's a continuation, clear the value of repeat
					if c is not None:
						working.append((d, s, None, None))
					else:
						working.append((d, s, None, r))
			else:
				raise NotImplementedError("Bad Machine Type")
		# If there are no elements on the working stack, we pop off the continuation stack,
    #  and make that the new working stack.  Next, to sequence things correctly, we 
		#  join all of the final nodes together into a single node, and set that node to
    #  the start node of the top element of the working set if the current start node is
    #  None.  It then restores the value of finish from the finish_stack.
		elif len(continuation) > 0:
			working = continuation.pop()
			n = nfa.addNode()
			states += 1
			nfa.mergeNodes(final, n)

			if len(working) > 0:
				(m, s, c, r) = working.pop()
				if s is None:
					working.append((m, n, c, r))
				else:
					working.append((m, s, c, r))

			final = final_stack.pop()

	# Expand the probed edges into start;use;finish edges
	nfa.expandEdges(probed)

	# Keep track of how many states were made
	max_states = max(max_states, states-temp_states)

	# Once both the working stack and the continuation stack are empty, return the generated
	#  NFA.
	return Machine(False, M_NFA, machine.prbs, machine.snds, machine.rcvs, machine.occs, machine.rep, nfa)

# Takes two NFAs, and generates the NFA cross-product
# assumes that they interact, and that you want to make the NFA
# generates errors from faulty as it builds the NFA.
# Baked-in reductions
#   If a node can only be reached by traversing a faulty edge, that node
#     is not in the final NFA
def NFA_Cross(nfa1, nfa2):
	global states
	global max_states
	temp_states = states
	nfa = NFA()

	# Do this like DFS, starting from the start nodes of the two NFA's we're building
  #  from, and only adding nodes if they can be reached through non-erroneous edges
	stack = [(nfa1.start, nfa2.start)]

	# Map from node pairs in original graph to node pairs in new graph
	node_map = {}
	node_map[stack[0]] = nfa.start

	# Build a map from nodes to edges for each of the nfas
	#nfa1.buildOutgoing()
	#nfa2.buildOutgoing()

	# While building the NFA, build up a new set of stats
	prbs = set([])
	snds = set([])
	rcvs = set([])
	occs = set([])

	while len(stack) > 0:
		n = stack.pop()

		#print n

		# First check the nodes that can be reached using only edges from the second NFA 
		for e in nfa2.outgoing[n[1]]:
			el = nfa2.edgeLabel[e]
			if (n[0], el.tail) in node_map:
				tgt = node_map[(n[0], el.tail)]
			else:
				tgt = nfa.addNode()
				node_map[(n[0], el.tail)] = tgt
				states += 1
				stack.append((n[0], el.tail))
			new_edge = Edge(node_map[n], el.label, tgt)
			nfa.addEdge(new_edge)
			(prbs, snds, rcvs, occs) = updateStats(prbs, snds, rcvs, occs, el.label)

		# Then check the nodes that can be reached using edges from the first NFA, or 
    #  diagonal edges from both NFAs. For diagonal edges, check for errors.
		for e1 in nfa1.outgoing[n[0]]:
			el1 = nfa1.edgeLabel[e1]
			if (el1.tail, n[1]) in node_map:
				tgt = node_map[(el1.tail, n[1])]
			else:
				tgt = nfa.addNode()
				node_map[(el1.tail, n[1])] = tgt
				states += 1
				stack.append((el1.tail, n[1]))
			new_edge = Edge(node_map[n], el1.label, tgt)
			nfa.addEdge(new_edge)
			(prbs, snds, rcvs, occs) = updateStats(prbs, snds, rcvs, occs, el1.label)
			for e2 in nfa2.outgoing[n[1]]:
				el2 = nfa2.edgeLabel[e2]
				label = composeLabels(el1.label, el2.label)
				(prbs, snds, rcvs, occs) = updateStats(prbs, snds, rcvs, occs, label)

				if (el1.tail, el2.tail) in node_map:
					tgt = node_map[(el1.tail, el2.tail)]
				else:
					tgt = nfa.addNode()
					node_map[(el1.tail, el2.tail)] = tgt
					if label.type not in [L_IST, L_INT, L_SER]:
						stack.append((el1.tail, el2.tail))
						states += 1
				new_edge = Edge(node_map[n], label, tgt)
				nfa.addEdge(new_edge)

	max_states = max(max_states, states-temp_states)

	return (prbs, snds, rcvs, occs, nfa)

# Simila to NFA_Cross, but takes a channel as an argument. Checks this
#  channel for compliance when edges are created.
def NFA_Cross_New(nfa1, nfa2, chan):
	global states
	global max_states
	temp_states = states
	nfa = NFA()

	stack = [(nfa1.start, nfa2.start)]

	node_map = {}
	node_map[stack[0]] = nfa.start

	prbs = set([])
	snds = set([])
	rcvs = set([])
	occs = set([])

	while len(stack) > 0:
		n = stack.pop()

		for e in nfa2.outgoing[n[1]]:
			add = None
			el = nfa2.edgeLabel[e]
			if (n[0], el.tail) in node_map:
				tgt = node_map[(n[0], el.tail)]
				add = False
			else:
				tgt = nfa.addNode()
				node_map[(n[0], el.tail)] = tgt
				add = True
			(compliant, l) = compliantLabel(el.label, chan)
			if compliant == 0:
				if add:
					node_map.pop((n[0], el.tail))
			else:
				if add:
					stack.append((n[0], el.tail))
					states += 1
				if compliant == 1:
					new_edge = Edge(node_map[n], l, tgt)
				elif compliant == 3:
					new_edge = Edge(node_map[n], l, node_map[n])
				else:
					assert compliant == 2
					new_edge = Edge(node_map[n], el.label, tgt)
				nfa.addEdge(new_edge)
				(prbs, snds, rcvs, occs) = updateStats(prbs, snds, rcvs, occs, new_edge.label)

		for e1 in nfa1.outgoing[n[0]]:
			el1 = nfa1.edgeLabel[e1]
			if (el1.tail, n[1]) in node_map:
				tgt = node_map[(el1.tail, n[1])]
				add = False
			else:
				tgt = nfa.addNode()
				node_map[(el1.tail, n[1])] = tgt
				add = True

			(compliant, l) = compliantLabel(el1.label, chan)
			if (compliant == 0):
				if add:
					node_map.pop((el1.tail, n[1]))
			else:
				if add:
					states += 1
					stack.append((el1.tail, n[1]))
				if compliant == 1:
					new_edge = Edge(node_map[n], l, tgt)
				elif compliant == 3:
					new_edge = Edge(node_map[n], l, node_map[n])
				else:
					assert compliant == 2
					new_edge = Edge(node_map[n], el1.label, tgt)
				nfa.addEdge(new_edge)
				(prbs, snds, rcvs, occs) = updateStats(prbs, snds, rcvs, occs, new_edge.label)

			for e2 in nfa2.outgoing[n[1]]:
				el2 = nfa2.edgeLabel[e2]

				label = composeLabels(el1.label, el2.label)

				add = True

				if (el1.tail, el2.tail) in node_map:
					tgt = node_map[(el1.tail, el2.tail)]
					add = False
				else:
					tgt = nfa.addNode()

					if label.type in [L_IST, L_INT, L_SER]:
						add = False
					else:
						node_map[(el1.tail, el2.tail)] = tgt
						add = True

				(compliant, l) = compliantLabel(label, chan)

				if (compliant == 0):
					if add:
						node_map.pop((el1.tail, el2.tail))
				else:
					if add:
						states += 1
						stack.append((el1.tail, el2.tail))
					if compliant == 1:
						new_edge = Edge(node_map[n], l, tgt)
					elif compliant == 2:
						new_edge = Edge(node_map[n], label, tgt)
					else:
						assert compliant == 3
						new_edge = Edge(node_map[n], l, node_map[n])
					nfa.addEdge(new_edge)
					(prbs, snds, rcvs, occs) = updateStats(prbs, snds, rcvs, occs, new_edge.label)

	nfa.filterCompliant(chan)

	nfa.reduceNFA()
	max_states = max(max_states, states-temp_states)

	return (prbs, snds, rcvs, occs, nfa)


# parallelizes two machines together
def Par_Machine(machine1, machine2, probed):
	assert not machine1.error
	assert not machine2.error
	rep = machine1.rep or machine2.rep
	# Check conditions between machine 1 and machine 2
	if machine1.type == M_PAR and machine2.type == M_PAR:
		# This is a pain to implement, mostly comes about from nested PAR
		raise NotImplementedError("Two PAR machines.")
	elif machine1.type == M_PAR or machine2.type == M_PAR:
		# Identify which machine is PAR
		if machine1.type == M_PAR:
			pm = machine1
			np = machine2
		else:
			pm = machine2
			np = machine1
		# Check to see where there is an interaction
		interact = checkInteract(pm, np)
		if interact:
			interacting = []
			not_interact = []
			for m in pm.content:
				if checkInteract(m, np):
					interacting.append(m)
				else:
					not_interact.append(m)
			# Compose them together into an NFA
			nfac = NFA_Machine(np, probed).content
			for i in interacting:
				nfai = NFA_Machine(i, probed).content
				(prbs, snds, rcvs, occs, nfac) = NFA_Cross(nfac, nfai)
			machine = Machine(False, M_NFA, prbs, snds, rcvs, occs, rep, nfac)
			# Build the Par_Machine back up out of the non-interacting parts and the NFA
			if len(not_interact) > 0: 
				for ni in not_interact:
					machine = Par_Machine(machine, ni, probed)
			return machine
		else:
			# No interactions, create a PAR
			content = machine1.content + [machine2]
			(prbs, snds, rcvs, occs) = buildCache(machine1, machine2)
			machine = Machine(False, M_PAR, prbs, snds, rcvs, occs, rep, content)
	else:
		# Neither of the machines is M_PAR, check to see if they interact
		interact = checkInteract(machine1, machine2)
		if interact:
			nfa1 = NFA_Machine(machine1, probed).content
			nfa2 = NFA_Machine(machine2, probed).content
			(prbs, snds, rcvs, occs, nfac) = NFA_Cross(nfa1, nfa2)
			machine = Machine(False, M_NFA, prbs, snds, rcvs, occs, rep, nfac)
		else:
			# No interactions, create a PAR
			content = [machine1, machine2]
			(prbs, snds, rcvs, occs) = buildCache(machine1, machine2)
			machine = Machine(False, M_PAR, prbs, snds, rcvs, occs, rep, content)
	return machine

## Sequences two machines together. If they are both sequences, joins the 
#    content lists.
def Seq_Machine(machine1, machine2, probed):
	assert not machine1.error
	assert not machine2.error
	assert canTerminate(machine1)
	(prbs, snds, rcvs, occs) = buildCache(machine1, machine2)
	if machine1.type == M_SEQ and machine2.type == M_SEQ:
		content = machine1.content + machine2.content
	elif machine1.type == M_SEQ:
		content = machine1.content + [machine2]
	elif machine2.type == M_SEQ:
		content = [machine1] + machine2.content
	else:
		content = [machine1, machine2]
	return Machine(False, M_SEQ, prbs, snds, rcvs, occs, False, content)

# translates a guard to a machine
def Guard_Machine(guard, exclusive, chans=set([])):
	global states
	type = guard[0]
	
	if type == "GUARD_e":
		if exclusive:
			return Machine(False, M_GRD, set([]), set([]), set([]), set([]), False, chans)
		else:
			return Machine(False, M_GRD, set([]), set([]), set([]), set([]), False, None)
	
	elif type == "GUARD_p":
		channel = guard[1]
		if exclusive:
			return Machine(False, M_GRD, set([channel]), set([]), set([]), set([]), False, chans - set([channel]))
		else:
			return Machine(False, M_GRD, set([channel]), set([]), set([]), set([]), False, None)
	
	elif type == "GUARD":
		channel = guard[1]
		if exclusive:
			machine1 = Machine(False, M_GRD, set([channel]), set([]), set([]), set([]), False, None)
			machine2 = Guard_Machine(guard[2])
			prbs = machine1.prbs.union(machine2.prbs)
			excl = chans - prbs
			machine = Machine(False, M_GRD, prbs, set([]), set([]), set([]), False, excl)
			return machine
		else:
			machine1 = Machine(False, M_GRD, set([channel]), set([]), set([]), set([]), False, None)
			machine2 = Guard_Machine(guard[2])
			if machine2.error: return machine2
			return Par_Machine(machine1, machine2, [])
	
	else:
		assert guard == "ELSE"
		return Machine(False, M_GRD, set([]), set([]), set([]), set([]), False, None)

# translates a program to a machine
def Program_Machine(program, probed):
	global max_states
	type = program[0]
	
	if type == "NEW":
		## NEW is where we can reduce programs, if appropriate.  This is handled
		##  by the call to filterCompliant
		channel = program[1]
		machine0 = Program_Machine(program[2], probed)
		if machine0.error: return machine0

		if machine0.type == M_NFA:
			machine0.content.filterCompliant(channel)
			max_states = max(max_states, machine0.content.count_states())
		elif machine0.type == M_PAR:
			# Cross together sub-machines
			if len(machine0.content) > 1:
				m = machine0.content[0]
				if m.type != M_NFA:
					m = NFA_Machine(m, probed)
					m = m.content
				else:
					m = m.content
				for mp in machine0.content[1:]:
					if mp.type == M_NFA:
						(prbs, snds, rcvs, occs, m) = NFA_Cross(m, mp.content)
					else:
						mp = NFA_Machine(mp, probed).content
						(prbs, snds, rcvs, occs, m) = NFA_Cross(m, mp)
				machine0 = Machine(False, M_NFA, prbs, snds, rcvs, occs, False, m)
				machine0.content.filterCompliant(channel)
			else:
				machine0 = machine0.content[0]
				if machine0.type == M_NFA:
					machine0.content.filterCompliant(channel)
				else:
					raise NotImplementedError("NEW around a non-NFA.")
				max_states = max(max_states, machine0.content.count_states())
		else:
			print(machine0)
			raise NotImplementedError("NEW around a non-NFA.")

		#print "NEW", chan, states, max_size

		return machine0

	elif type == "NEW_PAR":
		#print "NEW_PAR", program[1], states, max_states
		## Can create and reduce simultaneously, using NFA_Cross_New
		# Create the two individual sub-machines
		chan = program[1]
		machine0 = Program_Machine(program[2], probed)
		machine1 = Program_Machine(program[3], probed)
		rep = machine0.rep or machine1.rep

		assert machine0.type != M_PAR
		assert machine1.type != M_PAR

		if machine0.type == M_NFA:
			n0 = machine0.content
		else:
			n0 = NFA_Machine(machine0, probed).content

		if machine1.type == M_NFA:
			n1 = machine1.content
		else:
			n1 = NFA_Machine(machine1, probed).content

		(prbs, snds, rcvs, occs, m) = NFA_Cross_New(n0, n1, chan)
		return Machine(False, M_NFA, prbs, snds, rcvs, occs, rep, m)

	elif type == "WIDE_PAR":
		## Has a mapping between channels and the sub-programs that they're used in
    ##  and uses this mapping to do the reduction while trying to create as few
    ##  intermediate states as possible.
		ps = program[2]
		map_chan = program[1]

		for i in range(len(ps)):
			ps[i] = Program_Machine(ps[i], probed)

		# Collect the sizes up. Using the number of edges in the NFA as a proxy
    #  for the size of the NFA
		sizes = {}
		for i in range(len(ps)):
			if ps[i].type == M_NFA:
				sizes[i] = ps[i].content.count_states()
			else:
				sizes[i] = len(ps[i].rcvs)

		no_map = []
		run = True
		while run:
			run = False
			one = False
			two = False
			other = False
			for k in map_chan.keys():
				if len(map_chan[k]) == 1:
					run = True
					one = True
					break
			if not one:
				ks = []
				for k in map_chan.keys():
					if len(map_chan[k]) == 2:
						run = True
						two = True
						lmck = list(map_chan[k])
						ks.append((k, sizes[lmck[0]] * sizes[lmck[1]]))
				# Reduce the smallest
				if two:
					k = ks[0]
					for ki in ks[1:]:
						# Note: this is the line that has the largest affect on the number
						#  of explored states. Changing the < to a > will completely change
						#  the results, sometimes for the better, sometimes for the worse. 
						if ki[1] < k[1]:
							k = ki
					k = k[0]
			#print "Memory Usage: ", resource.getrusage(resource.RUSAGE_SELF)
			if not (one or two):
				for k in map_chan.keys():
					if len(map_chan[k]) > 2:
						run = True
						other = True
						lmck = list(map_chan[k])
						s = reduce(operator.mul, map(lambda x : sizes[x], lmck), 1)
						ks.append((k, s))
				# Expand the smallest
				if other:
					k = ks[0]
					for ki in ks[1:]:
						if ki[1] < k[1]:
							k = ki
					k = k[0]
			if one:
				#print "ONE", k, states, max_states
				# If there are any channels that are used in only a single process, reduce those 
				#  immediately.
				lmck = list(map_chan[k])
				p = ps[lmck[0]]
				if p.type == M_NFA:
					pass
				else:
					p = NFA_Machine(p, probed)
				p.content.filterCompliant(k)
				ps.append(p)
				sizes[len(ps)-1] = len(p.content.edges)
				map_chan = updateMapChan(map_chan, lmck[0], len(ps)-1)
				map_chan.pop(k)
				in_map = searchMapChan(map_chan, len(ps)-1)
				if not in_map:
					no_map.append(len(ps)-1)
			elif two:
				#print "TWO", k, states, max_states
				lmck = list(map_chan[k])
				p0 = ps[lmck[0]]
				p1 = ps[lmck[1]]
				if p0.type == M_NFA:
					np0 = p0.content
				else:
					p0 = NFA_Machine(p0, probed)
					np0 = p0.content
				if p1.type == M_NFA:
					np1 = p1.content
				else:
					p1 = NFA_Machine(p1, probed)
					np1 = p1.content
				rep = p0.rep or p1.rep
				(prbs, snds, rcvs, occs, m)  = NFA_Cross_New(np0, np1, k)
				p = Machine(False, M_NFA, prbs, snds, rcvs, occs, rep, m)
				ps.append(p)
				sizes[len(ps)-1] = len(p.content.edges)
				map_chan = updateMapChan(map_chan, lmck[0], len(ps)-1)
				map_chan = updateMapChan(map_chan, lmck[1], len(ps)-1)
				map_chan.pop(k)
				in_map = searchMapChan(map_chan, len(ps)-1)
				if not in_map:
					no_map.append(len(ps)-1)
			elif other:
				#print "OTHER", k, states, max_states
				lps = list(map_chan[k])
				rep = False
				# turn all of the processes into NFAs and collect rep
				for i in range(len(lps)):
					pp = ps[lps[i]]
					rep = rep or pp.rep
					if pp.type == M_NFA:
						lps[i] = pp.content
					else:
						lps[i] = NFA_Machine(pp, probed).content
				# find the largest
				l = lps[0]
				for pi in lps:
					if len(pi.edges) > len(l.edges):
						l = pi
				lps.remove(l)
				# join all but the largest together using NFA_Cross
				p = lps[0]
				for pi in lps[1:]:
					#	print "TEST", states, len(lps)
					(prbs, snds, rcvs, occs, p) = NFA_Cross(p, pi)
				# join the largest with the rest using NFA_Cross_New
				# and update the data structures
				(prbs, snds, rcvs, occs, m)  = NFA_Cross_New(p, l, k)
				p = Machine(False, M_NFA, prbs, snds, rcvs, occs, rep, m)
				ps.append(p)
				sizes[len(ps)-1] = len(p.content.edges)
				pis = list(map_chan[k])
				for pi in pis:
					map_chan = updateMapChan(map_chan, pi, len(ps)-1)
				map_chan.pop(k)
				in_map = searchMapChan(map_chan, len(ps)-1)
				if not in_map:
					no_map.append(len(ps)-1)
			elif len(no_map) == 1:
				return ps[no_map[0]]
			else:
				print no_map
				assert False
	
	elif type == "PAR":
		## PAR may require expansion of the NFA to correctly represent the system,
		##  this is handled by Par_machine.
		machine = Program_Machine(program[1][0], probed)
		if machine.error: return machine
		for p in program[1][1:]:
			machine2 = Program_Machine(p, probed)
			if machine2.error: return machine2
			machine = Par_Machine(machine, machine2, probed)
		return machine
	
	elif type == "COMM":
		## Basic communication actions are formed by simple lookup
		comm = program[1]
		commtype = comm[0]
		channel = comm[1]
		if commtype in ["SEND", "SEND_C"]:
			return Machine(False, M_SMP, set([]), set([channel]), set([]), set([]), False, None)
		else:
			assert commtype in ["RECV", "RECV_C"]
			return Machine(False, M_SMP, set([]), set([]), set([channel]), set([]), False, None)
	
	elif type == "SELECT_ONE":
		## Single selection is just a wait and can't cause problems by itself
		gc = program[1]
		assert gc[0] == "GC"
		guardM = Guard_Machine(gc[1], False)
		if guardM.error: return guardM
		progM = Program_Machine(gc[2], probed)
		if progM.error: return progM
		return Seq_Machine(guardM, progM, probed)
	
	elif type in ["SELECT_DET", "SELECT_UDET"]:
		## Multiple selection doesn't result in interaction, and so can't cause 
		##  problems by itself.
		gc_ = program[1]
		assert gc_[0] in ["GCDET", "GCUDET"]
		gcs = gc_[1]
		
		# If it's an exclusive selection statement, collect up all the guards first
		if gc_[0] == "GCDET":
			excl = set([])
			for gc in gcs:
				assert gc[0] == "GC"
				guardM = Guard_Machine(gc[1], False)
				excl = excl.union(guardM.prbs)
		
		# Build the initial guard and collection sets
		gc = gcs[0]
		assert gc[0] == "GC"
		if gc_[0] == "GCDET":
			guardM = Guard_Machine(gc[1], True, excl)
			if guardM.error: return guardsM
		else:
			guardM = Guard_Machine(gc[1], False)
			if guardM.error: return guardsM
		progM = Program_Machine(gc[2], probed)
		if progM.error: return progM
		prbs = progM.prbs
		snds = progM.snds
		rcvs = progM.rcvs
		occs = progM.occs
		content = [Seq_Machine(guardM, progM, probed)]

		# Build each guard and the associated program fragments
		for gc in gcs[1:]:
			assert gc[0] == "GC"
			if gc_[0] == "GCDET":
				guardM = Guard_Machine(gc[1], True, excl)
				if guardM.error: return guardM
			else:
				guardM = Guard_Machine(gc[1], False)
				if guardM.error: return guardM
			progM = Program_Machine(gc[2], probed)
			if progM.error: return progM
			prbs = prbs.union(progM.prbs)
			snds = snds.union(progM.snds)
			rcvs = rcvs.union(progM.rcvs)
			occs = occs.union(progM.occs)
			selM = Seq_Machine(guardM, progM, probed)
			content += [selM]
		
		return Machine(False, M_SEL, prbs, snds, rcvs, occs, False, content)
	
	elif type == "SEQ":
		## No errors or reductions can occur from simple sequencing
		machine = Program_Machine(program[1][0], probed)
		if machine.error: return machine
		if not canTerminate(machine): return machine
		for p in program[1][1:]:
			machine2 = Program_Machine(p, probed)
			if machine2.error: return machine2
			machine = Seq_Machine(machine, machine2, probed)
			if not canTerminate(machine): return machine
		return machine
	
	elif type == "REP":
		## No errors or reductions can occur from simple repetition
		machine0 = Program_Machine(program[1], probed)
		if machine0.error: return machine0
		machine0.rep = True
		return machine0
	
	else:
		assert program == "SKIP"
		return Machine(False, M_SMP, set([]), set([]), set([]), set([]), False, None)

# Count the states that would be generated by naive state space exploration
def countNaiveStates(program, probed):
	global states
	type = program[0]

	if type == "NEW":
		return countNaiveStates(program[2], probed)
	elif type == "PAR":
		sc = 1
		i = 1
		for p in program[1]:
			nfa = NFA_Machine(Program_Machine(p, probed), probed).content
			#print "Sub program", str(i), "states:", str(states)
			i += 1
			sc = sc*states
			states = 0
		return sc
	else:
		print type
		assert False

# Test
if __name__ == "__main__":
	input = sys.stdin.read()
	program = chp_parse.parser.parse(input)
	#print program
	sys.stdout.flush()
	probed = set(collectProbed(program))
	#new_program = newPar(copy.deepcopy(program))
	#print probed
	#machine0 = Program_Machine(program, probed)
	#print "Baseline States", states, max_states
	states = 0
	#print printGraphviz(machine0)
	#new_program = newPar(copy.deepcopy(program))
	#print new_program
	#machine1 = Program_Machine(new_program, probed)
	#print printGraphviz(machine1)
	#print "NEW PAR Explored States", states
	#print "NEW PAR Max_States", max_states
	#if machine1.type == M_NFA:
	#	print "NEW PAR size", machine1.content.count_states()
	states = 0
	max_states = 0
	new_program2 = widePar(copy.deepcopy(program))
	print countNaiveStates(copy.deepcopy(program), probed)
	#print new_program2
	if len(new_program2[1].keys()) > 0:
		machine2 = Program_Machine(new_program2, probed)
		print printGraphviz(machine2)
		print "Wide PAR States", states
		print "Wide PAR Max States", max_states
		if machine2.type == M_NFA:
			print "Wide PAR size", machine2.content.count_states()
	else:
		print "No reductions possible. Declare some scopes and try again."

	print "Memory Usage: " + str(psutil.Process().memory_info_ex().data) # USE FOR LINUX
	# print "Memory Usage: " + str(psutil.Process().memory_info().rss) # USE FOR WINDOWS

