#include "profile.h"
#include "dataset.h"

Profile *initProfile(int initspan, int maxspan) {
	Profile *p;
	int i, j;
	if(initspan > maxspan) {
		fprintf(stderr, "Error: initspan > maxspan in initProfile()\n");
	}
	p = (Profile*) malloc(sizeof(Profile));
	p->cols = initspan;
	p->span = initspan;
	p->maxspan = maxspan;
	p->maxspan = p->maxspan;

	p->isgap = (char*) calloc(sizeof(char), p->maxspan); 
	p->mat = (double**) malloc(sizeof(double*) * p->maxspan);
	for(i = 0; i < p->maxspan; i++) {
		p->mat[i] = (double*) malloc(sizeof(double) * 4);
	}
	for(i = 0; i < p->span; i++) {
		for(j = 0; j < NUMALPHAS; j++) {
			p->mat[i][j] = 0.0;
		}
	}
	return p;
}

void nilProfile(Profile *profile) {
	int i;
	free(profile->isgap);
	for( i = 0; i < profile->maxspan; i++) {
		free(profile->mat[i]);
	}
	free(profile->mat);
	free(profile);
}

void copyProfile(Profile *dest, Profile *src) {
	int i, j;
	if(DEBUG0) {
		if(dest->maxspan != src->maxspan) {
			fprintf(stderr, "Error: dest->maxspan != src->maxspan in copyProfile()\n");
			exit(1);
		}
		for(i = 0; i < src->span; i++) {
			for(j = 0; j < 4; j++) {
				if(src->isgap[i] && fabs(src->mat[i][j] - 1.0) > 0.000001){
					fprintf(stderr, "Error: values on gaps are not set to 1.0 in copyProfile()\n");
					exit(1);
				}
			}
		}
	}
	dest->cols = src->cols;
	dest->span = src->span;
	for(i = 0; i < src->span; i++) {
		dest->isgap[i] = src->isgap[i];
	}
	for(i = 0; i < src->span; i++) {
		for(j = 0; j < NUMALPHAS; j++) {
			dest->mat[i][j] = src->mat[i][j];
		}
	}
}

void resetProfile(Profile *profile, int initspan) {
	int i, j;
	if(DEBUG0) {
		if(initspan > profile->maxspan) {
			fprintf(stderr, "Error: initspan > maxspan in resetProfile()\n");
			exit(1);
		}
	}
	profile->cols = initspan;
	profile->span = initspan;
	for(i = 0; i < profile->span; i++) {
		profile->isgap[i] = 0;
	}
	for(i = 0; i < profile->span; i++) {
		for(j = 0; j < NUMALPHAS; j++) {
			profile->mat[i][j] = 0.0;
		}
	}
}
void setProfile(Profile *profile, double val) {
	int i, j;
	for(i = 0; i < profile->span; i++) {
		for(j = 0; j < 4; j++) {
			if(!profile->isgap[i]) {
				profile->mat[i][j] = val;
			}
		}
	}
	if(DEBUG0) {
		if(profile->isgap[0]) {
			fprintf(stderr, "Error: profile column 0 is a gap in setProfile()\n");
			exit(1);
		}
		if(profile->span > profile->maxspan) {
			fprintf(stderr, "Error: span out of bound in setProfile().\n");
			exit(1);
		}
		for(i = 0; i < profile->span; i++) {
			for(j = 0; j < NUMALPHAS; j++) {
				if(profile->isgap[i] && fabs(profile->mat[i][j] - 1.0) > 0.000001){
					fprintf(stderr, "Error: values on gaps are not set to 1.0 in setProfile()\n");
					exit(1);
				}
			}
		}
	}
}

//----------------------------------------------------------------------
// Column Shift Profile
//----------------------------------------------------------------------
//Shift profile on a set of sequences to the left
void shiftProfileLeft(Profile *profile, double *initval) {
	rmProfileCol(profile, profile->span-1);
	addFrontCol(profile, initval);
}
void shiftProfileRight(Profile *profile, double *initval) {
	rmProfileCol(profile, 0);
	addBackCol(profile, initval);
}

//----------------------------------------------------------------------
// Dimension change
//----------------------------------------------------------------------
void addFrontCol(Profile *profile, double *initval) {
	int i, j;
	if(DEBUG0) {
		if(profile->maxspan == profile->span) { //cannot expand span
			fprintf(stderr, "Error: maxspan == span in addFrontCol");
			exit(1);
		}
	}
	for( i = profile->span; i >= 1; i--) {
		for(j = 0; j < NUMALPHAS; j++) {
			profile->mat[i][j] = profile->mat[i-1][j];
		}
		profile->isgap[i] = profile->isgap[i-1];	
	}

	//set values for the new column
	for(i = 0; i < NUMALPHAS; i++) {
		profile->mat[0][i] = initval[i];
	}
	profile->isgap[0] = 0;

	profile->span++;
	profile->cols++;

}
void addBackCol(Profile *profile, double *initval) {
	int i;
	if(DEBUG0) {
		if(profile->maxspan == profile->span) { //cannot expand span
			fprintf(stderr, "Error: maxspan == span in addFrontCol");
			exit(1);
		}
	}

	profile->isgap[profile->span] = 0;
	for(i = 0; i < NUMALPHAS; i++) {
		profile->mat[profile->span][i] = initval[i];
	}

	profile->span++;
	profile->cols++;

}
void addProfileCol(Profile *profile, int pos, double *initval) {
	int i;
	if(DEBUG0) {
		if(profile->cols == profile->span) { //no gaps to fill
			fprintf(stderr, "Error: no gaps to fill in addProfileCol");
			exit(1);
		}
		if(pos <= 0 || pos >= profile->span - 1) { //addition case of boundaries
			fprintf(stderr, "Error: pos out of range in addProfileCol() %d\n", pos);
			exit(1);
		}
		if(!profile->isgap[pos]) {
			fprintf(stderr, "Error: pos is not a gap in addProfileCol()\n");
			exit(1);
		}
	}
	profile->isgap[pos] = 0;
	profile->cols++;
	for(i = 0; i < NUMALPHAS; i++) {
		profile->mat[pos][i] = initval[i];
	}
}
void rmProfileCol(Profile *profile, int pos) {
	int i,a;
	if(DEBUG0) {
		if(profile->cols <= 1) {
			fprintf(stderr, "Error: profile->cols <= 1 in rmProfilecol");
			exit(1);
		}
		if(pos < 0 || pos >= profile->span) {
			fprintf(stderr, "Error: pos out of range in rmProfileCol() %d\n", pos);
			exit(1);
		}
		if(profile->isgap[pos]) {
			fprintf(stderr, "Error: pos is a gap in rmProfileCol()\n");
			exit(1);
		}
	}
	if(pos == 0) {
		int startpos;
		for(startpos = 1; startpos < profile->span; startpos++) {
			if(!profile->isgap[startpos]) {
				break;
			}
		}
		for(i = startpos; i < profile->span; i++) {
			for(a = 0; a < NUMALPHAS; a++) {
				profile->mat[i - startpos][a] = profile->mat[i][a];
			}
			profile->isgap[i - startpos] = profile->isgap[i];

		}
		profile->span -= startpos;
	}
	else if(pos == profile->span - 1) {
		profile->span--;
		while(profile->isgap[profile->span -1]) {
			profile->span--;
		}
	}
	else { //pos is in the middle
		for(a = 0; a < NUMALPHAS; a++) {
			profile->mat[pos][a] = 1.0;
		}
		profile->isgap[pos] = 1; 
	}
	profile->cols--;

	if(DEBUG0) {
		if(profile->cols > profile->span) {
			fprintf(stderr, "Error: profile->cols > profile->span in rmProfileCol()\n");
			fprintf(stderr, "span %d, cols %d\n", profile->span, profile->cols);
			exit(1);
		}
	}
}
//----------------------------------------------------------------------
// Print
//----------------------------------------------------------------------
void printProfile(FILE *fptr, Profile *profile) {
	int x,y;

	for(y=0; y< NUMALPHAS;  y++)
	{
		fprintf(fptr, "%c   ", numToAlpha(y));
		for(x=0; x< profile->span; x++) {
			if(profile->isgap[x]) {
				fprintf(fptr, "  ** ");
				fprintf(fptr, " ");
			}
			else {
				fprintf(fptr, "%5.2lf",profile->mat[x][y]);
				fprintf(fptr, " ");
			}
		}
		fprintf(fptr, "\n\n");
	}
}

void printCountmat(FILE *fptr, Profile *countmat) {
	int x,y;

	for(y=0; y< NUMALPHAS;  y++)
	{
		fprintf(fptr, "%c   ", numToAlpha(y));
		for(x=0; x< countmat->span; x++) {
			if(countmat->isgap[x]) {
				fprintf(fptr, "  ** ");
				fprintf(fptr, " ");
			}
			else {
				//no decimal points
				fprintf(fptr, "%5.0lf",countmat->mat[x][y]);
				fprintf(fptr, " ");
			}
		}
		fprintf(fptr, "\n\n");
	}
}

