#include "print_results.h"
#include "gibbs_util.h"

static 
void printSiteProfile(double **pr, int span) {
	int x,y;

	for(y=0; y < NUMALPHAS; y++)
	{
		printf("%c   ", numToAlpha((char)y));
		for(x=0;  x < span; x++)
		{
			printf("%5.2lf",pr[x][y]);
			printf(" ");
		}
		printf("\n\n");
  }

}
/* Prints out a given motif profile along with its occurences in the sample  */

static
void printGapPos(FILE *fptr, Profile *profile ) {
	int i;
	for(i = 0; i < profile->span; i++) {
		if(profile->isgap[i]) {
			fprintf(fptr, "Gap position: %d\n", i);
		}
	}
}

static
void printAlignment(FILE *fptr, int *sites, int span, Dataset *data ) {
	int i,j;
	int extra = 5; //extra to view on left and right
	char upToLow = 'a' - 'A'; //offset to convert uppercase to lowercase
	for(i = 0; i < data->numseqs; i++) {
		fprintf(fptr, "%2d   ", i);
		for(j = -extra; j < span + extra; j++) {
			if(j == 0 || j == span) {
				fprintf(fptr, " ");
			}

			int pos = j + sites[i];
			if(pos < 0 || pos >= data->seqLen[i]) {
				fprintf(fptr, " "); //out of sequence range
			}
			else if(sites[i] <= pos && pos < sites[i] + span) {
				fprintf(fptr, "%c", numToAlpha(data->seqs[i][pos])); //upper case
			}
			else {
				//lower case
				fprintf(fptr, "%c", numToAlpha(data->seqs[i][pos]) + upToLow); 
			}
		}
		fprintf(fptr, "\n");
	}
}


void printProfileAndBestOcc(Profile *profile, Dataset *data)
{
	int i, x, y, pos;
	int maxind;
	double maxval, scoreCur;
	double profilescore;
	
	//profile generate by sites
	double **siteprof = (double**)malloc(sizeof(double*)*profile->maxspan); 
	for(x = 0; x< profile->span; x++) {
		siteprof[x] = (double*)malloc(sizeof(double)*data->numalphas);
	}

	for(x=0; x<profile->span; x++) {
		for (y=0; y<data->numalphas; y++) {
			siteprof[x][y] = 0.0;
		}
	}

	/* print profile */
	printf("\n");
	printf("Motif profile:\n\n"); 
	printProfile(stdout, profile);

	/* NOW, PRINT CONSENSUS MOTIF */
	printf("Motif span: %d\n", profile->span);
	printf("Motif columns: %d\n", profile->cols);
	printf("Consensus motif: ");
	for(x=0; x< profile->span; x++) {
		if(!profile->isgap[x]) {
			if(profile->mat[x][0] > 0.5) printf("A");
			else if(profile->mat[x][1] > 0.5) printf("C");
			else if(profile->mat[x][2] > 0.5) printf("G");
			else if(profile->mat[x][3] > 0.5) printf("T");
			else printf("N");
		}
		else {
			printf("_");
		}
	}
	printf("\n\n");

	if(profile->span != profile->cols) {
		printGapPos(stdout, profile);
		printf("\n");
	}

	double **oddsratioMat = (double**) malloc(sizeof(double*) * profile->span);
	for(i = 0; i < profile->span; i++) {
		oddsratioMat[i] = (double*) malloc(sizeof(double) * data->numalphas);
	}
	for(x = 0; x < profile->span; x++) {
		for(y= 0; y< data->numalphas; y++) {
			oddsratioMat[x][y] = profile->mat[x][y] / data->bgfreq[y];
		}
	}

	profilescore = 0.0;
	for(i=0; i< data->numseqs; i++)
	{
		maxind = -1; maxval = -DBL_MAX;
		for(pos=0; pos<data->seqLen[i]- profile->span +1; pos++)
		{
			scoreCur = 1.0;
			for(x=0; x< profile->span; x++) {
				scoreCur *= oddsratioMat[x][ ((int)data->seqs[i][pos+x]) ];
			}
			if(scoreCur > maxval) {
				maxval = scoreCur;
				maxind = pos;
			}
		}
		profilescore += log(maxval);
		for(x=0; x< profile->span; x++) {
			siteprof[x][ ((int)data->seqs[i][maxind+x]) ] += 1.0;  /* UK */
		}

		printf("Sequence %d: motif site is %d\n",i, maxind);
	}
		
	printf("\n");
	printf("Total site-confidence is %.4lf\n", profilescore);
	/* UK : print sites generated profile rather than EM's garbage...*/
	printf("sites generated profile (rather than EM's):\n");
	printf("\n");
	for(x=0; x< profile->span; x++) {
		for (y=0; y< data->numalphas; y++) {
			siteprof[x][y] /= data->numseqs;
		}
	}
	printSiteProfile(siteprof, profile->span);

	for(i = 0; i < profile->span; i++) {
		free(oddsratioMat[i]);
	}
	free(oddsratioMat);

	for(x = 0; x < profile->span; x++) {
		free(siteprof[x]);
	}
	free(siteprof);
}

void printCountmatAndSites(Profile *countmat, int *sites, Dataset *data)
{
	int i, x, y;
	
	Profile *profile = initProfile(countmat->span, countmat->maxspan);
	copyProfile(profile, countmat);
	for(x = 0; x < countmat->span; x++) {
		if( !profile->isgap[x]) {
			for(y = 0; y < data->numalphas; y++) {
				profile->mat[x][y] /= data->numseqs;
			}
		}
	}

	/* print profile */
	printf("\n");
	printf("Motif profile:\n\n"); 
	printProfile(stdout, profile);

	/* NOW, PRINT CONSENSUS MOTIF */
	printf("Motif span: %d\n", profile->span);
	printf("Motif columns: %d\n", profile->cols);
	printf("Consensus motif: ");
	for(x=0; x< profile->span; x++) {
		if(!profile->isgap[x]) {
			if(profile->mat[x][0] > 0.5) 
				printf("A");
			else if(profile->mat[x][1] > 0.5) 
				printf("C");
			else if(profile->mat[x][2] > 0.5) 
				printf("G");
			else if(profile->mat[x][3] > 0.5) 
				printf("T");
			else 
				printf("N");
		}
		else {
			printf("_");
		}
	}
	printf("\n\n");

	if(profile->span != profile->cols) {
		printGapPos(stdout, profile);
		printf("\n");
	}

	for(i=0; i< data->numseqs; i++) {
		printf("Sequence %d: motif site is %d\n",i, sites[i]);
	}
	printf("\n");
	printAlignment(stdout, sites, profile->span, data);
	printf("\n");
}
