(* Brittany Nkounkou *)
(* August 2020 *)
(* Traces *)

Require Export Events.

Set Implicit Arguments.

Module MTraces (env : Environment).
Module Export M := MEvents env.

Module opttrace.

(* option trace *)
CoInductive t : Type :=
| eps : t
| opt : option bag.t -> t -> t.

(* opttrace rewrite helper *)
Lemma match_ ot : ot = match ot with eps => eps | opt o ot' => opt o ot' end.
Proof.
  destruct ot; auto.
Qed.

End opttrace.

(* ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~ *)

Module slang.

(* structural language *)
Definition t : Type :=
  opttrace.t -> Prop.

(* language order: sublanguage *)
Definition le (L1 L2 : t) : Prop :=
  forall ot, L1 ot -> L2 ot.

(* ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~ *)

(* language containing the epsilon opttrace *)
CoInductive eps : t :=
| eps_eps : eps opttrace.eps
| eps_opt ot : eps ot -> eps (opttrace.opt None ot).

(* language containing a singleton-bag opttrace *)
CoInductive bag B : t :=
| bag_None ot : bag B ot -> bag B (opttrace.opt None ot)
| bag_Some ot : eps ot -> bag B (opttrace.opt (Some B) ot).

(* append a language to an opttrace *)
CoInductive app : opttrace.t -> t -> t :=
| app_eps (L : t) ot :
    L ot -> app opttrace.eps L ot
| app_opt o ot L ot' :
    app ot L ot' -> app (opttrace.opt o ot) L (opttrace.opt o ot').

(* language concatenation *)
Definition concat (L1 L2 : t) : t :=
  fun ot => exists ot1, L1 ot1 /\ app ot1 L2 ot.

(* merge two option bags together *)
CoInductive merge_opt : option bag.t -> option bag.t -> option bag.t -> Prop :=
| merge_None_None : merge_opt None None None
| merge_Some_None B : merge_opt (Some B) None (Some B)
| merge_None_Some B : merge_opt None (Some B) (Some B)
| merge_Some_Some B1 B2 :
    merge_opt (Some B1) (Some B2) (Some (bag.union B1 B2)).

(* merge two opttraces together *)
CoInductive merge : opttrace.t -> opttrace.t -> t :=
| merge_eps_eps : merge opttrace.eps opttrace.eps opttrace.eps
| merge_opt_opt o1 o2 o ot1 ot2 ot :
    merge_opt o1 o2 o -> merge ot1 ot2 ot ->
    merge (opttrace.opt o1 ot1) (opttrace.opt o2 ot2) (opttrace.opt o ot).

(* language concurrence *)
Definition concur (L1 L2 : t) : t :=
  fun ot => exists ot1 ot2, L1 ot1 /\ L2 ot2 /\ merge ot1 ot2 ot.

(* language union *)
CoInductive union (L1 L2 : t) : t :=
| union_None ot : union L1 L2 ot -> union L1 L2 (opttrace.opt None ot)
| union_l ot : L1 ot -> union L1 L2 ot
| union_r ot : L2 ot -> union L1 L2 ot.

(* repeatedly append a language to an opttrace *)
CoInductive star : opttrace.t -> t -> t :=
| star_eps_eps (L : t) :
    star opttrace.eps L opttrace.eps
| star_eps_None (L : t) ot :
    star opttrace.eps L ot -> star opttrace.eps L (opttrace.opt None ot)
| star_eps_star (L : t) o ot ot' :
    L (opttrace.opt o ot) -> star (opttrace.opt o ot) L ot' ->
    star opttrace.eps L ot'
| star_opt o ot (L : t) ot' :
    star ot L ot' -> star (opttrace.opt o ot) L (opttrace.opt o ot').

(* ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~ *)

(* sublanguage transitivity *)
Lemma le_trans L L' L'' : le L L' -> le L' L'' -> le L L''.
Proof.
  unfold le. eauto.
Qed.

(* sublanguages produce sublangauge concatenations *)
Lemma concat_le_le L1 L2 L1' L2' :
  le L1 L1' -> le L2 L2' -> le (concat L1 L2) (concat L1' L2').
Proof.
  do 4 intro. destruct H1. destruct H1. exists x. split; auto. clear H1.
  generalize dependent ot. generalize dependent x. generalize dependent L2'.
  generalize dependent L2. cofix F. intros. destruct H2; constructor; eauto.
Qed.

(* sublanguages produce sublangauge concurrences *)
Lemma concur_le_le L1 L2 L1' L2' :
  le L1 L1' -> le L2 L2' -> le (concur L1 L2) (concur L1' L2').
Proof.
  do 4 intro. destruct H1 as [?[?[?[]]]]. exists x, x0. auto.
Qed.

(* sublanguages produce sublangauge unions *)
Lemma union_le_le L1 L2 L1' L2' :
  le L1 L1' -> le L2 L2' -> le (union L1 L2) (union L1' L2').
Proof.
  intros. unfold le. cofix F. intros. destruct H1. constructor; auto.
  constructor 2; auto. constructor 3; auto.
Qed.

(* sublanguages produce sublangauge stars *)
Lemma star_le L L' : le L L' -> forall t, le (star t L) (star t L').
Proof.
  intro. unfold le. cofix F. intros. inversion H0; econstructor; eauto.
Qed.

End slang.

(* ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~ *)

Module trace.

(* trace *)
CoInductive t : Type :=
| eps : t
| bag : bag.t -> t -> t.

(* lift a trace to an opttrace *)
CoFixpoint lift t : opttrace.t :=
  match t with
  | eps => opttrace.eps
  | bag B t' => opttrace.opt (Some B) (lift t')
  end.

(* the epsilon trace lifts to the epsilon opttrace *)
Lemma lift_eps : lift eps = opttrace.eps.
Proof.
  rewrite opttrace.match_ at 1. auto.
Qed.

(* a bag trace lifts to a bag opttrace *)
Lemma lift_bag B t : lift (bag B t) = opttrace.opt (Some B) (lift t).
Proof.
  rewrite opttrace.match_ at 1. auto.
Qed.

(* ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~ *)

(* feasible trace *)
CoInductive Feas sig : t -> Prop :=
| Feas_eps :
    Feas sig eps
| Feas_bag B t :
    bag.Feas sig B -> Feas (bag.update sig B) t -> Feas sig (bag B t).

(* erroneous trace *)
Inductive Error sig : t -> Prop :=
| Error_Error B t :
    bag.Error sig B -> Error sig (bag B t)
| Error_Feas B t :
    bag.Feas sig B -> Error (bag.update sig B) t -> Error sig (bag B t).

End trace.

End MTraces.

(* (c) 2020 Brittany Ro Nkounkou *)
