(* Brittany Nkounkou *)
(* August 2020 *)
(* Petri Nets *)

Require Export Traces.
Require ListSet.

Set Implicit Arguments.

Module MPetris (env : Environment).
Module Export M := MTraces env.

(* the axiom of dependent choice *)
Axiom Dependent_Choice :
  forall X (R : X -> X -> Prop), (forall x, exists y, R x y) -> forall x,
  exists f, f 0 = x /\ forall n, R (f n) (f (S n)).

(* ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~ *)

Module place.

(* place *)
Inductive t : Type :=
| P : t
| L : t -> t
| R : t -> t.

(* decidable place equality *)
Definition dec : eq_dec t.
Proof.
  unfold eq_dec. decide equality.
Defined.

(* the left place of a selection petri net *)
Definition RLL p : place.t :=
  place.R (place.L (place.L p)).

(* the right place of a selection petri net *)
Definition RLR p : place.t :=
  place.R (place.L (place.R p)).

(* the inner place of a repetition petri net *)
Definition RL p : place.t :=
  place.R (place.L p).

End place.

(* ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~ *)

Module config.
Import ListSet.

(* configuration *)
Definition t : Type :=
  set place.t.

(* configuration order: subconfiguration *)
Definition le (c1 c2 : t) : Prop :=
  forall p, In p c1 -> In p c2.

(* configuration equality *)
Definition eq (c1 c2 : t) : Prop :=
  le c1 c2 /\ le c2 c1.

(* configuration union *)
Definition union : t -> t -> t :=
  set_union place.dec.

(* configuration difference *)
Definition diff : t -> t -> t :=
  set_diff place.dec.

(* decidable configuration order *)
Definition le_dec : forall c c' : t, { le c c' } + { ~ le c c' }.
Proof.
  intros.
  destruct (Forall_dec (fun p => In p c') (fun p => In_dec place.dec p c') c).
    left. unfold le, incl. eapply Forall_forall. auto.
    right. intro. apply n. apply Forall_forall. apply H.
Defined.

(* decidable configuration equality *)
Definition dec : forall c c' : t, { eq c c' } + { ~ eq c c' }.
Proof.
  intros. destruct (le_dec c c'). destruct (le_dec c' c). left. split; auto.
    right. intro. destruct H. contradiction.
    right. intro. destruct H. contradiction.
Defined.

(* ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~ *)

(* the left places of a configuration *)
Fixpoint L c : t :=
  match c with
  | [] => []
  | place.L p :: c' => p :: L c'
  | _ :: c' => L c'
  end.

(* the right places of a configuration *)
Fixpoint R c : t :=
  match c with
  | [] => []
  | place.R p :: c' => p :: R c'
  | _ :: c' => R c'
  end.

(* split configuration into left and right places, dropping any Ps *)
Fixpoint L_R c : t * t :=
  match c with
  | [] => ([], [])
  | p :: c' =>
    match L_R c' with (cl, cr) =>
      match p with
      | place.P => (cl, cr)
      | place.L p' => (p' :: cl, cr)
      | place.R p' => (cl, p' :: cr)
      end
    end
  end.

(* the RLL places of a configuration *)
Fixpoint RLL c : t :=
  match c with
  | [] => []
  | place.R (place.L (place.L p)) :: c' => p :: RLL c'
  | _ :: c' => RLL c'
  end.

(* the RLR places of a configuration *)
Fixpoint RLR c : t :=
  match c with
  | [] => []
  | place.R (place.L (place.R p)) :: c' => p :: RLR c'
  | _ :: c' => RLR c'
  end.

(* the RL places of a configuration *)
Fixpoint RL c : t :=
  match c with
  | [] => []
  | place.R (place.L p) :: c' => p :: RL c'
  | _ :: c' => RL c'
  end.

(* ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~ *)

(* subconfiguration reflexivity *)
Lemma le_refl c : le c c.
Proof.
  unfold le. auto.
Qed.

(* subconfiguration transitivity *)
Lemma le_trans c c' c'' : le c c' -> le c' c'' -> le c c''.
Proof.
  unfold le. auto.
Qed.

(* equality reflexivity *)
Lemma eq_refl c : eq c c.
Proof.
  unfold eq. split; apply le_refl.
Qed.

(* equality symmetry *)
Lemma eq_sym c c' : eq c c' -> eq c' c.
Proof.
  unfold eq. intuition.
Qed.

(* equality transitivity *)
Lemma eq_trans c c' c'' : eq c c' -> eq c' c'' -> eq c c''.
Proof.
  unfold eq. intuition; eapply le_trans; eauto.
Qed.

(* ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~ *)

(* left place inclusion *)
Lemma In_L p c : In p (L c) <-> In (place.L p) c.
Proof.
  induction c. split; intro; destruct H. simpl.
  destruct a; simpl; intuition; inversion H2; auto.
Qed.

(* right place inclusion *)
Lemma In_R p c : In p (R c) <-> In (place.R p) c.
Proof.
  induction c. split; intro; destruct H. simpl.
  destruct a; simpl; intuition; inversion H2; auto.
Qed.

(* RLL place inclusion *)
Lemma In_RLL p c : In p (RLL c) <-> In (place.R (place.L (place.L p))) c.
Proof.
  induction c. split; intro; destruct H. simpl.
  repeat (destruct a; simpl; try (intuition; inversion H2)).
Qed.

(* RLR place inclusion *)
Lemma In_RLR p c : In p (RLR c) <-> In (place.R (place.L (place.R p))) c.
Proof.
  induction c. split; intro; destruct H. simpl.
  repeat (destruct a; simpl; try (intuition; inversion H2)).
Qed.

(* RL place inclusion *)
Lemma In_RL p c : In p (RL c) <-> In (place.R (place.L p)) c.
Proof.
  induction c. split; intro; destruct H. simpl.
  repeat (destruct a; simpl; try (intuition; inversion H2)).
Qed.

(* ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~ *)

(* if f is injective, subconfigurations produce subconfiguration maps *)
Lemma le_map f c c' :
  (forall p p', f p = f p' -> p = p') -> le (map f c) (map f c') <-> le c c'.
Proof.
  unfold le. split; intros.
    specialize (H0 (f p)). repeat rewrite in_map_iff in H0.
      edestruct H0 as [?[]]; eauto. rewrite <- (H _ _ H2). auto.
    apply in_map_iff. rewrite in_map_iff in H1. destruct H1 as [?[]]. eauto.
Qed.

Lemma le_app_l c1 c2 c : le (map place.L c1 ++ map place.R c2) c <->
  le c1 (config.L c) /\ le c2 (config.R c).
Proof.
  unfold le. intuition.
    apply In_L. apply H. apply in_or_app. left. apply in_map. auto.
    apply In_R. apply H. apply in_or_app. right. apply in_map. auto.
    edestruct in_app_or; eauto; rewrite in_map_iff in H2; destruct H2 as [?[]].
      inversion H2. apply In_L; auto.
      inversion H2. apply In_R; auto.
Qed.

Lemma le_app_r c c1 c2 : le c (map place.L c1 ++ map place.R c2) <->
  ~ In place.P c /\ le (config.L c) c1 /\ le (config.R c) c2.
Proof.
  unfold le. intuition.
    edestruct in_app_or; eauto; clear - H1.
      induction c1; auto. inversion H1; auto. inversion H.
      induction c2; auto. inversion H1; auto. inversion H.
    rewrite In_L in H0. edestruct in_app_or; eauto; rewrite in_map_iff in H1.
      destruct H1 as [?[]]. inversion H1. rewrite <- H4. auto.
      destruct H1 as [?[]]. inversion H1.
    rewrite In_R in H0. edestruct in_app_or; eauto; rewrite in_map_iff in H1.
      destruct H1 as [?[]]. inversion H1.
      destruct H1 as [?[]]. inversion H1. rewrite <- H4. auto.
    apply in_app_iff. destruct p. destruct H0; auto.
      left. apply in_map, H, In_L, H1.
      right. apply in_map, H2, In_R, H1.
Qed.

Lemma le_app c1 c1' c2 c2' :
  le (map place.L c1 ++ map place.R c2) (map place.L c1' ++ map place.R c2') <->
  le c1 c1' /\ le c2 c2'.
Proof.
  unfold le. intuition.
    specialize (H (place.L p)). repeat rewrite in_app_iff in H.
      repeat rewrite in_map_iff in H. destruct H as [[?[]]|[?[]]]; eauto;
      inversion H. rewrite <- H3. auto.
    specialize (H (place.R p)). repeat rewrite in_app_iff in H.
      repeat rewrite in_map_iff in H. destruct H as [[?[]]|[?[]]]; eauto;
      inversion H. rewrite <- H3. auto.
    rewrite in_app_iff in *. repeat rewrite in_map_iff in *.
      destruct H as [[?[]]|[?[]]]; eauto.
Qed.

(* ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~ *)

(* the left places of subconfigurations are subconfigurations *)
Lemma le_L c c' : le c c' -> le (L c) (L c').
Proof.
  unfold le. intros. apply In_L, H, In_L, H0.
Qed.

(* the right places of subconfigurations are subconfigurations *)
Lemma le_R c c' : le c c' -> le (R c) (R c').
Proof.
  unfold le. intros. apply In_R, H, In_R, H0.
Qed.

(* the RLL places of subconfigurations are subconfigurations *)
Lemma le_RLL c c' : le c c' -> le (RLL c) (RLL c').
Proof.
  unfold le. intros. apply In_RLL, H, In_RLL, H0.
Qed.

(* the RLR places of subconfigurations are subconfigurations *)
Lemma le_RLR c c' : le c c' -> le (RLR c) (RLR c').
Proof.
  unfold le. intros. apply In_RLR, H, In_RLR, H0.
Qed.

(* the RL places of subconfigurations are subconfigurations *)
Lemma le_RL c c' : le c c' -> le (RL c) (RL c').
Proof.
  unfold le. intros. apply In_RL, H, In_RL, H0.
Qed.

(* ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~ *)

(* if f is injective, equal configurations produce equal maps *)
Lemma eq_map f c c' :
  (forall p p', f p = f p' -> p = p') -> eq (map f c) (map f c') <-> eq c c'.
Proof.
  unfold eq. split; intros; destruct H0. rewrite le_map in H0, H1; auto.
    split; try apply le_map; auto.
Qed.

Lemma eq_map_L c c' : eq (map place.L c) c' -> c' = map place.L (config.L c').
Proof.
  intro. destruct H. clear H. induction c'. auto. destruct a; simpl.
    absurd (In place.P (map place.L c)).
      intro. clear - H. induction c; auto. inversion H; auto. inversion H0.
      apply H0. apply in_eq.
    rewrite IHc' at 1. auto. do 2 intro. apply H0. right. auto.
    absurd (In (place.R a) (map place.L c)).
      intro. clear - H. induction c; auto. inversion H; auto. inversion H0.
      apply H0. apply in_eq.
Qed.

Lemma eq_map_R c c' : eq (map place.R c) c' -> c' = map place.R (config.R c').
Proof.
  intro. destruct H. clear H. induction c'. auto. destruct a; simpl.
    absurd (In place.P (map place.R c)).
      intro. clear - H. induction c; auto. inversion H; auto. inversion H0.
      apply H0. apply in_eq.
    absurd (In (place.L a) (map place.R c)).
      intro. clear - H. induction c; auto. inversion H; auto. inversion H0.
      apply H0. apply in_eq.
    rewrite IHc' at 1. auto. do 2 intro. apply H0. right. auto.
Qed.

Lemma eq_map_RLL c c' :
  eq (map place.RLL c) c' -> c' = map place.RLL (config.RLL c').
Proof.
  intro. destruct H. clear H. induction c'. auto. destruct a; simpl.
    absurd (In place.P (map place.RLL c)).
      intro. clear - H. induction c; auto. inversion H; auto. inversion H0.
      apply H0. apply in_eq.
    absurd (In (place.L a) (map place.RLL c)).
      intro. clear - H. induction c; auto. inversion H; auto. inversion H0.
      apply H0. apply in_eq.
    destruct a.
      absurd (In (place.R place.P) (map place.RLL c)).
        intro. clear - H. induction c; auto. inversion H; auto. inversion H0.
        apply H0. apply in_eq.
      destruct a.
        absurd (In (place.R (place.L place.P)) (map place.RLL c)).
          intro. clear - H. induction c; auto. inversion H; auto. inversion H0.
          apply H0. apply in_eq.
        rewrite IHc' at 1. auto. do 2 intro. apply H0. right. auto.
        absurd (In (place.R (place.L (place.R a))) (map place.RLL c)).
          intro. clear - H. induction c; auto. inversion H; auto. inversion H0.
          apply H0. apply in_eq.
      absurd (In (place.R (place.R a)) (map place.RLL c)).
        intro. clear - H. induction c; auto. inversion H; auto. inversion H0.
        apply H0. apply in_eq.
Qed.

Lemma eq_map_RLR c c' :
  eq (map place.RLR c) c' -> c' = map place.RLR (config.RLR c').
Proof.
  intro. destruct H. clear H. induction c'. auto. destruct a; simpl.
    absurd (In place.P (map place.RLR c)).
      intro. clear - H. induction c; auto. inversion H; auto. inversion H0.
      apply H0. apply in_eq.
    absurd (In (place.L a) (map place.RLR c)).
      intro. clear - H. induction c; auto. inversion H; auto. inversion H0.
      apply H0. apply in_eq.
    destruct a.
      absurd (In (place.R place.P) (map place.RLR c)).
        intro. clear - H. induction c; auto. inversion H; auto. inversion H0.
        apply H0. apply in_eq.
      destruct a.
        absurd (In (place.R (place.L place.P)) (map place.RLR c)).
          intro. clear - H. induction c; auto. inversion H; auto. inversion H0.
          apply H0. apply in_eq.
        absurd (In (place.R (place.L (place.L a))) (map place.RLR c)).
          intro. clear - H. induction c; auto. inversion H; auto. inversion H0.
          apply H0. apply in_eq.
        rewrite IHc' at 1. auto. do 2 intro. apply H0. right. auto.
      absurd (In (place.R (place.R a)) (map place.RLR c)).
        intro. clear - H. induction c; auto. inversion H; auto. inversion H0.
        apply H0. apply in_eq.
Qed.

Lemma eq_map_RL c c' :
  eq (map place.RL c) c' -> c' = map place.RL (config.RL c').
Proof.
  intro. destruct H. clear H. induction c'. auto. destruct a; simpl.
    absurd (In place.P (map place.RL c)).
      intro. clear - H. induction c; auto. inversion H; auto. inversion H0.
      apply H0. apply in_eq.
    absurd (In (place.L a) (map place.RL c)).
      intro. clear - H. induction c; auto. inversion H; auto. inversion H0.
      apply H0. apply in_eq.
    destruct a.
      absurd (In (place.R place.P) (map place.RL c)).
        intro. clear - H. induction c; auto. inversion H; auto. inversion H0.
        apply H0. apply in_eq.
      rewrite IHc' at 1. auto. do 2 intro. apply H0. right. auto.
      absurd (In (place.R (place.R a)) (map place.RL c)).
        intro. clear - H. induction c; auto. inversion H; auto. inversion H0.
        apply H0. apply in_eq.
Qed.

Lemma eq_app_r c c1 c2 : eq c (map place.L c1 ++ map place.R c2) <->
  ~ In place.P c /\ eq (config.L c) c1 /\ eq (config.R c) c2.
Proof.
  unfold eq. rewrite le_app_l, le_app_r. intuition.
Qed.

Lemma eq_app c1 c1' c2 c2' :
  eq (map place.L c1 ++ map place.R c2) (map place.L c1' ++ map place.R c2') <->
  eq c1 c1' /\ eq c2 c2'.
Proof.
  unfold eq. intuition; try solve [eapply le_app; eauto].
    rewrite le_app in H0. destruct H0. auto.
    rewrite le_app in H1. destruct H1. auto.
Qed.

(* ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~ *)

(* the left places of equal configurations are equal *)
Lemma eq_L c c' : eq c c' -> eq (L c) (L c').
Proof.
  unfold eq. intro; destruct H; split; apply le_L; auto.
Qed.

(* the right places of equal configurations are equal *)
Lemma eq_R c c' : eq c c' -> eq (R c) (R c').
Proof.
  unfold eq. intro; destruct H; split; apply le_R; auto.
Qed.

(* the RLL places of equal configurations are equal *)
Lemma eq_RLL c c' : eq c c' -> eq (RLL c) (RLL c').
Proof.
  unfold eq. intro; destruct H; split; apply le_RLL; auto.
Qed.

(* the RLR places of equal configurations are equal *)
Lemma eq_RLR c c' : eq c c' -> eq (RLR c) (RLR c').
Proof.
  unfold eq. intro; destruct H; split; apply le_RLR; auto.
Qed.

(* the RL places of equal configurations are equal *)
Lemma eq_RL c c' : eq c c' -> eq (RL c) (RL c').
Proof.
  unfold eq. intro; destruct H; split; apply le_RL; auto.
Qed.

(* ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~ *)

(* subconfigurations produce subconfiguration set_adds *)
Lemma set_add_le p c c' :
  le c c' -> le (set_add place.dec p c) (set_add place.dec p c').
Proof.
  intro. induction c; simpl; do 2 intro.
    apply set_add_intro2. destruct H0; inversion H0. auto.
    apply set_add_intro. destruct (place.dec p a). right. apply H, H0.
      destruct H0. right. apply H. rewrite H0. apply in_eq.
      eapply set_add_elim. apply IHc; auto. do 2 intro. apply H. right. auto.
Qed.

(* equal configurations produce equal set_adds *)
Lemma set_add_eq p c c' :
  eq c c' -> eq (set_add place.dec p c) (set_add place.dec p c').
Proof.
  intro. destruct H. split; apply set_add_le; auto.
Qed.

(* if f is injective, set_add/map order does not matter *)
Lemma set_add_map f p c :
  (forall p p', f p = f p' -> p = p') ->
  set_add place.dec (f p) (map f c) = map f (set_add place.dec p c).
Proof.
  intro. induction c. auto. simpl.
  destruct (place.dec (f p) (f a)), (place.dec p a).
    auto. destruct n. auto. rewrite e in n. contradiction. rewrite IHc. auto.
Qed.

Lemma set_add_app_l p c1 c2 :
  eq (set_add place.dec p (c1 ++ c2)) (set_add place.dec p c1 ++ c2).
Proof.
  induction c1; simpl.
    split; intro; rewrite set_add_iff; intuition. rewrite H0. apply in_eq.
      destruct H; auto.
    destruct (place.dec p a); simpl. apply eq_refl. split; do 2 intro.
      destruct H. left. auto. right. apply IHc1. auto.
      destruct H. left. auto. right. apply IHc1. auto.
Qed.

Lemma set_add_app_r p c1 c2 :
  eq (set_add place.dec p (c1 ++ c2)) (c1 ++ set_add place.dec p c2).
Proof.
  induction c1; simpl. apply eq_refl. destruct (place.dec p a); simpl.
    split; do 2 intro.
      destruct H. left. auto. right. apply IHc1. apply set_add_intro1. auto.
      destruct H. left. auto. edestruct set_add_elim. apply IHc1. eauto.
        left. transitivity p; auto. right. auto.
    split; do 2 intro.
      destruct H. left. auto. right. apply IHc1. auto.
      destruct H. left. auto. right. apply IHc1. auto.
Qed.

(* ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~ *)

(* a place is in configuration c union c' iff it is in c or c' *)
Lemma In_union p c c' : In p (union c c') <-> In p c \/ In p c'.
Proof.
  apply set_union_iff.
Qed.

(* subconfigurations produce subconfiguration unions *)
Lemma union_le_l c c' c'' : le c c' -> le (union c c'') (union c' c'').
Proof.
  do 3 intro. apply In_union. rewrite In_union in H0. destruct H0; auto.
Qed.

(* equal configurations produce equal unions *)
Lemma union_eq_l c c' c'' : eq c c' -> eq (union c c'') (union c' c'').
Proof.
  intro. split; apply union_le_l; apply H.
Qed.

(* if f is injective, union/map order does not matter *)
Lemma union_map f c c' :
  (forall p p', f p = f p' -> p = p') ->
  union (map f c) (map f c') = map f (union c c').
Proof.
  intro. induction c'. auto. simpl. rewrite IHc'. apply set_add_map. auto.
Qed.

Lemma union_app c1 c2 c1' c2' :
  eq
    (union (map place.L c1 ++ map place.R c2)
      (map place.L c1' ++ map place.R c2'))
    (map place.L (union c1 c1') ++ map place.R (union c2 c2')).
Proof.
  induction c1'; simpl. induction c2'; simpl. apply config.eq_refl.
    eapply eq_trans. apply set_add_eq. eauto. eapply eq_trans.
      apply set_add_app_r. rewrite <- set_add_map. apply eq_refl.
      intros. inversion H. auto.
    eapply eq_trans. apply set_add_eq. eauto. eapply eq_trans.
      apply set_add_app_l. rewrite <- set_add_map. apply eq_refl.
      intros. inversion H. auto.
Qed.

Lemma union_L c c' : union (L c) (L c') = L (union c c').
Proof.
  induction c'. auto. simpl. destruct a; simpl; rewrite IHc'; clear IHc'.
    induction (union c c'); auto. simpl. rewrite IHt0. destruct a; auto.
    induction (union c c'); auto. simpl. destruct a0; try rewrite IHt0; auto.
      simpl. destruct place.dec; simpl; auto. rewrite IHt0. auto.
    induction (union c c'); auto. simpl. destruct a0; try rewrite IHt0; auto.
      destruct place.dec; simpl; auto.
Qed.

Lemma union_R c c' : union (R c) (R c') = R (union c c').
Proof.
  induction c'. auto. simpl. destruct a; simpl; rewrite IHc'; clear IHc'.
    induction (union c c'); auto. simpl. rewrite IHt0. destruct a; auto.
    induction (union c c'); auto. simpl. destruct a0; try rewrite IHt0; auto.
      destruct place.dec; simpl; auto.
    induction (union c c'); auto. simpl. destruct a0; try rewrite IHt0; auto.
      simpl. destruct place.dec; simpl; auto. rewrite IHt0. auto.
Qed.

(* ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~ *)

(* a place is in configuration c diff c' iff it is in c and not in c' *)
Lemma In_diff p c c' : In p (diff c c') <-> In p c /\ ~ In p c'.
Proof.
  apply set_diff_iff.
Qed.

(* if configuration c = c', then c diff c' is empty *)
Lemma diff_eq c c' : eq c c' -> diff c c' = [].
Proof.
  intro. case_eq (diff c c'). auto. unfold diff. intros. absurd (In t0 c').
    eapply set_diff_elim2. rewrite H0. apply in_eq.
    apply H. eapply set_diff_elim1. rewrite H0. apply in_eq.
Qed.

(* subconfigurations produce subconfiguration diffs *)
Lemma diff_le_l c c' c'' : le c c' -> le (diff c c'') (diff c' c'').
Proof.
  do 3 intro. apply set_diff_intro. apply H. eapply set_diff_elim1; eauto.
  eapply set_diff_elim2; eauto.
Qed.

(* equal configurations produce equal diffs *)
Lemma diff_eq_l c c' c'' : eq c c' -> eq (diff c c'') (diff c' c'').
Proof.
  intro. split; apply diff_le_l; apply H.
Qed.

(* if f is injective, diff/map order does not matter *)
Lemma diff_map f c c' :
  (forall p p', f p = f p' -> p = p') ->
  diff (map f c) (map f c') = map f (diff c c').
Proof.
  intro. induction c. auto. simpl. case_eq (set_mem place.dec a c'); intro.
    rewrite set_mem_correct2. auto.
    apply in_map. eapply set_mem_correct1. eauto.
    rewrite set_mem_complete2. rewrite IHc. apply set_add_map. auto.
      intro. absurd (In a c'). eapply set_mem_complete1. eauto.
        rewrite in_map_iff in H1. destruct H1 as [?[]]. rewrite <- (H _ _ H1).
        auto.
Qed.

Lemma diff_app c1 c2 c1' c2' :
  eq
  (diff (map place.L c1 ++ map place.R c2) (map place.L c1' ++ map place.R c2'))
  (map place.L (diff c1 c1') ++ map place.R (diff c2 c2')).
Proof.
  induction c1; simpl. induction c2; simpl. apply config.eq_refl.
    case_eq (set_mem place.dec a c2'); intro.
      erewrite set_mem_correct2. auto. apply in_or_app. right. apply in_map.
        eapply set_mem_correct1. eauto.
      erewrite set_mem_complete2. rewrite <- set_add_map. apply set_add_eq.
        auto. intros. inversion H0. auto. intro. eapply set_mem_complete1.
        eauto. edestruct in_app_or; eauto. exfalso. clear - H1.
        induction c1'; auto. inversion H1; auto. inversion H.
        rewrite in_map_iff in H1. destruct H1 as [?[]]. inversion H1.
        rewrite <- H4. auto.
    case_eq (set_mem place.dec a c1'); intro.
      erewrite set_mem_correct2. auto. apply in_or_app. left. apply in_map.
        eapply set_mem_correct1. eauto.
      erewrite set_mem_complete2. eapply eq_trans. apply set_add_eq. eauto.
        rewrite <- set_add_map. apply set_add_app_l. intros. inversion H0. auto.
        intro. eapply set_mem_complete1. eauto. edestruct in_app_or; eauto.
        rewrite in_map_iff in H1. destruct H1 as [?[]]. inversion H1.
        rewrite <- H4. auto. exfalso. clear - H1. induction c2'; auto.
        inversion H1; auto. inversion H.
Qed.

Lemma diff_L c c' : diff (L c) (L c') = L (diff c c').
Proof.
  induction c. auto. simpl. destruct a; simpl; rewrite IHc; clear IHc.
    destruct set_mem; auto. induction (diff c c'); auto. simpl. rewrite IHt0.
      destruct a; auto.
    case_eq (set_mem place.dec a (L c')); intro.
      rewrite set_mem_correct2. auto. eapply In_L, set_mem_correct1. eauto.
      rewrite set_mem_complete2.
        induction (diff c c'); auto. simpl. destruct a0; try rewrite IHt0; auto.
          simpl. destruct place.dec; simpl; auto. rewrite IHt0. auto.
        intro. eapply set_mem_complete1. eauto. apply In_L. auto.
    destruct set_mem; auto. induction (diff c c'); auto. simpl.
      destruct a0; try rewrite IHt0; auto. destruct place.dec; simpl; auto.
Qed.

Lemma diff_R c c' : diff (R c) (R c') = R (diff c c').
Proof.
  induction c. auto. simpl. destruct a; simpl; rewrite IHc; clear IHc.
    destruct set_mem; auto. induction (diff c c'); auto. simpl. rewrite IHt0.
      destruct a; auto.
    destruct set_mem; auto. induction (diff c c'); auto. simpl.
      destruct a0; try rewrite IHt0; auto. destruct place.dec; simpl; auto.
    case_eq (set_mem place.dec a (R c')); intro.
      rewrite set_mem_correct2. auto. eapply In_R, set_mem_correct1. eauto.
      rewrite set_mem_complete2.
        induction (diff c c'); auto. simpl. destruct a0; try rewrite IHt0; auto.
          simpl. destruct place.dec; simpl; auto. rewrite IHt0. auto.
        intro. eapply set_mem_complete1. eauto. apply In_R. auto.
Qed.

(* ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~ *)

Lemma union_nil c : eq (union [] c) c.
Proof.
  split; do 2 intro. eapply set_union_emptyL. apply H. apply In_union. auto.
Qed.

Lemma diff_nil c : eq (diff c []) c.
Proof.
  split; do 2 intro. eapply set_diff_elim1; eauto. apply set_diff_intro. auto.
  apply in_nil.
Qed.

Lemma L_R_L_R c : L_R c = (L c, R c).
Proof.
  induction c. auto. simpl. rewrite IHc. destruct a; auto.
Qed.

Lemma L_app c1 c2 : config.L (map place.L c1 ++ map place.R c2) = c1.
Proof.
  induction c1; simpl. induction c2; auto. rewrite IHc1. auto.
Qed.

Lemma R_app c1 c2 : config.R (map place.L c1 ++ map place.R c2) = c2.
Proof.
  induction c1; auto. simpl. induction c2; auto. simpl. rewrite IHc2. auto.
Qed.

(* ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~ *)

Lemma L_map_L c : config.L (map place.L c) = c.
Proof.
  induction c. auto. simpl. rewrite IHc. auto.
Qed.

Lemma R_map_R c : config.R (map place.R c) = c.
Proof.
  induction c. auto. simpl. rewrite IHc. auto.
Qed.

Lemma RLL_map_RLL c : config.RLL (map place.RLL c) = c.
Proof.
  induction c. auto. simpl. rewrite IHc. auto.
Qed.

Lemma RLR_map_RLR c : config.RLR (map place.RLR c) = c.
Proof.
  induction c. auto. simpl. rewrite IHc. auto.
Qed.

Lemma RL_map_RL c : config.RL (map place.RL c) = c.
Proof.
  induction c. auto. simpl. rewrite IHc. auto.
Qed.

Lemma L_map_R c : config.L (map place.R c) = [].
Proof.
  induction c. auto. simpl. rewrite IHc. auto.
Qed.

Lemma R_map_L c : config.R (map place.L c) = [].
Proof.
  induction c. auto. simpl. rewrite IHc. auto.
Qed.

End config.

(* ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~ *)

Module trans.

(* transition *)
Definition t : Type :=
  config.t * option bag.t * config.t.

(* transition map *)
Definition map f (t' : t) : t :=
  match t' with
  | (ci, w, co) => (map f ci, w, map f co)
  end.

(* transition product *)
Definition prod tt : list t :=
  match tt with
  | ((ci, Some B, co), (ci', Some B', co')) =>
      [(ci ++ ci', Some (bag.union B B'), co ++ co')]
  | _ => []
  end.

(* transition map rewrite helper *)
Lemma map_eq f ci w co : (List.map f ci, w, List.map f co) = map f (ci, w, co).
Proof.
  auto.
Qed.

End trans.

(* ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~ *)

Module path.

(* path *)
CoInductive t : Type :=
| fin : list config.t -> t
| hop : list config.t -> config.t -> t -> t.

(* path rewrite helper *)
Lemma match_ pi :
  pi = match pi with fin l => fin l | hop l c pi' => hop l c pi' end.
Proof.
  destruct pi; auto.
Qed.

(* prepend a path with a list of event bags *)
Definition prep l pi : t :=
  match pi with
  | fin l' => fin (l ++ l')
  | hop l' c pi' => hop (l ++ l') c pi'
  end.

(* path map *)
CoFixpoint map f pi : t :=
  match pi with
  | fin l => fin (List.map f l)
  | hop l c pi' => hop (List.map f l) (f c) (map f pi')
  end.

(* rewrite helper for map applied to fin *)
Lemma map_fin f l :
  map f (fin l) = fin (List.map f l).
Proof.
  rewrite match_ at 1. auto.
Qed.

(* rewrite helper for map applied to hop *)
Lemma map_hop f l c pi :
  map f (hop l c pi) = hop (List.map f l) (f c) (map f pi).
Proof.
  rewrite match_ at 1. auto.
Qed.

(* part of list l that appears before first configuration c (if any) *)
Fixpoint before c l : list config.t :=
  match l with
  | [] => []
  | c' :: l' => if config.dec c c' then [] else c' :: before c l'
  end.

(* part of list l that appears after first configuration c (if any) *)
Fixpoint after c l : list config.t :=
  match l with
  | [] => []
  | c' :: l' => if config.dec c c' then l' else after c l'
  end.

End path.

(* ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~ *)

Module petri.

(* petri net *)
Record t : Type :=
  make {
    init : config.t;
    fin : config.t;
    T : list trans.t;
  }.

(* well-formed petri net *)
Record Wf N : Prop :=
  mkWf {
    Wf_init : init N <> [];
    Wf_fin : fin N <> [];
    Wf_T : forall ci B co, In (ci, B, co) (T N) -> ci <> [] /\ co <> [];
  }.

(* ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~ *)

(* firing *)
Inductive Firing N c w c' : Prop :=
| Firing_In ci co :
    In (ci, w, co) (T N) -> config.le ci c ->
    config.eq (config.union (config.diff c ci) co) c' ->
    Firing N c w c'.

(* empty firing chain *)
Inductive Efchain N c : list config.t -> config.t -> Prop :=
| Efchain_eq c' :
    config.eq c c' -> Efchain N c [] c'
| Efchain_Firing c' l c'' :
    Firing N c None c' -> Efchain N c' l c'' -> Efchain N c (c' :: l) c''.

(* trace generation *)
CoInductive Gen N c : path.t -> slang.t :=
| Gen_eps l :
    Efchain N c l (fin N) -> Gen N c (path.fin l) opttrace.eps
| Gen_None l c' pi ot :
    Efchain N c l c' -> Gen N c' pi ot ->
    Gen N c (path.hop l c' pi) (opttrace.opt None ot)
| Gen_Some l c' B c'' pi ot :
    Efchain N c l c' -> Firing N c' (Some B) c'' -> Gen N c'' pi ot ->
    Gen N c (path.hop l c'' pi) (opttrace.opt (Some B) ot).

(* petri net language *)
Definition Lang N : slang.t :=
  fun ot => exists pi, Gen N (init N) pi ot.

(* trace generation without explicit paths *)
CoInductive Gen' N c : slang.t :=
| Gen'_eps l :
    Efchain N c l (fin N) -> Gen' N c opttrace.eps
| Gen'_None l c' ot :
    Efchain N c l c' -> Gen' N c' ot -> Gen' N c (opttrace.opt None ot)
| Gen'_Some l c' B c'' ot :
    Efchain N c l c' -> Firing N c' (Some B) c'' -> Gen' N c'' ot ->
    Gen' N c (opttrace.opt (Some B) ot).

Lemma Gen'_Gen N : forall c pi ot, Gen N c pi ot -> Gen' N c ot.
Proof.
  cofix F. intros. destruct H; econstructor; eauto.
Qed.

(* ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~ *)

(* reachable configuration *)
Inductive Reach N : config.t -> Prop :=
| Reach_eq_init c : config.eq c (init N) -> Reach N c
| Reach_Firing c w c' : Reach N c -> Firing N c w c' -> Reach N c'.

(* the initial configuration is reachable *)
Lemma Reach_init N : Reach N (init N).
Proof.
  constructor. apply config.eq_refl.
Qed.

(* safe petri net *)
Definition Safe N : Prop :=
  forall c, Reach N c -> config.le (fin N) c -> config.le c (fin N).

(* empty firing chain to final configuration *)
Definition efc_fin : Type :=
  config.t -> bool.

(* nth bag operation *)
Definition nth_bag : Type :=
  config.t -> state.t -> bool ->
  (bag.t -> config.t -> nat -> option (bag.t * config.t * option nat)) ->
  nat -> option (bag.t * config.t * option nat).

(* efc_fin correctness *)
Record EfcFin N (ef : efc_fin) : Type :=
  mkEF {
    EF_sound :
      forall c, Reach N c -> ef c = true -> {l | Efchain N c l (fin N)};
    EF_complete :
      forall c l, Reach N c -> Efchain N c l (fin N) -> ef c = true;
  }.

(* nth_bag soundness *)
Definition nb_sound N (nb : nth_bag) : Type :=
  forall c sig sync F t n, Reach N c -> nb c sig sync F n = Some t ->
  {l & {c' & {B & {c'' & {n' | Efchain N c l c' /\ Firing N c' (Some B) c'' /\
  bag.Sat sig B /\ (sync = true -> bag.Sync B) /\ F B c'' n' = Some t}}}}}.

Definition f_None
  (f : nat -> option (bag.t * config.t * option nat)) : Prop :=
  forall n n', f n = None -> f n' = None.

Definition f_Some
  (f : nat -> option (bag.t * config.t * option nat)) : Type :=
  forall n t n'', f n = Some t -> {n' | exists p, f n' = Some (p, Some n'')}.

(* nth_bag completeness *)
Definition nb_complete N (nb : nth_bag) : Type :=
  forall c l c' B c'' sig sync F, Reach N c -> Efchain N c l c' ->
  Firing N c' (Some B) c'' -> bag.Sat sig B -> (sync = true -> bag.Sync B) ->
  (forall B c, f_None (F B c)) -> (forall B c, f_Some (F B c)) ->
  {c' : _ & Reach N c' & {l' : _ & Efchain N c' l' c'' & forall n', {n |
  (exists p, F B c' n' = Some (p, None)) -> nb c sig sync F n = F B c' n'}}}.

(* nth_bag correctness *)
Record NthBag N (nb : nth_bag) : Type :=
  mkNB {
    NB_sound : nb_sound N nb;
    NB_None_None : forall c sig sync F, (forall B c, f_None (F B c)) ->
      f_None (nb c sig sync F);
    NB_Some_Some : forall c sig sync F, (forall B c, f_None (F B c)) ->
      (forall B c, f_Some (F B c)) -> f_Some (nb c sig sync F);
    NB_complete : nb_complete N nb;
  }.

(* ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~ *)

Lemma Firing_eq_eq N ci w co c c' :
  In (ci, w, co) (T N) -> config.eq c ci -> config.eq c' co -> Firing N c w c'.
Proof.
  intros. econstructor; eauto. apply H0. rewrite config.diff_eq; auto.
  eapply config.eq_trans. apply config.union_nil. apply config.eq_sym. auto.
Qed.

Lemma Firing_eq_l N c w c' :
  (exists c'', Firing N c'' w c' /\ config.eq c c'') -> Firing N c w c'.
Proof.
  intro. destruct H as [?[]]. destruct H. econstructor; eauto.
    eapply config.le_trans; eauto. apply H0.
    eapply config.eq_trans; try apply H2. apply config.union_eq_l.
      apply config.diff_eq_l. auto.
Qed.

Lemma Firing_eq_r N c w c' c'' :
  Firing N c w c' -> config.eq c' c'' -> Firing N c w c''.
Proof.
  intros. destruct H. econstructor; eauto. eapply config.eq_trans; eauto.
Qed.

Lemma Efchain_eq_eq N ci co c c' :
  In (ci, None, co) (T N) -> config.eq c ci -> config.eq c' co ->
  Efchain N c [c'] c'.
Proof.
  intros. constructor. eapply Firing_eq_eq; eauto.
  constructor. apply config.eq_refl.
Qed.

Lemma Efchain_eq_l N c c' l c'' :
  Efchain N c l c'' -> config.eq c c' -> Efchain N c' l c''.
Proof.
  intros. destruct H.
    constructor. eapply config.eq_trans; eauto. apply config.eq_sym. auto.
    econstructor; eauto. eapply Firing_eq_l. exists c. split; auto.
      apply config.eq_sym. auto.
Qed.

Lemma Efchain_eq_r N c l c' :
  (exists c'', Efchain N c l c'' /\ config.eq c'' c') -> Efchain N c l c'.
Proof.
  intro. destruct H as [?[]]. induction H. constructor.
  eapply config.eq_trans; eauto. econstructor; eauto.
Qed.

Lemma Efchain_last N c l c' : Efchain N c l c' -> config.eq c' (last l c).
Proof.
  intro. induction H. apply config.eq_sym. auto.
  clear H0. induction l; auto. destruct l; auto.
Qed.

(* Efchain transitivity *)
Lemma Efchain_trans N c l c' l' c'' :
  Efchain N c l c' -> Efchain N c' l' c'' -> Efchain N c (l ++ l') c''.
Proof.
  intro. generalize dependent c''. induction H; intros.
    eapply Efchain_eq_l; eauto. apply config.eq_sym. auto.
    econstructor 2; eauto.
Qed.

(* prepend a generated trace with an empty firing chain *)
Lemma Gen_Efchain N c l c' pi ot :
  Efchain N c l c' -> Gen N c' pi ot -> Gen N c (path.prep l pi) ot.
Proof.
  intros. destruct H0; econstructor; eauto; eapply Efchain_trans; eauto.
Qed.

Lemma Gen'_Efchain N c l c' ot :
  Efchain N c l c' -> Gen' N c' ot -> Gen' N c ot.
Proof.
  intros. destruct H0; econstructor; try eapply Efchain_trans; eauto.
Qed.

(* reachable configurations are well-formed *)
Lemma Reach_nil N : Wf N -> ~ Reach N [].
Proof.
  do 2 intro. destruct H. inversion H0.
    destruct (init N). apply Wf_init0. auto.
      eapply in_nil. apply H. apply in_eq.
    destruct H1. edestruct Wf_T0; eauto. destruct co. apply H6. auto.
      eapply in_nil. apply H4. apply config.In_union. right. apply in_eq.
Qed.

Lemma Reach_eq N c c' : Reach N c -> config.eq c c' -> Reach N c'.
Proof.
  intros. destruct H.
    constructor. eapply config.eq_trans; eauto. apply config.eq_sym. auto.
    econstructor 2. eauto. destruct H1. econstructor; eauto.
      eapply config.eq_trans; eauto.
Qed.

(* append an empty firing chain to a reachable configuration *)
Lemma Reach_Efchain N c l c' : Reach N c -> Efchain N c l c' -> Reach N c'.
Proof.
  intros. induction H0. eapply Reach_eq; eauto.
  apply IHEfchain. econstructor 2; eauto.
Qed.

(* ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~ *)

Lemma DC' X (P : nat -> X -> Prop) R :
  (forall n x, P n x -> exists y, P (S n) y /\ R n x y) -> forall x, P 0 x ->
  exists f, f 0 = x /\ forall n, P n (f n) /\ R n (f n) (f (S n)).
Proof.
  intros. cut (exists f : nat -> {p | P (fst p) (snd p)},
      snd (proj1_sig (f 0)) = x /\ forall n, fst (proj1_sig (f n)) = n /\
      S (fst (proj1_sig (f n))) = fst (proj1_sig (f (S n))) /\
      R (fst (proj1_sig (f n))) (snd (proj1_sig (f n)))
      (snd (proj1_sig (f (S n))))).
    intro. destruct H1 as [?[]]. exists (fun n => snd (proj1_sig (x0 n))).
      split; auto. intro. destruct (H2 n) as [?[]]. rewrite H3 in H5.
      split; auto. destruct (x0 n); simpl in *. rewrite <- H3. auto.
    cut (exists f : nat -> {p | P (fst p) (snd p)}, f 0 = exist _ (0, x) H0 /\
        forall n, (fun t t' => S (fst (proj1_sig t)) = fst (proj1_sig t') /\
        R (fst (proj1_sig t)) (snd (proj1_sig t)) (snd (proj1_sig t')))
        (f n) (f (S n))).
      simpl. intro. destruct H1 as [?[]]. exists x0. split. rewrite H1. auto.
        intro. split. induction n. rewrite H1. auto. destruct (H2 n).
        rewrite IHn in H3. auto. destruct (H2 n). auto.
      eapply Dependent_Choice. intro. destruct x0. simpl. edestruct H as [?[]].
        apply p. exists (exist _ (S (fst x0), x1) H1). split; auto.
Qed.

Fixpoint skip ot n : opttrace.t :=
  match n with
  | 0 => ot
  | S n' =>
    match ot with
    | opttrace.eps => opttrace.eps
    | opttrace.opt o ot' => skip ot' n'
    end
  end.

Inductive Gen'' N ot n : list config.t * option config.t -> Prop :=
| Gen''_None l : Gen'' N ot n (l, None)
| Gen''_Some l c : Gen' N c (skip ot n) -> Gen'' N ot n (l, Some c).

Inductive Op N ot n :
  list config.t * option config.t -> list config.t * option config.t -> Prop :=
| Op_None l p : Op N ot n (l, None) p
| Op_Some_None l c l' :
    skip ot n = opttrace.eps -> Efchain N c l' (fin N) ->
    Op N ot n (l, Some c) (l', None)
| Op_Some_None_Some l c l' c' ot' :
    skip ot n = opttrace.opt None ot' -> Efchain N c l' c' ->
    Op N ot n (l, Some c) (l', Some c')
| Op_Some_Some_Some l c l' c' B c'' ot' :
    skip ot n = opttrace.opt (Some B) ot' -> Efchain N c l' c' ->
    Firing N c' (Some B) c'' -> Op N ot n (l, Some c) (l', Some c'').

CoFixpoint path_f f : path.t :=
  match f 0 with (l, o) =>
    match o with
    | None => path.fin l
    | Some c => path.hop l c (path_f (fun n => f (S n)))
    end
  end.

Lemma path_f_None f l : f 0 = (l, None) -> path_f f = path.fin l.
Proof.
  intro. rewrite path.match_ at 1. simpl. rewrite H. auto.
Qed.

Lemma path_f_Some f l c :
  f 0 = (l, Some c) -> path_f f = path.hop l c (path_f (fun n => f (S n))).
Proof.
  intro. rewrite path.match_ at 1. simpl. rewrite H. auto.
Qed.

Lemma Gen_Gen' N c ot : Gen' N c ot -> exists pi, Gen N c pi ot.
Proof.
  intro. cut (exists f, (exists l, f 0 = (l, Some c)) /\
      forall n, Gen'' N ot n (f n) /\ Op N ot n (f n) (f (S n))).
    intro. destruct H0 as [?[[]?]]. exists (path_f (fun n => x (S n))). clear H.
        generalize dependent ot. generalize dependent x0.
        generalize dependent x. generalize dependent c. cofix F. intros.
        destruct (H1 0). rewrite H0 in *. inversion H2; simpl in H6; rewrite H6.
      erewrite path_f_None; eauto. econstructor; eauto.
      erewrite path_f_Some; eauto. econstructor; eauto.
        eapply F with (x:=fun n => x (S n)); eauto. intro. rewrite H6 in H1.
        destruct (H1 (S n)). inversion H8. split; constructor. split.
        constructor; auto. rewrite <- H10 in H9. inversion H9.
        constructor; auto. econstructor; eauto. econstructor 4; eauto.
      erewrite path_f_Some; eauto. econstructor; eauto.
        eapply F with (x:=fun n => x (S n)); eauto. intro. rewrite H6 in H1.
        destruct (H1 (S n)). inversion H9. split; constructor. split.
        constructor; auto. rewrite <- H11 in H10. inversion H10.
        constructor; auto. econstructor; eauto. econstructor 4; eauto.
    cut (exists f, f 0 = ([], Some c) /\
        forall n, Gen'' N ot n (f n) /\ Op N ot n (f n) (f (S n))).
      intro. destruct H0 as [?[]]. eauto.
      apply DC'. intros. inversion H0. exists ([], None). split; constructor.
        clear - H1. generalize dependent ot. induction n; intros; simpl in H1.
          destruct H1. exists (l0, None). split; constructor; auto.
            exists (l0, Some c'). split; econstructor; simpl; eauto.
            exists (l0, Some c''). split. constructor; simpl; auto.
              econstructor 4; simpl; eauto.
          destruct ot.
            inversion H1. exists (l0, None). split; constructor; auto.
            edestruct IHn as [?[]]; eauto. exists x. inversion H.
              rewrite <- H2 in H0. inversion H0. split; constructor; auto.
              rewrite <- H3 in H0. inversion H0.
                split; econstructor; eauto.
                split. constructor; auto. econstructor 4; eauto.
        constructor. auto.
Qed.

End petri.

End MPetris.

(* (c) 2020 Brittany Ro Nkounkou *)
