(* Brittany Nkounkou *)
(* August 2020 *)
(* Sequence Petri Net *)

Require Export PetBag.

Set Implicit Arguments.

Module MPetSeq (env : Environment).
Module Export M := MPetBag env.

Module seq.

(* initial configuration *)
Definition init init1 : config.t :=
  map place.L init1.

(* final configuration *)
Definition fin fin2 : config.t :=
  map place.R fin2.

(* sequence petri net *)
Definition make N1 N2 : petri.t :=
  petri.make
    (init (petri.init N1))
    (fin (petri.fin N2))
    (map (trans.map place.L) (petri.T N1)
    ++ (map place.L (petri.fin N1), None, map place.R (petri.init N2))
    :: map (trans.map place.R) (petri.T N2)).

(* empty-firing-chain-to-fin decider *)
Definition ef ef1 ef2 init2 : petri.efc_fin :=
  fun c =>
  match c with
  | place.L _ :: _ => andb (ef1 (config.L c)) (ef2 init2)
  | _ => ef2 (config.R c)
  end.

(* nth-bag finder *)
Definition nb (ef1 : petri.efc_fin) nb1 nb2 init2 : petri.nth_bag :=
  fun c sig sync F n =>
  let F1 (B : bag.t) (c1' : config.t) := F B (map place.L c1') in
  let F2 (B : bag.t) (c2' : config.t) := F B (map place.R c2') in
  match c with
  | place.L _ :: _ =>
    match nb1 (config.L c) sig sync F1 n with
    | Some (p, o) => Some
      match o with
      | None => (p, None)
      | Some n' =>
        if ef1 (config.L c)
        then
          match nb2 init2 sig sync F2 n' with
          | Some t => t
          | None => (p, Some n')
          end
        else (p, Some n')
      end
    | None => if ef1 (config.L c) then nb2 init2 sig sync F2 n else None
    end
  | _ => nb2 (config.R c) sig sync F2 n
  end.

(* ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~ *)

(* the sequence petri net is well-formed *)
Lemma Wf N1 N2 : petri.Wf N1 -> petri.Wf N2 -> petri.Wf (make N1 N2).
Proof.
  intros. destruct H, H0. split.
    intro; apply Wf_init; eapply map_eq_nil; apply H.
    intro; apply Wf_fin0; eapply map_eq_nil; apply H.
    intros. destruct (in_app_or _ _ _ H).
      destruct (proj1 (in_map_iff _ _ _) H0) as [[[]][]].
        inversion H1. destruct (Wf_T _ _ _ H2). split.
          intro; apply H3; eapply map_eq_nil; eauto.
          intro; apply H7; eapply map_eq_nil; eauto.
      destruct H0. inversion H0. split.
          intro; apply Wf_fin; eapply map_eq_nil; eauto.
          intro; apply Wf_init0; eapply map_eq_nil; eauto.
      destruct (proj1 (in_map_iff _ _ _) H0) as [[[]][]].
        inversion H1. destruct (Wf_T0 _ _ _ H2). split.
          intro; apply H3; eapply map_eq_nil; eauto.
          intro; apply H7; eapply map_eq_nil; eauto.
Qed.

(* ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~ *)

(* any firing from the left petri net is in the sequence petri net *)
Lemma Firing_L N1 N2 c w c' :
  petri.Firing N1 c w c' ->
  petri.Firing (make N1 N2) (map place.L c) w (map place.L c').
Proof.
  assert (forall p p' : place.t, place.L p = place.L p' -> p = p').
    intros. inversion H. auto.
  intro. destruct H0. econstructor.
    apply in_or_app. left. apply in_map_iff. exists (ci, w, co).
      split; simpl; eauto.
    apply config.le_map; auto.
    rewrite config.diff_map, config.union_map; auto. apply config.eq_map; auto.
Qed.

(* any firing from the right petri net is in the sequence petri net *)
Lemma Firing_R N1 N2 c w c' :
  petri.Firing N2 c w c' ->
  petri.Firing (make N1 N2) (map place.R c) w (map place.R c').
Proof.
  assert (forall p p' : place.t, place.R p = place.R p' -> p = p').
    intros. inversion H. auto.
  intro. destruct H0. econstructor.
    apply in_or_app. right. right. apply in_map_iff. exists (ci, w, co).
      split; simpl; eauto.
    apply config.le_map; auto.
    rewrite config.diff_map, config.union_map; auto. apply config.eq_map; auto.
Qed.

(* any firing from a left configuration either
    ends on a left configuration and is a firing from the left petri net, or
    starts from the final configuration of the left petri net and ends on the
      initial configuration of the right petri net *)
Lemma Firing_L_ N1 N2 c w c' :
  petri.Wf N2 -> petri.Safe N1 -> petri.Reach N1 c ->
  petri.Firing (make N1 N2) (map place.L c) w c' ->
  c' = map place.L (config.L c') /\ petri.Firing N1 c w (config.L c') \/
  config.eq c (petri.fin N1) /\ config.eq c' (map place.R (petri.init N2)).
Proof.
  assert (forall p p' : place.t, place.L p = place.L p' -> p = p').
    intros. inversion H. auto.
  intros. destruct H3. edestruct in_app_or. apply H3.
    left. rewrite in_map_iff in H6. destruct H6 as [?[]]. destruct x, p.
      inversion H6. rewrite <- H9, <- H10, <- H11 in *.
      rewrite config.le_map in H4; auto.
      erewrite config.diff_map, config.union_map in H5; auto. split.
      eapply config.eq_map_L; eauto.
      erewrite config.eq_map_L, config.eq_map in H5; eauto. econstructor; eauto.
    right. destruct H6.
      inversion H6. rewrite <- H8, <- H10 in *.
        rewrite config.le_map in H4; auto. split. split; auto.
        apply config.eq_sym. eapply config.eq_trans; eauto.
        rewrite config.diff_eq. apply config.eq_sym, config.union_nil.
        apply config.eq_map, config.eq_sym; auto. split; auto.
      exfalso. rewrite in_map_iff in H6. destruct H6 as [?[]]. destruct x, p.
        inversion H6. rewrite <- H9 in H4. destruct H0. edestruct Wf_T; eauto.
        destruct t0. contradiction. absurd (In (place.R t0) (map place.L c)).
          clear. intro. induction c; auto. inversion H; auto. inversion H0.
          apply H4. apply in_eq.
Qed.

(* any firing from a right configuration ends on a right configuration and is a
    firing from the right petri net *)
Lemma Firing_R_ N1 N2 c w c' :
  petri.Wf N1 -> petri.Firing (make N1 N2) (map place.R c) w c' ->
  c' = map place.R (config.R c') /\ petri.Firing N2 c w (config.R c').
Proof.
  assert (forall p p' : place.t, place.R p = place.R p' -> p = p').
    intros. inversion H. auto.
  intros. destruct H1. edestruct in_app_or. apply H1.
    exfalso. destruct H0. rewrite in_map_iff in H4. destruct H4 as [?[]].
      destruct x, p. inversion H0. rewrite <- H6 in H2. edestruct Wf_T; eauto.
      destruct t0. contradiction. absurd (In (place.L t0) (map place.R c)).
        clear. intro. induction c; auto. inversion H; auto. inversion H0.
        apply H2. apply in_eq.
    destruct H4.
      exfalso. destruct H0. inversion H4. rewrite <- H5 in H2.
        destruct (petri.fin N1). contradiction.
        absurd (In (place.L t) (map place.R c)).
          clear. intro. induction c; auto. inversion H; auto. inversion H0.
          apply H2. apply in_eq.
      rewrite in_map_iff in H4. destruct H4 as [?[]]. destruct x, p.
        inversion H4. rewrite <- H7, <- H8, <- H9 in *.
        rewrite config.le_map in H2; auto.
        rewrite config.diff_map, config.union_map in H3; auto. split.
          eapply config.eq_map_R; eauto.
          erewrite config.eq_map_R, config.eq_map in H3; eauto.
            econstructor; eauto.
Qed.

(* any left-to-left firing is a firing from the left petri net *)
Lemma Firing_L_L N1 N2 c w c' :
  petri.Wf N2 -> petri.Firing (make N1 N2) (map place.L c) w (map place.L c') ->
  petri.Firing N1 c w c'.
Proof.
  assert (forall p p' : place.t, place.L p = place.L p' -> p = p').
    intros. inversion H. auto.
  intros. destruct H1. edestruct in_app_or. apply H1.
    rewrite in_map_iff in H4. destruct H4 as [?[]]. destruct x, p.
      inversion H4. rewrite <- H7, <- H8, <- H9 in *.
      rewrite config.le_map in H2; auto.
      rewrite config.diff_map, config.union_map, config.eq_map in H3; auto.
      econstructor; eauto.
    exfalso. destruct H0. destruct H4.
      inversion H0. rewrite <- H7 in H3. destruct (petri.init N2).
        contradiction. absurd (In (place.R t) (map place.L c')).
          clear. intro. induction c'; auto. inversion H; auto. inversion H0.
          apply H3. apply config.In_union. right. apply in_eq.
      rewrite in_map_iff in H0. destruct H0 as [?[]]. destruct x, p.
        inversion H0. rewrite <- H6 in H2. edestruct Wf_T; eauto. destruct t0.
          contradiction. absurd (In (place.R t0) (map place.L c)).
            clear. intro. induction c; auto. inversion H; auto. inversion H0.
            apply H2. apply in_eq.
Qed.

(* any right-to-right firing is a firing from the right petri net *)
Lemma Firing_R_R N1 N2 c w c' :
  petri.Wf N1 -> petri.Firing (make N1 N2) (map place.R c) w (map place.R c') ->
  petri.Firing N2 c w c'.
Proof.
  assert (forall p p' : place.t, place.R p = place.R p' -> p = p').
    intros. inversion H. auto.
  intros. destruct H1. edestruct in_app_or. apply H1.
    exfalso. destruct H0. rewrite in_map_iff in H4. destruct H4 as [?[]].
      destruct x, p. inversion H0. rewrite <- H6 in H2. edestruct Wf_T; eauto.
      destruct t0. contradiction. absurd (In (place.L t0) (map place.R c)).
        clear. intro. induction c; auto. inversion H; auto. inversion H0.
        apply H2. apply in_eq.
    destruct H4.
      exfalso. destruct H0. inversion H4. rewrite <- H5 in H2.
        destruct (petri.fin N1). contradiction.
        absurd (In (place.L t) (map place.R c)).
          clear. intro. induction c; auto. inversion H; auto. inversion H0.
          apply H2. apply in_eq.
      rewrite in_map_iff in H4. destruct H4 as [?[]]. destruct x, p.
        inversion H4. rewrite <- H7, <- H8, <- H9 in *.
        rewrite config.le_map in H2; auto.
        rewrite config.diff_map, config.union_map, config.eq_map in H3; auto.
        econstructor; eauto.
Qed.

(* there are no non-empty left-to-right firings *)
Lemma Firing_L_Some_R N1 N2 c B c' :
  petri.Wf N1 -> petri.Wf N2 ->
  ~ petri.Firing (make N1 N2) (map place.L c) (Some B) (map place.R c').
Proof.
  do 3 intro. inversion H1. edestruct in_app_or. apply H2.
    destruct H. rewrite in_map_iff in H5. destruct H5 as [?[]]. destruct x, p.
      inversion H. rewrite <- H9 in H4. edestruct Wf_T; eauto. destruct t.
      contradiction. absurd (In (place.L t) (map place.R c')).
        clear. intro. induction c'; auto. inversion H; auto. inversion H0.
        apply H4. apply config.In_union. right. apply in_eq.
    destruct H5. inversion H5.
      destruct H0. rewrite in_map_iff in H5. destruct H5 as [?[]].
        destruct x, p. inversion H0. rewrite <- H7 in H3. edestruct Wf_T; eauto.
        destruct t0. contradiction. absurd (In (place.R t0) (map place.L c)).
          clear. intro. induction c; auto. inversion H; auto. inversion H0.
          apply H3. apply in_eq.
Qed.

(* there are no right-to-left firings *)
Lemma Firing_R_L N1 N2 c w c' :
  petri.Wf N1 -> petri.Wf N2 ->
  ~ petri.Firing (make N1 N2) (map place.R c) w (map place.L c').
Proof.
  do 3 intro. inversion H1. edestruct in_app_or. apply H2.
    destruct H. rewrite in_map_iff in H5. destruct H5 as [?[]]. destruct x, p.
      inversion H. rewrite <- H7 in H3. edestruct Wf_T; eauto. destruct t0.
      contradiction. absurd (In (place.L t0) (map place.R c)).
        clear. intro. induction c; auto. inversion H; auto. inversion H0.
        apply H3. apply in_eq.
    destruct H0. destruct H5.
      inversion H0. rewrite <- H8 in H4. destruct (petri.init N2).
        contradiction. absurd (In (place.R t) (map place.L c')).
          clear. intro. induction c'; auto. inversion H; auto. inversion H0.
          apply H4. apply config.In_union. right. apply in_eq.
      rewrite in_map_iff in H0. destruct H0 as [?[]]. destruct x, p.
        inversion H0. rewrite <- H9 in H4. edestruct Wf_T; eauto. destruct t.
      contradiction. absurd (In (place.R t) (map place.L c')).
        clear. intro. induction c'; auto. inversion H; auto. inversion H0.
        apply H4. apply config.In_union. right. apply in_eq.
Qed.

(* ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~ *)

(* any empty firing chain from the left petri net is in the sequence petri
    net *)
Lemma Efchain_L N1 N2 c l c' :
  petri.Wf N2 -> petri.Efchain N1 c l c' ->
  petri.Efchain (make N1 N2) (map place.L c) (map (map place.L) l)
    (map place.L c').
Proof.
  intros. induction H0. constructor. apply config.eq_map.
    intros. inversion H1. auto. auto.
  constructor. apply Firing_L; auto. auto.
Qed.

(* any empty firing chain from the right petri net is in the sequence petri
    net *)
Lemma Efchain_R N1 N2 c l c' :
  petri.Wf N1 -> petri.Efchain N2 c l c' ->
  petri.Efchain (make N1 N2) (map place.R c) (map (map place.R) l)
    (map place.R c').
Proof.
  intros. induction H0. constructor. apply config.eq_map.
    intros. inversion H1. auto. auto.
  constructor. apply Firing_R; auto. auto.
Qed.

(* there are no right-to-left empty firing chains *)
Lemma Efchain_R_L N1 N2 c l c' :
  petri.Wf N1 -> petri.Wf N2 -> petri.Reach N2 (config.R c) ->
  petri.Efchain (make N1 N2) c l c' ->
  c = map place.R (config.R c) -> c' = map place.L (config.L c') -> False.
Proof.
  intros. induction H2.
    destruct (config.R c). eapply petri.Reach_nil; eauto.
      absurd (In (place.R t) c').
        rewrite H4. clear. intro. induction (config.L c'); auto.
          inversion H; auto. inversion H0.
        apply H2. rewrite H3. apply in_eq.
    rewrite H3 in H2. edestruct Firing_R_. apply H. eauto.
      apply IHEfchain; auto. econstructor 2; eauto.
Qed.

(* any left-to-left empty firing chain is from the left petri net *)
Lemma Efchain_L_L N1 N2 c l c' :
  petri.Wf N1 -> petri.Wf N2 -> petri.Safe N1 -> petri.Reach N1 (config.L c) ->
  petri.Efchain (make N1 N2) c l c' ->
  c = map place.L (config.L c) -> c' = map place.L (config.L c') ->
  petri.Efchain N1 (config.L c) (map config.L l) (config.L c').
Proof.
  intros. induction H3. constructor. apply config.eq_L; auto.
  rewrite H4 in H3. edestruct Firing_L_ as [[]|[]]; eauto.
    constructor; auto. apply IHEfchain; auto. econstructor 2; eauto.
    exfalso. edestruct Efchain_R_L. apply H. eauto. rewrite config.R_map_R.
      apply petri.Reach_init. eapply petri.Efchain_eq_l; eauto.
      rewrite config.R_map_R. auto. auto.
Qed.

(* any right-to-right empty firing chain is from the right petri net *)
Lemma Efchain_R_R N1 N2 c l c' :
  petri.Wf N1 -> petri.Efchain (make N1 N2) c l c' ->
  c = map place.R (config.R c) -> c' = map place.R (config.R c') ->
  petri.Efchain N2 (config.R c) (map config.R l) (config.R c').
Proof.
  intros. induction H0. constructor. apply config.eq_R; auto.
  rewrite H1 in H0. edestruct Firing_R_; eauto. constructor; auto.
Qed.

(* any left-to-right empty firing chain is
    an empty firing chain from the left petri net ending on its final
      configuration concatenated with
    an empty firing chain from the right petri net starting on its initial
      configuration *)
Lemma Efchain_L_R N1 N2 c l c' :
  petri.Wf N1 -> petri.Wf N2 -> petri.Safe N1 -> petri.Reach N1 (config.L c) ->
  petri.Efchain (make N1 N2) c l c' ->
  c = map place.L (config.L c) -> c' = map place.R (config.R c') ->
  petri.Efchain N1 (config.L c)
    (map config.L (path.before (map place.R (petri.init N2)) l))
    (petri.fin N1) /\
  petri.Efchain N2 (petri.init N2)
    (map config.R (path.after (map place.R (petri.init N2)) l)) (config.R c').
Proof.
  intros. induction H3.
    exfalso. destruct (config.L c). edestruct petri.Reach_nil. apply H. eauto.
      absurd (In (place.L t) c').
        rewrite H5. clear. intro. induction (config.R c'); auto.
          inversion H; auto. inversion H0.
        apply H3. rewrite H4. apply in_eq.
    rewrite H4 in H3. edestruct Firing_L_ as [[]|[]]; eauto.
      destruct IHEfchain; auto. econstructor 2; eauto. simpl.
          destruct config.dec.
        exfalso. destruct H0. destruct (petri.init N2). contradiction.
          absurd (In (place.R t) c').
            rewrite H7. clear. intro. induction (config.L c'); auto.
              inversion H; auto. inversion H0.
            apply e. apply in_eq.
        split; auto. constructor; auto.
      simpl. destruct config.dec. split. constructor; auto.
        rewrite <- config.R_map_R at 1. eapply Efchain_R_R; auto. apply H.
          eapply petri.Efchain_eq_l; eauto. rewrite config.R_map_R. auto.
        destruct n. apply config.eq_sym. auto.
Qed.

(* any empty firing chain from a right configuration ends on a right
    configuration and is an empty firing chain from the right petri net *)
Lemma Efchain_R_ N1 N2 c l c' : petri.Wf N1 ->
  petri.Efchain (make N1 N2) c l c' -> c = map place.R (config.R c) ->
  c' = map place.R (config.R c') /\
    petri.Efchain N2 (config.R c) (map config.R l) (config.R c').
Proof.
  intros. induction H0.
    split. eapply config.eq_map_R. rewrite <- H1. auto.
      constructor. apply config.eq_R. auto.
    rewrite H1 in H0. edestruct Firing_R_; eauto. destruct IHEfchain; auto.
      split; auto. constructor; auto.
Qed.

(* ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~ *)

(* traces generated from a right configuration are generated by the right petri
    net *)
Lemma Gen_R N1 N2 : petri.Wf N1 ->
  forall c pi ot, petri.Gen (make N1 N2) c pi ot ->
  c = map place.R (config.R c) ->
  petri.Gen N2 (config.R c) (path.map config.R pi) ot.
Proof.
  intro. cofix F. intros. destruct H0; simpl.
    rewrite path.map_fin. constructor. rewrite <- config.R_map_R.
      eapply Efchain_R_R; eauto. rewrite config.R_map_R. auto.
    rewrite path.map_hop. edestruct Efchain_R_; eauto. econstructor; eauto.
    rewrite path.map_hop. edestruct Efchain_R_; eauto. rewrite H4 in H2.
      edestruct Firing_R_; eauto. econstructor; eauto.
Qed.

(* traces generated by the right petri net are generated from a right
    configuration *)
Lemma Gen'_R N1 N2 : petri.Wf N1 ->
  forall c ot, petri.Gen' N2 c ot -> petri.Gen' (make N1 N2) (map place.R c) ot.
Proof.
  intro. cofix F. intros.
  destruct H0; econstructor; eauto; try apply Efchain_R; eauto.
  apply Firing_R; eauto.
Qed.

(* ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~ *)

(* any reachable configuration is either
    a reachable configuration from the left petri net, or
    a reachable configuration from the right petri net and the final
      configuration of the left petri net is reachable *)
Lemma Reach N1 N2 c :
  petri.Wf N1 -> petri.Wf N2 -> petri.Safe N1 -> petri.Reach (make N1 N2) c ->
  c = map place.L (config.L c) /\ petri.Reach N1 (config.L c) \/
  c = map place.R (config.R c) /\ petri.Reach N2 (config.R c) /\
    petri.Reach N1 (petri.fin N1).
Proof.
  intros. induction H2.
    left. split. eapply config.eq_map_L. apply config.eq_sym, H2.
      constructor. rewrite <- config.L_map_L. apply config.eq_L. auto.
    destruct IHReach as [[]|[?[]]].
      rewrite H4 in H3. edestruct Firing_L_ as [[]|[]]; eauto.
        left. split; auto. econstructor 2; eauto.
        right. split. eapply config.eq_map_R. apply config.eq_sym, H7. split.
          constructor. rewrite <- config.R_map_R. apply config.eq_R. auto.
          eapply petri.Reach_eq; eauto.
      right. rewrite H4 in H3. edestruct Firing_R_. apply H. eauto. split; auto.
        split; auto. econstructor 2; eauto.
Qed.

(* the first element of any reachable configuration is not the P place *)
Lemma Reach_P N1 N2 : petri.Wf N1 -> petri.Wf N2 -> petri.Safe N1 ->
  forall c, ~ petri.Reach (make N1 N2) (place.P :: c).
Proof.
  do 5 intro. destruct (Reach H H0 H1 H2) as [[]|[?[]]]; simpl in H3.
    destruct (config.L c); inversion H3.
    destruct (config.R c); inversion H3.
Qed.

(* if the first element of a reachable configuration is an L place,
    then it is a reachable configuration from the left petri net *)
Lemma Reach_L N1 N2 : petri.Wf N1 -> petri.Wf N2 -> petri.Safe N1 ->
  forall p c, petri.Reach (make N1 N2) (place.L p :: c) ->
  place.L p :: c = map place.L (config.L (place.L p :: c)) /\
  petri.Reach N1 (config.L (place.L p :: c)).
Proof.
  intros. destruct (Reach H H0 H1 H2); auto. destruct H3. simpl in H3.
  destruct (config.R c); inversion H3.
Qed.

(* if the first element of a reachable configuration is an R place,
    then it is a reachable configuration from the right petri net *)
Lemma Reach_R N1 N2 : petri.Wf N1 -> petri.Wf N2 -> petri.Safe N1 ->
  forall p c, petri.Reach (make N1 N2) (place.R p :: c) ->
  place.R p :: c = map place.R (config.R (place.R p :: c)) /\
  petri.Reach N2 (config.R (place.R p :: c)) /\ petri.Reach N1 (petri.fin N1).
Proof.
  intros. destruct (Reach H H0 H1 H2); auto. destruct H3. simpl in H3.
  destruct (config.L c); inversion H3.
Qed.

(* any reachable non-empty configuration is decidably either
    a reachable configuration from the left petri net, or
    a reachable configuration from the right petri net and the final
      configuration of the left petri net is reachable *)
Definition Reach_cons N1 N2 : petri.Wf N1 -> petri.Wf N2 -> petri.Safe N1 ->
  forall p c, petri.Reach (make N1 N2) (p :: c) ->
  {p :: c = map place.L (config.L (p :: c)) /\
    petri.Reach N1 (config.L (p :: c))} +
  {p :: c = map place.R (config.R (p :: c)) /\
    petri.Reach N2 (config.R (p :: c)) /\ petri.Reach N1 (petri.fin N1)}.
Proof.
  intros. destruct p. edestruct (Reach_P H H0); eauto.
    left. eapply (Reach_L H H0); eauto.
    right. eapply (Reach_R H H0); eauto.
Defined.

(* any reachable configuration from the left petri net is reachable in the
    sequence petri net *)
Lemma Reach_L' N1 N2 c :
  petri.Reach N1 c -> petri.Reach (make N1 N2) (map place.L c).
Proof.
  intro. induction H. constructor. apply config.eq_map; auto.
    intros. inversion H0. auto.
  econstructor 2; eauto. eapply Firing_L; eauto.
Qed.

(* if the final configuration of the left petri net is reachable,
    then any reachable configuration from the right petri net is reachable in
    the sequence petri net *)
Lemma Reach_R' N1 N2 c :
  petri.Reach N1 (petri.fin N1) -> petri.Reach N2 c ->
  petri.Reach (make N1 N2) (map place.R c).
Proof.
  intros. induction H0; econstructor 2; eauto.
    eapply Reach_L'; eauto. eapply petri.Firing_eq_eq.
      apply in_or_app. right. apply in_eq. apply config.eq_refl.
      apply config.eq_map; auto. intros. inversion H1. auto.
    eapply Firing_R; eauto.
Qed.

(* the sequence petri net is safe *)
Lemma Safe N1 N2 : petri.Wf N1 -> petri.Wf N2 ->
  petri.Safe N1 -> petri.Safe N2 -> petri.Safe (make N1 N2).
Proof.
  do 7 intro. destruct (Reach H H0 H1 H3) as [[]|[?[]]].
    exfalso. destruct H0. simpl in H4. unfold fin in H4.
      destruct (petri.fin N2). contradiction. absurd (In (place.R t) c).
        rewrite H5. clear. intro. induction (config.L c); auto.
          inversion H; auto. inversion H0.
        apply H4. apply in_eq.
    rewrite H5. simpl in *. unfold fin in *. apply config.le_map.
      intros. inversion H8. auto. apply H2; auto.
      rewrite H5, config.le_map in H4; auto. intros. inversion H8. auto.
Qed.

(* ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~ *)

(* the left part of a trace from a full path and trace *)
CoFixpoint opttrace_l pi ot : opttrace.t :=
  match pi with
  | path.hop _ (place.L _ :: _) pi' =>
    match ot with
    | opttrace.eps => opttrace.eps
    | opttrace.opt o ot' => opttrace.opt o (opttrace_l pi' ot')
    end
  | _ => opttrace.eps
  end.

(* rewrite helper for trace_l applied to path.fin *)
Lemma opttrace_l_fin l ot : opttrace_l (path.fin l) ot = opttrace.eps.
Proof.
  rewrite opttrace.match_ at 1. auto.
Qed.

(* rewrite helper for trace_l applied to path.hop *)
Lemma opttrace_l_hop l c pi ot :
  opttrace_l (path.hop l c pi) ot =
    match c with
    | place.L _ :: _ =>
      match ot with
      | opttrace.eps => opttrace.eps
      | opttrace.opt o ot' => opttrace.opt o (opttrace_l pi ot')
      end
    | _ => opttrace.eps
    end.
Proof.
  rewrite opttrace.match_ at 1. simpl. destruct c; auto. destruct t; auto.
  destruct ot; auto.
Qed.

(* the left part of a path from a full path *)
CoFixpoint path_l init2 pi : path.t :=
  match pi with
  | path.fin l => path.fin (map config.L (path.before (map place.R init2) l))
  | path.hop l c pi' =>
    match c with
    | place.L _ :: _ =>
        path.hop (map config.L l) (config.L c) (path_l init2 pi')
    | _ => path.fin (map config.L (path.before (map place.R init2) l))
    end
  end.

(* rewrite helper for path_l applied to path.fin *)
Lemma path_l_fin init2 l :
  path_l init2 (path.fin l) =
    path.fin (map config.L (path.before (map place.R init2) l)).
Proof.
  rewrite path.match_ at 1. auto.
Qed.

(* rewrite helper for path_l applied to path.hop *)
Lemma path_l_hop init2 l c pi :
  path_l init2 (path.hop l c pi) =
    match c with
    | place.L _ :: _ => path.hop (map config.L l) (config.L c) (path_l init2 pi)
    | _ => path.fin (map config.L (path.before (map place.R init2) l))
    end.
Proof.
  rewrite path.match_ at 1. simpl. destruct c; auto. destruct t; auto.
Qed.

(* ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~ *)

(* trace_l is generated by the left petri net *)
Lemma Gen_opttrace_l N1 N2 : petri.Wf N1 -> petri.Wf N2 -> petri.Safe N1 ->
  forall c pi ot, petri.Reach N1 (config.L c) ->
  petri.Gen (make N1 N2) c pi ot -> c = map place.L (config.L c) ->
  petri.Gen N1 (config.L c) (path_l (petri.init N2) pi) (opttrace_l pi ot).
Proof.
  do 3 intro. cofix F. intros. rewrite opttrace.match_. destruct H3; simpl.
    rewrite path_l_fin. constructor. edestruct Efchain_L_R. apply H. eauto.
      auto. eauto. eauto. auto. simpl. rewrite config.R_map_R. auto. auto.
    rewrite path_l_hop. assert (petri.Reach (make N1 N2) c').
      eapply petri.Reach_Efchain. apply Reach_L'; eauto. rewrite H4 in H3.
      eauto. destruct c'. edestruct petri.Reach_nil. apply Wf. apply H. eauto.
      auto. destruct t. edestruct Reach_P. apply H. eauto. auto. eauto.
        edestruct Reach_L. apply H. eauto. auto. eauto. constructor; auto.
          eapply Efchain_L_L; eauto.
        edestruct Reach_R. apply H. eauto. auto. eauto. constructor.
          edestruct Efchain_L_R. apply H. eauto. auto. eauto. eauto. auto. auto.
          auto.
    rewrite path_l_hop. assert (petri.Reach (make N1 N2) c').
      eapply petri.Reach_Efchain. apply Reach_L'; eauto. rewrite <- H4. eauto.
      assert (petri.Reach (make N1 N2) c''). econstructor 2; eauto.
      destruct c''. edestruct petri.Reach_nil. eapply Wf. apply H. eauto. auto.
      destruct t. edestruct Reach_P. apply H. eauto. auto. eauto.
        edestruct Reach_L. apply H. eauto. auto. eauto. rewrite H9 in H5.
            destruct (Reach H H0 H1 H7) as [[]|[?[]]]; rewrite H11 in H5.
          econstructor. eapply Efchain_L_L; eauto. eapply Firing_L_L; eauto.
            apply F; auto.
          edestruct Firing_R_L. apply H. eauto. eauto.
        edestruct Reach_R as [?[]]. apply H. eauto. auto. eauto.
            rewrite H9 in H5. destruct (Reach H H0 H1 H7) as [[]|[?[]]].
          rewrite H12 in H5. edestruct Firing_L_Some_R. apply H. eauto. eauto.
          constructor. edestruct Efchain_L_R. apply H. eauto. auto. eauto.
            eauto. auto. auto. auto.
Qed.

(* trace_l appended with the language of the right petri net includes the
    original trace *)
Lemma app_opttrace_l N1 N2 : petri.Wf N1 -> petri.Wf N2 -> petri.Safe N1 ->
  forall c pi ot, petri.Reach N1 (config.L c) ->
  petri.Gen (make N1 N2) c pi ot -> c = map place.L (config.L c) ->
  slang.app (opttrace_l pi ot) (petri.Lang N2) ot.
Proof.
  do 3 intro. cofix F. intros. destruct H3.
    rewrite opttrace_l_fin. constructor. eexists. econstructor.
      edestruct Efchain_L_R. apply H. eauto. auto. eauto. eauto. auto.
      simpl. rewrite config.R_map_R. auto. simpl in H6.
      rewrite config.R_map_R in H6. eauto.
    rewrite opttrace_l_hop. assert (petri.Reach (make N1 N2) c').
      eapply petri.Reach_Efchain. apply Reach_L'; eauto. rewrite H4 in H3.
      eauto. destruct c'. edestruct petri.Reach_nil. apply Wf. apply H. eauto.
      auto. destruct t. edestruct Reach_P. apply H. eauto. auto. eauto.
        edestruct Reach_L. apply H. eauto. auto. eauto. constructor; eauto.
        edestruct Reach_R. apply H. eauto. auto. eauto. constructor.
          edestruct Efchain_L_R. apply H. eauto. auto. eauto. eauto. auto. auto.
          eexists. constructor; eauto. eapply Gen_R; auto. apply H. eauto.
    rewrite opttrace_l_hop. assert (petri.Reach (make N1 N2) c').
      eapply petri.Reach_Efchain. apply Reach_L'; eauto. rewrite <- H4. eauto.
      assert (petri.Reach (make N1 N2) c''). econstructor 2; eauto.
      destruct c''. edestruct petri.Reach_nil. eapply Wf. apply H. eauto. auto.
      destruct t. edestruct Reach_P. apply H. eauto. auto. eauto.
        edestruct Reach_L. apply H. eauto. auto. eauto. constructor.
          eapply F; eauto.
        edestruct Reach_R as [?[]]. apply H. eauto. auto. eauto. constructor.
          eexists. rewrite <- config.R_map_R at 1. eapply Gen_R. apply H.
          destruct (Reach H H0 H1 H7) as [[]|[?[]]]; rewrite H12, H9 in H5.
            edestruct Firing_L_Some_R. apply H. eauto. eauto.
            econstructor; eauto. eapply Efchain_R; eauto. edestruct Efchain_L_R.
              apply H. eauto. auto. eauto. eauto. auto. auto. eauto.
              rewrite <- H9. eauto. rewrite config.R_map_R. auto.
Qed.

(* ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~ *)

(* Lang soundness *)
Lemma Lang_sound N1 N2 : petri.Wf N1 -> petri.Wf N2 -> petri.Safe N1 ->
  slang.le (petri.Lang (make N1 N2))
    (slang.concat (petri.Lang N1) (petri.Lang N2)).
Proof.
  do 5 intro. destruct H2. exists (opttrace_l x ot). split.
    exists (path_l (petri.init N2) x). rewrite <- config.L_map_L at 1.
      apply Gen_opttrace_l; auto. rewrite config.L_map_L.
      apply petri.Reach_init. rewrite config.L_map_L. auto.
    eapply app_opttrace_l; auto. apply H. auto. rewrite config.L_map_L.
      apply petri.Reach_init. auto. rewrite config.L_map_L. auto.
Qed.

(* Lang completeness *)
Lemma Lang_complete N1 N2 : petri.Wf N1 -> petri.Wf N2 ->
  slang.le (slang.concat (petri.Lang N1) (petri.Lang N2))
    (petri.Lang (make N1 N2)).
Proof.
  do 4 intro. destruct H1. destruct H1. destruct H1. apply petri.Gen_Gen'.
  generalize dependent ot. assert (petri.Gen' N1 (petri.init N1) x).
  eapply petri.Gen'_Gen; eauto. clear x0 H1. generalize dependent x. simpl.
  unfold init. generalize (petri.init N1). cofix F. intros.
  destruct H2; inversion H1.
    eapply petri.Gen'_Efchain.
      eapply petri.Efchain_trans. eapply Efchain_L; eauto.
        eapply petri.Efchain_eq_eq. apply in_or_app. right. apply in_eq.
        apply config.eq_refl. apply config.eq_refl.
      eapply Gen'_R; auto. destruct H3. eapply petri.Gen'_Gen; eauto.
    econstructor. eapply Efchain_L; eauto. eauto.
    econstructor. apply Efchain_L; eauto. apply Firing_L; eauto. eauto.
Qed.

(* ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~ *)

(* ef correctness *)
Definition EfcFin N1 N2 ef1 ef2 init2 :
  petri.Wf N1 -> petri.Wf N2 -> petri.Safe N1 ->
  petri.EfcFin N1 ef1 -> petri.EfcFin N2 ef2 -> petri.init N2 = init2 ->
  petri.EfcFin (make N1 N2) (ef ef1 ef2 init2).
Proof.
  unfold ef. intros. destruct X, X0. split; repeat intro.
    destruct c. edestruct petri.Reach_nil. eapply Wf. apply H. eauto. auto.
      destruct t. edestruct (Reach_P H H0); eauto.
        edestruct (Reach_L H H0); eauto. rewrite H5. edestruct EF_sound; eauto.
          apply (andb_prop _ _ H4). edestruct EF_sound0; eauto.
          apply petri.Reach_init. rewrite H2. apply (andb_prop _ _ H4).
          exists (map (map place.L) x ++ [map place.R (petri.init N2)] ++
          map (map place.R) x0). eapply petri.Efchain_trans.
          apply Efchain_L; eauto. eapply petri.Efchain_trans.
          eapply petri.Efchain_eq_eq. apply in_or_app. right. apply in_eq.
          apply config.eq_refl. apply config.eq_refl. apply Efchain_R; eauto.
        edestruct (Reach_R H H0) as [?[]]; eauto. rewrite H5.
          edestruct EF_sound0; eauto. exists (map (map place.R) x).
            apply Efchain_R; auto.
    destruct c. edestruct petri.Reach_nil. eapply Wf. apply H. eauto. auto.
      destruct t. edestruct (Reach_P H H0) ; eauto.
        edestruct (Reach_L H H0); eauto. apply andb_true_intro. simpl in H4.
          unfold fin in H4. edestruct Efchain_L_R. apply H. eauto. eauto. eauto.
          eauto. auto. rewrite config.R_map_R. auto. split.
          eapply EF_complete; eauto. rewrite <- H2.
          rewrite config.R_map_R in H8. eapply EF_complete0; eauto.
          apply petri.Reach_init.
        edestruct (Reach_R H H0) as [?[]]; eauto. eapply EF_complete0; eauto.
          rewrite <- config.R_map_R. eapply Efchain_R_R; auto. apply H. eauto.
            rewrite config.R_map_R. auto.
Defined.

(* nb soundness *)
Definition nb_sound N1 N2 ef1 nb1 nb2 init2 :
  petri.Wf N1 -> petri.Wf N2 -> petri.Safe N1 -> petri.EfcFin N1 ef1 ->
  petri.nb_sound N1 nb1 -> petri.nb_sound N2 nb2 -> petri.init N2 = init2 ->
  petri.nb_sound (make N1 N2) (nb ef1 nb1 nb2 init2).
Proof.
  unfold nb. repeat intro. destruct X.
  destruct c. edestruct petri.Reach_nil. eapply Wf. apply H. eauto. auto.
  destruct t0. edestruct (Reach_P H H0); eauto.
    edestruct (Reach_L H H0); eauto. rewrite H5.
        case_eq (nb1 (config.L (place.L t0 :: c)) sig sync
        (fun B c1' => F B (map place.L c1')) n); intros; rewrite H7 in H4.
      edestruct X0 as [?[?[?[?[?[?[?[?[]]]]]]]]]; eauto. destruct p. destruct o.
        case_eq (ef1 (config.L (place.L t0 :: c))); intro; rewrite H13 in H4.
          edestruct EF_sound; eauto. case_eq (nb2 init2 sig sync
              (fun B c2' => F B (map place.R c2')) n0); intros;
              rewrite H14 in H4; rewrite <- H4.
            edestruct X1 as [?[?[?[?[?[?[?[?[]]]]]]]]]; eauto. rewrite <- H2.
              apply petri.Reach_init. exists (map (map place.L) x4 ++
              [map place.R init2] ++ map (map place.R) x5), (map place.R x6),
              x7, (map place.R x8), x9. split. eapply petri.Efchain_trans.
              eapply Efchain_L; eauto. eapply petri.Efchain_trans.
              eapply petri.Efchain_eq_eq. apply in_or_app. right. rewrite <- H2.
              apply in_eq. apply config.eq_refl. rewrite H2.
              apply config.eq_refl. eapply Efchain_R; eauto. split; auto.
              apply Firing_R; auto.
            exists (map (map place.L) x), (map place.L x0), x1,
              (map place.L x2), x3. split. eapply Efchain_L; eauto. split; auto.
              apply Firing_L; auto.
          rewrite <- H4. exists (map (map place.L) x), (map place.L x0), x1,
            (map place.L x2), x3. split. eapply Efchain_L; eauto. split; auto.
            apply Firing_L; auto.
        rewrite <- H4. exists (map (map place.L) x), (map place.L x0), x1,
          (map place.L x2), x3. split. eapply Efchain_L; eauto. split; auto.
          apply Firing_L; auto.
      case_eq (ef1 (config.L (place.L t0 :: c))); intro; rewrite H8 in H4.
        edestruct EF_sound; eauto.
          edestruct X1 as [?[?[?[?[?[?[?[?[]]]]]]]]]; eauto.
          rewrite <- H2. apply petri.Reach_init. exists (map (map place.L) x ++
          [map place.R init2] ++ map (map place.R) x0), (map place.R x1), x2,
          (map place.R x3), x4. split. eapply petri.Efchain_trans.
          eapply Efchain_L; eauto. eapply petri.Efchain_trans.
          eapply petri.Efchain_eq_eq. apply in_or_app. right. rewrite <- H2.
          apply in_eq. apply config.eq_refl. rewrite H2. apply config.eq_refl.
          eapply Efchain_R; eauto. split; auto. apply Firing_R; auto.
        inversion H4.
    edestruct (Reach_R H H0) as [?[]]; eauto. rewrite H5.
      edestruct X1 as [?[?[?[?[?[?[?[?[]]]]]]]]]; eauto.
      exists (map (map place.R) x), (map place.R x0), x1, (map place.R x2), x3.
      split. eapply Efchain_R; eauto. split; auto. apply Firing_R; auto.
Defined.

(* nb completeness *)
Definition nb_complete N1 N2 ef1 nb1 nb2 init2 :
  petri.Wf N1 -> petri.Wf N2 -> petri.Safe N1 -> petri.EfcFin N1 ef1 ->
  petri.NthBag N1 nb1 -> petri.nb_complete N2 nb2 -> petri.init N2 = init2 ->
  petri.nb_complete (make N1 N2) (nb ef1 nb1 nb2 init2).
Proof.
  unfold nb. repeat intro. destruct X, X0.
  destruct c. edestruct petri.Reach_nil. eapply Wf. apply H. eauto. auto.
  assert (petri.Reach (make N1 N2) c'). eapply petri.Reach_Efchain; eauto.
  destruct c'. edestruct petri.Reach_nil. eapply Wf. apply H. eauto. auto.
  assert (petri.Reach (make N1 N2) c''). econstructor 2; eauto.
  destruct c''. edestruct petri.Reach_nil. eapply Wf. apply H. eauto. auto.
  destruct t. edestruct (Reach_P H H0); eauto.
    edestruct (Reach_L H H0); eauto.
        destruct (Reach_cons H H0 H1 H9) as [[]|[?[]]];
        destruct (Reach_cons H H0 H1 H10) as [[]|[?[]]].
      rewrite H13, H15 in H5. edestruct NB_complete with
        (F:=fun (B0 : bag.t) (c1' : config.t) => F B0 (map place.L c1')) as
        [??[]]. apply H12. eapply Efchain_L_L; eauto. eapply Firing_L_L; eauto.
        eauto. eauto. eauto. eauto.
        exists (map place.L x). apply Reach_L'; auto.
        exists (map (map place.L) x0). rewrite H15. apply Efchain_L; auto.
        intro. destruct (s n'). exists x1. intro. rewrite e0; eauto.
        destruct H17. rewrite H17. auto.
      rewrite H13, H15 in H5. edestruct Firing_L_Some_R. apply H. eauto. eauto.
      rewrite H13, H16 in H5. edestruct Firing_R_L. apply H. eauto. eauto.
      rewrite H13, H16 in H5. edestruct Efchain_L_R. apply H. eauto. auto.
        eauto. eauto. auto. auto. edestruct X1 with
        (F:=fun (B0 : bag.t) (c2' : config.t) => F B0 (map place.R c2')) as
        [??[]]. apply petri.Reach_init. eauto. eapply Firing_R_R. apply H.
        eauto. eauto. eauto. eauto. eauto.
        exists (map place.R x). apply Reach_R'; auto.
        exists (map (map place.R) x0). rewrite H16. apply Efchain_R; auto.
        intro. destruct (s n'). case_eq (nb1 (config.L (place.L t :: c)) sig
            sync (fun B0 c1' => F B0 (map place.L c1')) 0); intros.
          edestruct NB_Some_Some with (F:=fun (B0 : bag.t) (c1' : config.t) =>
            F B0 (map place.L c1')); eauto. exists x2. intro. destruct e1.
            rewrite H23. erewrite EF_complete; eauto. rewrite <- H2.
            rewrite e0; auto. destruct H22. rewrite H22. auto.
          exists x1. intro. rewrite NB_None_None; eauto.
            erewrite EF_complete; eauto. rewrite <- H2. rewrite e0; auto.
    edestruct (Reach_R H H0) as [?[]]; eauto.
        destruct (Reach_cons H H0 H1 H9) as [[]|[?[]]].
      edestruct Efchain_R_L. apply H. eauto. eauto. eauto. auto. auto.
      destruct (Reach_cons H H0 H1 H10) as [[]|[?[]]]; rewrite H14, H17 in H5.
        edestruct Firing_R_L. apply H. eauto. eauto.
        edestruct (Reach_R H H0); eauto. edestruct X1 with
          (F:=fun (B0 : bag.t) (c2' : config.t) => F B0 (map place.R c2')) as
          [??[]]. apply H12. eapply Efchain_R_R; auto. apply H. eauto. auto.
          eapply Firing_R_R. apply H. eauto. eauto. eauto. eauto. eauto.
          exists (map place.R x). apply Reach_R'; auto.
          exists (map (map place.R) x0). rewrite H17. apply Efchain_R; auto.
          intro. destruct (s n'). exists x1. intro. rewrite e0; auto.
Defined.

(* nb correctness *)
Definition NthBag N1 N2 ef1 nb1 nb2 init2 :
  petri.Wf N1 -> petri.Wf N2 -> petri.Safe N1 -> petri.EfcFin N1 ef1 ->
  petri.NthBag N1 nb1 -> petri.NthBag N2 nb2 -> petri.init N2 = init2 ->
  petri.NthBag (make N1 N2) (nb ef1 nb1 nb2 init2).
Proof.
  intros. inversion X0. destruct X1.
  split; try apply nb_sound; try apply nb_complete; auto;
  unfold nb; try unfold petri.f_None in *; try unfold petri.f_Some in *;
  repeat intro.
    destruct c; eauto. destruct t; eauto. simpl in *.
      case_eq (nb1 (t :: config.L c) sig sync (fun B c1' =>
          F B (map place.L c1')) n); intros; rewrite H5 in H4.
        destruct p. inversion H4.
        erewrite NB_None_None; eauto. destruct (ef1 (t :: config.L c)); auto.
          erewrite NB_None_None0; eauto.
    destruct c; eauto. destruct t0; eauto. simpl in *.
      case_eq (nb1 (t0 :: config.L c) sig sync (fun B c1' =>
          F B (map place.L c1')) n); intros; rewrite H5 in H4.
        destruct p. destruct (ef1 (t0 :: config.L c)).
          case_eq (nb2 init2 sig sync (fun B c2' =>
              F B (map place.R c2')) 0); intros.
            edestruct NB_Some_Some0 with
                (F:=fun (B : bag.t) (c2' : config.t) => F B (map place.R c2'));
                eauto.
              edestruct NB_Some_Some with
                (F:=fun (B : bag.t) (c1' : config.t) => F B (map place.L c1'));
                eauto.
              exists x0. destruct e0. rewrite H7. destruct e. rewrite e. eauto.
            edestruct NB_Some_Some with (F:=fun (B : bag.t) (c1' : config.t) =>
                F B (map place.L c1')); eauto.
              exists x. destruct e. rewrite H7. erewrite NB_None_None0; eauto.
        edestruct NB_Some_Some with (F:=fun (B : bag.t) (c1' : config.t) =>
            F B (map place.L c1')); eauto.
          exists x. destruct e. rewrite H6. eauto.
      destruct (ef1 (t0 :: config.L c)).
        edestruct NB_Some_Some0 with (F:=fun (B : bag.t) (c2' : config.t) =>
            F B (map place.R c2')); eauto.
          exists x. erewrite NB_None_None; eauto.
        inversion H4.
Defined.

End seq.

End MPetSeq.

(* (c) 2020 Brittany Ro Nkounkou *)
