(* Brittany Nkounkou *)
(* August 2020 *)
(* Selection Petri Net *)

Require Export PetPar.

Set Implicit Arguments.

Module MPetSel (env : Environment).
Module Export M := MPetPar env.

Module sel.

(* initial configuration *)
Definition init : config.t :=
  [place.L place.P].

(* final configuration *)
Definition fin : config.t :=
  [place.R (place.R place.P)].

(* selection petri net *)
Definition make N1 N2 : petri.t :=
  petri.make
    init
    fin
    ([(init, None, map place.RLL (petri.init N1));
      (init, None, map place.RLR (petri.init N2))]
      ++ map (trans.map place.RLL) (petri.T N1)
      ++ map (trans.map place.RLR) (petri.T N2)
      ++ [(map place.RLL (petri.fin N1), None, fin);
          (map place.RLR (petri.fin N2), None, fin)]).

(* empty-firing-chain-to-fin decider *)
Definition ef ef1 ef2 init1 init2 : petri.efc_fin :=
  fun c =>
  match c with
  | place.L _ :: _ => orb (ef1 init1) (ef2 init2)
  | place.R (place.L (place.L _)) :: _ => ef1 (config.RLL c)
  | place.R (place.L (place.R _)) :: _ => ef2 (config.RLR c)
  | _ => true
  end.

(* nth-bag finder *)
Definition nb nb1 nb2 init1 init2 : petri.nth_bag :=
  fun c sig sync F n =>
  let F1 (B : bag.t) (c1' : config.t) := F B (map place.RLL c1') in
  let F2 (B : bag.t) (c2' : config.t) := F B (map place.RLR c2') in
  match c with
  | place.L _ :: _ =>
    match nb1 init1 sig sync F1 n with
    | Some (p, o) => Some
      match o with
      | None => (p, None)
      | Some n' =>
        match nb2 init2 sig sync F2 n' with
        | Some t => t
        | None => (p, Some n')
        end
      end
    | None => nb2 init2 sig sync F2 n
    end
  | place.R (place.L (place.L _)) :: _ => nb1 (config.RLL c) sig sync F1 n
  | place.R (place.L (place.R _)) :: _ => nb2 (config.RLR c) sig sync F2 n
  | _ => None
  end.

(* ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~ *)

(* the selection petri net is well-formed *)
Lemma Wf N1 N2 : petri.Wf N1 -> petri.Wf N2 -> petri.Wf (make N1 N2).
Proof.
  intros. destruct H, H0. split; try discriminate. intros.
  destruct (in_app_or _ _ _ H).
    destruct H0.
      inversion H0. split; try discriminate.
        intro; apply Wf_init; eapply map_eq_nil; eauto.
      destruct H0; inversion H0. split; try discriminate.
        intro; apply Wf_init0; eapply map_eq_nil; eauto.
    destruct (in_app_or _ _ _ H0).
      destruct (proj1 (in_map_iff _ _ _) H1) as [[[]][]].
        inversion H2. destruct (Wf_T _ _ _ H3). split.
          intro; apply H4; eapply map_eq_nil; eauto.
          intro; apply H8; eapply map_eq_nil; eauto.
      destruct (in_app_or _ _ _ H1).
        destruct (proj1 (in_map_iff _ _ _) H2) as [[[]][]].
          inversion H3. destruct (Wf_T0 _ _ _ H4). split.
            intro; apply H5; eapply map_eq_nil; eauto.
            intro; apply H9; eapply map_eq_nil; eauto.
        destruct H2.
          inversion H2. split; try discriminate.
            intro; apply Wf_fin; eapply map_eq_nil; eauto.
          destruct H2; inversion H2. split; try discriminate.
            intro; apply Wf_fin0; eapply map_eq_nil; eauto.
Qed.

(* ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~ *)

(* the first element of any subconfiguration of init is not the P place *)
Lemma le_P_init c : ~ config.le (place.P :: c) init.
Proof.
  intro. edestruct H. apply in_eq. inversion H0. destruct H0.
Qed.

(* the first element of any subconfiguration of init is not an R place *)
Lemma le_R_init p c : ~ config.le (place.R p :: c) init.
Proof.
  intro. edestruct H. apply in_eq. inversion H0. destruct H0.
Qed.

(* the first element of any subconfiguration of fin is not the P place *)
Lemma le_P_fin c : ~ config.le (place.P :: c) fin.
Proof.
  intro. edestruct H. apply in_eq. inversion H0. destruct H0.
Qed.

(* the first element of any subconfiguration of fin is not an L place *)
Lemma le_L_fin p c : ~ config.le (place.L p :: c) fin.
Proof.
  intro. edestruct H. apply in_eq. inversion H0. destruct H0.
Qed.

(* the first element of any subconfiguration of fin is not the R P place *)
Lemma le_RP_fin c : ~ config.le (place.R place.P :: c) fin.
Proof.
  intro. edestruct H. apply in_eq. inversion H0. destruct H0.
Qed.

(* the first element of any subconfiguration of fin is not an R L place *)
Lemma le_RL_fin p c : ~ config.le (place.R (place.L p) :: c) fin.
Proof.
  intro. edestruct H. apply in_eq. inversion H0. destruct H0.
Qed.

(* ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~ *)

(* any firing from init is empty and ends on the initial configuration of
    either the left or the right petri net *)
Lemma Firing_init N1 N2 c w c' :
  petri.Wf N1 -> petri.Wf N2 -> config.eq c init ->
  petri.Firing (make N1 N2) c w c' ->
  w = None /\
  (config.eq c' (map place.RLL (petri.init N1)) \/
    config.eq c' (map place.RLR (petri.init N2))).
Proof.
  intros. inversion H2. edestruct in_app_or. apply H3.
    destruct H6.
      inversion H6. split. auto. left. apply config.eq_sym.
        eapply config.eq_trans; eauto. rewrite H10. rewrite config.diff_eq.
        apply config.eq_sym, config.union_nil. rewrite <- H8. auto.
      destruct H6; inversion H6. split. auto. right. apply config.eq_sym.
        eapply config.eq_trans; eauto. rewrite H10. rewrite config.diff_eq.
        apply config.eq_sym, config.union_nil. rewrite <- H8. auto.
    exfalso. edestruct in_app_or; eauto.
      rewrite in_map_iff in H7. destruct H7 as [?[]]. destruct x, p.
        inversion H7. rewrite <- H10 in H4. destruct H. edestruct Wf_T; eauto.
        destruct t0. contradiction. absurd (In (place.RLL t0) init).
          intro. destruct H13; auto. inversion H13.
          apply H1. apply H4. apply in_eq.
      edestruct in_app_or; eauto.
        rewrite in_map_iff in H8. destruct H8 as [?[]]. destruct x, p.
          inversion H8. rewrite <- H11 in H4. destruct H0.
          edestruct Wf_T; eauto. destruct t0. contradiction.
          absurd (In (place.RLR t0) init).
            intro. destruct H14; auto. inversion H14.
            apply H1. apply H4. apply in_eq.
        destruct H8.
          inversion H8. destruct H. destruct (petri.fin N1). contradiction.
            absurd (In (place.RLL t) init).
              intro. destruct H; auto. inversion H.
              apply H1. apply H4. rewrite <- H10. apply in_eq.
          destruct H8; inversion H8. destruct H0. destruct (petri.fin N2).
            contradiction. absurd (In (place.RLR t) init).
              intro. destruct H0; auto. inversion H0.
              apply H1. apply H4. rewrite <- H10. apply in_eq.
Qed.

(* there are no non-empty firings from init *)
Lemma Firing_init_Some N1 N2 c B c' :
  petri.Wf N1 -> petri.Wf N2 -> config.eq c init ->
  ~ petri.Firing (make N1 N2) c (Some B) c'.
Proof.
  do 4 intro. edestruct Firing_init. apply H. eauto. eauto. eauto. inversion H3.
Qed.

(* there are no firings to init *)
Lemma Firing_to_init N1 N2 c w c' :
  petri.Wf N1 -> petri.Wf N2 -> config.eq c' init ->
  ~ petri.Firing (make N1 N2) c w c'.
Proof.
  do 4 intro. inversion H2. edestruct in_app_or. apply H3.
    destruct H6.
      inversion H6. destruct H. destruct (petri.init N1). contradiction.
        absurd (In (place.RLL t) init).
          intro. destruct H; auto. inversion H.
          apply H1. apply H5. apply config.In_union. right. rewrite <- H10.
            apply in_eq.
      destruct H6; inversion H6. destruct H0. destruct (petri.init N2).
        contradiction. absurd (In (place.RLR t) init).
          intro. destruct H0; auto. inversion H0.
          apply H1. apply H5. apply config.In_union. right. rewrite <- H10.
            apply in_eq.
    edestruct in_app_or; eauto.
      rewrite in_map_iff in H7. destruct H7 as [?[]]. destruct x, p.
        inversion H7. rewrite <- H12 in H5. destruct H. edestruct Wf_T; eauto.
        destruct t. contradiction. absurd (In (place.RLL t) init).
          intro. destruct H13; auto. inversion H13.
          apply H1. apply H5. apply config.In_union. right. apply in_eq.
      edestruct in_app_or; eauto.
        rewrite in_map_iff in H8. destruct H8 as [?[]]. destruct x, p.
          inversion H8. rewrite <- H13 in H5. destruct H0.
          edestruct Wf_T; eauto. destruct t. contradiction.
          absurd (In (place.RLR t) init).
            intro. destruct H14; auto. inversion H14.
            apply H1. apply H5. apply config.In_union. right. apply in_eq.
        absurd (In (place.R (place.R place.P)) init).
          intro. destruct H9; auto. inversion H9.
          apply H1. apply H5. apply config.In_union. right. destruct H8.
            inversion H8. apply in_eq. destruct H8; inversion H8. apply in_eq.
Qed.

(* there are no firings from fin *)
Lemma Firing_fin N1 N2 c w c' :
  petri.Wf N1 -> petri.Wf N2 -> config.eq c fin ->
  ~ petri.Firing (make N1 N2) c w c'.
Proof.
  do 4 intro. inversion H2. edestruct in_app_or. apply H3.
    absurd (In (place.L place.P) fin).
      intro. destruct H7; auto. inversion H7.
      apply H1. apply H4. destruct H6. inversion H6. apply in_eq.
        destruct H6; inversion H6. apply in_eq.
    edestruct in_app_or; eauto.
      rewrite in_map_iff in H7. destruct H7 as [?[]]. destruct x, p.
        inversion H7. rewrite <- H10 in H4. destruct H. edestruct Wf_T; eauto.
        destruct t0. contradiction. absurd (In (place.RLL t0) fin).
          intro. destruct H13; auto. inversion H13.
          apply H1. apply H4. apply in_eq.
      edestruct in_app_or; eauto.
        rewrite in_map_iff in H8. destruct H8 as [?[]]. destruct x, p.
          inversion H8. rewrite <- H11 in H4. destruct H0.
          edestruct Wf_T; eauto. destruct t0. contradiction.
          absurd (In (place.RLR t0) fin).
            intro. destruct H14; auto. inversion H14.
            apply H1. apply H4. apply in_eq.
        destruct H8.
          inversion H8. destruct H. destruct (petri.fin N1). contradiction.
            absurd (In (place.RLL t) fin).
              intro. destruct H; auto. inversion H.
              apply H1. apply H4. rewrite <- H10. apply in_eq.
          destruct H8; inversion H8. destruct H0. destruct (petri.fin N2).
            contradiction. absurd (In (place.RLR t) fin).
              intro. destruct H0; auto. inversion H0.
              apply H1. apply H4. rewrite <- H10. apply in_eq.
Qed.

(* there are no non-empty firings to fin *)
Lemma Firing_Some_fin N1 N2 c B c' :
  petri.Wf N1 -> petri.Wf N2 -> config.eq c' fin ->
  ~ petri.Firing (make N1 N2) c (Some B) c'.
Proof.
  do 4 intro. inversion H2. edestruct in_app_or. apply H3.
    destruct H6. inversion H6. destruct H6; inversion H6.
    edestruct in_app_or; eauto.
      rewrite in_map_iff in H7. destruct H7 as [?[]]. destruct x, p.
        inversion H7. rewrite <- H12 in H5. destruct H. edestruct Wf_T; eauto.
        destruct t. contradiction. absurd (In (place.RLL t) fin).
          intro. destruct H13; auto. inversion H13.
          apply H1. apply H5. apply config.In_union. right. apply in_eq.
      edestruct in_app_or; eauto.
        rewrite in_map_iff in H8. destruct H8 as [?[]]. destruct x, p.
          inversion H8. rewrite <- H13 in H5. destruct H0.
          edestruct Wf_T; eauto. destruct t. contradiction.
          absurd (In (place.RLR t) fin).
            intro. destruct H14; auto. inversion H14.
            apply H1. apply H5. apply config.In_union. right. apply in_eq.
        destruct H8. inversion H8. destruct H8; inversion H8.
Qed.

(* any firing from the left petri net is in the selection petri net *)
Lemma Firing_RLL N1 N2 c w c' :
  petri.Firing N1 c w c' ->
  petri.Firing (make N1 N2) (map place.RLL c) w (map place.RLL c').
Proof.
  assert (forall p p' : place.t, place.RLL p = place.RLL p' -> p = p').
    intros. inversion H. auto.
  intro. destruct H0. econstructor.
    apply in_or_app. right. apply in_or_app. left. apply in_map_iff.
      exists (ci, w, co). split; simpl; eauto.
    apply config.le_map; auto.
    rewrite config.diff_map, config.union_map; auto. apply config.eq_map; auto.
Qed.

(* any firing from the right petri net is in the selection petri net *)
Lemma Firing_RLR N1 N2 c w c' :
  petri.Firing N2 c w c' ->
  petri.Firing (make N1 N2) (map place.RLR c) w (map place.RLR c').
Proof.
  assert (forall p p' : place.t, place.RLR p = place.RLR p' -> p = p').
    intros. inversion H. auto.
  intro. destruct H0. econstructor.
    apply in_or_app. right. apply in_or_app. right. apply in_or_app. left.
      apply in_map_iff. exists (ci, w, co). split; simpl; eauto.
    apply config.le_map; auto.
    rewrite config.diff_map, config.union_map; auto. apply config.eq_map; auto.
Qed.

(* any firing from a left configuration either
    ends on a left configuration and is a firing from the left petri net, or
    starts from the final configuration of the left petri net and ends on fin *)
Lemma Firing_RLL_ N1 N2 c w c' :
  petri.Wf N2 -> petri.Safe N1 -> petri.Reach N1 c ->
  petri.Firing (make N1 N2) (map place.RLL c) w c' ->
  c' = map place.RLL (config.RLL c') /\ petri.Firing N1 c w (config.RLL c') \/
  config.eq c (petri.fin N1) /\ config.eq c' fin.
Proof.
  assert (forall p p' : place.t, place.RLL p = place.RLL p' -> p = p').
    intros. inversion H. auto.
  intros. inversion H3. edestruct in_app_or. apply H4.
    absurd (In (place.L place.P) (map place.RLL c)).
      clear. intro. induction c; auto. inversion H; auto. inversion H0.
      apply H5. destruct H7. inversion H7. apply in_eq.
        destruct H7; inversion H7. apply in_eq.
    edestruct in_app_or; eauto.
      left. rewrite in_map_iff in H8. destruct H8 as [?[]]. destruct x, p.
        inversion H8. rewrite <- H11, <- H12, <- H13 in *.
        rewrite config.le_map in H5; auto.
        rewrite config.diff_map, config.union_map in H6; auto. split.
          eapply config.eq_map_RLL; eauto.
          erewrite config.eq_map_RLL, config.eq_map in H6; eauto.
            econstructor; eauto.
      right. edestruct in_app_or; eauto.
        rewrite in_map_iff in H9. destruct H9 as [?[]]. destruct x, p.
          inversion H9. rewrite <- H12 in H5. destruct H0.
          edestruct Wf_T; eauto. destruct t0. contradiction.
          absurd (In (place.RLR t0) (map place.RLL c)).
            clear. intro. induction c; auto. inversion H; auto. inversion H0.
            apply H5. apply in_eq.
        destruct H9.
          inversion H9. rewrite <- H11, <- H13 in *.
            rewrite config.le_map in H5; auto. rewrite config.diff_eq in H6.
              split. split; auto. apply config.eq_sym. auto.
              apply config.eq_map; auto. split; auto.
          destruct H9; inversion H9. destruct H0. destruct (petri.fin N2).
            contradiction. absurd (In (place.RLR t) (map place.RLL c)).
              clear. intro. induction c; auto. inversion H; auto. inversion H0.
              apply H5. rewrite <- H11. apply in_eq.
Qed.

(* any firing from a right configuration either
    ends on a right configuration and is a firing from the right petri net, or
    starts from the final configuration of the right petri net and ends on
    fin *)
Lemma Firing_RLR_ N1 N2 c w c' :
  petri.Wf N1 -> petri.Safe N2 -> petri.Reach N2 c ->
  petri.Firing (make N1 N2) (map place.RLR c) w c' ->
  c' = map place.RLR (config.RLR c') /\ petri.Firing N2 c w (config.RLR c') \/
  config.eq c (petri.fin N2) /\ config.eq c' fin.
Proof.
  assert (forall p p' : place.t, place.RLR p = place.RLR p' -> p = p').
    intros. inversion H. auto.
  intros. inversion H3. edestruct in_app_or. apply H4.
    absurd (In (place.L place.P) (map place.RLR c)).
      clear. intro. induction c; auto. inversion H; auto. inversion H0.
      apply H5. destruct H7. inversion H7. apply in_eq.
        destruct H7; inversion H7. apply in_eq.
    edestruct in_app_or; eauto.
      rewrite in_map_iff in H8. destruct H8 as [?[]]. destruct x, p.
        inversion H8. rewrite <- H11 in H5. destruct H0. edestruct Wf_T; eauto.
        destruct t0. contradiction.
        absurd (In (place.RLL t0) (map place.RLR c)).
          clear. intro. induction c; auto. inversion H; auto. inversion H0.
          apply H5. apply in_eq.
      edestruct in_app_or; eauto.
        left. rewrite in_map_iff in H9. destruct H9 as [?[]]. destruct x, p.
          inversion H9. rewrite <- H12, <- H13, <- H14 in *.
          rewrite config.le_map in H5; auto.
          rewrite config.diff_map, config.union_map in H6; auto. split.
            eapply config.eq_map_RLR; eauto.
            erewrite config.eq_map_RLR, config.eq_map in H6; eauto.
              econstructor; eauto.
        right. destruct H9.
          inversion H9. destruct H0. destruct (petri.fin N1). contradiction.
            absurd (In (place.RLL t) (map place.RLR c)).
              clear. intro. induction c; auto. inversion H; auto. inversion H0.
              apply H5. rewrite <- H11. apply in_eq.
          destruct H9; inversion H9. rewrite <- H11, <- H13 in *.
            rewrite config.le_map in H5; auto. rewrite config.diff_eq in H6.
              split. split; auto. apply config.eq_sym. auto.
              apply config.eq_map; auto. split; auto.
Qed.

(* any left-to-left firing is a firing from the left petri net *)
Lemma Firing_RLL_RLL N1 N2 c w c' :
  petri.Wf N2 ->
  petri.Firing (make N1 N2) (map place.RLL c) w (map place.RLL c') ->
  petri.Firing N1 c w c'.
Proof.
  assert (forall p p' : place.t, place.RLL p = place.RLL p' -> p = p').
    intros. inversion H. auto.
  intros. inversion H1. edestruct in_app_or. apply H2.
    absurd (In (place.L place.P) (map place.RLL c)).
      clear. intro. induction c; auto. inversion H; auto. inversion H0.
      apply H3. destruct H5. inversion H5. apply in_eq.
        destruct H5; inversion H5. apply in_eq.
    edestruct in_app_or; eauto.
      rewrite in_map_iff in H6. destruct H6 as [?[]]. destruct x, p.
        inversion H6. rewrite <- H9, <- H10, <- H11 in *.
        rewrite config.le_map in H3; auto.
        rewrite config.diff_map, config.union_map, config.eq_map in H4; auto.
        econstructor; eauto.
      edestruct in_app_or; eauto.
        rewrite in_map_iff in H7. destruct H7 as [?[]]. destruct x, p.
          inversion H7. rewrite <- H10 in H3. destruct H0.
          edestruct Wf_T; eauto. destruct t0. contradiction.
          absurd (In (place.RLR t0) (map place.RLL c)).
            clear. intro. induction c; auto. inversion H; auto. inversion H0.
            apply H3. apply in_eq.
        absurd (In (place.R (place.R place.P)) (map place.RLL c')).
          clear. intro. induction c'; auto. inversion H; auto. inversion H0.
          apply H4. apply config.In_union. right. destruct H7. inversion H7.
            apply in_eq. destruct H7; inversion H7. apply in_eq.
Qed.

(* any right-to-right firing is a firing from the right petri net *)
Lemma Firing_RLR_RLR N1 N2 c w c' :
  petri.Wf N1 ->
  petri.Firing (make N1 N2) (map place.RLR c) w (map place.RLR c') ->
  petri.Firing N2 c w c'.
Proof.
  assert (forall p p' : place.t, place.RLR p = place.RLR p' -> p = p').
    intros. inversion H. auto.
  intros. inversion H1. edestruct in_app_or. apply H2.
    absurd (In (place.L place.P) (map place.RLR c)).
      clear. intro. induction c; auto. inversion H; auto. inversion H0.
      apply H3. destruct H5. inversion H5. apply in_eq.
        destruct H5; inversion H5. apply in_eq.
    edestruct in_app_or; eauto.
      rewrite in_map_iff in H6. destruct H6 as [?[]]. destruct x, p.
        inversion H6. rewrite <- H9 in H3. destruct H0. edestruct Wf_T; eauto.
        destruct t0. contradiction.
        absurd (In (place.RLL t0) (map place.RLR c)).
          clear. intro. induction c; auto. inversion H; auto. inversion H0.
          apply H3. apply in_eq.
      edestruct in_app_or; eauto.
        rewrite in_map_iff in H7. destruct H7 as [?[]]. destruct x, p.
          inversion H7. rewrite <- H10, <- H11, <- H12 in *.
          rewrite config.le_map in H3; auto.
          rewrite config.diff_map, config.union_map, config.eq_map in H4; auto.
          econstructor; eauto.
        absurd (In (place.R (place.R place.P)) (map place.RLR c')).
          clear. intro. induction c'; auto. inversion H; auto. inversion H0.
          apply H4. apply config.In_union. right. destruct H7. inversion H7.
            apply in_eq. destruct H7; inversion H7. apply in_eq.
Qed.

(* there are no left-to-right firings *)
Lemma Firing_RLL_RLR N1 N2 c w c' :
  petri.Wf N1 -> petri.Wf N2 ->
  ~ petri.Firing (make N1 N2) (map place.RLL c) w (map place.RLR c').
Proof.
  do 3 intro. inversion H1. edestruct in_app_or. apply H2.
    absurd (In (place.L place.P) (map place.RLL c)).
      clear. intro. induction c; auto. inversion H; auto. inversion H0.
      apply H3. destruct H5. inversion H5. apply in_eq.
        destruct H5; inversion H5. apply in_eq.
    edestruct in_app_or; eauto.
      rewrite in_map_iff in H6. destruct H6 as [?[]]. destruct x, p.
        inversion H6. rewrite <- H11 in H4. destruct H. edestruct Wf_T; eauto.
        destruct t. contradiction. absurd (In (place.RLL t) (map place.RLR c')).
          clear. intro. induction c'; auto. inversion H; auto. inversion H0.
          apply H4. apply config.In_union. right. apply in_eq.
      edestruct in_app_or; eauto.
        rewrite in_map_iff in H7. destruct H7 as [?[]]. destruct x, p.
          inversion H7. rewrite <- H10 in H3. destruct H0.
          edestruct Wf_T; eauto. destruct t0. contradiction.
          absurd (In (place.RLR t0) (map place.RLL c)).
            clear. intro. induction c; auto. inversion H; auto. inversion H0.
            apply H3. apply in_eq.
        absurd (In (place.R (place.R place.P)) (map place.RLR c')).
          clear. intro. induction c'; auto. inversion H; auto. inversion H0.
          apply H4. apply config.In_union. right. destruct H7. inversion H7.
            apply in_eq. destruct H7; inversion H7. apply in_eq.
Qed.

(* there are no right-to-left firings *)
Lemma Firing_RLR_RLL N1 N2 c w c' :
  petri.Wf N1 -> petri.Wf N2 ->
  ~ petri.Firing (make N1 N2) (map place.RLR c) w (map place.RLL c').
Proof.
  do 3 intro. inversion H1. edestruct in_app_or. apply H2.
    absurd (In (place.L place.P) (map place.RLR c)).
      clear. intro. induction c; auto. inversion H; auto. inversion H0.
      apply H3. destruct H5. inversion H5. apply in_eq.
        destruct H5; inversion H5. apply in_eq.
    edestruct in_app_or; eauto.
      rewrite in_map_iff in H6. destruct H6 as [?[]]. destruct x, p.
        inversion H6. rewrite <- H9 in H3. destruct H. edestruct Wf_T; eauto.
        destruct t0. contradiction.
        absurd (In (place.RLL t0) (map place.RLR c)).
          clear. intro. induction c; auto. inversion H; auto. inversion H0.
          apply H3. apply in_eq.
      edestruct in_app_or; eauto.
        rewrite in_map_iff in H7. destruct H7 as [?[]]. destruct x, p.
          inversion H7. rewrite <- H12 in H4. destruct H0.
          edestruct Wf_T; eauto. destruct t. contradiction.
          absurd (In (place.RLR t) (map place.RLL c')).
            clear. intro. induction c'; auto. inversion H; auto. inversion H0.
            apply H4. apply config.In_union. right. apply in_eq.
        absurd (In (place.R (place.R place.P)) (map place.RLL c')).
          clear. intro. induction c'; auto. inversion H; auto. inversion H0.
          apply H4. apply config.In_union. right. destruct H7. inversion H7.
            apply in_eq. destruct H7; inversion H7. apply in_eq.
Qed.

(* any left-to-fin firing starts on the final configuration of the left petri
    net *)
Lemma Firing_RLL_fin N1 N2 c w c' :
  petri.Wf N2 -> petri.Safe N1 -> petri.Reach N1 c ->
  petri.Firing (make N1 N2) (map place.RLL c) w c' -> config.eq c' fin ->
  config.eq c (petri.fin N1).
Proof.
  intros. edestruct Firing_RLL_ as [[]|[]]; eauto.
    absurd (In (place.R (place.R place.P)) c').
      rewrite H4. clear. intro. induction (config.RLL c'); auto.
        inversion H; auto. inversion H0.
      apply H3. apply in_eq.
Qed.

(* any right-to-fin firing starts on the final configuration of the right petri
    net *)
Lemma Firing_RLR_fin N1 N2 c w c' :
  petri.Wf N1 -> petri.Safe N2 -> petri.Reach N2 c ->
  petri.Firing (make N1 N2) (map place.RLR c) w c' -> config.eq c' fin ->
  config.eq c (petri.fin N2).
Proof.
  intros. edestruct Firing_RLR_ as [[]|[]]; eauto.
    absurd (In (place.R (place.R place.P)) c').
      rewrite H4. clear. intro. induction (config.RLR c'); auto.
        inversion H; auto. inversion H0.
      apply H3. apply in_eq.
Qed.

(* ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~ *)

(* any empty firing chain from the left petri net is in the selection petri
    net *)
Lemma Efchain_RLL N1 N2 c l c' :
  petri.Efchain N1 c l c' ->
  petri.Efchain (make N1 N2) (map place.RLL c) (map (map place.RLL) l)
    (map place.RLL c').
Proof.
  intro. induction H. constructor. apply config.eq_map.
    intros. inversion H0. auto. auto.
  constructor. apply Firing_RLL; auto. auto.
Qed.

(* any empty firing chain from the right petri net is in the selection petri
    net *)
Lemma Efchain_RLR N1 N2 c l c' :
  petri.Efchain N2 c l c' ->
  petri.Efchain (make N1 N2) (map place.RLR c) (map (map place.RLR) l)
    (map place.RLR c').
Proof.
  intro. induction H. constructor. apply config.eq_map.
    intros. inversion H0. auto. auto.
  constructor. apply Firing_RLR; auto. auto.
Qed.

(* any empty firing chain from fin is a self-loop *)
Lemma Efchain_fin N1 N2 c l c' :
  petri.Wf N1 -> petri.Wf N2 -> config.eq c fin ->
  petri.Efchain (make N1 N2) c l c' -> config.eq c' fin.
Proof.
  intros. destruct H2. eapply config.eq_trans; eauto. apply config.eq_sym; auto.
  edestruct Firing_fin. apply H. eauto. eauto. eauto.
Qed.

(* any left-to-left empty firing chain is from the left petri net *)
Lemma Efchain_RLL_RLL N1 N2 c l c' :
  petri.Wf N1 -> petri.Wf N2 -> petri.Safe N1 ->
  petri.Reach N1 (config.RLL c) -> petri.Efchain (make N1 N2) c l c' ->
  c = map place.RLL (config.RLL c) -> c' = map place.RLL (config.RLL c') ->
  petri.Efchain N1 (config.RLL c) (map config.RLL l) (config.RLL c').
Proof.
  intros. induction H3. constructor. apply config.eq_RLL. auto.
  rewrite H4 in H3. edestruct Firing_RLL_ as [[]|[]]; eauto.
    simpl. constructor; auto. apply IHEfchain; auto. econstructor 2; eauto.
    absurd (In (place.R (place.R place.P)) c'').
      rewrite H5. clear. intro. induction (config.RLL c''); auto.
        inversion H; auto. inversion H0.
      eapply Efchain_fin. apply H. eauto. eauto. eauto. apply in_eq.
Qed.

(* any right-to-right empty firing chain is from the right petri net *)
Lemma Efchain_RLR_RLR N1 N2 c l c' :
  petri.Wf N1 -> petri.Wf N2 -> petri.Safe N2 ->
  petri.Reach N2 (config.RLR c) -> petri.Efchain (make N1 N2) c l c' ->
  c = map place.RLR (config.RLR c) -> c' = map place.RLR (config.RLR c') ->
  petri.Efchain N2 (config.RLR c) (map config.RLR l) (config.RLR c').
Proof.
  intros. induction H3. constructor. apply config.eq_RLR. auto.
  rewrite H4 in H3. edestruct Firing_RLR_ as [[]|[]]. apply H. eauto. eauto.
      eauto.
    simpl. constructor; auto. apply IHEfchain; auto. econstructor 2; eauto.
    absurd (In (place.R (place.R place.P)) c'').
      rewrite H5. clear. intro. induction (config.RLR c''); auto.
        inversion H; auto. inversion H0.
      eapply Efchain_fin. apply H. eauto. eauto. eauto. apply in_eq.
Qed.

(* there are no left-to-right empty firing chains *)
Lemma Efchain_RLL_RLR N1 N2 c l c' :
  petri.Wf N1 -> petri.Wf N2 -> petri.Safe N1 ->
  petri.Reach N1 (config.RLL c) -> petri.Efchain (make N1 N2) c l c' ->
  c = map place.RLL (config.RLL c) -> c' = map place.RLR (config.RLR c') ->
  False.
Proof.
  intros. induction H3.
    destruct (config.RLL c). eapply petri.Reach_nil. apply H. auto.
      absurd (In (place.RLL t) c').
        rewrite H5. clear. intro. induction (config.RLR c'); auto.
          inversion H; auto. inversion H0.
        apply H3. rewrite H4. apply in_eq.
    rewrite H4 in H3. edestruct Firing_RLL_ as [[]|[]]; eauto.
      apply IHEfchain; auto. econstructor 2; eauto.
      absurd (In (place.R (place.R place.P)) c'').
        rewrite H5. clear. intro. induction (config.RLR c''); auto.
          inversion H; auto. inversion H0.
        eapply Efchain_fin. apply H. eauto. eauto. eauto. apply in_eq.
Qed.

(* there are no right-to-left empty firing chains *)
Lemma Efchain_RLR_RLL N1 N2 c l c' :
  petri.Wf N1 -> petri.Wf N2 -> petri.Safe N2 ->
  petri.Reach N2 (config.RLR c) -> petri.Efchain (make N1 N2) c l c' ->
  c = map place.RLR (config.RLR c) -> c' = map place.RLL (config.RLL c') ->
  False.
Proof.
  intros. induction H3.
    destruct (config.RLR c). eapply petri.Reach_nil; eauto.
      absurd (In (place.RLR t) c').
        rewrite H5. clear. intro. induction (config.RLL c'); auto.
          inversion H; auto. inversion H0.
        apply H3. rewrite H4. apply in_eq.
    rewrite H4 in H3. edestruct Firing_RLR_ as [[]|[]]. apply H. eauto. eauto.
      eauto. apply IHEfchain; auto. econstructor 2; eauto.
      absurd (In (place.R (place.R place.P)) c'').
        rewrite H5. clear. intro. induction (config.RLL c''); auto.
          inversion H; auto. inversion H0.
        eapply Efchain_fin. apply H. eauto. eauto. eauto. apply in_eq.
Qed.

(* any left-to-fin empty firing chain includes one from the left petri net
    that ends on its final configuration *)
Lemma Efchain_RLL_fin N1 N2 c l c' :
  petri.Wf N1 -> petri.Wf N2 -> petri.Safe N1 ->
  petri.Reach N1 (config.RLL c) -> petri.Efchain (make N1 N2) c l c' ->
  c = map place.RLL (config.RLL c) -> config.eq c' fin ->
  petri.Efchain N1 (config.RLL c) (map config.RLL (removelast l))
    (petri.fin N1).
Proof.
  intros. destruct H3.
    absurd (In (place.R (place.R place.P)) c).
      rewrite H4. clear. intro. induction (config.RLL c); auto.
        inversion H; auto. inversion H0.
      apply H3. apply H5. apply in_eq.
    generalize dependent c. induction H6; intros.
      constructor. rewrite H6 in H4. eapply Firing_RLL_fin; eauto.
        eapply config.eq_trans; eauto.
      rewrite H7 in H4. edestruct Firing_RLL_ as [[]|[]]; eauto.
        constructor; auto. apply IHEfchain; eauto. econstructor 2; eauto.
        edestruct Firing_fin. apply H. eauto. eauto. eauto.
Qed.

(* any right-to-fin empty firing chain includes one from the right petri net
    that ends on its final configuration *)
Lemma Efchain_RLR_fin N1 N2 c l c' :
  petri.Wf N1 -> petri.Wf N2 -> petri.Safe N2 ->
  petri.Reach N2 (config.RLR c) -> petri.Efchain (make N1 N2) c l c' ->
  c = map place.RLR (config.RLR c) -> config.eq c' fin ->
  petri.Efchain N2 (config.RLR c) (map config.RLR (removelast l))
    (petri.fin N2).
Proof.
  intros. destruct H3.
    absurd (In (place.R (place.R place.P)) c).
      rewrite H4. clear. intro. induction (config.RLR c); auto.
        inversion H; auto. inversion H0.
      apply H3. apply H5. apply in_eq.
    generalize dependent c. induction H6; intros.
      constructor. rewrite H6 in H4. eapply Firing_RLR_fin; auto. apply H.
        eauto. eapply config.eq_trans; eauto.
      rewrite H7 in H4. edestruct Firing_RLR_ as [[]|[]]. apply H. eauto. eauto.
          eauto.
        constructor; auto. apply IHEfchain; eauto. econstructor 2; eauto.
        edestruct Firing_fin. apply H. eauto. eauto. eauto.
Qed.

(* any init-to-left empty firing chain includes one from the left petri net
    that starts on its initial configuration *)
Lemma Efchain_init_RLL N1 N2 c l c' :
  petri.Wf N1 -> petri.Wf N2 -> petri.Safe N1 -> petri.Safe N2 ->
  config.eq c init -> petri.Efchain (make N1 N2) c l c' ->
  c' = map place.RLL (config.RLL c') ->
  petri.Efchain N1 (petri.init N1) (map config.RLL (tl l)) (config.RLL c').
Proof.
  intros. destruct H4.
    absurd (In (place.L place.P) c').
      rewrite H5. clear. intro. induction (config.RLL c'); auto.
        inversion H; auto. inversion H0.
      apply H4. apply H3. apply in_eq.
    simpl. edestruct Firing_init as [_[]]. apply H. eauto. eauto. eauto.
      rewrite <- config.RLL_map_RLL at 1. eapply Efchain_RLL_RLL; auto; eauto.
        rewrite config.RLL_map_RLL. apply petri.Reach_init.
        eapply petri.Efchain_eq_l; eauto. rewrite config.RLL_map_RLL. auto.
      edestruct Efchain_RLR_RLL. apply H. eauto. auto.
        rewrite config.RLR_map_RLR. apply petri.Reach_init.
        eapply petri.Efchain_eq_l; eauto. rewrite config.RLR_map_RLR. auto.
        auto.
Qed.

(* any init-to-right empty firing chain includes one from the right petri net
    that starts on its initial configuration *)
Lemma Efchain_init_RLR N1 N2 c l c' :
  petri.Wf N1 -> petri.Wf N2 -> petri.Safe N1 -> petri.Safe N2 ->
  config.eq c init -> petri.Efchain (make N1 N2) c l c' ->
  c' = map place.RLR (config.RLR c') ->
  petri.Efchain N2 (petri.init N2) (map config.RLR (tl l)) (config.RLR c').
Proof.
  intros. destruct H4.
    absurd (In (place.L place.P) c').
      rewrite H5. clear. intro. induction (config.RLR c'); auto.
        inversion H; auto. inversion H0.
      apply H4. apply H3. apply in_eq.
    simpl. edestruct Firing_init as [_[]]. apply H. eauto. eauto. eauto.
      edestruct Efchain_RLL_RLR. apply H. eauto. auto.
        rewrite config.RLL_map_RLL. apply petri.Reach_init.
        eapply petri.Efchain_eq_l; eauto. rewrite config.RLL_map_RLL. auto.
        auto.
      rewrite <- config.RLR_map_RLR at 1. eapply Efchain_RLR_RLR; auto. apply H.
        rewrite config.RLR_map_RLR. apply petri.Reach_init.
        eapply petri.Efchain_eq_l; eauto. rewrite config.RLR_map_RLR. auto.
Qed.

(* any init-to-fin empty firing chain includes either
    one from the left petri net that starts on its initial configuration and
      ends on its final, or
    one from the right petri net that starts on its initial configuration and
      ends on its final *)
Lemma Efchain_init_fin N1 N2 c l c' :
  petri.Wf N1 -> petri.Wf N2 -> petri.Safe N1 -> petri.Safe N2 ->
  config.eq c init -> petri.Efchain (make N1 N2) c l c' -> config.eq c' fin ->
  petri.Efchain N1 (petri.init N1) (map config.RLL (removelast (tl l)))
    (petri.fin N1) \/
  petri.Efchain N2 (petri.init N2) (map config.RLR (removelast (tl l)))
    (petri.fin N2).
Proof.
  intros. destruct H4.
    edestruct le_L_fin. eapply config.le_trans. apply H3.
      eapply config.le_trans. apply H4. apply H5.
    simpl. edestruct Firing_init as [_[]]. apply H. eauto. eauto. eauto.
      left. rewrite <- config.RLL_map_RLL at 1. eapply Efchain_RLL_fin; eauto.
        rewrite config.RLL_map_RLL. apply petri.Reach_init.
        eapply petri.Efchain_eq_l; eauto. rewrite config.RLL_map_RLL. auto.
      right. rewrite <- config.RLR_map_RLR at 1. eapply Efchain_RLR_fin; auto.
        apply H. rewrite config.RLR_map_RLR. apply petri.Reach_init.
        eapply petri.Efchain_eq_l; eauto. rewrite config.RLR_map_RLR. auto.
        auto.
Qed.

(* any empty firing chain from a left configuration ends on a left
    configuration or fin and is an empty firing chain from the left petri net *)
Lemma Efchain_RLL_ N1 N2 c l c' :
  petri.Wf N1 -> petri.Wf N2 -> petri.Safe N1 ->
  petri.Reach N1 (config.RLL c) -> petri.Efchain (make N1 N2) c l c' ->
  c = map place.RLL (config.RLL c) ->
  c' = map place.RLL (config.RLL c') /\
    petri.Efchain N1 (config.RLL c) (map config.RLL l) (config.RLL c') \/
  config.eq c' fin /\
    petri.Efchain N1 (config.RLL c) (map config.RLL (removelast l))
      (petri.fin N1).
Proof.
  intros. induction H3.
    left. split. eapply config.eq_map_RLL. rewrite <- H4. auto.
      constructor. apply config.eq_RLL. auto.
    edestruct Firing_RLL_ as [[]|[]]; eauto. rewrite H4 in H3. eauto.
      destruct IHEfchain as [[]|[]]; auto. econstructor 2; eauto.
        left. split; auto. constructor; auto.
        right. split; auto. eapply Efchain_RLL_fin; eauto. constructor; auto.
      right. assert (config.eq c'' fin). eapply Efchain_fin. apply H. eauto.
        eauto. eauto. split; auto. eapply Efchain_RLL_fin; eauto.
        constructor; auto.
Qed.

(* any empty firing chain from a right configuration ends on a right
    configuration or fin and is an empty firing chain from the right petri
    net *)
Lemma Efchain_RLR_ N1 N2 c l c' :
  petri.Wf N1 -> petri.Wf N2 -> petri.Safe N2 ->
  petri.Reach N2 (config.RLR c) -> petri.Efchain (make N1 N2) c l c' ->
  c = map place.RLR (config.RLR c) ->
  c' = map place.RLR (config.RLR c') /\
    petri.Efchain N2 (config.RLR c) (map config.RLR l) (config.RLR c') \/
  config.eq c' fin /\
    petri.Efchain N2 (config.RLR c) (map config.RLR (removelast l))
      (petri.fin N2).
Proof.
  intros. induction H3.
    left. split. eapply config.eq_map_RLR. rewrite <- H4. auto.
      constructor. apply config.eq_RLR. auto.
    edestruct Firing_RLR_ as [[]|[]]. apply H. eauto. eauto. rewrite H4 in H3.
        eauto.
      destruct IHEfchain as [[]|[]]; auto. econstructor 2; eauto.
        left. split; auto. constructor; auto.
        right. split; auto. eapply Efchain_RLR_fin; auto. apply H.
          constructor; eauto. auto.
      right. assert (config.eq c'' fin). eapply Efchain_fin. apply H. eauto.
        eauto. eauto. split; auto. eapply Efchain_RLR_fin; auto. apply H.
        constructor; eauto. auto.
Qed.

(* ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~ *)

CoFixpoint path_l fin1 pi : path.t :=
  match pi with
  | path.fin l => path.fin (map config.RLL (removelast l))
  | path.hop l c pi' =>
    if config.dec c fin
    then path.hop (map config.RLL (removelast l)) fin1 (path_l fin1 pi')
    else path.hop (map config.RLL l) (config.RLL c) (path_l fin1 pi')
  end.

Lemma path_l_fin fin1 l :
  path_l fin1 (path.fin l) = path.fin (map config.RLL (removelast l)).
Proof.
  rewrite path.match_ at 1. auto.
Qed.

Lemma path_l_hop fin1 l c pi :
  path_l fin1 (path.hop l c pi) =
    if config.dec c fin
    then path.hop (map config.RLL (removelast l)) fin1 (path_l fin1 pi)
    else path.hop (map config.RLL l) (config.RLL c) (path_l fin1 pi).
Proof.
  rewrite path.match_ at 1. simpl. destruct config.dec; auto.
Qed.

(* traces generated from a left configuration are generated by the left petri
    net *)
Lemma Gen_RLL N1 N2 : petri.Wf N1 -> petri.Wf N2 -> petri.Safe N1 ->
  forall c pi ot, petri.Reach N1 (config.RLL c) ->
  petri.Gen (make N1 N2) c pi ot -> c = map place.RLL (config.RLL c) ->
  petri.Gen N1 (config.RLL c) (path_l (petri.fin N1) pi) ot.
Proof.
  do 3 intro. cofix F. intros. destruct H3.
    rewrite path_l_fin. constructor. eapply Efchain_RLL_fin; eauto.
      apply config.eq_refl.
    rewrite path_l_hop. edestruct Efchain_RLL_ as [[]|[]]. apply H. eauto. auto.
        eauto. eauto. auto.
      destruct config.dec.
        absurd (In (place.R (place.R place.P)) c'). rewrite H6. clear. intro.
          induction (config.RLL c'); auto. inversion H; auto. inversion H0.
          apply e. apply in_eq.
        assert (petri.Reach N1 (config.RLL c')).
          eapply petri.Reach_Efchain; eauto. constructor; auto.
      destruct config.dec; try contradiction. constructor; auto.
          generalize dependent ot. generalize dependent pi. clear H3 e.
          generalize dependent c'. cofix F'. intros. destruct H5.
        rewrite path_l_fin. constructor. destruct H3.
          constructor. apply config.eq_refl.
          edestruct Firing_fin. apply H. eauto. eauto. eauto.
        rewrite path_l_hop. destruct config.dec.
          constructor; eauto. destruct H3. constructor. apply config.eq_refl.
            edestruct Firing_fin. apply H. eauto. apply H6. eauto.
          destruct n. eapply Efchain_fin. apply H. eauto. eauto. eauto.
        rewrite path_l_hop. edestruct Firing_fin. apply H. eauto.
          eapply Efchain_fin. apply H. eauto. eauto. eauto. eauto.
    rewrite path_l_hop. destruct config.dec.
      edestruct Firing_Some_fin. apply H. eauto. eauto. eauto.
      edestruct Efchain_RLL_ as [[]|[]]. apply H. eauto. auto. eauto. eauto.
          auto.
        assert (petri.Reach N1 (config.RLL c')).
            eapply petri.Reach_Efchain; eauto. rewrite H7 in H5.
            edestruct Firing_RLL_ as [[]|[]]; eauto.
          econstructor; eauto. apply F; auto. econstructor 2; eauto.
          edestruct Firing_Some_fin. apply H. eauto. eauto. eauto.
        edestruct Firing_fin. apply H. eauto. eauto. eauto.
Qed.

(* ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~ *)

CoFixpoint path_r fin2 pi : path.t :=
  match pi with
  | path.fin l => path.fin (map config.RLR (removelast l))
  | path.hop l c pi' =>
    if config.dec c fin
    then path.hop (map config.RLR (removelast l)) fin2 (path_r fin2 pi')
    else path.hop (map config.RLR l) (config.RLR c) (path_r fin2 pi')
  end.

Lemma path_r_fin fin2 l :
  path_r fin2 (path.fin l) = path.fin (map config.RLR (removelast l)).
Proof.
  rewrite path.match_ at 1. auto.
Qed.

Lemma path_r_hop fin2 l c pi :
  path_r fin2 (path.hop l c pi) =
    if config.dec c fin
    then path.hop (map config.RLR (removelast l)) fin2 (path_r fin2 pi)
    else path.hop (map config.RLR l) (config.RLR c) (path_r fin2 pi).
Proof.
  rewrite path.match_ at 1. simpl. destruct config.dec; auto.
Qed.

(* traces generated from a right configuration are generated by the right petri
    net *)
Lemma Gen_RLR N1 N2 : petri.Wf N1 -> petri.Wf N2 -> petri.Safe N2 ->
  forall c pi ot, petri.Reach N2 (config.RLR c) ->
  petri.Gen (make N1 N2) c pi ot -> c = map place.RLR (config.RLR c) ->
  petri.Gen N2 (config.RLR c) (path_r (petri.fin N2) pi) ot.
Proof.
  do 3 intro. cofix F. intros. destruct H3.
    rewrite path_r_fin. constructor. eapply Efchain_RLR_fin. apply H. auto.
      auto. auto. eauto. auto. apply config.eq_refl.
    rewrite path_r_hop. edestruct Efchain_RLR_ as [[]|[]]. apply H. eauto. auto.
        eauto. eauto. auto.
      destruct config.dec.
        absurd (In (place.R (place.R place.P)) c'). rewrite H6. clear. intro.
          induction (config.RLR c'); auto. inversion H; auto. inversion H0.
          apply e. apply in_eq.
        assert (petri.Reach N2 (config.RLR c')).
          eapply petri.Reach_Efchain; eauto. constructor; auto.
      destruct config.dec; try contradiction. constructor; auto.
          generalize dependent ot. generalize dependent pi. clear H3 e.
          generalize dependent c'. cofix F'. intros. destruct H5.
        rewrite path_r_fin. constructor. destruct H3.
          constructor. apply config.eq_refl.
          edestruct Firing_fin. apply H. eauto. eauto. eauto.
        rewrite path_r_hop. destruct config.dec.
          constructor; eauto. destruct H3. constructor. apply config.eq_refl.
            edestruct Firing_fin. apply H. eauto. apply H6. eauto.
          destruct n. eapply Efchain_fin. apply H. eauto. eauto. eauto.
        rewrite path_r_hop. edestruct Firing_fin. apply H. eauto.
          eapply Efchain_fin. apply H. eauto. eauto. eauto. eauto.
    rewrite path_r_hop. destruct config.dec.
      edestruct Firing_Some_fin. apply H. eauto. eauto. eauto.
      edestruct Efchain_RLR_ as [[]|[]]. apply H. eauto. auto. eauto. eauto.
          auto.
        assert (petri.Reach N2 (config.RLR c')).
            eapply petri.Reach_Efchain; eauto. rewrite H7 in H5.
            edestruct Firing_RLR_ as [[]|[]]. apply H. eauto. eauto. eauto.
          econstructor; eauto. apply F; auto. econstructor 2; eauto.
          edestruct Firing_Some_fin. apply H. eauto. eauto. eauto.
        edestruct Firing_fin. apply H. eauto. eauto. eauto.
Qed.

(* ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~ *)

(* traces generated by the left petri net are generated from a left
    configuration *)
Lemma Gen'_RLL N1 N2 :
  forall c ot, petri.Gen' N1 c ot ->
  petri.Gen' (make N1 N2) (map place.RLL c) ot.
Proof.
  cofix F. intros. destruct H; econstructor; eauto.
    eapply petri.Efchain_trans. apply Efchain_RLL; eauto.
      eapply petri.Efchain_eq_eq. repeat (apply in_or_app; right). apply in_eq.
      apply config.eq_refl. apply config.eq_refl.
    apply Efchain_RLL; eauto.
    apply Efchain_RLL; eauto. apply Firing_RLL; eauto.
Qed.

(* traces generated by the right petri net are generated from a right
    configuration *)
Lemma Gen'_RLR N1 N2 :
  forall c ot, petri.Gen' N2 c ot ->
  petri.Gen' (make N1 N2) (map place.RLR c) ot.
Proof.
  cofix F. intros. destruct H; econstructor; eauto.
    eapply petri.Efchain_trans. apply Efchain_RLR; eauto.
      eapply petri.Efchain_eq_eq. repeat (apply in_or_app; right). right.
      apply in_eq. apply config.eq_refl. apply config.eq_refl.
    apply Efchain_RLR; eauto.
    apply Efchain_RLR; eauto. apply Firing_RLR; eauto.
Qed.

(* ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~ *)

(* any reachable configuration is either init,
    a reachable configuration from the left petri net,
    a reachable configuration from the right petri net, or fin *)
Lemma Reach N1 N2 c :
  petri.Wf N1 -> petri.Wf N2 -> petri.Safe N1 -> petri.Safe N2 ->
  petri.Reach (make N1 N2) c ->
  config.eq c init \/
  c = map place.RLL (config.RLL c) /\ petri.Reach N1 (config.RLL c) \/
  c = map place.RLR (config.RLR c) /\ petri.Reach N2 (config.RLR c) \/
  config.eq c fin.
Proof.
  intros. induction H3. left. auto. right. destruct IHReach as [|[[]|[[]|]]].
    edestruct Firing_init as [_[]]. apply H. eauto. eauto. eauto.
      left. split. eapply config.eq_map_RLL. apply config.eq_sym. eauto.
        eapply petri.Reach_eq. apply petri.Reach_init.
        erewrite config.eq_map_RLL in H6 at 1. apply config.eq_sym.
        eapply config.eq_map; eauto. intros. inversion H7. auto.
        apply config.eq_sym. eauto.
      right. left. split. eapply config.eq_map_RLR. apply config.eq_sym. eauto.
        eapply petri.Reach_eq. apply petri.Reach_init.
        erewrite config.eq_map_RLR in H6 at 1. apply config.eq_sym.
        eapply config.eq_map; eauto. intros. inversion H7. auto.
        apply config.eq_sym. eauto.
    rewrite H5 in H4. edestruct Firing_RLL_ as [[]|[]]. eauto. apply H1. eauto.
      eauto. left. split; auto. econstructor 2; eauto. eauto.
    rewrite H5 in H4. edestruct Firing_RLR_ as [[]|[]]. apply H. eauto. eauto.
      eauto. right. left. split; auto. econstructor 2; eauto. eauto.
    edestruct Firing_fin. apply H. eauto. eauto. eauto.
Qed.

(* the first element of any reachable configuration is not the P place *)
Lemma Reach_P N1 N2 :
  petri.Wf N1 -> petri.Wf N2 -> petri.Safe N1 -> petri.Safe N2 ->
  forall c, ~ petri.Reach (make N1 N2) (place.P :: c).
Proof.
  do 6 intro.
  destruct (Reach H H0 H1 H2 H3) as [|[[]|[[]|]]]; eauto; simpl in H4.
    eapply le_P_init, H4.
    destruct (config.RLL c); inversion H4.
    destruct (config.RLR c); inversion H4.
    eapply le_P_fin, H4.
Qed.

(* if the first element of a reachable configuration is an L place,
    then it is init *)
Lemma Reach_L N1 N2 :
  petri.Wf N1 -> petri.Wf N2 -> petri.Safe N1 -> petri.Safe N2 ->
  forall p c, petri.Reach (make N1 N2) (place.L p :: c) ->
  config.eq (place.L p :: c) init.
Proof.
  intros.
  destruct (Reach H H0 H1 H2 H3) as [|[[]|[[]|]]]; eauto; simpl in H4.
    destruct (config.RLL c); inversion H4.
    destruct (config.RLR c); inversion H4.
    edestruct le_L_fin. apply H4.
Qed.

(* the first element of any reachable configuration is not the R P place *)
Lemma Reach_RP N1 N2 :
  petri.Wf N1 -> petri.Wf N2 -> petri.Safe N1 -> petri.Safe N2 ->
  forall c, ~ petri.Reach (make N1 N2) (place.R place.P :: c).
Proof.
  do 6 intro.
  destruct (Reach H H0 H1 H2 H3) as [|[[]|[[]|]]]; eauto; simpl in H4.
    eapply le_R_init, H4.
    destruct (config.RLL c); inversion H4.
    destruct (config.RLR c); inversion H4.
    eapply le_RP_fin, H4.
Qed.

(* the first element of any reachable configuration is not the R L P place *)
Lemma Reach_RLP N1 N2 :
  petri.Wf N1 -> petri.Wf N2 -> petri.Safe N1 -> petri.Safe N2 ->
  forall c, ~ petri.Reach (make N1 N2) (place.R (place.L place.P) :: c).
Proof.
  do 6 intro.
  destruct (Reach H H0 H1 H2 H3) as [|[[]|[[]|]]]; eauto; simpl in H4.
    eapply le_R_init, H4.
    destruct (config.RLL c); inversion H4.
    destruct (config.RLR c); inversion H4.
    eapply le_RL_fin, H4.
Qed.

(* if the first element of a reachable configuration is an R L L place,
    then it is a reachable configuration from the left petri net *)
Lemma Reach_RLL N1 N2 :
  petri.Wf N1 -> petri.Wf N2 -> petri.Safe N1 -> petri.Safe N2 ->
  forall p c, petri.Reach (make N1 N2) (place.R (place.L (place.L p)) :: c) ->
  place.R (place.L (place.L p)) :: c =
    map place.RLL (config.RLL (place.R (place.L (place.L p)) :: c)) /\
  petri.Reach N1 (config.RLL (place.R (place.L (place.L p)) :: c)).
Proof.
  intros.
  destruct (Reach H H0 H1 H2 H3) as [|[[]|[[]|]]]; eauto; simpl in H4.
    edestruct le_R_init. apply H4.
    destruct (config.RLR c); inversion H4.
    edestruct le_RL_fin. apply H4.
Qed.

(* if the first element of a reachable configuration is an R L R place,
    then it is a reachable configuration from the right petri net *)
Lemma Reach_RLR N1 N2 :
  petri.Wf N1 -> petri.Wf N2 -> petri.Safe N1 -> petri.Safe N2 ->
  forall p c, petri.Reach (make N1 N2) (place.R (place.L (place.R p)) :: c) ->
  place.R (place.L (place.R p)) :: c =
    map place.RLR (config.RLR (place.R (place.L (place.R p)) :: c)) /\
  petri.Reach N2 (config.RLR (place.R (place.L (place.R p)) :: c)).
Proof.
  intros.
  destruct (Reach H H0 H1 H2 H3) as [|[[]|[[]|]]]; eauto; simpl in H4.
    edestruct le_R_init. apply H4.
    destruct (config.RLL c); inversion H4.
    edestruct le_RL_fin. apply H4.
Qed.

(* if the first element of a reachable configuration is an R R place,
    then it is fin *)
Lemma Reach_RR N1 N2 :
  petri.Wf N1 -> petri.Wf N2 -> petri.Safe N1 -> petri.Safe N2 ->
  forall p c, petri.Reach (make N1 N2) (place.R (place.R p) :: c) ->
  config.eq (place.R (place.R p) :: c) fin.
Proof.
  intros.
  destruct (Reach H H0 H1 H2 H3) as [|[[]|[[]|]]]; eauto; simpl in H4.
    edestruct le_R_init. apply H4.
    destruct (config.RLL c); inversion H4.
    destruct (config.RLR c); inversion H4.
Qed.

(* any reachable non-empty configuration is decidably either init,
    a reachable configuration from the left petri net,
    a reachable configuration from the right petri net, or fin *)
Definition Reach_cons N1 N2 :
  petri.Wf N1 -> petri.Wf N2 -> petri.Safe N1 -> petri.Safe N2 ->
  forall p c, petri.Reach (make N1 N2) (p :: c) ->
  {config.eq (p :: c) init} +
  {p :: c = map place.RLL (config.RLL (p :: c)) /\
    petri.Reach N1 (config.RLL (p :: c))} +
  {p :: c = map place.RLR (config.RLR (p :: c)) /\
    petri.Reach N2 (config.RLR (p :: c))} +
  {config.eq (p :: c) fin}.
Proof.
  intros. destruct p. edestruct (Reach_P H H0 H1); eauto.
    left. left. left. eapply (Reach_L H H0 H1); eauto.
    destruct p. edestruct (Reach_RP H H0 H1); eauto.
      destruct p. edestruct (Reach_RLP H H0 H1); eauto.
        left. left. right. eapply (Reach_RLL H H0 H1); eauto.
        left. right. eapply (Reach_RLR H H0 H1); eauto.
      right. eapply (Reach_RR H H0 H1); eauto.
Defined.

(* any reachable configuration from the left petri net is reachable in the
    selection petri net *)
Lemma Reach_RLL' N1 N2 c :
  petri.Reach N1 c -> petri.Reach (make N1 N2) (map place.RLL c).
Proof.
  intro. induction H; econstructor 2; eauto. apply petri.Reach_init.
    eapply petri.Firing_eq_eq. apply in_or_app. left. apply in_eq.
      apply config.eq_refl.
      apply config.eq_map; auto. intros. inversion H0. auto.
    apply Firing_RLL. eauto.
Qed.

(* any reachable configuration from the right petri net is reachable in the
    selection petri net *)
Lemma Reach_RLR' N1 N2 c :
  petri.Reach N2 c -> petri.Reach (make N1 N2) (map place.RLR c).
Proof.
  intro. induction H; econstructor 2; eauto. apply petri.Reach_init.
    eapply petri.Firing_eq_eq. apply in_or_app. left. right. apply in_eq.
      apply config.eq_refl.
      apply config.eq_map; auto. intros. inversion H0. auto.
    apply Firing_RLR. eauto.
Qed.

(* the selection petri net is safe *)
Lemma Safe N1 N2 : petri.Wf N1 -> petri.Wf N2 ->
  petri.Safe N1 -> petri.Safe N2 -> petri.Safe (make N1 N2).
Proof.
  do 7 intro. assert (In (place.R (place.R place.P)) c). apply H4. apply in_eq.
  destruct (Reach H H0 H1 H2 H3) as [|[[]|[[]|]]].
    exfalso. eapply le_R_init. eapply config.le_trans. apply H4. apply H6.
    exfalso. rewrite H6 in H5. clear - H5. induction (config.RLL c); auto.
      inversion H5; auto. inversion H.
    exfalso. rewrite H6 in H5. clear - H5. induction (config.RLR c); auto.
      inversion H5; auto. inversion H.
    apply H6.
Qed.

(* ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~ *)

(* Lang soundness *)
Lemma Lang_sound N1 N2 :
  petri.Wf N1 -> petri.Wf N2 -> petri.Safe N1 -> petri.Safe N2 ->
  slang.le (petri.Lang (make N1 N2))
    (slang.union (petri.Lang N1) (petri.Lang N2)).
Proof.
  do 4 intro. unfold slang.le. cofix F. intros. destruct H3. destruct H3.
    edestruct Efchain_init_fin. apply H. eauto. auto. auto.
        apply config.eq_refl. eauto. apply config.eq_refl.
      constructor 2. eexists. constructor. eauto.
      constructor 3. eexists. constructor. eauto.
    constructor. apply F. eexists. eapply petri.Gen_Efchain; eauto.
    assert (petri.Reach (make N1 N2) c').
        eapply petri.Reach_Efchain; eauto. apply petri.Reach_init.
        edestruct (Reach H H0 H1 H2 H6) as [|[[]|[[]|]]].
      edestruct Firing_init_Some. apply H. eauto. eauto. eauto.
      constructor 2. eexists. rewrite <- config.RLL_map_RLL at 1.
        eapply Gen_RLL; eauto. rewrite config.RLL_map_RLL.
        apply petri.Reach_init. econstructor; eauto. rewrite H7.
        apply Efchain_RLL. eapply Efchain_init_RLL; eauto. apply config.eq_refl.
        rewrite config.RLL_map_RLL. auto.
      constructor 3. eexists. rewrite <- config.RLR_map_RLR at 1.
        eapply Gen_RLR. apply H. auto. auto. rewrite config.RLR_map_RLR.
        apply petri.Reach_init. econstructor; eauto. rewrite H7.
        apply Efchain_RLR. eapply Efchain_init_RLR. apply H. auto. auto. auto.
        apply config.eq_refl. eauto. auto. rewrite config.RLR_map_RLR. auto.
      edestruct Firing_fin. apply H. eauto. eauto. eauto.
Qed.

(* Lang completeness *)
Lemma Lang_complete N1 N2 :
  slang.le (slang.union (petri.Lang N1) (petri.Lang N2))
    (petri.Lang (make N1 N2)).
Proof.
  do 2 intro. apply petri.Gen_Gen'. generalize dependent ot. cofix F. intros.
  destruct H. econstructor; eauto. constructor. apply config.eq_refl.
    eapply petri.Gen'_Efchain. eapply petri.Efchain_eq_eq. apply in_or_app.
      left. apply in_eq. apply config.eq_refl. apply config.eq_refl.
      apply Gen'_RLL. destruct H. eapply petri.Gen'_Gen. eauto.
    eapply petri.Gen'_Efchain. eapply petri.Efchain_eq_eq. apply in_or_app.
      left. right. apply in_eq. apply config.eq_refl. apply config.eq_refl.
      apply Gen'_RLR. destruct H. eapply petri.Gen'_Gen. eauto.
Qed.

(* ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~ *)

(* ef correctness *)
Definition EfcFin N1 N2 ef1 ef2 init1 init2 :
  petri.Wf N1 -> petri.Wf N2 -> petri.Safe N1 -> petri.Safe N2 ->
  petri.EfcFin N1 ef1 -> petri.EfcFin N2 ef2 ->
  petri.init N1 = init1 -> petri.init N2 = init2 ->
  petri.EfcFin (make N1 N2) (ef ef1 ef2 init1 init2).
Proof.
  unfold ef. intros. destruct X, X0. split; repeat intro.
    destruct c. edestruct petri.Reach_nil. eapply Wf. apply H. eauto. auto.
      destruct t. edestruct (Reach_P H H0 H1); eauto.
        case_eq (ef1 init1); intro.
          edestruct EF_sound; eauto. rewrite <- H3. apply petri.Reach_init.
            exists ([map place.RLL init1] ++ map (map place.RLL) x ++ [fin]).
            eapply petri.Efchain_trans. eapply petri.Efchain_eq_eq.
            apply in_or_app. left. apply in_eq. eapply (Reach_L H H0 H1); eauto.
            rewrite H3. apply config.eq_refl. eapply petri.Efchain_trans.
            apply Efchain_RLL; eauto. eapply petri.Efchain_eq_eq.
            repeat (apply in_or_app; right). apply in_eq. apply config.eq_refl.
            apply config.eq_refl.
          rewrite H7 in H6. edestruct EF_sound0; eauto. rewrite <- H4.
            apply petri.Reach_init.
            exists ([map place.RLR init2] ++ map (map place.RLR) x ++ [fin]).
            eapply petri.Efchain_trans. eapply petri.Efchain_eq_eq.
            apply in_or_app. left. right. apply in_eq.
            eapply (Reach_L H H0 H1); eauto. rewrite H4. apply config.eq_refl.
            eapply petri.Efchain_trans. apply Efchain_RLR; eauto.
            eapply petri.Efchain_eq_eq. repeat (apply in_or_app; right). right.
            apply in_eq. apply config.eq_refl. apply config.eq_refl.
        destruct t. edestruct (Reach_RP H H0 H1); eauto.
          destruct t. edestruct (Reach_RLP H H0 H1); eauto.
            edestruct (Reach_RLL H H0 H1); eauto. rewrite H7.
              edestruct EF_sound; eauto.
              exists (map (map place.RLL) x ++ [fin]).
              eapply petri.Efchain_trans. apply Efchain_RLL; eauto.
              eapply petri.Efchain_eq_eq. repeat (apply in_or_app; right).
              apply in_eq. apply config.eq_refl. apply config.eq_refl.
            edestruct (Reach_RLR H H0 H1); eauto. rewrite H7.
              edestruct EF_sound0; eauto.
              exists (map (map place.RLR) x ++ [fin]).
              eapply petri.Efchain_trans. apply Efchain_RLR; eauto.
              eapply petri.Efchain_eq_eq. repeat (apply in_or_app; right).
              right. apply in_eq. apply config.eq_refl. apply config.eq_refl.
          exists []. constructor. eapply (Reach_RR H H0 H1); eauto.
    destruct c; auto. destruct t; auto.
      rewrite <- H3, <- H4. edestruct Efchain_init_fin. apply H. eauto. auto.
          auto. eapply (Reach_L H H0 H1); eauto. eauto. apply config.eq_refl.
        erewrite EF_complete; eauto. apply petri.Reach_init.
        destruct (ef1 (petri.init N1)); auto. erewrite EF_complete0; eauto.
          apply petri.Reach_init.
      destruct t; auto. destruct t; auto.
        edestruct (Reach_RLL H H0 H1); eauto. eapply EF_complete; eauto.
          eapply Efchain_RLL_fin; eauto. apply config.eq_refl.
        edestruct (Reach_RLR H H0 H1); eauto. eapply EF_complete0; eauto.
          eapply Efchain_RLR_fin; auto. apply H. eauto. apply config.eq_refl.
Defined.

(* nb soundness *)
Definition nb_sound N1 N2 nb1 nb2 init1 init2 :
  petri.Wf N1 -> petri.Wf N2 -> petri.Safe N1 -> petri.Safe N2 ->
  petri.nb_sound N1 nb1 -> petri.nb_sound N2 nb2 -> petri.init N1 = init1 ->
  petri.init N2 = init2 -> petri.nb_sound (make N1 N2) (nb nb1 nb2 init1 init2).
Proof.
  unfold nb. repeat intro. destruct c. inversion H6. destruct t0. inversion H6.
    case_eq (nb1 init1 sig sync (fun B c1' => F B (map place.RLL c1')) n);
        intros; rewrite H7 in H6.
      edestruct X as [?[?[?[?[?[?[?[?[]]]]]]]]]; eauto. rewrite <- H3.
          apply petri.Reach_init. destruct p. destruct o.
        case_eq (nb2 init2 sig sync (fun B c2' => F B (map place.RLR c2')) n0);
            intros; rewrite H13 in H6; rewrite <- H6.
          edestruct X0 as [?[?[?[?[?[?[?[?[]]]]]]]]]; eauto. rewrite <- H4.
            apply petri.Reach_init.
            exists (map place.RLR (petri.init N2) :: map (map place.RLR) x4),
            (map place.RLR x5), x6, (map place.RLR x7), x8. split.
            constructor. eapply petri.Firing_eq_eq. apply in_or_app. left.
            right. apply in_eq. eapply (Reach_L H H0 H1); eauto.
            apply config.eq_refl. rewrite H4. apply Efchain_RLR; auto.
            split; auto. apply Firing_RLR; auto.
          exists (map place.RLL (petri.init N1) :: map (map place.RLL) x),
            (map place.RLL x0), x1, (map place.RLL x2), x3. split. constructor.
            eapply petri.Firing_eq_eq. apply in_or_app. left. apply in_eq.
            eapply (Reach_L H H0 H1); eauto. apply config.eq_refl. rewrite H3.
            apply Efchain_RLL; auto. split; auto. apply Firing_RLL; auto.
        rewrite <- H6.
          exists (map place.RLL (petri.init N1) :: map (map place.RLL) x),
          (map place.RLL x0), x1, (map place.RLL x2), x3. split. constructor.
          eapply petri.Firing_eq_eq. apply in_or_app. left. apply in_eq.
          eapply (Reach_L H H0 H1); eauto. apply config.eq_refl. rewrite H3.
          apply Efchain_RLL; auto. split; auto. apply Firing_RLL; auto.
      edestruct X0 as [?[?[?[?[?[?[?[?[]]]]]]]]]; eauto. rewrite <- H4.
        apply petri.Reach_init.
        exists (map place.RLR (petri.init N2) :: map (map place.RLR) x),
        (map place.RLR x0), x1, (map place.RLR x2), x3. split. constructor.
        eapply petri.Firing_eq_eq. apply in_or_app. left. right. apply in_eq.
        eapply (Reach_L H H0 H1); eauto. apply config.eq_refl. rewrite H4.
        apply Efchain_RLR; auto. split; auto. apply Firing_RLR; auto.
    destruct t0; try solve [inversion H6]. destruct t0. inversion H6.
      edestruct (Reach_RLL H H0 H1); eauto.
        edestruct X as [?[?[?[?[?[?[?[?[]]]]]]]]]; eauto.
        exists (map (map place.RLL) x), (map place.RLL x0), x1,
        (map place.RLL x2), x3. split. rewrite H7. apply Efchain_RLL; auto.
        split; auto. apply Firing_RLL; auto.
      edestruct (Reach_RLR H H0 H1); eauto.
        edestruct X0 as [?[?[?[?[?[?[?[?[]]]]]]]]]; eauto.
        exists (map (map place.RLR) x), (map place.RLR x0), x1,
        (map place.RLR x2), x3. split. rewrite H7. apply Efchain_RLR; auto.
        split; auto. apply Firing_RLR; auto.
Defined.

(* nb completeness *)
Definition nb_complete N1 N2 nb1 nb2 init1 init2 :
  petri.Wf N1 -> petri.Wf N2 -> petri.Safe N1 -> petri.Safe N2 ->
  petri.NthBag N1 nb1 -> petri.nb_complete N2 nb2 ->
  petri.init N1 = init1 -> petri.init N2 = init2 ->
  petri.nb_complete (make N1 N2) (nb nb1 nb2 init1 init2).
Proof.
  unfold nb. repeat intro. destruct X.
  destruct c. edestruct petri.Reach_nil. eapply Wf. apply H. eauto. auto.
  assert (petri.Reach (make N1 N2) c'). eapply petri.Reach_Efchain; eauto.
  destruct c'. edestruct petri.Reach_nil. eapply Wf. apply H. eauto. auto.
  assert (petri.Reach (make N1 N2) c''). econstructor 2; eauto.
  destruct c''. edestruct petri.Reach_nil. eapply Wf. apply H. eauto. auto.
  destruct t. edestruct (Reach_P H H0 H1); eauto.
    destruct (Reach_cons H H0 H1 H2 H11) as [[[|[]]|[]]|].
      edestruct Firing_init_Some. apply H. eauto. eauto. eauto.
      rewrite H13 in H7. destruct (Reach_cons H H0 H1 H2 H12) as [[[|[]]|[]]|].
        edestruct Firing_to_init. apply H. eauto. eauto. eauto.
        rewrite H15 in H7. edestruct NB_complete with
          (F:=fun (B0 : bag.t) (c1' : config.t) => F B0 (map place.RLL c1'))
          as [??[]]. apply petri.Reach_init. eapply Efchain_init_RLL; eauto.
          eapply (Reach_L H H0 H1); eauto. eapply Firing_RLL_RLL; eauto. eauto.
          eauto. eauto. eauto.
          exists (map place.RLL x). eapply Reach_RLL'; eauto.
          exists (map (map place.RLL) x0). rewrite H15.
            eapply Efchain_RLL; eauto.
          intro. destruct (s n'). exists x1. intro. rewrite <- H3.
          rewrite e0; eauto. destruct H17. rewrite H17. auto.
        rewrite H15 in H7. edestruct Firing_RLL_RLR. apply H. eauto. eauto.
        edestruct Firing_Some_fin. apply H. eauto. eauto. eauto.
      rewrite H13 in H7. destruct (Reach_cons H H0 H1 H2 H12) as [[[|[]]|[]]|].
        edestruct Firing_to_init. apply H. eauto. eauto. eauto.
        rewrite H15 in H7. edestruct Firing_RLR_RLL. apply H. eauto. eauto.
        rewrite H15 in H7. edestruct X0 with
          (F:=fun (B0 : bag.t) (c2' : config.t) => F B0 (map place.RLR c2'))
          as [??[]]. apply petri.Reach_init. eapply Efchain_init_RLR; auto.
          apply H. auto. eapply (Reach_L H H0 H1); eauto. eauto. auto.
          eapply Firing_RLR_RLR. apply H. eauto. eauto. eauto. eauto. eauto.
          exists (map place.RLR x). eapply Reach_RLR'; eauto.
          exists (map (map place.RLR) x0). rewrite H15.
            eapply Efchain_RLR; eauto.
          intro. destruct (s n'). case_eq (nb1 init1 sig sync
              (fun B0 c1' => F B0 (map place.RLL c1')) x1); intros.
            edestruct NB_Some_Some with (F:=fun (B0 : bag.t) (c1' : config.t) =>
              F B0 (map place.RLL c1')); eauto. exists x2. intro. destruct e1.
              rewrite H19. rewrite <- H4. rewrite e0; eauto. destruct H18.
              rewrite H18. auto.
            exists x1. intro. erewrite NB_None_None; eauto. rewrite <- H4.
              rewrite e0; eauto.
        edestruct Firing_Some_fin. apply H. eauto. eauto. eauto.
      edestruct Firing_fin. apply H. eauto. eauto. eauto.
    destruct t. edestruct (Reach_RP H H0 H1); eauto.
      destruct t. edestruct (Reach_RLP H H0 H1); eauto.
        edestruct (Reach_RLL H H0 H1); eauto.
            destruct (Reach_cons H H0 H1 H2 H11) as [[[|[]]|[]]|].
          edestruct Firing_init_Some. apply H. eauto. eauto. eauto.
          rewrite H15 in H7.
              destruct (Reach_cons H H0 H1 H2 H12) as [[[|[]]|[]]|].
            edestruct Firing_to_init. apply H. eauto. eauto. eauto.
            rewrite H17 in H7. edestruct NB_complete with
              (F:=fun (B0 : bag.t) (c1' : config.t) => F B0 (map place.RLL c1'))
              as [??[]]. apply H14. eapply Efchain_RLL_RLL; eauto.
              eapply Firing_RLL_RLL; eauto. eauto. eauto. eauto. eauto.
              exists (map place.RLL x). eapply Reach_RLL'; eauto.
              exists (map (map place.RLL) x0). rewrite H17.
                eapply Efchain_RLL; eauto.
              intro. destruct (s n'). exists x1. intro. rewrite e0; eauto.
            rewrite H17 in H7. edestruct Firing_RLL_RLR. apply H. eauto. eauto.
            edestruct Firing_Some_fin. apply H. eauto. eauto. eauto.
          edestruct Efchain_RLL_RLR. apply H. eauto. auto. eauto. eauto. auto.
            auto.
          edestruct Firing_fin. apply H. eauto. eauto. eauto.
        edestruct (Reach_RLR H H0 H1); eauto.
            destruct (Reach_cons H H0 H1 H2 H11) as [[[|[]]|[]]|].
          edestruct Firing_init_Some. apply H. eauto. eauto. eauto.
          edestruct Efchain_RLR_RLL. apply H. eauto. auto. eauto. eauto. auto.
            auto.
          rewrite H15 in H7.
              destruct (Reach_cons H H0 H1 H2 H12) as [[[|[]]|[]]|].
            edestruct Firing_to_init. apply H. eauto. eauto. eauto.
            rewrite H17 in H7. edestruct Firing_RLR_RLL. apply H. eauto. eauto.
            rewrite H17 in H7. edestruct X0 with
              (F:=fun (B0 : bag.t) (c2' : config.t) => F B0 (map place.RLR c2'))
              as [??[]]. apply H14. eapply Efchain_RLR_RLR; auto. apply H.
              eauto. auto. eapply Firing_RLR_RLR. apply H. eauto. eauto. eauto.
              eauto. eauto. exists (map place.RLR x). eapply Reach_RLR'; eauto.
              exists (map (map place.RLR) x0). rewrite H17.
                eapply Efchain_RLR; eauto.
              intro. destruct (s n'). exists x1. intro. rewrite e0; eauto.
            edestruct Firing_Some_fin. apply H. eauto. eauto. eauto.
          edestruct Firing_fin. apply H. eauto. eauto. eauto.
      edestruct Firing_fin. apply H. eauto. eapply Efchain_fin. apply H. eauto.
        eapply (Reach_RR H H0 H1); eauto. eauto. eauto.
Defined.

(* nb correctness *)
Definition NthBag N1 N2 nb1 nb2 init1 init2 :
  petri.Wf N1 -> petri.Wf N2 -> petri.Safe N1 -> petri.Safe N2 ->
  petri.NthBag N1 nb1 -> petri.NthBag N2 nb2 ->
  petri.init N1 = init1 -> petri.init N2 = init2 ->
  petri.NthBag (make N1 N2) (nb nb1 nb2 init1 init2).
Proof.
  intros. inversion X. destruct X0.
  split; try apply nb_sound; try apply nb_complete; auto;
  unfold nb; try unfold petri.f_None in *; try unfold petri.f_Some in *;
  repeat intro.
    destruct c; auto. destruct t; auto.
      case_eq (nb1 init1 sig sync (fun B c1' =>
          F B (map place.RLL c1')) n); intros; rewrite H7 in H6.
        destruct p. inversion H6.
        erewrite NB_None_None; eauto.
      destruct t; auto. destruct t; eauto.
    destruct c. inversion H6. destruct t0. inversion H6.
      case_eq (nb1 init1 sig sync (fun B c1' =>
          F B (map place.RLL c1')) n); intros; rewrite H7 in H6.
        destruct p.
          case_eq (nb2 init2 sig sync (fun B c2' =>
              F B (map place.RLR c2')) 0); intros.
            edestruct NB_Some_Some0 with (F:=
                fun (B : bag.t) (c2' : config.t) => F B (map place.RLR c2'));
                eauto.
              edestruct NB_Some_Some with (F:=
                fun (B : bag.t) (c1' : config.t) => F B (map place.RLL c1'));
                eauto.
              exists x0. destruct e0. rewrite H9. destruct e. rewrite e. eauto.
            edestruct NB_Some_Some with (F:=fun (B : bag.t) (c1' : config.t) =>
                F B (map place.RLL c1')); eauto.
              exists x. destruct e. rewrite H9. erewrite NB_None_None0; eauto.
        edestruct NB_Some_Some0 with (F:=fun (B : bag.t) (c2' : config.t) =>
            F B (map place.RLR c2')); eauto.
          exists x. erewrite NB_None_None; eauto.
      destruct t0; try solve [inversion H6]. destruct t0; eauto. inversion H6.
Defined.

End sel.

End MPetSel.

(* (c) 2020 Brittany Ro Nkounkou *)
