(* Brittany Nkounkou *)
(* August 2020 *)
(* Repetition Petri Net *)

Require Export PetSel.

Set Implicit Arguments.

Module MPetRep (env : Environment).
Module Export M := MPetSel env.

Module rep.

(* initial configuration *)
Definition init : config.t :=
  [place.L place.P].

(* final configuration *)
Definition fin : config.t :=
  [place.R (place.R place.P)].

(* repetition petri net *)
Definition make N0 : petri.t :=
  petri.make
    init
    fin
    ([(init, None, map place.RL (petri.init N0));
      (init, None, fin)]
      ++ map (trans.map place.RL) (petri.T N0)
      ++ [(map place.RL (petri.fin N0), None, init)]).

(* empty-firing-chain-to-fin decider *)
Definition ef ef0 : petri.efc_fin :=
  fun c =>
  match c with
  | place.R (place.L _) :: _ => ef0 (config.RL c)
  | _ => true
  end.

(* nth-bag finder *)
Definition nb (ef0 : petri.efc_fin) nb0 init0 : petri.nth_bag :=
  fun c sig sync F n =>
  let F0 (B : bag.t) (c0 : config.t) := F B (map place.RL c0) in
  match c with
  | place.L _ :: _ => nb0 init0 sig sync F0 n
  | place.R (place.L _) :: _ =>
    match nb0 (config.RL c) sig sync F0 n with
    | Some (p, o) => Some
      match o with
      | None => (p, None)
      | Some n' =>
        if ef0 (config.RL c)
        then
          match nb0 init0 sig sync F0 n' with
          | Some t => t
          | None => (p, Some n')
          end
        else (p, Some n')
      end
    | None => if ef0 (config.RL c) then nb0 init0 sig sync F0 n else None
    end
  | _ => None
  end.

(* ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~ *)

(* the repetition petri net is well-formed *)
Lemma Wf N : petri.Wf N -> petri.Wf (make N).
Proof.
  intros. destruct H. split; try discriminate. intros.
  destruct (in_app_or _ _ _ H).
    destruct H0.
      inversion H0. split; try discriminate.
        intro; apply Wf_init; eapply map_eq_nil; eauto.
      destruct H0; inversion H0. split; discriminate.
    destruct (in_app_or _ _ _ H0).
      destruct (proj1 (in_map_iff _ _ _) H1) as [[[]][]].
        inversion H2. destruct (Wf_T _ _ _ H3). split.
          intro; apply H4; eapply map_eq_nil; eauto.
          intro; apply H8; eapply map_eq_nil; eauto.
      destruct H1; inversion H1. split; try discriminate.
        intro; apply Wf_fin; eapply map_eq_nil; eauto.
Qed.

(* ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~ *)

(* the first element of any subconfiguration of init is not the P place *)
Lemma le_P_init c : ~ config.le (place.P :: c) init.
Proof.
  intro. edestruct H. apply in_eq. inversion H0. destruct H0.
Qed.

(* the first element of any subconfiguration of init is not and R place *)
Lemma le_R_init p c : ~ config.le (place.R p :: c) init.
Proof.
  intro. edestruct H. apply in_eq. inversion H0. destruct H0.
Qed.

(* the first element of any subconfiguration of fin is not the P place *)
Lemma le_P_fin c : ~ config.le (place.P :: c) fin.
Proof.
  intro. edestruct H. apply in_eq. inversion H0. destruct H0.
Qed.

(* the first element of any subconfiguration of fin is not an L place *)
Lemma le_L_fin p c : ~ config.le (place.L p :: c) fin.
Proof.
  intro. edestruct H. apply in_eq. inversion H0. destruct H0.
Qed.

(* the first element of any subconfiguration of fin is not the R P place *)
Lemma le_RP_fin c : ~ config.le (place.R place.P :: c) fin.
Proof.
  intro. edestruct H. apply in_eq. inversion H0. destruct H0.
Qed.

(* the first element of any subconfiguration of fin is not an R L place *)
Lemma le_RL_fin p c : ~ config.le (place.R (place.L p) :: c) fin.
Proof.
  intro. edestruct H. apply in_eq. inversion H0. destruct H0.
Qed.

(* ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~ *)

(* any firing from init is empty and ends on either the initial configuration of
    the inner right petri net or fin *)
Lemma Firing_init N c w c' :
  petri.Wf N -> config.eq c init -> petri.Firing (make N) c w c' ->
  w = None /\ (config.eq c' (map place.RL (petri.init N)) \/ config.eq c' fin).
Proof.
  intros. inversion H1. edestruct in_app_or. apply H2.
    destruct H5.
      inversion H5. split. auto. left. apply config.eq_sym.
        eapply config.eq_trans; eauto. rewrite H9. rewrite config.diff_eq.
        apply config.eq_sym, config.union_nil. rewrite <- H7. auto.
      destruct H5; inversion H5. split. auto. right. apply config.eq_sym.
        eapply config.eq_trans; eauto. rewrite H9. rewrite config.diff_eq.
        apply config.eq_sym, config.union_nil. rewrite <- H7. auto.
    exfalso. edestruct in_app_or; eauto.
      rewrite in_map_iff in H6. destruct H6 as [?[]]. destruct x, p.
        inversion H6. rewrite <- H9 in H3. destruct H. edestruct Wf_T; eauto.
        destruct t0. contradiction. absurd (In (place.RL t0) init).
          intro. destruct H12; auto. inversion H12.
          apply H0. apply H3. apply in_eq.
      destruct H6; inversion H6. destruct H. destruct (petri.fin N).
        contradiction. absurd (In (place.RL t) init).
          intro. destruct H; auto. inversion H.
          apply H0. apply H3. rewrite <- H8. apply in_eq.
Qed.

(* there are no non-empty firings from init *)
Lemma Firing_init_Some N c B c' :
  petri.Wf N -> config.eq c init -> ~ petri.Firing (make N) c (Some B) c'.
Proof.
  do 3 intro. edestruct Firing_init; eauto. inversion H2.
Qed.

(* there are no non-empty firings to init *)
Lemma Firing_Some_init N c B c' :
  petri.Wf N -> config.eq c' init -> ~ petri.Firing (make N) c (Some B) c'.
Proof.
  do 3 intro. inversion H1. edestruct in_app_or. apply H2.
    destruct H5. inversion H5. destruct H5; inversion H5.
    edestruct in_app_or; eauto.
      rewrite in_map_iff in H6. destruct H6 as [?[]]. destruct x, p.
        inversion H6. rewrite <- H11 in H4. destruct H. edestruct Wf_T; eauto.
        destruct t. contradiction. absurd (In (place.RL t) init).
          intro. destruct H12; auto. inversion H12.
          apply H0. apply H4. apply config.In_union. right. apply in_eq.
      destruct H6; inversion H6.
Qed.

(* there are no firings from fin *)
Lemma Firing_fin N c w c' :
  petri.Wf N -> config.eq c fin -> ~ petri.Firing (make N) c w c'.
Proof.
  do 3 intro. inversion H1. edestruct in_app_or. apply H2.
    absurd (In (place.L place.P) fin).
      intro. destruct H6; auto. inversion H6.
      apply H0. apply H3. destruct H5. inversion H5. apply in_eq.
        destruct H5; inversion H5. apply in_eq.
    edestruct in_app_or; eauto.
      rewrite in_map_iff in H6. destruct H6 as [?[]]. destruct x, p.
        inversion H6. rewrite <- H9 in H3. destruct H. edestruct Wf_T; eauto.
        destruct t0. contradiction. absurd (In (place.RL t0) fin).
          intro. destruct H12; auto. inversion H12.
          apply H0. apply H3. apply in_eq.
      destruct H6; inversion H6. destruct H. destruct (petri.fin N).
        contradiction. absurd (In (place.RL t) fin).
          intro. destruct H; auto. inversion H.
          apply H0. apply H3. rewrite <- H8. apply in_eq.
Qed.

(* there are no non-empty firings to fin *)
Lemma Firing_Some_fin N c B c' :
  petri.Wf N -> config.eq c' fin -> ~ petri.Firing (make N) c (Some B) c'.
Proof.
  do 3 intro. inversion H1. edestruct in_app_or. apply H2.
    destruct H5. inversion H5. destruct H5; inversion H5.
    edestruct in_app_or; eauto.
      rewrite in_map_iff in H6. destruct H6 as [?[]]. destruct x, p.
        inversion H6. rewrite <- H11 in H4. destruct H. edestruct Wf_T; eauto.
        destruct t. contradiction. absurd (In (place.RL t) fin).
          intro. destruct H12; auto. inversion H12.
          apply H0. apply H4. apply config.In_union. right. apply in_eq.
      destruct H6; inversion H6.
Qed.

(* any firing from the inner petri net is in the repetition petri net *)
Lemma Firing_RL N c w c' :
  petri.Firing N c w c' ->
  petri.Firing (make N) (map place.RL c) w (map place.RL c').
Proof.
  assert (forall p p' : place.t, place.RL p = place.RL p' -> p = p').
    intros. inversion H. auto.
  intro. destruct H0. econstructor.
    apply in_or_app. right. apply in_or_app. left. apply in_map_iff.
      exists (ci, w, co). split; simpl; eauto.
    apply config.le_map; auto.
    rewrite config.diff_map, config.union_map; auto. apply config.eq_map; auto.
Qed.

(* any firing from an inner configuration either
    ends on an inner configuration and is a firing from the inner petri net, or
    starts from the final configuration of the inner petri net and ends on
    init *)
Lemma Firing_RL_ N c w c' :
  petri.Safe N -> petri.Reach N c ->
  petri.Firing (make N) (map place.RL c) w c' ->
  c' = map place.RL (config.RL c') /\ petri.Firing N c w (config.RL c') \/
  config.eq c (petri.fin N) /\ config.eq c' init.
Proof.
  assert (forall p p' : place.t, place.RL p = place.RL p' -> p = p').
    intros. inversion H. auto.
  intros. inversion H2. edestruct in_app_or. apply H3.
    absurd (In (place.L place.P) (map place.RL c)).
      clear. intro. induction c; auto. inversion H; auto. inversion H0.
      apply H4. destruct H6. inversion H6. apply in_eq.
        destruct H6; inversion H6. apply in_eq.
    edestruct in_app_or; eauto.
      left. rewrite in_map_iff in H7. destruct H7 as [?[]]. destruct x, p.
        inversion H7. rewrite <- H10, <- H11, <- H12 in *.
        rewrite config.le_map in H4; auto.
        rewrite config.diff_map, config.union_map in H5; auto. split.
          eapply config.eq_map_RL; eauto.
          erewrite config.eq_map_RL, config.eq_map in H5; eauto.
            econstructor; eauto.
      right. destruct H7; inversion H7. rewrite <- H9, <- H11 in *.
        rewrite config.le_map in H4; auto. rewrite config.diff_eq in H5.
          split. split; auto. apply config.eq_sym. auto.
          apply config.eq_map; auto. split; auto.
Qed.

(* any inner-to-inner firing is a firing from the inner petri net *)
Lemma Firing_RL_RL N c w c' :
  petri.Firing (make N) (map place.RL c) w (map place.RL c') ->
  petri.Firing N c w c'.
Proof.
  assert (forall p p' : place.t, place.RL p = place.RL p' -> p = p').
    intros. inversion H. auto.
  intros. inversion H0. edestruct in_app_or. apply H1.
    absurd (In (place.L place.P) (map place.RL c)).
      clear. intro. induction c; auto. inversion H; auto. inversion H0.
      apply H2. destruct H4. inversion H4. apply in_eq.
        destruct H4; inversion H4. apply in_eq.
    edestruct in_app_or; eauto.
      rewrite in_map_iff in H5. destruct H5 as [?[]]. destruct x, p.
        inversion H5. rewrite <- H8, <- H9, <- H10 in *.
        rewrite config.le_map in H2; auto.
        rewrite config.diff_map, config.union_map, config.eq_map in H3; auto.
        econstructor; eauto.
      absurd (In (place.L place.P) (map place.RL c')).
        clear. intro. induction c'; auto. inversion H; auto. inversion H0.
        apply H3. apply config.In_union. right. destruct H5; inversion H5.
          apply in_eq.
Qed.

(* ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~ *)

(* any empty firing chain from the inner petri net is in the repetition petri
    net *)
Lemma Efchain_RL N c l c' :
  petri.Efchain N c l c' ->
  petri.Efchain (make N) (map place.RL c) (map (map place.RL) l)
    (map place.RL c').
Proof.
  intro. induction H. constructor. apply config.eq_map.
    intros. inversion H0. auto. auto.
  constructor. apply Firing_RL; auto. auto.
Qed.

(* any empty firing chain from fin is a self-loop *)
Lemma Efchain_fin N c l c': petri.Wf N ->
  config.eq c fin -> petri.Efchain (make N) c l c' -> config.eq c' fin.
Proof.
  intros. destruct H1. eapply config.eq_trans; eauto. apply config.eq_sym; auto.
  edestruct Firing_fin; eauto.
Qed.

(* returns the sublist of l that comes after the last occurrence of init *)
Fixpoint afterlastinit l : list config.t :=
  match l with
  | [] => []
  | c :: l' =>
    if Exists_dec _ l' (config.dec init)
    then afterlastinit l'
    else if config.dec init c then l' else c :: l'
  end.

(* helper induction principle for list *)
Fixpoint list_rect2
  (P : list config.t -> Type) (f : P []) (f0 : forall c0, P [c0])
  (f1 : forall c1 c2 l, P (c2 :: l) -> P l -> P (c1 :: c2 :: l)) l : P l :=
  match l with
  | [] => f
  | c :: l' =>
    match l' as l' return P l' -> P (c :: l') with
    | [] => fun _ => f0 c
    | c' :: l'' => fun f' => f1 c c' l'' f' (list_rect2 P f f0 f1 l'')
    end (list_rect2 P f f0 f1 l')
  end.

(* any inner-to-inner empty firing chain is either from the inner petri net, or
    an empty firing chain from the inner petri net ending on its final
      configuration concatenated with
    an empty firing chain from the inner petri net starting on its initial
      configuration *)
Definition Efchain_RL_RL N c l c' :
  petri.Wf N -> petri.Safe N -> petri.Reach N (config.RL c) ->
  petri.Efchain (make N) c l c' ->
  c = map place.RL (config.RL c) -> c' = map place.RL (config.RL c') ->
  {petri.Efchain N (config.RL c) (map config.RL l) (config.RL c') /\
      ~ Exists (config.eq init) l} +
  {petri.Efchain N (config.RL c) (map config.RL (path.before init l))
      (petri.fin N) /\
    petri.Efchain N (petri.init N) (map config.RL (tl (afterlastinit l)))
      (config.RL c') /\
    Exists (config.eq init) l}.
Proof.
  intros. generalize dependent c. induction l using list_rect2; intros.
    left. split. constructor. apply config.eq_RL. inversion H2. auto.
      intro. inversion H5.
    left. inversion H2. rewrite H3 in H7.
        edestruct Firing_RL_ as [[]|[]]; eauto.
      split.
        constructor. rewrite H10 in H7. eapply Firing_RL_RL; eauto.
          constructor. apply config.eq_RL. inversion H9. auto.
        intro. inversion H12.
          absurd (In (place.L place.P) c0).
            rewrite H10. clear. intro. induction (config.RL c0); auto.
              inversion H; auto. inversion H0.
            apply H14. apply in_eq.
          inversion H14.
      absurd (In (place.L place.P) c').
        rewrite H4. clear. intro. induction (config.RL c'); auto.
          inversion H; auto. inversion H0.
        inversion H9. apply H12. apply H11. apply in_eq.
    simpl. destruct config.dec.
      right. inversion H2. split.
        constructor. rewrite H3 in H7. edestruct Firing_RL_ as [[]|[]]; eauto.
          absurd (In (place.L place.P) c1).
            rewrite H10. clear. intro. induction (config.RL c1); auto.
              inversion H; auto. inversion H0.
            apply e. apply in_eq.
        split; auto. inversion H9. edestruct Firing_init as [_[]]; eauto.
            apply config.eq_sym. auto.
          destruct config.dec.
            absurd (In (place.L place.P) (map place.RL (petri.init N))).
              clear. intro. induction (petri.init N); auto. inversion H; auto.
                inversion H0.
              apply H15. apply e0. apply in_eq.
            edestruct IHl0 as [[]|[?[]]]. rewrite config.RL_map_RL.
                apply petri.Reach_init. eapply petri.Efchain_eq_l; eauto.
                rewrite config.RL_map_RL. auto.
              destruct Exists_dec. contradiction.
                eapply petri.Efchain_eq_l; eauto. rewrite config.RL_map_RL.
                apply config.eq_refl.
              destruct Exists_dec. auto. contradiction.
          absurd (In (place.R (place.R place.P)) c').
            rewrite H4. clear. intro. induction (config.RL c'); auto.
              inversion H; auto. inversion H0.
            eapply Efchain_fin; eauto. apply in_eq.
      assert (c1 = map place.RL (config.RL c1) /\
          petri.Firing N (config.RL c) None (config.RL c1)). inversion H2.
          rewrite H3 in H7. edestruct Firing_RL_ as [[]|[]]; eauto. destruct n.
          apply config.eq_sym. auto. destruct H5. edestruct IHl as [[]|[?[]]].
          inversion H2. econstructor 2; eauto. inversion H2. auto. auto.
        left. split. constructor; auto. intro. inversion H9; contradiction.
        right. split. constructor; auto. split; auto. simpl in H8.
          destruct config.dec, Exists_dec; auto. inversion H9; contradiction.
Defined.

(* any inner-to-fin empty firing chain includes one from the inner petri net
    that ends on its final configuration *)
Lemma Efchain_RL_init N c l c':
  petri.Safe N -> petri.Reach N (config.RL c) ->
  petri.Efchain (make N) c l c' ->
  c = map place.RL (config.RL c) -> config.eq c' init ->
  petri.Efchain N (config.RL c) (map config.RL (path.before init l))
    (petri.fin N) /\ Exists (config.eq init) l.
Proof.
  intros. induction H1; intros.
    absurd (In (place.L place.P) c).
      rewrite H2. clear. intro. induction (config.RL c); auto.
        inversion H; auto. inversion H0.
      apply H1. apply H3. apply in_eq.
    rewrite H2 in H1. edestruct Firing_RL_ as [[]|[]]; eauto.
      simpl. destruct config.dec.
        absurd (In (place.L place.P) c').
          rewrite H5. clear. intro. induction (config.RL c'); auto.
            inversion H; auto. inversion H0.
          apply e. apply in_eq.
        destruct IHEfchain; auto. econstructor 2; eauto. split; auto.
          constructor; auto.
      simpl. destruct config.dec. split; auto. constructor. auto. destruct n.
        apply config.eq_sym. auto.
Qed.

(* any inner-to-fin empty firing chain includes one from the inner petri net
    that ends on its final configuration *)
Lemma Efchain_RL_fin N c l c':
  petri.Safe N -> petri.Reach N (config.RL c) ->
  petri.Efchain (make N) c l c' ->
  c = map place.RL (config.RL c) -> config.eq c' fin ->
  petri.Efchain N (config.RL c) (map config.RL (path.before init l))
    (petri.fin N) /\ Exists (config.eq init) l.
Proof.
  intros. induction H1; intros.
    absurd (In (place.R (place.R place.P)) c).
      rewrite H2. clear. intro. induction (config.RL c); auto.
        inversion H; auto. inversion H0.
      apply H1. apply H3. apply in_eq.
    rewrite H2 in H1. edestruct Firing_RL_ as [[]|[]]; eauto.
      simpl. destruct config.dec.
        absurd (In (place.L place.P) c').
          rewrite H5. clear. intro. induction (config.RL c'); auto.
            inversion H; auto. inversion H0.
          apply e. apply in_eq.
        destruct IHEfchain; auto. econstructor 2; eauto. split; auto.
          constructor; auto.
      simpl. destruct config.dec. split; auto. constructor. auto. destruct n.
        apply config.eq_sym. auto.
Qed.

(* any init-to-inner empty firing chain includes one from the inner petri net
    that starts on its initial configuration *)
Lemma Efchain_init_RL N c l c' :
  petri.Wf N -> petri.Safe N -> config.eq c init ->
  petri.Efchain (make N) c l c'-> c' = map place.RL (config.RL c') ->
  petri.Efchain N (petri.init N) (map config.RL (tl (afterlastinit l)))
    (config.RL c').
Proof.
  intros. destruct H2.
    absurd (In (place.L place.P) c').
      rewrite H3. clear. intro. induction (config.RL c'); auto.
        inversion H; auto. inversion H0.
      apply H2. apply H1. apply in_eq.
    edestruct Firing_init as [_[]]; eauto.
      edestruct Efchain_RL_RL as [[]|[?[]]]. eauto. eauto.
          rewrite config.RL_map_RL. apply petri.Reach_init.
          eapply petri.Efchain_eq_l; eauto. rewrite config.RL_map_RL. auto.
          auto.
        rewrite <- config.RL_map_RL at 1. simpl.
          destruct Exists_dec. contradiction. destruct config.dec; auto.
          absurd (In (place.L place.P) (map place.RL (petri.init N))).
            clear. intro. induction (petri.init N); auto.
              inversion H; auto. inversion H0.
            apply H5. apply e. apply in_eq.
        simpl. destruct Exists_dec. auto. contradiction.
      absurd (In (place.R (place.R place.P)) c'').
        rewrite H3. clear. intro. induction (config.RL c''); auto.
          inversion H; auto. inversion H0.
        eapply Efchain_fin; eauto. apply in_eq.
Qed.

(* ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~ *)

(* any reachable configuration is either init,
    a reachable configuration from the inner petri net, or fin *)
Lemma Reach N c :
  petri.Wf N -> petri.Safe N -> petri.Reach (make N) c ->
  config.eq c init \/
  c = map place.RL (config.RL c) /\ petri.Reach N (config.RL c) \/
  config.eq c fin.
Proof.
  intros. induction H1. left. auto. destruct IHReach as [|[[]|]].
    edestruct Firing_init as [_[]]; eauto. right. left. split.
      eapply config.eq_map_RL. apply config.eq_sym. eauto.
      eapply petri.Reach_eq. apply petri.Reach_init.
      erewrite config.eq_map_RL in H4 at 1. apply config.eq_sym.
      eapply config.eq_map; eauto. intros. inversion H5. auto.
      apply config.eq_sym. eauto.
    rewrite H3 in H2. edestruct Firing_RL_ as [[]|[]]; eauto. right. left.
      split; auto. econstructor 2; eauto.
    edestruct Firing_fin; eauto.
Qed.

(* the first element of any reachable configuration is not the P place *)
Lemma Reach_P N c :
  petri.Wf N -> petri.Safe N -> ~ petri.Reach (make N) (place.P :: c).
Proof.
  do 3 intro. edestruct Reach as [|[[]|]]; eauto; simpl in H2.
    eapply le_P_init, H2.
    destruct (config.RL c); inversion H2.
    eapply le_P_fin, H2.
Qed.

(* if the first element of a reachable configuration is an L place,
    then it is init *)
Lemma Reach_L N p c :
  petri.Wf N -> petri.Safe N -> petri.Reach (make N) (place.L p :: c) ->
  config.eq (place.L p :: c) init.
Proof.
  intros. edestruct Reach as [|[[]|]]; eauto; simpl in H2.
    destruct (config.RL c); inversion H2.
    edestruct le_L_fin. apply H2.
Qed.

(* the first element of any reachable configuration is not the R P place *)
Lemma Reach_RP N c :
  petri.Wf N -> petri.Safe N -> ~ petri.Reach (make N) (place.R place.P :: c).
Proof.
  do 3 intro. edestruct Reach as [|[[]|]]; eauto; simpl in H2.
    eapply le_R_init, H2.
    destruct (config.RL c); inversion H2.
    eapply le_RP_fin, H2.
Qed.

(* if the first element of a reachable configuration is an R L place,
    then it is a reachable configuration from the inner petri net *)
Lemma Reach_RL N p c :
  petri.Wf N -> petri.Safe N ->
  petri.Reach (make N) (place.R (place.L p) :: c) ->
  place.R (place.L p) :: c =
    map place.RL (config.RL (place.R (place.L p) :: c)) /\
  petri.Reach N (config.RL (place.R (place.L p) :: c)).
Proof.
  intros. edestruct Reach as [|[|]]; eauto.
    edestruct le_R_init. apply H2.
    edestruct le_RL_fin. apply H2.
Qed.

(* if the first element of a reachable configuration is an R R place,
    then it is fin *)
Lemma Reach_RR N p c :
  petri.Wf N -> petri.Safe N ->
  petri.Reach (make N) (place.R (place.R p) :: c) ->
  config.eq (place.R (place.R p) :: c) fin.
Proof.
  intros. edestruct Reach as [|[[]|]]; eauto; simpl in H2.
    edestruct le_R_init. apply H2.
    destruct (config.RL c); inversion H2.
Qed.

(* any reachable non-empty configuration is decidably either init,
    a reachable configuration from the inner petri net, or fin *)
Definition Reach_cons N p c :
  petri.Wf N -> petri.Safe N -> petri.Reach (make N) (p :: c) ->
  {config.eq (p :: c) init} +
  {p :: c = map place.RL (config.RL (p :: c)) /\
    petri.Reach N (config.RL (p :: c))} +
  {config.eq (p :: c) fin}.
Proof.
  intros. destruct p. edestruct Reach_P; eauto.
    left. left. eapply Reach_L; eauto.
    destruct p. edestruct Reach_RP; eauto.
      left. right. eapply Reach_RL; eauto.
      right. eapply Reach_RR; eauto.
Defined.

(* any reachable configuration from the inner petri net is reachable in the
    repetition petri net *)
Lemma Reach_RL' N c :
  petri.Reach N c -> petri.Reach (make N) (map place.RL c).
Proof.
  intro. induction H; econstructor 2; eauto. apply petri.Reach_init.
    eapply petri.Firing_eq_eq. apply in_or_app. left. apply in_eq.
      apply config.eq_refl.
      apply config.eq_map; auto. intros. inversion H0. auto.
    apply Firing_RL. eauto.
Qed.

(* the repetition petri net is safe *)
Lemma Safe N : petri.Wf N -> petri.Safe N -> petri.Safe (make N).
Proof.
  do 5 intro. assert (In (place.R (place.R place.P)) c). apply H2. apply in_eq.
  edestruct Reach as [|[[]|]]; eauto.
    exfalso. eapply le_R_init. eapply config.le_trans. apply H2. apply H4.
    exfalso. rewrite H4 in H3. clear - H3. induction (config.RL c); auto.
      inversion H3; auto. inversion H.
    apply H4.
Qed.

(* ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~ *)

(* the left part of a trace from a full path and trace *)
CoFixpoint opttrace_l pi ot : opttrace.t :=
  match pi with
  | path.fin _ => opttrace.eps
  | path.hop l _ pi' =>
    if Exists_dec _ l (config.dec init)
    then opttrace.eps
    else
      match ot with
      | opttrace.eps => opttrace.eps
      | opttrace.opt o ot' => opttrace.opt o (opttrace_l pi' ot')
      end
  end.

(* rewrite helper for trace_l applied to path.fin *)
Lemma opttrace_l_fin l ot : opttrace_l (path.fin l) ot = opttrace.eps.
Proof.
  rewrite opttrace.match_ at 1. auto.
Qed.

(* rewrite helper for trace_l applied to path.hop *)
Lemma opttrace_l_hop l c pi ot :
  opttrace_l (path.hop l c pi) ot =
    if Exists_dec _ l (config.dec init)
    then opttrace.eps
    else
      match ot with
      | opttrace.eps => opttrace.eps
      | opttrace.opt o ot' => opttrace.opt o (opttrace_l pi ot')
      end.
Proof.
  rewrite opttrace.match_ at 1. simpl. destruct Exists_dec; auto.
  destruct ot; auto.
Qed.

(* the left part of a path from a full path *)
CoFixpoint path_l pi : path.t :=
  match pi with
  | path.fin l => path.fin (map config.RL (path.before init l))
  | path.hop l c pi' =>
    if Exists_dec _ l (config.dec init)
    then path.fin (map config.RL (path.before init l))
    else path.hop (map config.RL l) (config.RL c) (path_l pi')
  end.

(* rewrite helper for path_l applied to path.fin *)
Lemma path_l_fin l :
  path_l (path.fin l) = path.fin (map config.RL (path.before init l)).
Proof.
  rewrite path.match_ at 1. auto.
Qed.

(* rewrite helper for path_l applied to path.hop *)
Lemma path_l_hop l c pi :
  path_l (path.hop l c pi) =
    if Exists_dec _ l (config.dec init)
    then path.fin (map config.RL (path.before init l))
    else path.hop (map config.RL l) (config.RL c) (path_l pi).
Proof.
  rewrite path.match_ at 1. simpl. destruct Exists_dec; auto.
Qed.

(* ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~ *)

(* trace_l is generated by the inner petri net *)
Lemma Gen_opttrace_l N : petri.Wf N -> petri.Safe N ->
  forall c pi ot, petri.Reach N (config.RL c) -> petri.Gen (make N) c pi ot ->
  c = map place.RL (config.RL c) ->
  petri.Gen N (config.RL c) (path_l pi) (opttrace_l pi ot).
Proof.
  do 2 intro. cofix F. intros. destruct H2.
    rewrite path_l_fin, opttrace_l_fin. constructor.
      eapply Efchain_RL_fin; eauto. apply config.eq_refl.
    rewrite path_l_hop, opttrace_l_hop. assert (petri.Reach (make N) c').
        eapply petri.Reach_Efchain; eauto. rewrite H3. apply Reach_RL'; auto.
        destruct (Reach H H0 H5) as [|[[]|]].
      edestruct Efchain_RL_init; eauto. destruct Exists_dec; try contradiction.
        constructor; auto.
      edestruct Efchain_RL_RL as [[]|[?[]]]. eauto. auto. apply H1. eauto. auto.
          auto.
        destruct Exists_dec; try contradiction. constructor; auto.
        destruct Exists_dec; try contradiction. constructor; auto.
      edestruct Efchain_RL_fin; eauto. destruct Exists_dec; try contradiction.
        constructor; auto.
    rewrite path_l_hop, opttrace_l_hop. assert (petri.Reach (make N) c').
        eapply petri.Reach_Efchain; eauto. rewrite H3. apply Reach_RL'; auto.
        destruct (Reach H H0 H6) as [|[[]|]].
      edestruct Firing_init_Some; eauto.
      edestruct Efchain_RL_RL as [[]|[?[]]]. eauto. auto. apply H1. eauto. auto.
          auto.
        destruct Exists_dec; try contradiction. rewrite H7 in H4.
            edestruct Firing_RL_ as [[]|[]]; eauto.
          econstructor; eauto. apply F; auto. econstructor 2; eauto.
          edestruct Firing_Some_init; eauto.
        destruct Exists_dec; try contradiction. constructor. auto.
      edestruct Firing_fin; eauto.
Qed.

(* ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~ *)

(* Lang soundness *)
Lemma Lang_sound N : petri.Wf N -> petri.Safe N ->
  slang.le (petri.Lang (make N)) (slang.star opttrace.eps (petri.Lang N)).
Proof.
  do 4 intro. destruct H1. generalize dependent ot. generalize dependent x.
  cofix F. intros. destruct H1. constructor.
    constructor. eapply F. eapply petri.Gen_Efchain; eauto.
    assert (petri.Reach (make N) c'). eapply petri.Reach_Efchain; eauto.
        apply petri.Reach_init. edestruct Reach as [|[[]|]]; eauto.
      edestruct Firing_init_Some; eauto.
      rewrite H5 in H2. edestruct Firing_RL_ as [[]|[]]; eauto.
        assert (petri.Reach N (config.RL c'')). econstructor 2; eauto.
            constructor 3 with (Some B) (opttrace_l pi ot).
          eexists. econstructor; eauto. eapply Efchain_init_RL; eauto.
            apply config.eq_refl. apply Gen_opttrace_l; auto.
          constructor. generalize dependent ot. generalize dependent pi.
              clear H2 H8. generalize dependent c''. cofix F'. intros.
              destruct H3.
            rewrite opttrace_l_fin. constructor.
            rewrite opttrace_l_hop. assert (petri.Reach (make N) c'0).
                eapply petri.Reach_Efchain. eapply Reach_RL'. eauto.
                rewrite H7 in H2. eauto. edestruct Reach as [|[[]|]]; eauto.
              edestruct Efchain_RL_init; eauto.
                destruct Exists_dec; try contradiction. constructor. eapply F.
                  eapply petri.Gen_Efchain. constructor. apply config.eq_sym.
                  eauto. eauto.
              edestruct Efchain_RL_RL as [[]|[?[]]]. eauto. auto. apply H9.
                  eauto. auto. auto.
                destruct Exists_dec; try contradiction. constructor. eauto.
                destruct Exists_dec; try contradiction. constructor. eapply F.
                  eapply petri.Gen_Efchain. eapply petri.Efchain_trans.
                  eapply petri.Efchain_eq_eq. apply in_or_app. left.
                  apply in_eq. apply config.eq_refl. apply config.eq_refl.
                  apply Efchain_RL; eauto. rewrite H10 in H3. eauto.
              edestruct Efchain_RL_fin; eauto.
                  destruct Exists_dec; try contradiction. constructor.
                  generalize dependent ot. generalize dependent pi. clear H2 H8.
                  generalize dependent c'0. cofix F''. intros. destruct H3.
                constructor.
                constructor. eapply F''. eapply Efchain_fin; eauto. eauto.
                edestruct Firing_fin. eauto. eapply Efchain_fin; eauto. eauto.
            rewrite opttrace_l_hop. assert (petri.Reach (make N) c'0).
                eapply petri.Reach_Efchain. eapply Reach_RL'. eauto.
                rewrite H7 in H2. eauto. edestruct Reach as [|[[]|]]; eauto.
              edestruct Firing_init_Some; eauto.
              rewrite H11 in H3. edestruct Firing_RL_ as [[]|[]]; eauto.
                edestruct Efchain_RL_RL as [[]|[?[]]]. eauto. auto. apply H9.
                    eauto. auto. auto.
                  destruct Exists_dec; try contradiction. constructor.
                    eapply F'; eauto. econstructor 2; eauto.
                  destruct Exists_dec; try contradiction. eapply F.
                    econstructor. eapply petri.Efchain_trans.
                    eapply petri.Efchain_eq_eq. apply in_or_app. left.
                    apply in_eq. apply config.eq_refl. apply config.eq_refl.
                    apply Efchain_RL; eauto. eauto. eauto.
                edestruct Firing_Some_init; eauto.
              edestruct Firing_fin; eauto.
        edestruct Firing_Some_init; eauto.
      edestruct Firing_fin; eauto.
Qed.

(* Lang completeness *)
Lemma Lang_complete N :
  slang.le (slang.star opttrace.eps (petri.Lang N)) (petri.Lang (make N)).
Proof.
  do 2 intro. apply petri.Gen_Gen'. generalize dependent ot. cofix F. intros.
  inversion H.
    econstructor. eapply petri.Efchain_eq_eq. apply in_or_app. left. right.
      apply in_eq. apply config.eq_refl. apply config.eq_refl.
    econstructor; auto. constructor. apply config.eq_refl.
    destruct H0. assert (petri.Gen' N (petri.init N) (opttrace.opt o ot0)).
        eapply petri.Gen'_Gen; eauto. inversion H1. inversion H2.
      econstructor. eapply petri.Efchain_trans. eapply petri.Efchain_eq_eq.
          apply in_or_app. left. apply in_eq. apply config.eq_refl.
          apply config.eq_refl. apply Efchain_RL; eauto. clear - F H9 H13.
          generalize dependent ot'0. generalize dependent ot0.
          generalize dependent c'. cofix F'. intros. inversion H9.
        rewrite <- H in H13. inversion H13. econstructor.
          eapply petri.Efchain_trans. apply Efchain_RL; eauto.
          eapply petri.Efchain_trans. eapply petri.Efchain_eq_eq.
          repeat (apply in_or_app; right). apply in_eq. apply config.eq_refl.
          apply config.eq_refl. eapply petri.Efchain_eq_eq. apply in_or_app.
          left. right. apply in_eq. apply config.eq_refl. apply config.eq_refl.
        rewrite <- H0 in H13. inversion H13.
          econstructor. eapply petri.Efchain_trans. apply Efchain_RL; eauto.
          eapply petri.Efchain_eq_eq. repeat (apply in_or_app; right).
          apply in_eq. apply config.eq_refl. apply config.eq_refl. auto.
        rewrite <- H1 in H13. inversion H13. destruct H.
            assert (petri.Gen' N (petri.init N) (opttrace.opt o ot)).
            eapply petri.Gen'_Gen; eauto. inversion H0. inversion H5.
          econstructor. eapply petri.Efchain_trans. apply Efchain_RL; eauto.
              eapply petri.Efchain_trans. eapply petri.Efchain_eq_eq.
              repeat (apply in_or_app; right). apply in_eq.
              apply config.eq_refl. apply config.eq_refl.
              eapply petri.Efchain_trans. eapply petri.Efchain_eq_eq.
              apply in_or_app. left. apply in_eq. apply config.eq_refl.
              apply config.eq_refl. apply Efchain_RL; eauto. eauto.
          econstructor. eapply petri.Efchain_trans. apply Efchain_RL; eauto.
              eapply petri.Efchain_trans. eapply petri.Efchain_eq_eq.
              repeat (apply in_or_app; right). apply in_eq.
              apply config.eq_refl. apply config.eq_refl.
              eapply petri.Efchain_trans. eapply petri.Efchain_eq_eq.
              apply in_or_app. left. apply in_eq. apply config.eq_refl.
              apply config.eq_refl. apply Efchain_RL; eauto.
              apply Firing_RL; eauto. eauto.
        rewrite <- H0 in H13. inversion H13.
          econstructor. apply Efchain_RL; eauto. eauto.
          econstructor. apply Efchain_RL; eauto. apply Firing_RL; eauto. eauto.
      econstructor. eapply petri.Efchain_trans. eapply petri.Efchain_eq_eq.
          apply in_or_app. left. apply in_eq. apply config.eq_refl.
          apply config.eq_refl. apply Efchain_RL; eauto.
          apply Firing_RL; eauto. clear - F H9 H14. generalize dependent ot'0.
          generalize dependent ot0. generalize dependent c''. cofix F'. intros.
          inversion H9.
        rewrite <- H in H14. inversion H14. econstructor.
          eapply petri.Efchain_trans. apply Efchain_RL; eauto.
          eapply petri.Efchain_trans. eapply petri.Efchain_eq_eq.
          repeat (apply in_or_app; right). apply in_eq. apply config.eq_refl.
          apply config.eq_refl. eapply petri.Efchain_eq_eq. apply in_or_app.
          left. right. apply in_eq. apply config.eq_refl. apply config.eq_refl.
        rewrite <- H0 in H14. inversion H14.
          econstructor. eapply petri.Efchain_trans. apply Efchain_RL; eauto.
          eapply petri.Efchain_eq_eq. repeat (apply in_or_app; right).
          apply in_eq. apply config.eq_refl. apply config.eq_refl. auto.
        rewrite <- H1 in H14. inversion H14. destruct H.
            assert (petri.Gen' N (petri.init N) (opttrace.opt o ot)).
            eapply petri.Gen'_Gen; eauto. inversion H0. inversion H5.
          econstructor. eapply petri.Efchain_trans. apply Efchain_RL; eauto.
              eapply petri.Efchain_trans. eapply petri.Efchain_eq_eq.
              repeat (apply in_or_app; right). apply in_eq.
              apply config.eq_refl. apply config.eq_refl.
              eapply petri.Efchain_trans. eapply petri.Efchain_eq_eq.
              apply in_or_app. left. apply in_eq. apply config.eq_refl.
              apply config.eq_refl. apply Efchain_RL; eauto. eauto.
          econstructor. eapply petri.Efchain_trans. apply Efchain_RL; eauto.
              eapply petri.Efchain_trans. eapply petri.Efchain_eq_eq.
              repeat (apply in_or_app; right). apply in_eq.
              apply config.eq_refl. apply config.eq_refl.
              eapply petri.Efchain_trans. eapply petri.Efchain_eq_eq.
              apply in_or_app. left. apply in_eq. apply config.eq_refl.
              apply config.eq_refl. apply Efchain_RL; eauto.
              apply Firing_RL; eauto. eauto.
        rewrite <- H0 in H14. inversion H14.
          econstructor. apply Efchain_RL; eauto. eauto.
          econstructor. apply Efchain_RL; eauto. apply Firing_RL; eauto. eauto.
Qed.

(* ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~ *)

(* ef correctness *)
Definition EfcFin N ef0 :
  petri.Wf N -> petri.Safe N -> petri.EfcFin N ef0 ->
  petri.EfcFin (make N) (ef ef0).
Proof.
  unfold ef. intros. destruct X. split; repeat intro.
    destruct c. edestruct petri.Reach_nil. eapply Wf; eauto. auto.
      destruct t. edestruct Reach_P; eauto.
        exists [fin]. eapply petri.Efchain_eq_eq. apply in_or_app. left. right.
          apply in_eq. eapply Reach_L; eauto. apply config.eq_refl.
        destruct t. edestruct Reach_RP; eauto.
          edestruct Reach_RL; eauto. rewrite H3. edestruct EF_sound; eauto.
            exists (map (map place.RL) x ++ [init; fin]).
            eapply petri.Efchain_trans. apply Efchain_RL; eauto.
            constructor. eapply petri.Firing_eq_eq.
            repeat (apply in_or_app; right). apply in_eq. apply config.eq_refl.
            apply config.eq_refl. eapply petri.Efchain_eq_eq. apply in_or_app.
            left. right. apply in_eq. apply config.eq_refl.
            apply config.eq_refl.
          exists []. constructor. eapply Reach_RR; eauto.
    destruct c; auto. destruct t; auto. destruct t; auto.
      edestruct Reach_RL; eauto. eapply EF_complete; eauto.
      eapply Efchain_RL_fin; eauto. apply config.eq_refl.
Defined.

(* nb soundness *)
Definition nb_sound N ef0 nb0 init0 :
  petri.Wf N -> petri.Safe N -> petri.EfcFin N ef0 -> petri.nb_sound N nb0 ->
  petri.init N = init0 -> petri.nb_sound (make N) (nb ef0 nb0 init0).
Proof.
  unfold nb. repeat intro. destruct X.
  destruct c. inversion H3. destruct t0. inversion H3.
    edestruct X0 as [?[?[?[?[?[?[?[?[]]]]]]]]]; eauto. rewrite <- H1.
      apply petri.Reach_init.
      exists (map place.RL init0 :: map (map place.RL) x), (map place.RL x0),
      x1, (map place.RL x2), x3. split. constructor. eapply petri.Firing_eq_eq.
      apply in_or_app. left. apply in_eq. eapply Reach_L; eauto. rewrite H1.
      apply config.eq_refl. apply Efchain_RL; auto. split; auto.
      apply Firing_RL; auto.
    destruct t0; try solve [inversion H3]. edestruct Reach_RL; eauto.
        case_eq (nb0 (config.RL (place.R (place.L t0) :: c)) sig sync
         (fun B c0 => F B (map place.RL c0)) n); intros; rewrite H6 in H3.
      edestruct X0 as [?[?[?[?[?[?[?[?[]]]]]]]]]; eauto. destruct p. destruct o.
        case_eq (ef0 (config.RL (place.R (place.L t0) :: c))); intro;
            rewrite H12 in H3.
          edestruct EF_sound; eauto. case_eq (nb0 init0 sig sync
              (fun B c0 => F B (map place.RL c0)) n0); intros;
              rewrite H13 in H3; rewrite <- H3.
            edestruct X0 as [?[?[?[?[?[?[?[?[]]]]]]]]]. apply petri.Reach_init.
              rewrite H1. eauto. exists (map (map place.RL) x4 ++
              [init; map place.RL init0] ++ map (map place.RL) x5),
              (map place.RL x6), x7, (map place.RL x8), x9. split.
              eapply petri.Efchain_trans. rewrite H4. eapply Efchain_RL; eauto.
              eapply petri.Efchain_trans. constructor.
              eapply petri.Firing_eq_eq. repeat (apply in_or_app; right).
              apply in_eq. apply config.eq_refl. apply config.eq_refl.
              eapply petri.Efchain_eq_eq. apply in_or_app. left. apply in_eq.
              apply config.eq_refl. rewrite H1. apply config.eq_refl.
              rewrite <- H1. eapply Efchain_RL; eauto. split; auto.
              apply Firing_RL; auto.
            exists (map (map place.RL) x), (map place.RL x0), x1,
              (map place.RL x2), x3. split. rewrite H4. eapply Efchain_RL; auto.
              split; auto. apply Firing_RL; auto.
          rewrite <- H3. exists (map (map place.RL) x), (map place.RL x0), x1,
            (map place.RL x2), x3. split. rewrite H4. eapply Efchain_RL; auto.
            split; auto. apply Firing_RL; auto.
        rewrite <- H3. exists (map (map place.RL) x), (map place.RL x0), x1,
          (map place.RL x2), x3. split. rewrite H4. eapply Efchain_RL; auto.
          split; auto. apply Firing_RL; auto.
      case_eq (ef0 (config.RL (place.R (place.L t0) :: c))); intro;
          rewrite H7 in H3.
        edestruct EF_sound; eauto. edestruct X0 as [?[?[?[?[?[?[?[?[]]]]]]]]].
          apply petri.Reach_init. rewrite H1. eauto.
          exists (map (map place.RL) x ++ [init; map place.RL init0] ++
          map (map place.RL) x0), (map place.RL x1), x2, (map place.RL x3), x4.
          split. eapply petri.Efchain_trans. rewrite H4.
          eapply Efchain_RL; eauto. eapply petri.Efchain_trans. constructor.
          eapply petri.Firing_eq_eq. repeat (apply in_or_app; right).
          apply in_eq. apply config.eq_refl. apply config.eq_refl.
          eapply petri.Efchain_eq_eq. apply in_or_app. left. apply in_eq.
          apply config.eq_refl. rewrite H1. apply config.eq_refl. rewrite <- H1.
          eapply Efchain_RL; eauto. split; auto. apply Firing_RL; auto.
        inversion H3.
Defined.

(* nb completeness *)
Definition nb_complete N ef0 nb0 init0 :
  petri.Wf N -> petri.Safe N -> petri.EfcFin N ef0 -> petri.NthBag N nb0 ->
  petri.init N = init0 -> petri.nb_complete (make N) (nb ef0 nb0 init0).
Proof.
  unfold nb. repeat intro. destruct X, X0.
  destruct c. edestruct petri.Reach_nil. eapply Wf; eauto. auto.
  assert (petri.Reach (make N) c'). eapply petri.Reach_Efchain; eauto.
  destruct c'. edestruct petri.Reach_nil. eapply Wf; eauto. auto.
  assert (petri.Reach (make N) c''). econstructor 2; eauto.
  destruct c''. edestruct petri.Reach_nil. eapply Wf; eauto. auto.
  destruct t. edestruct Reach_P; eauto.
    destruct (Reach_cons H H0 H8) as [[|[]]|].
      edestruct Firing_init_Some; eauto.
      rewrite H10 in H4. destruct (Reach_cons H H0 H9) as [[|[]]|].
        edestruct Firing_Some_init; eauto.
        rewrite H12 in H4. edestruct NB_complete with
          (F:=fun (B0 : bag.t) (c0 : config.t) => F B0 (map place.RL c0)) as
          [??[]]. apply petri.Reach_init. eapply Efchain_init_RL; eauto.
          eapply Reach_L; eauto. eapply Firing_RL_RL; eauto. eauto. eauto.
          eauto. eauto. exists (map place.RL x). eapply Reach_RL'; eauto.
          exists (map (map place.RL) x0). rewrite H12. eapply Efchain_RL; eauto.
          intro. destruct (s n'). exists x1. intro. rewrite <- H1.
          rewrite e0; eauto.
        edestruct Firing_Some_fin; eauto.
      edestruct Firing_fin; eauto.
    destruct t. edestruct Reach_RP; eauto.
      edestruct Reach_RL; eauto. destruct (Reach_cons H H0 H8) as [[|[]]|].
        edestruct Firing_init_Some; eauto.
        rewrite H12 in H4. destruct (Reach_cons H H0 H9) as [[|[]]|].
          edestruct Firing_Some_init; eauto.
          rewrite H14 in H4. edestruct Efchain_RL_RL as [[]|[?[]]]. eauto. auto.
              apply H11. eauto. auto. auto.
            edestruct NB_complete with (F:=fun (B0 : bag.t) (c0 : config.t) =>
              F B0 (map place.RL c0)) as [??[]]. apply H11. eauto.
              eapply Firing_RL_RL; eauto. eauto. eauto. eauto. eauto.
              exists (map place.RL x). eapply Reach_RL'; eauto.
              exists (map (map place.RL) x0). rewrite H14.
                eapply Efchain_RL; eauto.
              intro. destruct (s n'). exists x1. intro. rewrite e0; eauto.
              destruct H18. rewrite H18. auto.
            edestruct NB_complete with (F:=fun (B0 : bag.t) (c0 : config.t) =>
              F B0 (map place.RL c0)) as [??[]]. apply petri.Reach_init. eauto.
              eapply Firing_RL_RL; eauto. eauto. eauto. eauto. eauto.
              exists (map place.RL x). eapply Reach_RL'; eauto.
              exists (map (map place.RL) x0). rewrite H14.
                eapply Efchain_RL; eauto.
              intro. destruct (s n').
              case_eq (nb0 (config.RL (place.R (place.L t) :: c)) sig sync
                  (fun B0 c0 => F B0 (map place.RL c0)) 0); intros.
                edestruct NB_Some_Some with
                  (F:=fun (B0 : bag.t) (c0 : config.t) =>
                  F B0 (map place.RL c0)); eauto. exists x2. intro.
                  destruct e1. rewrite H21. erewrite EF_complete; eauto.
                  rewrite <- H1. rewrite e0; eauto. destruct H20. rewrite H20.
                  auto.
                exists x1. intro. erewrite NB_None_None; eauto.
                  erewrite EF_complete; eauto. rewrite <- H1. rewrite e0; eauto.
          edestruct Firing_Some_fin; eauto.
        edestruct Firing_fin; eauto.
      edestruct Firing_fin; eauto. eapply Efchain_fin; eauto.
        eapply Reach_RR; eauto.
Defined.

(* nb correctness *)
Definition NthBag N ef0 nb0 init0 :
  petri.Wf N -> petri.Safe N -> petri.EfcFin N ef0 -> petri.NthBag N nb0 ->
  petri.init N = init0 -> petri.NthBag (make N) (nb ef0 nb0 init0).
Proof.
  intros. inversion X. inversion X0.
  split; try apply nb_sound; try apply nb_complete; auto;
  unfold nb; try unfold petri.f_None in *; try unfold petri.f_Some in *;
  repeat intro.
    destruct c; auto. destruct t; eauto. destruct t; auto.
        case_eq (nb0 (config.RL (place.R (place.L t) :: c)) sig sync
        (fun B c0 => F B (map place.RL c0)) n); intros; rewrite H4 in H3.
      destruct p. inversion H3.
      erewrite NB_None_None; eauto.
        case_eq (ef0 (config.RL (place.R (place.L t) :: c))); intro;
        rewrite H5 in H3; eauto.
    destruct c. inversion H3. destruct t0. inversion H3.
      edestruct NB_Some_Some with
        (F:=fun (B : bag.t) (c0 : config.t) => F B (map place.RL c0)); eauto.
      destruct t0; try solve [inversion H3].
          case_eq (nb0 (config.RL (place.R (place.L t0) :: c)) sig sync
          (fun B c0 => F B (map place.RL c0)) n); intros; rewrite H4 in H3.
        destruct p. case_eq (ef0 (config.RL (place.R (place.L t0) :: c)));
            intro; rewrite H5 in H3.
          case_eq (nb0 init0 sig sync (fun B c0 => F B (map place.RL c0)) 0);
              intros.
            edestruct NB_Some_Some with (F:=fun (B : bag.t) (c0 : config.t) =>
                F B (map place.RL c0)); eauto. clear H6.
              edestruct NB_Some_Some with
                  (F:=fun (B : bag.t) (c0 : config.t) => F B (map place.RL c0));
                  eauto.
                exists x0. destruct e0. rewrite H6. destruct e. rewrite e.
                  eauto.
            edestruct NB_Some_Some with
                (F:= fun (B : bag.t) (c0 : config.t) => F B (map place.RL c0));
                eauto.
              exists x. destruct e. rewrite H7. erewrite NB_None_None; eauto.
          edestruct NB_Some_Some with (F:=fun (B : bag.t) (c0 : config.t) =>
              F B (map place.RL c0)); eauto.
            exists x. destruct e. rewrite H6. eauto.
        case_eq (ef0 (config.RL (place.R (place.L t0) :: c))); intro;
            rewrite H5 in H3.
          edestruct NB_Some_Some with (F:=fun (B : bag.t) (c0 : config.t) =>
              F B (map place.RL c0)); eauto.
            exists x. erewrite NB_None_None; eauto.
          inversion H3.
Defined.

End rep.

End MPetRep.

(* (c) 2020 Brittany Ro Nkounkou *)
