(* Brittany Nkounkou *)
(* August 2020 *)
(* Parallel Petri Net *)

Require Export PetSeq.

Set Implicit Arguments.

Module MPetPar (env : Environment).
Module Export M := MPetSeq env.

Module par.

(* initial configuration *)
Definition init init1 init2 : config.t :=
  map place.L init1 ++ map place.R init2.

(* initial configuration *)
Definition fin fin1 fin2 : config.t :=
  map place.L fin1 ++ map place.R fin2.

(* parallel petri net *)
Definition make N1 N2 : petri.t :=
  petri.make
    (init (petri.init N1) (petri.init N2))
    (fin (petri.fin N1) (petri.fin N2))
    (map (trans.map place.L) (petri.T N1)
      ++ map (trans.map place.R) (petri.T N2)
      ++ flat_map trans.prod (list_prod
          (map (trans.map place.L) (petri.T N1))
          (map (trans.map place.R) (petri.T N2)))).

(* empty-firing-chain-to-fin decider *)
Definition ef ef1 ef2 : petri.efc_fin :=
  fun c =>
  match config.L_R c with (c1, c2) => andb (ef1 c1) (ef2 c2) end.

(* nth-bag finder *)
Definition nb nb1 nb2 : petri.nth_bag :=
  fun c sig sync F n =>
  match config.L_R c with (c1, c2) =>
    let F1 (B : bag.t) (c1' : config.t) :=
      F B (map place.L c1' ++ map place.R c2) in
    let F2 (B : bag.t) (c2' : config.t) :=
      F B (map place.L c1 ++ map place.R c2') in
    let nb12 n default :=
      let F1' (B1 : bag.t) (c1' : config.t) n' :=
        let F2' (B2 : bag.t) (c2' : config.t) n'' :=
          let B := bag.union B1 B2 in
          if implb sync (bag.sync B)
          then F B (map place.L c1' ++ map place.R c2') n''
          else None in
        nb2 c2 sig false F2' n' in
      match nb1 c1 sig false F1' n with
      | Some t => Some t
      | None => default
      end in
    match nb1 c1 sig sync F1 n with
    | Some (p, None) => Some (p, None)
    | Some (p1, Some n') =>
      match nb2 c2 sig sync F2 n' with
      | Some (p, None) => Some (p, None)
      | Some (p2, Some n'') => nb12 n'' (Some (p2, Some n''))
      | None => nb12 n' (Some (p1, Some n'))
      end
    | None =>
      match nb2 c2 sig sync F2 n with
      | Some (p, None) => Some (p, None)
      | Some (p, Some n') => nb12 n' (Some (p, Some n'))
      | None => nb12 n None
      end
    end
  end.

(* ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~ *)

(* the parallel petri net is well-formed *)
Lemma Wf N1 N2 : petri.Wf N1 -> petri.Wf N2 -> petri.Wf (make N1 N2).
Proof.
  intros. destruct H, H0. split.
    intro; apply Wf_init; eapply map_eq_nil; destruct (app_eq_nil _ _ H);
      eauto.
    intro; apply Wf_fin; eapply map_eq_nil; destruct (app_eq_nil _ _ H);
      eauto.
    intros. destruct (in_app_or _ _ _ H).
      destruct (proj1 (in_map_iff _ _ _) H0) as [[[]][]].
        inversion H1. destruct (Wf_T _ _ _ H2). split.
          intro; apply H3; eapply map_eq_nil; eauto.
          intro; apply H7; eapply map_eq_nil; eauto.
      destruct (in_app_or _ _ _ H0).
        destruct (proj1 (in_map_iff _ _ _) H1) as [[[]][]].
          inversion H2. destruct (Wf_T0 _ _ _ H3). split.
            intro; apply H4; eapply map_eq_nil; eauto.
            intro; apply H8; eapply map_eq_nil; eauto.
        destruct (proj1 (in_flat_map _ _ _) H1) as [[][]].
          destruct (proj1 (in_prod_iff _ _ _ _) H2).
          destruct p as [[]?], p0 as [[]?]. simpl in H3.
          destruct o; try inversion H3. destruct o0; destruct H3; inversion H3.
          destruct (proj1 (in_map_iff _ _ _) H4) as [[[]][]]. inversion H6.
          destruct (proj1 (in_map_iff _ _ _) H5) as [[[]][]]. inversion H11.
          destruct (Wf_T _ _ _ H10), (Wf_T0 _ _ _ H15). split.
            intro; apply H16; eapply map_eq_nil; destruct (app_eq_nil _ _ H23);
              eauto.
            intro; apply H20; eapply map_eq_nil; destruct (app_eq_nil _ _ H23);
              eauto.
Qed.

(* ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~ *)

(* any firing from the left petri net is in the parallel petri net *)
Lemma Firing_L N1 N2 c1 c2 w c1' :
  petri.Firing N1 c1 w c1' ->
  petri.Firing (make N1 N2) (map place.L c1 ++ map place.R c2) w
    (map place.L c1' ++ map place.R c2).
Proof.
  intro. destruct H. constructor 1 with (map place.L ci ++ map place.R [])
      (map place.L co ++ map place.R []).
    apply in_or_app. left. apply in_map_iff. exists (ci, w, co).
      split; simpl; auto. repeat rewrite app_nil_r. auto.
    apply config.le_app. split; auto. do 2 intro. destruct H2.
    eapply config.eq_trans. eapply config.union_eq_l. apply config.diff_app.
    eapply config.eq_trans. eapply config.union_app. apply config.eq_app_r.
      split. eapply config.eq_app_r. apply config.eq_refl. split.
      rewrite config.L_app. auto. rewrite config.R_app. simpl.
      apply config.diff_nil.
Qed.

(* any firing from the right petri net is in the parallel petri net *)
Lemma Firing_R N1 N2 c1 c2 w c2' :
  petri.Firing N2 c2 w c2' ->
  petri.Firing (make N1 N2) (map place.L c1 ++ map place.R c2) w
    (map place.L c1 ++ map place.R c2').
Proof.
  intro. destruct H. constructor 1 with (map place.L [] ++ map place.R ci)
      (map place.L [] ++ map place.R co).
    apply in_or_app. right. apply in_or_app. left. apply in_map_iff.
      exists (ci, w, co). split; simpl; auto.
    apply config.le_app. split; auto. do 2 intro. destruct H2.
    eapply config.eq_trans. eapply config.union_eq_l. apply config.diff_app.
    eapply config.eq_trans. eapply config.union_app. apply config.eq_app_r.
      split. eapply config.eq_app_r. apply config.eq_refl. split.
      rewrite config.L_app. simpl. apply config.diff_nil. rewrite config.R_app.
      auto.
Qed.

(* the merge of any firing from the left petri net with any from the right is in
    the parallel petri net *)
Lemma Firing_union N1 N2 c1 c2 B1 B2 c1' c2' :
  petri.Firing N1 c1 (Some B1) c1' -> petri.Firing N2 c2 (Some B2) c2' ->
  petri.Firing (make N1 N2) (map place.L c1 ++ map place.R c2)
    (Some (bag.union B1 B2)) (map place.L c1' ++ map place.R c2').
Proof.
  intros. destruct H, H0. constructor 1 with (map place.L ci ++ map place.R ci0)
      (map place.L co ++ map place.R co0).
    apply in_or_app. right. apply in_or_app. right. apply in_flat_map.
      eexists. split. apply in_prod; apply in_map; eauto. apply in_eq.
    apply config.le_app. split; auto.
    eapply config.eq_trans. eapply config.union_eq_l. apply config.diff_app.
    eapply config.eq_trans. eapply config.union_app. apply config.eq_app_r.
      split. eapply config.eq_app_r. apply config.eq_refl. split.
      rewrite config.L_app. auto. rewrite config.R_app. auto.
Qed.

(* any empty firing is from either the left petri net or the right petri net *)
Definition Firing_None N1 N2 c c' :
  petri.Firing (make N1 N2) c None c' ->
  {petri.Firing N1 (config.L c) None (config.L c') /\
    config.eq (config.R c) (config.R c')} +
  {config.eq (config.L c) (config.L c') /\
    petri.Firing N2 (config.R c) None (config.R c')}.
Proof.
  intro. assert (exists ci co, In (ci, None, co)
    (map (trans.map place.L) (petri.T N1) ++
    map (trans.map place.R) (petri.T N2)) /\ config.le ci c /\
    config.eq (config.union (config.diff c ci) co) c'). destruct H.
    exists ci, co. split; auto. simpl in H. apply in_or_app.
    edestruct in_app_or; eauto. edestruct in_app_or; eauto.
    rewrite in_flat_map in H3. destruct H3 as [?[]].
    destruct x, p, p, o, p0, p, o; destruct H4; inversion H4.
  assert (incl (petri.T N1) (petri.T N1)). apply incl_refl.
  generalize dependent H0. generalize dependent H1.
  generalize (petri.T N1) at 1 3. induction l; intros.
    right. destruct H0 as [?[?[?[]]]]. simpl in H0. rewrite in_map_iff in H0.
        destruct H0 as [?[]]. destruct x1, p. inversion H0.
        rewrite <- H6, H7, <- H8 in *. split.
      apply config.eq_sym. eapply config.eq_trans. apply config.eq_L.
        apply config.eq_sym. eauto. rewrite <- config.union_L, <- config.diff_L.
        repeat rewrite config.L_map_R. simpl. apply config.diff_nil.
      econstructor; eauto.
        rewrite <- config.R_map_R at 1. apply config.le_R. auto.
        apply config.eq_sym. eapply config.eq_trans. apply config.eq_R.
          apply config.eq_sym. eauto.
          rewrite <- config.union_R, <- config.diff_R.
          repeat rewrite config.R_map_R. apply config.eq_refl.
    destruct a, p, o.
      apply IHl. do 2 intro. apply H1. right. auto. destruct H0 as [?[?[]]].
        destruct H0; eauto. inversion H0.
      destruct (config.le_dec (map place.L t0) c).
        destruct (config.dec (config.union (config.diff c (map place.L t0))
            (map place.L t)) c').
          left. split.
            econstructor. apply H1. apply in_eq.
              rewrite <- config.L_map_L at 1. apply config.le_L. auto.
              apply config.eq_sym. eapply config.eq_trans. apply config.eq_L.
                apply config.eq_sym. eauto.
                rewrite <- config.union_L, <- config.diff_L.
                repeat rewrite config.L_map_L. apply config.eq_refl.
            apply config.eq_sym. eapply config.eq_trans. apply config.eq_R.
              apply config.eq_sym. eauto.
              rewrite <- config.union_R, <- config.diff_R.
              repeat rewrite config.R_map_L. simpl. apply config.diff_nil.
          apply IHl. do 2 intro. apply H1. right. auto.
            destruct H0 as [?[?[?[]]]]. destruct H0; eauto.
            inversion H0. rewrite H5, H6 in n. contradiction.
        apply IHl. do 2 intro. apply H1. right. auto.
          destruct H0 as [?[?[?[]]]]. destruct H0; eauto.
          inversion H0. rewrite H5 in n. contradiction.
Defined.

(* any firing with a bag is either from the left petri net,
    from the right petri net, or the merge of one from each *)
Definition Firing_Some N1 N2 c B c' :
  petri.Firing (make N1 N2) c (Some B) c' ->
  {petri.Firing N1 (config.L c) (Some B) (config.L c') /\
    config.eq (config.R c) (config.R c')} +
  {config.eq (config.L c) (config.L c') /\
    petri.Firing N2 (config.R c) (Some B) (config.R c')} +
  {B1 & {B2 | B = bag.union B1 B2 /\
    petri.Firing N1 (config.L c) (Some B1) (config.L c') /\
    petri.Firing N2 (config.R c) (Some B2) (config.R c')}}.
Proof.
  intro. assert (exists ci co, In (ci, Some B, co) (petri.T (make N1 N2)) /\
    config.le ci c /\ config.eq (config.union (config.diff c ci) co) c').
    destruct H. eauto. simpl in H0. rewrite flat_map_concat_map in H0.
  assert (incl (petri.T N1) (petri.T N1)). apply incl_refl.
  generalize dependent H0. generalize dependent H1.
  generalize (petri.T N1) at 1 3. induction l; intros.
    simpl in H0. assert (incl (petri.T N2) (petri.T N2)). apply incl_refl.
      generalize dependent H0. generalize dependent H2.
      generalize (petri.T N2) at 1 3. induction l; intros.
        simpl in H0. right. clear H1 H2.
          assert (incl (petri.T N1) (petri.T N1)). apply incl_refl.
          generalize dependent H0. generalize dependent H1.
          generalize (petri.T N1) at 1 3. induction l; intros; simpl in H0.
            exfalso. destruct H0 as [?[?[]]]. auto.
            rewrite map_app, concat_app in H0.
              assert (incl (petri.T N2) (petri.T N2)). apply incl_refl.
              generalize dependent H0. generalize dependent H2.
              generalize (petri.T N2) at 1 3. induction l0; intros; simpl in H0.
                apply IHl. do 2 intro. apply H1. right. auto. auto.
                destruct a, p, o, a0, p, o; simpl in H0.
                  destruct (list_eq_dec event.dec B (bag.union t1 t4)).
                    destruct
                        (config.le_dec (map place.L t0 ++ map place.R t3) c).
                      destruct (config.dec (config.union (config.diff c
                          (map place.L t0 ++ map place.R t3))
                          (map place.L t ++ map place.R t2)) c').
                        exists t1, t4. split; auto. split.
                          econstructor; eauto. apply H1. apply in_eq.
                            rewrite <- config.L_map_L at 1. apply config.le_L.
                              eapply config.le_trans; eauto. do 2 intro.
                              apply in_or_app. auto.
                            apply config.eq_sym. eapply config.eq_trans.
                              apply config.eq_L. apply config.eq_sym. eauto.
                              rewrite <- config.union_L, <- config.diff_L.
                              repeat rewrite config.L_app. apply config.eq_refl.
                          econstructor; eauto. apply H2. apply in_eq.
                            rewrite <- config.R_map_R at 1. apply config.le_R.
                              eapply config.le_trans; eauto. do 2 intro.
                              apply in_or_app. auto.
                            apply config.eq_sym. eapply config.eq_trans.
                              apply config.eq_R. apply config.eq_sym. eauto.
                              rewrite <- config.union_R, <- config.diff_R.
                              repeat rewrite config.R_app. apply config.eq_refl.
                        apply IHl0. do 2 intro. apply H2. right. auto.
                          destruct H0 as [?[?[?[]]]]. destruct H0; eauto.
                          inversion H0. rewrite H6, H8 in n. contradiction.
                      apply IHl0. do 2 intro. apply H2. right. auto.
                        destruct H0 as [?[?[?[]]]]. destruct H0; eauto.
                        inversion H0. rewrite H6 in n. contradiction.
                    apply IHl0. do 2 intro. apply H2. right. auto.
                      destruct H0 as [?[?[?[]]]]. destruct H0; eauto.
                      inversion H0. rewrite H7 in n. contradiction.
                  apply IHl0. do 2 intro. apply H2. right. auto. auto.
                  apply IHl0. do 2 intro. apply H2. right. auto. auto.
                  apply IHl0. do 2 intro. apply H2. right. auto. auto.
        destruct a, p, o.
          destruct (list_eq_dec event.dec B t1).
            destruct (config.le_dec (map place.R t0) c).
              destruct (config.dec (config.union
                  (config.diff c (map place.R t0)) (map place.R t)) c').
                left. right. split.
                  apply config.eq_sym. eapply config.eq_trans.
                    apply config.eq_L. apply config.eq_sym. eauto.
                    rewrite <- config.union_L, <- config.diff_L.
                    repeat rewrite config.L_map_R. simpl. apply config.diff_nil.
                  econstructor; eauto. apply H2. rewrite e. apply in_eq.
                    rewrite <- config.R_map_R at 1. apply config.le_R. auto.
                    apply config.eq_sym. eapply config.eq_trans.
                      apply config.eq_R. apply config.eq_sym. eauto.
                      rewrite <- config.union_R, <- config.diff_R.
                      repeat rewrite config.R_map_R. apply config.eq_refl.
                apply IHl. do 2 intro. apply H2. right. auto.
                  destruct H0 as [?[?[?[]]]]. destruct H0; eauto. inversion H0.
                  rewrite H6, H8 in n. contradiction.
              apply IHl. do 2 intro. apply H2. right. auto.
                destruct H0 as [?[?[?[]]]]. destruct H0; eauto. inversion H0.
                rewrite H6 in n. contradiction.
            apply IHl. do 2 intro. apply H2. right. auto.
              destruct H0 as [?[?[?[]]]]. destruct H0; eauto. inversion H0.
              rewrite H7 in n. contradiction.
          apply IHl. do 2 intro. apply H2. right. auto.
            destruct H0 as [?[?[?[]]]]. destruct H0; eauto. inversion H0.
      destruct a, p, o.
        destruct (list_eq_dec event.dec B t1).
          destruct (config.le_dec (map place.L t0) c).
            destruct (config.dec (config.union
                (config.diff c (map place.L t0)) (map place.L t)) c').
              left. left. split.
                econstructor; eauto. apply H1. rewrite e. apply in_eq.
                  rewrite <- config.L_map_L at 1. apply config.le_L. auto.
                  apply config.eq_sym. eapply config.eq_trans.
                    apply config.eq_L. apply config.eq_sym. eauto.
                    rewrite <- config.union_L, <- config.diff_L.
                    repeat rewrite config.L_map_L. apply config.eq_refl.
                apply config.eq_sym. eapply config.eq_trans.
                  apply config.eq_R. apply config.eq_sym. eauto.
                  rewrite <- config.union_R, <- config.diff_R.
                  repeat rewrite config.R_map_L. simpl. apply config.diff_nil.
              apply IHl. do 2 intro. apply H1. right. auto.
                destruct H0 as [?[?[?[]]]]. destruct H0; eauto. inversion H0.
                rewrite H5, H7 in n. contradiction.
            apply IHl. do 2 intro. apply H1. right. auto.
              destruct H0 as [?[?[?[]]]]. destruct H0; eauto. inversion H0.
              rewrite H5 in n. contradiction.
          apply IHl. do 2 intro. apply H1. right. auto.
            destruct H0 as [?[?[?[]]]]. destruct H0; eauto. inversion H0.
            rewrite H6 in n. contradiction.
        apply IHl. do 2 intro. apply H1. right. auto.
          destruct H0 as [?[?[?[]]]]. destruct H0; eauto. inversion H0.
Defined.

(* any firing is either from the left petri net, from the right petri net,
    or some merge of one from each *)
Lemma Firing N1 N2 c w c' :
  petri.Firing (make N1 N2) c w c' ->
  (~ In place.P c -> ~ In place.P c') /\
  (petri.Firing N1 (config.L c) w (config.L c') /\
    config.eq (config.R c) (config.R c') \/
  config.eq (config.L c) (config.L c') /\
    petri.Firing N2 (config.R c) w (config.R c') \/
  exists w1 w2, petri.Firing N1 (config.L c) w1 (config.L c') /\
    petri.Firing N2 (config.R c) w2 (config.R c')).
Proof.
  intro. split.
    do 2 intro. apply H0. destruct H. edestruct config.In_union. destruct H4.
      apply H3. eauto. rewrite config.In_diff in H4. destruct H4. auto. exfalso.
      edestruct in_app_or. apply H.
        induction (petri.T N1); auto. inversion H6; auto. destruct a, p.
          inversion H7. rewrite <- H11 in H4. clear - H4. induction t; auto.
          inversion H4; auto. inversion H.
        edestruct in_app_or; eauto; clear H6.
          induction (petri.T N2); auto. inversion H7; auto. destruct a, p.
            inversion H6. rewrite <- H11 in H4. clear - H4. induction t; auto.
            inversion H4; auto. inversion H.
          rewrite in_flat_map in H7. destruct H7 as [?[]]. destruct x.
            rewrite in_prod_iff in H6. destruct H6.
            rewrite in_map_iff in H6, H8. destruct H6 as [?[]], H8 as [?[]].
            rewrite <- H6, <- H8 in H7.
            destruct x, p1, o, x0, p1, o; simpl in H7; auto. destruct H7; auto.
            inversion H7. rewrite <- H14 in H4. edestruct config.eq_app_r.
            destruct H11. apply config.eq_refl. apply H11. eauto.
    destruct w. edestruct Firing_Some as [[]|[?[?[]]]]; eauto.
      edestruct Firing_None; eauto.
Qed.

(* ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~ *)

(* the concatentation of any empty firing chain from the left petri net with any
    from the right is in the parallel petri net *)
Lemma Efchain N1 N2 c1 c2 l1 l2 c1' c2' :
  petri.Efchain N1 c1 l1 c1' -> petri.Efchain N2 c2 l2 c2' ->
  petri.Efchain (make N1 N2) (map place.L c1 ++ map place.R c2)
    (map (fun c1 => map place.L c1 ++ map place.R c2) l1 ++
      map (fun c2 => map place.L c1' ++ map place.R c2) l2)
    (map place.L c1' ++ map place.R c2').
Proof.
  intros. apply petri.Efchain_trans with (map place.L c1' ++ map place.R c2).
    induction H; constructor; auto. apply config.eq_app. split; auto.
      apply config.eq_refl. apply Firing_L; auto.
    induction H0; constructor; auto. apply config.eq_app. split; auto.
      apply config.eq_refl. apply Firing_R; auto.
Qed.

(* helper function for Efchain *)
Lemma Efchain_app_r N1 N2 c l1 l2 c1' c2' :
  ~ In place.P c -> petri.Efchain N1 (config.L c) l1 c1' ->
  petri.Efchain N2 (config.R c) l2 c2' ->
  petri.Efchain (make N1 N2) c
    (map (fun c1 => map place.L c1 ++ map place.R (config.R c)) l1 ++
      map (fun c2 => map place.L c1' ++ map place.R c2) l2)
        (map place.L c1' ++ map place.R c2').
Proof.
  intros. eapply petri.Efchain_eq_l. eapply Efchain; eauto.
  apply config.eq_sym, config.eq_app_r. split; auto.
  split; apply config.eq_refl.
Qed.

(* any empty firing chain consists of one from the left petri net and one from
    the right *)
Lemma Efchain_L_R N1 N2 c l c' :
  petri.Efchain (make N1 N2) c l c' ->
  {l1 & {l2 | petri.Efchain N1 (config.L c) l1 (config.L c') /\
    petri.Efchain N2 (config.R c) l2 (config.R c')}}.
Proof.
  generalize dependent c. induction l; intros.
    exists [], []. inversion H. split; constructor.
      apply config.eq_L. auto.
      apply config.eq_R. auto.
    edestruct IHl as [?[?[]]]. inversion H. eauto.
      assert (petri.Firing (make N1 N2) c None a). inversion H. auto.
      edestruct Firing_None as [[]|[]]; eauto.
        exists (config.L a :: x), x0. split; auto. constructor; auto.
          eapply petri.Efchain_eq_l; eauto. apply config.eq_sym. auto.
        exists x, (config.R a :: x0). split; auto.
          eapply petri.Efchain_eq_l; eauto. apply config.eq_sym. auto.
          constructor; auto.
Qed.

(* ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~ *)

(* any reachable configuration does not include the P place,
    its left part is reachable in the left petri net, and
    its right part is reachable in the right petri net *)
Lemma Reach N1 N2 c : petri.Reach (make N1 N2) c ->
  ~ In place.P c /\ petri.Reach N1 (config.L c) /\ petri.Reach N2 (config.R c).
Proof.
  intro. induction H.
    simpl in H. unfold init in H. rewrite config.eq_app_r in H.
      destruct H as [?[]]. split; auto. split; constructor; auto.
    destruct IHReach as [?[]]. edestruct Firing as [?[[]|[[]|[?[?[]]]]]]; eauto;
    split; auto.
      split. econstructor 2; eauto. eapply petri.Reach_eq; eauto.
      split. eapply petri.Reach_eq; eauto. econstructor 2; eauto.
      split; econstructor 2; eauto.
Qed.

(* the merge of any reachable configuration from the left petri net with any
    from the right is reachable in the parallel petri net *)
Lemma Reach_L_R N1 N2 c1 c2 :
  petri.Reach N1 c1 -> petri.Reach N2 c2 ->
  petri.Reach (make N1 N2) (map place.L c1 ++ map place.R c2).
Proof.
  intros. induction H.
    induction H0. constructor. simpl. unfold init. apply config.eq_app. auto.
      econstructor 2; eauto. apply Firing_R; eauto.
    econstructor 2; eauto. eapply Firing_L; eauto.
Qed.

(* the parallel petri net is safe *)
Lemma Safe N1 N2 : petri.Safe N1 -> petri.Safe N2 -> petri.Safe (make N1 N2).
Proof.
  do 5 intro. edestruct Reach as [?[]]; eauto. simpl in *. unfold fin in *.
  apply config.le_app_r. split; auto. rewrite config.le_app_l in H2.
  destruct H2; auto.
Qed.

(* ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~ *)

(* the left part of a trace from a full path and trace *)
Definition opttrace_l N1 N2 :
  forall c pi ot, petri.Gen (make N1 N2) c pi ot -> opttrace.t.
Proof.
  cofix F. intros. destruct pi. apply opttrace.eps. destruct ot.
    apply opttrace.eps.
    destruct o.
      assert (petri.Firing (make N1 N2) (last l c) (Some t0) t).
        inversion H. apply petri.Firing_eq_l. eexists. split; eauto.
        apply config.eq_sym. eapply petri.Efchain_last. eauto.
        edestruct Firing_Some as [[]|[?[]]]. eauto.
          apply (opttrace.opt (Some t0)). eapply F. inversion H; eauto.
          apply (opttrace.opt None). eapply F. inversion H; eauto.
          apply (opttrace.opt (Some x)). eapply F. inversion H; eauto.
      apply (opttrace.opt None). eapply F. inversion H; eauto.
Defined.

Lemma opttrace_l_eps N1 N2 c l E :
  @opttrace_l N1 N2 c (path.fin l) opttrace.eps (petri.Gen_eps E) =
  opttrace.eps.
Proof.
  rewrite opttrace.match_ at 1. auto.
Qed.

Lemma opttrace_l_opt_None N1 N2 c l c' pi ot E G :
  @opttrace_l N1 N2 c (path.hop l c' pi) (opttrace.opt None ot)
    (@petri.Gen_None _ c l c' pi ot E G) =
  opttrace.opt None (opttrace_l G).
Proof.
  rewrite opttrace.match_ at 1. auto.
Qed.

Lemma opttrace_l_opt_Some N1 N2 c l c' B c'' pi ot E F G :
  @opttrace_l N1 N2 c (path.hop l c'' pi) (opttrace.opt (Some B) ot)
    (@petri.Gen_Some _ c l c' B c'' pi ot E F G) =
  match
    Firing_Some
      (eq_ind_r
         (fun l0 : list config.t =>
          c'' = c'' ->
          pi = pi ->
          opttrace.opt (Some B) ot = opttrace.opt (Some B) ot ->
          petri.Efchain (make N1 N2) c l0 c' ->
          petri.Firing (make N1 N2) c' (Some B) c'' ->
          petri.Gen (make N1 N2) c'' pi ot ->
          petri.Firing (make N1 N2) (last l c) (Some B) c'')
         (fun H1 : c'' = c'' =>
          eq_ind_r
            (fun t : config.t =>
             pi = pi ->
             opttrace.opt (Some B) ot = opttrace.opt (Some B) ot ->
             petri.Efchain (make N1 N2) c l c' ->
             petri.Firing (make N1 N2) c' (Some B) t ->
             petri.Gen (make N1 N2) t pi ot ->
             petri.Firing (make N1 N2) (last l c) (Some B) c'')
            (fun H2 : pi = pi =>
             eq_ind_r
               (fun t : path.t =>
                opttrace.opt (Some B) ot = opttrace.opt (Some B) ot ->
                petri.Efchain (make N1 N2) c l c' ->
                petri.Firing (make N1 N2) c' (Some B) c'' ->
                petri.Gen (make N1 N2) c'' t ot ->
                petri.Firing (make N1 N2) (last l c) (Some B) c'')
               (fun H4 : opttrace.opt (Some B) ot = opttrace.opt (Some B) ot =>
                eq_ind_r
                  (fun t : bag.t =>
                   ot = ot ->
                   petri.Efchain (make N1 N2) c l c' ->
                   petri.Firing (make N1 N2) c' (Some t) c'' ->
                   petri.Gen (make N1 N2) c'' pi ot ->
                   petri.Firing (make N1 N2) (last l c) (Some B) c'')
                  (fun H5 : ot = ot =>
                   eq_ind_r
                     (fun t : opttrace.t =>
                      petri.Efchain (make N1 N2) c l c' ->
                      petri.Firing (make N1 N2) c' (Some B) c'' ->
                      petri.Gen (make N1 N2) c'' pi t ->
                      petri.Firing (make N1 N2) (last l c) (Some B) c'')
                     (fun (H0 : petri.Efchain (make N1 N2) c l c')
                        (H6 : petri.Firing (make N1 N2) c' (Some B) c'')
                        (_ : petri.Gen (make N1 N2) c'' pi ot) =>
                      petri.Firing_eq_l
                        (ex_intro
                           (fun c''0 : config.t =>
                            petri.Firing (make N1 N2) c''0 (Some B) c'' /\
                            config.eq (last l c) c''0) c'
                           (conj H6 (config.eq_sym (petri.Efchain_last H0)))))
                            H5)
                  (f_equal
                     (fun e : opttrace.t =>
                      match e with
                      | opttrace.opt (Some t0) _ => t0
                      | _ => B
                      end) H4)
                  (f_equal
                     (fun e : opttrace.t =>
                      match e with
                      | opttrace.eps => ot
                      | opttrace.opt _ t => t
                      end) H4)) H2) H1) eq_refl eq_refl eq_refl eq_refl E F G)
  with
  | inl (left _) => opttrace.opt (Some B) (opttrace_l G)
  | inl (right _) => opttrace.opt None (opttrace_l G)
  | inr (existT _ x (exist _ _ _)) => opttrace.opt (Some x) (opttrace_l G)
  end.
Proof.
  rewrite opttrace.match_ at 1. simpl. destruct Firing_Some as [[]|[?[]]]; auto.
Qed.

(* ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~ *)

(* the right part of a trace from a full path and trace *)
Definition opttrace_r N1 N2 :
  forall c pi ot, petri.Gen (make N1 N2) c pi ot -> opttrace.t.
Proof.
  cofix F. intros. destruct pi. apply opttrace.eps. destruct ot.
    apply opttrace.eps.
    destruct o.
      assert (petri.Firing (make N1 N2) (last l c) (Some t0) t).
        inversion H. apply petri.Firing_eq_l. eexists. split; eauto.
        apply config.eq_sym. eapply petri.Efchain_last. eauto.
        edestruct Firing_Some as [[]|[?[]]]. eauto.
          apply (opttrace.opt None). eapply F. inversion H; eauto.
          apply (opttrace.opt (Some t0)). eapply F. inversion H; eauto.
          apply (opttrace.opt (Some x0)). eapply F. inversion H; eauto.
      apply (opttrace.opt None). eapply F. inversion H; eauto.
Defined.

Lemma opttrace_r_eps N1 N2 c l E :
  @opttrace_r N1 N2 c (path.fin l) opttrace.eps (petri.Gen_eps E) =
  opttrace.eps.
Proof.
  rewrite opttrace.match_ at 1. auto.
Qed.

Lemma opttrace_r_opt_None N1 N2 c l c' pi ot E G :
  @opttrace_r N1 N2 c (path.hop l c' pi) (opttrace.opt None ot)
    (@petri.Gen_None _ c l c' pi ot E G) =
  opttrace.opt None (opttrace_r G).
Proof.
  rewrite opttrace.match_ at 1. auto.
Qed.

Lemma opttrace_r_opt_Some N1 N2 c l c' B c'' pi ot E F G :
  @opttrace_r N1 N2 c (path.hop l c'' pi) (opttrace.opt (Some B) ot)
    (@petri.Gen_Some _ c l c' B c'' pi ot E F G) =
  match
    Firing_Some
      (eq_ind_r
         (fun l0 : list config.t =>
          c'' = c'' ->
          pi = pi ->
          opttrace.opt (Some B) ot = opttrace.opt (Some B) ot ->
          petri.Efchain (make N1 N2) c l0 c' ->
          petri.Firing (make N1 N2) c' (Some B) c'' ->
          petri.Gen (make N1 N2) c'' pi ot ->
          petri.Firing (make N1 N2) (last l c) (Some B) c'')
         (fun H1 : c'' = c'' =>
          eq_ind_r
            (fun t : config.t =>
             pi = pi ->
             opttrace.opt (Some B) ot = opttrace.opt (Some B) ot ->
             petri.Efchain (make N1 N2) c l c' ->
             petri.Firing (make N1 N2) c' (Some B) t ->
             petri.Gen (make N1 N2) t pi ot ->
             petri.Firing (make N1 N2) (last l c) (Some B) c'')
            (fun H2 : pi = pi =>
             eq_ind_r
               (fun t : path.t =>
                opttrace.opt (Some B) ot = opttrace.opt (Some B) ot ->
                petri.Efchain (make N1 N2) c l c' ->
                petri.Firing (make N1 N2) c' (Some B) c'' ->
                petri.Gen (make N1 N2) c'' t ot ->
                petri.Firing (make N1 N2) (last l c) (Some B) c'')
               (fun H4 : opttrace.opt (Some B) ot = opttrace.opt (Some B) ot =>
                eq_ind_r
                  (fun t : bag.t =>
                   ot = ot ->
                   petri.Efchain (make N1 N2) c l c' ->
                   petri.Firing (make N1 N2) c' (Some t) c'' ->
                   petri.Gen (make N1 N2) c'' pi ot ->
                   petri.Firing (make N1 N2) (last l c) (Some B) c'')
                  (fun H5 : ot = ot =>
                   eq_ind_r
                     (fun t : opttrace.t =>
                      petri.Efchain (make N1 N2) c l c' ->
                      petri.Firing (make N1 N2) c' (Some B) c'' ->
                      petri.Gen (make N1 N2) c'' pi t ->
                      petri.Firing (make N1 N2) (last l c) (Some B) c'')
                     (fun (H0 : petri.Efchain (make N1 N2) c l c')
                        (H6 : petri.Firing (make N1 N2) c' (Some B) c'')
                        (_ : petri.Gen (make N1 N2) c'' pi ot) =>
                      petri.Firing_eq_l
                        (ex_intro
                           (fun c''0 : config.t =>
                            petri.Firing (make N1 N2) c''0 (Some B) c'' /\
                            config.eq (last l c) c''0) c'
                           (conj H6 (config.eq_sym (petri.Efchain_last H0)))))
                            H5)
                  (f_equal
                     (fun e : opttrace.t =>
                      match e with
                      | opttrace.opt (Some t0) _ => t0
                      | _ => B
                      end) H4)
                  (f_equal
                     (fun e : opttrace.t =>
                      match e with
                      | opttrace.eps => ot
                      | opttrace.opt _ t => t
                      end) H4)) H2) H1) eq_refl eq_refl eq_refl eq_refl E F G)
  with
  | inl (left _) => opttrace.opt None (opttrace_r G)
  | inl (right _) => opttrace.opt (Some B) (opttrace_r G)
  | inr (existT _ x (exist _ x0 _)) => opttrace.opt (Some x0) (opttrace_r G)
  end.
Proof.
  rewrite opttrace.match_ at 1. simpl. destruct Firing_Some as [[]|[?[]]]; auto.
Qed.


(* ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~ *)

(* the left part of a path from a full path and trace *)
Definition path_l N1 N2 :
  forall c pi ot, petri.Gen (make N1 N2) c pi ot -> path.t.
Proof.
  cofix F. intros. destruct pi, ot.
    assert (petri.Efchain (make N1 N2) c l (petri.fin (make N1 N2))).
      inversion H. auto. edestruct Efchain_L_R as [?[]]. eauto.
      apply (path.fin x).
    apply (path.fin []).
    apply (path.fin []).
    assert (petri.Efchain (make N1 N2) c l (last l c)). inversion H.
      apply petri.Efchain_eq_r. eexists. split; eauto.
        eapply petri.Efchain_last. eauto.
      apply petri.Efchain_eq_r. eexists. split; eauto.
        eapply petri.Efchain_last. eauto.
      edestruct Efchain_L_R as [?[]]. eauto.
      apply (path.hop x (config.L t)). eapply F. inversion H; eauto.
Defined.

Lemma path_l_eps N1 N2 c l E :
  @path_l N1 N2 c (path.fin l) opttrace.eps (petri.Gen_eps E) =
  (let (x, s) :=
     Efchain_L_R
       (eq_ind_r
          (fun l0 : list config.t =>
           opttrace.eps = opttrace.eps ->
           petri.Efchain (make N1 N2) c l0 (fin (petri.fin N1) (petri.fin N2))
            ->
           petri.Efchain (make N1 N2) c l (fin (petri.fin N1) (petri.fin N2)))
          (fun (_ : opttrace.eps = opttrace.eps)
             (H1 : petri.Efchain (make N1 N2) c l
              (fin (petri.fin N1) (petri.fin N2))) => H1)
          eq_refl eq_refl E) in
   let (x0, _) := s in path.fin x).
Proof.
  rewrite path.match_ at 1. simpl. destruct Efchain_L_R as [?[]]. auto.
Qed.

Lemma path_l_opt_None N1 N2 c l c' pi ot E G :
  @path_l N1 N2 c (path.hop l c' pi) (opttrace.opt None ot)
    (@petri.Gen_None _ c l c' pi ot E G) =
  (let (x, s) :=
     Efchain_L_R
       (eq_ind_r
          (fun l0 : list config.t =>
           c' = c' ->
           pi = pi ->
           opttrace.opt None ot = opttrace.opt None ot ->
           petri.Efchain (make N1 N2) c l0 c' ->
           petri.Gen (make N1 N2) c' pi ot ->
           petri.Efchain (make N1 N2) c l (last l c))
          (fun H1 : c' = c' =>
           eq_ind_r
             (fun t : config.t =>
              pi = pi ->
              opttrace.opt None ot = opttrace.opt None ot ->
              petri.Efchain (make N1 N2) c l t ->
              petri.Gen (make N1 N2) t pi ot ->
              petri.Efchain (make N1 N2) c l (last l c))
             (fun H3 : pi = pi =>
              eq_ind_r
                (fun t : path.t =>
                 opttrace.opt None ot = opttrace.opt None ot ->
                 petri.Efchain (make N1 N2) c l c' ->
                 petri.Gen (make N1 N2) c' t ot ->
                 petri.Efchain (make N1 N2) c l (last l c))
                (fun H2 : opttrace.opt None ot = opttrace.opt None ot =>
                 eq_ind None
                   (fun _ : option bag.t =>
                    ot = ot ->
                    petri.Efchain (make N1 N2) c l c' ->
                    petri.Gen (make N1 N2) c' pi ot ->
                    petri.Efchain (make N1 N2) c l (last l c))
                   (fun H5 : ot = ot =>
                    eq_ind_r
                      (fun t : opttrace.t =>
                       petri.Efchain (make N1 N2) c l c' ->
                       petri.Gen (make N1 N2) c' pi t ->
                       petri.Efchain (make N1 N2) c l (last l c))
                      (fun (H0 : petri.Efchain (make N1 N2) c l c')
                         (_ : petri.Gen (make N1 N2) c' pi ot) =>
                       petri.Efchain_eq_r
                         (ex_intro
                            (fun c'' : config.t =>
                             petri.Efchain (make N1 N2) c l c'' /\
                             config.eq c'' (last l c))
                            c' (conj H0 (petri.Efchain_last H0)))) H5) None
                   (f_equal
                      (fun e : opttrace.t =>
                       match e with
                       | opttrace.eps => None
                       | opttrace.opt o _ => o
                       end) H2)
                   (f_equal
                      (fun e : opttrace.t =>
                       match e with
                       | opttrace.eps => ot
                       | opttrace.opt _ t => t
                       end) H2)) H3) H1) eq_refl eq_refl eq_refl eq_refl E G) in
   let (x0, _) := s in
   path.hop x (config.L c') (path_l G)).
Proof.
  rewrite path.match_ at 1. simpl. destruct Efchain_L_R as [?[]]. auto.
Qed.

Lemma path_l_opt_Some N1 N2 c l c' B c'' pi ot E F G :
  @path_l N1 N2 c (path.hop l c'' pi) (opttrace.opt (Some B) ot)
    (@petri.Gen_Some _ c l c' B c'' pi ot E F G) =
  (let (x, s) :=
     Efchain_L_R
       (eq_ind_r
          (fun l0 : list config.t =>
           c'' = c'' ->
           pi = pi ->
           opttrace.opt (Some B) ot = opttrace.opt (Some B) ot ->
           petri.Efchain (make N1 N2) c l0 c' ->
           petri.Firing (make N1 N2) c' (Some B) c'' ->
           petri.Gen (make N1 N2) c'' pi ot ->
           petri.Efchain (make N1 N2) c l (last l c))
          (fun H1 : c'' = c'' =>
           eq_ind_r
             (fun t : config.t =>
              pi = pi ->
              opttrace.opt (Some B) ot = opttrace.opt (Some B) ot ->
              petri.Efchain (make N1 N2) c l c' ->
              petri.Firing (make N1 N2) c' (Some B) t ->
              petri.Gen (make N1 N2) t pi ot ->
              petri.Efchain (make N1 N2) c l (last l c))
             (fun H2 : pi = pi =>
              eq_ind_r
                (fun t : path.t =>
                 opttrace.opt (Some B) ot = opttrace.opt (Some B) ot ->
                 petri.Efchain (make N1 N2) c l c' ->
                 petri.Firing (make N1 N2) c' (Some B) c'' ->
                 petri.Gen (make N1 N2) c'' t ot ->
                 petri.Efchain (make N1 N2) c l (last l c))
                (fun H4 : opttrace.opt (Some B) ot = opttrace.opt (Some B) ot =>
                 eq_ind (Some B)
                   (fun _ : option bag.t =>
                    ot = ot ->
                    petri.Efchain (make N1 N2) c l c' ->
                    petri.Firing (make N1 N2) c' (Some B) c'' ->
                    petri.Gen (make N1 N2) c'' pi ot ->
                    petri.Efchain (make N1 N2) c l (last l c))
                   (fun H5 : ot = ot =>
                    eq_ind_r
                      (fun t : opttrace.t =>
                       petri.Efchain (make N1 N2) c l c' ->
                       petri.Firing (make N1 N2) c' (Some B) c'' ->
                       petri.Gen (make N1 N2) c'' pi t ->
                       petri.Efchain (make N1 N2) c l (last l c))
                      (fun (H0 : petri.Efchain (make N1 N2) c l c')
                         (_ : petri.Firing (make N1 N2) c' (Some B) c'')
                         (_ : petri.Gen (make N1 N2) c'' pi ot) =>
                       petri.Efchain_eq_r
                         (ex_intro
                            (fun c''0 : config.t =>
                             petri.Efchain (make N1 N2) c l c''0 /\
                             config.eq c''0 (last l c))
                            c' (conj H0 (petri.Efchain_last H0)))) H5) 
                   (Some B)
                   (f_equal
                      (fun e : opttrace.t =>
                       match e with
                       | opttrace.eps => Some B
                       | opttrace.opt o _ => o
                       end) H4)
                   (f_equal
                      (fun e : opttrace.t =>
                       match e with
                       | opttrace.eps => ot
                       | opttrace.opt _ t => t
                       end) H4)) H2) H1) eq_refl eq_refl eq_refl eq_refl E F G)
   in let (x0, _) := s in
   path.hop x (config.L c'') (path_l G)).
Proof.
  rewrite path.match_ at 1. simpl. destruct Efchain_L_R as [?[]]. auto.
Qed.

(* ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~ *)

(* the right part of a path from a full path and trace *)
Definition path_r N1 N2 :
  forall c pi ot, petri.Gen (make N1 N2) c pi ot -> path.t.
Proof.
  cofix F. intros. destruct pi, ot.
    assert (petri.Efchain (make N1 N2) c l (petri.fin (make N1 N2))).
      inversion H. auto. edestruct Efchain_L_R as [?[]]. eauto.
      apply (path.fin x0).
    apply (path.fin []).
    apply (path.fin []).
    assert (petri.Efchain (make N1 N2) c l (last l c)). inversion H.
      apply petri.Efchain_eq_r. eexists. split; eauto.
        eapply petri.Efchain_last. eauto.
      apply petri.Efchain_eq_r. eexists. split; eauto.
        eapply petri.Efchain_last. eauto.
      edestruct Efchain_L_R as [?[]]. eauto.
      apply (path.hop x0 (config.R t)). eapply F. inversion H; eauto.
Defined.

Lemma path_r_eps N1 N2 c l E :
  @path_r N1 N2 c (path.fin l) opttrace.eps (petri.Gen_eps E) =
  (let (x, s) :=
     Efchain_L_R
       (eq_ind_r
          (fun l0 : list config.t =>
           opttrace.eps = opttrace.eps ->
           petri.Efchain (make N1 N2) c l0 (fin (petri.fin N1) (petri.fin N2))
            ->
           petri.Efchain (make N1 N2) c l (fin (petri.fin N1) (petri.fin N2)))
          (fun (_ : opttrace.eps = opttrace.eps)
             (H1 : petri.Efchain (make N1 N2) c l
              (fin (petri.fin N1) (petri.fin N2))) => H1)
          eq_refl eq_refl E) in
   let (x0, _) := s in path.fin x0).
Proof.
  rewrite path.match_ at 1. simpl. destruct Efchain_L_R as [?[]]. auto.
Qed.

Lemma path_r_opt_None N1 N2 c l c' pi ot E G :
  @path_r N1 N2 c (path.hop l c' pi) (opttrace.opt None ot)
    (@petri.Gen_None _ c l c' pi ot E G) =
  (let (x, s) :=
     Efchain_L_R
       (eq_ind_r
          (fun l0 : list config.t =>
           c' = c' ->
           pi = pi ->
           opttrace.opt None ot = opttrace.opt None ot ->
           petri.Efchain (make N1 N2) c l0 c' ->
           petri.Gen (make N1 N2) c' pi ot ->
           petri.Efchain (make N1 N2) c l (last l c))
          (fun H1 : c' = c' =>
           eq_ind_r
             (fun t : config.t =>
              pi = pi ->
              opttrace.opt None ot = opttrace.opt None ot ->
              petri.Efchain (make N1 N2) c l t ->
              petri.Gen (make N1 N2) t pi ot ->
              petri.Efchain (make N1 N2) c l (last l c))
             (fun H3 : pi = pi =>
              eq_ind_r
                (fun t : path.t =>
                 opttrace.opt None ot = opttrace.opt None ot ->
                 petri.Efchain (make N1 N2) c l c' ->
                 petri.Gen (make N1 N2) c' t ot ->
                 petri.Efchain (make N1 N2) c l (last l c))
                (fun H2 : opttrace.opt None ot = opttrace.opt None ot =>
                 eq_ind None
                   (fun _ : option bag.t =>
                    ot = ot ->
                    petri.Efchain (make N1 N2) c l c' ->
                    petri.Gen (make N1 N2) c' pi ot ->
                    petri.Efchain (make N1 N2) c l (last l c))
                   (fun H5 : ot = ot =>
                    eq_ind_r
                      (fun t : opttrace.t =>
                       petri.Efchain (make N1 N2) c l c' ->
                       petri.Gen (make N1 N2) c' pi t ->
                       petri.Efchain (make N1 N2) c l (last l c))
                      (fun (H0 : petri.Efchain (make N1 N2) c l c')
                         (_ : petri.Gen (make N1 N2) c' pi ot) =>
                       petri.Efchain_eq_r
                         (ex_intro
                            (fun c'' : config.t =>
                             petri.Efchain (make N1 N2) c l c'' /\
                             config.eq c'' (last l c))
                            c' (conj H0 (petri.Efchain_last H0)))) H5) None
                   (f_equal
                      (fun e : opttrace.t =>
                       match e with
                       | opttrace.eps => None
                       | opttrace.opt o _ => o
                       end) H2)
                   (f_equal
                      (fun e : opttrace.t =>
                       match e with
                       | opttrace.eps => ot
                       | opttrace.opt _ t => t
                       end) H2)) H3) H1) eq_refl eq_refl eq_refl eq_refl E G) in
   let (x0, _) := s in
   path.hop x0 (config.R c') (path_r G)).
Proof.
  rewrite path.match_ at 1. simpl. destruct Efchain_L_R as [?[]]. auto.
Qed.

Lemma path_r_opt_Some N1 N2 c l c' B c'' pi ot E F G :
  @path_r N1 N2 c (path.hop l c'' pi) (opttrace.opt (Some B) ot)
    (@petri.Gen_Some _ c l c' B c'' pi ot E F G) =
  (let (x, s) :=
     Efchain_L_R
       (eq_ind_r
          (fun l0 : list config.t =>
           c'' = c'' ->
           pi = pi ->
           opttrace.opt (Some B) ot = opttrace.opt (Some B) ot ->
           petri.Efchain (make N1 N2) c l0 c' ->
           petri.Firing (make N1 N2) c' (Some B) c'' ->
           petri.Gen (make N1 N2) c'' pi ot ->
           petri.Efchain (make N1 N2) c l (last l c))
          (fun H1 : c'' = c'' =>
           eq_ind_r
             (fun t : config.t =>
              pi = pi ->
              opttrace.opt (Some B) ot = opttrace.opt (Some B) ot ->
              petri.Efchain (make N1 N2) c l c' ->
              petri.Firing (make N1 N2) c' (Some B) t ->
              petri.Gen (make N1 N2) t pi ot ->
              petri.Efchain (make N1 N2) c l (last l c))
             (fun H2 : pi = pi =>
              eq_ind_r
                (fun t : path.t =>
                 opttrace.opt (Some B) ot = opttrace.opt (Some B) ot ->
                 petri.Efchain (make N1 N2) c l c' ->
                 petri.Firing (make N1 N2) c' (Some B) c'' ->
                 petri.Gen (make N1 N2) c'' t ot ->
                 petri.Efchain (make N1 N2) c l (last l c))
                (fun H4 : opttrace.opt (Some B) ot = opttrace.opt (Some B) ot =>
                 eq_ind (Some B)
                   (fun _ : option bag.t =>
                    ot = ot ->
                    petri.Efchain (make N1 N2) c l c' ->
                    petri.Firing (make N1 N2) c' (Some B) c'' ->
                    petri.Gen (make N1 N2) c'' pi ot ->
                    petri.Efchain (make N1 N2) c l (last l c))
                   (fun H5 : ot = ot =>
                    eq_ind_r
                      (fun t : opttrace.t =>
                       petri.Efchain (make N1 N2) c l c' ->
                       petri.Firing (make N1 N2) c' (Some B) c'' ->
                       petri.Gen (make N1 N2) c'' pi t ->
                       petri.Efchain (make N1 N2) c l (last l c))
                      (fun (H0 : petri.Efchain (make N1 N2) c l c')
                         (_ : petri.Firing (make N1 N2) c' (Some B) c'')
                         (_ : petri.Gen (make N1 N2) c'' pi ot) =>
                       petri.Efchain_eq_r
                         (ex_intro
                            (fun c''0 : config.t =>
                             petri.Efchain (make N1 N2) c l c''0 /\
                             config.eq c''0 (last l c))
                            c' (conj H0 (petri.Efchain_last H0)))) H5) 
                   (Some B)
                   (f_equal
                      (fun e : opttrace.t =>
                       match e with
                       | opttrace.eps => Some B
                       | opttrace.opt o _ => o
                       end) H4)
                   (f_equal
                      (fun e : opttrace.t =>
                       match e with
                       | opttrace.eps => ot
                       | opttrace.opt _ t => t
                       end) H4)) H2) H1) eq_refl eq_refl eq_refl eq_refl E F G)
   in let (x0, _) := s in
   path.hop x0 (config.R c'') (path_r G)).
Proof.
  rewrite path.match_ at 1. simpl. destruct Efchain_L_R as [?[]]. auto.
Qed.

(* ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~ *)

CoFixpoint path_merge fin1 c1 c2 pi1 pi2 : path.t :=
  match pi1, pi2 with
  | path.fin l1, path.fin l2 =>
      path.fin
        (map (fun c1 => map place.L c1 ++ map place.R c2) l1 ++
        map (fun c2 => map place.L fin1 ++ map place.R c2) l2)
  | path.hop l1 c1' pi1', path.hop l2 c2' pi2' =>
      path.hop
        (map (fun c1 => map place.L c1 ++ map place.R c2) l1 ++
        map (fun c2 => map place.L (last l1 c1) ++ map place.R c2) l2)
        (map place.L c1' ++ map place.R c2') (path_merge fin1 c1' c2' pi1' pi2')
  | _, _ => path.fin []
  end.

Lemma path_merge_fin_fin fin1 c1 c2 l1 l2 :
  path_merge fin1 c1 c2 (path.fin l1) (path.fin l2) =
      path.fin
        (map (fun c1 => map place.L c1 ++ map place.R c2) l1 ++
        map (fun c2 => map place.L fin1 ++ map place.R c2) l2).
Proof.
  rewrite path.match_ at 1. auto.
Qed.

Lemma path_merge_hop_hop fin1 c1 c2 l1 l2 c1' c2' pi1 pi2 :
  path_merge fin1 c1 c2 (path.hop l1 c1' pi1) (path.hop l2 c2' pi2) =
      path.hop
        (map (fun c1 => map place.L c1 ++ map place.R c2) l1 ++
        map (fun c2 => map place.L (last l1 c1) ++ map place.R c2) l2)
        (map place.L c1' ++ map place.R c2') (path_merge fin1 c1' c2' pi1 pi2).
Proof.
  rewrite path.match_ at 1. auto.
Qed.

(* ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~ *)

(* opttrace_l is generated by the left petri net *)
Lemma Gen_opttrace_l N1 N2 :
  forall c pi ot G,
  petri.Gen N1 (config.L c) (@path_l N1 N2 c pi ot G) (opttrace_l G).
Proof.
  cofix F. intros. destruct G.
    rewrite path_l_eps. destruct Efchain_L_R as [?[?[]]].
      rewrite opttrace_l_eps. constructor. unfold fin in H.
      rewrite config.L_app in H. auto.
    rewrite path_l_opt_None. destruct Efchain_L_R as [?[?[]]].
      rewrite opttrace_l_opt_None. constructor; auto. apply petri.Efchain_eq_r.
      eexists. split; eauto. apply config.eq_L. apply config.eq_sym.
      eapply petri.Efchain_last; eauto.
    rewrite path_l_opt_Some. destruct Efchain_L_R as [?[?[]]].
      rewrite opttrace_l_opt_Some.
      destruct Firing_Some as [[[]|[]]|[?[?[?[]]]]]; econstructor; eauto.
      apply petri.Efchain_eq_r. eexists. split; eauto.
Qed.

(* opttrace_r is generated by the right petri net *)
Lemma Gen_opttrace_r N1 N2 :
  forall c pi ot G,
  petri.Gen N2 (config.R c) (@path_r N1 N2 c pi ot G) (opttrace_r G).
Proof.
  cofix F. intros. destruct G.
    rewrite path_r_eps. destruct Efchain_L_R as [?[?[]]].
      rewrite opttrace_r_eps. constructor. unfold fin in H0.
      rewrite config.R_app in H0. auto.
    rewrite path_r_opt_None. destruct Efchain_L_R as [?[?[]]].
      rewrite opttrace_r_opt_None. constructor; auto. apply petri.Efchain_eq_r.
      eexists. split; eauto. apply config.eq_R. apply config.eq_sym.
      eapply petri.Efchain_last; eauto.
    rewrite path_r_opt_Some. destruct Efchain_L_R as [?[?[]]].
      rewrite opttrace_r_opt_Some.
      destruct Firing_Some as [[[]|[]]|[?[?[?[]]]]]; econstructor; eauto.
      apply petri.Efchain_eq_r. eexists. split; eauto.
Qed.

(* ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~ *)

(* Lang soundness *)
Lemma Lang_sound N1 N2 :
  slang.le (petri.Lang (make N1 N2))
    (slang.concur (petri.Lang N1) (petri.Lang N2)).
Proof.
  do 2 intro. destruct H. exists (opttrace_l H), (opttrace_r H). split.
    eexists. erewrite <- config.L_app at 1. apply Gen_opttrace_l. split.
    eexists. erewrite <- config.R_app at 1. apply Gen_opttrace_r.
  generalize dependent ot. generalize dependent x.
  generalize (petri.init (make N1 N2)). cofix F. intros.
  destruct H.
    rewrite opttrace_l_eps, opttrace_r_eps. constructor.
    rewrite opttrace_l_opt_None, opttrace_r_opt_None. constructor; auto.
      constructor.
    rewrite opttrace_l_opt_Some, opttrace_r_opt_Some.
      destruct Firing_Some as [[[]|[]]|[?[?[?[]]]]]; econstructor; eauto.
      constructor. constructor. rewrite H0. constructor.
Qed.

(* Lang completeness *)
Lemma Lang_complete N1 N2 :
  slang.le (slang.concur (petri.Lang N1) (petri.Lang N2))
    (petri.Lang (make N1 N2)).
Proof.
  do 2 intro. destruct H as [?[?[?[]]]]. destruct H, H0.
  exists (path_merge (petri.fin N1) (petri.init N1) (petri.init N2) x1 x2).
  generalize dependent ot. generalize dependent x0. generalize dependent x.
  generalize dependent x2. generalize dependent x1. simpl. unfold init.
  generalize (petri.init N2). generalize (petri.init N1). cofix F. intros.
  destruct H1.
    inversion H. inversion H0. rewrite path_merge_fin_fin. constructor. simpl.
      unfold fin. apply Efchain; auto.
    destruct H1; inversion H; inversion H0; rewrite path_merge_hop_hop.
      constructor; eauto. assert (config.eq c' (last l t)).
        eapply petri.Efchain_last; eauto. apply petri.Efchain_eq_r. eexists.
        split. apply Efchain; eauto. apply petri.Efchain_eq_r. eexists.
        split; eauto. apply config.eq_app. split. apply config.eq_sym. auto.
        apply config.eq_refl.
      econstructor; eauto. assert (config.eq c' (last l t)).
        eapply petri.Efchain_last; eauto. apply petri.Efchain_eq_r. eexists.
        split. apply Efchain; eauto. apply petri.Efchain_eq_r. eexists.
        split; eauto. apply config.eq_app. split. apply config.eq_sym. eauto.
        apply config.eq_refl. apply Firing_L; eauto.
      econstructor; eauto. assert (config.eq c' (last l t)).
        eapply petri.Efchain_last; eauto. apply petri.Efchain_eq_r. eexists.
        split. apply Efchain; eauto. apply petri.Efchain_eq_r. eexists.
        split; eauto. apply config.eq_app. split. apply config.eq_sym. eauto.
        apply config.eq_refl. apply Firing_R; eauto.
      econstructor; eauto. assert (config.eq c' (last l t)).
        eapply petri.Efchain_last; eauto. apply petri.Efchain_eq_r. eexists.
        split. apply Efchain; eauto. apply petri.Efchain_eq_r. eexists.
        split; eauto. apply config.eq_app. split. apply config.eq_sym. eauto.
        apply config.eq_refl. apply Firing_union; eauto.
Qed.

(* ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~ *)

(* ef correctness *)
Definition EfcFin N1 N2 ef1 ef2 :
  petri.EfcFin N1 ef1 -> petri.EfcFin N2 ef2 ->
  petri.EfcFin (make N1 N2) (ef ef1 ef2).
Proof.
  unfold ef. intros. destruct X, X0. split; repeat intro.
    edestruct Reach as [?[]]; eauto. rewrite config.L_R_L_R in H0.
      edestruct andb_prop; eauto. edestruct EF_sound; eauto.
      edestruct EF_sound0; eauto.
      exists (map (fun c1 => map place.L c1 ++ map place.R (config.R c)) x ++
      map (fun c2 => map place.L (petri.fin N1) ++ map place.R c2) x0).
      eapply Efchain_app_r; eauto.
    edestruct Reach as [_[]]; eauto. rewrite config.L_R_L_R.
      apply andb_true_intro. edestruct Efchain_L_R as [?[?[]]]; eauto.
      simpl in H3, H4. unfold fin in H3, H4. rewrite config.L_app in H3.
      rewrite config.R_app in H4. eauto.
Defined.

(* nb soundness *)
Definition nb_sound N1 N2 nb1 nb2 :
  petri.nb_sound N1 nb1 -> petri.nb_sound N2 nb2 ->
  petri.nb_sound (make N1 N2) (nb nb1 nb2).
Proof.
  unfold nb. repeat intro. edestruct Reach as [?[]]; eauto.
  rewrite config.L_R_L_R in H0.
  case_eq (nb1 (config.L c) sig sync (fun B c1' =>
      F B (map place.L c1' ++ map place.R (config.R c))) n); intros;
      rewrite H4 in H0.
    edestruct X as [?[?[?[?[?[?[?[?[]]]]]]]]]; eauto. destruct p. destruct o.
      case_eq (nb2 (config.R c) sig sync (fun B c2' =>
          F B (map place.L (config.L c) ++ map place.R c2')) n0); intros;
          rewrite H10 in H0.
        edestruct X0 as [?[?[?[?[?[?[?[?[]]]]]]]]]; eauto. destruct p0.
            destruct o.
          case_eq (nb1 (config.L c) sig false (fun B1 c1' n' =>
              nb2 (config.R c) sig false (fun B2 c2' n'' =>
              if implb sync (bag.sync (bag.union B1 B2))
              then F (bag.union B1 B2) (map place.L c1' ++ map place.R c2') n''
              else None) n') n1); intros; rewrite H16 in H0; rewrite <- H0.
            edestruct X as [?[?[?[?[?[?[?[?[]]]]]]]]]; eauto. simpl in H21.
              edestruct X0 as [?[?[?[?[?[?[?[?[]]]]]]]]]; eauto. simpl in H26.
              case_eq (implb sync (bag.sync (bag.union x11 x16))); intro;
                rewrite H27 in H26; try solve [inversion H26].
              exists
              (map (fun c1 => map place.L c1 ++ map place.R (config.R c)) x9 ++
              map (fun c2 => map place.L x10 ++ map place.R c2) x14),
              (map place.L x10 ++ map place.R x15), (bag.union x11 x16),
              (map place.L x12 ++ map place.R x17), x18. split.
              eapply Efchain_app_r; eauto. split. eapply Firing_union; eauto.
              split. apply bag.Sat_union; auto. split; auto. intro.
              rewrite H28 in H27. apply bag.sync_true; auto.
            exists
              (map (fun c1 => map place.L c1 ++ map place.R (config.R c)) [] ++
              map (fun c2 => map place.L (config.L c) ++ map place.R c2) x4),
              (map place.L (config.L c) ++ map place.R x5), x6,
              (map place.L (config.L c) ++ map place.R x7), x8. split.
              eapply Efchain_app_r; eauto. constructor. apply config.eq_refl.
              split; auto. eapply Firing_R; eauto.
          rewrite <- H0.
            exists
            (map (fun c1 => map place.L c1 ++ map place.R (config.R c)) [] ++
            map (fun c2 => map place.L (config.L c) ++ map place.R c2) x4),
            (map place.L (config.L c) ++ map place.R x5), x6,
            (map place.L (config.L c) ++ map place.R x7), x8. split.
            eapply Efchain_app_r; eauto. constructor. apply config.eq_refl.
            split; auto. eapply Firing_R; eauto.
        case_eq (nb1 (config.L c) sig false (fun B1 c1' n' =>
            nb2 (config.R c) sig false (fun B2 c2' n'' =>
            if implb sync (bag.sync (bag.union B1 B2))
            then F (bag.union B1 B2) (map place.L c1' ++ map place.R c2') n''
            else None) n') n0); intros; rewrite H11 in H0; rewrite <- H0.
          edestruct X as [?[?[?[?[?[?[?[?[]]]]]]]]]; eauto. simpl in H16.
            edestruct X0 as [?[?[?[?[?[?[?[?[]]]]]]]]]; eauto. simpl in H21.
            case_eq (implb sync (bag.sync (bag.union x6 x11))); intro;
              rewrite H22 in H21; try solve [inversion H21].
            exists
            (map (fun c1 => map place.L c1 ++ map place.R (config.R c)) x4 ++
            map (fun c2 => map place.L x5 ++ map place.R c2) x9),
            (map place.L x5 ++ map place.R x10), (bag.union x6 x11),
            (map place.L x7 ++ map place.R x12), x13. split.
            eapply Efchain_app_r; eauto. split. eapply Firing_union; eauto.
            split. apply bag.Sat_union; auto. split; auto. intro.
            rewrite H23 in H22. apply bag.sync_true; auto.
          exists
            (map (fun c1 => map place.L c1 ++ map place.R (config.R c)) x ++
            map (fun c2 => map place.L x0 ++ map place.R c2) []),
            (map place.L x0 ++ map place.R (config.R c)), x1,
            (map place.L x2 ++ map place.R (config.R c)), x3. split.
            eapply Efchain_app_r; eauto. constructor. apply config.eq_refl.
            split; auto. eapply Firing_L; eauto.
      rewrite <- H0.
        exists (map (fun c1 => map place.L c1 ++ map place.R (config.R c)) x ++
        map (fun c2 => map place.L x0 ++ map place.R c2) []),
        (map place.L x0 ++ map place.R (config.R c)), x1,
        (map place.L x2 ++ map place.R (config.R c)), x3. split.
        eapply Efchain_app_r; eauto. constructor. apply config.eq_refl.
        split; auto. eapply Firing_L; eauto.
    case_eq (nb2 (config.R c) sig sync (fun B c2' =>
        F B (map place.L (config.L c) ++ map place.R c2')) n); intros;
        rewrite H5 in H0.
      edestruct X0 as [?[?[?[?[?[?[?[?[]]]]]]]]]; eauto. destruct p. destruct o.
        case_eq (nb1 (config.L c) sig false (fun B1 c1' n' =>
            nb2 (config.R c) sig false (fun B2 c2' n'' =>
            if implb sync (bag.sync (bag.union B1 B2))
            then F (bag.union B1 B2) (map place.L c1' ++ map place.R c2') n''
            else None) n') n0); intros; rewrite H11 in H0; rewrite <- H0.
          edestruct X as [?[?[?[?[?[?[?[?[]]]]]]]]]; eauto. simpl in H16.
            edestruct X0 as [?[?[?[?[?[?[?[?[]]]]]]]]]; eauto. simpl in H21.
            case_eq (implb sync (bag.sync (bag.union x6 x11))); intro;
              rewrite H22 in H21; try solve [inversion H21].
            exists
            (map (fun c1 => map place.L c1 ++ map place.R (config.R c)) x4 ++
            map (fun c2 => map place.L x5 ++ map place.R c2) x9),
            (map place.L x5 ++ map place.R x10), (bag.union x6 x11),
            (map place.L x7 ++ map place.R x12), x13. split.
            eapply Efchain_app_r; eauto. split. eapply Firing_union; eauto.
            split. apply bag.Sat_union; auto. split; auto. intro.
            rewrite H23 in H22. apply bag.sync_true; auto.
          exists
            (map (fun c1 => map place.L c1 ++ map place.R (config.R c)) [] ++
            map (fun c2 => map place.L (config.L c) ++ map place.R c2) x),
            (map place.L (config.L c) ++ map place.R x0), x1,
            (map place.L (config.L c) ++ map place.R x2), x3. split.
            eapply Efchain_app_r; eauto. constructor. apply config.eq_refl.
            split; auto. eapply Firing_R; eauto.
        rewrite <- H0.
          exists
          (map (fun c1 => map place.L c1 ++ map place.R (config.R c)) [] ++
          map (fun c2 => map place.L (config.L c) ++ map place.R c2) x),
          (map place.L (config.L c) ++ map place.R x0), x1,
          (map place.L (config.L c) ++ map place.R x2), x3. split.
          eapply Efchain_app_r; eauto. constructor. apply config.eq_refl.
          split; auto. eapply Firing_R; eauto.
      case_eq (nb1 (config.L c) sig false (fun B1 c1' n' =>
          nb2 (config.R c) sig false (fun B2 c2' n'' =>
          if implb sync (bag.sync (bag.union B1 B2))
          then F (bag.union B1 B2) (map place.L c1' ++ map place.R c2') n''
          else None) n') n); intros; rewrite H6 in H0; rewrite <- H0.
        edestruct X as [?[?[?[?[?[?[?[?[]]]]]]]]]; eauto. simpl in H11.
          edestruct X0 as [?[?[?[?[?[?[?[?[]]]]]]]]]; eauto. simpl in H16.
          case_eq (implb sync (bag.sync (bag.union x1 x6))); intro;
            rewrite H17 in H16; try solve [inversion H16].
          exists
          (map (fun c1 => map place.L c1 ++ map place.R (config.R c)) x ++
          map (fun c2 => map place.L x0 ++ map place.R c2) x4),
          (map place.L x0 ++ map place.R x5), (bag.union x1 x6),
          (map place.L x2 ++ map place.R x7), x8. split.
          eapply Efchain_app_r; eauto. split. eapply Firing_union; eauto. split.
          apply bag.Sat_union; auto. split; auto. intro. rewrite H18 in H17.
          apply bag.sync_true; auto.
        inversion H0.
Defined.

(* nb completeness *)
Definition nb_complete N1 N2 nb1 nb2 :
  petri.NthBag N1 nb1 -> petri.NthBag N2 nb2 ->
  petri.nb_complete (make N1 N2) (nb nb1 nb2).
Proof.
  unfold nb. repeat intro. destruct X, X0.
  edestruct Reach as [_[]]; eauto. rewrite config.L_R_L_R.
  assert (petri.Reach (make N1 N2) c'). eapply petri.Reach_Efchain; eauto.
  edestruct Reach as [_[]]; eauto.
  assert (petri.Reach (make N1 N2) c''). econstructor 2; eauto.
  edestruct Reach as [?[]]; eauto.
  edestruct Efchain_L_R as [?[?[]]]; eauto.
  edestruct Firing_Some as [[[]|[]]|[?[?[?[]]]]]; eauto.
    edestruct NB_complete with (F:=fun (B0 : bag.t) (c1' : config.t) =>
      F B0 (map place.L c1' ++ map place.R (config.R c))) as [??[]].
      apply H5. eauto. eauto. eauto. eauto. eauto. eauto.
      exists (map place.L x1 ++ map place.R (config.R c)).
        apply Reach_L_R; auto.
      exists (map (fun c1 => map place.L c1 ++ map place.R (config.R c)) x2 ++
        map (fun c2 => map place.L (config.L c'') ++ map place.R c2) x0).
        apply petri.Efchain_eq_r. eexists. split. eapply Efchain; eauto.
        apply config.eq_sym, config.eq_app_r. split; auto. split.
        apply config.eq_refl. apply config.eq_sym. auto.
      intro. destruct (s n'). exists x3. intro. rewrite e0; auto. destruct H18.
      rewrite H18. auto.
    edestruct NB_complete0 with (F:=fun (B0 : bag.t) (c2' : config.t) =>
      F B0 (map place.L (config.L c) ++ map place.R c2')) as [??[]].
      apply H6. eauto. eauto. eauto. eauto. eauto. eauto.
      exists (map place.L (config.L c) ++ map place.R x1).
        apply Reach_L_R; auto.
      exists (map (fun c1 => map place.L c1 ++ map place.R x1) x ++
        map (fun c2 => map place.L (config.L c'') ++ map place.R c2) x2).
        apply petri.Efchain_eq_r. eexists. split. eapply Efchain; eauto.
        eapply petri.Efchain_eq_r; eauto. apply config.eq_sym, config.eq_app_r.
        split; auto. split; apply config.eq_refl.
      intro. destruct (s n'). case_eq (nb1 (config.L c) sig sync
          (fun B0 c1' => F B0 (map place.L c1' ++ map place.R (config.R c))) 0);
          intros.
        edestruct NB_Some_Some with (F:=fun (B0 : bag.t) (c1' : config.t) =>
          F B0 (map place.L c1' ++ map place.R (config.R c))); eauto.
          exists x4. intro. destruct e1. rewrite H20. rewrite e0; auto.
          destruct H19. rewrite H19. auto.
        exists x3. erewrite NB_None_None; eauto. intro. rewrite e0; eauto.
          destruct H19. rewrite H19. auto.
    rewrite H16 in *. rewrite <- bag.Sat_union in H2. destruct H2.
      rewrite <- bag.sync_true in H3.
      assert (forall B0 c0 B1 c1, petri.f_None (fun n : nat =>
        if implb sync (bag.sync (bag.union B0 B1))
        then F (bag.union B0 B1) (map place.L c0 ++ map place.R c1) n
        else None)). destruct sync; simpl; auto. intros.
        destruct (bag.sync (bag.union B0 B1)); auto. intro. auto.
      assert (forall B0 c0 B1 c1, petri.f_Some (fun n : nat =>
        if implb sync (bag.sync (bag.union B0 B1))
        then F (bag.union B0 B1) (map place.L c0 ++ map place.R c1) n
        else None)). destruct sync; simpl; auto. intros.
        destruct (bag.sync (bag.union B0 B1)); auto. do 4 intro. inversion H21.
      edestruct NB_complete with (sync:=false)
        (F:=fun (B1 : bag.t) (c1' : config.t) n' => nb2 (config.R c) sig false
        (fun (B2 : bag.t) (c2' : config.t) n =>
        if implb sync (bag.sync (bag.union B1 B2))
        then F (bag.union B1 B2) (map place.L c1' ++ map place.R c2') n
        else None) n') as [??[]]. apply H5. eauto. eauto. eauto. intro.
        discriminate. eauto. eauto.
      edestruct NB_complete0 with (sync:=false)
        (F:=fun (B2 : bag.t) (c2' : config.t) n =>
        if implb sync (bag.sync (bag.union x1 B2))
        then F (bag.union x1 B2) (map place.L x3 ++ map place.R c2') n
        else None) as [??[]]. apply H6. eauto. eauto. eauto. intro.
        discriminate. eauto. eauto.
      exists (map place.L x3 ++ map place.R x5). apply Reach_L_R; auto.
      exists (map (fun c1 => map place.L c1 ++ map place.R x5) x4 ++
        map (fun c2 => map place.L (config.L c'') ++ map place.R c2) x6).
        apply petri.Efchain_eq_r. eexists. split. eapply Efchain; eauto.
        apply config.eq_sym, config.eq_app_r. split; auto.
        split; apply config.eq_refl.
      intro. destruct (s0 n'). destruct (s x7). case_eq (nb2 (config.R c) sig
          sync (fun B0 c2' =>
          F B0 (map place.L (config.L c) ++ map place.R c2')) 0); intros.
        edestruct NB_Some_Some0 with (F:=fun (B0 : bag.t) (c2' : config.t) =>
            F B0 (map place.L (config.L c) ++ map place.R c2')); eauto.
            case_eq (nb1 (config.L c) sig sync (fun B0 c1' =>
            F B0 (map place.L c1' ++ map place.R (config.R c))) 0); intros.
          edestruct NB_Some_Some with (F:=fun (B0 : bag.t) (c1' : config.t) =>
            F B0 (map place.L c1' ++ map place.R (config.R c))); eauto.
            exists x10. intro. destruct e4. rewrite H24. destruct e3.
            rewrite e3. rewrite e2. rewrite e1. destruct H23. rewrite H23.
            destruct sync; auto. rewrite H3; auto.
            destruct sync; auto. rewrite H3; auto.
            rewrite e1. destruct sync; auto. rewrite H3; auto.
            destruct sync; auto. rewrite H3; auto.
          exists x9. intro. erewrite NB_None_None; eauto. destruct e3.
            rewrite H24. rewrite e2. rewrite e1. destruct H23. rewrite H23.
            destruct sync; auto. rewrite H3; auto.
            destruct sync; auto. rewrite H3; auto.
            rewrite e1. destruct sync; auto. rewrite H3; auto.
            destruct sync; auto. rewrite H3; auto.
        case_eq (nb1 (config.L c) sig sync (fun B0 c1' =>
            F B0 (map place.L c1' ++ map place.R (config.R c))) 0); intros.
          edestruct NB_Some_Some with (F:=fun (B0 : bag.t) (c1' : config.t) =>
            F B0 (map place.L c1' ++ map place.R (config.R c))); eauto.
            exists x9. intro. destruct e3. rewrite H24.
            erewrite NB_None_None0; eauto. rewrite e2. rewrite e1. destruct H23.
            rewrite H23. destruct sync; auto. rewrite H3; auto.
            destruct sync; auto. rewrite H3; auto.
            rewrite e1. destruct sync; auto. rewrite H3; auto.
            destruct sync; auto. rewrite H3; auto.
          exists x8. intro. erewrite NB_None_None; eauto.
            erewrite NB_None_None0; eauto. rewrite e2. rewrite e1. destruct H23.
            rewrite H23. destruct sync; auto. rewrite H3; auto.
            destruct sync; auto. rewrite H3; auto.
            rewrite e1. destruct sync; auto. rewrite H3; auto.
            destruct sync; auto. rewrite H3; auto.
Defined.

(* nb correctness *)
Definition NthBag N1 N2 nb1 nb2 :
  petri.Wf N1 -> petri.Wf N2 -> petri.NthBag N1 nb1 -> petri.NthBag N2 nb2 ->
  petri.NthBag (make N1 N2) (nb nb1 nb2).
Proof.
  intros. inversion X. inversion X0.
  split; try apply nb_sound; try apply nb_complete; auto;
  unfold nb; try unfold petri.f_None in *; try unfold petri.f_Some in *;
  repeat intro.
    rewrite config.L_R_L_R in *. case_eq (nb1 (config.L c) sig sync
        (fun B c1' => F B (map place.L c1' ++ map place.R (config.R c))) n);
        intros; rewrite H3 in H2.
      destruct p. destruct o.
        destruct nb2 in H2.
          destruct p0. destruct o.
            destruct nb1 in H2; inversion H2.
            inversion H2.
          destruct nb1 in H2; inversion H2.
        inversion H2.
      erewrite NB_None_None; eauto. case_eq (nb2 (config.R c) sig sync
          (fun B c2' => F B (map place.L (config.L c) ++ map place.R c2')) n);
          intros; rewrite H4 in H2.
        destruct p. destruct o.
          destruct nb1 in H2; inversion H2.
          inversion H2.
        erewrite NB_None_None0; eauto. case_eq (nb1 (config.L c) sig false
            (fun B1 c1' n' => nb2 (config.R c) sig false (fun B2 c2' n'' =>
            if implb sync (bag.sync (bag.union B1 B2))
            then F (bag.union B1 B2) (map place.L c1' ++ map place.R c2') n''
            else None) n') n); intros; rewrite H5 in H2.
          inversion H2.
          erewrite NB_None_None; eauto. intros. erewrite NB_None_None0; eauto.
            intros. destruct sync; simpl in *; eauto.
            destruct (bag.sync (bag.union B B0)); simpl in *; eauto.
    rewrite config.L_R_L_R in *. case_eq (nb1 (config.L c) sig sync
        (fun B c1' => F B (map place.L c1' ++ map place.R (config.R c))) n);
        intros; rewrite H3 in H2.
      case_eq (nb2 (config.R c) sig sync (fun (B : bag.t) (c2' : config.t) =>
          F B (map place.L (config.L c) ++ map place.R c2')) 0); intros.
        case_eq (nb1 (config.L c) sig false (fun B1 c1' n' =>
            nb2 (config.R c) sig false (fun B2 c2' n =>
            if implb sync (bag.sync (bag.union B1 B2))
            then F (bag.union B1 B2) (map place.L c1' ++ map place.R c2') n
            else None) n') 0); intros.
          edestruct NB_Some_Some with (F:=
              fun (B1 : bag.t) (c1 : config.t) n' => nb2 (config.R c) sig
              false (fun (B2 : bag.t) (c2 : config.t) n =>
              if implb sync (bag.sync (bag.union B1 B2))
              then F (bag.union B1 B2) (map place.L c1 ++ map place.R c2) n
              else None) n'); eauto. intros. erewrite NB_None_None0; eauto.
              intros. destruct sync; simpl in *; eauto.
              destruct (bag.sync (bag.union B B0)); simpl in *; eauto.
              intros. edestruct NB_Some_Some0 with (F:=
                fun (B2 : bag.t) (c2 : config.t) n =>
                if implb sync (bag.sync (bag.union B B2))
                then F (bag.union B B2) (map place.L c0 ++ map place.R c2) n
                else None); eauto. intros. destruct sync; simpl in *; eauto.
              destruct (bag.sync (bag.union B B0)); simpl in *; eauto.
              intros. destruct sync; simpl in *; eauto.
              destruct (bag.sync (bag.union B B0)); simpl in *; eauto.
              inversion H7.
            edestruct NB_Some_Some0 with (F:=
                fun (B : bag.t) (c2' : config.t) =>
                F B (map place.L (config.L c) ++ map place.R c2')); eauto.
              edestruct NB_Some_Some with (F:=
                  fun (B : bag.t) (c1' : config.t) =>
                  F B (map place.L c1' ++ map place.R (config.R c))); eauto.
                exists x1. destruct e1. rewrite H6. destruct e0. rewrite e0.
                  destruct e. rewrite e. eauto.
          edestruct NB_Some_Some0 with (F:=
              fun (B : bag.t) (c2' : config.t) =>
              F B (map place.L (config.L c) ++ map place.R c2')); eauto.
            edestruct NB_Some_Some with (F:=
                fun (B : bag.t) (c1' : config.t) =>
                F B (map place.L c1' ++ map place.R (config.R c))); eauto.
              exists x0. destruct e0. rewrite H6. destruct e. rewrite e.
                erewrite NB_None_None; eauto.
                intros. erewrite NB_None_None0; eauto.
                intros. destruct sync; simpl in *; eauto.
                destruct (bag.sync (bag.union B B0)); simpl in *; eauto.
        case_eq (nb1 (config.L c) sig false (fun B1 c1' n' =>
            nb2 (config.R c) sig false (fun B2 c2' n =>
            if implb sync (bag.sync (bag.union B1 B2))
            then F (bag.union B1 B2) (map place.L c1' ++ map place.R c2') n
            else None) n') 0); intros.
          edestruct NB_Some_Some with (F:=
              fun (B1 : bag.t) (c1 : config.t) n' => nb2 (config.R c) sig
              false (fun (B2 : bag.t) (c2 : config.t) n =>
              if implb sync (bag.sync (bag.union B1 B2))
              then F (bag.union B1 B2) (map place.L c1 ++ map place.R c2) n
              else None) n'); eauto. intros. erewrite NB_None_None0; eauto.
              intros. destruct sync; simpl in *; eauto.
              destruct (bag.sync (bag.union B B0)); simpl in *; eauto.
              intros. edestruct NB_Some_Some0 with (F:=
                fun (B2 : bag.t) (c2 : config.t) n =>
                if implb sync (bag.sync (bag.union B B2))
                then F (bag.union B B2) (map place.L c0 ++ map place.R c2) n
                else None); eauto. intros. destruct sync; simpl in *; eauto.
              destruct (bag.sync (bag.union B B0)); simpl in *; eauto.
              intros. destruct sync; simpl in *; eauto.
              destruct (bag.sync (bag.union B B0)); simpl in *; eauto.
              inversion H7.
            edestruct NB_Some_Some with (F:= fun (B : bag.t) (c1' : config.t) =>
                F B (map place.L c1' ++ map place.R (config.R c))); eauto.
              exists x0. destruct e0. rewrite H6. erewrite NB_None_None0; eauto.
                destruct e. rewrite e. eauto.
          edestruct NB_Some_Some with (F:=
              fun (B : bag.t) (c1' : config.t) =>
              F B (map place.L c1' ++ map place.R (config.R c))); eauto.
            exists x. destruct e. rewrite H6. erewrite NB_None_None0; eauto.
              erewrite NB_None_None; eauto.
              intros. erewrite NB_None_None0; eauto.
              intros. destruct sync; simpl in *; eauto.
              destruct (bag.sync (bag.union B B0)); simpl in *; eauto.
      case_eq (nb2 (config.R c) sig sync
          (fun (B : bag.t) (c2' : config.t) =>
          F B (map place.L (config.L c) ++ map place.R c2')) n); intros;
          rewrite H4 in H2.
        case_eq (nb1 (config.L c) sig false (fun B1 c1' n' =>
            nb2 (config.R c) sig false (fun B2 c2' n =>
            if implb sync (bag.sync (bag.union B1 B2))
            then F (bag.union B1 B2) (map place.L c1' ++ map place.R c2') n
            else None) n') 0); intros.
          edestruct NB_Some_Some with (F:=
              fun (B1 : bag.t) (c1 : config.t) n' => nb2 (config.R c) sig
              false (fun (B2 : bag.t) (c2 : config.t) n =>
              if implb sync (bag.sync (bag.union B1 B2))
              then F (bag.union B1 B2) (map place.L c1 ++ map place.R c2) n
              else None) n'); eauto. intros. erewrite NB_None_None0; eauto.
              intros. destruct sync; simpl in *; eauto.
              destruct (bag.sync (bag.union B B0)); simpl in *; eauto.
              intros. edestruct NB_Some_Some0 with (F:=
                fun (B2 : bag.t) (c2 : config.t) n =>
                if implb sync (bag.sync (bag.union B B2))
                then F (bag.union B B2) (map place.L c0 ++ map place.R c2) n
                else None); eauto. intros. destruct sync; simpl in *; eauto.
              destruct (bag.sync (bag.union B B0)); simpl in *; eauto.
              intros. destruct sync; simpl in *; eauto.
              destruct (bag.sync (bag.union B B0)); simpl in *; eauto.
              inversion H7.
            edestruct NB_Some_Some0 with (F:=
                fun (B : bag.t) (c2' : config.t) =>
                F B (map place.L (config.L c) ++ map place.R c2')); eauto.
              exists x0. erewrite NB_None_None; eauto. destruct e0. rewrite H6.
                destruct e. rewrite e. eauto.
          edestruct NB_Some_Some0 with (F:=fun (B : bag.t) (c2' : config.t) =>
              F B (map place.L (config.L c) ++ map place.R c2')); eauto.
            exists x. erewrite NB_None_None; eauto. destruct e. rewrite H6.
              erewrite NB_None_None; eauto.
              intros. erewrite NB_None_None0; eauto.
              intros. destruct sync; simpl in *; eauto.
              destruct (bag.sync (bag.union B B0)); simpl in *; eauto.
        case_eq (nb1 (config.L c) sig false (fun B1 c1' n' =>
            nb2 (config.R c) sig false (fun B2 c2' n =>
            if implb sync (bag.sync (bag.union B1 B2))
            then F (bag.union B1 B2) (map place.L c1' ++ map place.R c2') n
            else None) n') n); intros; rewrite H5 in H2.
          edestruct NB_Some_Some with (F:=
              fun (B1 : bag.t) (c1 : config.t) n' => nb2 (config.R c) sig
              false (fun (B2 : bag.t) (c2 : config.t) n =>
              if implb sync (bag.sync (bag.union B1 B2))
              then F (bag.union B1 B2) (map place.L c1 ++ map place.R c2) n
              else None) n'); eauto. intros. erewrite NB_None_None0; eauto.
              intros. destruct sync; simpl in *; eauto.
              destruct (bag.sync (bag.union B B0)); simpl in *; eauto.
              intros. edestruct NB_Some_Some0 with (F:=
                fun (B2 : bag.t) (c2 : config.t) n =>
                if implb sync (bag.sync (bag.union B B2))
                then F (bag.union B B2) (map place.L c0 ++ map place.R c2) n
                else None); eauto. intros. destruct sync; simpl in *; eauto.
              destruct (bag.sync (bag.union B B0)); simpl in *; eauto.
              intros. destruct sync; simpl in *; eauto.
              destruct (bag.sync (bag.union B B0)); simpl in *; eauto.
              inversion H7.
            exists x. erewrite NB_None_None; eauto.
              erewrite NB_None_None0; eauto. destruct e. rewrite H6. eauto.
          inversion H2.
Defined.

End par.

End MPetPar.

(* (c) 2020 Brittany Ro Nkounkou *)
