(* Brittany Nkounkou *)
(* August 2020 *)
(* Epsilon Petri Net *)

Require Export Petris.

Set Implicit Arguments.

Module MPetEps (env : Environment).
Module Export M := MPetris env.

Module eps.

(* initial configuration *)
Definition init : config.t :=
  [place.P].

(* final configuration *)
Definition fin : config.t :=
  [place.P].

(* epsilon petri net *)
Definition make : petri.t :=
  petri.make
    init
    fin
    [].

(* empty-firing-chain-to-fin decider *)
Definition ef : petri.efc_fin :=
  fun _ =>
  true.

(* nth-bag finder *)
Definition nb : petri.nth_bag :=
  fun _ _ _ _ _ =>
  None.

(* ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~ *)

(* the epsilon petri net is well-formed *)
Lemma Wf : petri.Wf make.
Proof.
  split; try discriminate. intros. destruct H.
Qed.

(* ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~ *)

(* there are no firings *)
Lemma Firing c w c' : ~ petri.Firing make c w c'.
Proof.
  intro. destruct H. destruct H.
Qed.

(* any empty firing chain is a self-loop *)
Lemma Efchain c l c' : petri.Efchain make c l c' -> config.eq c c'.
Proof.
  intro. destruct H. auto. edestruct Firing; eauto.
Qed.

(* any reachable configuration is init *)
Lemma Reach c : petri.Reach make c -> config.eq c init.
Proof.
  intro. destruct H. auto. edestruct Firing; eauto.
Qed.

(* the epsilon petri net is safe *)
Lemma Safe : petri.Safe make.
Proof.
  do 3 intro. eapply Reach; eauto.
Qed.

(* ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~ *)

CoFixpoint path_opttrace ot : path.t :=
  match ot with
  | opttrace.eps => path.fin []
  | opttrace.opt _ ot' => path.hop [] init (path_opttrace ot')
  end.

Lemma path_eps : path_opttrace opttrace.eps = path.fin [].
Proof.
  rewrite path.match_ at 1. auto.
Qed.

Lemma path_opt o t :
  path_opttrace (opttrace.opt o t) = path.hop [] init (path_opttrace t).
Proof.
  rewrite path.match_ at 1. auto.
Qed.

(* ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~ *)

(* Lang soundness *)
Lemma Lang_sound : slang.le (petri.Lang make) slang.eps.
Proof.
  unfold slang.le, petri.Lang. cofix F. intros. destruct H. destruct H.
  constructor. constructor. apply F. eexists. eapply petri.Gen_Efchain; eauto.
  edestruct Firing; eauto.
Qed.

(* Lang completeness *)
Lemma Lang_complete : slang.le slang.eps (petri.Lang make).
Proof.
  do 2 intro. exists (path_opttrace ot). generalize dependent ot. cofix F.
  intros. destruct H.
    rewrite path_eps. constructor. constructor. apply config.eq_refl.
    rewrite path_opt. constructor; auto. constructor. apply config.eq_refl.
Qed.

(* ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~ *)

(* ef correctness *)
Definition EfcFin : petri.EfcFin make ef.
Proof.
  unfold ef. split; repeat intro; auto.
  exists []. constructor. eapply Reach; eauto.
Defined.

(* nb correctness *)
Definition NthBag : petri.NthBag make nb.
Proof.
  unfold nb. split; repeat intro; auto. inversion H0. inversion H0.
  exfalso. edestruct Firing; eauto.
Defined.

End eps.

End MPetEps.

(* (c) 2020 Brittany Ro Nkounkou *)
