(* Brittany Nkounkou *)
(* August 2020 *)
(* Singleton-Bag Petri Net *)

Require Export PetEps.

Set Implicit Arguments.

Module MPetBag (env : Environment).
Module Export M := MPetEps env.

Module bag.

(* initial configuration *)
Definition init : config.t :=
  [place.L place.P].

(* final configuration *)
Definition fin : config.t :=
  [place.R place.P].

(* singleton-bag petri net *)
Definition make B : petri.t :=
  petri.make
    init
    fin
    [(init, Some B, fin)].

(* empty-firing-chain-to-fin decider *)
Definition ef : petri.efc_fin :=
  fun c =>
  match c with
  | place.L _ :: _ => false
  | _ => true
  end.

(* nth-bag finder *)
Definition nb B : petri.nth_bag :=
  fun c sig sync F n =>
  match c with
  | place.L _ :: _ =>
    if andb (bag.sat sig B) (implb sync (bag.sync B))
    then F B fin n
    else None
  | _ => None
  end.

(* ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~ *)

(* the singleton-bag petri net is well-formed *)
Lemma Wf B : petri.Wf (make B).
Proof.
  split; try discriminate. intros. destruct H; inversion H. split; discriminate.
Qed.

(* ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~ *)

(* the first element of any subconfiguration of init is not the P place *)
Lemma le_P_init c : ~ config.le (place.P :: c) init.
Proof.
  intro. edestruct H. apply in_eq. inversion H0. destruct H0.
Qed.

(* the first element of any subconfiguration of init is not an R place *)
Lemma le_R_init p c : ~ config.le (place.R p :: c) init.
Proof.
  intro. edestruct H. apply in_eq. inversion H0. destruct H0.
Qed.

(* the first element of any subconfiguration of fin is not the P place *)
Lemma le_P_fin c : ~ config.le (place.P :: c) fin.
Proof.
  intro. edestruct H. apply in_eq. inversion H0. destruct H0.
Qed.

(* the first element of any subconfiguration of fin is not an L place *)
Lemma le_L_fin p c : ~ config.le (place.L p :: c) fin.
Proof.
  intro. edestruct H. apply in_eq. inversion H0. destruct H0.
Qed.

(* ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~ *)

(* there are no empty firings *)
Lemma Firing_None B c c' :
  ~ petri.Firing (make B) c None c'.
Proof.
  intro. destruct H. destruct H; inversion H.
Qed.

(* any firing from init is non-empty and ends on fin *)
Lemma Firing_init B c w c' :
  config.eq c init ->
  (petri.Firing (make B) c w c' <-> w = Some B /\ config.eq c' fin).
Proof.
  split; intro.
    destruct H0. destruct H0; inversion H0. split; auto.
      eapply config.eq_trans. apply config.eq_sym. apply H2.
      rewrite <- H4, <- H6. rewrite config.diff_eq. apply config.union_nil.
      auto.
    destruct H0. rewrite H0. eapply petri.Firing_eq_eq; eauto. apply in_eq.
Qed.

(* there are no firings from fin *)
Lemma Firing_fin B c w c' :
  config.eq c fin -> ~ petri.Firing (make B) c w c'.
Proof.
  do 2 intro. destruct H0. destruct H0; inversion H0. rewrite <- H4 in H1.
  eapply le_L_fin. eapply config.le_trans; eauto. apply H.
Qed.

(* any empty firing chain is a self-loop *)
Lemma Efchain B c l c' : petri.Efchain (make B) c l c' -> config.eq c c'.
Proof.
  intro. destruct H. auto. edestruct Firing_None; eauto.
Qed.

(* ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~ *)

(* any reachable configuration is either init or fin *)
Lemma Reach B c : petri.Reach (make B) c -> config.eq c init \/ config.eq c fin.
Proof.
  intro. induction H. auto. destruct IHReach. right. eapply Firing_init; eauto.
  edestruct Firing_fin; eauto.
Qed.

(* the first element of any reachable configuration is not the P place *)
Lemma Reach_P B c : ~ petri.Reach (make B) (place.P :: c).
Proof.
  intro. edestruct Reach; eauto. eapply le_P_init, H0. eapply le_P_fin, H0.
Qed.

(* if the first element of a reachable configuration is an L place,
    then the configuration is init *)
Lemma Reach_L B p c :
  petri.Reach (make B) (place.L p :: c) -> config.eq (place.L p :: c) init.
Proof.
  intro. edestruct Reach; eauto. exfalso. eapply le_L_fin, H0.
Qed.

(* if the first element of a reachable configuration is an R place,
    then the configuration is fin *)
Lemma Reach_R B p c :
  petri.Reach (make B) (place.R p :: c) -> config.eq (place.R p :: c) fin.
Proof.
  intro. edestruct Reach; eauto. exfalso. eapply le_R_init, H0.
Qed.

(* the singleton-bag petri net is safe *)
Lemma Safe B : petri.Safe (make B).
Proof.
  do 3 intro. edestruct Reach; eauto.
    edestruct le_R_init. eapply config.le_trans. apply H0. apply H1.
    apply H1.
Qed.

(* ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~ *)

CoFixpoint path_opttrace c ot : path.t :=
  match ot with
  | opttrace.eps => path.fin []
  | opttrace.opt None ot' => path.hop [] c (path_opttrace c ot')
  | opttrace.opt (Some _) ot' => path.hop [] fin (path_opttrace fin ot')
  end.

Lemma path_eps c : path_opttrace c opttrace.eps = path.fin [].
Proof.
  rewrite path.match_ at 1. auto.
Qed.

Lemma path_None ot c :
  path_opttrace c (opttrace.opt None ot) = path.hop [] c (path_opttrace c ot).
Proof.
  rewrite path.match_ at 1. auto.
Qed.

Lemma path_Some B ot c :
  path_opttrace c (opttrace.opt (Some B) ot) =
  path.hop [] fin (path_opttrace fin ot).
Proof.
  rewrite path.match_ at 1. auto.
Qed.

(* ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~ *)

(* Lang soundness *)
Lemma Lang_sound B : slang.le (petri.Lang (make B)) (slang.bag B).
Proof.
  unfold slang.le, petri.Lang. cofix F. intros. destruct H. destruct H.
    edestruct le_R_init. eapply Efchain. apply H.
    constructor. apply F. eexists. eapply petri.Gen_Efchain; eauto.
    rewrite Firing_init in H0.
      destruct H0. inversion H0. constructor. generalize dependent ot.
          generalize dependent pi. generalize dependent c''. cofix F'. intros.
          destruct H1.
        constructor.
        constructor. eapply F'. apply config.eq_sym. eapply Efchain.
          eapply petri.Efchain_eq_l; eauto. eauto.
        edestruct Firing_fin. apply config.eq_sym. eapply Efchain.
          eapply petri.Efchain_eq_l; eauto. eauto.
      apply config.eq_sym. eapply Efchain; eauto.
Qed.

(* Lang completeness *)
Lemma Lang_complete B : slang.le (slang.bag B) (petri.Lang (make B)).
Proof.
  do 2 intro. exists (path_opttrace init ot). generalize dependent ot. simpl.
  cofix F. intros. destruct H.
    rewrite path_None. constructor; auto. constructor. apply config.eq_refl.
    rewrite path_Some. econstructor. constructor. apply config.eq_refl.
        eapply petri.Firing_eq_eq. apply in_eq. apply config.eq_refl.
        apply config.eq_refl. generalize dependent ot. cofix F'. intros.
        destruct H.
      rewrite path_eps. constructor. constructor. apply config.eq_refl.
      rewrite path_None. constructor; auto. constructor. apply config.eq_refl.
Qed.

(* ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~ *)

(* ef correctness *)
Definition EfcFin B : petri.EfcFin (make B) ef.
Proof.
  unfold ef. split; repeat intro.
    destruct c. edestruct petri.Reach_nil. apply Wf. eauto.
      destruct t. edestruct Reach_P; eauto. discriminate.
      exists []. constructor. eapply Reach_R; eauto.
    destruct c; auto. destruct t; auto. edestruct le_L_fin. apply (Efchain H0).
Defined.

(* nb soundness *)
Definition nb_sound B : petri.nb_sound (make B) (nb B).
Proof.
  unfold nb. repeat intro. destruct c. inversion H0.
  destruct t0; try solve [inversion H0].
  case_eq (bag.sat sig B && implb sync (bag.sync B))%bool; intro;
      rewrite H1 in H0.
    edestruct andb_prop; eauto. exists [], init, B, fin, n. split.
      constructor. eapply Reach_L; eauto. split. apply Firing_init.
      apply config.eq_refl. split; auto. apply config.eq_refl. split.
      apply bag.sat_true. auto. split. intro. rewrite H4 in H3.
      apply bag.sync_true. auto. auto.
    inversion H0.
Defined.

(* nb completeness *)
Definition nb_complete B : petri.nb_complete (make B) (nb B).
Proof.
  unfold nb. repeat intro.
  destruct c. edestruct petri.Reach_nil. apply Wf. eauto.
  destruct t. edestruct Reach_P; eauto.
    exists fin. econstructor 2. apply petri.Reach_init. apply Firing_init.
      apply config.eq_refl. split; eauto. apply config.eq_refl.
      rewrite Firing_init in H1. destruct H1. inversion H1. rewrite H7 in *.
      exists []. constructor. apply config.eq_sym. auto.
      intro. exists n'. intro. rewrite <- bag.sat_true in H2.
      rewrite <- bag.sync_true in H3. rewrite H2. destruct sync; auto.
      rewrite H3; auto. eapply config.eq_trans. apply config.eq_sym.
      eapply Efchain; eauto. eapply Reach_L; eauto.
    exfalso. eapply Firing_fin; eauto. eapply config.eq_trans.
      apply config.eq_sym. eapply Efchain; eauto. eapply Reach_R; eauto.
Defined.

(* nb correctness *)
Definition NthBag B : petri.NthBag (make B) (nb B).
Proof.
  split; try apply nb_sound; try apply nb_complete;
  unfold nb; try unfold petri.f_None; try unfold petri.f_Some; repeat intro.
    destruct c; auto. destruct t; auto.
      destruct (bag.sat sig B && implb sync (bag.sync B))%bool; eauto.
    destruct c. inversion H0. destruct t0. inversion H0.
      destruct (bag.sat sig B && implb sync (bag.sync B))%bool. eauto.
      inversion H0. inversion H0.
Defined.

End bag.

End MPetBag.

(* (c) 2020 Brittany Ro Nkounkou *)
