(* Brittany Nkounkou *)
(* August 2020 *)
(* Simtrace Observer *)

Require Export SimAlg.

Set Implicit Arguments.

Module MObsrvr (env : Environment).
Module Export M := MSimAlg env.

Module obsguard.

(* observation guard *)
Inductive t : Type :=
| bool : bool -> t
| sprb : chan.t -> bool -> t
| rprb : chan.t -> bool -> t
| dprb : chan.t -> option val.t -> expr.t -> option val.t -> t
| expr : expr.t -> option val.t -> expr.t -> option val.t -> t
| neg : t -> t
| and : t -> t -> t
| or : t -> t -> t.

(* obsguard of a (regular) guard *)
Fixpoint of_guard sig G : t :=
  match G with
  | guard.bool b => bool b
  | guard.sprb A => sprb A (state.get_sprb sig A)
  | guard.rprb A => rprb A (state.get_rprb sig A)
  | guard.dprb A e => dprb A (state.get_dprb sig A) e (state.eval_expr sig e)
  | guard.expr e1 e2 =>
      expr e1 (state.eval_expr sig e1) e2 (state.eval_expr sig e2)
  | guard.neg G => neg (of_guard sig G)
  | guard.and G1 G2 => and (of_guard sig G1) (of_guard sig G2)
  | guard.or G1 G2 => or (of_guard sig G1) (of_guard sig G2)
  end.

End obsguard.

(* ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~ *)

Module obsevent.

(* observation event *)
Inductive t : Type :=
| assn : dvar.t -> expr.t -> option val.t -> t
| send : chan.t -> option (expr.t * option val.t) -> t
| recv : chan.t -> option (dvar.t * option val.t) -> t
| wait : obsguard.t -> t
| detv : t.

(* obsevent of a (regular) event *)
Definition of_event sig E : t :=
  match E with
  | event.assn x e => assn x e (state.eval_expr sig e)
  | event.send A o => send A
    match o with
    | Some e => Some (e, state.eval_expr sig e)
    | None => None
    end
  | event.recv A o => recv A
    match o with
    | None => None
    | Some x => Some (x, state.get_dprb sig A)
    end
  | event.wait G => wait (obsguard.of_guard sig G)
  | event.detv => detv
  end.

End obsevent.

(* ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~ *)

(* observation *)
Inductive observation : Type :=
| Eps : observation
| Bag : list obsevent.t -> observation -> observation
| More : observation
| Dead : observation.

(* observation of a simtrace *)
Fixpoint observe' n sig st : observation :=
  match st with
  | simtrace.eps => Eps
  | simtrace.bag B st' =>
    match n with
    | 0 => More
    | S n' =>
        Bag (map (obsevent.of_event sig) B) (observe' n' (bag.update sig B) st')
    end
  | simtrace.dead => Dead
  end.
Definition observe n st : observation :=
  observe' n state.zero st.

(* ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~ *)

(* value notations *)
Notation "()" := (@None val.t) (at level 0).
Notation "( k )" := (@Some val.t k) (at level 0, format "( k )").

(* guard notations *)
Notation ".true" := (guard.bool true) (at level 0).
Notation ".false" := (guard.bool false) (at level 0).
Notation "- A" := (guard.sprb A) (at level 35, right associativity).
Notation "^ A" := (guard.rprb A) (at level 35, right associativity).
Notation "# A = e" := (guard.dprb A e) (at level 70).
Notation "e = f" := (guard.expr e f) (at level 70).
Notation "~ G" :=
  (guard.neg G) (at level 75, right associativity).
Notation "G /\ H" := (guard.and G H) (at level 80, right associativity).
Notation "G \/ H" := (guard.or G H) (at level 85, right associativity).

(* obsguard notations *)
Notation ".T" := (obsguard.bool true) (at level 0).
Notation ".F" := (obsguard.bool false) (at level 0).
Notation "-- A (T)" :=
  (obsguard.sprb A true) (at level 35, right associativity, format "-- A (T)").
Notation "-- A (F)" :=
  (obsguard.sprb A false) (at level 35, right associativity, format "-- A (F)").
Notation "^^ A (T)" :=
  (obsguard.rprb A true) (at level 35, right associativity, format "^^ A (T)").
Notation "^^ A (F)" :=
  (obsguard.rprb A false) (at level 35, right associativity, format "^^ A (F)").
Notation "## A o == e p" :=
  (obsguard.dprb A o e p) (at level 70, format "## A o == e p").
Notation "e o == f p" :=
  (obsguard.expr e o f p) (at level 70, format "e o == f p").
Notation "~~ G" :=
  (obsguard.neg G) (at level 75, right associativity, format "~~ G").
Notation "G / \ H" :=
  (obsguard.and G H) (at level 80, right associativity, format "G / \ H").
Notation "G \ / H" :=
  (obsguard.or G H) (at level 85, right associativity, format "G \ / H").

(* obsevent notations *)
Notation "x <- e o" := (obsevent.assn x e o) (at level 0, format "x <- e o").
Notation "A !^ e o" :=
  (obsevent.send A (Some (e, o))) (at level 0, format "A !^ e o").
Notation "A !!" := (obsevent.send A None) (at level 0, format "A !!").
Notation "A ?^" := (obsevent.recv A None) (at level 0, format "A ?^").
Notation "A ?? x o" :=
  (obsevent.recv A (Some (x, o))) (at level 0, format "A ?? x o").
Notation "[ G ]" := (obsevent.wait G) (at level 0, G at level 200).
Notation ".detv" := (obsevent.detv) (at level 0).

(* communication notations *)
Notation "A ! e" := (comm.send A e) (at level 35).
Notation "A ? v" := (comm.recv A v) (at level 35).
Notation "C * D" := (comm.sim C D) (at level 40, left associativity).

(* program notations *)
Notation ".skip" := prgm.skip (at level 0).
Notation "v :== e" := (prgm.assn v e) (at level 40).
Notation "' C " := (prgm.comm C) (at level 43).
Notation "P ; Q" := (prgm.seq P Q) (at level 45).
Notation "P || Q" := (prgm.par P Q) (at level 50, left associativity).
Notation "[ G -> P [] H -> Q ]" :=
  (prgm.dsel G P H Q) (at level 40, G at level 85, H at level 85).
Notation "[ G -> P | H -> Q ]" :=
  (prgm.nsel G P H Q) (at level 40, G at level 85, H at level 85).
Notation "*[ G -> P ]" := (prgm.rep G P) (at level 45, G at level 85).

(* list notations *)
Notation "{}" := nil.
Notation "{ x }" := (cons x nil) (at level 0, x at level 99, format "{ x }").
Notation "{ x ; y ; .. ; z }" :=
  (cons x (cons y .. (cons z nil) ..)) (format "{ x ; y ; .. ; z }").
Notation "( x y .. z o )" := (Bag x (Bag y .. (Bag z o) ..)).

End MObsrvr.

(* (c) 2020 Brittany Ro Nkounkou *)
