(* Brittany Nkounkou *)
(* August 2020 *)
(* The First Asynchronous Microprocessor Environment *)

Require Export Obsrvr.

Set Implicit Arguments.

Module Type Imem.
  Parameter imem : nat -> nat.
End Imem.

Module MFamEnv (Export I : Imem) <: Environment.

Module val <: DecidableType.

(* value *)
Definition t : Type :=
  nat.

(* decidable value equality*)
Definition dec : eq_dec t :=
  PeanoNat.Nat.eq_dec.

End val.

Definition add : val.t := 0.
Definition sub : val.t := 1.
Definition ld : val.t := 2.
Definition st : val.t := 3.
Definition ldx : val.t := 4.
Definition stx : val.t := 5.
Definition lda : val.t := 6.
Definition stpc : val.t := 7.
Definition jmp : val.t := 8.
Definition brch : val.t := 9.

(* ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~ *)

(* data variable *)
Inductive dvar_t : Type :=
| pc | i | offset | pc' | i' | f | f'
| aop | ax | ay | az | mx | my | ma | mw
| dmem0 | dmem1 | dmem2 | dmem3
| b : val.t -> dvar_t | r : val.t -> dvar_t
| ignE2 | ignPc | ignXof | ignMC
| ignXs | ignYs | ignZWs | ignZAs | ignZRs.

Module dvar <: DecidableType.

Definition t : Type := dvar_t.

(* decidable data variable equality *)
Definition dec : eq_dec t.
Proof.
  unfold eq_dec, t. decide equality; apply val.dec.
Defined.

End dvar.

(* ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~ *)

(* channel *)
Inductive chan_t : Type :=
| ID | PCI1 | PCI2 | PCA1 | PCA2 | Xpc | Ypc | Xof
| E1 | E2 | X | Y | Xs | Ys | AC | MC1 | MC2 | MC3
| ZAs | ZRs | ZWs | F | ZA | ZM | MDl | MDs.

Module chan <: DecidableType.

Definition t : Type := chan_t.

(* decidable channel equality *)
Definition dec : eq_dec t.
Proof.
  unfold eq_dec, t. decide equality.
Defined.

End chan.

(* ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~ *)

(* expression *)
Inductive expr_t : Type :=
| val : val.t -> expr_t
| dvar : dvar.t -> expr_t
| plus : expr_t -> expr_t -> expr_t
| imem : dvar.t -> expr_t
| mod4 : expr_t -> expr_t
| op : dvar.t -> expr_t
| x : dvar.t -> expr_t
| y : dvar.t -> expr_t
| z : dvar.t -> expr_t
| cc : dvar.t -> expr_t
| aluz : dvar.t -> dvar.t -> dvar.t -> expr_t
| aluf : dvar.t -> dvar.t -> dvar.t -> expr_t.

Module expr <: DecidableType.

Definition t : Type := expr_t.

(* decidable expression equality *)
Definition dec : eq_dec t.
Proof.
  unfold eq_dec, t. decide equality; try apply val.dec; try apply dvar.dec.
Defined.

End expr.

(* ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~ *)

(* data variable inclusion in expression *)
Inductive expr_In' : dvar.t -> expr.t -> Prop :=
| expr_In_dvar v : expr_In' v (dvar v)
| expr_In_plus v e1 e2 :
    expr_In' v e1 \/ expr_In' v e2 -> expr_In' v (plus e1 e2)
| expr_In_imem v : expr_In' v (imem v)
| expr_In_mod4 v e : expr_In' v e -> expr_In' v (mod4 e)
| expr_In_op v : expr_In' v (op v)
| expr_In_x v : expr_In' v (x v)
| expr_In_y v : expr_In' v (y v)
| expr_In_z v : expr_In' v (z v)
| expr_In_cc v : expr_In' v (cc v)
| expr_In_aluz v v1 v2 v3 :
    v = v1 \/ v = v2 \/ v = v3 -> expr_In' v (aluz v1 v2 v3)
| expr_In_aluf v v1 v2 v3 :
    v = v1 \/ v = v2 \/ v = v3 -> expr_In' v (aluf v1 v2 v3).
Definition expr_In : dvar.t -> expr.t -> Prop := expr_In'.

(* expression evaluation *)
Fixpoint expr_eval (f : dvar.t -> option val.t) (e : expr.t) : option val.t :=
  match e with
  | val k => Some k
  | dvar v => f v
  | plus e1 e2 =>
    match expr_eval f e1, expr_eval f e2 with
    | Some k1, Some k2 => Some (k1 + k2)
    | _, _ => None
    end
  | imem v => match f v with Some k => Some (I.imem k) | _ => None end
  | mod4 e =>
    match expr_eval f e with Some k => Some (Nat.modulo k 4) | _ => None end
  | op v =>
    match f v with Some k => Some (Nat.modulo (Nat.div k 1000) 10) | _ => None
    end
  | x v =>
    match f v with Some k => Some (Nat.modulo (Nat.div k 100) 10) | _ => None
    end
  | y v =>
    match f v with Some k => Some (Nat.modulo (Nat.div k 10) 10) | _ => None end
  | z v => match f v with Some k => Some (Nat.modulo k 10) | _ => None end
  | cc v => match f v with Some k => Some (Nat.modulo k 10) | _ => None end
  | aluz v1 v2 v3 =>
    match f v1, f v2, f v3 with
    | Some k1, Some k2, Some k3 => Some
      (if val.dec k3 add then k1 + k2
        else if val.dec k3 sub then k1 - k2
        else 0)
    | _, _, _ => None
    end
  | aluf v1 v2 v3 =>
    match f v1, f v2, f v3 with
    | Some k1, Some k2, Some k3 => Some
      (if val.dec k3 sub
        then if PeanoNat.Nat.ltb k1 k2 then 1 else 0
        else 0)
    | _, _, _ => None
    end
  end.

(* expression evaluation equality *)
Lemma expr_eval_eq f f' e :
  (forall x, expr_In x e -> f x = f' x) -> expr_eval f e = expr_eval f' e.
Proof.
  intro.
  induction e; auto; simpl; try repeat rewrite H; auto; try constructor; auto.
  rewrite IHe1, IHe2; auto; intros; apply H; constructor; auto.
  rewrite IHe; auto. intros. apply H. constructor. auto.
Qed.

End MFamEnv.

(* (c) 2020 Brittany Ro Nkounkou *)
