(* Brittany Nkounkou *)
(* August 2020 *)
(* The First Asynchronous Microprocessor *)

Require Export FamEnv.

Set Implicit Arguments.

Module MFamDef (I : Imem).
Module env := MFamEnv I.
Module Export M := MObsrvr env.

(* data variable notations *)
Notation "dmem[0]" := dmem0 (at level 0).
Notation "dmem[1]" := dmem1 (at level 0).
Notation "dmem[2]" := dmem2 (at level 0).
Notation "dmem[3]" := dmem3 (at level 0).
Notation ".b0" := (b 0) (at level 0).
Notation ".b1" := (b 1) (at level 0).
Notation ".b2" := (b 2) (at level 0).
Notation ".b3" := (b 3) (at level 0).
Notation ".r0" := (r 0) (at level 0).
Notation ".r1" := (r 1) (at level 0).
Notation ".r2" := (r 2) (at level 0).
Notation ".r3" := (r 3) (at level 0).

(* expression notations *)
Notation "$ k" := (val k) (at level 0, format "$ k").
Notation "& v" := (dvar v) (at level 0, format "& v").
Notation "e + f" := (plus e f) (at level 50, left associativity).
Notation "imem[ v ]" := (imem v) (at level 0, format "imem[ v ]").
Notation "e %4" := (mod4 e) (at level 0, format "e %4").
Notation "v .op" := (op v) (at level 0, format "v .op").
Notation "v .x" := (x v) (at level 0, format "v .x").
Notation "v .y" := (y v) (at level 0, format "v .y").
Notation "v .z" := (z v) (at level 0, format "v .z").
Notation "v .cc" := (cc v) (at level 0, format "v .cc").

(* ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~ *)

Definition IMEM : prgm.t :=
  *[ .true -> [ ^ID -> 'ID!(imem[pc])
              | .false -> .skip ] ].

(* ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~ *)

Definition offset' e : guard.t :=
  e = $ldx \/ e = $stx \/ e = $lda \/ e = $brch.

Definition FETCH : prgm.t :=
  *[ .true -> 'PCI1!($0); 'ID?(i); 'PCI2!($0);
              [ offset' (op i) -> 'PCI1!($0); 'ID?(offset); 'PCI2!($0)
             [] ~ (offset' (op i)) -> .skip ];
              'E1!&i; 'E2?(ignE2) ].

(* ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~ *)

Definition PCADD : prgm.t :=
  pc :== $0;
  (  *[ .true -> [ -PCI1 \/ -PCA1 ->
                   [ -PCI1 -> 'PCI1?(ignPc); pc' :== (&pc + $1);
                              'PCI2?(ignPc); pc :== &pc'
                   [] -PCA1 -> 'PCA1?(ignPc); pc' :== (&pc + &offset);
                               'PCA2?(ignPc); pc :== &pc' ]
                [] -Xpc \/ -Ypc -> [ -Xpc -> 'X!&pc*Xpc?(ignPc)
                                  [] -Ypc -> 'Y?(pc)*Ypc?(ignPc) ] ] ]
  || *[ .true -> [ -Xof -> 'X!&offset*Xof?(ignXof)
                 | .false -> .skip ] ]).

(* ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~ *)

Definition alu e : guard.t :=
  e = $add \/ e = $sub.

Definition EXECa : prgm.t :=
  [ alu(i'.op) -> 'Xs!($0)*Ys!($0)*AC!(i'.op)*ZAs!($0)
 [] i'.op = $ld \/ i'.op = $st ->
    [ i'.op = $ld -> 'Xs!($0)*Ys!($0)*MC1!($0)*ZRs!($0)
   [] i'.op = $st -> 'Xs!($0)*Ys!($0)*MC2!($0)*ZWs!($0) ] ].

Definition EXECb1 : prgm.t :=
  [ i'.op = $ldx \/ i'.op = $stx ->
    [ i'.op = $ldx -> 'Xof!($0)*Ys!($0)*MC1!($0)*ZRs!($0)
   [] i'.op = $stx -> 'Xof!($0)*Ys!($0)*MC2!($0)*ZWs!($0) ]
 [] i'.op = $lda -> 'Xof!($0)*Ys!($0)*MC3!($0)*ZRs!($0) ].

Definition cond e1 e2 : guard.t :=
  ~ e1 = $0 \/ ~ e2 = $0.

Definition EXECb2 : prgm.t :=
  [ i'.op = $stpc \/ i'.op = $jmp ->
    [ i'.op = $stpc -> 'Xpc!($0)*Ys!($0)*AC!($add)*ZAs!($0)
   [] i'.op = $jmp -> 'Ypc!($0)*Ys!($0) ]
 [] i'.op = $brch -> 'F?(f');
                     [ cond &f' (i'.cc) -> 'PCA1!($0); 'PCA2!($0)
                    [] ~ (cond &f' (i'.cc)) -> .skip ] ].

Definition EXECb : prgm.t :=
  [ i'.op = $ldx \/ i'.op = $stx \/ i'.op = $lda -> EXECb1
 [] i'.op = $stpc \/ i'.op = $jmp \/ i'.op = $brch -> EXECb2 ].

Definition EXEC : prgm.t :=
  *[ .true -> 'E1?(i');
              [ alu(i'.op) \/ i'.op = $ld \/ i'.op = $st -> 'E2!($0); EXECa
             [] i'.op = $ldx \/ i'.op = $stx \/ i'.op = $lda \/
                i'.op = $stpc \/ i'.op = $jmp \/ i'.op = $brch ->
                  EXECb; 'E2!($0) ] ].

(* ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~ *)

Definition ALU : prgm.t :=
  f :== $0;
  *[ .true -> [ -AC -> 'AC?(aop)*X?(ax)*Y?(ay);
                       (az :== aluz ax ay aop || f :== aluf ax ay aop); 'ZA!&az
             [] ^F -> 'F!&f ] ].

(* ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~ *)

Definition MU : prgm.t :=
  *[ .true -> [ -MC1 \/ -MC2 ->
                [ -MC1 -> 'X?(mx)*Y?(my)*MC1?(ignMC);
                          ma :== (&mx + &my); 'MDl?(mw); 'ZM?(mw)
               [] -MC2 -> 'X?(mx)*Y?(my)*MC2?(ignMC)*ZM?(mw);
                          ma :== (&mx + &my); 'MDs!&mw ]
             [] -MC3 -> 'X?(mx)*Y?(my)*MC3?(ignMC);
                        ma :== (&mx + &my); 'ZM!&ma ] ].

(* ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~ *)

Definition DMEM : prgm.t :=
  dmem[0] :== $0; dmem[1] :== $0; dmem[2] :== $0; dmem[3] :== $0;
  *[ .true -> [ ^MDl -> [ &ma%4 = $0 \/ &ma%4 = $1 ->
                          [ &ma%4 = $0 -> 'MDl!&dmem[0]
                         [] &ma%4 = $1 -> 'MDl!&dmem[1] ]
                       [] &ma%4 = $2 \/ &ma%4 = $3 ->
                          [ &ma%4 = $2 -> 'MDl!&dmem[2]
                         [] &ma%4 = $3 -> 'MDl!&dmem[3] ] ]
             [] -MDs -> [ &ma%4 = $0 \/ &ma%4 = $1 ->
                          [ &ma%4 = $0 -> 'MDs?(dmem[0])
                         [] &ma%4 = $1 -> 'MDs?(dmem[1]) ]
                       [] &ma%4 = $2 \/ &ma%4 = $3 ->
                          [ &ma%4 = $2 -> 'MDs?(dmem[2])
                         [] &ma%4 = $3 -> 'MDs?(dmem[3]) ] ] ] ].

(* ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~ *)

Definition REG k : prgm.t :=
  b k :== $0; r k :== $0;
  ( *[ .true -> [ &(b k) = $0 /\ $k = (i'.x)%4 /\ -Xs ->
                     'X!&(r k)*Xs?(ignXs)
                 | .false -> .skip ] ]
  || *[ .true -> [ &(b k) = $0 /\ $k = (i'.y)%4 /\ -Ys ->
                     'Y!&(r k)*Ys?(ignYs)
                 | .false -> .skip ] ]
  || *[ .true -> [ &(b k) = $0 /\ $k = (i'.z)%4 /\ -ZWs ->
                     'ZM!&(r k)*ZWs?(ignZWs)
                 | .false -> .skip ] ]
  || *[ .true -> [ &(b k) = $0 /\ $k = (i'.z)%4 /\ -ZAs ->
                     b k :== $1; 'ZAs?(ignZAs); 'ZA?(r k) ; b k :== $0
                 | .false -> .skip ] ]
  || *[ .true -> [ &(b k) = $0 /\ $k = (i'.z)%4 /\ -ZRs ->
                     b k :== $1; 'ZRs?(ignZRs); 'ZM?(r k) ; b k :== $0
                 | .false -> .skip ] ]).

(* ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~ *)

(* top-level program syntax *)
Definition FAM : prgm.t :=
  IMEM || FETCH || PCADD || EXEC || ALU || MU || DMEM
  || REG 0 || REG 1 || REG 2 || REG 3.

End MFamDef.

(* (c) 2020 Brittany Ro Nkounkou *)
