(* Brittany Nkounkou *)
(* August 2020 *)
(* Events *)

Require Export States.
Require Import Omega Permutation.

Set Implicit Arguments.

Module MEvents (env : Environment).
Module Export M := MStates env.

Module event <: DecidableType.

(* event *)
Inductive t' : Type :=
| assn : dvar.t -> expr.t -> t'
| send : chan.t -> option expr.t -> t'
| recv : chan.t -> option dvar.t -> t'
| wait : guard.t -> t'
| detv : t'.
Definition t : Type := t'.

(* decidable event equality *)
Definition dec : eq_dec t.
Proof.
  unfold eq_dec, t. repeat decide equality;
    try apply dvar.dec; try apply expr.dec; try apply chan.dec.
Defined.

(* state update with event *)
Definition update sig E : state.t :=
  match E with
  | assn x e => state.set_dvar sig x (state.eval_expr sig e)
  | send A (Some e) =>
      state.set_sprb (state.set_dprb sig A (state.eval_expr sig e)) A true
  | send A None => state.set_sprb sig A false
  | recv A None => state.set_rprb sig A true
  | recv A (Some x) =>
      state.set_rprb
        (state.set_dprb (state.set_dvar sig x (state.get_dprb sig A)) A None)
        A false
  | wait G => sig
  | detv => sig
  end.

(* ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~ *)

(* uninitialized event *)
Inductive Uninit sig : t -> Prop :=
| Uninit_assn x e :
    state.eval_expr sig e = None -> Uninit sig (assn x e)
| Uninit_send A e :
    state.eval_expr sig e = None -> Uninit sig (send A (Some e))
| Uninit_recv A x :
    state.get_dprb sig A = None -> Uninit sig (recv A (Some x)).

(* event that reads a mutable *)
Inductive Reads : mut.t -> t -> Prop :=
| Reads_assn x y e : expr_In x e -> Reads (mut.dvar x) (assn y e)
| Reads_send x A e : expr_In x e -> Reads (mut.dvar x) (send A (Some e))
| Reads_recv x A : Reads (mut.dprb A) (recv A (Some x)).

(* event that writes to a mutable *)
Inductive Writes : mut.t -> t -> Prop :=
| Writes_assn x e : Writes (mut.dvar x) (assn x e)
| Writes_send_dprb A e : Writes (mut.dprb A) (send A (Some e))
| Writes_send_sprb A o : Writes (mut.sprb A) (send A o)
| Writes_recv_dvar A x : Writes (mut.dvar x) (recv A (Some x))
| Writes_recv_dprb A x : Writes (mut.dprb A) (recv A (Some x))
| Writes_recv_rprb A o : Writes (mut.rprb A) (recv A o).

(* two conflicting events *)
Inductive Conflict E1 E2 : Prop :=
| Reads_Writes X : Reads X E1 -> Writes X E2 -> Conflict E1 E2
| Writes_Reads X : Writes X E1 -> Reads X E2 -> Conflict E1 E2
| Writes_Writes X : Writes X E1 -> Writes X E2 -> Conflict E1 E2.

(* ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~ *)

(* equal states produce equal event updates *)
Lemma eq_update_eq sig sig' E :
  state.eq sig sig' -> state.eq (update sig E) (update sig' E).
Proof.
  intro. destruct E; simpl; auto.
    erewrite state.eq_eval_expr_eq; eauto. apply state.eq_set_dvar_eq. auto.
    destruct o.
      erewrite state.eq_eval_expr_eq; eauto. apply state.eq_set_sprb_eq.
        apply state.eq_set_dprb_eq. auto.
      apply state.eq_set_sprb_eq. auto.
    destruct o.
      erewrite state.eq_get_dprb_eq; eauto. apply state.eq_set_rprb_eq.
        apply state.eq_set_dprb_eq. apply state.eq_set_dvar_eq. auto.
      apply state.eq_set_rprb_eq. auto.
Qed.

(* if two events are not conflicting, then their update order does not matter *)
Fact update_update sig E1 E2 :
  ~ Conflict E1 E2 ->
  state.eq (update (update sig E1) E2) (update (update sig E2) E1).
Proof.
  intro. destruct E1, E2; simpl; try apply state.eq_refl.
    repeat rewrite state.eval_expr_set_dvar. apply state.set_dvar_set_dvar.
      intro. apply H. rewrite H0. eapply Writes_Writes; constructor.
      intro. apply H. eapply Reads_Writes; constructor; auto.
      intro. apply H. eapply Writes_Reads; constructor; auto.
    destruct o.
      rewrite state.eval_expr_set_dvar. rewrite state.eval_expr_set_sprb.
        rewrite state.eval_expr_set_dprb. apply state.eq_sym.
        eapply state.eq_trans. apply state.set_dvar_set_sprb.
        apply state.eq_set_sprb_eq. apply state.set_dvar_set_dprb.
          intro. apply H. eapply Writes_Reads; constructor; auto.
      rewrite state.eval_expr_set_sprb. apply state.eq_sym.
        apply state.set_dvar_set_sprb.
    destruct o.
      rewrite state.get_dprb_set_dvar. rewrite state.eval_expr_set_rprb.
        rewrite state.eval_expr_set_dprb. rewrite state.eval_expr_set_dvar.
        apply state.eq_sym. eapply state.eq_trans.
        apply state.set_dvar_set_rprb. apply state.eq_set_rprb_eq.
        eapply state.eq_trans. apply state.set_dvar_set_dprb.
        apply state.eq_set_dprb_eq. apply state.set_dvar_set_dvar.
          intro. apply H. rewrite H0. eapply Writes_Writes; constructor.
          intro. apply H. eapply Reads_Writes; constructor; auto.
      rewrite state.eval_expr_set_rprb. apply state.eq_sym.
        apply state.set_dvar_set_rprb.
    destruct o.
      rewrite state.eval_expr_set_sprb. rewrite state.eval_expr_set_dprb.
        rewrite state.eval_expr_set_dvar. eapply state.eq_trans.
        apply state.set_dvar_set_sprb. apply state.eq_set_sprb_eq.
        apply state.set_dvar_set_dprb.
          intro. apply H. eapply Reads_Writes; constructor; auto.
      rewrite state.eval_expr_set_sprb. apply state.set_dvar_set_sprb.
    destruct o, o0.
      repeat rewrite state.eval_expr_set_sprb.
        repeat rewrite state.eval_expr_set_dprb. eapply state.eq_trans.
        apply state.eq_set_sprb_eq. apply state.eq_sym.
        apply state.set_sprb_set_dprb. eapply state.eq_trans.
        apply state.set_sprb_set_sprb.
          intro. apply H. rewrite H0. eapply Writes_Writes; constructor; auto.
        apply state.eq_set_sprb_eq. apply state.eq_sym. eapply state.eq_trans.
        apply state.eq_sym. apply state.set_sprb_set_dprb.
        apply state.eq_set_sprb_eq. apply state.set_dprb_set_dprb.
          intro. apply H. rewrite H0. eapply Writes_Writes; constructor; auto.
      rewrite state.eval_expr_set_sprb. eapply state.eq_trans.
        apply state.set_sprb_set_sprb.
          intro. apply H. rewrite H0. eapply Writes_Writes; constructor; auto.
        apply state.eq_set_sprb_eq. apply state.set_sprb_set_dprb.
      rewrite state.eval_expr_set_sprb. apply state.eq_sym.
        eapply state.eq_trans. apply state.set_sprb_set_sprb.
          intro. apply H. rewrite H0. eapply Writes_Writes; constructor; auto.
        apply state.eq_set_sprb_eq. apply state.set_sprb_set_dprb.
      apply state.set_sprb_set_sprb.
          intro. apply H. rewrite H0. eapply Writes_Writes; constructor; auto.
    destruct o, o0.
      rewrite state.get_dprb_set_sprb. rewrite state.get_dprb_set_dprb.
        rewrite state.eval_expr_set_rprb. rewrite state.eval_expr_set_dprb.
        rewrite state.eval_expr_set_dvar. apply state.eq_sym.
        eapply state.eq_trans. apply state.eq_set_sprb_eq. apply state.eq_sym.
        apply state.set_rprb_set_dprb. eapply state.eq_trans.
        apply state.eq_sym. apply state.set_rprb_set_sprb.
        apply state.eq_set_rprb_eq. eapply state.eq_trans.
        apply state.eq_set_sprb_eq. apply state.set_dprb_set_dprb.
          intro. apply H. rewrite H0. eapply Writes_Writes; constructor; auto.
        eapply state.eq_trans. apply state.set_sprb_set_dprb.
        apply state.eq_set_dprb_eq. apply state.eq_sym. eapply state.eq_trans.
        apply state.set_dvar_set_sprb. apply state.eq_set_sprb_eq.
        apply state.set_dvar_set_dprb.
          intro. apply H. eapply Reads_Writes; constructor; auto.
          intro. apply H. rewrite H0. eapply Writes_Writes; constructor; auto.
      rewrite state.eval_expr_set_rprb. eapply state.eq_trans.
        apply state.set_rprb_set_sprb. apply state.eq_set_sprb_eq.
          apply state.set_rprb_set_dprb.
      rewrite state.get_dprb_set_sprb. apply state.eq_sym.
        eapply state.eq_trans. apply state.eq_sym.
        apply state.set_rprb_set_sprb. apply state.eq_set_rprb_eq.
        apply state.set_sprb_set_dprb.
      apply state.set_rprb_set_sprb.
    destruct o.
      rewrite state.eval_expr_set_rprb. rewrite state.eval_expr_set_dprb.
        rewrite state.eval_expr_set_dvar. rewrite state.get_dprb_set_dvar.
        eapply state.eq_trans. apply state.set_dvar_set_rprb.
        apply state.eq_set_rprb_eq. eapply state.eq_trans.
        apply state.set_dvar_set_dprb. apply state.eq_set_dprb_eq.
        apply state.set_dvar_set_dvar.
          intro. apply H. rewrite H0. eapply Writes_Writes; constructor.
          intro. apply H. eapply Writes_Reads; constructor; auto.
      rewrite state.eval_expr_set_rprb. apply state.set_dvar_set_rprb.
    destruct o, o0.
      rewrite state.eval_expr_set_rprb. rewrite state.eval_expr_set_dprb.
        rewrite state.eval_expr_set_dvar. rewrite state.get_dprb_set_sprb.
        rewrite state.get_dprb_set_dprb. eapply state.eq_trans.
        apply state.eq_set_sprb_eq. apply state.eq_sym.
        apply state.set_rprb_set_dprb. eapply state.eq_trans.
        apply state.eq_sym. apply state.set_rprb_set_sprb.
        apply state.eq_set_rprb_eq. eapply state.eq_trans.
        apply state.eq_set_sprb_eq. apply state.set_dprb_set_dprb.
          intro. apply H. rewrite H0. eapply Writes_Writes; constructor; auto.
        eapply state.eq_trans. apply state.set_sprb_set_dprb.
        apply state.eq_set_dprb_eq. apply state.eq_sym. eapply state.eq_trans.
        apply state.set_dvar_set_sprb. apply state.eq_set_sprb_eq.
        apply state.set_dvar_set_dprb.
          intro. apply H. rewrite H0. eapply Writes_Writes; constructor; auto.
          intro. apply H. eapply Writes_Reads; constructor; auto.
      rewrite state.get_dprb_set_sprb. eapply state.eq_trans.
        apply state.eq_sym. apply state.set_rprb_set_sprb.
        apply state.eq_set_rprb_eq. apply state.set_sprb_set_dprb.
      rewrite state.eval_expr_set_rprb. apply state.eq_sym.
        eapply state.eq_trans. apply state.set_rprb_set_sprb.
        apply state.eq_set_sprb_eq. apply state.set_rprb_set_dprb.
      apply state.eq_sym. apply state.set_rprb_set_sprb.
    destruct o, o0.
      repeat rewrite state.get_dprb_set_rprb.
        repeat rewrite state.get_dprb_set_dprb.
        repeat rewrite state.get_dprb_set_dvar. eapply state.eq_trans.
        apply state.eq_set_rprb_eq. apply state.eq_set_dprb_eq.
        apply state.set_dvar_set_rprb. eapply state.eq_trans.
        apply state.eq_set_rprb_eq. apply state.eq_sym.
        apply state.set_rprb_set_dprb. eapply state.eq_trans.
        apply state.set_rprb_set_rprb.
          intro. apply H. rewrite H0. eapply Writes_Writes; constructor; auto.
        apply state.eq_set_rprb_eq. apply state.eq_sym. eapply state.eq_trans.
        apply state.eq_set_dprb_eq. apply state.set_dvar_set_rprb.
        eapply state.eq_trans. apply state.eq_sym.
        apply state.set_rprb_set_dprb. apply state.eq_set_rprb_eq.
        eapply state.eq_trans. apply state.eq_set_dprb_eq.
        apply state.set_dvar_set_dprb. eapply state.eq_trans.
        apply state.set_dprb_set_dprb.
          intro. apply H. rewrite H0. eapply Writes_Writes; constructor; auto.
        apply state.eq_set_dprb_eq. apply state.eq_sym. eapply state.eq_trans.
        apply state.set_dvar_set_dprb. apply state.eq_set_dprb_eq.
        apply state.set_dvar_set_dvar.
          intro. apply H. rewrite H0. eapply Writes_Writes; constructor; auto.
          intro. apply H. rewrite H0. eapply Writes_Writes; constructor; auto.
          intro. apply H. rewrite H0. eapply Writes_Writes; constructor; auto.
      rewrite state.get_dprb_set_rprb. eapply state.eq_trans.
        apply state.set_rprb_set_rprb.
          intro. apply H. rewrite H0. eapply Writes_Writes; constructor; auto.
        apply state.eq_set_rprb_eq. eapply state.eq_trans.
        apply state.set_rprb_set_dprb. apply state.eq_set_dprb_eq.
        apply state.eq_sym. apply state.set_dvar_set_rprb.
      rewrite state.get_dprb_set_rprb. apply state.eq_sym.
        eapply state.eq_trans. apply state.set_rprb_set_rprb.
          intro. apply H. rewrite H0. eapply Writes_Writes; constructor; auto.
        apply state.eq_set_rprb_eq. eapply state.eq_trans.
        apply state.set_rprb_set_dprb. apply state.eq_set_dprb_eq.
        apply state.eq_sym. apply state.set_dvar_set_rprb.
      apply state.set_rprb_set_rprb.
          intro. apply H. rewrite H0. eapply Writes_Writes; constructor; auto.
Qed.

End event.

(* ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~ *)

Module bag.

(* bags of events *)
Definition t : Type :=
  list event.t.

(* event-bag union *)
Definition union : t -> t -> t :=
  @app event.t.

(* event-bag order: subbag *)
Definition le (B1 B2 : t) : Prop :=
  forall E, count_occ event.dec B1 E <= count_occ event.dec B2 E.

(* state update with bag of events *)
Definition update sig B : state.t :=
  fold_left event.update B sig.

(* ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~ *)

(* synchronized bag of events *)
Definition Sync B : Prop :=
  forall A, In (event.send A None) B <-> exists x, In (event.recv A (Some x)) B.

(* satisfied bag of events *)
Definition Sat sig B : Prop :=
  forall G, In (event.wait G) B -> state.eval_guard sig G = Some true.

(* feasible bag of events *)
Definition Feas sig B : Prop :=
  Sync B /\ Sat sig B.

(* invalid bag of events *)
Definition Inval sig B : Prop :=
  exists G, In (event.wait G) B /\ state.eval_guard sig G = None.

(* uninitialized bag of events *)
Definition Uninit sig B : Prop :=
  exists E, In E B /\ event.Uninit sig E.

(* interfering bag of events *)
Definition Int B : Prop :=
  In event.detv B \/ exists E1 E2, le [E1; E2] B /\ event.Conflict E1 E2.

(* unstable bag of events *)
Definition Unstable sig B : Prop :=
  exists B', le B' B /\ ~ Sat (update sig B') B.

(* erroneous bag of events *)
Definition Error sig B : Prop :=
  Sync B /\
  (Inval sig B \/ Sat sig B /\ (Uninit sig B \/ Int B \/ Unstable sig B)).

(* ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~ *)

(* the first two elements of an event bag are a subbag of the bag *)
Lemma le_hd_hd E1 E2 B : le [E1; E2] (E1 :: E2 :: B).
Proof.
  intro. simpl. destruct (event.dec E1 E), (event.dec E2 E); omega.
Qed.

(* the tail of an event bag is a subbag of the bag *)
Lemma le_tl E B : le B (E :: B).
Proof.
  intro. simpl. destruct (event.dec E E0); auto.
Qed.

(* a subbag of an event bag is a subbag of a permutation of the event bag *)
Lemma le_Permutation B B' B'' : le B B' -> Permutation B' B'' -> le B B''.
Proof.
  intros. intro. specialize (H E). generalize dependent B.
  induction H0; auto; simpl; intros.
    destruct (event.dec x E); auto. edestruct le_lt_or_eq. eauto.
      apply le_S. apply IHPermutation. apply lt_n_Sm_le. auto.
      rewrite H1. apply le_n_S. auto.
    destruct (event.dec x E), (event.dec y E); auto.
Qed.

(* equal states produce equal bag updates *)
Lemma eq_update_eq B :
  forall sig sig', state.eq sig sig' -> state.eq (update sig B) (update sig' B).
Proof.
  unfold update. induction B; simpl; intros; auto. apply IHB.
  apply event.eq_update_eq. auto.
Qed.

(* equal noninterfering bags produce the same updated state *)
Fact update_Permutation sig B : ~ Int B ->
  forall B', Permutation B B' -> state.eq (update sig B) (update sig B').
Proof.
  intros. generalize dependent sig.
  induction H0; intro; simpl.
    apply state.eq_refl.
    apply IHPermutation.
      intro. apply H. destruct H1 as [|[?[?[]]]]. left. right. auto.
        right. exists x0, x1. split; auto. intro. etransitivity. apply H1.
          apply le_tl.
    apply eq_update_eq. apply event.update_update. intro. apply H. right.
      exists y, x; split; auto. apply le_hd_hd.
    eapply state.eq_trans.
      apply IHPermutation1; auto.
      apply IHPermutation2. intro. apply H. destruct H0 as [|[?[?[]]]].
        left. apply Permutation_in with l'; auto. symmetry. auto.
        right. exists x, x0. split; auto. eapply le_Permutation; eauto.
          symmetry. auto.
Qed.

(* ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~ *)

(* decider for whether or not E is a receive down event *)
Definition recv_dec A E :
  {exists x, E = event.recv A (Some x)} +
  {~ exists x, E = event.recv A (Some x)}.
Proof.
  destruct E; try (right; intro; destruct H; discriminate).
  destruct o; try (right; intro; destruct H; discriminate).
  destruct (chan.dec A t0). rewrite e. eauto.
  right. intro. apply n. destruct H. inversion H. auto.
Defined.

(* decider for whether or not event bag B is synchronized *)
Definition sync' (B : bag.t) E : bool :=
  match E with
  | event.send A None =>
      if Exists_dec _ B (recv_dec A) then true else false
  | event.recv A (Some _) =>
      if in_dec event.dec (event.send A None) B then true else false
  | _ => true
  end.
Definition sync (B : bag.t) : bool :=
  forallb (sync' B) B.

(* decider for whether or not event bag B is satisfied *)
Definition sat' sig E : bool :=
  match E with
  | event.wait G =>
    match state.eval_guard sig G with
    | Some true => true
    | _ => false
    end
  | _ => true
  end.
Definition sat sig (B : bag.t) : bool :=
  forallb (sat' sig) B.

(* ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~ *)

(* event bags B1 and B2 are each satisfied iff B1 union B2 is satisfied *)
Lemma Sat_union sig (B1 B2 : bag.t) :
  Sat sig B1 /\ Sat sig B2 <-> Sat sig (bag.union B1 B2).
Proof.
  unfold Sat; intuition.
    edestruct in_app_or; eauto.
    apply H. apply in_or_app. auto.
    apply H. apply in_or_app. auto.
Qed.

(* sync correctness *)
Lemma sync_true B : sync B = true <-> Sync B.
Proof.
  etransitivity. apply forallb_forall. unfold sync', Sync. intuition.
    specialize (H _ H0). simpl in H. destruct Exists_dec in H.
      rewrite Exists_exists in e. destruct e as [?[?[]]]. rewrite H2 in H1.
      eauto. discriminate.
    destruct H0. specialize (H _ H0). simpl in H. destruct in_dec in H.
      auto. discriminate.
    destruct x; auto; destruct o; auto. destruct Exists_dec; auto.
      destruct n. apply Exists_exists. edestruct H. destruct H1. eauto. eauto.
      destruct in_dec; auto. destruct n. apply H. eauto.
Qed.

(* sat correctness *)
Lemma sat_true sig B : sat sig B = true <-> Sat sig B.
Proof.
  etransitivity. eapply forallb_forall. unfold sat', Sat. intuition.
    specialize (H _ H0). simpl in H. destruct (state.eval_guard sig G).
      destruct b. auto. inversion H. inversion H.
    destruct x; auto. rewrite H; auto.
Qed.

End bag.

End MEvents.

(* (c) 2020 Brittany Ro Nkounkou *)
