(* Brittany Nkounkou *)
(* August 2020 *)
(* Construction *)

Require Export PetRep.

Set Implicit Arguments.

Module MConstr (env : Environment).
Module Export M := MPetRep env.

Module comm.

(* communication program *)
Inductive t : Type :=
| send : chan.t -> expr.t -> t
| recv : chan.t -> dvar.t -> t
| sim : t -> t -> t.

(* initiation behavior *)
Fixpoint Init C : slang.t :=
  match C with
  | send A e => slang.bag [event.send A (Some e)]
  | recv A _ => slang.bag [event.recv A None]
  | sim C1 C2 => slang.concur (Init C1) (Init C2)
  end.

(* completion behavior *)
Fixpoint Comp C : bag.t :=
  match C with
  | send A e => [event.send A None]
  | recv A x => [event.recv A (Some x)]
  | sim C1 C2 => bag.union (Comp C1) (Comp C2)
  end.

(* ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~ *)

(* initial configuration *)
Fixpoint init C : config.t :=
  match C with
  | send _ _ => bag.init
  | recv _ _ => bag.init
  | sim C1 C2 => par.init (init C1) (init C2)
  end.

(* petri-net construction *)
Fixpoint constr C : petri.t :=
  match C with
  | send A e => bag.make [event.send A (Some e)]
  | recv A _ => bag.make [event.recv A None]
  | sim C1 C2 => par.make (constr C1) (constr C2)
  end.

(* empty-firing-chain-to-fin decider *)
Fixpoint ef C : petri.efc_fin :=
  match C with
  | send _ _ => bag.ef
  | recv _ _ => bag.ef
  | sim C1 C2 => par.ef (ef C1) (ef C2)
  end.

(* nth-bag finder *)
Fixpoint nb C : petri.nth_bag :=
  match C with
  | send A e => bag.nb [event.send A (Some e)]
  | recv A _ => bag.nb [event.recv A None]
  | sim C1 C2 => par.nb (nb C1) (nb C2)
  end.

(* ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~ *)

(* init is the initial configuration of constr *)
Lemma init_constr C : init C = petri.init (constr C).
Proof.
  induction C; simpl; auto. rewrite IHC1, IHC2. auto.
Qed.

(* constr is well-formed *)
Lemma Wf C : petri.Wf (constr C).
Proof.
  induction C; simpl; try apply bag.Wf. apply par.Wf; auto.
Qed.

(* constr is safe *)
Lemma Safe C : petri.Safe (constr C).
Proof.
  induction C; simpl; try apply bag.Safe. apply par.Safe; auto.
Qed.

(* ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~ *)

(* Lang soundness *)
Lemma Lang_sound C :
  slang.le (petri.Lang (constr C)) (Init C).
Proof.
  induction C; simpl; try apply bag.Lang_sound.
  eapply slang.le_trans. apply par.Lang_sound. eapply slang.concur_le_le; eauto.
Qed.

(* Lang completeness *)
Lemma Lang_complete C :
  slang.le (Init C) (petri.Lang (constr C)).
Proof.
  induction C; simpl; try apply bag.Lang_complete.
  eapply slang.le_trans. eapply slang.concur_le_le; eauto.
    apply par.Lang_complete.
Qed.

(* ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~ *)

(* ef correctness *)
Lemma EfcFin C : petri.EfcFin (constr C) (ef C).
Proof.
  induction C; simpl; try apply bag.EfcFin. apply par.EfcFin; auto.
Qed.

(* nb correctness *)
Lemma NthBag C : petri.NthBag (constr C) (nb C).
Proof.
  induction C; simpl; try apply bag.NthBag. apply par.NthBag; auto; apply Wf.
Qed.

End comm.

(* ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~ *)
 
Module prgm.

(* program *)
Inductive t : Type :=
| skip : t
| assn : dvar.t -> expr.t -> t
| comm : comm.t -> t
| seq : t -> t -> t
| par : t -> t -> t
| nsel : guard.t -> t -> guard.t -> t -> t
| dsel : guard.t -> t -> guard.t -> t -> t
| rep : guard.t -> t -> t.

(* raw behavior *)
Fixpoint Raw P : slang.t :=
  match P with
  | skip => slang.eps
  | assn x e => slang.bag [event.assn x e]
  | comm C => slang.concat (comm.Init C) (slang.bag (comm.Comp C))
  | seq P1 P2 => slang.concat (Raw P1) (Raw P2)
  | par P1 P2 => slang.concur (Raw P1) (Raw P2)
  | nsel G1 P1 G2 P2 =>
      slang.union
        (slang.concat (slang.bag [event.wait G1]) (Raw P1))
        (slang.concat (slang.bag [event.wait G2]) (Raw P2))
  | dsel G1 P1 G2 P2 =>
      slang.union (slang.union
        (slang.concat (slang.bag [event.wait G1]) (Raw P1))
        (slang.concat (slang.bag [event.wait G2]) (Raw P2)))
        (slang.bag (bag.union (bag.union [event.wait G1] [event.wait G2])
          [event.detv]))
  | rep G P =>
      slang.concat
        (slang.star opttrace.eps
          (slang.concat (slang.bag [event.wait G]) (Raw P)))
        (slang.bag [event.wait (guard.neg G)])
  end.

(* feasible behavior *)
Definition Feas P : trace.t -> Prop :=
  fun t => Raw P (trace.lift t) /\ trace.Feas state.zero t.

(* erroneous behavior *)
Definition Error P : trace.t -> Prop :=
  fun t => Raw P (trace.lift t) /\ trace.Error state.zero t.

(* ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~ *)

(* initial configuration *)
Fixpoint init P : config.t :=
  match P with
  | skip => eps.init
  | assn _ _ => bag.init
  | comm C => seq.init (comm.init C)
  | seq P1 _ => seq.init (init P1)
  | par P1 P2 => par.init (init P1) (init P2)
  | nsel _ _ _ _ => sel.init
  | dsel _ _ _ _ => sel.init
  | rep _ _ => seq.init (rep.init)
  end.

(* petri-net construction *)
Fixpoint constr P : petri.t :=
  match P with
  | skip => eps.make
  | assn x e => bag.make [event.assn x e]
  | comm C => seq.make (comm.constr C) (bag.make (comm.Comp C))
  | seq P1 P2 => seq.make (constr P1) (constr P2)
  | par P1 P2 => par.make (constr P1) (constr P2)
  | nsel G1 P1 G2 P2 =>
      sel.make
        (seq.make (bag.make [event.wait G1]) (constr P1))
        (seq.make (bag.make [event.wait G2]) (constr P2))
  | dsel G1 P1 G2 P2 =>
      sel.make (sel.make
        (seq.make (bag.make [event.wait G1]) (constr P1))
        (seq.make (bag.make [event.wait G2]) (constr P2)))
      (bag.make
        (bag.union (bag.union [event.wait G1] [event.wait G2]) [event.detv]))
  | rep G P =>
      seq.make (rep.make (seq.make (bag.make [event.wait G]) (constr P)))
        (bag.make [event.wait (guard.neg G)])
  end.

(* empty-firing-chain-to-fin decider *)
Fixpoint ef P : petri.efc_fin :=
  match P with
  | skip => eps.ef
  | assn _ _ => bag.ef
  | comm C => seq.ef (comm.ef C) bag.ef bag.init
  | seq P1 P2 => seq.ef (ef P1) (ef P2) (init P2)
  | par P1 P2 => par.ef (ef P1) (ef P2)
  | nsel _ P1 _ P2 =>
      sel.ef
        (seq.ef bag.ef (ef P1) (init P1))
        (seq.ef bag.ef (ef P2) (init P2))
        (seq.init bag.init)
        (seq.init bag.init)
  | dsel _ P1 _ P2 =>
      sel.ef (sel.ef
        (seq.ef bag.ef (ef P1) (init P1))
        (seq.ef bag.ef (ef P2) (init P2))
        (seq.init bag.init)
        (seq.init bag.init))
        bag.ef sel.init bag.init
  | rep _ P => seq.ef (rep.ef (seq.ef bag.ef (ef P) (init P))) bag.ef bag.init
  end.

(* nth-bag finder *)
Fixpoint nb P : petri.nth_bag :=
  match P with
  | skip => eps.nb
  | assn x e => bag.nb [event.assn x e]
  | comm C => seq.nb (comm.ef C) (comm.nb C) (bag.nb (comm.Comp C)) bag.init
  | seq P1 P2 => seq.nb (ef P1) (nb P1) (nb P2) (init P2)
  | par P1 P2 => par.nb (nb P1) (nb P2)
  | nsel G1 P1 G2 P2 =>
      sel.nb
        (seq.nb bag.ef (bag.nb [event.wait G1]) (nb P1) (init P1))
        (seq.nb bag.ef (bag.nb [event.wait G2]) (nb P2) (init P2))
        (seq.init bag.init)
        (seq.init bag.init)
  | dsel G1 P1 G2 P2 =>
      sel.nb (sel.nb
        (seq.nb bag.ef (bag.nb [event.wait G1]) (nb P1) (init P1))
        (seq.nb bag.ef (bag.nb [event.wait G2]) (nb P2) (init P2))
        (seq.init bag.init)
        (seq.init bag.init))
        (bag.nb
          (bag.union (bag.union [event.wait G1] [event.wait G2]) [event.detv]))
        sel.init bag.init
  | rep G P =>
      seq.nb (rep.ef (seq.ef bag.ef (ef P) (init P)))
        (rep.nb (seq.ef bag.ef (ef P) (init P))
          (seq.nb bag.ef (bag.nb [event.wait G]) (nb P) (init P))
          (seq.init bag.init))
        (bag.nb [event.wait (guard.neg G)]) bag.init
  end.

(* ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~ *)

(* init is the initial configuration of constr *)
Lemma init_constr P : init P = petri.init (constr P).
Proof.
  induction P; simpl; auto; try rewrite comm.init_constr;
  try rewrite IHP1; try rewrite IHP2; try rewrite IHCP; auto.
Qed.

(* constr is well-formed *)
Lemma Wf P : petri.Wf (constr P).
Proof.
  induction P; simpl;
    try apply eps.Wf;
    try apply par.Wf;
    try apply sel.Wf;
    try apply sel.Wf;
    try apply seq.Wf;
    try apply rep.Wf;
    try apply seq.Wf;
    try apply comm.Wf;
    try apply bag.Wf;
    auto.
Qed.

(* constr is safe *)
Lemma Safe P : petri.Safe (constr P).
Proof.
  induction P; simpl;
    try apply eps.Safe;
    try apply par.Safe;
    try apply sel.Safe;
    try apply sel.Safe;
    try apply seq.Safe;
    try apply rep.Safe;
    try apply seq.Safe;
    try apply comm.Safe;
    try apply bag.Safe;
    try apply comm.Wf;
    try apply sel.Wf;
    try apply rep.Wf;
    try apply seq.Wf;
    try apply bag.Wf;
    try apply Wf;
    auto.
Qed.

(* ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~ *)

(* Lang soundness *)
Lemma Lang_sound P :
  slang.le (petri.Lang (constr P)) (Raw P).
Proof.
  induction P; simpl;
    try apply eps.Lang_sound;
    try apply bag.Lang_sound;
    try eapply slang.le_trans;
    try apply seq.Lang_sound;
    try eapply slang.concat_le_le;
    try eapply comm.Lang_sound;
    try apply par.Lang_sound;
    try eapply slang.concur_le_le;
    try apply sel.Lang_sound;
    try eapply slang.union_le_le;
    try apply bag.Lang_sound;
    try apply comm.Safe;
    try apply sel.Safe;
    try apply rep.Safe;
    try apply seq.Safe;
    try apply bag.Safe;
    try apply Safe;
    try apply comm.Wf;
    try apply sel.Wf;
    try apply rep.Wf;
    try apply seq.Wf;
    try apply bag.Wf;
    try apply Wf;
    auto;
    try eapply slang.le_trans;
    try apply seq.Lang_sound;
    try eapply slang.concat_le_le;
    try apply sel.Lang_sound;
    try eapply slang.union_le_le;
    try apply rep.Lang_sound;
    try eapply slang.star_le;
    try apply bag.Lang_sound;
    try apply seq.Safe;
    try apply bag.Safe;
    try apply Safe;
    try apply seq.Wf;
    try apply bag.Wf;
    try apply Wf;
    auto;
    try eapply slang.le_trans;
    try apply seq.Lang_sound;
    try eapply slang.concat_le_le;
    try apply bag.Lang_sound;
    try apply bag.Safe;
    try apply Safe;
    try apply bag.Wf;
    try apply Wf;
    auto.
Qed.

(* Lang completeness *)
Lemma Lang_complete P :
  slang.le (Raw P) (petri.Lang (constr P)).
Proof.
  induction P; simpl;
    try apply eps.Lang_complete;
    try apply bag.Lang_complete;
    try eapply slang.le_trans;
    try eapply slang.concat_le_le;
    try apply seq.Lang_complete;
    try eapply comm.Lang_complete;
    try eapply slang.concur_le_le;
    try apply par.Lang_complete;
    try eapply slang.union_le_le;
    try apply sel.Lang_complete;
    try apply bag.Lang_complete;
    auto;
    try apply comm.Wf;
    try apply rep.Wf;
    try apply seq.Wf;
    try apply bag.Wf;
    try apply Wf;
    auto;
    try eapply slang.le_trans;
    try eapply slang.concat_le_le;
    try apply seq.Lang_complete;
    try eapply slang.union_le_le;
    try apply sel.Lang_complete;
    try eapply slang.star_le;
    try apply rep.Lang_complete;
    try apply bag.Lang_complete;
    try apply bag.Wf;
    try apply Wf;
    auto;
    try eapply slang.le_trans;
    try eapply slang.concat_le_le;
    try apply seq.Lang_complete;
    try apply bag.Lang_complete;
    try apply bag.Wf;
    try apply Wf;
    auto.
Qed.

(* ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~ *)

(* ef correctness *)
Lemma EfcFin P : petri.EfcFin (constr P) (ef P).
Proof.
  induction P; simpl;
    try apply eps.EfcFin;
    try apply par.EfcFin;
    try apply sel.EfcFin;
    try apply sel.EfcFin;
    try apply seq.EfcFin;
    try apply rep.EfcFin;
    try apply seq.EfcFin;
    try apply comm.EfcFin;
    try apply bag.EfcFin;
    try apply comm.Safe;
    try apply sel.Safe;
    try apply rep.Safe;
    try apply seq.Safe;
    try apply bag.Safe;
    try apply Safe;
    try apply comm.Wf;
    try (symmetry; apply init_constr);
    try apply sel.Wf;
    try apply rep.Wf;
    try apply seq.Wf;
    try apply bag.Wf;
    try apply Wf;
    auto.
Qed.

(* nb correctness *)
Lemma NthBag P : petri.NthBag (constr P) (nb P).
Proof.
  induction P; simpl;
    try apply eps.NthBag;
    try apply par.NthBag;
    try apply sel.NthBag;
    try apply sel.NthBag;
    try apply seq.NthBag;
    try apply rep.NthBag;
    try apply seq.NthBag;
    try apply comm.NthBag;
    try apply bag.NthBag;
    try apply comm.EfcFin;
    try apply rep.EfcFin;
    try apply seq.EfcFin;
    try apply bag.EfcFin;
    try apply EfcFin;
    try apply comm.Safe;
    try apply sel.Safe;
    try apply rep.Safe;
    try apply seq.Safe;
    try apply bag.Safe;
    try apply Safe;
    try apply comm.Wf;
    try apply sel.Wf;
    try apply rep.Wf;
    try apply seq.Wf;
    try apply bag.Wf;
    try apply Wf;
    try (symmetry; apply init_constr);
    auto.
Qed.

End prgm.

End MConstr.

(* (c) 2020 Brittany Ro Nkounkou *)
