(* Brittany Nkounkou *)
(* August 2020 *)
(* A FIFO Buffer *)

Require Export Obsrvr.

Set Implicit Arguments.

Module BfrEnv <: Environment.

Module val <: DecidableType.

(* value *)
Definition t : Type :=
  nat.

(* decidable value equality*)
Definition dec : eq_dec t :=
  PeanoNat.Nat.eq_dec.

End val.

(* ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~ *)

(* data variable *)
Inductive dvar_t : Type :=
| x | y | z.

Module dvar <: DecidableType.

Definition t : Type := dvar_t.

(* decidable data variable equality *)
Definition dec : eq_dec t.
Proof.
  unfold eq_dec, t. decide equality.
Defined.

End dvar.

(* ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~ *)

(* channel *)
Inductive chan_t : Type :=
| A | B | C.

Module chan <: DecidableType.

Definition t : Type := chan_t.

(* decidable channel equality *)
Definition dec : eq_dec t.
Proof.
  unfold eq_dec, t. decide equality.
Defined.

End chan.

(* ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~ *)

(* expression *)
Inductive expr_t : Type :=
| val : val.t -> expr_t
| dvar : dvar.t -> expr_t.

Module expr <: DecidableType.

Definition t : Type := expr_t.

(* decidable expression equality *)
Definition dec : eq_dec t.
Proof.
  unfold eq_dec, t. decide equality; try apply val.dec; try apply dvar.dec.
Defined.

End expr.

(* ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~ *)

(* data variable inclusion in expression *)
Inductive expr_In' : dvar.t -> expr.t -> Prop :=
| expr_In_dvar v : expr_In' v (dvar v).
Definition expr_In : dvar.t -> expr.t -> Prop := expr_In'.

(* expression evaluation *)
Fixpoint expr_eval (f : dvar.t -> option val.t) (e : expr.t) : option val.t :=
  match e with
  | val k => Some k
  | dvar v => f v
  end.

(* expression evaluation equality *)
Lemma expr_eval_eq f f' e :
  (forall x, expr_In x e -> f x = f' x) -> expr_eval f e = expr_eval f' e.
Proof.
  intro.
  induction e; auto; simpl; try repeat rewrite H; auto; try constructor; auto.
Qed.

End BfrEnv.

(* ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~ *)

Module Import M := MObsrvr BfrEnv.

(* PRINT ASSUMPTIONS OF FACT event.update_update *)
Print Assumptions event.update_update.

(* PRINT ASSUMPTIONS OF FACT bag.update_Permutation *)
Print Assumptions bag.update_Permutation.

(* PRINT ASSUMPTIONS OF THEOREM sim_sound *)
Print Assumptions sim_sound.

(* PRINT ASSUMPTIONS OF THEOREM sim_complete *)
Print Assumptions sim_complete.

(* expression notations *)
Notation "$ k" := (val k) (at level 0, format "$ k").
Notation "& v" := (dvar v) (at level 0, format "& v").

(* FIFO buffer with three pieces of data passed through *)
Definition BFR : prgm.t :=
  'A!($1); 'A!($2); 'A!($3)
  || 'A?(x); 'B!(&x); 'A?(x); 'B!(&x); 'A?(x); 'B!(&x)
  || 'B?(y); 'C!(&y); 'B?(y); 'C!(&y); 'B?(y); 'C!(&y)
  || 'C?(z); 'C?(z); 'C?(z).

(* PRINT RESULT OF (observe 100 (sim BFR (const 0))) *)
Compute (observe 100 (sim BFR (const 0))).

(* PRINT RESULT OF (observe 100 (sim BFR (const 100))) *)
Compute (observe 100 (sim BFR (const 100))).

(* FIFO buffer with one piece of data passed through *)
Definition BFR2 : prgm.t :=
  'A!($0)
  || 'A?(x); 'B!(&x)
  || 'B?(y); 'C!(&y)
  || 'C?(z).

(* PRINT RESULT OF (observe 100 (sim BFR2 (const 0))) *)
Compute (observe 100 (sim BFR2 (const 0))).

(* PRINT RESULT OF (observe 100 (sim BFR2 (const 100))) *)
Compute (observe 100 (sim BFR2 (const 100))).

(* (c) 2020 Brittany Ro Nkounkou *)
