/*
 * rcwGenerator.cpp
 *
 * Kiyan Ahmadizadeh
 * Cornell University 2010
 */

#include "rcwGenerator.h"
#include <fstream>
#include <climits>
#include <cmath>
#include <iostream>
using namespace std;

rcwGenerator::rcwGenerator(char* gisfname, char* costsfname, int maxSuit) {
	this->numParcel = 0;
	this->numTerr = 0;
	this->maxSuitability = maxSuit;
	double tempX, tempY;
	int tempSuit, tempParc;
	char cma;
	// First open the GIS file.
	ifstream gisData (gisfname, ifstream::in);
	ifstream costRangeData (costsfname, ifstream::in);
	while (gisData.good()) {
		gisData >> tempX >> cma >> tempY >> cma >> tempSuit >> cma >> tempParc;
		this->xCoords.push_back(tempX);
		this->yCoords.push_back(tempY);
		this->baseSuitabilities.push_back(tempSuit);
		this->suitabilities.push_back(tempSuit);
		this->terrToParcel.push_back(tempParc);
		// Add parcel indexes previously unseen.
		if(!(this->parcels.count(tempParc))) {
			this->parcels.insert(tempParc);
			this->parcelIndex[tempParc] = this->numParcel;
			this->numParcel++;
		}
		this->numTerr++;
	}
	gisData.close();

	// Initialize initial territories.
	for (int i=0; i < this->numTerr; i++) {
		this->initialTerritories.push_back(false);
	}
	// Read in the cost ranges.
	int cmin, cmax;
	for (int i=0; i < this->maxSuitability; i++) {
		costRangeData >> cmin >> cma >> cmax;
		this->costMins.push_back(cmin);
		this->costMaxes.push_back(cmax);
	}
}

void rcwGenerator::generateInstance(long seed, int numInitials) {
	srand(seed);
	// Perturb suitabilities by -1, 0, or +1
	for (int i=0; i < this->numTerr; i++) {
		if (this->baseSuitabilities[i] == this->maxSuitability)
			this->suitabilities[i] = this->baseSuitabilities[i] -
				(rand() % 2);
		else if (this->baseSuitabilities[i] == 0 )
			this->suitabilities[i] = 0;
		else
			this->suitabilities[i] = this->baseSuitabilities[i] +
				((rand() % 3) - 1);
	}
	// Record suitability data for parcels;
	vector<parcel_record*> sortedParcels;
	for (int i=0; i < this->numParcel; i++) {
		sortedParcels.push_back((new parcel_record));
		sortedParcels[i]->parcel_index = i;
		sortedParcels[i]->suit_sum = 0;
		sortedParcels[i]->terr_count = 0;
	}
	for (int i=0; i < this->numTerr; i++) {
		sortedParcels[this->parcelIndex[this->terrToParcel[i]]]->terr_count++;
		sortedParcels[this->parcelIndex[this->terrToParcel[i]]]->suit_sum += this->suitabilities[i];
	}
	vector<double> avgParcelSuits;
	for (int i=0; i < this->numParcel; i++) {
		avgParcelSuits.push_back(round(sortedParcels[i]->suit_sum / sortedParcels[i]->terr_count));
	}
	// Use the suitability data to assign a cost to each parcel.
	this->parcelCosts.clear();
	double tempCost;
	for (int i=0; i < this->numParcel; i++) {
		tempCost = (rand() % (this->costMaxes[(int)avgParcelSuits[i]] -
				this->costMins[(int)avgParcelSuits[i]])) +
				this->costMins[(int)avgParcelSuits[i]];
		parcelCosts.push_back(tempCost);
	}
	// Sort the parcels by average suitability.
	sort(sortedParcels.begin(), sortedParcels.end(), pr_lessthan);
	set<int> initialTerrCandidates;
	double initWinMaxX = 0;
	double initWinMaxY = 0;
	double initWinMinX = INT_MAX;
	double initWinMinY = INT_MAX;
	int bestParcel = sortedParcels[sortedParcels.size()-1]->parcel_index;
	// Add territories from best parcel as candidates for initial terr.
	for (int i=0; i < this->numTerr; i++) {
		if (this->parcelIndex[this->terrToParcel[i]] == bestParcel) {
			this->initialTerritories[i] = false;
			initialTerrCandidates.insert(i);
			if (this->xCoords[i] > initWinMaxX)
				initWinMaxX = xCoords[i];
			if (this->xCoords[i] < initWinMinX)
				initWinMinX = xCoords[i];
			if (this->yCoords[i] > initWinMaxY)
				initWinMaxY = yCoords[i];
			if (this->yCoords[i] < initWinMinY)
				initWinMinY = yCoords[i];
		}
	}
	// Increase the search window until there are enough territories
	// to randomly choose the initials from.
	while (initialTerrCandidates.size() < numInitials) {
		initWinMaxX += SEARCH_RADIUS_INCR;
		initWinMinX -= SEARCH_RADIUS_INCR;
		initWinMaxY += SEARCH_RADIUS_INCR;
		initWinMinY += SEARCH_RADIUS_INCR;
		for (int i=0; i < this->numTerr; i++) {
			if (this->xCoords[i] >= initWinMinX &&
					this->xCoords[i] <= initWinMaxX &&
					this->yCoords[i] >= initWinMinY &&
					this->yCoords[i] <= initWinMaxY)
				initialTerrCandidates.insert(i);
		}
	}
	// Now randomly choose some initial candidates to be the
	// actual initial territories.
	vector<int> initCands;
	for (set<int>::iterator itr = initialTerrCandidates.begin();
			itr != initialTerrCandidates.end(); itr++) {
		initCands.push_back(*itr);
	}
	int choice;
	set<int> choices;
	for (int numChosen = 0; numChosen < numInitials; numChosen++) {
		while(choices.count(choice = rand() % initCands.size()));
		choices.insert(choice);
		this->initialTerritories[initCands[choice]] = true;
		this->parcelCosts[this->parcelIndex[this->terrToParcel[initCands[choice]]]] = 0;
	}
}

void rcwGenerator::printCurrentMap() {
	// Print number of territories and parcels.
	cout << this->numTerr << endl;
	cout << this->numParcel << endl;

	// Now output the cost of each parcel on one csv line.
	for (int i=0; i < this->numParcel; i++) {
		cout << this->parcelCosts[i];
		if (i != this->numParcel-1)
			cout << ",";
	}
	cout << endl;

	// For each territory, print
	// xcoord, ycoord, suitability, parcel, initial_flag
	for (int i=0; i < this->numTerr; i++) {
		cout << this->xCoords[i] << ",";
		cout << this->yCoords[i] << ",";
		cout << this->suitabilities[i] << ",";
		cout << this->parcelIndex[this->terrToParcel[i]] << ",";
		if (this->initialTerritories[i])
			cout << "1" << endl;
		else
			cout << "0" << endl;
	}
}

rcwGenerator::~rcwGenerator() {
	// TODO Auto-generated destructor stub
}
