#! /usr/bin/env python

###############################################################################
#   File: SSHRunner.py
#
#   This file is part of prunner
#
#  This program is free software: you can redistribute it and/or modify
#  it under the terms of the GNU General Public License as published by
#  the Free Software Foundation, either version 3 of the License, or
#  (at your option) any later version.
#
#  This program is distributed in the hope that it will be useful,
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#  GNU General Public License for more details.
#
#  You should have received a copy of the GNU General Public License
#  along with this program.  If not, see <http://www.gnu.org/licenses/>.
################################################################################

import os, sys
import paramiko

class SSHRunner:
    """ Run remote command through SSH connection """
    def __init__(self, hostname, username, password=None, keyfile=None, 
                 port=22, timeout=None, log=None):
        """ log: log file to record the output from stderr """
        #self.hostname = hostname
        #self.username = username
        #self.port     = port
        #self.timeout  = timeout
        self._sshclient = paramiko.SSHClient()
        self._sshclient.set_missing_host_key_policy(paramiko.AutoAddPolicy())
        self._log = log
        if keyfile:
            rsaKey = paramiko.RSAKey.from_private_key_file(
                    os.path.expanduser(os.path.normpath(keyfile)),
                    password = password)
            self._sshclient.connect(hostname, port=port, username=username, 
                    pkey=rsaKey, timeout=timeout)
        else:
            self._sshclient.connect(hostname, port=port, username=username,
                    password=password, timeout=timeout)

    def __del__(self):
        self._sshclient.close()

    def close(self):
        self._sshclient.close()

    def run_command(self, cmd):
        try:
            stdin, stdout, stderr = self._sshclient.exec_command(cmd)
            txt = stdout.readline()
            while txt:
                sys.stdout.write(txt)
                txt = stdout.readline()
            # record output of stderr
            #if self._log:
            errcnt = 0
            txt = stderr.readline()
            while txt:
                if self._log:
                    if errcnt == 0: self._log.newCommand(cmd)
                    self._log.errorMsg(txt)
                errcnt += 1
                txt = stderr.readline()
            return errcnt == 0
        except SSHException, err:
            if self._log:
                self._log.newCommand(cmd)
                self._log.errorMsg('Fail to run. %s' % err.__str__())
            return False

