###############################################################################
#   File: Clusters.py
#
#   This file is part of prunner
#
#  This program is free software: you can redistribute it and/or modify
#  it under the terms of the GNU General Public License as published by
#  the Free Software Foundation, either version 3 of the License, or
#  (at your option) any later version.
#
#  This program is distributed in the hope that it will be useful,
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#  GNU General Public License for more details.
#
#  You should have received a copy of the GNU General Public License
#  along with this program.  If not, see <http://www.gnu.org/licenses/>.
################################################################################
import getpass
import tempfile
import ConfigParser
from Cluster import *

#
# All the clusters
#
class Clusters(object):
    def __init__(self, configfile, jobs):
        """ All the working clusters

        clusterfile: the text file to store clusters. The format of this file is
                     hostname <num_threads>

        """
        config = ConfigParser.SafeConfigParser()
        config.read(configfile)

        self._username = config.get('ssh', 'username')
        if config.has_option('ssh', 'keyfile'):
            keyfile = config.get('ssh', 'keyfile')
        else:
            keyfile = None

        if config.has_option('ssh', 'port'):
            port = int(config.get('ssh', 'port'))
        else:
            port = 22

        if config.has_option('ssh', 'timeout'):
            timeout = int(config.get('ssh', 'timeout'))
        else:
            timeout = None
        self._password = getpass.getpass('SSH password: ');
        logdir   = tempfile.mkdtemp(prefix='prunner-tmp')

        polltime = int(config.get('clusters', 'queue_poll_timeout'))
        clusterfile = config.get('clusters', 'clusterfile')

        self._clusters = []
        self._clusterMap = {}       # map from the name of a cluster to the cluster instance
        fd = open(clusterfile, 'r')
        txt = fd.readline()
        while txt:
            txt = txt.strip()
            if txt and txt[0] != '#':
                # check file format
                toks = txt.split()
                if len(toks) != 2:
                    raise Exception, "Incorrect cluster file format."
                cls = Cluster(jobs, polltime, toks[0].lower(), 
                              int(toks[1]), logdir, self._username, self._password, 
                              keyfile, port, timeout)
                if cls.clustername in self._clusterMap:
                    raise Exception, "Cluster is repeated in the flie %s." % clusterfile
                self._clusters.append(cls)
                self._clusterMap[cls.clustername] = cls
            txt = fd.readline();
        fd.close()

    @property
    def auth(self):
        return { self._username : self._password }

    @property
    def clusters(self):
        return self._clusters

    # ------ enable/disable a cluster thread -------
    def activateClusterThread(self, cname):
        toks = cname.split('::')
        if len(toks) == 2:
            self._clusterMap[toks[0]].activateThread(int(toks[1]))

    def deactivateClusterThread(self, cname):
        toks = cname.split('::')
        if len(toks) == 2:
            self._clusterMap[toks[0]].deactivateThread(int(toks[1]))

    def getCluster(self, name):
        return self._clusterMap[name.lower()]
