###############################################################################
#   File: ClusterThread.py
#
#   This file is part of prunner
#
#  This program is free software: you can redistribute it and/or modify
#  it under the terms of the GNU General Public License as published by
#  the Free Software Foundation, either version 3 of the License, or
#  (at your option) any later version.
#
#  This program is distributed in the hope that it will be useful,
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#  GNU General Public License for more details.
#
#  You should have received a copy of the GNU General Public License
#  along with this program.  If not, see <http://www.gnu.org/licenses/>.
################################################################################
import threading
from SSHRunner import *
import time
import random

class ClusterThread(threading.Thread):

    def __init__(self, cluster, id, jobs, pollTimeout, 
                 hostname, username, password=None, 
                 keyfile=None, port=22, timeout=None, log=None,
                 **kwds):
        threading.Thread.__init__(self, **kwds)
        self.setDaemon(True)

        self._cluster = cluster
        self._id = id            # id: (name, id)
        self._jobs = jobs
        self._pollTimeout = pollTimeout
        self._finishedJobs = 0
        self._errcnt = 0
        self._log = log
        self._sshrunner = SSHRunner(hostname, username, password, keyfile,
                                    port, timeout, log)
        self._terminated = threading.Event()
        self._runEvt = threading.Event()    # control running or pause the current thread
        self._runEvt.set()

        print "Run thread %s-%d" % id
        self.start()

    def currentErrorMsg(self):
        cmd = None
        errmsg = ''
        for txt in self._log.readLog():
            if txt.startswith('$$$'):
                if cmd:
                    yield (cmd, errmsg)
                cmd = txt[3:]
                errmsg = ''
            else:
                errmsg += txt
        if cmd: yield (cmd, errmsg)

    @property
    def id(self):
        return self._id

    def hasError(self):
        return self._errcnt > 0

    @property
    def finishedJobs(self):
        return self._finishedJobs

    def disable(self):
        self._runEvt.clear()

    def enable(self):
        self._runEvt.set()

    def isPaused(self):
        return not self._runEvt.isSet()

    def run(self):
        """ Repeatedly process the job queue until it is told to be exit """
        time.sleep(random.randint(0,10))
        while not self._terminated.isSet():
            self._runEvt.wait()     # wait here if the thread is paused
            cmd = self._jobs.nextCommand(self._pollTimeout)     # get next command from queue
            if not cmd: continue

            self._cluster.cmdStarted(self._id, cmd)
            # run command
            if not self._sshrunner.run_command(cmd): 
                self._errcnt += 1
            # callback after command is finished
            self._cluster.cmdFinished(self._id, cmd)
            self._finishedJobs += 1

