###############################################################################
#   File: Cluster.py
#
#   This file is part of prunner
#
#  This program is free software: you can redistribute it and/or modify
#  it under the terms of the GNU General Public License as published by
#  the Free Software Foundation, either version 3 of the License, or
#  (at your option) any later version.
#
#  This program is distributed in the hope that it will be useful,
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#  GNU General Public License for more details.
#
#  You should have received a copy of the GNU General Public License
#  along with this program.  If not, see <http://www.gnu.org/licenses/>.
################################################################################
import threading
from FileLog import *
from ClusterThread import *

class Cluster(object):

    def __init__(self, jobs, polltimeout, hostname, nthread, logdir,
                 username, password=None, keyfile=None, port=22, timeout=None):
        self._hostname = hostname
        self._name     = hostname.split('.')[0]
        self._nthread  = nthread
        self._threads  = []         # multiple threads running on this cluster
        self._finishedJobs = 0
        self._runningCmd = []       # running command on each thread
        self._lock = threading.Lock()
        # create cluster thread
        for id in xrange(nthread):
            if logdir:
                log = FileLog('%s/%s.%d.log' % (logdir, self._name, id))
            else:
                log = None
            self._runningCmd.append(None)
            self._threads.append(
                        ClusterThread(self, (self._name, id), jobs, polltimeout,
                        hostname, username, password, keyfile, port, timeout,
                        log))

    def getClusterThread(self, id):
        return self._threads[id]

    def activateThread(self, id):
        """ - if the thread is not paused, ignore the calling 
            - if it is paused but still running, simply set the runEvt event
            - otherwise, restart the thread
        """
        if not self._threads[id].isPaused(): return
        self._threads[id].enable()

    def deactivateThread(self, id):
        """ - if the thread is paused, ignore the calling
            - otherwise, set it paused
        """
        if self._threads[id].isPaused(): return
        self._threads[id].disable()

    @property
    def clustername(self):
        """ Get the cluster's representative name """
        return self._name

    @property
    def hostname(self):
        return self._hostname

    @property
    def numFinishedJobs(self):
        return self._finishedJobs

    def cmdFinished(self, id, cmd):
        self._runningCmd[id[1]] = None
        self._lock.acquire()
        self._finishedJobs += 1
        self._lock.release()

    def cmdStarted(self, id, cmd):
        print "LOG: %s-%d: run command %s" % (id[0], id[1], cmd)
        self._runningCmd[id[1]] = cmd

    def threadStatus(self):
        for td in self._threads:
            tid  = td.id[1]
            rcmd = self._runningCmd[td.id[1]]
            if td.isPaused():
                sts = 'S'
                if not rcmd: rcmd = ''
            else:
                if rcmd:
                    sts = 'R'
                else:
                    sts = 'I'
                    rcmd = ''
            if td.hasError():
                sts += 'E'
            yield (td.id[0], tid, sts, td.finishedJobs, rcmd)

