###############################################################################
#   File: __init__.py
#
#   This file is part of prunner
#
#  This program is free software: you can redistribute it and/or modify
#  it under the terms of the GNU General Public License as published by
#  the Free Software Foundation, either version 3 of the License, or
#  (at your option) any later version.
#
#  This program is distributed in the hope that it will be useful,
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#  GNU General Public License for more details.
#
#  You should have received a copy of the GNU General Public License
#  along with this program.  If not, see <http://www.gnu.org/licenses/>.
################################################################################
import sys, os
from MonitorPage import *
from ManagePage import *
from ErrorPage import *
from PlotGen import *
import cStringIO

if sys.version_info < (2, 4):
    raise RuntimeError('You need python 2.3 for this module.')

__author__ = "Changxi Zheng<cxzheng@cs.cornell.edu>"
__date__ = "Jan 2010"

__all__ = ['MonitorPage', 'ManagePage', 'ErrorPage', 'PlotGen']

CURRENT_DIR = os.path.dirname(os.path.abspath(__file__))

import cherrypy

# --------------- Error Page --------------
class Errors:
    """ page to display error messages """
    def __init__(self, prunner):
        self._vars = { 'cname': 'cluster', 'cid': 0, 'clusters': prunner }
        self._errorpage = ErrorPage(searchList = [ self._vars ])

    @cherrypy.expose
    def default(self, cname, cid):
        self._vars['cname'] = cname
        self._vars['cid']   = int(cid)
        return self._errorpage.__str__()

# --------------- Management Page --------------
class Manage:
    """ page for enabling/disabling cluster, and submitting jobs """
    def __init__(self, prunner):
        self._prunner = prunner
        self._vars = { 'message' : None, 'clusters': prunner }
        self._managepage = ManagePage(searchList = [self._vars])

    @cherrypy.expose
    def index(self):
        return self._managepage.__str__()

    @cherrypy.expose
    def enable_clusters(self, selection=None):
        if selection is None: return self._managepage.__str__()
        if isinstance(selection, str):
            self._prunner.clusters.activateClusterThread(selection)
        else:
            for cname in selection:
                self._prunner.clusters.activateClusterThread(cname)
        return self._managepage.__str__()

    @cherrypy.expose
    def disable_clusters(self, selection=None):
        if selection is None: return self._managepage.__str__()
        if isinstance(selection, str):
            self._prunner.clusters.deactivateClusterThread(selection)
        else:
            for cname in selection:
                self._prunner.clusters.deactivateClusterThread(cname)
        return self._managepage.__str__()

    @cherrypy.expose
    def submit_cmd_txt(self, commands=None):
        if commands is None: return;

        lno = 1
        errmsg = None
        instr = cStringIO.StringIO(commands)
        txt = instr.readline()
        while txt:
            txt = txt.strip()
            if len(txt) > 0 and txt[0] != '#' and not self._prunner.jobs.addJobsFromCmd(txt):
                errmsg = "Warning: Incorrect command syntax at line: %d" % lno
                break
            lno += 1
            txt = instr.readline()
        instr.close()
        # return the current page 
        if errmsg:
            self._vars['message'] = errmsg
            ret = self._managepage.__str__()
            self._vars['message'] = None
            return ret
        else:
            return self._managepage.__str__()

    @cherrypy.expose
    def submit_cmd_file(self, cmdFile=None):
        """ submit additional jobs from uploaded file """
        # make sure the input argument is not None
        if cmdFile is None:
            self._vars['message'] = 'Empty command file is provided'
            ret = self._managepage.__str__()
            self._vars['message'] = None
            return ret
        # read the file and add jobs
        lno = 1
        errmsg = None
        txt = cmdFile.file.readline()
        while txt:
            txt = txt.strip()
            if len(txt) > 0 and txt[0] != '#' and not self._prunner.jobs.addJobsFromCmd(txt):
                errmsg = "Warning: Incorrect command syntax at line: %d" % lno
                break
            txt = cmdFile.file.readline()
        # return the current page 
        if errmsg:
            self._vars['message'] = errmsg
            ret = self._managepage.__str__()
            self._vars['message'] = None
            return ret
        else:
            return self._managepage.__str__()

# --------------- Monitor Page --------------
class Root:
    """ the default page is to show the cluster running information """

    # page to the error information
    errors = None
    # page to manage the clusters and submit jobs
    manage = None

    def __init__(self, prunner):
        self.errors = Errors(prunner)
        self.manage = Manage(prunner)
        self._prunner = prunner
        self._monitorpage = MonitorPage(searchList = [{'clusters': prunner}])

    @cherrypy.expose
    def index(self):
        return self.monitor()

    @cherrypy.expose
    def monitor(self):
        plot_cluster_finished_jobs(
                self._prunner.clusterNames(),
                self._prunner.clusterFinishedJobs(),
                CURRENT_DIR+'/data/img/finished_jobs.png')
        return self._monitorpage.__str__()
