% This experimental code is published in connection with a scientific publication, Silent error detection in numerical time-stepping schemes,
% by Austin R. Benson, Sven Schmit, and Robert Schreiber, to appear in International Journal of High Performance Computing Applications in 2014.
% This code is made available solely to allow the readers of that publication to verify and reproduce the results described. 
% This experimental code is published "as is", with no representation, warranty, indemnification of any kind. 
% Hewlett-Packard excludes all liability that may result from the use of this experimental code in any form.

function [ U, T, n_diff, err, LTE ] = ...
        ode_ab45_FT(u0, t0, tmax, f, h, ind, t_fault, fault_amt, x_fault, f_type)

order = size(u0,2); 
T = t0:h:tmax;
NT = length(T);

U = zeros(NT, order);
U(1,:) = u0;

n_diff = zeros(NT, 1);

err = -1;
LTE = -1;

% Runge-Kutta for the first 2 steps
for j = 1:3
    % Calculate slopes
    k1 = f(T(j), U(j,:));
    k2 = f(T(j)+h/2, U(j,:)+h*k1/2);
    k3 = f(T(j)+h/2, U(j,:)+h*k2/2);
    k4 = f(T(j)+h, U(j,:)+h*k3);
    % Calculate update
    rk4_update = h/6*(k1 + 2*k2 + 2*k3 + k4);
    
    U(j+1,:) = U(j,:) + rk4_update;
end

% initialize f's
fj1 = f(T(4), U(4,:));
fj2 = f(T(3), U(3,:));
fj3 = f(T(2), U(2,:));
fj4 = f(T(1), U(1,:));

% Adams-Bashforth updates
for j = 5:length(T)
    fj = f(T(j), U(j, :));    
    if j == t_fault && strcmp(f_type, 'func_eval');
        fj(x_fault) = fj(x_fault) * fault_amt;
    end
    
    if j == t_fault && strcmp(f_type, 'prev_soln')
        fj1_save = fj1;
        fj2_save = fj2;
        fj3_save = fj3;
        fj4_save = fj4;        
        switch(ind)
            case 1
                fj1 = fj1(x_fault) * fault_amt;
            case 2
                fj2 = fj2(x_fault) * fault_amt;
            case 3
                fj3 = fj3(x_fault) * fault_amt;
            case 4
                fj4 = fj4(x_fault) * fault_amt;                
        end
    end
    
    % calculate updates
    %ab1_update = h* fj;
    %ab2_update = h/2*  (3* fj - fj1);
    %ab3_update = h/12* (23* fj - 16* fj1 + 5 * fj2);
    ab4_update = h/24* (55* fj - 59* fj1 +37 * fj2 - 9 * fj3 );
    ab5_update = h/720*(1901* fj - 2774* fj1 + 2616* fj2 - 1274* fj3 + 251* fj4);
    
    % check difference
    n_diff(j) = norm(ab5_update - ab4_update, 'inf');
    
    if j == t_fault
        fj_good = f(T(j), U(j,:));
        if strcmp(f_type, 'prev_soln')
            ab5_update_good = h/720*(1901* fj_good - 2774* fj1_save + 2616* fj2_save - 1274* fj3_save + 251* fj4_save);
            ab4_update_good = h/24*(55* fj_good - 59* fj1_save +37 * fj2_save - 9 * fj3_save );
        else
            ab5_update_good = h/720*(1901* fj_good - 2774* fj1 + 2616* fj2 - 1274* fj3 + 251* fj4);
            ab4_update_good = h/24*(55* fj_good - 59* fj1 +37 * fj2 - 9 * fj3 );
        end
        err = norm(ab5_update - ab5_update_good, 'inf');
        LTE = norm(ab5_update_good - ab4_update_good, 'inf');        
    end    
    
    % add update
    U(j+1,:) = U(j,:) + ab5_update;
    
    % Update f's
    fj4 = fj3;
    fj3 = fj2;    
    fj2 = fj1;
    fj1 = fj;
end

end
