% This experimental code is published in connection with a scientific publication, Silent error detection in numerical time-stepping schemes,
% by Austin R. Benson, Sven Schmit, and Robert Schreiber, to appear in International Journal of High Performance Computing Applications in 2014.
% This code is made available solely to allow the readers of that publication to verify and reproduce the results described. 
% This experimental code is published "as is", with no representation, warranty, indemnification of any kind. 
% Hewlett-Packard excludes all liability that may result from the use of this experimental code in any form.

function [U_all, V_all, u_diff, v_diff, u_expl_diff, v_expl_diff] = ...
    navierstokes(Re, dt, tf, lx, ly, nx, ny, f, fault_amt, t_fault, x_fault)
%MIT18086_NAVIERSTOKES
%    Solves the incompressible Navier-Stokes equations in a
%    rectangular domain with prescribed velocities along the
%    boundary. The solution method is finite differencing on
%    a staggered grid with implicit diffusion and a Chorin
%    projection method for the pressure.
%    Visualization is done by a colormap-isoline plot for
%    pressure and normalized quiver and streamline plot for
%    the velocity field.
%    The standard setup solves a lid driven cavity problem.

% 07/2007 by Benjamin Seibold
%            http://www-math.mit.edu/~seibold/
% Feel free to modify for teaching and learning.
%-----------------------------------------------------------------------
% Re      % Reynolds number
% dt      % time step
% tf      % final time
% lx      % width of box
% ly      % height of box
% nx      % number of x-gridpoints
% ny      % number of y-gridpoints
% f       % external forces
%-----------------------------------------------------------------------
nt = ceil(tf/dt);
dt = tf/nt;
x = linspace(0,lx,nx+1);
hx = lx/nx;
y = linspace(0,ly,ny+1);
hy = ly/ny;
%-----------------------------------------------------------------------
% initial conditions
U = zeros(nx-1,ny)+1; V = zeros(nx,ny-1)+1;

% boundary conditions
uN = x*0+1;
vN = avg(x)*0;

%uS = sin(2 * pi * x); 
uS = x*0;
vS = avg(x)*0;

uW = avg(y)*0;
vW = y*0;

uE = avg(y)*0;
vE = y*0;

% data for post-processing
U_all = zeros(nt, nx-1, ny);
V_all = zeros(nt, nx, ny-1);
u_diff = zeros(nt, 1);
v_diff = u_diff;
u_expl_diff = u_diff;
v_expl_diff = u_diff;

%-----------------------------------------------------------------------
Ubc = dt/Re*([2*uS(2:end-1)' zeros(nx-1,ny-2) 2*uN(2:end-1)']/hx^2+...
      [uW;zeros(nx-3,ny);uE]/hy^2);
Vbc = dt/Re*([vS' zeros(nx,ny-3) vN']/hx^2+...
      [2*vW(2:end-1);zeros(nx-2,ny-1);2*vE(2:end-1)]/hy^2);

Lp = kron(speye(ny),K1(nx,hx,1))+kron(K1(ny,hy,1),speye(nx));
Lp(1,1) = 3/2*Lp(1,1);
perp = symamd(Lp); Rp = chol(Lp(perp,perp)); Rpt = Rp';
Lu = speye((nx-1)*ny)+dt/Re*(kron(speye(ny),K1(nx-1,hx,2))+...
     kron(K1(ny,hy,3),speye(nx-1)));
peru = symamd(Lu); Ru = chol(Lu(peru,peru)); Rut = Ru';
Lv = speye(nx*(ny-1))+dt/Re*(kron(speye(ny-1),K1(nx,hx,3))+...
     kron(K1(ny-1,hy,2),speye(nx)));
perv = symamd(Lv); Rv = chol(Lv(perv,perv)); Rvt = Rv';
Lq = kron(speye(ny-1),K1(nx-1,hx,2))+kron(K1(ny-1,hy,2),speye(nx-1));
perq = symamd(Lq); Rq = chol(Lq(perq,perq));

for k = 1:nt

   % treat nonlinear terms
   gamma = min(1.2*dt*max(max(max(abs(U)))/hx,max(max(abs(V)))/hy),1);
   
   Ue = [uW;U;uE];
   Ue = [2*uS'-Ue(:,1) Ue 2*uN'-Ue(:,end)];
   

   Ve = [vS' V vN'];
   Ve = [2*vW-Ve(1,:);Ve;2*vE-Ve(end,:)];
   Ua = avg(Ue')';
   Ud = diff(Ue')'/2;
   Va = avg(Ve);
   Vd = diff(Ve)/2;
   UVx = diff(Ua.*Va-gamma*abs(Ua).*Vd)/hx;
   UVy = diff((Ua.*Va-gamma*Ud.*abs(Va))')'/hy;
   
   Ua = avg(Ue(:,2:end-1));
   Ud = diff(Ue(:,2:end-1))/2;
   Va = avg(Ve(2:end-1,:)')';
   Vd = diff(Ve(2:end-1,:)')'/2;


   U2x = diff(Ua.^2-gamma*abs(Ua).*Ud)/hx;
   

   V2y = diff((Va.^2-gamma*abs(Va).*Vd)')'/hy;
   
   FU = f(x(2:end-1), y(2:end), k * dt);
   FV = f(x(2:end), y(2:end-1), k * dt);

   U = U-dt*(UVy(2:end-1,:)+U2x) + dt * FU;
   V = V-dt*(UVx(:,2:end-1)+V2y) + dt * FV;
   
   % implicit viscosity
   rhs = reshape(U+Ubc,[],1);
   
   U_save = U;
   V_save = V;
   
   u(peru) = Ru\(Rut\rhs(peru));
   U = reshape(u,nx-1,ny);
   rhs = reshape(V+Vbc,[],1);
   v(perv) = Rv\(Rvt\rhs(perv));
   V = reshape(v,nx,ny-1);
   
   % explicit solve for check
   c1 = dt / (hx^2) / Re;
   c2 = dt / (hy^2) / Re;
   % second derivative in x direction
   U_expl = c1 * diff([uW; U; uE], 2, 1);
   V_expl = c1 * diff([vW(2:end-1); V; vE(2:end-1)], 2, 1);   
   % second derivative in y direction
   U_expl = U_expl + c2 * diff([uS(2:end-1)', U, uN(2:end-1)'], 2, 2);
   V_expl = V_expl + c2 * diff([vS', V, vN'], 2, 2);   
   % left-hand-side temporal derivative add-on
   U_expl = U_expl + U_save;
   V_expl = V_expl + V_save;
   
   u_expl_diff(k) = max(max(abs(U - U_expl)));
   v_expl_diff(k) = max(max(abs(V - V_expl)));
   
   % pressure correction
   rhs = reshape(diff([uW;U;uE])/hx+diff([vS' V vN']')'/hy,[],1);
   
   p(perp) = -Rp\(Rpt\rhs(perp));
   P = reshape(p,nx,ny);
   U = U-diff(P)/hx;
   V = V-diff(P')'/hy;
   
   % Computations for fault detection
   U_all(k, :, :) = U;
   V_all(k, :, :) = V;
   if (k > 2)
       u_diff(k) = interp_diff(U_all, k);
       v_diff(k) = interp_diff(V_all, k);
   end
   
    % Check
    % (ignore the viscosity term)
    %u_diff(k) = norm(U - U_save, 'inf');
    %v_diff(k) = norm(V - V_save, 'inf');
   
end
end


%=======================================================================
function val = inject_fault(val, x_fault, fault_amt)
    if val(x_fault(1), x_fault(2)) == 0
        fprintf('multiplying zero value!');
    end
    val(x_fault(1), x_fault(2)) = val(x_fault(1), x_fault(2)) * fault_amt;
end

function val =  interp_diff(A, k)
    if 1
    A = abs(A(k, :, :) - 2 * A(k-1, :, :) + A(k-2, :, :));
    val = max(A(:));
    end        
        
    if 0
    A = A(k, :, :) - 2 * A(k-1, :, :) + A(k-2, :, :);
    val = norm(reshape(A, [size(A, 2), size(A, 3)]), 'inf');
    end
    
    if 0
    A = A(k, :, :) - A(k-1, :, :);
    val = norm(reshape(A, [size(A, 2), size(A, 3)]), 'inf');
    end
end

function B = avg(A,k)
if nargin<2, k = 1; end
if size(A,1)==1, A = A'; end
if k<2, B = (A(2:end,:)+A(1:end-1,:))/2; else B = avg(A,k-1); end
if size(A,2)==1, B = B'; end
end

function A = K1(n,h,a11)
% a11: Neumann=1, Dirichlet=2, Dirichlet mid=3;
A = spdiags([-1 a11 0;ones(n-2,1)*[-1 2 -1];0 a11 -1],-1:1,n,n)'/h^2;
end