/*
 * pmread.c -- 
 * 
 * Code for read Portable Pixmap and Portable Graymap image.
 * Large parts are grabbed from pbmplus software, so:
 *
 * Copyright (C) 1988 by Jef Poskanzer.
 *
 * Permission to use, copy, modify, and distribute this software and its
 * documentation for any purpose and without fee is hereby granted, provided
 * that the above copyright notice appear in all copies and that both that
 * copyright notice and this permission notice appear in supporting
 * documentation.  This software is provided "as is" without express or
 * implied warranty.
 * 
 * Copyright (c) 1994 Kongji Huang and Brian C. Smith.
 * Cornell University
 * All rights reserved.
 * 
 * Permission to use, copy, modify, and distribute this software and its
 * documentation for any purpose, without fee, and without written agreement is
 * hereby granted, provided that the above copyright notice and the following
 * two paragraphs appear in all copies of this software.
 * 
 * IN NO EVENT SHALL CORNELL UNIVERSITY BE LIABLE TO ANY PARTY FOR
 * DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT
 * OF THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF CORNELL
 * UNIVERSITY HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 * 
 * CORNELL UNIVERSITY SPECIFICALLY DISCLAIMS ANY WARRANTIES,
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY
 * AND FITNESS FOR A PARTICULAR PURPOSE.  THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND CORNELL UNIVERSITY HAS NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS, OR MODIFICATIONS.
 */

#include <stdio.h>
#include <malloc.h>
#include "jpeg.h"
#include "mcu.h"
#include "proto.h"
#include "pnmtoljpg.h"

#define PM_MAXVAL 65536

/* Magic constants. */
#define PBM_MAGIC1 'P'
#define PBM_MAGIC2 '1'
#define RPBM_MAGIC2 '4'
#define PBM_FORMAT (PBM_MAGIC1 * 256 + PBM_MAGIC2)
#define RPBM_FORMAT (PBM_MAGIC1 * 256 + RPBM_MAGIC2)
#define PBM_TYPE PBM_FORMAT

#define PGM_MAGIC1 'P'
#define PGM_MAGIC2 '2'
#define RPGM_MAGIC2 '5'
#define PGM_FORMAT (PGM_MAGIC1 * 256 + PGM_MAGIC2)
#define RPGM_FORMAT (PGM_MAGIC1 * 256 + RPGM_MAGIC2)
#define PGM_TYPE PGM_FORMAT

#define PPM_MAGIC1 'P'
#define PPM_MAGIC2 '3'
#define RPPM_MAGIC2 '6'
#define PPM_FORMAT (PPM_MAGIC1 * 256 + PPM_MAGIC2)
#define RPPM_FORMAT (PPM_MAGIC1 * 256 + RPPM_MAGIC2)
#define PPM_TYPE PPM_FORMAT

/* Macro for turning a format number into a type number. */
#define PBM_FORMAT_TYPE(f) ((f) == PBM_FORMAT || (f) == RPBM_FORMAT ?	\
        PBM_TYPE : -1)

#define PGM_FORMAT_TYPE(f) ((f) == PGM_FORMAT || (f) == RPGM_FORMAT ?	\
        PGM_TYPE : PBM_FORMAT_TYPE(f))

#define PPM_FORMAT_TYPE(f) ((f) == PPM_FORMAT || (f) == RPPM_FORMAT ?	\
        PPM_TYPE : PGM_FORMAT_TYPE(f))

char
pbm_getc( file )
    FILE* file;
{
    register int ich;
    register char ch;

    ich = getc( file );
    inputFileBytes++;
    if ( ich == EOF )
        fprintf(stderr,"EOF / read error" );
    ch = (char) ich;

    if ( ch == '#' )
        {
        do
            {
            ich = getc( file );
            inputFileBytes++;
            if ( ich == EOF )
                fprintf(stderr,"EOF / read error" );
            ch = (char) ich;
            }
        while ( ch != '\n' && ch != '\r' );
        }

    return ch;
}

unsigned char
pbm_getrawbyte( file )
    FILE* file;
{
    register int iby;

    iby = getc( file );
    if ( iby == EOF )
        fprintf(stderr,"EOF / read error" );
    return (unsigned char) iby;
}

int
pbm_getint( file )
    FILE* file;
{
    register char ch;
    register int i;

    do
        {
        ch = pbm_getc( file );
        }
    while ( ch == ' ' || ch == '\t' || ch == '\n' || ch == '\r' );

    if ( ch < '0' || ch > '9' )
        fprintf(stderr,"junk in file where an integer should be" );

    i = 0;
    do
        {
        i = i * 10 + ch - '0';
        ch = pbm_getc( file );
        }
    while ( ch >= '0' && ch <= '9' );

    return i;
}

int
pbm_readmagicnumber( file )
    FILE* file;
{
    int ich1, ich2;

    ich1 = getc( file );
    if ( ich1 == EOF )
        fprintf(stderr,"EOF / read error reading magic number" );
    ich2 = getc( file );
    if ( ich2 == EOF )
        fprintf(stderr,"EOF / read error reading magic number" );
    return ich1 * 256 + ich2;
}

void
pbm_readpbminitrest( file, colsP, rowsP )
    FILE* file;
    int* colsP;
    int* rowsP;
{
    /* Read size. */
    *colsP = pbm_getint( file );
    *rowsP = pbm_getint( file );
}

void
pgm_readpgminitrest( file, colsP, rowsP, maxvalP )
    FILE* file;
    int* colsP;
    int* rowsP;
    int* maxvalP;
    {
    int maxval;

    /* Read size. */
    *colsP = pbm_getint( file );
    *rowsP = pbm_getint( file );

    /* Read maxval. */
    maxval = pbm_getint( file );
    if ( maxval > PM_MAXVAL )
        fprintf(stderr,"maxval is too large\n" );
    *maxvalP = maxval;
}

void
ppm_readppminitrest( file, colsP, rowsP, maxvalP )
    FILE* file;
    int* colsP;
    int* rowsP;
    int* maxvalP;
    {
    int maxval;

    /* Read size. */
    *colsP = pbm_getint( file );
    *rowsP = pbm_getint( file );

    /* Read maxval. */
    maxval = pbm_getint( file );
    if ( maxval > PM_MAXVAL )
        fprintf(stderr, "maxval is too large\n");
    *maxvalP = maxval;
    }

void
ppm_readppminit( file, colsP, rowsP, maxvalP, formatP )
    FILE* file;
    int* colsP;
    int* rowsP;
    int* formatP;
    int* maxvalP;
{
    /* Check magic number. */
    *formatP = pbm_readmagicnumber( file );
    switch ( PPM_FORMAT_TYPE(*formatP) )
        {
        case PPM_TYPE:
        ppm_readppminitrest( file, colsP, rowsP, maxvalP );
        break;

        case PGM_TYPE:
        pgm_readpgminitrest( file, colsP, rowsP, maxvalP );
        break;

        case PBM_TYPE:
        pbm_readpbminitrest( file, colsP, rowsP );
        *maxvalP = 1;
        break;

        default:
        fprintf(stderr,"bad magic number - not a ppm, pgm, or pbm file\n");
        exit(-1);
    }
}

/*
 *--------------------------------------------------------------
 *
 * PmRead --
 *
 *	Read the source image file pointed by inFile.
 *
 * Results:
 *      Source image parameters such as width, height, sample
 *	precision and number of color components are passed
 *	out by enInfo.
 *
 * Side effects:
 *	Source image is read into mcuTable, input image size
 *	is counted in inputFileBytes.
 *
 *--------------------------------------------------------------
 */
void
PmRead(enInfo,inFile)
    CompressInfo *enInfo;
    FILE *inFile;
{
    int rows,cols,format,maxMcu;
    int maxval,precision;
    long r,g,b;
    MCU mcu;
    JpegComponentInfo *compPtr;
    int i,j,k,ci;

    /*
     * Initialize inputFileBytes to 2 bytes since magic number is 2 bytes.
     * Read magic number, cols, rows, and max color level. Convert the
     * color level to precision bits.
     */
    inputFileBytes=2;
    ppm_readppminit( inFile, &cols, &rows, &maxval, &format );
    if ((maxval<MinPrecisionValue) || (maxval>MaxPrecisionValue)) {
       fprintf(stderr,"Precision not supported.\n");
       exit(-1);
    }

    if (maxval==255) {
       precision=8;
    } else {
       precision=0;
       while ((maxval>>precision++) != 0); 
       precision--;
    }

    /* 
     * Store the image width, height, precision and number of color
     * components in compression data structure pointed by enInfo.
     */
    enInfo->imageWidth=cols;
    enInfo->imageHeight=rows;
    enInfo->dataPrecision=precision;
    switch ( PPM_FORMAT_TYPE(format) )
        {
        case PPM_TYPE:
             if (verbose) {
                fprintf(stderr,"ppm file\n%d %d\n%d\n",cols,rows,maxval);
             }
             enInfo->numComponents=3;
             break;

        case PGM_TYPE:
             if (verbose) {
                fprintf(stderr,"pgm file\n%d %d\n%d\n",cols,rows,maxval);
             }
             enInfo->numComponents=1;
             break;

        case PBM_TYPE:
             if (verbose) {
                fprintf(stderr,"pbm file\n%d %d\n%d\n",cols,rows,maxval);
             }
             fprintf(stderr,"pbm format not supported.\n");
             exit(-1);
             break;

        default:
        fprintf(stderr,"bad magic number - not a ppm, pgm, or pbm file" );
    }

    enInfo->compInfo = (JpegComponentInfo *) malloc
            (enInfo->numComponents * sizeof (JpegComponentInfo));

    /*
     * Because JPEG standard (DIS) defined downsampling, in our
     * CompressInfo structure, we have sampling factor and an
     * index array MCUmembership, in case a MCU contains several
     * samples of one color component. But in lossless JPEG, one
     * normally don't use downsampling.
     */
    for (ci = 0; ci < enInfo->numComponents; ci++) {
        compPtr = &(enInfo->compInfo[ci]);

        compPtr->componentId = ci;
        compPtr->componentIndex = ci;
        compPtr->hSampFactor = 1;
        compPtr->vSampFactor = 1;
    }

    /*
     * Set curCompInfo equal to compInfo since these is only
     * one scan.
     */
    enInfo->compsInScan=enInfo->numComponents;
    for (ci = 0; ci < enInfo->compsInScan; ci++) {
        enInfo->curCompInfo[ci] = &(enInfo->compInfo[ci]);
    }

    /*
     * Prepare array indexing MCU components into curCompInfo.
     */
    if (enInfo->compsInScan == 1) {
        enInfo->MCUmembership[0] = 0;
    } else {
        short ci, mcublks;

        if (enInfo->compsInScan > 4) {
            fprintf (stderr, "Too many components for interleaved scan");
            exit (-1);
        }

        for (ci = 0; ci < enInfo->compsInScan; ci++) {
            enInfo->MCUmembership[ci] = ci;
        }
    }

    /*
     * Alloc mucTable. Read source image into the table.
     * Apply point transform if Pt!=0. Update the input
     * file size - inputFileBytes.
     */ 
    maxMcu = enInfo->imageWidth * enInfo->imageHeight;
    InitMcuTable (maxMcu, enInfo->compsInScan);

    for (i = 0; i < maxMcu; i++) {
        mcu = MakeMCU(enInfo);
        for (j = 0; j < enInfo->compsInScan; j++) {
            mcu[j] = (short)pbm_getrawbyte( inFile );
            if (enInfo->Pt!=0) {
               mcu[j] >>= enInfo->Pt;
            }
        }
    }
    inputFileBytes+=maxMcu*enInfo->compsInScan;
}
