/*
 * tocxWidget.cpp -- 
 * 
 *      Implements the "tocx" widget.
 *
 * Copyright (c) 1997 Cornell University.
 *
 * See the file "license.terms" for information on usage and redistribution
 * of this file, and for a DISCLAIMER OF ALL WARRANTIES.
 * 
 */

#include "stdafx.h"
#include <afxole.h>
#include "tocxInt.h"
#include "tocxMfc.h"
#include <tkWinInt.h>	/* Tk private header file. Needed to access
			 * internal data structures to find out Windows
			 * specific information (such as HWND) about Tk
			 * windows.
			 */

/*
 * Forward declarations for procedures defined later in this file:
 */

static int              tocxBind(Tcl_Interp *interp, CtocxObject *tocxPtr,
                            int argc, char ** argv);
static int              tocxConfigure(Tcl_Interp *interp,
			    CtocxObject *tocxPtr,
                            int argc, char ** argv, int flags);
static int              tocxSetProperty(Tcl_Interp *interp, CtocxObject
			    *tocxPtr, char *property_name,
			    char *property_value);
static int              tocxGetProperty(Tcl_Interp *interp, CtocxObject *tocxPtr,
                            char *property_name);
static int              tocxConfigure(Tcl_Interp *interp, CtocxObject *tocxPtr,
                            int argc, char ** argv, int flags);
static void             tocxCmdDeletedProc _ANSI_ARGS_ ((
                            ClientData clientData));
static int              tocxConfigureInfo(Tcl_Interp *interp,
                            CtocxObject *tocxPtr);
static int              tocxConfigureInfo1(Tcl_Interp *interp,
                            CtocxObject *tocxPtr, char *property_name);
static int              tocxConfigureWidget(Tcl_Interp *interp,
                            CtocxObject *tocxPtr, int argc, char ** argv);
static int              tocxEventInfo(Tcl_Interp *interp, CtocxObject*tocxPtr);
static int              tocxEventInfo1(Tcl_Interp *interp, CtocxObject *tocxPtr,
                            char * name, FUNCDESC * descPtr);
static void             tocxEventProc(ClientData clientData, XEvent *eventPtr);
static int              tocxInfo(Tcl_Interp *interp, CtocxObject *tocxPtr,
                            int argc, char **argv);
static int              tocxGetPutIndexedProp(Tcl_Interp *interp,
                            CtocxObject *tocxPtr, int isGet, char * name,
                            int argc, char **argv);
static int              tocxGetPutInfo(Tcl_Interp *interp, CtocxObject *tocxPtr,
                            int isGet);
static int              tocxGetPutInfo1(Tcl_Interp *interp, CtocxObject *tocxPtr,
                            int isGet, char * name, TocxPropertyInfo *infoPtr);
static TocxPropertyInfo*    tocxGetPropertyInfo _ANSI_ARGS_((Tcl_Interp * interp,
                            CtocxObject * tocxPtr, char * name, int indexed));
static int              tocxMethodInfo(Tcl_Interp *interp, CtocxObject*tocxPtr);
static int              tocxMethodInfo1(Tcl_Interp *interp, CtocxObject *tocxPtr,
                            char * name, FUNCDESC * descPtr);
static int              tocxWidgetCmd _ANSI_ARGS_((ClientData clientData,
                            Tcl_Interp *, int argc, char **argv));
static int ProcessInfoCmd(Tcl_Interp *interp,int argc,char **argv);

static HRESULT          CallMethod(Tcl_Interp * interp, CtocxObject * tocxPtr,
                            TocxFuncInfo *fiPtr, int argc, char **argv);

/*
 * Global Data 
 */

static int child_id = 1234;

#define tocx_WIDTH  ((TocxPropertyInfo *)0x00000001)
#define tocx_HEIGHT ((TocxPropertyInfo *)0x00000002)


/*
 *----------------------------------------------------------------------
 * tocxGetPropertyInfo --
 *
 *      Returns a TocxPropertyInfo pointer to the named property.
 *
 * Results:
 *      NULL if not found.
 *
 * Side effects:
 *      If not found, interp->result stores error message.
 *----------------------------------------------------------------------
 */

TocxPropertyInfo *
tocxGetPropertyInfo(
    Tcl_Interp * interp,
    CtocxObject * tocxPtr,
    char * name,
    int indexed)                /* Is this an indexed property? */
{
    int len = strlen(name);
    TocxPropertyInfo * infoPtr;

    for (infoPtr=tocxPtr->propHead; infoPtr; infoPtr=infoPtr->next) {
        if (infoPtr->indexed == indexed && stricmp(name, infoPtr->name) ==0) {
            return infoPtr;
        }
    }

    if (!indexed) {
        if (strncmp(name, "-width", len) == 0) {
            return tocx_WIDTH;
        } else if (strncmp(name, "-height", len) == 0) {
            return tocx_HEIGHT;
        }
    }

    if (interp != NULL) {
        Tcl_AppendResult(interp, "unknown option \"", name, "\"",
                (char*)NULL);
    }
    return NULL;
}

static TocxMethodInfo*
tocxGetFunc(
    CtocxObject * tocxPtr,
    char * name)
{
    int len = strlen(name);
    TocxMethodInfo * infoPtr;

    for(infoPtr=tocxPtr->funcHead; infoPtr; infoPtr=infoPtr->next) {
        if (stricmp(name, infoPtr->name) ==0) {
            return infoPtr;
        }
    }
    return NULL;
}

static FUNCDESC*
tocxGetEventDesc(
    CtocxObject * tocxPtr,
    char * name)
{
    int len = strlen(name);
    TocxEventInfo * infoPtr;

    for(infoPtr=tocxPtr->eventHead; infoPtr; infoPtr=infoPtr->next) {
        if (stricmp(name, infoPtr->name) ==0) {
            return infoPtr->desc;
        }
    }
    return NULL;
}

/*
 *--------------------------------------------------------------
 *
 * Tocx_TocxCmd --
 *
 *      This procedure is invoked to process the "tocx" Tcl
 *      command. It creates a new "tocx" widget.
 *
 * Results:
 *      A standard Tcl result.
 *
 * Side effects:
 *      A new widget is created and configured.
 *
 *--------------------------------------------------------------
 */

int
Tocx_TocxCmd(
    ClientData clientData,	/* Stores the handle to the Tk main window. */
    Tcl_Interp *interp,		/* Current interpreter. */
    int argc,                   /* Number of arguments. */
    char **argv)                /* Argument strings. */
{
    Tk_Window main = (Tk_Window) clientData;
    Tk_Window tkwin = NULL;
    CtocxObject *tocxPtr = NULL;
    RECT rect;
    HRESULT hr = 0;

    static int inited = 0;

    if (!inited) {
	/*
	 * Enable MFC support for Control Containment
	 */
	AfxEnableControlContainer();
    }

    if (argc != 3) {
        Tcl_AppendResult(interp, "wrong # args: should be \"",
                argv[0], " pathName ClassName ?options? \"", (char *) NULL);
        return TCL_ERROR;
    }

    tkwin = Tk_CreateWindowFromPath(interp, main, argv[1], (char *)NULL);
    if (tkwin == NULL) {
		Tcl_AppendResult(interp, "tocx Tk Window Creation failed: ",
                Tocx_ErrorCode(hr), (char*)NULL);
        return TCL_ERROR;
    }

    /*
     * Tk delays the creation of the window in an idle handler. However,
     * we need the HWND below in the tocxPtr->parent->Attach() call,
     * so we force the creation of the window.
     */
    Tk_MakeWindowExist(tkwin);

    tocxPtr = new CtocxObject;
    tocxPtr->tkwin  = tkwin;
    tocxPtr->interp = interp;
    tocxPtr->parent = new CWnd;
    tocxPtr->width  = 0;
    tocxPtr->height = 0;
    tocxPtr->funcHead = NULL;
    tocxPtr->funcTail = NULL;
    tocxPtr->propHead = NULL;
    tocxPtr->propTail = NULL;
    tocxPtr->eventHead = NULL;
    tocxPtr->eventTail = NULL;
    tocxPtr->typeInfoList = NULL;
    tocxPtr->bindList = NULL;
    tocxPtr->widgetCmd = NULL;
    tocxPtr->m_pdisp = NULL;

    tocxPtr->parent->Attach(TkWinGetHWND(Tk_WindowId(tkwin)));

    rect.left = 0;
    rect.top = 0;
    rect.right = 200;
    rect.bottom = 200;

    hr = tocxPtr->Create(argv[2], argv[2], 0, rect, tocxPtr->parent,
	    child_id);
    if (SUCCEEDED(hr)) {
        child_id++;
    } else {
        Tcl_AppendResult(interp, "tocx Object creation failed: ",
                Tocx_ErrorCode(hr), (char*)NULL);
        goto error;
    }

    if (tocxPtr->UpdateInfoTable() != TCL_OK) {
	Tcl_AppendResult(interp, "tocx information table creation failed: ",
                Tocx_ErrorCode(hr), (char*)NULL);
        goto error;
    }
    tocxPtr->ShowWindow(SW_SHOW);
    tocxPtr->widgetCmd = Tcl_CreateCommand(interp,
	    Tk_PathName(tocxPtr->tkwin), tocxWidgetCmd, (ClientData)tocxPtr,
	    tocxCmdDeletedProc);

    Tk_CreateEventHandler(tocxPtr->tkwin, ExposureMask|StructureNotifyMask,
            tocxEventProc, (ClientData) tocxPtr);

    if (tocxConfigureWidget(interp, tocxPtr, argc-3, argv+3) != TCL_OK) {
        Tk_DestroyWindow(tocxPtr->tkwin);
		Tcl_AppendResult(interp, "tocx Object configuration failed: ",
                Tocx_ErrorCode(hr), (char*)NULL);
        return TCL_ERROR;
    }

    interp->result = Tk_PathName(tocxPtr->tkwin);

    return TCL_OK;

  error:
    if (tkwin != NULL) {
        Tk_DestroyWindow(tkwin);
    }
    if (tocxPtr != NULL) {
        delete tocxPtr;
    }

    return TCL_ERROR;
}


/*
 *--------------------------------------------------------------
 *
 * tocxWidgetCmd --
 *
 *      This procedure is invoked to process the Tcl command
 *      that corresponds to a widget managed by this module.
 *      See the user documentation for details on what it does.
 *
 * Results:
 *      A standard Tcl result.
 *
 * Side effects:
 *      See the user documentation.
 *
 *--------------------------------------------------------------
 */

static int
tocxWidgetCmd(
    ClientData clientData,      /* Info about the OCX. */
    Tcl_Interp * interp,        /* Current interpreter. */
    int argc,                   /* Number of arguments. */
    char **argv)                /* Argument strings. */
{
    CtocxObject *tocxPtr = (CtocxObject *) clientData;
    int result = TCL_OK;
    int len;

    if (argc < 2) {
        Tcl_AppendResult(interp, "wrong # args: should be \"",
                argv[0], " command ?arg arg ...?\"", (char *) NULL);
        return TCL_ERROR;
    }

    len = strlen(argv[1]);

    if (strncmp(argv[1], "bind", len) == 0) {
        return tocxBind(interp, tocxPtr, argc, argv);
    }
    else if (strncmp(argv[1], "configure", len) == 0) {
        if (argc == 2) {
            return tocxConfigureInfo(interp, tocxPtr);
        } else if (argc == 3) {
            return tocxConfigureInfo1(interp, tocxPtr, argv[2]);
        } else {
            return tocxConfigureWidget(interp, tocxPtr, argc-2, argv+2);
        }
    }
    else if (strncmp(argv[1], "get", len) == 0) {
        if (argc < 4) {
            Tcl_AppendResult(interp, "wrong # args: should be ",
                argv[0], " get option index ?...?", NULL);
            return TCL_ERROR;
        }
        return tocxGetPutIndexedProp(interp, tocxPtr, 1, argv[2], argc-3,
		argv+3);
    }
    else if (strncmp(argv[1], "info", len) == 0) {
        return tocxInfo(interp, tocxPtr, argc, argv);
    }
    else if (strncmp(argv[1], "put", len) == 0) {
        if (argc < 5) {
            Tcl_AppendResult(interp, "wrong # args: should be ",
                argv[0], " put option index ?...? value", NULL);
            return TCL_ERROR;
        }
        return tocxGetPutIndexedProp(interp, tocxPtr, 0, argv[2], argc-3,
		argv+3);
    }
    else {
        TocxMethodInfo *minfo;
        char * methodName;
        int domethodUsed;

        if (strncmp(argv[1], "domethod", len) == 0) {
            if (tocxPtr->funcHead == NULL) {
                Tcl_AppendResult(interp, "this tocx control does not ",
			"support methods", NULL);
                return TCL_ERROR;
            }

            if (argc <= 2) {
                Tcl_AppendResult(interp, "wrong # of arguments: should be ",
                        "?domethod? methodname ?arg ...?", NULL);
                return TCL_ERROR;
            }

            methodName = argv[2];
            argc -= 3;
            argv += 3;
            domethodUsed = 1;
        } else {
            methodName = argv[1];
            argc -= 2;
            argv += 2;
            domethodUsed = 0;
        }

        minfo = tocxGetFunc(tocxPtr, methodName);
        if (minfo != NULL) {
            return CallMethod(interp, tocxPtr, minfo->info, argc, argv);
        }
        else {
            /*
             * Print out a list of available commands. This is tricky
             * because if the user says "domethod", the available
             * options will *NOT* include the standard "configure",
             * "info" and "domethod".
             *
             * Note that we assume that tocxPtr->funcHead is non NULL
             * if (domethodUsed == 1); the condition is checked above.
             *
             * The error results can be converted into a Tcl list by
             *  " or " and ", " into the space character 
             */

            TocxMethodInfo * infoPtr;

            Tcl_AppendResult(interp, "wrong option \"", methodName,
                        "\", should be: ", NULL);

            if (domethodUsed == 0) {
                Tcl_AppendResult(interp, "bind, configure, domethod, ",
                        "get, info", NULL);
                if (tocxPtr->funcHead == NULL) {
                    Tcl_AppendResult(interp, " or put", NULL);
                    return TCL_ERROR;
                } else {
                    Tcl_AppendResult(interp, ", put", NULL);
                }
            }
            for(infoPtr=tocxPtr->funcHead; infoPtr;
                    infoPtr=infoPtr->next) {
                if (infoPtr->next == NULL) {
                    Tcl_AppendResult(interp, " or ", NULL);
                } else {
                    Tcl_AppendResult(interp, ", ",   NULL);
                }
                Tcl_AppendResult(interp, infoPtr->name, NULL);
            }
            return TCL_ERROR;
        }
    }
}

/*
 *----------------------------------------------------------------------
 * tocxBind --
 *
 *      Creates, displays or deletes an event binding.
 *
 * Results:
 *      Standard Tcl result. See user documentation.
 *
 * Side effects:
 *      See user documentation.
 *----------------------------------------------------------------------
 */

static int
tocxBind(
    Tcl_Interp * interp,        /* Current interpreter. */
    CtocxObject * tocxPtr,        /* Info about tocx widget. */
    int argc,                   /* Number of arguments. */
    char **argv)                /* Argument strings. */
{
    FUNCDESC * descPtr;
    TocxEventBinding * bindPtr, *lastBindPtr;

    if (argc < 3 || argc > 4) {
        Tcl_AppendResult(interp, "wrong # args: should be ",
                argv[0], " bind event ?script?", NULL);
        return TCL_ERROR;
    }

    descPtr = tocxGetEventDesc(tocxPtr, argv[2]);
    if (descPtr == NULL) {
        Tcl_AppendResult(interp, "unknown event \"", argv[2], "\"", NULL);
        return TCL_ERROR;
    }

    for (lastBindPtr=tocxPtr->bindList, bindPtr=tocxPtr->bindList;
             bindPtr;
             lastBindPtr=bindPtr, bindPtr=bindPtr->next) {
        if (bindPtr->memid == descPtr->memid) {
            break;
        }
    }

    if (argc == 3) {
        /*
         * Returns the current binding
         */
        if (bindPtr != NULL) {
            Tcl_AppendResult(interp, bindPtr->cmdPrefix, NULL);
        }
        return TCL_OK;
    }
    else if (strlen(argv[3]) == 0) {
        if (bindPtr != NULL) {
            /*
             * Deletes the binding.
             */
            if (lastBindPtr == bindPtr) {
                tocxPtr->bindList = bindPtr->next;
            } else {
                lastBindPtr->next = bindPtr->next;
            }
            ckfree(bindPtr->cmdPrefix);
            ckfree((char*)bindPtr);
        }
        return TCL_OK;
    }
    else {
        /*
         * Create or modify the binding.
         */
        if (bindPtr != NULL) {
            ckfree(bindPtr->cmdPrefix);
        } else {
            bindPtr = (TocxEventBinding *)ckalloc(sizeof(TocxEventBinding));
            bindPtr->memid = descPtr->memid;
            bindPtr->next = tocxPtr->bindList;
            tocxPtr->bindList = bindPtr;
        }

        bindPtr->cmdPrefix = (char*)ckalloc(strlen(argv[3])+1);
        strcpy(bindPtr->cmdPrefix, argv[3]);

        return TCL_OK;
    }
}

/*
 *----------------------------------------------------------------------
 * CallMethod --
 *
 *      Call a method or a property get/put(ref) function.
 *
 * Results:
 *      Standard TCL result.
 *
 * Side effects:
 *      None.
 *----------------------------------------------------------------------
 */

static HRESULT
CallMethod(
    Tcl_Interp * interp,        /* Current interpreter. */
    CtocxObject * tocxPtr,        /* Info about tocx widget. */
    TocxFuncInfo *fiPtr,    /* Info about the function. */
    int argc,                   /* Number of arguments. */
    char **argv)                /* Argument strings. */
{
    DISPPARAMS params;
    VARIANTARG *vargs;
    VARIANT varResult;
    HRESULT hr;
    unsigned int uArgErr;
    int code, i;
    FUNCDESC *desc = fiPtr->desc;
    code = TCL_OK;
    vargs = NULL;

    params.rgdispidNamedArgs = NULL;
    params.cNamedArgs = 0;
    params.rgvarg = NULL;
    params.cArgs = 0;

    if (desc->cParamsOpt == -1) {
        /*
         * (ToDo) tocxAuto, pp263
         */
        Tcl_AppendResult(interp, "SafeArray optargs not supported", NULL);
        return TCL_ERROR;
    } else {
        char buff[100];
        if (argc < desc->cParams - desc->cParamsOpt) {
            sprintf(buff, "%d", argc);
            Tcl_AppendResult(interp, "Got ", buff, " argument(s); ", NULL);
            sprintf(buff, "%d", desc->cParams - desc->cParamsOpt);
            Tcl_AppendResult(interp, "expected (at least) ", buff, NULL);
            return TCL_ERROR;
        } else if (argc > desc->cParams) {
            sprintf(buff, "%d", argc);
            Tcl_AppendResult(interp, "Got ", buff, " argument(s); ", NULL);
            sprintf(buff, "%d", desc->cParams);
            Tcl_AppendResult(interp, "expected at most ", buff, NULL);
            return TCL_ERROR;
        }
    }

    /*
     * We don't support call-by-name for normal methods. However, the
     * property get/put must be called in a call-by-name convention,
     * with a single named argument whose DISPID is set according the
     * get/set operation
     */
    if (desc->invkind == INVOKE_PROPERTYPUT) {
        params.rgdispidNamedArgs = (DISPID*)ckalloc(sizeof(DISPID) * 1);
        params.cNamedArgs = 1;
        params.rgdispidNamedArgs[0] = DISPID_PROPERTYPUT;
    }

    if (argc > 0) {
        vargs = (VARIANTARG *)ckalloc(sizeof(VARIANTARG) * argc);
        params.rgvarg = vargs;
        params.cArgs = argc;

        for (i=0; i<argc; i++) {
            VariantInit(&vargs[i]);
        }
        for (i=0; i<argc; i++) {
            /*
             * Note: the arguments are passed in reversed order.
             */
            if (Tocx_StringToVariant(interp, argv[argc-i-1],
                    fiPtr->argTypes[i], &vargs[i])!=TCL_OK) {
                code = TCL_ERROR;
                goto done;
            }
        }
    }

    /*
     * Invoke the method
     */
    hr = tocxPtr->m_pdisp->Invoke(desc->memid, IID_NULL, LOCALE_SYSTEM_DEFAULT,
            desc->invkind, &params, &varResult, NULL, &uArgErr);
    if (FAILED(hr)) {
        Tcl_AppendResult(interp, "method invokation failed: ",
                Tocx_ErrorCode(hr), NULL);
        code = TCL_ERROR;
        goto done;
    }

  done:
    if (params.rgdispidNamedArgs != NULL) {
        ckfree((char*)(params.rgdispidNamedArgs));
    }
    if (vargs != NULL) {
        for (i=0; i<argc; i++) {
            VariantClear(&vargs[i]);
        }
        ckfree((char*)vargs);
    }
    if (code == TCL_OK) {
        code = Tocx_VariantToString(interp, fiPtr->type, &varResult);
    }
    return code;
}

/*
 *----------------------------------------------------------------------
 * tocxGetPutIndexedProp --
 *
 *      Gets or puts an indexed property.
 *
 * Results:
 *      Standard Tcl result.
 *
 * Side effects:
 *      Property may be chaned as the result of a put call.
 *----------------------------------------------------------------------
 */

static int
tocxGetPutIndexedProp(
    Tcl_Interp *interp,         /* Current interpreter. */
    CtocxObject *tocxPtr,         /* Info about the tocx widget. */
    int isGet,                  /* True if this is a PROPERTYGET call. */
    char * name,                /* Name of the indexed property. */
    int argc,                   /* Number of arguments. */
    char **argv)                /* Argument strings. */
{
#if 0
    TocxPropertyInfo * infoPtr;
    FUNCDESC *descPtr;

    infoPtr = tocxGetPropertyInfo(interp, tocxPtr, name, 1);
    if (infoPtr == NULL) {
        return TCL_ERROR;
    }
    if (isGet) {
        if (infoPtr->funcGet == NULL) {
            Tcl_AppendResult(interp, "\"", name, "\" does not support ",
                    "PROPERTYGET", NULL);
            return TCL_ERROR;
        }
        descPtr = infoPtr->funcGet;
    } else {
        if (infoPtr->funcPut == NULL) {
            Tcl_AppendResult(interp, "\"", name, "\" does not support ",
                    "PROPERTYPUT", NULL);
            return TCL_ERROR;
        }
        descPtr = infoPtr->funcPut;
    }

    return CallMethod(interp, tocxPtr, descPtr, argc, argv);
#else
    return 0;
#endif
}

/*
 *----------------------------------------------------------------------
 * tocxGetPutProperty --
 *
 *      Gets or puts the (non-indexed) property. The main work of type
 *      conversion is done by CallMethod().
 *
 * Results:
 *      Standard Tcl result.
 *
 * Side effects:
 *      If property get, value will be stored in interp->result.
 *----------------------------------------------------------------------
 */

static int
tocxGetPutProperty(
    Tcl_Interp *interp,		/* Current Interpreter. */
    CtocxObject *tocxPtr,       /* Info about tocx widget. */
    TocxPropertyInfo *propInfo,     /* Info about the property. */
    char *value)                /* New value for the property. NULL for
                                 * property get */
{
    if (propInfo->typeInfoPtr != NULL) {
        /*
         * This property is defined as a "variable". No property
         * get/set TocxFuncInfo has been defined. So we create one
         * on the fly.
         */
        TocxFuncInfo funcInfo;
        FUNCDESC desc;
        TocxTypeInfo *argTypes[1];

        if (value == NULL) {
            /*
             * Get property
             */
            desc.invkind = INVOKE_PROPERTYGET;
            desc.cParams = 0;
            funcInfo.argTypes = NULL;
	    funcInfo.type = propInfo->typeInfoPtr;
        } else {
            ELEMDESC elemDesc;

            /*
             * Put property
             */
            elemDesc.tdesc.vt = propInfo->typeInfoPtr->tdesc->vt;

            desc.invkind = INVOKE_PROPERTYPUT;
            desc.cParams = 1;
            desc.lprgelemdescParam = &elemDesc;

            argTypes[0] = propInfo->typeInfoPtr;
            funcInfo.argTypes = argTypes;
	    funcInfo.type = NULL;
        }

        desc.cParamsOpt = 0;
        desc.memid = propInfo->memid;
        funcInfo.desc = &desc;

        return CallMethod(interp, tocxPtr, &funcInfo, desc.cParams, &value);
    } else {
	/*
	 * This is a get/set property. We call the appropriate property access
	 * function.
	 */
        if (value == NULL) {
            /*
             * Get property
             */
            if (propInfo->funcGet == NULL) {
                Tcl_AppendResult(interp, "*unreadable*", NULL);
                return TCL_OK;
            } else {
                return CallMethod(interp, tocxPtr, propInfo->funcGet,
                        0, NULL);
            }
        } else {
            /*
             * Put property
             */
            if (propInfo->funcPut == NULL) {
                Tcl_AppendResult(interp, "option \"", propInfo->name,
                        "\" is readonly", NULL);
                return TCL_ERROR;
            } else {
                return CallMethod(interp, tocxPtr, propInfo->funcPut,
                        1, &value);
            }
        }
    }
}

static int
tocxSetProperty(
    Tcl_Interp *interp,         /* Current Interpreter. */
    CtocxObject *tocxPtr,         /* Info about tocx widget. */
    char *property_name,        /* Name of the option to change. */
    char *property_value,       /* New value for the option. */
    int * sizeChangedPtr)       /* This integer will be set to true if the
                                 * option changes the size of the widget. */
{
    TocxPropertyInfo * propInfo;

    propInfo = tocxGetPropertyInfo(interp, tocxPtr, property_name, 0);
    if (propInfo == NULL) {
        return TCL_ERROR;
    }
    if (propInfo == tocx_WIDTH || propInfo == tocx_HEIGHT) {
        int n;

        if (Tcl_GetInt(interp, property_value, &n) != TCL_OK) {
            return TCL_ERROR;
        }
        if (propInfo == tocx_WIDTH) {
            tocxPtr->width = n;
        } else {
            tocxPtr->height = n;
        }

        *sizeChangedPtr = 1;
        return TCL_OK;
    } else {
        return tocxGetPutProperty(interp, tocxPtr, propInfo, property_value);
    }
}

/*----------------------------------------------------------------------
 * tocxConfigureWidget --
 *
 *      This procedure is called to process an argv/argc list in
 *      conjunction with the Tk option database to configure (or
 *      reconfigure) a tocx widget.
 *
 * Results:
 *      The return value is a standard Tcl result.  If TCL_ERROR is
 *      returned, then interp->result contains an error message.
 *
 * Side effects:
 *      Configuration information, such as colors, border width,
 *      etc. get set for tocxPtr;  old resources get freed,
 *      if there were any.
 *
 *----------------------------------------------------------------------
 */

static int
tocxConfigureWidget(
    Tcl_Interp *interp,
    CtocxObject *tocxPtr,
    int argc,
    char ** argv)
{
    int i;
    int changed;

    if ((argc % 2) == 0) {
        changed = 0;
        for (i=0; i<argc; i+=2) {
            if (tocxSetProperty(interp, tocxPtr, argv[i], argv[i+1], &changed)
                    != TCL_OK) {
                return TCL_ERROR;
            }
        }
        if (changed) {
            int w = tocxPtr->width;
            int h = tocxPtr->height;

            if (w <= 0) {
                w = 1;
            }
            if (h <= 0) {
                h = 1;
            }
            Tk_GeometryRequest(tocxPtr->tkwin, w, h);
        }
        return TCL_OK;
    } else {
        char * last = argv[argc-1];
        if (tocxGetPropertyInfo(interp, tocxPtr, last, 0) == NULL) {
            return TCL_ERROR;
        } else {
            Tcl_AppendResult(interp, "value missing for \"", last, "\"",
                    (char*)NULL);
            return TCL_ERROR;
        }
    }
}

static int
tocxConfigureInfo(
    Tcl_Interp *interp,
    CtocxObject *tocxPtr)
{
    TocxPropertyInfo *infoPtr;
    Tcl_DString dstring;

    Tcl_DStringInit(&dstring);

    for(infoPtr=tocxPtr->propHead; infoPtr; infoPtr=infoPtr->next) {
        if (!infoPtr->indexed) {
            Tcl_ResetResult(interp);
            tocxConfigureInfo1(interp, tocxPtr, infoPtr->name);
            Tcl_DStringAppendElement(&dstring, interp->result);
        }
    }

    Tcl_ResetResult(interp);
    tocxConfigureInfo1(interp, tocxPtr, "-width");
    Tcl_DStringAppendElement(&dstring, interp->result);

    Tcl_ResetResult(interp);
    tocxConfigureInfo1(interp, tocxPtr, "-height");
    Tcl_DStringAppendElement(&dstring, interp->result);

    Tcl_SetResult(interp, dstring.string, TCL_VOLATILE);
    Tcl_DStringFree(&dstring);

    return TCL_OK;
}

static int
tocxConfigureInfo1(
    Tcl_Interp *interp,
    CtocxObject *tocxPtr,
    char *property_name)
{
    TocxPropertyInfo * propInfo;
    char buff[100];

    propInfo = tocxGetPropertyInfo(interp, tocxPtr, property_name, 0);
    if (propInfo == NULL) {
        return TCL_ERROR;
    }

    if (propInfo == tocx_WIDTH) {
        sprintf(buff, "%d", tocxPtr->width);
        Tcl_AppendElement(interp, "-width");
        Tcl_AppendElement(interp, "width");
        Tcl_AppendElement(interp, "Width");
        Tcl_AppendElement(interp, "0");
        Tcl_AppendElement(interp, buff);        
    } else if (propInfo == tocx_HEIGHT) {
        sprintf(buff, "%d", tocxPtr->height);
        Tcl_AppendElement(interp, "-height");
        Tcl_AppendElement(interp, "height");
        Tcl_AppendElement(interp, "Height");
        Tcl_AppendElement(interp, "0");
        Tcl_AppendElement(interp, buff);        
    } else {
        Tcl_DString dstring;
        int code;

        Tcl_DStringInit(&dstring);
        code = tocxGetPutProperty(interp, tocxPtr, propInfo, NULL);

        /*
         * Return value: list of
         *      command-line switch     (same as name)
         *      name
         *      class                   (tocx type)
         *      default value           (not supported, make it "")
         *      current value
         */
        Tcl_DStringAppendElement(&dstring, property_name);
        Tcl_DStringAppendElement(&dstring, property_name);
	if (propInfo->typeInfoPtr != NULL) { 
	    Tcl_DStringAppendElement(&dstring,
		    Tocx_TypeName(propInfo->typeInfoPtr));
	} else { 
	    // Do something
	}
        Tcl_DStringAppendElement(&dstring, "");
        if (code == TCL_OK) {
            Tcl_DStringAppendElement(&dstring, interp->result);
        } else {
            Tcl_DStringAppendElement(&dstring, interp->result);
        }

        Tcl_SetResult(interp, dstring.string, TCL_VOLATILE);
        Tcl_DStringFree(&dstring);
    }

    return TCL_OK;
}

/*
 *----------------------------------------------------------------------
 * tocxInfo --
 *
 *      Returns information of the tocx object.
 *
 * Results:
 *      Standard Tcl result.
 *
 * Side effects:
 *      None.
 *----------------------------------------------------------------------
 */

static int
tocxInfo(
    Tcl_Interp *interp,         /* Current interpreter. */
    CtocxObject *tocxPtr,       /* Info about tocx widget. */
    int argc,                   /* Number of arguments */
    char **argv)                /* Argument list */
{
    int len;

    if (argc < 3 || argc > 4) {
        Tcl_AppendResult(interp, "wrong number of arguments, should be: ",
                argv[0], " info option ?opt?", NULL);
        return TCL_ERROR;
    }

    len = strlen(argv[2]);
    
    if (strncmp(argv[2], "event", len) == 0) {
        if (argc == 3) {
            return tocxEventInfo(interp, tocxPtr);
        } else {
            return tocxEventInfo1(interp, tocxPtr, argv[3], NULL);
        }
    }
    else if (strncmp(argv[2], "get", len) == 0) {
        if (argc == 3) {
            return tocxGetPutInfo(interp, tocxPtr, 1);
        } else {
            return tocxGetPutInfo1(interp, tocxPtr, 1, argv[3], NULL);
        }
    }
    else if (strncmp(argv[2], "method", len) == 0) {
        if (argc == 3) {
            return tocxMethodInfo(interp, tocxPtr);
        } else {
            return tocxMethodInfo1(interp, tocxPtr, argv[3], NULL);
        }
    }
    else if (strncmp(argv[2], "option", len) == 0) {
        return TCL_OK;
    }
    else if (strncmp(argv[2], "put", len) == 0) {
        if (argc == 3) {
            return tocxGetPutInfo(interp, tocxPtr, 1);
        } else {
            return tocxGetPutInfo1(interp, tocxPtr, 1, argv[3], NULL);
        }
    }
    else {
        Tcl_AppendResult(interp, "unknown option \"", argv[2],
                "\", should be: event, get, method, option or put", NULL);
        return TCL_ERROR;
    }
}

/*
 *----------------------------------------------------------------------
 * tocxGetPutInfo --
 *
 *      Returns information about the property get (OR put) functions
 *      of the indexed properties.
 *
 * Results:
 *      Stardard Tcl result.
 *
 * Side effects:
 *      None.
 *----------------------------------------------------------------------
 */

static int
tocxGetPutInfo(
    Tcl_Interp *interp,         /* Current interpreter. */
    CtocxObject *tocxPtr,         /* Info about tocx widget. */
    int isGet)                  /* Whether we want info for get or put funcs*/
{
    TocxPropertyInfo *infoPtr;
    Tcl_DString dstring;

    Tcl_DStringInit(&dstring);

    for (infoPtr=tocxPtr->propHead; infoPtr; infoPtr=infoPtr->next) {
        if (infoPtr->indexed) {
            if (((isGet)  && (infoPtr->funcGet != NULL)) ||
                ((!isGet) && (infoPtr->funcPut != NULL))) {

                Tcl_ResetResult(interp);
                tocxGetPutInfo1(interp, tocxPtr, isGet, infoPtr->name,
		        infoPtr);
                Tcl_DStringAppendElement(&dstring, interp->result);
            }
        }
    }

    Tcl_SetResult(interp, dstring.string, TCL_VOLATILE);
    Tcl_DStringFree(&dstring);

    return TCL_OK;
}

/*
 *----------------------------------------------------------------------
 * tocxGetPutInfo1 --
 *
 *      Returns information about the property get (OR put) function
 *      of an indexed property.
 *
 * Results:
 *      Stardard Tcl result.
 *
 * Side effects:
 *      None.
 *----------------------------------------------------------------------
 */

static int
tocxGetPutInfo1(
    Tcl_Interp *interp,         /* Current interpreter. */
    CtocxObject *tocxPtr,       /* Info about tocx widget. */
    int isGet,                  /* Whether we want info for get or put func.*/
    char * name,                /* Name of the property.*/
    TocxPropertyInfo *infoPtr)      /* (Optional) info about the property. */
{
    FUNCDESC * descPtr;

    if (infoPtr == NULL) {
        infoPtr = tocxGetPropertyInfo(interp, tocxPtr, name, 1);
        if (infoPtr == NULL) {
            return TCL_ERROR;
        }
    }
    if (!infoPtr->indexed) {
        Tcl_AppendResult(interp, "\"", name, "\" is not an indexed property",
                NULL);
        return TCL_ERROR;
    }
    if (isGet) {
        if (infoPtr->funcGet == NULL) {
            Tcl_AppendResult(interp, "\"", name, "\" does not support ",
                    "PROPERTYGET", NULL);
            return TCL_ERROR;
        }
        descPtr = infoPtr->funcGet->desc;
    } else {
        if (infoPtr->funcPut == NULL) {
            Tcl_AppendResult(interp, "\"", name, "\" does not support ",
                    "PROPERTYPUT", NULL);
            return TCL_ERROR;
        }
        descPtr = infoPtr->funcPut->desc;
    }

    return tocxMethodInfo1(interp, tocxPtr, name, descPtr);
}

/*
 *----------------------------------------------------------------------
 * tocxMethodInfo --
 *
 *      Returns information about all the property get (OR put) functions.
 *
 * Results:
 *      Stardard Tcl result.
 *
 * Side effects:
 *      None.
 *----------------------------------------------------------------------
 */

static int
tocxMethodInfo(
    Tcl_Interp *interp,         /* Current interpreter. */
    CtocxObject *tocxPtr)         /* Info about tocx widget. */
{
    TocxMethodInfo *infoPtr;
    Tcl_DString dstring;

    Tcl_DStringInit(&dstring);

    for (infoPtr=tocxPtr->funcHead; infoPtr; infoPtr=infoPtr->next) {
        Tcl_ResetResult(interp);
	tocxMethodInfo1(interp, tocxPtr, infoPtr->name,
		infoPtr->info->desc);
        Tcl_DStringAppendElement(&dstring, interp->result);
    }

    Tcl_SetResult(interp, dstring.string, TCL_VOLATILE);
    Tcl_DStringFree(&dstring);
    return TCL_OK;
}

/*
 *----------------------------------------------------------------------
 * tocxMethodInfo1 --
 *
 *      Returns information about one method.
 *
 * Results:
 *      Stardard Tcl result.
 *
 * Side effects:
 *      None.
 *----------------------------------------------------------------------
 */

static int
tocxMethodInfo1(
    Tcl_Interp *interp,         /* Current interpreter. */
    CtocxObject *tocxPtr,       /* Info about tocx widget. */
    char * name,                /* Name of the method.*/
    FUNCDESC * descPtr)         /* (Optional) Info about the func. */
{
    int i;

    if (descPtr == NULL) {
        descPtr = tocxGetFunc(tocxPtr, name)->info->desc;
        if (descPtr == NULL) {
            Tcl_AppendResult(interp, "unknown method \"", name, "\"", NULL);
            return TCL_ERROR;
        }
    }

    Tcl_AppendElement(interp, Tocx_ElemDescName(&descPtr->elemdescFunc));
    Tcl_AppendElement(interp, name);

    for (i=descPtr->cParams-1; i>=0; i--) {
        /*
         * The arguments are listed in reversed order in lprgelemdescParam
         */
        Tcl_AppendElement(interp,
                Tocx_ElemDescName(&descPtr->lprgelemdescParam[i]));
    }
    return TCL_OK;
}

/*
 *----------------------------------------------------------------------
 * tocxEventInfo --
 *
 *      Returns information about all the property get (OR put) functions.
 *
 * Results:
 *      Stardard Tcl result.
 *
 * Side effects:
 *      None.
 *----------------------------------------------------------------------
 */

static int
tocxEventInfo(
    Tcl_Interp *interp,         /* Current interpreter. */
    CtocxObject *tocxPtr)         /* Info about tocx widget. */
{
    TocxEventInfo *infoPtr;
    Tcl_DString dstring;

    Tcl_DStringInit(&dstring);

    for (infoPtr=tocxPtr->eventHead; infoPtr; infoPtr=infoPtr->next) {
        Tcl_ResetResult(interp);
        tocxEventInfo1(interp, tocxPtr, infoPtr->name, infoPtr->desc);
        Tcl_DStringAppendElement(&dstring, interp->result);
    }

    Tcl_SetResult(interp, dstring.string, TCL_VOLATILE);
    Tcl_DStringFree(&dstring);

    return TCL_OK;
}

/*
 *----------------------------------------------------------------------
 * tocxEventInfo1 --
 *
 *      Returns information about one event.
 *
 * Results:
 *      Stardard Tcl result.
 *
 * Side effects:
 *      None.
 *----------------------------------------------------------------------
 */

static int
tocxEventInfo1(
    Tcl_Interp *interp,         /* Current interpreter. */
    CtocxObject *tocxPtr,         /* Info about tocx widget. */
    char * name,                /* Name of the event.*/
    FUNCDESC * descPtr)         /* (Optional) Info about the func. */
{
    if (descPtr == NULL) {
        descPtr = tocxGetEventDesc(tocxPtr, name);
        if (descPtr == NULL) {
            Tcl_AppendResult(interp, "unknown event \"", name, "\"", NULL);
            return TCL_ERROR;
        }
    }

    return tocxMethodInfo1(interp, tocxPtr, name, descPtr);
}

/*
 *--------------------------------------------------------------
 *
 * tocxEventProc --
 *
 *      This procedure is invoked by the Tk dispatcher for various
 *      events on tocxs.
 *
 * Results:
 *      None.
 *
 * Side effects:
 *      When the window gets deleted, internal structures get
 *      cleaned up.  When it gets exposed, it is redisplayed.
 *
 *--------------------------------------------------------------
 */

static void
tocxEventProc(ClientData clientData,XEvent * eventPtr)
{
    CtocxObject *tocxPtr = (CtocxObject *) clientData; 

    if (eventPtr->type == Expose) {
        /*
         * Nothing to be done. The tocx widget should receive an Expose message
         * and will redraw itself.
         */
    }
    else if (eventPtr->type == ConfigureNotify) {
        tocxPtr->SetWindowPos(NULL, 0, 0, Tk_Width(tocxPtr->tkwin), 
                Tk_Height(tocxPtr->tkwin), 
                SWP_NOZORDER|SWP_NOACTIVATE|SWP_NOMOVE);
    }
    else if (eventPtr->type == DestroyNotify) {
        if (tocxPtr->tkwin != NULL) {
            tocxPtr->tkwin = NULL;
            Tcl_DeleteCommand(tocxPtr->interp,
                    Tcl_GetCommandName(tocxPtr->interp,
                    tocxPtr->widgetCmd));
        }
#if 1
        if (tocxPtr->m_pdisp != NULL) {
            tocxPtr->m_pdisp->Release();
        }

        /*
         * ToDo: free event, func and prop lists.
         *       free bindList.
         */
        delete tocxPtr;
#else
        Tcl_EventuallyFree((ClientData) tocxPtr, tocxDestroy);
#endif
    }
}

/*
 *----------------------------------------------------------------------
 *
 * tocxCmdDeletedProc --
 *
 *      This procedure is invoked when a widget command is deleted.  If
 *      the widget isn't already in the process of being destroyed,
 *      this command destroys it.
 *
 * Results:
 *      None.
 *
 * Side effects:
 *      The widget is destroyed.
 *
 *----------------------------------------------------------------------
 */

static void
tocxCmdDeletedProc(ClientData clientData)
{
    CtocxObject *tocxPtr = (CtocxObject *) clientData;
    Tk_Window tkwin = tocxPtr->tkwin;

    /*
     * This procedure could be invoked either because the window was
     * destroyed and the command was then deleted (in which case tkwin
     * is NULL) or because the command was deleted, and then this procedure
     * destroys the widget.
     */

    if (tkwin != NULL) {
        tocxPtr->tkwin = NULL;
        Tk_DestroyWindow(tkwin);
    }
}


/*
 *----------------------------------------------------------------------
 * Tocx_StringToVariant --
 *
 *      Translate a TCL string into a VARIANTARG used to pass as an
 *      argument to a method.
 *
 * Results:
 *      Standard TCL result.
 *
 * Side effects:
 *      TCL_OK: contents of varg is changed (possibly allocated). Must be
 *              freed by the caller.
 *      TCL_ERROR: contents of varg is not changed.
 *----------------------------------------------------------------------
 */

int
Tocx_StringToVariant(
    Tcl_Interp * interp,        /* Current interpreter */
    char * string,              /* String to translate into VARIANTARG */
    TocxTypeInfo *tiPtr,      /* Type to translate to */
    VARIANTARG * vargPtr)       /* Holds return value */
{
    VARIANTARG src;
    int code;
    HRESULT hr;
    VARTYPE vt;

    if (tiPtr->toVarProc != NULL) {
        code = tiPtr->toVarProc(interp, string, tiPtr, vargPtr);
    } else {
        vt = tiPtr->rootvt;
        code = TCL_OK;
        VariantInit(&src);

        CString cstr = string;
        src.vt = VT_BSTR;
        src.bstrVal = cstr.AllocSysString();

        switch (vt) {
        case VT_UI1:
        case VT_I2:
        case VT_I4:
        case VT_R4:
        case VT_R8:
        case VT_BOOL:
        case VT_CY:
        case VT_DATE:
        case VT_BSTR:
            hr = VariantChangeType(vargPtr, &src, 0, vt);
            if (FAILED(hr)) {
                Tcl_AppendResult(interp, "Type conversion to \"",
                        Tocx_TypeName(tiPtr), "\" failed: ",
                        Tocx_ErrorCode(hr), NULL);
                code = TCL_ERROR;
            }
            break;

        default:
            Tcl_AppendResult(interp, "Cannot handle variable type \"",
                    Tocx_TypeName(tiPtr), "\"", NULL);
            code = TCL_ERROR;
        }
    }

    VariantClear(&src);
    return code;
}

/*
 *----------------------------------------------------------------------
 * Tocx_VariantToString --
 *
 *      Converts a Variant to a Tcl string.
 *
 * Results:
 *      Standard Tcl result.
 *
 * Side effects:
 *      interp->result is set to the converted string, or the error
 *      value.
 *----------------------------------------------------------------------
 */

int
Tocx_VariantToString(
    Tcl_Interp * interp,        /* Current interpreter. */
    TocxTypeInfo *tiPtr,      /* Type to translate to */
    VARIANT * varPtr)           /* Variant to convert from. */
{
    VARIANT dst;
    int code;
    HRESULT hr;

    if (tiPtr && tiPtr->toVarProc != NULL) {
        code = tiPtr->toStrProc(interp, tiPtr, varPtr);
    } 
	// Disabled by SCK
	/* else if (tiPtr == NULL) {
	Tcl_ResetResult(interp);
	code = TCL_OK;
    } */ 
	else {
        code = TCL_OK;
        VariantInit(&dst);

        switch (varPtr->vt) {
        case VT_EMPTY:
            Tcl_ResetResult(interp);
            break;

        case VT_UI1:
        case VT_I2:
        case VT_I4:
        case VT_R4:
        case VT_R8:
        case VT_BOOL:
        case VT_CY:
        case VT_DATE:
        case VT_BSTR:
            hr = VariantChangeType(&dst, varPtr, 0, VT_BSTR);
            if (FAILED(hr)) {
                Tcl_AppendResult(interp, "Type conversion from \"",
                        Tocx_VTName(varPtr->vt), "\" failed: ",
                        Tocx_ErrorCode(hr), NULL);
                code = TCL_ERROR;
            } else {
                CString str = dst.bstrVal;
                Tcl_SetResult(interp, str.GetBuffer(0), TCL_VOLATILE);
                str.ReleaseBuffer();
            }
            break;

        default:
            Tcl_AppendResult(interp, "Cannot handle return type \"",
                    Tocx_VTName(varPtr->vt), "\"", NULL);
            code = TCL_ERROR;
        }
    }

    VariantClear(&dst);
    return code;
}

