/*
 * tocxUtil.cpp -- 
 * 
 *      Utility functions for the Tocx package.
 *
 * Copyright (c) 1997 Cornell University.
 *
 * See the file "license.terms" for information on usage and redistribution
 * of this file, and for a DISCLAIMER OF ALL WARRANTIES.
 * 
 */

#include "tocx.h"


/*
 *----------------------------------------------------------------------
 * Tocx_ErrorCode --
 *
 *      Returns the OLE error code in a character string.
 *
 * Results:
 *      A static string that represent the error code.
 *
 * Side effects:
 *      None.
 *----------------------------------------------------------------------
 */

char *
Tocx_ErrorCode(
    HRESULT hr)			/* OLE Result code. */
{
    switch (GetScode(hr)) {
    case S_OK:
        return "S_OK";
    case DISP_E_BADPARAMCOUNT:
        return "DISP_E_BADPARAMCOUNT";
    case DISP_E_BADVARTYPE:
        return "DISP_E_BADVARTYPE";
    case DISP_E_EXCEPTION:
        return "DISP_E_EXCEPTION";
    case DISP_E_MEMBERNOTFOUND:
        return "DISP_E_MEMBERNOTFOUND";
    case DISP_E_NONAMEDARGS:
        return "DISP_E_NONAMEDARGS";
    case DISP_E_OVERFLOW:
        return "DISP_E_OVERFLOW";
    case DISP_E_PARAMNOTFOUND:
        return "DISP_E_PARAMNOTFOUND";
    case DISP_E_TYPEMISMATCH:
        return "DISP_E_TYPEMISMATCH";
    case DISP_E_UNKNOWNINTERFACE:
        return "DISP_E_UNKNOWNINTERFACE";
    case DISP_E_UNKNOWNLCID:
        return "DISP_E_UNKNOWNLCID";
    case DISP_E_PARAMNOTOPTIONAL:
        return "DISP_E_PARAMNOTOPTIONAL";
    }

    return "unknown SCODE";
}

/*
 *----------------------------------------------------------------------
 * Tocx_TypeName --
 *
 *      Return string name of a TocxTypeInfo.
 *
 * Results:
 *      Statically allocated string name. Does not need to be freed.
 *
 * Side effects:
 *      None.
 *----------------------------------------------------------------------
 */

char *
Tocx_TypeName(
    TocxTypeInfo *tiPtr)      /* Type information. */
{
    if (tiPtr->name) {
        return tiPtr->name;
    } else {
	return Tocx_VTName(tiPtr->rootvt);
    }
}

/*
 *----------------------------------------------------------------------
 * Tocx_VTName --
 *
 *      Return string name of an VT.
 *
 * Results:
 *      Statically allocated string name. Does not need to be freed.
 *
 * Side effects:
 *      None.
 *----------------------------------------------------------------------
 */

char *
Tocx_VTName(
    VARTYPE vt)     		 /* Type information. */
{
    switch (vt) {
    case VT_EMPTY:
	return "Empty";
    case VT_NULL:
	return "Null";
    case VT_I2:
	return "short";
    case VT_I4:
	return "int";
    case VT_R4:
	return "float";
    case VT_R8:
	return "double";
    case VT_CY:
	return "currency";
    case VT_DATE:
	return "Date";
    case VT_BSTR:
	return "string";
    case VT_DISPATCH:
	return "IDispatch *";
    case VT_ERROR:
	return "error";      
    case VT_BOOL:
	return "bool";
    case VT_VARIANT:
	return "variant";
    case VT_UNKNOWN:
	return "IUnknown *";
    case VT_BYREF:
	return "void *";
    case VT_ARRAY:
	return "array";
    case VT_UI1:
	return "unsigned char";
    case 24:
	return "void";
    case VT_USERDEFINED:
	return "USERDEFINED";
    default:
	return "unrecognized type";
    }
}

/*
 *----------------------------------------------------------------------
 * Tocx_ElemDescName --
 *
 *      Return name of the element type.
 *
 * Results:
 *      String description of the type, in C language syntax.
 *
 * Side effects:
 *      None.
 *----------------------------------------------------------------------
 */

char *
Tocx_ElemDescName(
    ELEMDESC * edescPtr)
{
    switch (edescPtr->tdesc.vt) {
    case VT_EMPTY:
        return "Empty";
    case VT_NULL:
        return "Null";
    case VT_I2:
        return "short";
    case VT_I4:
        return "int";
    case VT_R4:
        return "float";
    case VT_R8:
        return "double";
    case VT_CY:
        return "currency";
    case VT_DATE:
        return "Date";
    case VT_BSTR:
        return "string";
    case VT_DISPATCH:
        return "IDispatch *";
    case VT_ERROR:
        return "error";  
    case VT_BOOL:
        return "bool";
    case VT_VARIANT:
        return "variant";
    case VT_UNKNOWN:
        return "IUnknown *";
    case VT_DECIMAL:
        return "DECIMAL";
    case VT_I1:
        return "char";
    case VT_UI1:
        return "unsigned char";
    case VT_UI2:
        return "unsigned short";
    case VT_UI4:
        return "unsigned int";
    case VT_I8:
        return "long64";
    case VT_UI8:
        return "unsigned long64";
    case VT_INT:
        return "int";
    case VT_UINT:
        return "unsigned int";
    case VT_VOID:
        return "void";
    case VT_HRESULT:
        return "HRESULT";
    case VT_PTR:
        return "PTR";
    case VT_SAFEARRAY:
        return "SAFEARRAY";
    case VT_CARRAY:
        return "CARRAY";
    case VT_USERDEFINED:
        return "USERDEFINED";
    case VT_LPSTR:
        return "LPSTR";
    case VT_LPWSTR:
        return "LPWSTR";
    case VT_FILETIME:
        return "FILETIME";
    case VT_BLOB:
        return "BLOB";
    case VT_STREAM:
        return "STREAM";
    case VT_STORAGE:
        return "STORAGE";
    case VT_STREAMED_OBJECT:
        return "STREAMED_OBJECT";
    case VT_STORED_OBJECT:
        return "STORED_OBJECT";
    case VT_BLOB_OBJECT:
        return "BLOB_OBJECT";
    case VT_CF:
        return "CF";
    case VT_CLSID:
        return "CLSID";
    default:
        return "(unknown type)";
    }

/*
 * The following are BITMASKS and ignored for the time being.
 *
 *   case VT_VECTOR:
 *   case VT_ARRAY:
 *   case VT_BYREF:
 *   case VT_RESERVED:
 *   case VT_ILLEGAL:
 *   case VT_ILLEGALMASKED:
 *   case VT_TYPEMASK:
 *
 */
}
