/*
 * ctocxobj.cpp --
 * 
 *	C++ class implementation of the CtocxObject object.
 *
 * Copyright (c) 1995-1996 Cornell University.
 *
 * See the file "license.terms" for information on usage and redistribution
 * of this file, and for a DISCLAIMER OF ALL WARRANTIES.
 *
 */


#include "stdafx.h"
#include <afxdisp.h>	/* MFC ole automation classes */
#include <olectlid.h>	/* OLE control ID's */
#include <afxpriv.h>	/* Private MFC header. It is needed for symbols
			 * and typedefs used by OnCmdMsg().
			 */
#include "tocxInt.h"
#include "tocxMfc.h"

/*
 * MFC macro to make the CtocxObject class "DYNCREATE".
 */

IMPLEMENT_DYNCREATE(CtocxObject, CWnd)

/*
 * Bit-masks used to determine whether a TypeInfo is the default event source.
 */

#define IMPLTYPE_MASK \
    (IMPLTYPEFLAG_FDEFAULT | IMPLTYPEFLAG_FSOURCE | IMPLTYPEFLAG_FRESTRICTED)

#define DISPATCH_DEF_SRC \
    (IMPLTYPEFLAG_FDEFAULT | IMPLTYPEFLAG_FSOURCE)

#define DISPATCH_DEF \
    (IMPLTYPEFLAG_FDEFAULT)


/*
 *----------------------------------------------------------------------
 * CtocxObject::GetClassInfo --
 *
 *	Returns the ITypeInfo interface that described the COCLASS 
 *	of this control. We need this to find out the ITypeInfo of
 *	the dispinterface of this class.
 *
 * Results:
 *	HRESULT.
 *
 * Side effects:
 *	AddRef() to the ITypeInfo returned. Caller should call Release().
 *----------------------------------------------------------------------
 */

HRESULT
CtocxObject::GetClassInfo(
    ITypeInfo **classInfo)	/* Returns the ITypeInfo of the coclass. */
{
    LPUNKNOWN pUnknown;
    LPPROVIDECLASSINFO pPCI = NULL;
    LPTYPEINFO pClassInfo = NULL;
    HRESULT hr= ResultFromScode(S_OK);

    *classInfo = NULL;

    pUnknown = GetControlUnknown();
    ASSERT(pUnknown != NULL);

    /*
     * Get the IProvideClassInfo interface, which knows about the
     * default TypeInfo of the OCX.
     */
    if (FAILED(hr = pUnknown->QueryInterface(IID_IProvideClassInfo,
	   (LPVOID*)&pPCI))) {
	goto done;
    }
    ASSERT(pPCI != NULL);

    // Get the default typeinfo of the OCX.
    if (FAILED(hr = pPCI->GetClassInfo(&pClassInfo))) {
	goto done;
    }
    ASSERT(pClassInfo != NULL);

    *classInfo = pClassInfo;

done:
    return hr;
}

/*
 *----------------------------------------------------------------------
 * CtocxObject::GetDispatchInfo --
 *
 *	Returns the ITypeInfo of a dispinterface of a coclass. The
 *	type dispinterface should satisfy the "mask" argument.
 *
 *	NOTE: In some cases the dispinterface CANNOT be queried with
 *	m_pdisp->GetTypeInfo(); that may return a ITypeInfo of the
 *	*interface* instead of a *dispinterface* of the coclass, if
 *	that class supports dual interface. (I am not sure why this
 *	can happen; might be an tocx/MFC bug.)
 *
 * Results:
 *	HRESULT.
 *
 * Side effects:
 *	AddRef() to the ITypeInfo returned. Caller should call Release().
 *----------------------------------------------------------------------
 */

HRESULT
CtocxObject::GetDispatchInfo(
    ITypeInfo *pClassInfo,	/* ITypeInfo of the coclass. */
    ITypeInfo **typeInfo,	/* Returns ITypeInfo of the dispinterface. */
    int mask)			/* Specify which dispinterface. */
{
    LPTYPEATTR pClassAttr = NULL;
    LPTYPEINFO pTypeInfo = NULL;
    HRESULT hr = ResultFromScode(S_OK);
    int i;

    *typeInfo = NULL;

    if (FAILED(hr = pClassInfo->GetTypeAttr(&pClassAttr))) {
	goto done;
    }
    ASSERT(pClassAttr != NULL);
    ASSERT(pClassAttr->typekind == TKIND_COCLASS);

    /*
     * Search for typeinfo that satisfies the mask.
     */
    for (i = 0; i < pClassAttr->cImplTypes; i++) {
	int nFlags;
	HREFTYPE hRefType;

	if (FAILED(pClassInfo->GetImplTypeFlags(i, &nFlags))) {
	    continue;
	}
	if ((nFlags & IMPLTYPE_MASK) != mask) {
	    continue;
	}

	/*
	 * Found it.  Get its ITypeInfo pointer.
	 */
	if (SUCCEEDED(pClassInfo->GetRefTypeOfImplType(i,
		&hRefType)) &&
	    SUCCEEDED(pClassInfo->GetRefTypeInfo(hRefType,
		&pTypeInfo))) {

	    LPTYPEATTR pTypeAttr = NULL;
	    ASSERT(pTypeInfo != NULL);

	    if (FAILED(hr = pTypeInfo->GetTypeAttr(&pTypeAttr))) {
		goto done;
	    }
	    ASSERT(pTypeAttr);
	    if (pTypeAttr->typekind == TKIND_DISPATCH) {
		pTypeInfo->ReleaseTypeAttr(pTypeAttr);
		*typeInfo = pTypeInfo;

		goto done;
	    } else {
		pTypeInfo->ReleaseTypeAttr(pTypeAttr);
	    }
	}
    }

done:
    pClassInfo->ReleaseTypeAttr(pClassAttr);
    return hr;
}

/*
 *----------------------------------------------------------------------
 * CtocxObject::UpdateInfoTable --
 *
 *	Gets information about the properties, methods and events of
 *	an OCX.
 *
 * Results:
 *	Standard TCL resule.
 *
 * Side effects:
 *	When successful, the func, prop and event lists are initialized.
 *----------------------------------------------------------------------
 */

int
CtocxObject::UpdateInfoTable()
{
    LPUNKNOWN pUnknown;
    ITypeInfo *classInfo = NULL;/* Info about the coclass. */
    ITypeInfo *dispInfo	 = NULL;/* Info about the default dispinterface. */
    ITypeInfo *eventInfo = NULL;/* Info about the event dispinterface. */
    HRESULT hr =  ResultFromScode(S_OK);
    LPTYPEATTR pDispAttr = NULL;
    LPTYPEATTR pEventAttr = NULL;
    BSTR bstrName;
    int i;
    int indexed;

    /*
     * Get the IDispatch interface of the control. This interface will be
     * used to invoke methods and properties of the control.
     */
    pUnknown = GetControlUnknown();
    // ASSERT(pUnknown != NULL);
	if (pUnknown == NULL) return TCL_ERROR;

    if (FAILED(hr = pUnknown->QueryInterface(IID_IDispatch,
	    (void **)&m_pdisp))) {
	goto done;
    }

    /*
     * Get the typeinfo's of the control; from these we can determine the
     * methods, events and properties supported by the control.
     */

    if (FAILED(hr = GetClassInfo(&classInfo))) {
	goto done;
    }
    if (FAILED(hr = GetDispatchInfo(classInfo, &dispInfo, DISPATCH_DEF))) {
	goto done;
    }
    if (FAILED(hr = GetDispatchInfo(classInfo, &eventInfo,DISPATCH_DEF_SRC))){
	goto done;
    }

    /*
     * Handle the property and methods.
     */
    if (dispInfo != NULL) {
	if (FAILED(hr = dispInfo->GetTypeAttr(&pDispAttr))) {
	    goto done;
	}
	ASSERT(pDispAttr);
#if 0
	/*
	 * #if0
	 * We don't need the containing type lib for the time being ...
	 */
	unsigned int tindex;
	if (FAILED(hr = dispInfo->GetContainingTypeLib(&ptlib, &tindex))) {
	    goto done;
	}
	ASSERT(ptlib);

	ptlib->Release();
#endif

	/*
	 * Parse the func'. Initialize the methods according to the 
	 */
	for (i = 0; i<pDispAttr->cFuncs; i++) {
	    FUNCDESC * desc;
	    TocxMethodInfo *infoPtr;

	    if (FAILED(hr = dispInfo->GetFuncDesc(i, &desc))) {
		goto done;
	    }

	    if (desc->invkind == INVOKE_FUNC) {
		/*
		 * This is a method. Add it to the method list.
		 */

		if (pDispAttr->wTypeFlags & TYPEFLAG_FDUAL) {
		    if (desc->oVft < 28) {
			/*
			 * We don't expose the first seven functions in
			 * a dual interface to the Tcl script because of
			 * potential dangers. These are:
			 *	IUnknown::QueryInterface
			 *	IUnknown::AddRef
			 *	IUnknown::Release
			 *	IDispatch::GetTypeInfoCount
			 *	IDispatch::GetTypeInfo
			 *	IDispatch::GetIDsOfNames
			 *	IDispatch::Invoke
			 */
			dispInfo->ReleaseFuncDesc(desc);
			continue;
		    }
		}

		if (FAILED(hr = dispInfo->GetDocumentation(desc->memid,
			&bstrName, NULL, NULL, NULL))) {
		    dispInfo->ReleaseFuncDesc(desc);
		    goto done;
		}

		infoPtr = new TocxMethodInfo;
		infoPtr->name = strdup(CString(bstrName));
		infoPtr->info = MakeFuncInfo(dispInfo, desc);

		if (funcHead == NULL) {
		    funcHead = infoPtr;
		    funcTail = infoPtr;
		} else {
		    funcTail->next = infoPtr;
		    funcTail = infoPtr;
		}
		infoPtr->next = NULL;

		SysFreeString(bstrName);
	    } else {
		/*
		 * This is a property get/set function. We use info about
		 * this function to discover the ELEMDESC of the
		 * property accessed by this function.
		 */
		indexed = 0;

		switch (desc->invkind) {
		case INVOKE_PROPERTYPUT:
		    if (desc->cParams > 1) {
			indexed = 1;
		    }
		    break;

		case INVOKE_PROPERTYPUTREF:
		    if (desc->cParams > 1) {
			indexed = 1;
		    }
		    break;

		case INVOKE_PROPERTYGET:
		    if (desc->cParams > 0) {
			indexed = 1;
		    }
		    break;

		default:
		    ASSERT(0);
		}

		if (FAILED(hr = dispInfo->GetDocumentation(desc->memid,
			&bstrName, NULL, NULL, NULL))) {
		    dispInfo->ReleaseFuncDesc(desc);
		    goto done;
		}

		AddProperty(dispInfo, bstrName, desc->memid, NULL, desc,
		    indexed);
		SysFreeString(bstrName);
	    }
	}

	for (i = 0;  i<pDispAttr->cVars; i++) {
	    VARDESC * desc;

	    if (FAILED(dispInfo->GetVarDesc(i, &desc))) {
		break;
	    }

	    if (FAILED(dispInfo->GetDocumentation(desc->memid, &bstrName,
		   NULL, NULL, NULL))) {
		continue;
	    }

	    AddProperty(dispInfo, bstrName, desc->memid, desc, NULL, 0);
	    SysFreeString(bstrName);
	}
    }

    /*
     * Handle the events.
     */

    if (eventInfo != NULL) {
	if (FAILED(hr = eventInfo->GetTypeAttr(&pEventAttr))) {
	    goto done;
	}
	ASSERT(pEventAttr);

	/*
	 * Parse the func'. Initialize the methods according to the 
	 */
	for (i = 0; i<pEventAttr->cFuncs; i++) {
	    FUNCDESC * desc;
	    TocxEventInfo *infoPtr;

	    if (FAILED(hr = eventInfo->GetFuncDesc(i, &desc))) {
		goto done;
	    }

	    if (desc->invkind == INVOKE_FUNC) {
		if (FAILED(hr = eventInfo->GetDocumentation(desc->memid,
			&bstrName, NULL, NULL, NULL))) {
		    eventInfo->ReleaseFuncDesc(desc);
		    goto done;
		}

		infoPtr = new TocxEventInfo;
		infoPtr->name = _strdup(CString(bstrName));
		infoPtr->desc = desc;

		if (eventHead == NULL) {
		    eventHead = infoPtr;
		    eventTail = infoPtr;
		} else {
		    eventTail->next = infoPtr;
		    eventTail = infoPtr;
		}
		infoPtr->next = NULL;

		SysFreeString(bstrName);
	    }
	}
    }

done:
    if (classInfo != NULL) {
	classInfo->Release();
    }
    if (dispInfo != NULL) {
	if (pDispAttr != NULL) {
	    dispInfo->ReleaseTypeAttr(pDispAttr);
	}
	dispInfo->Release();
    }
    if (eventInfo != NULL) {
	if (pEventAttr != NULL) {
	    eventInfo->ReleaseTypeAttr(pEventAttr);
	}
	eventInfo->Release();
    }

    if (SUCCEEDED(hr)) {
	return TCL_OK;
    } else {
	return TCL_ERROR;
    }
}

/*
 *----------------------------------------------------------------------
 * CtocxObject::AddProperty --
 *
 *	Adds a property to the list of properties of this control.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	Property list is changed.
 *----------------------------------------------------------------------
 */

void
CtocxObject::AddProperty(
    LPTYPEINFO pTypeInfo,	/* Type infomation of this coclass. */
    BSTR bstrName,		/* Name of the property. */
    MEMBERID memid,		/* MEMID of the property. */
    VARDESC *vDesc,		/* VARDESC of the property. */
    FUNCDESC *fDesc,		/* FUNCDESC of a property accessor func. */
    int indexed)		/* Is this an indexed property. */
{
    TocxPropertyInfo *infoPtr;

    CString cstr = bstrName;
    char * name = cstr.GetBuffer(0);

    ASSERT(!( (vDesc != NULL) && (fDesc != NULL) ));
    ASSERT(!( (vDesc == NULL) && (fDesc == NULL) ));

    for (infoPtr=propHead; infoPtr; infoPtr=infoPtr->next) {
	if (stricmp(name, infoPtr->name) ==0) {
	    break;
	}
    }
    
    if (infoPtr == NULL) {
	infoPtr = (TocxPropertyInfo *)ckalloc(sizeof(TocxPropertyInfo));
	infoPtr->name = _strdup(name);
	infoPtr->memid = memid;
	infoPtr->typeInfoPtr = NULL;
	infoPtr->funcGet = NULL;
	infoPtr->funcPut = NULL;
	infoPtr->funcPutRef = NULL;
	infoPtr->indexed = indexed;
    } else {
#if 0
	/*
	 * The following code checks whether the disp members are
	 * redefined -- which usually won't happen for "well-behaved"
	 * OCX'es.
	 *
	 * #if0
	 * This code is currently incompleted and is commented out
	 */

	if (infoPtr->memid != memid) {
	    error "memid not matched";
	}
	if (fDesc) {
	    switch (fDesc->invkind) {
	    case INVOKE_PROPERTYPUT:
		if (infoPtr->fDescPut) {
		    error "fDescPut redefined";
		}
		break;
	    case INVOKE_PROPERTYPUTREF:
		if (infoPtr->fDescPutRef) {
		    error "fDescPutRef redefined";
		}
		break;
	    case INVOKE_PROPERTYGET:
		if (infoPtr->fDescGet) {
		    error "fDescGet redefined";
		}
		break;
	    }
	}
	if (infoPtr->indexed != indexed) {
	    error "indexed not matched";
	}
#endif
    }

    if (vDesc) {
	/*
	 * This is a normal property
	 */
	infoPtr->typeInfoPtr = InterpretTypeDesc(pTypeInfo, 
		&vDesc->elemdescVar.tdesc);
    } else {
	/*
	 * This is a property accessed by a function.
	 */
	switch (fDesc->invkind) {
	case INVOKE_PROPERTYPUT:
	    infoPtr->funcPut = MakeFuncInfo(pTypeInfo, fDesc);
	    break;
	case INVOKE_PROPERTYPUTREF:
	    infoPtr->funcPutRef = MakeFuncInfo(pTypeInfo, fDesc);
	    break;
	case INVOKE_PROPERTYGET:
	    infoPtr->funcGet = MakeFuncInfo(pTypeInfo, fDesc);
	    break;
	}
    }

    if (propHead == NULL) {
	propHead = infoPtr;
	propTail = infoPtr;
    } else {
	propTail->next = infoPtr;
	propTail = infoPtr;
    }
    infoPtr->next = NULL;

    cstr.ReleaseBuffer();
}

/*
 *----------------------------------------------------------------------
 * CtocxObject::InterpretTypeDesc --
 *
 *	Stores the typeinfo in a format that's easily accessible from
 *	Tcl.
 *
 * Results:
 *	    - All VT_USERDEFINE types are deferenced.
 *	    - VT_PTR and VT_SAFEARRAY types are interpreted.
 *	    - Stock properties are handled by special translation functions.
 *
 *	Exactly one TocxTypeInfo structure is maintained for each
 *	VARDESC of an OCX class.
 *
 * Side effects:
 *	New TocxTypeInfo is added to the object's typeInfoList.
 *----------------------------------------------------------------------
 */

TocxTypeInfo *
CtocxObject::InterpretTypeDesc(
    LPTYPEINFO pTypeInfo,	/* TYPEINFO of this coclass. */
    TYPEDESC * tdesc)		/* Info about the variable. */
{
    TocxTypeInfo * tiPtr;

#if 0
    /*
     * It may be unsafe to cache tdesc because it may refer to different
     * types ....
     */
    for (tiPtr = typeInfoList; tiPtr; tiPtr = tiPtr->next) {
	if (tiPtr->tdesc == tdesc) {
	    return tiPtr;
	}
    }
#endif

    tiPtr = (TocxTypeInfo *)ckalloc(sizeof(TocxTypeInfo));
    tiPtr->tdesc = tdesc;
    tiPtr->rootvt = VT_ILLEGAL;
    tiPtr->toVarProc = NULL;
    tiPtr->toStrProc = NULL;
    tiPtr->name = NULL;
    tiPtr->pointerCount = 0;
    tiPtr->numEnums = 0;
    tiPtr->enums = 0;

    InterpretTypeDescHelper(pTypeInfo, tdesc, 0, tiPtr);

    tiPtr->next = typeInfoList;
    typeInfoList = tiPtr;

    return tiPtr;
}

/*
 *----------------------------------------------------------------------
 * CtocxObject::InterpretTypeDescHelper --
 *
 *	Copies extra info into the TocxTypeInfo structure if possible.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	tiPtr is changed when more info can be found about this type.
 *----------------------------------------------------------------------
 */

void
CtocxObject::InterpretTypeDescHelper(
    LPTYPEINFO pTypeInfo,	/* TYPEINFO of this coclass. */
    TYPEDESC* tdesc,		/* TYPEDESC of type to interpret. */
    int aliasCount,		/* Count of nested levels of TKIND_ALIAS. */
    TocxTypeInfo * tiPtr)	/* [out] Stores info about this type. */
{
    LPTYPEINFO pTypeInfoRef = NULL;

    switch (tdesc->vt) {
    case VT_USERDEFINED:
	/*
	 * It's an alias: Expand the alias.
	 */
	if (SUCCEEDED(pTypeInfo->GetRefTypeInfo(tdesc->hreftype,
	        &pTypeInfoRef))) {

	    ASSERT(pTypeInfoRef != NULL);
	    LPTYPEATTR pTypeAttr = NULL;

	    if (tiPtr->name == NULL) {
		BSTR bstrName = NULL;
		BSTR bstrDoc  = NULL;
		BSTR bstrHelp = NULL;
		DWORD dwHelpID;

		if (SUCCEEDED(pTypeInfoRef->GetDocumentation(MEMBERID_NIL,
		        &bstrName, &bstrDoc, &dwHelpID, &bstrHelp))) {
		    tiPtr->name = _strdup(CString(bstrName));

		    SysFreeString(bstrName);
		    SysFreeString(bstrDoc);
		    SysFreeString(bstrHelp);
		}
	    }

	    if (SUCCEEDED(pTypeInfoRef->GetTypeAttr(&pTypeAttr))) {
		ASSERT(pTypeAttr != NULL);

		switch (pTypeAttr->typekind) {
		case TKIND_ALIAS:
		    /*
		     * Type expanded to another alias!
		     */
		    InterpretTypeDescHelper(pTypeInfoRef,
			    &pTypeAttr->tdescAlias, aliasCount+1, tiPtr);
		    if (aliasCount == 0 && tiPtr->toStrProc == NULL) {
			/*
			 * This check is important. For example, if
			 * 	A is an alias of B
			 *	B is an alias of C
			 *	C is an alias of VT_I4
			 * We want to used the converter for type A.
			 */
			MakeAliasConverter(pTypeInfoRef, pTypeAttr, tiPtr);
		    }
		    break;

		case TKIND_ENUM:
		    MakeEnumConverter(pTypeInfoRef, pTypeAttr, tiPtr);
		    break;

		case TKIND_DISPATCH:
		    tiPtr->rootvt = VT_DISPATCH;
		    MakeDispatchConverter(pTypeInfoRef, pTypeAttr, tiPtr);
		    break;
		}
		pTypeInfoRef->ReleaseTypeAttr(pTypeAttr);
	    }	

	    pTypeInfoRef->Release();
	}
	break;

    case VT_PTR:
        /*
	 * It's a pointer: Dereference and try to interpret with one less
	 * level of indirection.
	 *
	 * [BUG] The ++ is not correct if the ptr points to an alias which
	 * is a pointer to something else. This may not be very common,
	 * though ....
	 */
        ASSERT(tdesc->lptdesc != NULL);
	++ tiPtr->pointerCount;
        InterpretTypeDescHelper(pTypeInfo, tdesc->lptdesc, aliasCount, tiPtr);
	break;

    default:
	tiPtr->rootvt = tdesc->vt;
    }
}

/*
 *----------------------------------------------------------------------
 * CtocxObject::MakeAliasConverter --
 *
 *	Hooks up a variable type with the alias converter.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	tiPtr->toVarProc and toStrProc are changed.
 *----------------------------------------------------------------------
 */

void
CtocxObject::MakeAliasConverter(
    LPTYPEINFO pTypeInfo,	/* TYPEINFO of this coclass. */
    LPTYPEATTR pTypeAttr,	/* TYPEATTR of the type. */
    TocxTypeInfo * tiPtr)	/* [out] Stores info about this type. */

{
    if (tiPtr->toStrProc != NULL) {
	return;
    }

    if (pTypeAttr->guid == GUID_COLOR) {
	tiPtr->toVarProc = Tocx_StringToVariant_Color;
	tiPtr->toStrProc = Tocx_VariantToString_Color;
    }
}

/*
 *----------------------------------------------------------------------
 * CtocxObject::MakeDispatchConverter --
 *
 *	Hooks up a variable type with the IDispatch converter.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	tiPtr->toVarProc and toStrProc are changed.
 *----------------------------------------------------------------------
 */

void
CtocxObject::MakeDispatchConverter(
    LPTYPEINFO pTypeInfo,	/* TYPEINFO of this coclass. */
    LPTYPEATTR pTypeAttr,	/* TYPEATTR of the type. */
    TocxTypeInfo * tiPtr)	/* [out] Stores info about this type. */

{
    if (tiPtr->toStrProc != NULL) {
	return;
    }
}

/*
 *----------------------------------------------------------------------
 * CtocxObject::MakeEnumConverter --
 *
 *	Hooks up a variable type with the enumeration converter.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	tiPtr->toVarProc and toStrProc are changed.
 *----------------------------------------------------------------------
 */

void
CtocxObject::MakeEnumConverter(
    LPTYPEINFO pTypeInfo,	/* TYPEINFO of this coclass. */
    LPTYPEATTR pTypeAttr,	/* TYPEATTR of the type. */
    TocxTypeInfo * tiPtr)	/* [out] Stores info about this type. */

{
    int i;

    if (tiPtr->toStrProc != NULL) {
	return;
    }

    tiPtr->toVarProc = Tocx_StringToVariant_Enum;
    tiPtr->toStrProc = Tocx_VariantToString_Enum;
    tiPtr->numEnums = pTypeAttr->cVars;
    tiPtr->enums = (TocxEnum*)ckalloc(sizeof(TocxEnum) * tiPtr->numEnums);

    for (i=0; i<tiPtr->numEnums; i++) {
	VARDESC * desc = NULL;
	BSTR bstrName;

	if (FAILED(pTypeInfo->GetVarDesc(i, &desc))) {
	    goto failed;
	}

	if (FAILED(pTypeInfo->GetDocumentation(desc->memid, &bstrName,
	        NULL, NULL, NULL))) {
	    goto failed;
	}
	if (desc->varkind != VAR_CONST) {
	    goto failed;
	}

	tiPtr->enums[i].name  = _strdup(CString(bstrName));
	tiPtr->enums[i].value = *desc->lpvarValue;
	SysFreeString(bstrName);

	if (tiPtr->rootvt == VT_ILLEGAL) {
	    tiPtr->rootvt = desc->elemdescVar.tdesc.vt;
	}

	pTypeInfo->ReleaseVarDesc(desc);
	continue;

      failed:
	if (desc) {
	    pTypeInfo->ReleaseVarDesc(desc);
	}
	tiPtr->enums[i].name = NULL;
    }
}

/*
 *----------------------------------------------------------------------
 * CtocxObject::MakeFuncInfo --
 *
 *	Creates a TocxFuncInfo data structure for a member function.
 *
 * Results:
 *	The pointer to the TocxFuncInfo structure.
 *
 * Side effects:
 *	Fields are initialized.
 *----------------------------------------------------------------------
 */

TocxFuncInfo *
CtocxObject::MakeFuncInfo(
    LPTYPEINFO pTypeInfo,	/* TYPEINFO of this coclass. */
    FUNCDESC * desc)		/* FUNCDESC of the method. */
{
    int i;
    int num = desc->cParams + desc->cParamsOpt;
    TocxFuncInfo *infoPtr = (TocxFuncInfo *)
	   ckalloc(sizeof(TocxFuncInfo));

    infoPtr->desc = desc;
    infoPtr->type = InterpretTypeDesc(pTypeInfo, &desc->elemdescFunc.tdesc);
    infoPtr->argTypes = (TocxTypeInfo **)ckalloc(sizeof(TocxTypeInfo*));

    for (i=0; i<num; i++) {
	infoPtr->argTypes[i] = InterpretTypeDesc(pTypeInfo, 
	        &desc->lprgelemdescParam[i].tdesc);
    }

    return infoPtr;
}

/*
 *----------------------------------------------------------------------
 * CtocxObject::OnCmdMsg --
 *
 *	Handles events from the OCX. If an event handler has been
 *	installed for a certain event, fires it up.
 *
 * Results:
 *	True iff the event has been handled.
 *
 * Side effects:
 *	None.
 *----------------------------------------------------------------------
 */

BOOL
CtocxObject::OnCmdMsg(
    UINT nID,
    int nCode,
    void* pExtra,
    AFX_CMDHANDLERINFO* pHandlerInfo)
{
    AFX_EVENT *pEvent;
    TocxEventBinding * bindPtr;
    DISPPARAMS * params;
    int code;

    if (pHandlerInfo != NULL) {
	goto use_default;
    }

    if (nCode == CN_EVENT) {
	/*
	 * The information about the event is stored in the AFX_EVENT
	 * structure. This structure is declared in mfc/AfxPriv2.h as of
	 * MFC 4.2.
	 */
	pEvent = (AFX_EVENT*)pExtra;
	if (pEvent == NULL) {
	    goto use_default;
	}

	switch (pEvent->m_eventKind) {
	case AFX_EVENT::event:
	    for (bindPtr=bindList; bindPtr; bindPtr=bindPtr->next) {
		if (bindPtr->memid == pEvent->m_dispid) {
		    break;
		}
	    }

	    if (bindPtr == NULL) {
		/*
		 * This event is not bound to Tcl script.
		 */
		goto use_default;
	    }

	    params = pEvent->m_pDispParams;
	    code = TCL_OK;

	    if (params->cArgs == 0) {
		code = Tcl_Eval(interp, bindPtr->cmdPrefix);
	    } else {
		/*
		 * Translate additional event parameters to Tcl strings.
		 */
		Tcl_DString dstring;
		int i;
		
		Tcl_DStringInit(&dstring);
		Tcl_DStringAppend(&dstring, bindPtr->cmdPrefix, -1);
		for (i=params->cArgs-1; i>=0; i--) {
		    Tcl_ResetResult(interp);
		    if (Tocx_VariantToString(interp,NULL,&params->rgvarg[i])
			    != TCL_OK) {
			code = TCL_ERROR;
			goto done;
		    }
		    Tcl_DStringAppendElement(&dstring, interp->result);
		}
		
		code = Tcl_Eval(interp, dstring.string);
	      done:
		Tcl_DStringFree(&dstring);
	    }

	    if (code != TCL_OK) {
		/*
		 * ToDo: error reporting back to the Control.
		 */
		Tcl_AddErrorInfo(interp,
			"\n    (event binding executed by Tocx)");
		Tcl_BackgroundError(interp);
	    }
	    Tcl_ResetResult(interp);
	    return 1;

	case AFX_EVENT::propChanged:
	case AFX_EVENT::propRequest:
	case AFX_EVENT::propDSCNotify:
	default:
	    /*
	     * Not handled yet
	     */
	    break;
	}
    }

  use_default:
    return CWnd::OnCmdMsg(nID, nCode, pExtra, pHandlerInfo);
}
