/*
 * OleConv.c --
 * 
 *	Type converter for the Tocx package.
 *
 * Copyright (c) 1995-1996 Cornell University.
 *
 * See the file "license.terms" for information on usage and redistribution
 * of this file, and for a DISCLAIMER OF ALL WARRANTIES.
 *
 */

#include "stdafx.h"
#include "tocx.h"


/*
 *----------------------------------------------------------------------
 * Tocx_StringToVariant_Enum --
 *
 *	Converts a Tcl string to an OLE Variant that represents an
 *	OLE_COLOR.
 *
 * Results:
 *	Standard Tcl result.
 *
 * Side effects:
 *	None.
 *----------------------------------------------------------------------
 */

int
Tocx_StringToVariant_Color(
    Tcl_Interp * interp,	/* Current interpreter. */
    char * string,		/* String to convert from. */
    TocxTypeInfo *tiPtr,	/* Type information. */
    VARIANTARG * vargPtr)	/* [out] VARIANT to convert to. */
{
    Tk_Window tkwin = Tk_MainWindow(interp);
    XColor * colorPtr;
    COLORREF rgb;

    colorPtr = Tk_GetColor(interp, tkwin, string);
    if (colorPtr != NULL) {
	rgb = RGB(colorPtr->red/256, colorPtr->green/256, colorPtr->blue/256);
	Tk_FreeColor(colorPtr);
    }
    else {
	int v;
	if (Tcl_GetInt(interp, string, &v) != TCL_OK) {
	    Tcl_ResetResult(interp);
	    Tcl_AppendResult(interp, "unknown color value \"", string, "\"",
		    NULL);
	    return TCL_ERROR;
	}
	rgb = (COLORREF)v;
    }

    vargPtr->vt    = VT_UI4;
    vargPtr->ulVal = (ULONG) rgb;
    return TCL_OK;
}

/*
 *----------------------------------------------------------------------
 * Tocx_VariantToString_Enum --
 *
 *	Converts an OLE Variant that represents an OLE_COLOR to a Tcl
 *	string.
 *
 * Results:
 *	Standard Tcl result.
 *
 * Side effects:
 *	None.
 *----------------------------------------------------------------------
 */

int
Tocx_VariantToString_Color(
    Tcl_Interp * interp,	/* Current interpreter. */
    TocxTypeInfo *tiPtr,	/* Type information. */
    VARIANT *varPtr)		/* VARIANT to convert from. */
{
    Tk_Window tkwin = Tk_MainWindow(interp);
    XColor * colorPtr;
    XColor colorVal;
    COLORREF rgb;

    switch (varPtr->vt) {
    case VT_UI4:
    case VT_I4:
	rgb = (COLORREF)varPtr->ulVal;

	colorVal.red   = GetRValue(rgb)  * 256;
	colorVal.green = GetGValue(rgb) * 256;
	colorVal.blue  = GetBValue(rgb) * 256;

	colorPtr = Tk_GetColorByValue(tkwin, &colorVal);
	Tcl_AppendResult(interp, Tk_NameOfColor(colorPtr), NULL);
	Tk_FreeColor(colorPtr);

	break;
    default:
	Tcl_AppendResult(interp, "unknown color", NULL);
    }

    return TCL_OK;
}

/*
 *----------------------------------------------------------------------
 * Tocx_StringToVariant_Enum --
 *
 *	Converts an Enum string to an OLE Variant.
 *
 * Results:
 *	Standard Tcl result.
 *
 * Side effects:
 *	None.
 *----------------------------------------------------------------------
 */

int
Tocx_StringToVariant_Enum(
    Tcl_Interp * interp,	/* Current interpreter. */
    char * string,		/* String to convert from. */
    TocxTypeInfo *tiPtr,	/* Type information. */
    VARIANTARG * vargPtr)	/* [out] VARIANT to convert to. */
{
    int i;

    for (i=0; i<tiPtr->numEnums; i++) {
	if (tiPtr->enums[i].name != NULL) {
	    if (strcmp(string, tiPtr->enums[i].name) == 0) {
		*vargPtr = tiPtr->enums[i].value;
		return TCL_OK;
	    }
	}
    }
    Tcl_ResetResult(interp);
    Tcl_AppendResult(interp, "unknown value \"", string, "\"",
	NULL);
    if (tiPtr->rootvt != VT_ILLEGAL) {
	char * sep = "; must be ";

	for (i=0; i<tiPtr->numEnums; i++) {
	    if (tiPtr->enums[i].name != NULL) {
		Tcl_AppendResult(interp, sep, tiPtr->enums[i].name, NULL);
		if (i == (tiPtr->numEnums - 2)) {
		    sep = " or ";
		} else {
		    sep = ", ";
		}
	    }
	}
    }

    return TCL_ERROR;
}

/*
 *----------------------------------------------------------------------
 * Tocx_VariantToString_Enum --
 *
 *	Converts a OLE Variant to an Enum string.
 *
 * Results:
 *	Standard Tcl result.
 *
 * Side effects:
 *	None.
 *----------------------------------------------------------------------
 */

int
Tocx_VariantToString_Enum(
    Tcl_Interp * interp,	/* Current interpreter. */
    TocxTypeInfo *tiPtr,	/* Type information. */
    VARIANT *varPtr)		/* VARIANT to convert from. */
{
    int i;
    VARIANT v;
    HRESULT hr;

    /*
     * (ToDo) Variable equality is not implemented for many types (Currency,
     * etc)
     */

    for (i=0; i < tiPtr->numEnums; i++) {
	VariantInit(&v);
	hr = VariantChangeType(&v, varPtr, 0, tiPtr->enums[i].value.vt);
	if (FAILED(hr)) {
	    continue;
	}
	switch (tiPtr->enums[i].value.vt) {
	case VT_BOOL:
	    if (tiPtr->enums[i].value.boolVal == v.boolVal) {
		goto matched;
	    }
	    break;
	case VT_I1:
	case VT_UI1:
	    if (tiPtr->enums[i].value.bVal == v.bVal) {
		goto matched;
	    }
	    break;
	case VT_I2:
	case VT_UI2:
	    if (tiPtr->enums[i].value.iVal == v.iVal) {
		goto matched;
	    }
	    break;
	case VT_I4:
	case VT_UI4:
	    if (tiPtr->enums[i].value.lVal == v.lVal) {
		goto matched;
	    }
	    break;
	case VT_INT:
	case VT_UINT:
	    if (tiPtr->enums[i].value.lVal == v.lVal) {
		goto matched;
	    }
	    break;
	case VT_I8:
	case VT_UI8:
	    if (tiPtr->enums[i].value.lVal == v.lVal) {
		goto matched;
	    }
	    break;
	case VT_R4:
	    if (tiPtr->enums[i].value.fltVal == v.fltVal) {
		goto matched;
	    }
	    break;
	case VT_R8:
	    if (tiPtr->enums[i].value.dblVal == v.dblVal) {
		goto matched;
	    }
	    break;
	case VT_CY:
	    break;
	case VT_DATE:
	    if (tiPtr->enums[i].value.date == v.date) {
		goto matched;
	    }
	    break;
	case VT_BSTR:
	    break;
	case VT_DECIMAL:
	    break;
	}

	VariantClear(&v);
	continue;

      matched:
	VariantClear(&v);
	if (tiPtr->enums[i].name != NULL) {
	    Tcl_AppendResult(interp, tiPtr->enums[i].name, NULL);
	    return TCL_OK;
	}
    }

    Tcl_AppendResult(interp, "??", NULL);
    return TCL_OK;
}
