#!/usr/local/bin/wish -f
# spam.tcl --
#
#	Brian Smith's package manager.  This file manages packages for
#	Tcl/Tk

# 	Juliean Galak added an undo feature and hierarchial packaging

# Notes about the organization of this file:
#   Major sections are delimited by a string of "-" characters -- e.g. "-------"
#   At the beginning of each major section, the purpose and routines in that
#   section are described.
#

# -----------------------------------------------------------------------
#
# Global Variables
#
# This section describes and initializes the globals used in SPAM.
# All global variables are stored in the spamGlobals array.

# The following variables are used for debugging:
set spamGlobals(Debug)		1;
set spamGlobals(DebugLevel)	5;
set spamGlobals(DebugProc)	Global;
set spamGlobals(DebugStack)	Global;

# The following variables contain information used to undo
set spamGlobals(UndoStack)	{};
set spamGlobals(UndoLength)	1;

# The following variables contain information about the packages
set spamGlobals(currPackage)	{};	# The name of the currently selected package
set spamGlobals(installed)	{};	# A list of currently installed packages
					# Set in SpamGetPackages during SpamInit
# Notes about packages:
#   If pkg is the package name (e.g., Tcl7.5""), then:
#     spamGlobals(<pkg>,name) holds the actual package name (e.g., "Dp/Core")
#     spamGlobals(<pkg>,version) holds the package version (e.g., "1.0")
#     spamGlobals(<pkg>,readme) holds the name of the readme file
#     spamGlobals(<pkg>,requires) is a list of required packages
#     spamGlobals(<pkg>,url) holds the url of the source site (for updates)
#     spamGlobals(<pkg>,objFiles) lists the object files (shared libs/DLLs)
#     spamGlobals(<pkg>,tclFiles) lists the Tcl source files
#     spamGlobals(<pkg>,dataFiles) lists the data files used by the package

#this is a list of all the packages
set spamGlobals(rootList) {};

set ver [info tclversion]
lappend spamGlobals(installed) Tcl$ver
set spamGlobals(Tcl$ver,name)		Tcl
set spamGlobals(Tcl$ver,version)	$ver
set spamGlobals(Tcl$ver,readme)		{}
set spamGlobals(Tcl$ver,requires)	{}
set spamGlobals(Tcl$ver,url)		{}
set spamGlobals(Tcl$ver,objFiles)	{}
set spamGlobals(Tcl$ver,tclFiles)	{}
set spamGlobals(Tcl$ver,dataFiles)	{}

set ver $tk_version
lappend spamGlobals(installed) Tk$ver
set spamGlobals(Tk$ver,name)	Tk
set spamGlobals(Tk$ver,version)	$ver
set spamGlobals(Tk$ver,readme)		{}
set spamGlobals(Tk$ver,requires)	{}
set spamGlobals(Tk$ver,url)		{}
set spamGlobals(Tk$ver,objFiles)	{}
set spamGlobals(Tk$ver,tclFiles)	{}
set spamGlobals(Tk$ver,dataFiles)	{}

set spamGlobals(currPackage)	Tcl$ver
# -----------------------------------------------------------------------
#
# Debugging stuff:
#    o DebugEnter is called when a procedure is entered
#    o DebugLeave when it's left, and
#    o Debug to print a message (with formatting)
#    o FatalError prints a message an exits
#

#
# FatalError
#    Called when all hope is lost.  Just prints an error and
#    exits.
#
# Arguments:
#    message	Message to print before exiting
#
proc FatalError message {
    global spamGlobals
    puts stderr "Fatal error: $message"
    puts stderr "Stack: $spamGlobals(DebugStack)"
    puts stderr "exiting..."
    exit
}

#
# Debug
#    Called to print a debugging message.  Indents it, too!
#
# Arguments:
#    args	Message to print
#
proc Debug args {
    global spamGlobals
    if $spamGlobals(Debug) {
	set level $spamGlobals(DebugLevel)
        puts [format "%24s: %${level}s%s" $spamGlobals(DebugProc) {} $args]
    }
}

#
# DebugEnter
#    Should be called when we enter a routine.  Updates the call stack,
#    increments the stack depth variable (DebugLevel), and prints the
#    name of the procedure.
#
# Arguments:
#    procName	Name of procedure being entered.
#
proc DebugEnter procName {
    global spamGlobals
    lappend spamGlobals(DebugStack) $procName
    set spamGlobals(DebugProc) $procName
    incr spamGlobals(DebugLevel) 2
    Debug "Entering $procName"
}

#
# DebugLeave
#    Counterpart of DebugEnter.  Should be called when we leave a routine.
#    Updates the call stack, decrements the stack depth variable (DebugLevel),
#    and prints the name of the procedure.
#
# Arguments:
#    none
#
proc DebugLeave {} {
    global spamGlobals
    set len [llength $spamGlobals(DebugStack)]
    incr len -1
    set procName [lrange $spamGlobals(DebugStack) $len $len]
    incr len -1
    set spamGlobals(DebugStack) [lrange $spamGlobals(DebugStack) 0 $len]
    set spamGlobals(DebugProc) [lrange $spamGlobals(DebugStack) $len $len]
    Debug "Leaving $procName"
    incr spamGlobals(DebugLevel) -2
}

# -----------------------------------------------------------------------
#
# Undo commands
#
# This section allows to undo a faulty installation.
#
# UndoAdd
#	Adds to the Undo stack
#
# Arguments:
#    cmd
#	The command to execute to undo the current command.
#
proc UndoAdd cmd {
    global spamGlobals
    DebugEnter UndoAdd
    Debug $cmd
    lappend spamGlobals(UndoStack) $cmd
    incr spamGlobals(UndoLength) 1
    DebugLeave
}

proc UndoAll {} {
    global spamGlobals
    DebugEnter UndoAll
    Debug $spamGlobals(UndoStack)
    for {set i $spamGlobals(UndoLength)} {$i > 0} {incr i -1}  {
	set len $i
	incr len -1

	# first, undo the command
	set code [lindex $spamGlobals(UndoStack) $len]
	Debug "i=$i len=$len proc=$code"
	eval $code

	#second, delete the command from the list
	set spamGlobals(UndoStack) [lrange $spamGlobals(UndoStack) 0 $len]
    }
    DebugLeave
}

# -----------------------------------------------------------------------
#
# User Interface
#
# This section of code is all oriented around creating the UI.
# The main entry point is SpamCreateUI, which calls all the rest of the
# routines in this section.

#
# SpamCreateMenus
#    Create the menu bar across the top of the window
#
# Arguments:
#    None
#
proc SpamCreateMenus {} {
    global spamGlobals
    DebugEnter SpamCreateMenus
    frame .menu -relief raised -bd 2
    pack .menu -side top -fill x

    # File Menu
    set m .menu.file.m
    menubutton .menu.file -text File -menu $m -underline 0
    pack .menu.file -side left
    menu $m
    $m add command -label "Install..." -underline 0 -command DoFileInstall
    $m add command -label "Uninstall" -underline 1 -command DoFileUninstall
    $m add separator
    $m add command -label "Exit" -underline 1 -command DoFileExit

    DebugLeave
}

#
# SpamCreateWindow
#    Create the main window -- this window lists the installed packages, and has a
#    README section.
#
# Arguments:
#    none
#
proc SpamCreateWindow {} {
    global spamGlobals
    DebugEnter SpamCreateWindow

    set w .browser
    frame $w
    pack $w -side bottom -fill both -expand yes
    scrollbar $w.scroll -command "$w.text yview"
    text $w.text -relief sunken -bd 2 -yscrollcommand "$w.scroll set" -setgrid 1 -height 12
    pack $w.scroll -side right -fill y
    pack $w.text -expand yes -fill both
    set spamGlobals(Browser) $w.text
    UpdateBrowser

    # Create the README section of the frame
    set w .readme
    frame $w
    pack $w -side top -fill both -expand yes
    scrollbar $w.scroll -command "$w.text yview"
    text $w.text -relief sunken -bd 2 -yscrollcommand "$w.scroll set" -setgrid 1 -height 10
    set spamGlobals(Readme) $w.text
    pack $w.scroll -side right -fill y
    pack $w.text -expand yes -fill both
    UpdateReadme

    DebugLeave
}

#
# UpdateReadme -
#   Updates the readme window of the Info window.
#   Called whenever the current package changes.
#   The associated widget is a text widget.
#
proc UpdateReadme {} {
	global spamGlobals
	DebugEnter UpdateReadme
	set w $spamGlobals(Readme)
	set currPkg [lindex $spamGlobals(currPackage) 0]
	
	if {[lsearch $spamGlobals(rootList) $currPkg] < 0} {
		set filename $spamGlobals($currPkg,readme)
	} else {
		set filename ""
	}


	$w delete 1.0 end
	$w insert end "Information on the $currPkg package\n\n"

	set m .menu.file.m
	set pkg [lindex $spamGlobals(currPackage) 0]

	set name $spamGlobals($pkg,name)

	if {[lsearch $spamGlobals(rootList) $pkg] < 0} {
		set name $spamGlobals($pkg,name)
	} else {
		set name $pkg
	}

	if {($name == "Tcl") || ($name == "Tk")} {
	$m entryconfigure Uninstall -state disabled
	} else {
	$m entryconfigure Uninstall -state normal
	}



   Debug $filename
   if {[string length $filename] &&
       [file exists $filename] &&
       [file readable $filename]} {
	set f [open $filename r]
	$w insert end [read $f]
	close $f
   } else {
	$w insert end "No information available"
   }	


   wm title . "SPAM -- $currPkg"
   DebugLeave
}

#
# UpdateBrowser -
#   Updates the browser window.
#   Called whenever the list of installed packages changes.
#
proc UpdateBrowser {} {
    global spamGlobals
    DebugEnter UpdateBrowser
    set w $spamGlobals(Browser)
    $w delete @0,0 end
    $w insert end "Packages installed:\n\n"
    $w insert end [format "%-16s%-16s%-32s\n" "package" "version" "requires"]
    if {[winfo depth $w] > 1} {
        set hot "-background #808080 -relief raised -borderwidth 1"
        set normal "-background {} -relief flat"
    } else {
        set hot "-foreground white -background black"
        set normal "-foreground {} -background {}"
    }

    set spamGlobals(installed) [lsort $spamGlobals(installed)]
    foreach pkg $spamGlobals(installed) {
	set ver $spamGlobals($pkg,version)
	set req $spamGlobals($pkg,requires)

	# this is the code for displaying rooted stuff.
	set split [file split $pkg]
	set lsplit [llength $split]
	if {$lsplit == 1} {

	    $w tag bind $pkg <Any-Enter> "$w tag configure $pkg $hot"
	    $w tag bind $pkg <Any-Leave> "$w tag configure $pkg $normal"
	    $w tag bind $pkg <Button-1> "BrowserCallback $pkg"

	    $w insert end [format "%-19s%-16s%-29s\n" $pkg $ver $req] $pkg
	} else {
	    Debug "Halves: $split"
	    set rt [$w search [lindex $split 0] 0.0]
	    Debug "rt: $rt"
	    if {$rt > 0} {
		$w tag bind $pkg <Any-Enter> "$w tag configure $pkg $hot"
		$w tag bind $pkg <Any-Leave> "$w tag configure $pkg $normal"
		$w tag bind $pkg <Button-1> "BrowserCallback $pkg"
		$w insert [expr $rt + 1] \
			[format "%-19s%-16s%-29s\n" [join [list "  " [lindex $split 1]]] $ver $req] $pkg
	    } else {
		set r [lindex $split 0]
		$w tag bind $r <Any-Enter> "$w tag configure $r $hot"
		$w tag bind $r <Any-Leave> "$w tag configure $r $normal"
		$w tag bind $r <Button-1> "BrowserCallback $r"
		$w insert end [format "%-19s\n" $r] $r
		set spamGlobals(rootList) [concat $spamGlobals(rootList) $r]
		$w tag bind $pkg <Any-Enter> "$w tag configure $pkg $hot"
		$w tag bind $pkg <Any-Leave> "$w tag configure $pkg $normal"
		$w tag bind $pkg <Button-1> "BrowserCallback $pkg"
		$w insert end [format "%-19s%-16s%-29s\n" [join [list "  " [lindex $split 1]]] $ver $req] $pkg
	    }
	}
    }
    DebugLeave
}

#
# BrowserCall	back --
#    This procedure is called when the user clicks on a package in the browser window.
#    It is responsible for updating the UI to show details of the selected package
#
# Arguments:
#    index	The index of the character that the user clicked on.
#
proc BrowserCallback pkg {
    global spamGlobals
    DebugEnter "BrowserCallback $pkg"
    set w $spamGlobals(Browser)
	
    if {[lsearch $spamGlobals(rootList) $pkg] < 0} {
	set spamGlobals(currPackage) [list $pkg]
    } else {
	set l {}
	foreach item $spamGlobals(installed) {    
	    if {[string first $pkg $spamGlobals($item,name)] == 0} {
		set l [concat $l $item]
	    }
	}
	set spamGlobals(currPackage) $l
    }

    UpdateReadme
    DebugLeave
}

#
# SpamCreateUI
#    Create the SPAM User Interface in the main window (".")
#	1. Create the menus
#	2. Create & show the browse windows
#
proc SpamCreateUI {} {
    global spamGlobals
    DebugEnter SpamCreateUI
    SpamCreateMenus
    SpamCreateWindow
    set currPkg $spamGlobals(currPackage)
    wm title . "SPAM -- $currPkg"
    wm iconname . "SPAM"
    DebugLeave
}

# -----------------------------------------------------------------------
#
# Init
#
# This section contains the initialization rountines:
#   o SpamInit is the main entry point.
#   o SpamInstall makes sure the spam library files are around, and that we
#     can write those files we must
#   o SpamGetPackages figures out what packages are currently installed.

#
# SpamInstall
#    Initializes the SPAM package manager.  This function makes sure
#    that the SPAM installation is ok, and installs the needed files
#    if not.
#
#    It adds a spam.tcl file to [info library] and modifies the
#    init.tcl file there.  It will fail if the read permissions
#    are not correct.

proc SpamInstall {} {
    global spamGlobals
    DebugEnter "SpamInstall"
    set tclLib [info library]
    set spamFile $tclLib/spam.tcl
    if {![file writable $tclLib]} {
	FatalError "You must have permission to write $tclLib to run SPAM"
    }
    if [file exists $spamFile] {
	if {![file writable $spamFile]} {
	    FatalError "You must have permission to write $spamFile to run SPAM"
        }
	Debug "$spamFile exists"
	DebugLeave
	return
    }
    if {![file writable $tclLib/init.tcl]} {
	FatalError "You must have permission to write $tclLib/init.tcl to install SPAM"
    }
    SpamWriteMasterFile
    Debug "Modifying $tclLib/init.tcl"
    if [catch {set f [open $tclLib/init.tcl a]} err] {
	FatalError "Error editing $tclLib/init.tcl: $err"
    }
    puts $f ""
    puts $f {source [info library]/spam.tcl}
    close $f
    DebugLeave
}

#
# SpamGetPackages
#    Get a list of the current packages
#
proc SpamGetPackages {} {
    global spamGlobals
    DebugEnter SpamGetPackages

    source [info library]/spam.tcl
    set spamGlobals(currPackage) [lindex $spamGlobals(installed) 0]

    DebugLeave
}

#
# SpamInit
#    Initializes the SPAM package manager.  This consists of the
#    following steps:
#	1. Create the main window,
#	2. Make sure the SPAM files are installed and we have
#	   permission to write them.
#	3. Get a list of all installed packages from spam.tcl

proc SpamInit {} {
    global spamGlobals
    DebugEnter SpamInit
    SpamInstall
    SpamGetPackages
    SpamCreateUI
    DebugLeave
}


# -----------------------------------------------------------------------
#
# UI Callbacks
#
# This section of code contains the UI callbacks. This includes:
#    The pulldown menu callbacks (DoFileOpen, etc)
#

#
# The next 3 functions, DoFileInstall, FsboxExit, and FsboxOpenFile,
# work together to open a file to install.  DoFileInstall is called
# first, when the menu item File->Install is selected.  It posts a
# dialog, which then calls either FsboxExit or FsboxOpenFile, depending
# on whether the user hit "Open" or "Cancel"
#
proc DoFileInstall {} {
   global spamGlobals
   DebugEnter DoFileInstall
   # Pop up a file dialog, look for files named "spam"
   if {[info commands .fileSelectBox] != ""} {
	return;
   }
   toplevel .fileSelectBox
   wm title .fileSelectBox "Open File";

   fsbox .fileSelectBox.fsbox {{"Spam Files" *.spm} {"All Files" *}}    
   pack .fileSelectBox.fsbox -pady 5 -padx 5 -in .fileSelectBox \
	-expand yes -fill both;
   wm geometry  .fileSelectBox 25x8
   DebugLeave
}

proc FsboxExit {} {
   pack forget .fsbox;
   destroy .fileSelectBox;
}

proc FsboxOpenFile {file} {
   global spamGlobals;
   DebugEnter FsboxOpenFile
   FsboxExit;
   InstallPackage $file
   DebugLeave
}

#
# Uninstall the current package
#
#	o Pops up dialog to confirm the deletion
#	o If OK, delete directory, remove from list of installed, rewrite master
#	  file, and update UI.

proc DoFileUninstall {} {
    global spamGlobals tcl_library
    DebugEnter DoFileUninstall

    foreach pkg $spamGlobals(currPackage) {
	set name $spamGlobals($pkg,name)
	if {($name == "Tcl") || ($name == "Tk")} {
	    tk_dialog .error "Error" "Can't uninstall $pkg" {} 0 Ok
	    return
	}
	if [tk_dialog .query "Verify" "Uninstall $pkg?" {} 1 Ok Cancel] {
	    return
	}

	# This gets around a bug in Win32 networking.
	set olddir [pwd]
	cd "$tcl_library/.."
	file delete -force [string tolower ${pkg}]
	cd $olddir

	# Delete the package from the installed list
	set idx [lsearch -exact $spamGlobals(installed) $pkg]
	set len [llength $spamGlobals(installed)]
	if {$idx > 0} {
	    set new [lrange $spamGlobals(installed) 0 [expr $idx-1]]
	} else {
	    set new {}
	}
	if {$idx < ($len-1)} {
	    set new [concat $new [lrange $spamGlobals(installed) [expr $idx+1] end]]
	}
	set spamGlobals(installed) $new

	# Reset the vars in spamGlobals
	unset spamGlobals($pkg,name)
	unset spamGlobals($pkg,version)
	unset spamGlobals($pkg,readme)
	unset spamGlobals($pkg,requires)
	unset spamGlobals($pkg,url)
	unset spamGlobals($pkg,objFiles)
	unset spamGlobals($pkg,tclFiles)
    }

    set spamGlobals(currPackage) [list [lindex $spamGlobals(installed) 0]]
	 
    # Rewrite the master file and update the UI
    SpamWriteMasterFile
    UpdateReadme
    UpdateBrowser
	
    DebugLeave
}

proc DoFileExit {} {
   global spamGlobals
   DebugEnter DoFileExit
   exit
}

#
# InstallPackage --
#
#   Install the package whose name is passed in.  Assumes it's not
#   currently installed.
#
#   What to do to install a package:
#	1) Read the spam file from the dir.  This sets:
#	   var	sample		description
#	   name	Example		name of package (no spaces)
#	   version	1.0			version number of package
#	   files	{example.tcl}	list of files in package
#	   requires	{{Tcl 7.5}}		List of packages required
#	   url	ftp://...		URL for updates
#	   info	README.txt		file that describes package.
#
#	2) Make the directory [info library]/$pkg
#	3) Copy $packageFiles, $infoFileName to this dir ($dir)
#	4) run pkg_mkindex on this dir
#	5) update spamGlobals
#	6) Update master file so all Tcl/Tk shells see new package
#	7) Update the UI
#
# Arguments
#
#    srcdir	Pathname of directory containing package to be installed.
#

proc InstallPackage spamFile {
    global spamGlobals tcl_platform
    DebugEnter InstallPackage 
 
    set srcdir [file dirname $spamFile]
    if {[string length $srcdir] == 0} {
	set srcdir .
    }
    
    foreach dirname {tclFileDir objFileDir dataFileDir} {
	if [info exists $dirname] {set $dirname .}
    }

    # Instead of a normal spam file, it is possible to have a master file.  
    # this file has a list of other files to use
    # if it exists, it defines a packageRoot function that points to other spam files.

    if [file readable $spamFile] {
	set t [open $spamFile r]
	if {[gets $t]=="#SPAM ROOT FILE"} {
	    source $spamFile
	} else {
	    set packageRoot [list $spamFile]
	}
	close $t
    } else {
	tk_dialog .error "File not found" "Error: File not found" {} 0 Ok
	return
    }

    foreach sFile $packageRoot {

	#
	# Step 1: Source and verify the Spam file
	#
	if [file readable $sFile] {
	    source $sFile
	} else {
	    tk_dialog .error "File not found" "Error: File not found" {} 0 Ok
	    return
	}
    
	# Make sure the file defined the right variables!
	foreach var {name version tclFiles objFiles dataFiles requires url info} {
	    if {![info exists $var]} {
	       tk_dialog .error "Invalid File" \
		    "Error: Invalid spam file $spamFile.\nDoesn't define $var" {} 0 Ok
	       return
	    }
	}
	set pkg ${name}${version}

	# Make sure the package isn't already installed
	Debug "lsearch -exact $spamGlobals(installed) $pkg"
	if {[lsearch -exact $spamGlobals(installed) $pkg] != -1} {
	    Debug "$spamGlobals($pkg,version) == $version"
	    if {$spamGlobals($pkg,version) == $version} {
		    tk_dialog .error "Already Installed" \
		    "Error: Package $pkg is already installed" {} 0 Ok
	       return
	    }
	}

	#
	# Step 2: Make the directory (or verify that it exists)
	#
	# (2/19) Added tolower so that directory is always lowercase [MP]
	#
	set tcl_root [file dirname [info library]]
	set olddir [pwd]
	cd "$tcl_root"
	set libdir "[string tolower ${pkg}]"

	if [file isdirectory $libdir] {
	    set rval [tk_dialog .error "Directory Exists" \
		"Warning: Directory $libdir already exists. 
Press Delete to replace it,
Press Cancel to abort the installation" {} 1 Delete Cancel]
	    if {$rval == 1} {
		UndoAll
		return;
	    }
	    if [catch {file delete -force $libdir} err] {
		tk_dialog .error "Error Deleting Directory" \
		    "Error: Couldn't delete $libdir ($err).  Aborting installation" {} 0 Ok
		UndoAll
		return;
	    }
	}
	Debug "mkdir $libdir"
	if [catch {file mkdir $libdir} err] {
	    tk_dialog .error "Error Creating Directory" \
		"Error: Couldn't create $libdir ($err). Aborting installation" {} 0 Ok
	    UndoAll
	    return
	}
	UndoAdd "file delete -force \"$tcl_root/$libdir\""

	cd "$olddir"
	set libdir "$tcl_root/$libdir"

	#
	# Step 3: Copy the object files, tcl files, data files, and info (the
	# readme file
	#
	if [info exists info] {
	    Debug "cp $info $libdir/$info"
	    if [catch {file copy -force $info $libdir/$info} err] {
		tk_dialog .error "Error copying files" \
			"Error: Couldn't copy $info to $libdir ($err)" {} 0 Ok
		UndoAll	
		return
	    }
	}

	if [info exists objFileDir] {
	    set tmpDir $srcdir/$objFileDir
	} else {
	    set tmpDir $srcdir      
	}
	Debug "tmpDir = $tmpDir"
	Debug "objFileDir = $objFileDir"
	foreach file $objFiles {
	    Debug "cp $tmpDir/$file $libdir/$file"
	    if [catch {file copy -force $tmpDir/$file $libdir/$file} err] {
		tk_dialog .error "Error copying files" \
			"Error: Couldn't copy $tmpDir/$file to $libdir ($err)" {} 0 Ok
		UndoAll	
		return
	    }
	}

	if [info exists tclFileDir] {
	    set tmpDir $srcdir/$tclFileDir
	} else {
	    set tmpDir $srcdir      
	}
	foreach file $tclFiles {
	    Debug "cp $tmpDir/$file $libdir/$file"
	    if [catch {file copy -force $tmpDir/$file $libdir/$file} err] {
		tk_dialog .error "Error copying files" \
			"Error: Couldn't copy $tmpDir/$file to $libdir ($err)" {} 0 Ok
		UndoAll	
		return
	    }
	}

	if [info exists dataFileDir] {
	    set tmpDir $srcdir/$dataFileDir
	} else {
	    set tmpDir $srcdir      
	}
	foreach file $dataFiles {
	    Debug "cp $tmpDir/$file $libdir/$file"
	    if [catch {file copy -force $tmpDir/$file $libdir/$file} err] {
		tk_dialog .error "Error copying files" \
			"Error: Couldn't copy $tmpDir/$file to $libdir ($err)" {} 0 Ok
		UndoAll	
		return
	    }
	}

	# Added chmod here because of file permission problems
	# (2/19) [MP]
	if {$tcl_platform(platform) == "unix"} {
		foreach f [glob "$libdir/*"] {
			exec chmod 755 $f
		}
	}

	#
	# Step 4: Update our variables
	#
	lappend spamGlobals(installed) $pkg
	set spamGlobals($pkg,name) $name
	set spamGlobals($pkg,version) $version
	if [string length $info] {
	    set spamGlobals($pkg,readme) $libdir/$info
	} else {
	    set spamGlobals($pkg,readme) {}
	}
	set spamGlobals($pkg,requires) $requires
	set spamGlobals($pkg,url) $url
	set spamGlobals($pkg,objFiles) $objFiles
	set spamGlobals($pkg,tclFiles) $tclFiles
	set spamGlobals($pkg,dataFiles) $dataFiles

	#
	# Step 5: update $tcl_library/spam.tcl:
	#
	SpamWriteMasterFile

	# Update the screen...
	UpdateBrowser
    }
    DebugLeave
}
	
# SpamWriteMasterFile
#	Writes the master file ("spam.tcl" in [info library]) so that
#	new tcl shells that start will see our changes.
#
# Arguments:
#    var	meaning
#    ...
#
proc SpamWriteMasterFile {} {
    global spamGlobals
    DebugEnter SpamWriteMasterFile

    set tcl_library [info library]
    if [file exists $tcl_library/spam.tcl] {file copy -force -- $tcl_library/spam.tcl $tcl_library/spam.bak}
    Debug "Just copied $tcl_library /spam.tcl"
    UndoAdd "file rename -force -- $tcl_library/spam.bak $tcl_library/spam.tcl"
    if [catch {
	    set f [open $tcl_library/spam.tcl w]
	    puts $f {# spam.tcl --
#
# This file is part of SPAM (Smith's PAckage Manager).  It
# maintains a list of all packages, and is automatically
# generated.  Edit it at your own peril!

proc spamPkgSetup {dir obj tcl} {
    foreach f $obj {load "$dir/$f"}
    # Added 'uplevel #0' on advice of Peter Onion (ponion@srd.bt.co.uk)
    foreach f $tcl {uplevel #0 source [list "$dir/$f"]}
}

	    }
	    puts $f "set spamGlobals(installed) [list $spamGlobals(installed)]"

	    foreach pkg $spamGlobals(installed) {
		puts $f ""
		puts $f "set spamGlobals($pkg,name) $spamGlobals($pkg,name)"
		puts $f "set spamGlobals($pkg,version) $spamGlobals($pkg,version)"
		puts $f "set spamGlobals($pkg,readme) [list $spamGlobals($pkg,readme)]"
		puts $f "set spamGlobals($pkg,requires) [list $spamGlobals($pkg,requires)]"
		puts $f "set spamGlobals($pkg,url) [list $spamGlobals($pkg,url)]"
		puts $f "set spamGlobals($pkg,objFiles) [list $spamGlobals($pkg,objFiles)]"
		puts $f "set spamGlobals($pkg,tclFiles) [list $spamGlobals($pkg,tclFiles)]"
		puts $f "set spamGlobals($pkg,dataFiles) [list $spamGlobals($pkg,dataFiles)]"
		set name  $spamGlobals($pkg,name)
		set ver $spamGlobals($pkg,version)
		if {($name == "Tcl") || ($name == "Tk")} {
		   continue
		}
		set libdir "[file dirname "$tcl_library"]/[string tolower "${pkg}"]"
		set objFiles $spamGlobals($pkg,objFiles)
		set tclFiles $spamGlobals($pkg,tclFiles)
		set dataFiles $spamGlobals($pkg,dataFiles)
		puts $f "package ifneeded $name $ver \"spamPkgSetup [list $libdir]  \
			[list $objFiles] [list $tclFiles]\""
	    }
	    close $f
	  } err] {
	    tk_dialog .error "Error Writing Master File" \
	    "Error: Couldn't write $tcl_library/spam.tcl ($err)" {} 0 Ok
	    UndoAll
    }
    DebugLeave
}

# -----------------------------------------------------------------------
#
# This section contains code for a simple file dialog
#
proc slistbox {path} {
    frame $path;
    set p $path;

    frame $p.top;
    frame $p.bot;

    listbox $p.lbox -relief sunken -setgrid 1 \
	-yscrollcommand "$p.vert set" -xscrollcommand "$p.horiz set";
    scrollbar $p.vert -orient vertical -relief sunken \
	-command "$p.lbox yview";
    scrollbar $p.horiz -orient horizontal -relief sunken \
	-command "$p.lbox xview";
    frame $p.nop -width 19 -height 19;

    pack $p.vert $p.lbox -in $p.top -fill both -side right;
    pack configure $p.lbox -expand yes;
    pack $p.horiz $p.nop -in $p.bot -fill both -side left;
    pack configure $p.horiz -expand yes;

    pack $p.bot $p.top -in $p -fill both -side bottom;
    pack configure $p.top -expand yes;
    return $path;
}


proc fsbox { path {flist {}} } {
    global fspath currentDir filter filterList; 

    # defaults
    
    set fspath $path;
    set currentDir [pwd];
    set filterList $flist;

    if {"$flist" == {}} {
	set filterList [list "All Files\t" *];
    }
    
    set filter [lindex $filterList 0];
    
    # toplevel frames
    
    frame $fspath;
    frame $fspath.boxes;
    frame $fspath.options;
    frame $fspath.sep -borderwidth 1 -height 2 -relief sunken;
    frame $fspath.buttons;
    
   ##### Boxes frame #####
    
    set f $fspath.boxes.files;
    set d $fspath.boxes.dirs;
    
    frame $f;
    frame $d;
    
    label $f.l -text Files: -anchor w;
    entry $f.e -relief sunken -width 1;
    slistbox $f.list;
    
    label $d.l -text Directories: -anchor w;
    label $d.e -text "$currentDir" -width 1 -anchor w;
    slistbox $d.list;
    
    pack $f.l $f.e $f.list -side top -pady 1 -fill both;
    pack configure $f.list -expand yes;
    
    pack $d.l $d.e $d.list -side top -pady 1 -fill both;
    pack configure $d.list -expand yes;
    
    pack $f $d -in $fspath.boxes -side left -expand yes -fill both -padx 10;
    
   ##### Options frame #####
    
    set o $fspath.options;
    
    menubutton $o.filter -relief groove -pady 3 -padx 3 -width 1\
	-menu $o.filter.m -text "    [Fdesc]" -anchor w;
    menu $o.filter.m;

#   label $o.hidden -pady 3 -padx 3 -width 1;
   
    pack $o.filter -expand yes -fill both -side left \
 	-in $o -padx 10;
  
    foreach fpair $filterList {
	set fdesc [lindex $fpair 0];
	set fspec [lindex $fpair 1];
	$o.filter.m add radiobutton -label "$fdesc   $fspec" \
	    -variable filter -value $fpair \
	    -command { 
		$fspath.options.filter configure -text "    [Fdesc]";
		fillBoxes $currentDir [Fspec]; 
	    }
    }
    
   ##### Buttons frame #####
    
    set b $fspath.buttons;
    
    frame $b.ok -borderwidth 1 -relief sunken;
    button $b.ok.b -text OK -pady 5 -padx 8 -command {
	set selFile [$fspath.boxes.files.e get];
	cd $currentDir;
	if {[file isfile $selFile]} {
	    FsboxOpenFile $selFile;
	} else {
	    $fspath.boxes.files.e delete 0 end;
	    $fspath.boxes.files.e insert 0 "Filename Not Found";
	    update;
	    after 500
	    $fspath.boxes.files.e delete 0 end;
	    $fspath.boxes.files.e insert 0 $selFile;
	}
    }            
    
    button $b.cancel -text Cancel -pady 5 -padx 8 -command FsboxExit;
    
    pack $b.ok.b -in $b.ok -pady 5 -padx 5;
    pack $b.ok $b.cancel -expand yes -in $b -side left;
    
    # Pack widgets
    
    pack $b $fspath.sep $o $fspath.boxes -side bottom -fill both \
	-pady 5 -padx 5 -in $fspath;
    pack configure $fspath.boxes -expand yes;
    
    # Make neccesary bindings
    
    focus $f.e;
    
    bind $f.e <Return> {keyEnter};
    bind $f.e <Tab> {keyTab};
    
    bind $f.list.lbox <1> {
	set selFile [%W get [%W index @%x,%y]]
	$fspath.boxes.files.e delete 0 end;
	$fspath.boxes.files.e insert 0 $selFile;
    };

    bind $f.list.lbox <B1-Motion> {
	set selFile [%W get [%W index @%x,%y]]
	$fspath.boxes.files.e delete 0 end;
	$fspath.boxes.files.e insert 0 $selFile;
    };
	
    bind $f.list.lbox <Double-1> {FsboxOpenFile $selFile; break};

    bind $d.list.lbox <Double-1> {
	set selDir [%W get [%W index @%x,%y]]
	changeDir $selDir;
	$fspath.boxes.files.e delete 0 end;
	fillBoxes $currentDir [Fspec];
    };
    
    fillBoxes $currentDir [Fspec];
}


# Takes a directory and a filter-spec, and returns a sorted list of the
# files that match the filter-spec in that directory.

proc GetFiles {dir fspec} {
    cd $dir;
    set fileList {};
    set fspec [split $fspec \;\,\:\ ];
    foreach filter $fspec {
	set fileList [concat $fileList [glob -nocomplain $filter]];
    }
    set fileList [lsort $fileList];
    return $fileList;
}


proc fillBoxes {dir fspec} {
    global fspath;
    
    set fileList {};
    set dirList {};
    
    set files [GetFiles $dir $fspec];
    set dirs [GetFiles $dir *];

    foreach f $files {
	if {[file isfile $f] && [file readable $f]} {
	    lappend fileList [file tail $f];
	}
    }
        
    foreach d $dirs {
	if {[file isdirectory $d] && [file readable $d]} {
	    lappend dirList [file tail $d]/;
	}
    }
    
    set dirList [concat ./ ../ $dirList];
    
    set flbox $fspath.boxes.files.list.lbox;
    set dlbox $fspath.boxes.dirs.list.lbox;
    
    $flbox delete 0 end;
    $dlbox delete 0 end;
    eval $flbox insert 0 $fileList;
    eval $dlbox insert 0 $dirList;
}

proc keyEnter {} {
    global filter currentDir openFileCommand fspath;
    
    set entry $fspath.boxes.files.e;
    set file [$entry get];
    
    # Check to see if we have a full pathname in $file
    
    if {[file dirname $file] != "."} {
	set path [file dirname $file];
	set fspec [file tail $file];
    } else {
	set path $currentDir;
	set fspec $file;
    }
    
    if {"$fspec" == ""} {
	set fspec .;
    }
    
    set size [llength [GetFiles $path $fspec]];

    if {[string first * $fspec] != -1 ||
	[string first \; $fspec] != -1 ||
	[string first , $fspec] != -1 ||
	[string first \  $fspec] != -1} {
	set filtering 1;
    } else {
	set filtering 0;
    }
	    
    $entry delete 0 end;

    if {$size == 0 && !$filtering} {
	$entry delete 0 end;
	$entry insert 0 "Filename Not Found.";
	update;
	after 500
	$entry delete 0 end;
	$entry insert 0 $file;
	fillBoxes $currentDir [Fspec];
	return;
    } elseif {$size > 1 || $filtering} {
	changeDir $path;
	newFilter $fspec;
	fillBoxes $currentDir [Fspec];
	return;
    } else {
	if {[file isdirectory $fspec]} {
	    changeDir $fspec;
	    fillBoxes $currentDir [Fspec];
	    return;
	} else {
	    FsboxOpenFile $fspec;
	    return;
	}
    }
}

proc keyTab {} {
    global currentDir fspath;
    
    set fentry [$fspath.boxes.files.e get];
    
    # used in case we try to access ~user when user does not exist.

    set noUser [catch {file dirname $fentry} path];
    if {$noUser} {
	return;
    }
    
    if {$path != "."} {
	set path [file dirname $fentry];
	set file [file tail $fentry];
    } else {
	set path $currentDir;
	set file $fentry;
    }
    
    # used in case they give a path that doesn't exist

    set noPath [catch {cd $path}]
    if {$noPath} {
	return;
    }

    set f [glob -nocomplain $file*];
    set size [llength $f];
    
    if {$size == 0} {
	return;
    } elseif {$size == 1} {
	$fspath.boxes.files.e delete 0 end;
	if {[file dirname $fentry] == "."} {
	    $fspath.boxes.files.e insert 0 $f;
	} else {
	    if {$path == "/"} {
		$fspath.boxes.files.e insert 0 /$f;
	    } else {
		$fspath.boxes.files.e insert 0 $path/$f;
	    }		
	}
	fillBoxes $path $file*;
	return;
    } else {
	fillBoxes $path $file*;
	return;
    }
}

proc newFilter {f} {
    global filter fspath;
    set filter [list "User Defined" $f];
    $fspath.options.filter configure -text "    [Fdesc]";
}

proc Fspec {} {
    global filter;
    return [lindex $filter 1];
}

proc Fdesc {} {
    global filter;
    return "[lindex $filter 0]   [lindex $filter 1]";
}

proc changeDir {newDir} {
    global currentDir fspath;

    cd $newDir;
    set currentDir [pwd];
    $fspath.boxes.dirs.e conf -text "$currentDir";
}

# -----------------------------------------------------------------------
#
# Templates
#

# SpamXXX
#    <Description>
#
# Arguments:
#    var	meaning
#    ...
#
proc SpamXXX {} {
    global spamGlobals
    DebugEnter SpamXXX
    DebugLeave
}

SpamInit
