/* 
 * tkCanvas3d.c --
 *
 *	This module implements canvas3d widgets as an extension 
 *	for the Tk toolkit. A canvas3d displays a background and a 
 *	collection of graphical objects such as boxes, spheres, 
 *	cylinders, cones, 3d meshes. Tcl/Tk plays the role of parser/
 *	manager in the whole system, whereas the actual graphics 
 *	rendering is provided by separately written graphics engine
 *	library.
 *
 *	Note : This file is one of the most important one, however,
 *	all decisions regarding this file were left till the end.
 *	Lots of changes are still required, lots of code is redundant. 
 *	I originally started with tkCanvas.c code, however, a large 
 *	part of it is now replaced. A neat surgery over this file still
 *	remains. This disclaimer in no way means that this file is 
 *	useless, it just means, it is not perfect!
 */

#include <default.h>
#include "tkCanvas3dInt.h"

/*
 * The structure defined below is used to keep track of a tag search
 * in progress.  Only the "prevPtr" field should be accessed by anyone
 * other than StartTagSearch and NextItem.
 */

typedef struct TagSearch {
    TkCanvas3d *canvas3dPtr;	/* Canvas3d widget being searched. */
    Tk_Uid tag;			/* Tag to search for.   0 means return
				 * all items. */
    Tk_Item3d *prevPtr;		/* Item just before last one found (or NULL
				 * if last one found was first in the item
				 * list of canvas3dPtr). */
    Tk_Item3d *currentPtr;	/* Pointer to last item returned. */
    int searchOver;		/* Non-zero means NextItem should always
				 * return NULL. */
} TagSearch;

/*
 * Information used for argv parsing.
 */

static Tk_ConfigSpec configSpecs[] = {
    {TK_CONFIG_PIXELS, "-height", "height", "Height",
	DEF_CANVAS_HEIGHT, Tk_Offset(TkCanvas3d, height), 0},
    {TK_CONFIG_PIXELS, "-width", "width", "Width",
	DEF_CANVAS_WIDTH, Tk_Offset(TkCanvas3d, width), 0},
    {TK_CONFIG_END, (char *) NULL, (char *) NULL, (char *) NULL,
	(char *) NULL, 0, 0}
};

/*
 * List of all the item types known at present:
 */

static Tk_Item3dType *typeList = NULL;	/* NULL means initialization hasn't
					 * been done yet. */

/*
 * canvas3d item types. All the items are required to be added to the
 * typeList in InitCanvas3d(), while they are actually defined in their
 * respective files.
 */

extern Tk_Item3dType tkBoxType, tkConeType, tkCylinType, tkSphrType;
extern Tk_Item3dType tkPolygonType, tkMeshType;
extern Tk_Item3dType tkSurfType, tkMatrixType, tkLightType, tkViewType;

/*
 * Various Tk_Uid's used by this module (set up during initialization):
 */

static Tk_Uid allUid = NULL;
static Tk_Uid currentUid = NULL;

/*
 * Statistics counters:
 */

static int numIdSearches;
static int numSlowSearches;

/*
 * Prototypes for procedures defined later in this file:
 */

static void		Canvas3dBindProc _ANSI_ARGS_((ClientData clientData,
			    XEvent *eventPtr));
static void		Canvas3dCmdDeletedProc _ANSI_ARGS_((
			    ClientData clientData));
static void		Canvas3dDoEvent _ANSI_ARGS_((TkCanvas3d *canvas3dPtr,
			    XEvent *eventPtr));
static void		Canvas3dEventProc _ANSI_ARGS_((ClientData clientData,
			    XEvent *eventPtr));
static Tk_Item3d *	Canvas3dFindClosest _ANSI_ARGS_((TkCanvas3d *canvas3dPtr,
			    double coords[2]));
static void		Canvas3dSetOrigin _ANSI_ARGS_((TkCanvas3d *canvas3dPtr,
			    int xOrigin, int yOrigin));
static int		Canvas3dWidgetCmd _ANSI_ARGS_((ClientData clientData,
			    Tcl_Interp *interp, int argc, char **argv));
static int		ConfigureCanvas3d _ANSI_ARGS_((Tcl_Interp *interp,
			    TkCanvas3d *canvas3dPtr, int argc, char **argv,
			    int flags));
static void		DestroyCanvas3d _ANSI_ARGS_((char *memPtr));
static void		DisplayCanvas3d _ANSI_ARGS_((ClientData clientData));
static void		DoItem _ANSI_ARGS_((Tcl_Interp *interp,
			    Tk_Item3d *itemPtr, Tk_Uid tag));
static int		FindItems _ANSI_ARGS_((Tcl_Interp *interp,
			    TkCanvas3d *canvas3dPtr, int argc, char **argv,
			    char *newTag, char *cmdName, char *option));
static int		FindArea _ANSI_ARGS_((Tcl_Interp *interp,
			    TkCanvas3d *canvas3dPtr, char **argv, Tk_Uid uid,
			    int enclosed));
static double		GridAlign _ANSI_ARGS_((double coord, double spacing));
static void		InitCanvas3d _ANSI_ARGS_((void));
static Tk_Item3d *	NextItem _ANSI_ARGS_((TagSearch *searchPtr));
static void		PickCurrentItem _ANSI_ARGS_((TkCanvas3d *canvas3dPtr,
			    XEvent *eventPtr));
static void		RelinkItems _ANSI_ARGS_((TkCanvas3d *canvas3dPtr,
			    char *tag, Tk_Item3d *prevPtr));
static Tk_Item3d *	StartTagSearch _ANSI_ARGS_((TkCanvas3d *canvas3dPtr,
			    char *tag, TagSearch *searchPtr));




/*
 *--------------------------------------------------------------
 *
 * Canvas3d_Init --
 *
 *	This procedure is invoked to create a new "canvas3d" Tcl
 *	command from the Tcl_AppInit() in winMain.c.
 *
 * Results:
 *	A standard Tcl result.
 *
 * Side effects:
 *	See the user documentation.
 *
 *--------------------------------------------------------------
 */

int
Canvas3d_Init(interp)
    Tcl_Interp *interp;		/* Current interpreter. */
{
    Tcl_CreateCommand(interp, "canvas3d", Tk_Canvas3dCmd, 
	    (ClientData) Tk_MainWindow(interp), 
	    (Tcl_CmdDeleteProc *) NULL);

    return TCL_OK;
}


/*
 *--------------------------------------------------------------
 *
 * Tk_Canvas3dCmd --
 *
 *	This procedure is invoked to process the "canvas3d" Tcl
 *	command.  See the user documentation for details on what
 *	it does.
 *
 * Results:
 *	A standard Tcl result.
 *
 * Side effects:
 *	See the user documentation.
 *
 *--------------------------------------------------------------
 */

int
Tk_Canvas3dCmd(clientData, interp, argc, argv)
    ClientData clientData;		/* Main window associated with
				 * interpreter. */
    Tcl_Interp *interp;		/* Current interpreter. */
    int argc;			/* Number of arguments. */
    char **argv;		/* Argument strings. */
{
    Tk_Window tkwin = (Tk_Window) clientData;
    TkCanvas3d *canvas3dPtr;
    Tk_Window new;

    if (typeList == NULL) {
	InitCanvas3d();
    }

    if (argc < 2) {
	Tcl_AppendResult(interp, "wrong # args: should be \"",
		argv[0], " pathName ?options?\"", (char *) NULL);
	return TCL_ERROR;
    }

    new = Tk_CreateWindowFromPath(interp, tkwin, argv[1], (char *) NULL);
    if (new == NULL) {
	return TCL_ERROR;
    }

    /*
     * Initialize fields that won't be initialized by ConfigureCanvas3d,
     * or which ConfigureCanvas3d expects to have reasonable values
     * (e.g. resource pointers).
     */

    canvas3dPtr = (TkCanvas3d *) ckalloc(sizeof(TkCanvas3d));
    canvas3dPtr->tkwin = new;
    canvas3dPtr->display = Tk_Display(new);
    canvas3dPtr->interp = interp;
    canvas3dPtr->widgetCmd = Tcl_CreateCommand(interp,
	    Tk_PathName(canvas3dPtr->tkwin), Canvas3dWidgetCmd,
	    (ClientData) canvas3dPtr, Canvas3dCmdDeletedProc);
    canvas3dPtr->firstItemPtr = NULL;
    canvas3dPtr->lastItemPtr = NULL;
    canvas3dPtr->facePtr = (geInterface *) ckalloc(sizeof(geInterface));
    canvas3dPtr->facePtr->d3dapp = NULL;
    canvas3dPtr->borderWidth = 0;
    canvas3dPtr->bgBorder = NULL;
    canvas3dPtr->relief = TK_RELIEF_FLAT;
    canvas3dPtr->highlightWidth = 0;
    canvas3dPtr->highlightBgColorPtr = NULL;
    canvas3dPtr->highlightColorPtr = NULL;
    canvas3dPtr->inset = 0;
    canvas3dPtr->width = None;
    canvas3dPtr->height = None;
    canvas3dPtr->confine = 0;
    canvas3dPtr->xOrigin = canvas3dPtr->yOrigin = 0;
    canvas3dPtr->drawableXOrigin = canvas3dPtr->drawableYOrigin = 0;
    canvas3dPtr->bindingTable = NULL;
    canvas3dPtr->currentItemPtr = NULL;
    canvas3dPtr->newCurrentPtr = NULL;
    canvas3dPtr->closeEnough = 0.0;
    canvas3dPtr->pickEvent.type = LeaveNotify;
    canvas3dPtr->pickEvent.xcrossing.x = 0;
    canvas3dPtr->pickEvent.xcrossing.y = 0;
    canvas3dPtr->state = 0;
    canvas3dPtr->scanX = 0;
    canvas3dPtr->scanXOrigin = 0;
    canvas3dPtr->scanY = 0;
    canvas3dPtr->scanYOrigin = 0;
    canvas3dPtr->hotPtr = NULL;
    canvas3dPtr->hotPrevPtr = NULL;
    canvas3dPtr->cursor = None;
    canvas3dPtr->pixelsPerMM = WidthOfScreen(Tk_Screen(new));
    canvas3dPtr->pixelsPerMM /= WidthMMOfScreen(Tk_Screen(new));
    canvas3dPtr->flags = 0;
    canvas3dPtr->nextId = 1;
    canvas3dPtr->psInfoPtr = NULL;


    Tk_SetClass(canvas3dPtr->tkwin, "Canvas3d");
    Tk_CreateEventHandler(canvas3dPtr->tkwin,
	    ExposureMask|StructureNotifyMask|FocusChangeMask,
	    Canvas3dEventProc, (ClientData) canvas3dPtr);
    Tk_CreateEventHandler(canvas3dPtr->tkwin, KeyPressMask|KeyReleaseMask
	    |ButtonPressMask|ButtonReleaseMask|EnterWindowMask
	    |LeaveWindowMask|PointerMotionMask, Canvas3dBindProc,
	    (ClientData) canvas3dPtr);
    if (ConfigureCanvas3d(interp, canvas3dPtr, argc-2, argv+2, 0) != TCL_OK) {
	goto error;
    }

    interp->result = Tk_PathName(canvas3dPtr->tkwin);
    return TCL_OK;

    error:
    Tk_DestroyWindow(canvas3dPtr->tkwin);
    return TCL_ERROR;
}
 
/*
 *--------------------------------------------------------------
 *
 * Canvas3dWidgetCmd --
 *
 *	This procedure is invoked to process the Tcl command
 *	that corresponds to a widget managed by this module.
 *	See the user documentation for details on what it does.
 *
 * Results:
 *	A standard Tcl result.
 *
 * Side effects:
 *	See the user documentation.
 *
 *--------------------------------------------------------------
 */

static int
Canvas3dWidgetCmd(clientData, interp, argc, argv)
    ClientData clientData;		/* Information about canvas3d
					 * widget. */
    Tcl_Interp *interp;			/* Current interpreter. */
    int argc;				/* Number of arguments. */
    char **argv;			/* Argument strings. */
{
    TkCanvas3d *canvas3dPtr = (TkCanvas3d *) clientData;
    size_t length;
    int c, result;
    Tk_Item3d *itemPtr = NULL;		/* Initialization needed only to
					 * prevent compiler warning. */
    TagSearch search;
    

    if (argc < 2) {
	Tcl_AppendResult(interp, "wrong # args: should be \"",
		argv[0], " option ?arg arg ...?\"", (char *) NULL);
	return TCL_ERROR;
    }
    Tcl_Preserve((ClientData) canvas3dPtr);
    result = TCL_OK;
    c = argv[1][0];
    length = strlen(argv[1]);
    if ((c == 'a') && (strncmp(argv[1], "addtag", length) == 0)) {
	if (argc < 4) {
	    Tcl_AppendResult(interp, "wrong # args: should be \"",
		    argv[0], " addtags tag searchCommand ?arg arg ...?\"",
		    (char *) NULL);
	    goto error;
	}
	result = FindItems(interp, canvas3dPtr, argc-3, argv+3, argv[2], argv[0],
		" addtag tag");
    }  else if ((c == 'a') && (strncmp(argv[1], "antiAlias", length) == 0)
	    && (length >= 3)) {
	int aliasFlag;

	if (argc != 3) {
	    Tcl_AppendResult(interp, "wrong # args: should be \"",
		    argv[0], " antiAlias onOrOff\"",
		    (char *) NULL);
	    goto error;
	}
	if (GE_GetConstBoolean(interp, argv[2], &aliasFlag) != TCL_OK){
	    goto error;
	}
	GE_RenderAntiAlias(canvas3dPtr->facePtr, aliasFlag);
    } else if ((c == 'b') && (strncmp(argv[1], "bind", length) == 0)
	    && (length >= 2)) {
	ClientData object;

	if ((argc < 3) || (argc > 5)) {
	    Tcl_AppendResult(interp, "wrong # args: should be \"",
		    argv[0], " bind tagOrId ?sequence? ?command?\"",
		    (char *) NULL);
	    goto error;
	}

	/*
	 * Figure out what object to use for the binding (individual
	 * item vs. tag).
	 */

	object = 0;
	if (isdigit(UCHAR(argv[2][0]))) {
	    int id;
	    char *end;

	    id = strtoul(argv[2], &end, 0);
	    if (*end != 0) {
		goto bindByTag;
	    }
	    for (itemPtr = canvas3dPtr->firstItemPtr; itemPtr != NULL;
		    itemPtr = itemPtr->nextPtr) {
		if (itemPtr->id == id) {
		    object = (ClientData) itemPtr;
		    break;
		}
	    }
	    if (object == 0) {
		Tcl_AppendResult(interp, "item \"", argv[2],
			"\" doesn't exist", (char *) NULL);
		goto error;
	    }
	} else {
	    bindByTag:
	    object = (ClientData) Tk_GetUid(argv[2]);
	}

	/*
	 * Make a binding table if the canvas3d doesn't already have
	 * one.
	 */

	if (canvas3dPtr->bindingTable == NULL) {
	    canvas3dPtr->bindingTable = Tk_CreateBindingTable(interp);
	}

	if (argc == 5) {
	    int append = 0;
	    unsigned long mask;

	    if (argv[4][0] == 0) {
		result = Tk_DeleteBinding(interp, canvas3dPtr->bindingTable,
			object, argv[3]);
		goto done;
	    }
	    if (argv[4][0] == '+') {
		argv[4]++;
		append = 1;
	    }
	    mask = Tk_CreateBinding(interp, canvas3dPtr->bindingTable,
		    object, argv[3], argv[4], append);
	    if (mask == 0) {
		goto error;
	    }
	    if (mask & (unsigned) ~(ButtonMotionMask|Button1MotionMask
		    |Button2MotionMask|Button3MotionMask|Button4MotionMask
		    |Button5MotionMask|ButtonPressMask|ButtonReleaseMask
		    |EnterWindowMask|LeaveWindowMask|KeyPressMask
		    |KeyReleaseMask|PointerMotionMask)) {
		Tk_DeleteBinding(interp, canvas3dPtr->bindingTable,
			object, argv[3]);
		Tcl_ResetResult(interp);
		Tcl_AppendResult(interp, "requested illegal events; ",
			"only key, button, motion, and enter/leave ",
			"events may be used", (char *) NULL);
		goto error;
	    }
	} else if (argc == 4) {
	    char *command;
    
	    command = Tk_GetBinding(interp, canvas3dPtr->bindingTable,
		    object, argv[3]);
	    if (command == NULL) {
		goto error;
	    }
	    interp->result = command;
	} else {
	    Tk_GetAllBindings(interp, canvas3dPtr->bindingTable, object);
	}
    } else if ((c == 'c') && (strcmp(argv[1], "canvas3dx") == 0)) {
	int x;
	double grid;

	if ((argc < 3) || (argc > 4)) {
	    Tcl_AppendResult(interp, "wrong # args: should be \"",
		    argv[0], " canvas3dx screenx ?gridspacing?\"",
		    (char *) NULL);
	    goto error;
	}
	if (Tk_GetPixels(interp, canvas3dPtr->tkwin, argv[2], &x) != TCL_OK) {
	    goto error;
	}
	if (argc == 4) {
	    if (Tk_CanvasGetCoord(interp, (Tk_Canvas3d) canvas3dPtr, argv[3],
		    &grid) != TCL_OK) {
		goto error;
	    }
	} else {
	    grid = 0.0;
	}
	x += canvas3dPtr->xOrigin;
	Tcl_PrintDouble(interp, GridAlign((double) x, grid), interp->result);
    } else if ((c == 'c') && (strcmp(argv[1], "canvas3dy") == 0)) {
	int y;
	double grid;

	if ((argc < 3) || (argc > 4)) {
	    Tcl_AppendResult(interp, "wrong # args: should be \"",
		    argv[0], " canvas3dy screeny ?gridspacing?\"",
		    (char *) NULL);
	    goto error;
	}
	if (Tk_GetPixels(interp, canvas3dPtr->tkwin, argv[2], &y) != TCL_OK) {
	    goto error;
	}
	if (argc == 4) {
	    if (Tk_CanvasGetCoord(interp, (Tk_Canvas3d) canvas3dPtr,
		    argv[3], &grid) != TCL_OK) {
		goto error;
	    }
	} else {
	    grid = 0.0;
	}
	y += canvas3dPtr->yOrigin;
	Tcl_PrintDouble(interp, GridAlign((double) y, grid), interp->result);
    } else if ((c == 'c') && (strncmp(argv[1], "clears", length) == 0)
	    && (length >= 3)) {
	int clearsFlag;

	if (argc != 3) {
	    Tcl_AppendResult(interp, "wrong # args: should be \"",
		    argv[0], " clears onOrOff\"",
		    (char *) NULL);
	    goto error;
	}
	if (GE_GetConstBoolean(interp, argv[2], &clearsFlag) != TCL_OK){
	    goto error;
	}
	GE_ViewClears(canvas3dPtr->facePtr, clearsFlag);
    } else if ((c == 'c') && (strncmp(argv[1], "configure", length) == 0)
	    && (length >= 3)) {
	if (argc == 2) {
	    result = Tk_ConfigureInfo(interp, canvas3dPtr->tkwin, configSpecs,
		    (char *) canvas3dPtr, (char *) NULL, 0);
	} else if (argc == 3) {
	    result = Tk_ConfigureInfo(interp, canvas3dPtr->tkwin, configSpecs,
		    (char *) canvas3dPtr, argv[2], 0);
	} else {
	    result = ConfigureCanvas3d(interp, canvas3dPtr, argc-2, argv+2,
		    TK_CONFIG_ARGV_ONLY);
	}
    }  else if ((c == 'c') && (strncmp(argv[1], "create", length) == 0)
	    && (length >= 2)) {
	Tk_Item3dType *typePtr;
	Tk_Item3dType *matchPtr = NULL;
	Tk_Item3d *itemPtr;

	if (argc < 3) {
	    Tcl_AppendResult(interp, "wrong # args: should be \"",
		    argv[0], " create type ?arg arg ...?\"", (char *) NULL);
	    goto error;
	}
	c = argv[2][0];
	length = strlen(argv[2]);
	/* Whether an valid item name is provided... */
	for (typePtr = typeList; typePtr != NULL; typePtr = typePtr->nextPtr) {
	    if ((c == typePtr->name[0])
		    && (strncmp(argv[2], typePtr->name, length) == 0)) {
		if (matchPtr != NULL) {
		    badType:
		    Tcl_AppendResult(interp,
			    "unknown or ambiguous item type \"",
			    argv[2], "\"", (char *) NULL);
		    goto error;
		}
		matchPtr = typePtr;
	    }
	}
	if (matchPtr == NULL) {
	    goto badType;
	}
	typePtr = matchPtr;
	itemPtr = (Tk_Item3d *) ckalloc((unsigned) typePtr->itemSize + sizeof(int));
	itemPtr->id = canvas3dPtr->nextId;
	canvas3dPtr->nextId++;
	itemPtr->tagPtr = itemPtr->staticTagSpace;
	itemPtr->tagSpace = TK_TAG_SPACE;
	itemPtr->numTags = 0;
	itemPtr->typePtr = typePtr;
	if ((*typePtr->createProc)(interp, (Tk_Canvas3d) canvas3dPtr,
		itemPtr, argc-3, argv+3) != TCL_OK) {
	    ckfree((char *) itemPtr);
	    goto error;
	}
	itemPtr->nextPtr = NULL;
	canvas3dPtr->hotPtr = itemPtr;
	canvas3dPtr->hotPrevPtr = canvas3dPtr->lastItemPtr;
	if (canvas3dPtr->lastItemPtr == NULL) {
	    canvas3dPtr->firstItemPtr = itemPtr;
	} else {
	    canvas3dPtr->lastItemPtr->nextPtr = itemPtr;
	}
	canvas3dPtr->lastItemPtr = itemPtr;
	Tk_Canvas3dEventuallyRedraw((Tk_Canvas3d) canvas3dPtr,
		itemPtr->x1, itemPtr->y1, itemPtr->x2, itemPtr->y2);
	canvas3dPtr->flags |= REPICK_NEEDED;
	sprintf(interp->result, "%d", itemPtr->id);
    }  else if ((c == 'd') && (strncmp(argv[1], "delete", length) == 0)
	    && (length >= 2)) {
	int i;

	for (i = 2; i < argc; i++) {
	    for (itemPtr = StartTagSearch(canvas3dPtr, argv[i], &search);
		itemPtr != NULL; itemPtr = NextItem(&search)) {
		Tk_Canvas3dEventuallyRedraw((Tk_Canvas3d) canvas3dPtr,
			itemPtr->x1, itemPtr->y1, itemPtr->x2, itemPtr->y2);
		if (canvas3dPtr->bindingTable != NULL) {
		    Tk_DeleteAllBindings(canvas3dPtr->bindingTable,
			    (ClientData) itemPtr);
		}
		(*itemPtr->typePtr->deleteProc)((Tk_Canvas3d) canvas3dPtr, itemPtr);
		if (itemPtr->tagPtr != itemPtr->staticTagSpace) {
		    ckfree((char *) itemPtr->tagPtr);
		}
		if (search.prevPtr == NULL) {
		    canvas3dPtr->firstItemPtr = itemPtr->nextPtr;
		    if (canvas3dPtr->firstItemPtr == NULL) {
			canvas3dPtr->lastItemPtr = NULL;
		    }
		} else {
		    search.prevPtr->nextPtr = itemPtr->nextPtr;
		}
		if (canvas3dPtr->lastItemPtr == itemPtr) {
		    canvas3dPtr->lastItemPtr = search.prevPtr;
		}
		ckfree((char *) itemPtr);
		if (itemPtr == canvas3dPtr->currentItemPtr) {
		    canvas3dPtr->currentItemPtr = NULL;
		    canvas3dPtr->flags |= REPICK_NEEDED;
		}
		if (itemPtr == canvas3dPtr->newCurrentPtr) {
		    canvas3dPtr->newCurrentPtr = NULL;
		    canvas3dPtr->flags |= REPICK_NEEDED;
		}
		if ((itemPtr == canvas3dPtr->hotPtr)
			|| (itemPtr == canvas3dPtr->hotPrevPtr)) {
		    canvas3dPtr->hotPtr = NULL;
		}
	    }
	}
    } else if ((c == 'd') && (strncmp(argv[1], "dither", length) == 0)
	    && (length >= 3)) {
	int ditherFlag;

	if (argc != 3) {
	    Tcl_AppendResult(interp, "wrong # args: should be \"",
		    argv[0], " dither onOrOff\"",
		    (char *) NULL);
	    goto error;
	}
	if (GE_GetConstBoolean(interp, argv[2], &ditherFlag) != TCL_OK){
	    goto error;
	}
	GE_RenderDither(canvas3dPtr->facePtr, ditherFlag);
    } else if ((c == 'd') && (strncmp(argv[1], "dtag", length) == 0)
	    && (length >= 2)) {
	Tk_Uid tag;
	int i;

	if ((argc != 3) && (argc != 4)) {
	    Tcl_AppendResult(interp, "wrong # args: should be \"",
		    argv[0], " dtag tagOrId ?tagToDelete?\"",
		    (char *) NULL);
	    goto error;
	}
	if (argc == 4) {
	    tag = Tk_GetUid(argv[3]);
	} else {
	    tag = Tk_GetUid(argv[2]);
	}
	for (itemPtr = StartTagSearch(canvas3dPtr, argv[2], &search);
		itemPtr != NULL; itemPtr = NextItem(&search)) {
	    for (i = itemPtr->numTags-1; i >= 0; i--) {
		if (itemPtr->tagPtr[i] == tag) {
		    itemPtr->tagPtr[i] = itemPtr->tagPtr[itemPtr->numTags-1];
		    itemPtr->numTags--;
		}
	    }
	}
    } else if ((c == 'f') && (strncmp(argv[1], "fill", length) == 0)
	    && (length >= 3)) {
	int fillType;

	if (argc != 4) {
	    Tcl_AppendResult(interp, "wrong # args: should be \"",
		    argv[0], " fill tagOrId fillType\"",
		    (char *) NULL);
	    goto error;
	}
	/* Parse the fill type string. */
	if (GE_GetConstFill(interp, argv[3], &fillType) != TCL_OK){
	    goto error;
	}
	for (itemPtr = StartTagSearch(canvas3dPtr, argv[2], &search);
		itemPtr != NULL; itemPtr = NextItem(&search)) {
	    if (*itemPtr->typePtr->fillProc == NULL) {
		continue;
	    }
	    (void) (*itemPtr->typePtr->fillProc)((Tk_Canvas) canvas3dPtr,
		    itemPtr, fillType);
	   }
    } else if ((c == 'f') && (strncmp(argv[1], "filter", length) == 0)
	    && (length >= 3)) {
	int filterType;

	if (argc != 3) {
	    Tcl_AppendResult(interp, "wrong # args: should be \"",
		    argv[0], " filter filterType\"",
		    (char *) NULL);
	    goto error;
	}
	if (GE_GetConstFilter(interp, argv[2], &filterType) != TCL_OK){
	    goto error;
	}
	GE_RenderFilter(canvas3dPtr->facePtr, filterType);
    }  else if ((c == 'g') && (strncmp(argv[1], "gettags", length) == 0)) {
	if (argc != 3) {
	    Tcl_AppendResult(interp, "wrong # args: should be \"",
		    argv[0], " gettags tagOrId\"", (char *) NULL);
	    goto error;
	}
	itemPtr = StartTagSearch(canvas3dPtr, argv[2], &search);
	if (itemPtr != NULL) {
	    int i;
	    for (i = 0; i < itemPtr->numTags; i++) {
		Tcl_AppendElement(interp, (char *) itemPtr->tagPtr[i]);
	    }
	}
    } else if ((c == 'f') && (strncmp(argv[1], "fog", length) == 0)
	    && (length >= 3)) {
	int fogFlag;

	if (argc != 3) {
	    Tcl_AppendResult(interp, "wrong # args: should be \"",
		    argv[0], " fog onOrOff\"",
		    (char *) NULL);
	    goto error;
	}
	if (GE_GetConstBoolean(interp, argv[2], &fogFlag) != TCL_OK){
	    goto error;
	}
	GE_RenderFog(canvas3dPtr->facePtr, fogFlag);
    } else if ((c == 'i') && (strncmp(argv[1], "itemconfigure", length) == 0)
	    && (length >= 6)) {
	if (argc < 3) {
	    Tcl_AppendResult(interp, "wrong # args: should be \"",
		    argv[0], " itemconfigure tagOrId ?option value ...?\"",
		    (char *) NULL);
	    goto error;
	}
	for (itemPtr = StartTagSearch(canvas3dPtr, argv[2], &search);
		itemPtr != NULL; itemPtr = NextItem(&search)) {
	    if (argc == 3) {
		result = Tk_ConfigureInfo(canvas3dPtr->interp, canvas3dPtr->tkwin,
			itemPtr->typePtr->configSpecs, (char *) itemPtr,
			(char *) NULL, 0);
	    } else if (argc == 4) {
		result = Tk_ConfigureInfo(canvas3dPtr->interp, canvas3dPtr->tkwin,
			itemPtr->typePtr->configSpecs, (char *) itemPtr,
			argv[3], 0);
	    } else {
		Tk_Canvas3dEventuallyRedraw((Tk_Canvas3d) canvas3dPtr,
			itemPtr->x1, itemPtr->y1, itemPtr->x2, itemPtr->y2);
		result = (*itemPtr->typePtr->configProc)(interp,
			(Tk_Canvas3d) canvas3dPtr, itemPtr, argc-3, argv+3,
			TK_CONFIG_ARGV_ONLY);
		Tk_Canvas3dEventuallyRedraw((Tk_Canvas3d) canvas3dPtr,
			itemPtr->x1, itemPtr->y1, itemPtr->x2, itemPtr->y2);
		canvas3dPtr->flags |= REPICK_NEEDED;
	    }
	    if ((result != TCL_OK) || (argc < 5)) {
		break;
	    }
	}
    } else if ((c == 'p') && (strncmp(argv[1], "perspCor", length) == 0)
	    && (length >= 3)) {
	int perspFlag;

	if (argc != 3) {
	    Tcl_AppendResult(interp, "wrong # args: should be \"",
		    argv[0], " perspCor onOrOff\"",
		    (char *) NULL);
	    goto error;
	}
	if (GE_GetConstBoolean(interp, argv[2], &perspFlag) != TCL_OK){
	    goto error;
	}
	GE_RenderPerspCor(canvas3dPtr->facePtr, perspFlag);
    } else if ((c == 'r') && (strncmp(argv[1], "rotate", length) == 0)
	    && (length >= 3)) {
	double angle;
	int axis;
	
	if (argc != 5) {
	    Tcl_AppendResult(interp, "wrong # args: should be \"",
		    argv[0], " rotate tagOrId axis angle\"",
		    (char *) NULL);
	    goto error;
	}
	if ((GE_GetConstAxis(interp, argv[3], &axis) != TCL_OK)
		|| (Tcl_GetDouble(interp, argv[4], &angle) != TCL_OK)) {
	    goto error;
	}
	if ((angle == 0.0)) {
	    goto error;
	}
	for (itemPtr = StartTagSearch(canvas3dPtr, argv[2], &search);
		itemPtr != NULL; itemPtr = NextItem(&search)) {
	    (void) (*itemPtr->typePtr->rotateProc)((Tk_Canvas) canvas3dPtr,
		    itemPtr, axis, angle);
	   }
    } else if ((c == 's') && (strncmp(argv[1], "scale", length) == 0)
	    && (length >= 3)) {
	double xScale, yScale, zScale;

	if (argc != 6) {
	    Tcl_AppendResult(interp, "wrong # args: should be \"",
		    argv[0], " scale tagOrId xScale yScale zScale\"",
		    (char *) NULL);
	    goto error;
	}
	if ((Tcl_GetDouble(interp, argv[3], &xScale) != TCL_OK)
		|| (Tcl_GetDouble(interp, argv[4], &yScale) != TCL_OK)
		|| (Tcl_GetDouble(interp, argv[5], &zScale) != TCL_OK)) {
	    goto error;
	}
	for (itemPtr = StartTagSearch(canvas3dPtr, argv[2], &search);
		itemPtr != NULL; itemPtr = NextItem(&search)) {
	    (void) (*itemPtr->typePtr->scaleProc)((Tk_Canvas) canvas3dPtr,
		    itemPtr, xScale, yScale, zScale);
	   }
    }  else if ((c == 's') && (strncmp(argv[1], "shade", length) == 0)
	    && (length >= 3)) {
	int shadeType;

	if (argc != 3) {
	    Tcl_AppendResult(interp, "wrong # args: should be \"",
		    argv[0], " shade shadeType\"",
		    (char *) NULL);
	    goto error;
	}
	if (GE_GetConstShade(interp, argv[2], &shadeType) != TCL_OK){
	    goto error;
	}
	GE_RenderShade(canvas3dPtr->facePtr, shadeType);
    } else if ((c == 't') && (strncmp(argv[1], "transform", length) == 0)) {
	int matrixId;

	if (argc != 4) {
	    Tcl_AppendResult(interp, "wrong # args: should be \"",
		    argv[0], " transform tagOrId matrixId\"",
		    (char *) NULL);
	    goto error;
	}
	if (Tcl_GetInt(interp, argv[3], &matrixId) != TCL_OK) {
	    goto error;
	}
	for (itemPtr = StartTagSearch(canvas3dPtr, argv[2], &search);
		itemPtr != NULL; itemPtr = NextItem(&search)) {
	    (void) (*itemPtr->typePtr->transformProc)((Tk_Canvas3d) canvas3dPtr,
		    itemPtr,  matrixId);
	}
    } else if ((c == 't') && (strncmp(argv[1], "translate", length) == 0)) {
	double xAmount, yAmount, zAmount;

	if (argc != 6) {
	    Tcl_AppendResult(interp, "wrong # args: should be \"",
		    argv[0], " translate tagOrId xAmount yAmount zAmount\"",
		    (char *) NULL);
	    goto error;
	}
	if ((Tcl_GetDouble(interp, argv[3], &xAmount) != TCL_OK) 
		|| (Tcl_GetDouble(interp, argv[4], &yAmount) != TCL_OK)
		||(Tcl_GetDouble(interp, argv[5], &zAmount) != TCL_OK)) {
	    goto error;
	}
	for (itemPtr = StartTagSearch(canvas3dPtr, argv[2], &search);
		itemPtr != NULL; itemPtr = NextItem(&search)) {
	    (void) (*itemPtr->typePtr->translateProc)((Tk_Canvas3d) canvas3dPtr,
		    itemPtr,  xAmount, yAmount, zAmount);
	}
    } else if ((c == 'u') && (strncmp(argv[1], "update", length) == 0)) {
	    XEvent *eventPtr = (XEvent *)ckalloc(sizeof(XEvent));
	    GE_UpdateClears(canvas3dPtr);
	    eventPtr->type = ConfigureNotify;
	    Canvas3dEventProc(canvas3dPtr, eventPtr);
    } else if ((c == 'w') && (strncmp(argv[1], "wrap", length) == 0)
	    && (length >= 3)) {
	int wrapStyle;
	
	if (argc != 4) {
	    Tcl_AppendResult(interp, "wrong # args: should be \"",
		    argv[0], " wrap tagOrId wrap_style\"",
		    (char *) NULL);
	    goto error;
	}
	if (GE_GetConstWrap(interp, argv[3], &wrapStyle) != TCL_OK){
	    goto error;
	}
	for (itemPtr = StartTagSearch(canvas3dPtr, argv[2], &search);
		itemPtr != NULL; itemPtr = NextItem(&search)) {
	    if (*itemPtr->typePtr->wrapProc == NULL) {
		continue;
	    }
	    (void) (*itemPtr->typePtr->wrapProc)((Tk_Canvas) canvas3dPtr,
		    itemPtr, wrapStyle);
	   }
    } else if ((c == 'z') && (strncmp(argv[1], "zBuffers", length) == 0)
	    && (length >= 3)) {
	int zFlag;

	if (argc != 3) {
	    Tcl_AppendResult(interp, "wrong # args: should be \"",
		    argv[0], " zBuffers onOrOff\"",
		    (char *) NULL);
	    goto error;
	}
	if (GE_GetConstBoolean(interp, argv[2], &zFlag) != TCL_OK){
	    goto error;
	}
	GE_RenderZbuffer(canvas3dPtr->facePtr, zFlag);
    } else {
	Tcl_AppendResult(interp, "bad option \"", argv[1],
		"\": must be addtag, bind, ",
		"canvas3dx, canvas3dy, configure, create, ",
		"dchars, delete, dtag, ",
		"gettags, itemconfigure, ",
		"rotate, scale, ",
		"transform, translate, update",
		(char *) NULL);  
	goto error;
    }
    done:
    Tcl_Release((ClientData) canvas3dPtr);
    return result;

    error:
    Tcl_Release((ClientData) canvas3dPtr);
    return TCL_ERROR;
}
 
/*
 *----------------------------------------------------------------------
 *
 * DestroyCanvas3d --
 *
 *	This procedure is invoked by Tcl_EventuallyFree or Tcl_Release
 *	to clean up the internal structure of a canvas3d at a safe time
 *	(when no-one is using it anymore).
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	Everything associated with the canvas3d is freed up.
 *
 *----------------------------------------------------------------------
 */

static void
DestroyCanvas3d(memPtr)
    char *memPtr;		/* Info about canvas3d widget. */
{
    TkCanvas3d *canvas3dPtr = (TkCanvas3d *) memPtr;
    Tk_Item3d *itemPtr;

    /*
     * Free up all of the items in the canvas3d.
     */

    for (itemPtr = canvas3dPtr->firstItemPtr; itemPtr != NULL;
	    itemPtr = canvas3dPtr->firstItemPtr) {
	canvas3dPtr->firstItemPtr = itemPtr->nextPtr;
	(*itemPtr->typePtr->deleteProc)((Tk_Canvas3d) canvas3dPtr, itemPtr);
	if (itemPtr->tagPtr != itemPtr->staticTagSpace) {
	    ckfree((char *) itemPtr->tagPtr);
	}
	ckfree((char *) itemPtr);
    }

    /*
     * Free up all the stuff that requires special handling,
     * then let Tk_FreeOptions handle all the standard option-related
     * stuff.
     */
    if (canvas3dPtr->bindingTable != NULL) {
	Tk_DeleteBindingTable(canvas3dPtr->bindingTable);
    }
    Tk_FreeOptions(configSpecs, (char *) canvas3dPtr, canvas3dPtr->display, 0);
    ckfree((char *) canvas3dPtr);
}
 
/*
 *----------------------------------------------------------------------
 *
 * ConfigureCanvas3d --
 *
 *	This procedure is called to process an argv/argc list, plus
 *	the Tk option database, in order to configure (or
 *	reconfigure) a canvas3d widget.
 *
 * Results:
 *	The return value is a standard Tcl result.  If TCL_ERROR is
 *	returned, then interp->result contains an error message.
 *
 * Side effects:
 *	Configuration information, such as colors, border width,
 *	etc. get set for canvas3dPtr;  old resources get freed,
 *	if there were any.
 *
 *----------------------------------------------------------------------
 */

static int
ConfigureCanvas3d(interp, canvas3dPtr, argc, argv, flags)
    Tcl_Interp *interp;		/* Used for error reporting. */
    TkCanvas3d *canvas3dPtr;	/* Information about widget;  may or may
				 * not already have values for some fields. */
    int argc;			/* Number of valid entries in argv. */
    char **argv;		/* Arguments. */
    int flags;			/* Flags to pass to Tk_ConfigureWidget. */
{
    XEvent *eventPtr = (XEvent *)ckalloc(sizeof(XEvent));

    if (Tk_ConfigureWidget(interp, canvas3dPtr->tkwin, configSpecs,
	    argc, argv, (char *) canvas3dPtr, flags) != TCL_OK) {
	return TCL_ERROR;
    }

    /*
     * Reset the desired dimensions for the window.
     */

    Tk_GeometryRequest(canvas3dPtr->tkwin, canvas3dPtr->width + 2*canvas3dPtr->inset,
	    canvas3dPtr->height + 2*canvas3dPtr->inset);
    if (canvas3dPtr->facePtr->d3dapp != NULL) {
	GE_DestroyAllItems(canvas3dPtr);
	if (!D3DAppWindow(canvas3dPtr->width + 2*canvas3dPtr->inset,
	    canvas3dPtr->height + 2*canvas3dPtr->inset)) {
		//error
                        }
	GE_ReCreateAllItems(interp, canvas3dPtr);
	Tcl_DoWhenIdle(DisplayCanvas3d, (ClientData) canvas3dPtr);
	/*GE_UpdateClears(canvas3dPtr);
	eventPtr->type = ConfigureNotify;
	Canvas3dEventProc(canvas3dPtr, eventPtr);*/
    }
 
    return TCL_OK;
}
 
/*
 *--------------------------------------------------------------
 *
 * DisplayCanvas3d --
 *
 *	This procedure redraws the contents of a canvas3d window.
 *	It is invoked as a do-when-idle handler, so it only runs
 *	when there's nothing else for the application to do.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	Information appears on the screen.
 *
 *--------------------------------------------------------------
 */

static void
DisplayCanvas3d(clientData)
    ClientData clientData;	/* Information about widget. */
{
    TkCanvas3d *canvas3dPtr = (TkCanvas3d *) clientData;
    Tk_Window tkwin = canvas3dPtr->tkwin;
    Tk_Item3d *itemPtr;
	
    /************************************************************************/
    /*
     * Initialize the graphics engine.
     */
    if (canvas3dPtr->facePtr->d3dapp == NULL) {
        GE_Init(canvas3dPtr);
    }
    	/*
	 * Scan through the item list, redrawing those items that need it.
	 * An item must be redraw if either (a) it intersects the smaller
	 * on-screen area or (b) it intersects the full canvas3d area and its
	 * type requests that it be redrawn always (e.g. so subwindows can
	 * be unmapped when they move off-screen).
	 */
    
	for (itemPtr = canvas3dPtr->firstItemPtr; itemPtr != NULL;
		itemPtr = itemPtr->nextPtr) {	    
	    /* Items such as matrix, surface, etc.do not have displayProc. */
	    if (itemPtr->typePtr->displayProc == NULL) {
		continue;
	    }
	    (*itemPtr->typePtr->displayProc)((Tk_Canvas3d) canvas3dPtr, itemPtr );

	}
        /************************************************************************/

	
}
 
/*
 *--------------------------------------------------------------
 *
 * Canvas3dEventProc --
 *
 *	This procedure is invoked by the Tk dispatcher for various
 *	events on canvas3ds.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	When the window gets deleted, internal structures get
 *	cleaned up.  When it gets exposed, it is redisplayed.
 *
 *--------------------------------------------------------------
 */

static void
Canvas3dEventProc(clientData, eventPtr)
    ClientData clientData;	/* Information about window. */
    XEvent *eventPtr;		/* Information about event. */
{
    TkCanvas3d *canvas3dPtr = (TkCanvas3d *) clientData;
    
    HWND hWnd;
    Tk_Window tkwin = canvas3dPtr->tkwin;

    if (eventPtr->type == Expose) {
	int x, y;

	x = eventPtr->xexpose.x + canvas3dPtr->xOrigin;
	y = eventPtr->xexpose.y + canvas3dPtr->yOrigin;
	Tk_Canvas3dEventuallyRedraw((Tk_Canvas3d) canvas3dPtr, x, y,
		x + eventPtr->xexpose.width,
		y + eventPtr->xexpose.height);
	if ((eventPtr->xexpose.x < canvas3dPtr->inset)
		|| (eventPtr->xexpose.y < canvas3dPtr->inset)
		|| ((eventPtr->xexpose.x + eventPtr->xexpose.width)
		    > (Tk_Width(canvas3dPtr->tkwin) - canvas3dPtr->inset))
		|| ((eventPtr->xexpose.y + eventPtr->xexpose.height)
		    > (Tk_Height(canvas3dPtr->tkwin) - canvas3dPtr->inset))) {
	    canvas3dPtr->flags |= REDRAW_BORDERS;

    /***********************Maybe junk, Maybe useful!**********************/
	    //in Expose
	    hWnd = TkWinGetHWND(Tk_WindowId(canvas3dPtr->tkwin));
	    WindowProc(hWnd,WM_NCPAINT,0L, 0L );
    /***********************Maybe junk, Maybe useful!**********************/
	}
    } else if (eventPtr->type == DestroyNotify) {
	if (canvas3dPtr->tkwin != NULL) {
	    canvas3dPtr->tkwin = NULL;
	    Tcl_DeleteCommand(canvas3dPtr->interp,
		    Tcl_GetCommandName(canvas3dPtr->interp,
		    canvas3dPtr->widgetCmd));
	}
	if (canvas3dPtr->flags & REDRAW_PENDING) {
	    Tcl_CancelIdleCall(DisplayCanvas3d, (ClientData) canvas3dPtr);
	}
	Tcl_EventuallyFree((ClientData) canvas3dPtr, DestroyCanvas3d);
    } else if (eventPtr->type == ConfigureNotify) {
	

	Canvas3dSetOrigin(canvas3dPtr, canvas3dPtr->xOrigin, canvas3dPtr->yOrigin);
	Tk_Canvas3dEventuallyRedraw((Tk_Canvas3d) canvas3dPtr, canvas3dPtr->xOrigin,
		canvas3dPtr->yOrigin,
		canvas3dPtr->xOrigin + Tk_Width(canvas3dPtr->tkwin),
		canvas3dPtr->yOrigin + Tk_Height(canvas3dPtr->tkwin));
	canvas3dPtr->flags |= REDRAW_BORDERS;

	/***********************Maybe junk, Maybe useful!**********************/
	    // in Configure
	    hWnd =  TkWinGetHWND(Tk_WindowId(canvas3dPtr->tkwin));
	    WindowProc(hWnd,WM_NCPAINT,0L, 0L );

	/**************************************************************/
    }  else if (eventPtr->type == UnmapNotify) {
	Tk_Item3d *itemPtr;

	/*
	 * Special hack:  if the canvas3d is unmapped, then must notify
	 * all items with "alwaysRedraw" set, so that they know that
	 * they are no longer displayed.
	 */

	for (itemPtr = canvas3dPtr->firstItemPtr; itemPtr != NULL;
		itemPtr = itemPtr->nextPtr) {
		(*itemPtr->typePtr->displayProc)((Tk_Canvas3d) canvas3dPtr,
			itemPtr);
	}
    }
}
 
/*
 *----------------------------------------------------------------------
 *
 * Canvas3dCmdDeletedProc --
 *
 *	This procedure is invoked when a widget command is deleted.  If
 *	the widget isn't already in the process of being destroyed,
 *	this command destroys it.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	The widget is destroyed.
 *
 *----------------------------------------------------------------------
 */

static void
Canvas3dCmdDeletedProc(clientData)
    ClientData clientData;	/* Pointer to widget record for widget. */
{
    TkCanvas3d *canvas3dPtr = (TkCanvas3d *) clientData;
    Tk_Window tkwin = canvas3dPtr->tkwin;

    /*
     * This procedure could be invoked either because the window was
     * destroyed and the command was then deleted (in which case tkwin
     * is NULL) or because the command was deleted, and then this procedure
     * destroys the widget.
     */

    if (tkwin != NULL) {
	canvas3dPtr->tkwin = NULL;
	Tk_DestroyWindow(tkwin);
    }
}
 
/*
 *--------------------------------------------------------------
 *
 * Tk_Canvas3dEventuallyRedraw --
 *
 *	Arrange for part or all of a canvas3d widget to redrawn at
 *	some convenient time in the future.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	The screen will eventually be refreshed.
 *
 *--------------------------------------------------------------
 */

void
Tk_Canvas3dEventuallyRedraw(canvas3d, x1, y1, x2, y2)
    Tk_Canvas3d canvas3d;	/* Information about widget. */
    int x1, y1;			/* Upper left corner of area to redraw.
				 * Pixels on edge are redrawn. */
    int x2, y2;			/* Lower right corner of area to redraw.
				 * Pixels on edge are not redrawn. */
{
    TkCanvas3d *canvas3dPtr = (TkCanvas3d *) canvas3d;
    
    Tcl_DoWhenIdle(DisplayCanvas3d, (ClientData) canvas3dPtr);	
}
 
/*
 *--------------------------------------------------------------
 *
 * Tk_CreateItemType --
 *
 *	This procedure may be invoked to add a new kind of canvas
 *	element to the core item types supported by Tk.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	From now on, the new item type will be useable in canvas
 *	widgets (e.g. typePtr->name can be used as the item type
 *	in "create" widget commands).  If there was already a
 *	type with the same name as in typePtr, it is replaced with
 *	the new type.
 *
 *--------------------------------------------------------------
 */

void
Tk_CreateItem3dType(typePtr)
    Tk_Item3dType *typePtr;		/* Information about item type;
					 * storage must be statically
					 * allocated (must live forever). */
{
    Tk_Item3dType *typePtr2, *prevPtr;

    if (typeList == NULL) {
	InitCanvas3d();
    }

    /*
     * If there's already an item type with the given name, remove it.
     */

    for (typePtr2 = typeList, prevPtr = NULL; typePtr2 != NULL;
	    prevPtr = typePtr2, typePtr2 = typePtr2->nextPtr) {
	if (strcmp(typePtr2->name, typePtr->name) == 0) {
	    if (prevPtr == NULL) {
		typeList = typePtr2->nextPtr;
	    } else {
		prevPtr->nextPtr = typePtr2->nextPtr;
	    }
	    break;
	}
    }
    typePtr->nextPtr = typeList;
    typeList = typePtr;
}
 
/*
 *----------------------------------------------------------------------
 *
 * Tk_GetItem3dTypes --
 *
 *	This procedure returns a pointer to the list of all item
 *	types.
 *
 * Results:
 *	The return value is a pointer to the first in the list
 *	of item types currently supported by canvases.
 *
 * Side effects:
 *	None.
 *
 *----------------------------------------------------------------------
 */

Tk_Item3dType *
Tk_GetItem3dTypes()
{
    if (typeList == NULL) {
	InitCanvas3d();
    }
    return typeList;
}
 
/*
 *--------------------------------------------------------------
 *
 * InitCanvas3d --
 *
 *	This procedure is invoked to perform once-only-ever
 *	initialization for the module, such as setting up
 *	the type table.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	None.
 *
 *--------------------------------------------------------------
 */

static void
InitCanvas3d()
{
    if (typeList != NULL) {
	return;
    }
    typeList = &tkBoxType;
    tkBoxType.nextPtr = &tkConeType;
    tkConeType.nextPtr = &tkCylinType;
    tkCylinType.nextPtr = &tkSphrType;
    tkSphrType.nextPtr = &tkPolygonType;
    tkPolygonType.nextPtr = &tkMeshType;
    tkMeshType.nextPtr = &tkMatrixType;
    tkMatrixType.nextPtr = &tkSurfType;
    tkSurfType.nextPtr = &tkLightType;
    tkLightType.nextPtr = &tkViewType;
    tkViewType.nextPtr = NULL;

    allUid = Tk_GetUid("all");
    currentUid = Tk_GetUid("current");
}
 
/*
 *--------------------------------------------------------------
 *
 * StartTagSearch --
 *
 *	This procedure is called to initiate an enumeration of
 *	all items in a given canvas3d that contain a given tag.
 *
 * Results:
 *	The return value is a pointer to the first item in
 *	canvas3dPtr that matches tag, or NULL if there is no
 *	such item.  The information at *searchPtr is initialized
 *	such that successive calls to NextItem will return
 *	successive items that match tag.
 *
 * Side effects:
 *	SearchPtr is linked into a list of searches in progress
 *	on canvas3dPtr, so that elements can safely be deleted
 *	while the search is in progress.  EndTagSearch must be
 *	called at the end of the search to unlink searchPtr from
 *	this list.
 *
 *--------------------------------------------------------------
 */

static Tk_Item3d *
StartTagSearch(canvas3dPtr, tag, searchPtr)
    TkCanvas3d *canvas3dPtr;		/* Canvas3d whose items are to be
					 * searched. */
    char *tag;				/* String giving tag value. */
    TagSearch *searchPtr;		/* Record describing tag search;
					 * will be initialized here. */
{
    int id;
    Tk_Item3d *itemPtr, *prevPtr;
    Tk_Uid *tagPtr;
    Tk_Uid uid;
    int count;

    /*
     * Initialize the search.
     */

    searchPtr->canvas3dPtr = canvas3dPtr;
    searchPtr->searchOver = 0;

    /*
     * Find the first matching item in one of several ways. If the tag
     * is a number then it selects the single item with the matching
     * identifier.  In this case see if the item being requested is the
     * hot item, in which case the search can be skipped.
     */

    if (isdigit(UCHAR(*tag))) {
	char *end;

	numIdSearches++;
	id = strtoul(tag, &end, 0);
	if (*end == 0) {
	    itemPtr = canvas3dPtr->hotPtr;
	    prevPtr = canvas3dPtr->hotPrevPtr;
	    if ((itemPtr == NULL) || (itemPtr->id != id) || (prevPtr == NULL)
		    || (prevPtr->nextPtr != itemPtr)) {
		numSlowSearches++;
		for (prevPtr = NULL, itemPtr = canvas3dPtr->firstItemPtr;
			itemPtr != NULL;
			prevPtr = itemPtr, itemPtr = itemPtr->nextPtr) {
		    if (itemPtr->id == id) {
			break;
		    }
		}
	    }
	    searchPtr->prevPtr = prevPtr;
	    searchPtr->searchOver = 1;
	    canvas3dPtr->hotPtr = itemPtr;
	    canvas3dPtr->hotPrevPtr = prevPtr;
	    return itemPtr;
	}
    }

    searchPtr->tag = uid = Tk_GetUid(tag);
    if (uid == allUid) {

	/*
	 * All items match.
	 */

	searchPtr->tag = NULL;
	searchPtr->prevPtr = NULL;
	searchPtr->currentPtr = canvas3dPtr->firstItemPtr;
	return canvas3dPtr->firstItemPtr;
    }

    /*
     * None of the above.  Search for an item with a matching tag.
     */

    for (prevPtr = NULL, itemPtr = canvas3dPtr->firstItemPtr; itemPtr != NULL;
	    prevPtr = itemPtr, itemPtr = itemPtr->nextPtr) {
	for (tagPtr = itemPtr->tagPtr, count = itemPtr->numTags;
		count > 0; tagPtr++, count--) {
	    if (*tagPtr == uid) {
		searchPtr->prevPtr = prevPtr;
		searchPtr->currentPtr = itemPtr;
		return itemPtr;
	    }
	}
    }
    searchPtr->prevPtr = prevPtr;
    searchPtr->searchOver = 1;
    return NULL;
}
 
/*
 *--------------------------------------------------------------
 *
 * NextItem --
 *
 *	This procedure returns successive items that match a given
 *	tag;  it should be called only after StartTagSearch has been
 *	used to begin a search.
 *
 * Results:
 *	The return value is a pointer to the next item that matches
 *	the tag specified to StartTagSearch, or NULL if no such
 *	item exists.  *SearchPtr is updated so that the next call
 *	to this procedure will return the next item.
 *
 * Side effects:
 *	None.
 *
 *--------------------------------------------------------------
 */

static Tk_Item3d *
NextItem(searchPtr)
    TagSearch *searchPtr;		/* Record describing search in
					 * progress. */
{
    Tk_Item3d *itemPtr, *prevPtr;
    int count;
    Tk_Uid uid;
    Tk_Uid *tagPtr;

    /*
     * Find next item in list (this may not actually be a suitable
     * one to return), and return if there are no items left.
     */

    prevPtr = searchPtr->prevPtr;
    if (prevPtr == NULL) {
	itemPtr = searchPtr->canvas3dPtr->firstItemPtr;
    } else {
	itemPtr = prevPtr->nextPtr;
    }
    if ((itemPtr == NULL) || (searchPtr->searchOver)) {
	searchPtr->searchOver = 1;
	return NULL;
    }
    if (itemPtr != searchPtr->currentPtr) {
	/*
	 * The structure of the list has changed.  Probably the
	 * previously-returned item was removed from the list.
	 * In this case, don't advance prevPtr;  just return
	 * its new successor (i.e. do nothing here).
	 */
    } else {
	prevPtr = itemPtr;
	itemPtr = prevPtr->nextPtr;
    }

    /*
     * Handle special case of "all" search by returning next item.
     */

    uid = searchPtr->tag;
    if (uid == NULL) {
	searchPtr->prevPtr = prevPtr;
	searchPtr->currentPtr = itemPtr;
	return itemPtr;
    }

    /*
     * Look for an item with a particular tag.
     */

    for ( ; itemPtr != NULL; prevPtr = itemPtr, itemPtr = itemPtr->nextPtr) {
	for (tagPtr = itemPtr->tagPtr, count = itemPtr->numTags;
		count > 0; tagPtr++, count--) {
	    if (*tagPtr == uid) {
		searchPtr->prevPtr = prevPtr;
		searchPtr->currentPtr = itemPtr;
		return itemPtr;
	    }
	}
    }
    searchPtr->prevPtr = prevPtr;
    searchPtr->searchOver = 1;
    return NULL;
}
 
/*
 *--------------------------------------------------------------
 *
 * DoItem --
 *
 *	This is a utility procedure called by FindItems.  It
 *	either adds itemPtr's id to the result forming in interp,
 *	or it adds a new tag to itemPtr, depending on the value
 *	of tag.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	If tag is NULL then itemPtr's id is added as a list element
 *	to interp->result;  otherwise tag is added to itemPtr's
 *	list of tags.
 *
 *--------------------------------------------------------------
 */

static void
DoItem(interp, itemPtr, tag)
    Tcl_Interp *interp;			/* Interpreter in which to (possibly)
					 * record item id. */
    Tk_Item3d *itemPtr;			/* Item to (possibly) modify. */
    Tk_Uid tag;				/* Tag to add to those already
					 * present for item, or NULL. */
{
    Tk_Uid *tagPtr;
    int count;

    /*
     * Handle the "add-to-result" case and return, if appropriate.
     */

    if (tag == NULL) {
	char msg[30];
	sprintf(msg, "%d", itemPtr->id);
	Tcl_AppendElement(interp, msg);
	return;
    }

    for (tagPtr = itemPtr->tagPtr, count = itemPtr->numTags;
	    count > 0; tagPtr++, count--) {
	if (tag == *tagPtr) {
	    return;
	}
    }

    /*
     * Grow the tag space if there's no more room left in the current
     * block.
     */

    if (itemPtr->tagSpace == itemPtr->numTags) {
	Tk_Uid *newTagPtr;

	itemPtr->tagSpace += 5;
	newTagPtr = (Tk_Uid *) ckalloc((unsigned)
		(itemPtr->tagSpace * sizeof(Tk_Uid)));
	memcpy((VOID *) newTagPtr, (VOID *) itemPtr->tagPtr,
		(itemPtr->numTags * sizeof(Tk_Uid)));
	if (itemPtr->tagPtr != itemPtr->staticTagSpace) {
	    ckfree((char *) itemPtr->tagPtr);
	}
	itemPtr->tagPtr = newTagPtr;
	tagPtr = &itemPtr->tagPtr[itemPtr->numTags];
    }

    /*
     * Add in the new tag.
     */

    *tagPtr = tag;
    itemPtr->numTags++;
}
 
/*
 *--------------------------------------------------------------
 *
 * FindItems --
 *
 *	This procedure does all the work of implementing the
 *	"find" and "addtag" options of the canvas3d widget command,
 *	which locate items that have certain features (location,
 *	tags, position in display list, etc.).
 *
 * Results:
 *	A standard Tcl return value.  If newTag is NULL, then a
 *	list of ids from all the items that match argc/argv is
 *	returned in interp->result.  If newTag is NULL, then
 *	the normal interp->result is an empty string.  If an error
 *	occurs, then interp->result will hold an error message.
 *
 * Side effects:
 *	If newTag is non-NULL, then all the items that match the
 *	information in argc/argv have that tag added to their
 *	lists of tags.
 *
 *--------------------------------------------------------------
 */

static int
FindItems(interp, canvas3dPtr, argc, argv, newTag, cmdName, option)
    Tcl_Interp *interp;			/* Interpreter for error reporting. */
    TkCanvas3d *canvas3dPtr;		/* Canvas3d whose items are to be
					 * searched. */
    int argc;				/* Number of entries in argv.  Must be
					 * greater than zero. */
    char **argv;			/* Arguments that describe what items
					 * to search for (see user doc on
					 * "find" and "addtag" options). */
    char *newTag;			/* If non-NULL, gives new tag to set
					 * on all found items;  if NULL, then
					 * ids of found items are returned
					 * in interp->result. */
    char *cmdName;			/* Name of original Tcl command, for
					 * use in error messages. */
    char *option;			/* For error messages:  gives option
					 * from Tcl command and other stuff
					 * up to what's in argc/argv. */
{
    int c;
    size_t length;
    TagSearch search;
    Tk_Item3d *itemPtr;
    Tk_Uid uid;

    if (newTag != NULL) {
	uid = Tk_GetUid(newTag);
    } else {
	uid = NULL;
    }
    c = argv[0][0];
    length = strlen(argv[0]);
    if ((c == 'a') && (strncmp(argv[0], "above", length) == 0)
	    && (length >= 2)) {
	Tk_Item3d *lastPtr = NULL;
	if (argc != 2) {
	    Tcl_AppendResult(interp, "wrong # args: should be \"",
		    cmdName, option, " above tagOrId", (char *) NULL);
	    return TCL_ERROR;
	}
	for (itemPtr = StartTagSearch(canvas3dPtr, argv[1], &search);
		itemPtr != NULL; itemPtr = NextItem(&search)) {
	    lastPtr = itemPtr;
	}
	if ((lastPtr != NULL) && (lastPtr->nextPtr != NULL)) {
	    DoItem(interp, lastPtr->nextPtr, uid);
	}
    } else if ((c == 'a') && (strncmp(argv[0], "all", length) == 0)
	    && (length >= 2)) {
	if (argc != 1) {
	    Tcl_AppendResult(interp, "wrong # args: should be \"",
		    cmdName, option, " all", (char *) NULL);
	    return TCL_ERROR;
	}

	for (itemPtr = canvas3dPtr->firstItemPtr; itemPtr != NULL;
		itemPtr = itemPtr->nextPtr) {
	    DoItem(interp, itemPtr, uid);
	}
    } else if ((c == 'b') && (strncmp(argv[0], "below", length) == 0)) {
	if (argc != 2) {
	    Tcl_AppendResult(interp, "wrong # args: should be \"",
		    cmdName, option, " below tagOrId", (char *) NULL);
	    return TCL_ERROR;
	}
	itemPtr = StartTagSearch(canvas3dPtr, argv[1], &search);
	if (search.prevPtr != NULL) {
	    DoItem(interp, search.prevPtr, uid);
	}
    } else if ((c == 'c') && (strncmp(argv[0], "closest", length) == 0)) {
	double closestDist;
	Tk_Item3d *startPtr, *closestPtr;
	double coords[2], halo;
	int x1, y1, x2, y2;

	if ((argc < 3) || (argc > 5)) {
	    Tcl_AppendResult(interp, "wrong # args: should be \"",
		    cmdName, option, " closest x y ?halo? ?start?",
		    (char *) NULL);
	    return TCL_ERROR;
	}
	if ((Tk_CanvasGetCoord(interp, (Tk_Canvas3d) canvas3dPtr, argv[1],
		&coords[0]) != TCL_OK) || (Tk_CanvasGetCoord(interp,
		(Tk_Canvas3d) canvas3dPtr, argv[2], &coords[1]) != TCL_OK)) {
	    return TCL_ERROR;
	}
	if (argc > 3) {
	    if (Tk_CanvasGetCoord(interp, (Tk_Canvas3d) canvas3dPtr, argv[3],
		    &halo) != TCL_OK) {
		return TCL_ERROR;
	    }
	    if (halo < 0.0) {
		Tcl_AppendResult(interp, "can't have negative halo value \"",
			argv[3], "\"", (char *) NULL);
		return TCL_ERROR;
	    }
	} else {
	    halo = 0.0;
	}

	/*
	 * Find the item at which to start the search.
	 */

	startPtr = canvas3dPtr->firstItemPtr;
	if (argc == 5) {
	    itemPtr = StartTagSearch(canvas3dPtr, argv[4], &search);
	    if (itemPtr != NULL) {
		startPtr = itemPtr;
	    }
	}

	/*
	 * The code below is optimized so that it can eliminate most
	 * items without having to call their item-specific procedures.
	 * This is done by keeping a bounding box (x1, y1, x2, y2) that
	 * an item's bbox must overlap if the item is to have any
	 * chance of being closer than the closest so far.
	 */

	itemPtr = startPtr;
	if (itemPtr == NULL) {
	    return TCL_OK;
	}
	/*closestDist = (*itemPtr->typePtr->pointProc)((Tk_Canvas3d) canvas3dPtr,
		itemPtr, coords) - halo;*/
	if (closestDist < 0.0) {
	    closestDist = 0.0;
	}
	while (1) {
	    double newDist;

	    /*
	     * Update the bounding box using itemPtr, which is the
	     * new closest item.
	     */

	    x1 = (int)(coords[0] - closestDist - halo - 1);
	    y1 = (int)(coords[1] - closestDist - halo - 1);
	    x2 = (int)(coords[0] + closestDist + halo + 1);
	    y2 = (int)(coords[1] + closestDist + halo + 1);
	    closestPtr = itemPtr;

	    /*
	     * Search for an item that beats the current closest one.
	     * Work circularly through the canvas3d's item list until
	     * getting back to the starting item.
	     */

	    while (1) {
		itemPtr = itemPtr->nextPtr;
		if (itemPtr == NULL) {
		    itemPtr = canvas3dPtr->firstItemPtr;
		}
		if (itemPtr == startPtr) {
		    DoItem(interp, closestPtr, uid);
		    return TCL_OK;
		}
		if ((itemPtr->x1 >= x2) || (itemPtr->x2 <= x1)
			|| (itemPtr->y1 >= y2) || (itemPtr->y2 <= y1)) {
		    continue;
		}
		/*newDist = (*itemPtr->typePtr->pointProc)((Tk_Canvas3d) canvas3dPtr,
			itemPtr, coords) - halo;*/
		if (newDist < 0.0) {
		    newDist = 0.0;
		}
		if (newDist <= closestDist) {
		    closestDist = newDist;
		    break;
		}
	    }
	}
    } else if ((c == 'e') && (strncmp(argv[0], "enclosed", length) == 0)) {
	if (argc != 5) {
	    Tcl_AppendResult(interp, "wrong # args: should be \"",
		    cmdName, option, " enclosed x1 y1 x2 y2", (char *) NULL);
	    return TCL_ERROR;
	}
	return FindArea(interp, canvas3dPtr, argv+1, uid, 1);
    } else if ((c == 'o') && (strncmp(argv[0], "overlapping", length) == 0)) {
	if (argc != 5) {
	    Tcl_AppendResult(interp, "wrong # args: should be \"",
		    cmdName, option, " overlapping x1 y1 x2 y2",
		    (char *) NULL);
	    return TCL_ERROR;
	}
	return FindArea(interp, canvas3dPtr, argv+1, uid, 0);
    } else if ((c == 'w') && (strncmp(argv[0], "withtag", length) == 0)) {
		if (argc != 2) {
	    Tcl_AppendResult(interp, "wrong # args: should be \"",
		    cmdName, option, " withtag tagOrId", (char *) NULL);
	    return TCL_ERROR;
	}
	for (itemPtr = StartTagSearch(canvas3dPtr, argv[1], &search);
		itemPtr != NULL; itemPtr = NextItem(&search)) {
	    DoItem(interp, itemPtr, uid);
	}
    } else  {
	Tcl_AppendResult(interp, "bad search command \"", argv[0],
		"\": must be above, all, below, closest, enclosed, ",
		"overlapping, or withtag", (char *) NULL);
	return TCL_ERROR;
    }
    return TCL_OK;
}
 
/*
 *--------------------------------------------------------------
 *
 * FindArea --
 *
 *	This procedure implements area searches for the "find"
 *	and "addtag" options.
 *
 * Results:
 *	A standard Tcl return value.  If newTag is NULL, then a
 *	list of ids from all the items overlapping or enclosed
 *	by the rectangle given by argc is returned in interp->result.
 *	If newTag is NULL, then the normal interp->result is an
 *	empty string.  If an error occurs, then interp->result will
 *	hold an error message.
 *
 * Side effects:
 *	If uid is non-NULL, then all the items overlapping
 *	or enclosed by the area in argv have that tag added to
 *	their lists of tags.
 *
 *--------------------------------------------------------------
 */

static int
FindArea(interp, canvas3dPtr, argv, uid, enclosed)
    Tcl_Interp *interp;			/* Interpreter for error reporting
					 * and result storing. */
    TkCanvas3d *canvas3dPtr;		/* Canvas3d whose items are to be
					 * searched. */
    char **argv;			/* Array of four arguments that
					 * give the coordinates of the
					 * rectangular area to search. */
    Tk_Uid uid;				/* If non-NULL, gives new tag to set
					 * on all found items;  if NULL, then
					 * ids of found items are returned
					 * in interp->result. */
    int enclosed;			/* 0 means overlapping or enclosed
					 * items are OK, 1 means only enclosed
					 * items are OK. */
{
    double rect[4], tmp;
    int x1, y1, x2, y2;
    Tk_Item3d *itemPtr;

    if ((Tk_CanvasGetCoord(interp, (Tk_Canvas3d) canvas3dPtr, argv[0],
		&rect[0]) != TCL_OK)
	    || (Tk_CanvasGetCoord(interp, (Tk_Canvas3d) canvas3dPtr, argv[1],
		&rect[1]) != TCL_OK)
	    || (Tk_CanvasGetCoord(interp, (Tk_Canvas3d) canvas3dPtr, argv[2],
		&rect[2]) != TCL_OK)
	    || (Tk_CanvasGetCoord(interp, (Tk_Canvas3d) canvas3dPtr, argv[3],
		&rect[3]) != TCL_OK)) {
	return TCL_ERROR;
    }
    if (rect[0] > rect[2]) {
	tmp = rect[0]; rect[0] = rect[2]; rect[2] = tmp;
    }
    if (rect[1] > rect[3]) {
	tmp = rect[1]; rect[1] = rect[3]; rect[3] = tmp;
    }

    /*
     * Use an integer bounding box for a quick test, to avoid
     * calling item-specific code except for items that are close.
     */

    x1 = (int)(rect[0]-1.0);
    y1 = (int)(rect[1]-1.0);
    x2 = (int)(rect[2]+1.0);
    y2 = (int)(rect[3]+1.0);
    for (itemPtr = canvas3dPtr->firstItemPtr; itemPtr != NULL;
	    itemPtr = itemPtr->nextPtr) {
	if ((itemPtr->x1 >= x2) || (itemPtr->x2 <= x1)
		|| (itemPtr->y1 >= y2) || (itemPtr->y2 <= y1)) {
	    continue;
	}
	/*if ((*itemPtr->typePtr->areaProc)((Tk_Canvas3d) canvas3dPtr, itemPtr, rect)
		>= enclosed) {
	    DoItem(interp, itemPtr, uid);
	}*/
    }
    return TCL_OK;
}
 
/*
 *--------------------------------------------------------------
 *
 * RelinkItems --
 *
 *	Move one or more items to a different place in the
 *	display order for a canvas3d.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	The items identified by "tag" are moved so that they
 *	are all together in the display list and immediately
 *	after prevPtr.  The order of the moved items relative
 *	to each other is not changed.
 *
 *--------------------------------------------------------------
 */

static void
RelinkItems(canvas3dPtr, tag, prevPtr)
    TkCanvas3d *canvas3dPtr;	/* Canvas3d to be modified. */
    char *tag;			/* Tag identifying items to be moved
				 * in the redisplay list. */
    Tk_Item3d *prevPtr;		/* Reposition the items so that they
				 * go just after this item (NULL means
				 * put at beginning of list). */
{
    Tk_Item3d *itemPtr;
    TagSearch search;
    Tk_Item3d *firstMovePtr, *lastMovePtr;

    /*
     * Find all of the items to be moved and remove them from
     * the list, making an auxiliary list running from firstMovePtr
     * to lastMovePtr.  Record their areas for redisplay.
     */

    firstMovePtr = lastMovePtr = NULL;
    for (itemPtr = StartTagSearch(canvas3dPtr, tag, &search);
	    itemPtr != NULL; itemPtr = NextItem(&search)) {
	if (itemPtr == prevPtr) {
	    /*
	     * Item after which insertion is to occur is being
	     * moved!  Switch to insert after its predecessor.
	     */

	    prevPtr = search.prevPtr;
	}
	if (search.prevPtr == NULL) {
	    canvas3dPtr->firstItemPtr = itemPtr->nextPtr;
	} else {
	    search.prevPtr->nextPtr = itemPtr->nextPtr;
	}
	if (canvas3dPtr->lastItemPtr == itemPtr) {
	    canvas3dPtr->lastItemPtr = search.prevPtr;
	}
	if (firstMovePtr == NULL) {
	    firstMovePtr = itemPtr;
	} else {
	    lastMovePtr->nextPtr = itemPtr;
	}
	lastMovePtr = itemPtr;
	Tk_Canvas3dEventuallyRedraw((Tk_Canvas3d) canvas3dPtr, itemPtr->x1, itemPtr->y1,
		itemPtr->x2, itemPtr->y2);
	canvas3dPtr->flags |= REPICK_NEEDED;
    }

    /*
     * Insert the list of to-be-moved items back into the canvas3d's
     * at the desired position.
     */

    if (firstMovePtr == NULL) {
	return;
    }
    if (prevPtr == NULL) {
	lastMovePtr->nextPtr = canvas3dPtr->firstItemPtr;
	canvas3dPtr->firstItemPtr = firstMovePtr;
    } else {
	lastMovePtr->nextPtr = prevPtr->nextPtr;
	prevPtr->nextPtr = firstMovePtr;
    }
    if (canvas3dPtr->lastItemPtr == prevPtr) {
	canvas3dPtr->lastItemPtr = lastMovePtr;
    }
}
 
/*
 *--------------------------------------------------------------
 *
 * Canvas3dBindProc --
 *
 *	This procedure is invoked by the Tk dispatcher to handle
 *	events associated with bindings on items.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	Depends on the command invoked as part of the binding
 *	(if there was any).
 *
 *--------------------------------------------------------------
 */

static void
Canvas3dBindProc(clientData, eventPtr)
    ClientData clientData;		/* Pointer to canvas3d structure. */
    XEvent *eventPtr;			/* Pointer to X event that just
					 * happened. */
{
    TkCanvas3d *canvas3dPtr = (TkCanvas3d *) clientData;

    Tcl_Preserve((ClientData) canvas3dPtr);

    /*
     * This code below keeps track of the current modifier state in
     * canvas3dPtr>state.  This information is used to defer repicks of
     * the current item while buttons are down.
     */

    if ((eventPtr->type == ButtonPress) || (eventPtr->type == ButtonRelease)) {
	int mask;

	switch (eventPtr->xbutton.button) {
	    case Button1:
		mask = Button1Mask;
		break;
	    case Button2:
		mask = Button2Mask;
		break;
	    case Button3:
		mask = Button3Mask;
		break;
	    case Button4:
		mask = Button4Mask;
		break;
	    case Button5:
		mask = Button5Mask;
		break;
	    default:
		mask = 0;
		break;
	}

	/*
	 * For button press events, repick the current item using the
	 * button state before the event, then process the event.  For
	 * button release events, first process the event, then repick
	 * the current item using the button state *after* the event
	 * (the button has logically gone up before we change the
	 * current item).
	 */

	if (eventPtr->type == ButtonPress) {
	    /*
	     * On a button press, first repick the current item using
	     * the button state before the event, the process the event.
	     */

	    canvas3dPtr->state = eventPtr->xbutton.state;
	    PickCurrentItem(canvas3dPtr, eventPtr);
	    canvas3dPtr->state ^= mask;
	    Canvas3dDoEvent(canvas3dPtr, eventPtr);
	} else {
	    /*
	     * Button release: first process the event, with the button
	     * still considered to be down.  Then repick the current
	     * item under the assumption that the button is no longer down.
	     */

	    canvas3dPtr->state = eventPtr->xbutton.state;
	    Canvas3dDoEvent(canvas3dPtr, eventPtr);
	    eventPtr->xbutton.state ^= mask;
	    canvas3dPtr->state = eventPtr->xbutton.state;
	    PickCurrentItem(canvas3dPtr, eventPtr);
	    eventPtr->xbutton.state ^= mask;
	}
	goto done;
    } else if ((eventPtr->type == EnterNotify)
	    || (eventPtr->type == LeaveNotify)) {
	canvas3dPtr->state = eventPtr->xcrossing.state;
	PickCurrentItem(canvas3dPtr, eventPtr);
	goto done;
    } else if (eventPtr->type == MotionNotify) {
	canvas3dPtr->state = eventPtr->xmotion.state;
	PickCurrentItem(canvas3dPtr, eventPtr);
    }
    Canvas3dDoEvent(canvas3dPtr, eventPtr);

    done:
    Tcl_Release((ClientData) canvas3dPtr);
}
 
/*
 *--------------------------------------------------------------
 *
 * PickCurrentItem --
 *
 *	Find the topmost item in a canvas3d that contains a given
 *	location and mark the the current item.  If the current
 *	item has changed, generate a fake exit event on the old
 *	current item and a fake enter event on the new current
 *	item.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	The current item for canvas3dPtr may change.  If it does,
 *	then the commands associated with item entry and exit
 *	could do just about anything.  A binding script could
 *	delete the canvas3d, so callers should protect themselves
 *	with Tcl_Preserve and Tcl_Release.
 *
 *--------------------------------------------------------------
 */

static void
PickCurrentItem(canvas3dPtr, eventPtr)
    TkCanvas3d *canvas3dPtr;		/* Canvas3d widget in which to select
					 * current item. */
    XEvent *eventPtr;			/* Event describing location of
					 * mouse cursor.  Must be EnterWindow,
					 * LeaveWindow, ButtonRelease, or
					 * MotionNotify. */
{
    double coords[2];
    int buttonDown;

    /*
     * Check whether or not a button is down.  If so, we'll log entry
     * and exit into and out of the current item, but not entry into
     * any other item.  This implements a form of grabbing equivalent
     * to what the X server does for windows.
     */

    buttonDown = canvas3dPtr->state
	    & (Button1Mask|Button2Mask|Button3Mask|Button4Mask|Button5Mask);
    if (!buttonDown) {
	canvas3dPtr->flags &= ~LEFT_GRABBED_ITEM;
    }

    /*
     * Save information about this event in the canvas3d.  The event in
     * the canvas3d is used for two purposes:
     *
     * 1. Event bindings: if the current item changes, fake events are
     *    generated to allow item-enter and item-leave bindings to trigger.
     * 2. Reselection: if the current item gets deleted, can use the
     *    saved event to find a new current item.
     * Translate MotionNotify events into EnterNotify events, since that's
     * what gets reported to item handlers.
     */

    if (eventPtr != &canvas3dPtr->pickEvent) {
	if ((eventPtr->type == MotionNotify)
		|| (eventPtr->type == ButtonRelease)) {
	    canvas3dPtr->pickEvent.xcrossing.type = EnterNotify;
	    canvas3dPtr->pickEvent.xcrossing.serial = eventPtr->xmotion.serial;
	    canvas3dPtr->pickEvent.xcrossing.send_event
		    = eventPtr->xmotion.send_event;
	    canvas3dPtr->pickEvent.xcrossing.display = eventPtr->xmotion.display;
	    canvas3dPtr->pickEvent.xcrossing.window = eventPtr->xmotion.window;
	    canvas3dPtr->pickEvent.xcrossing.root = eventPtr->xmotion.root;
	    canvas3dPtr->pickEvent.xcrossing.subwindow = None;
	    canvas3dPtr->pickEvent.xcrossing.time = eventPtr->xmotion.time;
	    canvas3dPtr->pickEvent.xcrossing.x = eventPtr->xmotion.x;
	    canvas3dPtr->pickEvent.xcrossing.y = eventPtr->xmotion.y;
	    canvas3dPtr->pickEvent.xcrossing.x_root = eventPtr->xmotion.x_root;
	    canvas3dPtr->pickEvent.xcrossing.y_root = eventPtr->xmotion.y_root;
	    canvas3dPtr->pickEvent.xcrossing.mode = NotifyNormal;
	    canvas3dPtr->pickEvent.xcrossing.detail = NotifyNonlinear;
	    canvas3dPtr->pickEvent.xcrossing.same_screen
		    = eventPtr->xmotion.same_screen;
	    canvas3dPtr->pickEvent.xcrossing.state = eventPtr->xmotion.state;
	} else  {
	    canvas3dPtr->pickEvent = *eventPtr;
	}
    }

    /*
     * If this is a recursive call (there's already a partially completed
     * call pending on the stack;  it's in the middle of processing a
     * Leave event handler for the old current item) then just return;
     * the pending call will do everything that's needed.
     */

    if (canvas3dPtr->flags & REPICK_IN_PROGRESS) {
	return;
    }

    /*
     * A LeaveNotify event automatically means that there's no current
     * object, so the check for closest item can be skipped.
     */

    coords[0] = canvas3dPtr->pickEvent.xcrossing.x + canvas3dPtr->xOrigin;
    coords[1] = canvas3dPtr->pickEvent.xcrossing.y + canvas3dPtr->yOrigin;
    if (canvas3dPtr->pickEvent.type != LeaveNotify) {
	canvas3dPtr->newCurrentPtr = Canvas3dFindClosest(canvas3dPtr, coords);
    } else {
	canvas3dPtr->newCurrentPtr = NULL;
    }

    if ((canvas3dPtr->newCurrentPtr == canvas3dPtr->currentItemPtr)
	    && !(canvas3dPtr->flags & LEFT_GRABBED_ITEM)) {
	/*
	 * Nothing to do:  the current item hasn't changed.
	 */

	return;
    }

    /*
     * Simulate a LeaveNotify event on the previous current item and
     * an EnterNotify event on the new current item.  Remove the "current"
     * tag from the previous current item and place it on the new current
     * item.
     */

    if ((canvas3dPtr->newCurrentPtr != canvas3dPtr->currentItemPtr)
	    && (canvas3dPtr->currentItemPtr != NULL)
	    && !(canvas3dPtr->flags & LEFT_GRABBED_ITEM)) {
	XEvent event;
	Tk_Item3d *itemPtr = canvas3dPtr->currentItemPtr;
	int i;

	event = canvas3dPtr->pickEvent;
	event.type = LeaveNotify;

	/*
	 * If the event's detail happens to be NotifyInferior the
	 * binding mechanism will discard the event.  To be consistent,
	 * always use NotifyAncestor.
	 */

	event.xcrossing.detail = NotifyAncestor;
	canvas3dPtr->flags |= REPICK_IN_PROGRESS;
	Canvas3dDoEvent(canvas3dPtr, &event);
	canvas3dPtr->flags &= ~REPICK_IN_PROGRESS;

	/*
	 * The check below is needed because there could be an event
	 * handler for <LeaveNotify> that deletes the current item.
	 */

	if ((itemPtr == canvas3dPtr->currentItemPtr) && !buttonDown) {
	    for (i = itemPtr->numTags-1; i >= 0; i--) {
		if (itemPtr->tagPtr[i] == currentUid) {
		    itemPtr->tagPtr[i] = itemPtr->tagPtr[itemPtr->numTags-1];
		    itemPtr->numTags--;
		    break;
		}
	    }
	}
    
	/*
	 * Note:  during Canvas3dDoEvent above, it's possible that
	 * canvas3dPtr->newCurrentPtr got reset to NULL because the
	 * item was deleted.
	 */
    }
    if ((canvas3dPtr->newCurrentPtr != canvas3dPtr->currentItemPtr) && buttonDown) {
	canvas3dPtr->flags |= LEFT_GRABBED_ITEM;
	return;
    }

    /*
     * Special note:  it's possible that canvas3dPtr->newCurrentPtr ==
     * canvas3dPtr->currentItemPtr here.  This can happen, for example,
     * if LEFT_GRABBED_ITEM was set.
     */

    canvas3dPtr->flags &= ~LEFT_GRABBED_ITEM;
    canvas3dPtr->currentItemPtr = canvas3dPtr->newCurrentPtr;
    if (canvas3dPtr->currentItemPtr != NULL) {
	XEvent event;

	DoItem((Tcl_Interp *) NULL, canvas3dPtr->currentItemPtr, currentUid);
	event = canvas3dPtr->pickEvent;
	event.type = EnterNotify;
	event.xcrossing.detail = NotifyAncestor;
	Canvas3dDoEvent(canvas3dPtr, &event);
    }
}
 
/*
 *----------------------------------------------------------------------
 *
 * Canvas3dFindClosest --
 *
 *	Given x and y coordinates, find the topmost canvas3d item that
 *	is "close" to the coordinates.
 *
 * Results:
 *	The return value is a pointer to the topmost item that is
 *	close to (x,y), or NULL if no item is close.
 *
 * Side effects:
 *	None.
 *
 *----------------------------------------------------------------------
 */

static Tk_Item3d *
Canvas3dFindClosest(canvas3dPtr, coords)
    TkCanvas3d *canvas3dPtr;		/* Canvas3d widget to search. */
    double coords[2];			/* Desired x,y position in canvas3d,
					 * not screen, coordinates.) */
{
    Tk_Item3d *itemPtr;
    Tk_Item3d *bestPtr;
    int x1, y1, x2, y2;

    x1 = (int)(coords[0] - canvas3dPtr->closeEnough);
    y1 = (int)(coords[1] - canvas3dPtr->closeEnough);
    x2 = (int)(coords[0] + canvas3dPtr->closeEnough);
    y2 = (int)(coords[1] + canvas3dPtr->closeEnough);

    bestPtr = NULL;
    for (itemPtr = canvas3dPtr->firstItemPtr; itemPtr != NULL;
	    itemPtr = itemPtr->nextPtr) {
	if ((itemPtr->x1 > x2) || (itemPtr->x2 < x1)
		|| (itemPtr->y1 > y2) || (itemPtr->y2 < y1)) {
	    continue;
	}
	/*if ((*itemPtr->typePtr->pointProc)((Tk_Canvas3d) canvas3dPtr,
		itemPtr, coords) <= canvas3dPtr->closeEnough) {
	    bestPtr = itemPtr;
	}*/
    }
    return bestPtr;
}
 
/*
 *--------------------------------------------------------------
 *
 * Canvas3dDoEvent --
 *
 *	This procedure is called to invoke binding processing
 *	for a new event that is associated with the current item
 *	for a canvas3d.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	Depends on the bindings for the canvas3d.  A binding script
 *	could delete the canvas3d, so callers should protect themselves
 *	with Tcl_Preserve and Tcl_Release.
 *
 *--------------------------------------------------------------
 */

static void
Canvas3dDoEvent(canvas3dPtr, eventPtr)
    TkCanvas3d *canvas3dPtr;		/* Canvas3d widget in which event
					 * occurred. */
    XEvent *eventPtr;			/* Real or simulated X event that
					 * is to be processed. */
{
#define NUM_STATIC 3
    ClientData staticObjects[NUM_STATIC];
    ClientData *objectPtr;
    int numObjects, i;
    Tk_Item3d *itemPtr;

    if (canvas3dPtr->bindingTable == NULL) {
	return;
    }

    itemPtr = canvas3dPtr->currentItemPtr;
    if (itemPtr == NULL) {
	return;
    }

    /*
     * Set up an array with all the relevant objects for processing
     * this event.  The relevant objects are (a) the event's item,
     * (b) the tags associated with the event's item, and (c) the
     * tag "all".  If there are a lot of tags then malloc an array
     * to hold all of the objects.
     */

    numObjects = itemPtr->numTags + 2;
    if (numObjects <= NUM_STATIC) {
	objectPtr = staticObjects;
    } else {
	objectPtr = (ClientData *) ckalloc((unsigned)
		(numObjects * sizeof(ClientData)));
    }
    objectPtr[0] = (ClientData) allUid;
    for (i = itemPtr->numTags-1; i >= 0; i--) {
	objectPtr[i+1] = (ClientData) itemPtr->tagPtr[i];
    }
    objectPtr[itemPtr->numTags+1] = (ClientData) itemPtr;

    /*
     * Invoke the binding system, then free up the object array if
     * it was malloc-ed.
     */

    if (canvas3dPtr->tkwin != NULL) {
	Tk_BindEvent(canvas3dPtr->bindingTable, eventPtr, canvas3dPtr->tkwin,
		numObjects, objectPtr);
    }
    if (objectPtr != staticObjects) {
	ckfree((char *) objectPtr);
    }
}
 

/*
 *--------------------------------------------------------------
 *
 * GridAlign --
 *
 *	Given a coordinate and a grid spacing, this procedure
 *	computes the location of the nearest grid line to the
 *	coordinate.
 *
 * Results:
 *	The return value is the location of the grid line nearest
 *	to coord.
 *
 * Side effects:
 *	None.
 *
 *--------------------------------------------------------------
 */

static double
GridAlign(coord, spacing)
    double coord;		/* Coordinate to grid-align. */
    double spacing;		/* Spacing between grid lines.   If <= 0
				 * then no alignment is done. */
{
    if (spacing <= 0.0) {
	return coord;
    }
    if (coord < 0) {
	return -((int) ((-coord)/spacing + 0.5)) * spacing;
    }
    return ((int) (coord/spacing + 0.5)) * spacing;
}
 

/*
 *--------------------------------------------------------------
 *
 * Canvas3dSetOrigin --
 *
 *	This procedure is invoked to change the mapping between
 *	canvas3d coordinates and screen coordinates in the canvas3d
 *	window.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	The canvas3d will be redisplayed to reflect the change in
 *	view.  
 *
 *--------------------------------------------------------------
 */

static void
Canvas3dSetOrigin(canvas3dPtr, xOrigin, yOrigin)
    TkCanvas3d *canvas3dPtr;	/* Information about canvas3d. */
    int xOrigin;		/* New X origin for canvas3d (canvas3d x-coord
				 * corresponding to left edge of canvas3d
				 * window). */
    int yOrigin;		/* New Y origin for canvas3d (canvas3d y-coord
				 * corresponding to top edge of canvas3d
				 * window). */
{
    if ((xOrigin == canvas3dPtr->xOrigin) && (yOrigin == canvas3dPtr->yOrigin)) {
	return;
    }

    /*
     * Tricky point: must redisplay not only everything that's visible
     * in the window's final configuration, but also everything that was
     * visible in the initial configuration.  This is needed because some
     * item types, like windows, need to know when they move off-screen
     * so they can explicitly undisplay themselves.
     */

    Tk_Canvas3dEventuallyRedraw((Tk_Canvas3d) canvas3dPtr,
	    canvas3dPtr->xOrigin, canvas3dPtr->yOrigin,
	    canvas3dPtr->xOrigin + Tk_Width(canvas3dPtr->tkwin),
	    canvas3dPtr->yOrigin + Tk_Height(canvas3dPtr->tkwin));
    canvas3dPtr->xOrigin = xOrigin;
    canvas3dPtr->yOrigin = yOrigin;
    Tk_Canvas3dEventuallyRedraw((Tk_Canvas3d) canvas3dPtr,
	    canvas3dPtr->xOrigin, canvas3dPtr->yOrigin,
	    canvas3dPtr->xOrigin + Tk_Width(canvas3dPtr->tkwin),
	    canvas3dPtr->yOrigin + Tk_Height(canvas3dPtr->tkwin));
}
