/* 
 * tkCanv3dView.c --
 *
 *	This file implements viewport items for canvas3d widgets.
 *
 */

#include "tkCanvas3dInt.h"


/*
 * Prototypes for procedures defined in this file. 
 */

static int		ConfigureView _ANSI_ARGS_((Tcl_Interp *interp,
			    Tk_Canvas3d canvas3d, Tk_Item3d *itemPtr, int argc,
			    char **argv, int flags));
static int		CreateView _ANSI_ARGS_((Tcl_Interp *interp,
			    Tk_Canvas3d canvas3d, struct Tk_Item3d *itemPtr,
			    int argc, char **argv));
static void		DeleteView _ANSI_ARGS_((Tk_Canvas3d canvas3d,
			    Tk_Item3d *itemPtr));
static void		DisplayView _ANSI_ARGS_((Tk_Canvas3d canvas3d,
			    Tk_Item3d *itemPtr));
static void		RotateView _ANSI_ARGS_((Tk_Canvas3d canvas3d, Tk_Item3d *itemPtr, 
			    int axis, double angle));
static void		ScaleView _ANSI_ARGS_((Tk_Canvas3d canvas3d,
			    Tk_Item3d *itemPtr, double xScale,
			    double yScale, double zScale));
static void		TransformView _ANSI_ARGS_((Tk_Canvas3d canvas3d,
			    Tk_Item3d *itemPtr, int matrixId));
static void		TranslateView _ANSI_ARGS_((Tk_Canvas3d canvas3d,
			    Tk_Item3d *itemPtr, double xTrans, double yTrans, double zTrans));
static void		FillView _ANSI_ARGS_((Tk_Canvas3d canvas3d, Tk_Item3d *itemPtr, 
			    int fillType));
static void		WrapView _ANSI_ARGS_((Tk_Canvas3d canvas3d, Tk_Item3d *itemPtr, 
			    int wrapStyle));

static int		Tk_Canvas3dSurfParseProc _ANSI_ARGS_((
			    ClientData clientData, Tcl_Interp *interp,
			    Tk_Window tkwin, char *value, char *widgRec,
			    int offset));
static char *		Tk_Canvas3dSurfPrintProc _ANSI_ARGS_((
			    ClientData clientData, Tk_Window tkwin,
			    char *widgRec, int offset,
			    Tcl_FreeProc **freeProcPtr));
static int		Tk_Canvas3dMatrixParseProc _ANSI_ARGS_((
			    ClientData clientData, Tcl_Interp *interp,
			    Tk_Window tkwin, char *value, char *widgRec,
			    int offset));
static char *		Tk_Canvas3dMatrixPrintProc _ANSI_ARGS_((
			    ClientData clientData, Tk_Window tkwin,
			    char *widgRec, int offset,
			    Tcl_FreeProc **freeProcPtr));
static int		Tk_Canvas3dFrustumParseProc _ANSI_ARGS_((
			    ClientData clientData, Tcl_Interp *interp,
			    Tk_Window tkwin, char *value, char *widgRec,
			    int offset));
static char *		Tk_Canvas3dFrustumPrintProc _ANSI_ARGS_((
			    ClientData clientData, Tk_Window tkwin,
			    char *widgRec, int offset,
			    Tcl_FreeProc **freeProcPtr));
static int		Tk_Canvas3dWindowParseProc _ANSI_ARGS_((
			    ClientData clientData, Tcl_Interp *interp,
			    Tk_Window tkwin, char *value, char *widgRec,
			    int offset));
static char *		Tk_Canvas3dWindowPrintProc _ANSI_ARGS_((
			    ClientData clientData, Tk_Window tkwin,
			    char *widgRec, int offset,
			    Tcl_FreeProc **freeProcPtr));
static int		Tk_Canvas3dPositionParseProc _ANSI_ARGS_((
			    ClientData clientData, Tcl_Interp *interp,
			    Tk_Window tkwin, char *value, char *widgRec,
			    int offset));
static char *		Tk_Canvas3dPositionPrintProc _ANSI_ARGS_((
			    ClientData clientData, Tk_Window tkwin,
			    char *widgRec, int offset,
			    Tcl_FreeProc **freeProcPtr));


/*
 * Information used for parsing configuration specs.  If you change any
 * of the default strings, be sure to change the corresponding default
 * values in CreateView.
 */

static Tk_CustomOption tagsOption = {Tk_Canvas3dTagsParseProc,
    Tk_Canvas3dTagsPrintProc, (ClientData) NULL
};
static Tk_CustomOption surfOption = {Tk_Canvas3dSurfParseProc,
    Tk_Canvas3dSurfPrintProc, (ClientData) NULL
};
static Tk_CustomOption matrixOption = {Tk_Canvas3dMatrixParseProc,
    Tk_Canvas3dMatrixPrintProc, (ClientData) NULL
};
static Tk_CustomOption frustumOption = {Tk_Canvas3dFrustumParseProc,
    Tk_Canvas3dFrustumPrintProc, (ClientData) NULL
};
static Tk_CustomOption windowOption = {Tk_Canvas3dWindowParseProc,
    Tk_Canvas3dWindowPrintProc, (ClientData) NULL
};
static Tk_CustomOption positionOption = {Tk_Canvas3dPositionParseProc,
    Tk_Canvas3dPositionPrintProc, (ClientData) NULL
};
static Tk_ConfigSpec configSpecs[] = {
    {TK_CONFIG_CUSTOM, "-background", (char *) NULL, (char *) NULL,
	NULL, Tk_Offset(ViewItem, background), TK_CONFIG_NULL_OK, &surfOption},
    {TK_CONFIG_CUSTOM, "-frustum", (char *) NULL, (char *) NULL,
	NULL, 0, TK_CONFIG_NULL_OK, &frustumOption}, 
    {TK_CONFIG_CUSTOM, "-normal", (char *) NULL, (char *) NULL,
	"0.0 0.0 1.0", Tk_Offset(ViewItem, normal), TK_CONFIG_NULL_OK, &positionOption},
    {TK_CONFIG_CUSTOM, "-proj", (char *) NULL, (char *) NULL,
	NULL, Tk_Offset(ViewItem, projMat), TK_CONFIG_NULL_OK, &matrixOption},
    {TK_CONFIG_CUSTOM, "-up", (char *) NULL, (char *) NULL,
	"0.0 1.0 0.0", Tk_Offset(ViewItem, up), TK_CONFIG_NULL_OK, &positionOption},	
    {TK_CONFIG_CUSTOM, "-view", (char *) NULL, (char *) NULL,
	NULL, Tk_Offset(ViewItem, viewMat), TK_CONFIG_NULL_OK, &matrixOption}, 
    {TK_CONFIG_CUSTOM, "-viewpoint", (char *) NULL, (char *) NULL,
	"0.0 0.0 10.0", Tk_Offset(ViewItem, viewpoint), TK_CONFIG_NULL_OK, &positionOption},
    {TK_CONFIG_CUSTOM, "-window", (char *) NULL, (char *) NULL,
	NULL, 0 , TK_CONFIG_NULL_OK, &windowOption}, 
    {TK_CONFIG_CUSTOM, "-world", (char *) NULL, (char *) NULL,
	NULL, Tk_Offset(ViewItem, worldMat), TK_CONFIG_NULL_OK, &matrixOption},
    {TK_CONFIG_END, (char *) NULL, (char *) NULL, (char *) NULL,
	(char *) NULL, 0, 0}
};

/*
 * The structures below defines the view item type by means
 * of procedures that can be invoked by generic item code.
 */

Tk_Item3dType tkViewType = {
    "viewport",				/* name */
    sizeof(ViewItem),			/* itemSize */
    CreateView,				/* createProc */
    configSpecs,			/* configSpecs */
    ConfigureView,			/* configureProc */
    DeleteView,				/* deleteProc */
    DisplayView,			/* displayProc */
    (Tk_Item3dExBufProc *)NULL,		/* exbufProc */
    (Tk_Item3dRotateProc *)NULL,	/* rotateProc */
    (Tk_Item3dScaleProc *)NULL,		/* scaleProc */
    (Tk_Item3dTransformProc *)NULL,	/* transformProc */
    (Tk_Item3dTranslateProc *)NULL,	/* translateProc */
    (Tk_Item3dFillProc *)NULL,		/* fillProc */
    (Tk_Item3dWrapProc *)NULL,		/* wrapProc */
    (Tk_Item3dType *) NULL		/* nextPtr */
};

 
/*
 *--------------------------------------------------------------
 *
 * CreateView --
 *
 *	This procedure is invoked to create a new view item in
 *	a canvas3d.
 *
 * Results:
 *	A standard Tcl return value.  If an error occurred in
 *	creating the item, then an error message is left in
 *	interp->result;  in this case itemPtr is left uninitialized,
 *	so it can be safely freed by the caller.
 *
 * Side effects:
 *	A new view item is created.
 *
 *--------------------------------------------------------------
 */

static int
CreateView(interp, canvas3d, itemPtr, argc, argv)
    Tcl_Interp *interp;			/* Interpreter for error reporting. */
    Tk_Canvas3d canvas3d;		/* Canvas3d to hold new item. */
    Tk_Item3d *itemPtr;			/* Record to hold new item;  header
					 * has been initialized by caller. */
    int argc;				/* Number of arguments in argv. */
    char **argv;			/* Arguments describing view. */
{
    ViewItem *viewPtr = (ViewItem *) itemPtr;

    /*
     * Carry out initialization that is needed to set defaults and to
     * allow proper cleanup after errors during the the remainder of
     * this procedure.
     */

    viewPtr->canvas3d = canvas3d;
    viewPtr->itemCode = ITEM_VIEWPORT;
    itemPtr->itemCode = ITEM_VIEWPORT;
    viewPtr->background = (SurfItem *)ckalloc(sizeof(SurfItem));
    viewPtr->background = NULL;
    viewPtr->halfHeight = 0.0;
    viewPtr->backPlanePos = 0.0; 
    viewPtr->frontPlanePos = 0.0;
    viewPtr->scaleX = 0.0;
    viewPtr->scaleY = 0.0;
    viewPtr->originX = 0.0;
    viewPtr->originY = 0.0;
    viewPtr->viewpoint.x = 0.0;
    viewPtr->viewpoint.y = 0.0;
    viewPtr->viewpoint.z = 10.0;
    viewPtr->normal.x = 0.0;
    viewPtr->normal.y = 0.0;
    viewPtr->normal.z = 1.0;
    viewPtr->up.x = 0.0;
    viewPtr->up.y = 1.0;
    viewPtr->up.z = 0.0;
    viewPtr->projMat = (MatrixItem *)NULL;
    viewPtr->worldMat = (MatrixItem *)NULL;
    viewPtr->viewMat = (MatrixItem *)NULL;

    if (ConfigureView(interp, canvas3d, itemPtr, argc, argv, 0) == TCL_OK) {
	return TCL_OK;
    } else {
	DeleteView(canvas3d, itemPtr);
	return TCL_ERROR;
    }
}
  
/*
 *--------------------------------------------------------------
 *
 * ConfigureView --
 *
 *	This procedure is invoked to configure various aspects
 *	of a view item such as its viewpoint.
 *
 * Results:
 *	A standard Tcl result code.  If an error occurs, then
 *	an error message is left in interp->result.
 *
 * Side effects:
 *	None.
 *
 *--------------------------------------------------------------
 */

static int
ConfigureView(interp, canvas3d, itemPtr, argc, argv, flags)
    Tcl_Interp *interp;		/* Used for error reporting. */
    Tk_Canvas3d canvas3d;	/* Canvas3d containing itemPtr. */
    Tk_Item3d *itemPtr;		/* View item to reconfigure. */
    int argc;			/* Number of elements in argv.  */
    char **argv;		/* Arguments describing things to configure. */
    int flags;			/* Flags to pass to Tk_ConfigureWidget. */
{
    ViewItem *viewPtr = (ViewItem *) itemPtr;
    geInterface *facePtr = Tk_Canvas3dFace(canvas3d);
    Tk_Window tkwin = Tk_Canvas3dTkwin(canvas3d);

    if (Tk_ConfigureWidget(interp, tkwin, configSpecs, argc, argv,
	    (char *) viewPtr, flags) != TCL_OK) {
	return TCL_ERROR;
    }

    /* 
     * Set the camera according to configure options.
     */
    GE_SetCamera(facePtr, viewPtr, viewPtr->viewpoint, 
	    viewPtr->normal, viewPtr->up);

    /* 
     * If view matrix is not specified by user, then create it from
     * the projection and world matrices (if they both exist!)
     */

    /*if (viewPtr->viewMat == NULL) {
	if ((viewPtr->projMat != NULL) && (viewPtr->worldMat != NULL)) {
	    viewPtr->viewMat = (MatrixItem *)ckalloc (sizeof(MatrixItem));
	    viewPtr->viewMat->canvas3d = canvas3d;
	    GE_MakeViewMatrix(viewPtr->projMat, viewPtr->worldMat,
			viewPtr->viewMat);
	}
    }*/

    /*
     * Render the projection, world and view matrices with the configure 
     * options.
     */
    if ((viewPtr->projMat != NULL) 
	&& (viewPtr->worldMat != NULL) 
	&& (viewPtr->viewMat != NULL)) {
	    GE_RenderProjWorldView(*facePtr, viewPtr->projMat, 
		viewPtr->worldMat, viewPtr->viewMat);
    }

    /* 
     * If surface specified by -background option, then create a 
     * view background.
     */
    if (viewPtr->background != NULL ) {
	GE_CreateViewBackground(*facePtr, viewPtr->background);
    }
    return TCL_OK;
}
 
/*
 *--------------------------------------------------------------
 *
 * DeleteView --
 *
 *	This procedure is called to clean up the data structure
 *	associated with a view item.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	Resources associated with itemPtr are released.
 *
 *--------------------------------------------------------------
 */

static void
DeleteView(canvas3d, itemPtr)
    Tk_Canvas3d canvas3d;	/* Info about overall canvas3d widget. */
    Tk_Item3d *itemPtr;		/* Item that is being deleted. */
{
    ViewItem *viewPtr = (ViewItem *) itemPtr;
}
  
/*
 *--------------------------------------------------------------
 *
 * DisplayView --
 *
 *	This procedure is invoked to draw a view item in a given
 *	canvas3d.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	ItemPtr is drawn in drawable using the transformation
 *	information in canvas3d.
 *
 *--------------------------------------------------------------
 */

static void
DisplayView(canvas3d, itemPtr)
    Tk_Canvas3d canvas3d;		/* Canvas3d that contains item. */
    Tk_Item3d *itemPtr;			/* Item to be displayed. */
{
    ViewItem *viewPtr = (ViewItem *) itemPtr;
}



/*
 *--------------------------------------------------------------
 *
 * Tk_Canvas3dSurfParseProc --
 *
 *	This procedure is invoked during option processing to handle
 *	SurfItem  options (-background) for the viewport item.
 *
 * Results:
 *	A standard Tcl return value.
 *
 * Side effects:
 *	None.
 *
 *--------------------------------------------------------------
 */

static int
Tk_Canvas3dSurfParseProc(clientData, interp, tkwin, value, widgRec, offset)
    ClientData clientData;		/* Not used.*/
    Tcl_Interp *interp;			/* Used for reporting errors. */
    Tk_Window tkwin;			/* Window containing canvas3d widget. */
    char *value;			/* Value of option (list of tag
					 * names). */
    char *widgRec;			/* Pointer to record for item. */
    int offset;				/* Offset into item (ignored). */
{
    ViewItem *viewPtr = (ViewItem *) widgRec;
    int id;
    Tk_Item3d *itemPtr, *prevPtr;
    TkCanvas3d *canvas3dPtr = (TkCanvas3d *) viewPtr->canvas3d;

    if (offset != Tk_Offset(ViewItem, background)) {
	panic("Canvas3dSurfParseProc received bogus offset");
    }
    
    /* 
     * Get the Surface Item Id from command option input value. 
     */
    if (Tcl_GetInt(interp, value, &id) == TCL_ERROR ) {
	Tcl_AppendResult(interp, 
	    " : bad value for -surface option",(char *)NULL);
		return TCL_ERROR;
    } 
    
    /*
     * Search for the surface in the list of items.
     */
    for (prevPtr = NULL, itemPtr = canvas3dPtr->firstItemPtr;
			itemPtr != NULL;
			prevPtr = itemPtr, itemPtr = itemPtr->nextPtr) {
		    if (itemPtr->id == id) {
			if (itemPtr->itemCode != ITEM_SURFACE) {
			    Tcl_AppendResult(interp, 
				"bad value for -background option",(char *)NULL);
			    return TCL_ERROR;
			}
			viewPtr->background = (SurfItem *)itemPtr;
			break;
		    }
    }
    
    return TCL_OK; 
}


/*
 *--------------------------------------------------------------
 *	   
 * Tk_Canvas3dSurfPrintProc --
 *
 *	This procedure is invoked by the Tk configuration code
 *	to produce a printable string for the "-background" configuration
 *	option for canvas3d items.
 *
 * Results:
 *	The return value is a string describing the current 
 *	"-background" option value for the viewport item.
 *
 * Side effects:
 *	None.
 *
 *--------------------------------------------------------------
 */
static char *
Tk_Canvas3dSurfPrintProc(clientData, tkwin, widgRec, offset, freeProcPtr)
    ClientData clientData;		/* Ignored. */
    Tk_Window tkwin;			/* Window containing canvas3d widget. */
    char *widgRec;			/* Pointer to record for item. */
    int offset;				/* Ignored. */
    Tcl_FreeProc **freeProcPtr;		/* Pointer to variable to fill in with
					 * information about how to reclaim
					 * storage for return string. */
{
    ViewItem *viewPtr = (ViewItem *) widgRec;
    char *buffer;

    if(viewPtr->background == NULL) {
	buffer = (char *)ckalloc(5 * sizeof(char));
	sprintf(buffer, "None");
    } else {
	buffer = (char *) ckalloc(150);
	sprintf(buffer, "%.2f %.2f %.2f {%.2f %.2f %.2f %.2f} {%.2f %.2f %.2f %.2f} {%.2f %.2f %.2f %.2f} %i %s",
	    viewPtr->background->ambientIntensity, viewPtr->background->shininess,
	    viewPtr->background->transparency,
	    viewPtr->background->diffuseColor.r, viewPtr->background->diffuseColor.g,
	    viewPtr->background->diffuseColor.b, viewPtr->background->diffuseColor.a,
	    viewPtr->background->emissiveColor.r,viewPtr->background->emissiveColor.g,
	    viewPtr->background->emissiveColor.b, viewPtr->background->emissiveColor.a,
	    viewPtr->background->specularColor.r, viewPtr->background->specularColor.g,
	    viewPtr->background->specularColor.b, viewPtr->background->specularColor.a,
	    viewPtr->background->ramp, viewPtr->background->textureString);
    }

    *freeProcPtr = TCL_DYNAMIC;
    return buffer;  
} 
 
/*
 *--------------------------------------------------------------
 *
 * Tk_Canvas3dMatrixParseProc --
 *
 *	This procedure is invoked during option processing to handle
 *	SurfItem  options (-view, -proj or -world) for viewport item.
 *
 * Results:
 *	A standard Tcl return value.
 *
 * Side effects:
 *	None.
 *
 *--------------------------------------------------------------
 */

static int
Tk_Canvas3dMatrixParseProc(clientData, interp, tkwin, value, widgRec, offset)
    ClientData clientData;		/* Not used.*/
    Tcl_Interp *interp;			/* Used for reporting errors. */
    Tk_Window tkwin;			/* Window containing canvas3d widget. */
    char *value;			/* Value of option (list of tag
					 * names). */
    char *widgRec;			/* Pointer to record for item. */
    int offset;				/* Offset into item (ignored). */
{
    ViewItem *viewPtr = (ViewItem *) widgRec;
    int id;
    Tk_Item3d *itemPtr, *prevPtr;
    TkCanvas3d *canvas3dPtr = (TkCanvas3d *) viewPtr->canvas3d;

    if ((offset != Tk_Offset(ViewItem, projMat))
	&& (offset != Tk_Offset(ViewItem, worldMat))
	&& (offset != Tk_Offset(ViewItem, viewMat))) {
	panic("Canvas3dMatrixParseProc received bogus offset");
    }
    /* 
     * Get the matrix item id from input option value.
     */
    Tcl_GetInt(interp, value, &id);
    for (prevPtr = NULL, itemPtr = canvas3dPtr->firstItemPtr;
			itemPtr != NULL;
			prevPtr = itemPtr, itemPtr = itemPtr->nextPtr) {
	/*
	 * Search through the list of items for the required matrix.
	 */
	if (itemPtr->id == id) {
	    if (itemPtr->itemCode != ITEM_MATRIX) {
		Tcl_AppendResult(interp, 
			"bad matrix value",(char *)NULL);
			return TCL_ERROR;
	    }
	    /* 
	     * If the parsing is being done for -proj
	     */
	    if (offset == Tk_Offset(ViewItem, projMat))  {
		viewPtr->halfHeight = 0.0;
		viewPtr->backPlanePos = 0.0;
		viewPtr->frontPlanePos = 0.0;
		viewPtr->projMat = (MatrixItem *)itemPtr;
	    return TCL_OK; 	
	    }
	    /* 
	     * If the parsing is being done for -world
	     */
	    if (offset == Tk_Offset(ViewItem, worldMat))  {
		viewPtr->scaleX = viewPtr->scaleY = 0.0;
		viewPtr->originX = viewPtr->originY = 0.0;
		viewPtr->worldMat = (MatrixItem *)itemPtr;
	    return TCL_OK; 	
	    }
	    /* 
	     * If the parsing is being done for -view
	     */
	    if (offset == Tk_Offset(ViewItem, viewMat))  { 
		viewPtr->viewMat = (MatrixItem *)itemPtr;
	    return TCL_OK; 	
	    }
	}
    }
    
    return TCL_OK; 
}

/*
 *--------------------------------------------------------------
 *	   
 * Tk_Canvas3dMatrixPrintProc --
 *
 *	This procedure is invoked by the Tk configuration code
 *	to produce a printable string for the -proj, -world or
 *	-view configuration options for canvas3d items.
 *
 * Results:
 *	The return value is a string describing the current 
 *	-proj, -world or -view option value for the viewport item.
 *
 * Side effects:
 *	None.
 *
 *--------------------------------------------------------------
 */
static char *
Tk_Canvas3dMatrixPrintProc(clientData, tkwin, widgRec, offset, freeProcPtr)
    ClientData clientData;		/* Ignored. */
    Tk_Window tkwin;			/* Window containing canvas3d widget. */
    char *widgRec;			/* Pointer to record for item. */
    int offset;				/* Ignored. */
    Tcl_FreeProc **freeProcPtr;		/* Pointer to variable to fill in with
					 * information about how to reclaim
					 * storage for return string. */
{
    ViewItem *viewPtr = (ViewItem *) widgRec;
    char *buffer;

    buffer = (char *) ckalloc(100 * sizeof(char));
    /*
     * If printing for -proj value.
     */
    if (offset == Tk_Offset(ViewItem, projMat)) {
	if (viewPtr->projMat == NULL) {
	    sprintf(buffer, "None");
	    return buffer;
	}
	sprintf(buffer, "{%.2f %.2f %.2f %.2f} {%.2f %.2f %.2f %.2f} {%.2f %.2f %.2f %.2f} {%.2f %.2f %.2f %.2f}", 
		viewPtr->projMat->matrix.a1, viewPtr->projMat->matrix.a2,
		viewPtr->projMat->matrix.a3, viewPtr->projMat->matrix.a4,
		viewPtr->projMat->matrix.b1, viewPtr->projMat->matrix.b2,
		viewPtr->projMat->matrix.b3, viewPtr->projMat->matrix.b4,
		viewPtr->projMat->matrix.c1, viewPtr->projMat->matrix.c2,
		viewPtr->projMat->matrix.c3, viewPtr->projMat->matrix.c4,
		viewPtr->projMat->matrix.d1, viewPtr->projMat->matrix.d2,
		viewPtr->projMat->matrix.d3, viewPtr->projMat->matrix.d4);
    } else if (offset == Tk_Offset(ViewItem, worldMat)) {
	/* 
	 * If printing for -world option
	 */
	if (viewPtr->worldMat == NULL) {
	    sprintf(buffer, "None");
	    return buffer;
	}
	sprintf(buffer, "{%.2f %.2f %.2f %.2f} {%.2f %.2f %.2f %.2f} {%.2f %.2f %.2f %.2f} {%.2f %.2f %.2f %.2f}", 
		viewPtr->worldMat->matrix.a1, viewPtr->worldMat->matrix.a2,
		viewPtr->worldMat->matrix.a3, viewPtr->worldMat->matrix.a4,
		viewPtr->worldMat->matrix.b1, viewPtr->worldMat->matrix.b2,
		viewPtr->worldMat->matrix.b3, viewPtr->worldMat->matrix.b4,
		viewPtr->worldMat->matrix.c1, viewPtr->worldMat->matrix.c2,
		viewPtr->worldMat->matrix.c3, viewPtr->worldMat->matrix.c4,
		viewPtr->worldMat->matrix.d1, viewPtr->worldMat->matrix.d2,
		viewPtr->worldMat->matrix.d3, viewPtr->worldMat->matrix.d4);
    } else if (offset == Tk_Offset(ViewItem, viewMat)) {
	/*
	 * If printing for -view option.
	 */
	if (viewPtr->viewMat == NULL) {
	    sprintf(buffer, "None");
	    return buffer;
	}
	sprintf(buffer, "{%.2f %.2f %.2f %.2f} {%.2f %.2f %.2f %.2f} {%.2f %.2f %.2f %.2f} {%.2f %.2f %.2f %.2f}", 
		viewPtr->viewMat->matrix.a1, viewPtr->viewMat->matrix.a2,
		viewPtr->viewMat->matrix.a3, viewPtr->viewMat->matrix.a4,
		viewPtr->viewMat->matrix.b1, viewPtr->viewMat->matrix.b2,
		viewPtr->viewMat->matrix.b3, viewPtr->viewMat->matrix.b4,
		viewPtr->viewMat->matrix.c1, viewPtr->viewMat->matrix.c2,
		viewPtr->viewMat->matrix.c3, viewPtr->viewMat->matrix.c4,
		viewPtr->viewMat->matrix.d1, viewPtr->viewMat->matrix.d2,
		viewPtr->viewMat->matrix.d3, viewPtr->viewMat->matrix.d4);
    } else {
	sprintf(buffer, "None");
    }
	
    *freeProcPtr = TCL_DYNAMIC;
    return buffer;
}

/*
 *--------------------------------------------------------------
 *
 * Tk_Canvas3dFrustumParseProc --
 *
 *	This procedure is invoked during option processing to handle
 *	SurfItem  options (-frustum) for viewport item.
 *
 * Results:
 *	A standard Tcl return value.
 *
 * Side effects:
 *	None.
 *
 *--------------------------------------------------------------
 */

static int
Tk_Canvas3dFrustumParseProc(clientData, interp, tkwin, value, widgRec, offset)
    ClientData clientData;		/* Not used.*/
    Tcl_Interp *interp;			/* Used for reporting errors. */
    Tk_Window tkwin;			/* Window containing canvas3d widget. */
    char *value;			/* Value of option (list of tag
					 * names). */
    char *widgRec;			/* Pointer to record for item. */
    int offset;				/* Offset into item (ignored). */
{
    ViewItem *viewPtr = (ViewItem *) widgRec;
    int argc;
    char **argv = NULL;
    TkCanvas3d *canvas3dPtr = (TkCanvas3d *)viewPtr->canvas3d;
    /*
     * Break the value up into the individual values.
     */
    if (Tcl_SplitList(interp, value, &argc, &argv) != TCL_OK) {
	syntaxError:
	Tcl_ResetResult(interp);
	Tcl_AppendResult(interp, "bad frustum value \"", value,
		"\": must be list of format",
		" { halfHeight backPlanePosition frontPlanePosition }",
		(char *) NULL);
	if (argv != NULL) {
	    ckfree((char *) argv);
	}
	return TCL_ERROR;
    }
    if (argc != 3) {
	goto syntaxError;
    }
    /*
     * Convert all the string input values to doubles.
     */
    if ((Tcl_GetDouble(interp, argv[0], &viewPtr->halfHeight) != TCL_OK)
	    || (Tcl_GetDouble(interp, argv[1], &viewPtr->backPlanePos)
		!= TCL_OK)
	    || (Tcl_GetDouble(interp, argv[2], &viewPtr->frontPlanePos)
		!= TCL_OK)) {
	goto syntaxError;
    } else {
	if(viewPtr->projMat != NULL) {
	    ckfree((char *)viewPtr->projMat);
	}
	/*
	 * Prepare a projection matrix depending on the frustum values.
	 */
	viewPtr->projMat = (MatrixItem *)ckalloc(sizeof(MatrixItem));
	viewPtr->projMat->canvas3d = (Tk_Canvas3d)canvas3dPtr;
	GE_MakeProjMatrix(*canvas3dPtr->facePtr, 
	    viewPtr->halfHeight, 
	    viewPtr->backPlanePos, 
	    viewPtr->frontPlanePos, viewPtr->projMat);
  
    }

    ckfree((char *) argv);
    return TCL_OK; 
}

/*
 *--------------------------------------------------------------
 *	   
 * Tk_Canvas3dFrustumPrintProc --
 *
 *	This procedure is invoked by the Tk configuration code
 *	to produce a printable string for the "-frustum" configuration
 *	option for canvas3d items.
 *
 * Results:
 *	The return value is a string describing the current 
 *	"-frustum" option value for the viewport item.
 *
 * Side effects:
 *	None.
 *
 *--------------------------------------------------------------
 */
static char *
Tk_Canvas3dFrustumPrintProc(clientData, tkwin, widgRec, offset, freeProcPtr)
    ClientData clientData;		/* Ignored. */
    Tk_Window tkwin;			/* Window containing canvas3d widget. */
    char *widgRec;			/* Pointer to record for item. */
    int offset;				/* Ignored. */
    Tcl_FreeProc **freeProcPtr;		/* Pointer to variable to fill in with
					 * information about how to reclaim
					 * storage for return string. */
{
    ViewItem *viewPtr = (ViewItem *) widgRec;
    char *buffer;

    buffer = (char *) ckalloc(120);
    sprintf(buffer, "%.2f %.2f %.2f", viewPtr->halfHeight,
	    viewPtr->backPlanePos, viewPtr->frontPlanePos);

    *freeProcPtr = TCL_DYNAMIC;
    return buffer;  
} 

/*
 *--------------------------------------------------------------
 *
 * Tk_Canvas3dWindowParseProc --
 *
 *	This procedure is invoked during option processing to handle
 *	SurfItem  options (-window) for viewport item.
 *
 * Results:
 *	A standard Tcl return value.
 *
 * Side effects:
 *	None.
 *
 *--------------------------------------------------------------
 */

static int
Tk_Canvas3dWindowParseProc(clientData, interp, tkwin, value, widgRec, offset)
    ClientData clientData;		/* Not used.*/
    Tcl_Interp *interp;			/* Used for reporting errors. */
    Tk_Window tkwin;			/* Window containing canvas3d widget. */
    char *value;			/* Value of option (list of tag
					 * names). */
    char *widgRec;			/* Pointer to record for item. */
    int offset;				/* Offset into item (ignored). */
{
    ViewItem *viewPtr = (ViewItem *) widgRec;
    int argc;
    char **argv = NULL;
    TkCanvas3d *canvas3dPtr = (TkCanvas3d *) viewPtr->canvas3d;
    /*
     * Break the value up into the individual values.
     */
    if (Tcl_SplitList(interp, value, &argc, &argv) != TCL_OK) {
	syntaxError:
	Tcl_ResetResult(interp);
	Tcl_AppendResult(interp, "bad window value \"", value,
		"\": must be list of format",
		" { scaleX scaleY originX originY }",
		(char *) NULL);
	if (argv != NULL) {
	    ckfree((char *) argv);
	}
	return TCL_ERROR;
    }
    if (argc != 4) {
	goto syntaxError;
    }

    /*
     * Convert the string input values to doubles.
     */
    if ((Tcl_GetDouble(interp, argv[0], &viewPtr->scaleX) != TCL_OK)
	|| (Tcl_GetDouble(interp, argv[1], &viewPtr->scaleY) != TCL_OK)
	|| (Tcl_GetDouble(interp, argv[2], &viewPtr->originX) != TCL_OK)
	|| (Tcl_GetDouble(interp, argv[3], &viewPtr->originY) != TCL_OK)) {
	goto syntaxError;
    } else {
	if(viewPtr->worldMat != NULL) {
	    ckfree((char *)viewPtr->worldMat);
	}
	viewPtr->worldMat = (MatrixItem *)ckalloc(sizeof(MatrixItem));
	viewPtr->worldMat->canvas3d = (Tk_Canvas3d)canvas3dPtr;
	/*
	 * Make a world matrix based on the -window scaling values/
	 */
	GE_MakeWorldMatrix(*canvas3dPtr->facePtr, 
	    viewPtr->scaleX, viewPtr->scaleY, 
	    viewPtr->originX, viewPtr->originY,  
	    viewPtr->worldMat);
  
    }

    ckfree((char *) argv);
    return TCL_OK; 
}

/*
 *--------------------------------------------------------------
 *	   
 * Tk_Canvas3dWindowPrintProc --
 *
 *	This procedure is invoked by the Tk configuration code
 *	to produce a printable string for the "-window" configuration
 *	option for canvas3d items.
 *
 * Results:
 *	The return value is a string describing the current 
 *	"-window" option value for the viewport item.
 *
 * Side effects:
 *	None.
 *
 *--------------------------------------------------------------
 */

static char *
Tk_Canvas3dWindowPrintProc(clientData, tkwin, widgRec, offset, freeProcPtr)
    ClientData clientData;		/* Ignored. */
    Tk_Window tkwin;			/* Window containing canvas3d widget. */
    char *widgRec;			/* Pointer to record for item. */
    int offset;				/* Ignored. */
    Tcl_FreeProc **freeProcPtr;		/* Pointer to variable to fill in with
					 * information about how to reclaim
					 * storage for return string. */
{
    ViewItem *viewPtr = (ViewItem *) widgRec;
    char *buffer;

    buffer = (char *) ckalloc(120);
    sprintf(buffer, "%.2f %.2f %.2f %.2f", viewPtr->scaleX,
	    viewPtr->scaleY, viewPtr->originX, viewPtr->originY);

    *freeProcPtr = TCL_DYNAMIC;
    return buffer;  
} 
  
/*
 *--------------------------------------------------------------
 *
 * Tk_Canvas3dPositionParseProc --
 *
 *	This procedure is invoked during option processing to handle
 *	SurfItem  options (-viewpoint, -normal or -up) for the 
 *	viewport item.
 *
 * Results:
 *	A standard Tcl return value.
 *
 * Side effects:
 *	None.
 *
 *--------------------------------------------------------------
 */
static int
Tk_Canvas3dPositionParseProc(clientData, interp, tkwin, value, widgRec, offset)
    ClientData clientData;		/* Not used.*/
    Tcl_Interp *interp;			/* Used for reporting errors. */
    Tk_Window tkwin;			/* Window containing canvas3d widget. */
    char *value;			/* Value of option (list of tag
					 * names). */
    char *widgRec;			/* Pointer to record for item. */
    int offset;				/* Offset into item (ignored). */
{
    ViewItem *viewPtr = (ViewItem *) widgRec;
    double x, y, z;
    int argc;
    char **argv = NULL;

    /*
     * Break the value up into the individual values.
     */
    if (Tcl_SplitList(interp, value, &argc, &argv) != TCL_OK) {
	syntaxError:
	Tcl_ResetResult(interp);
	Tcl_AppendResult(interp, "bad value \"", value,
		"\": must be list with three numbers", (char *) NULL);
	if (argv != NULL) {
	    ckfree((char *) argv);
	}
	return TCL_ERROR;
    }
    if (argc != 3) {
	goto syntaxError;
    }

    /* 
     * Convert the string values parsed to doubles.
     */
    if ((Tcl_GetDouble(interp, argv[0], &x) != TCL_OK)
	    || (Tcl_GetDouble(interp, argv[1], &y)
		!= TCL_OK)
	    || (Tcl_GetDouble(interp, argv[2], &z)
		!= TCL_OK)) {
	goto syntaxError;
    }
    /*
     * If parsing is for the -viewpoint option. 
     */
    if (offset == Tk_Offset(ViewItem, viewpoint))  {
	viewPtr->viewpoint.x = x;
	viewPtr->viewpoint.y = y;
	viewPtr->viewpoint.z = z;
    } else if (offset == Tk_Offset(ViewItem, normal))  {
	/* 
	 * If the parsing is for -normal option.
	 */
	viewPtr->normal.x = x;
	viewPtr->normal.y = y;
	viewPtr->normal.z = z;
    } else if (offset == Tk_Offset(ViewItem, up))  {
	/*
	 * If the parsing is for -up option.
	 */
	viewPtr->up.x = x;
	viewPtr->up.y = y;
	viewPtr->up.z = z;
    } else {
	Tcl_AppendResult(interp, "bad offset value in PositionParseProc",
	    (char *) NULL);
	ckfree((char *) argv);
	return TCL_ERROR;
    }

    ckfree((char *) argv);
    return TCL_OK;   
}

/*
 *--------------------------------------------------------------
 *	   
 * Tk_Canvas3dPositionPrintProc --
 *
 *	This procedure is invoked by the Tk configuration code
 *	to produce a printable string for the -viewpoint, -normal or
 *	-up configuration option for canvas3d items.
 *
 * Results:
 *	The return value is a string describing the current 
 *	-viewpoint, -normal or -up option value for the viewport item.
 *
 * Side effects:
 *	None.
 *
 *--------------------------------------------------------------
 */
static char *
Tk_Canvas3dPositionPrintProc(clientData, tkwin, widgRec, offset, freeProcPtr)
    ClientData clientData;		/* Ignored. */
    Tk_Window tkwin;			/* Window containing canvas3d widget. */
    char *widgRec;			/* Pointer to record for item. */
    int offset;				/* Ignored. */
    Tcl_FreeProc **freeProcPtr;		/* Pointer to variable to fill in with
					 * information about how to reclaim
					 * storage for return string. */
{
    ViewItem *viewPtr = (ViewItem *) widgRec;
    char *buffer;

    buffer = (char *) ckalloc(120);

    if (offset == Tk_Offset(ViewItem, viewpoint))  {
	sprintf(buffer, "%.2f %.2f %.2f", viewPtr->viewpoint.x,
	    viewPtr->viewpoint.y, viewPtr->viewpoint.z);
    } else if (offset == Tk_Offset(ViewItem, normal))  {
	sprintf(buffer, "%.2f %.2f %.2f", viewPtr->normal.x,
	    viewPtr->normal.y, viewPtr->normal.z);
    } else if (offset == Tk_Offset(ViewItem, up))  {
	sprintf(buffer, "%.2f %.2f %.2f", viewPtr->up.x,
	    viewPtr->up.y, viewPtr->up.z);
    } else {
	sprintf(buffer, "None");
    }

    *freeProcPtr = TCL_DYNAMIC;
    return buffer;
    
}
